/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2014  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* math.c  April 1999

Copyright (c) 1999-2014  D. R. Williamson

Note: In some Unix versions, file /usr/include/mathcalls.h shows calls
to math functions.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "lib.h"
#include "mat.h"
#include "math1.h"
#include "mmath.h"
#include "mem.h"
#include "sys.h"
#include "tag.h"
#include "tex.h"
#include "sparse.h"
#include "wapp.h"

#ifdef LAPACK
   #include "lapack.h"
#endif

#ifdef ESSL
   #include "essl.h"
#endif

/*--------------------------------------------------------------------*/

/* This loop appears in several matrix-multiply functions, including
   mpy(), mpyad(), mpyad1(), starm().

Computing C=D+A*B where C is initialized to D: */

#define MPYLOOP \
   for(;k<cB;k++) { \
      iC1=k*rA; \
      iCn=iC1+rA; \
      jB1=k*cA; \
      jBn=jB1+cA; \
      jA1=0; \
      for(j=jB1;j<jBn;j++) { \
         Bj=*(B+j); \
         iA=jA1*rA-iC1; \
         for(i=iC1;i<iCn;i++) *(C+i)+=*(A+i+iA)*Bj; \
         jA1++; \
      } \
   }
/*--------------------------------------------------------------------*/

/* Term-by-term operations for byterm1(): */
enum byterm1 {NOT,NOTT,EQ0,GT0,LT0,NE0,INT,INT4,INT8,ABS1,UINT4,UINT8,
      INT2,UINT2,REAL2,REAL4,UREAL2,UREAL4,LN,EXP,LOG10,TENPOW,LOG2,
      TWOPOW,COS,SIN,TAN,COSH,SINH,TANH,LOG5,FIVEPOW,
      CNT1}; /* CNT1 is the count */
struct {
   double (*fexe)(double x);
   char *fnam;
} fbyterm1[CNT1];

struct {
   double* (*fexe)(double x);
   char *fnam;
} fbyterm1x[CNT1];

/* How a term-by-term math function for byterm1() is implemented, using
   cosine as an example:

   1. Place the name of the operation, like COS, in enum byterm1.

   2. Create a function that operates on one double number and returns
      a double number result, in this case the cosine:

         double _cos(double x) { return(cos(x)); }

   3. Define the interface function for the native word that will fire
      byterm1() with option COS.  This is simply a call to bytem1 with 
      COS:

         int cos1() // cos (hA --- hA1) 
         { 
             return(byterm1(COS)); 
         } 

   4. In mathinit(), define fterm1[COS].fexe and fterm1[COS].fnam for 
      COS and link them to function _cos() as follows:

         fbyterm1[COS].fexe=(double (*)()) _cos;
         fbyterm1[COS].fnam="_cos";

      Note that .fnam in quotes is not required to match the function
      name; it is used for the name on the stack, and is most informa-
      tive when it is close to the native word that is invoked.

   5. Then do the usual: add entries in math1.h for the new functions
      _cos() and cos1(), and add an entry in word.p for new word cos.

   Function byterm1() will execute _cos() through fbyterm1[COS].fexe. 
   If there is an error, fbyterm1[COS].fnam will provide the name to
   report.

   For functions of two double numbers in byterm2(), the process is
   similar.  See, for example, ATAN2.
*/

/* Term-by-term operations for byterm2(): */
enum byterm2 {AND,OR,XOR,EQ,GE,GT,LE,LT,NE,MAX1,MIN1,ATAN2,
      CNT2}; /* CNT2 is the count */
struct {
   double (*fexe)(double x, double y);
   char *fnam;
} fbyterm2[CNT2];

/*--------------------------------------------------------------------*/

/* Unions used for term-by-term bit operations: */

union { 
   double x; unsigned int c[2]; 
} static X;

union { 
   double y; unsigned int d[2]; 
} static Y;

/*--------------------------------------------------------------------*/

double _abs(double x) { return(fabs(x)); }

int abs1() /* abs (hA --- hA1) */
{ 
   double *A,xi,xr;
   int k=0,len;

/* Function byterm1() is more than twice as slow for real MATs, so it 
   is being replaced by the MAT loop below to flip negative signs.

   if(!is_complex(tos)) 
      return(byterm1(ABS1)); 

*/
   if(tos->typ==NUM)
      return( /* real or complex NUM */
         popdx(&xr,&xi) &&
         pushd(sqrt(xr*xr + xi*xi))
      );
   if(tos->typ==MAT && !is_complex(tos)) { /* real MAT */
      cop(); /* copy required, as A is in the catalog */
      A=tos->mat;
      len=tos->row*tos->col;
      for(;k<len;k++) *(A+k)=fabs(*(A+k));
      return 1;
   }
   return( /* complex MAT */
      dup1s() &&

      conj1() &&
      starby() &&

      cmplxdbl() &&
      drop() &&

      pushd(0.5) &&
      power1()
   );
} 

int across() /* across (hA --- hV) */
/* Tue Nov 19 04:29:42 PST 2013
   Adapted from totals() in this file.

   Returning vector V containing the summation across each row of A.

   Row (element) n of V holds the summation of values across row n 
   of A. 

   Testing.  Function totals() is well tested.  To verify this func-
   tion, its results should match the high level phrase "bend totals."

      Copy and drop the following lines at the ready prompt:

      seed0 seedset \
      1000 1000 random (hA) dup across swap (hA) bend totals - null? \
      IF " real matrix agrees" ELSE " real matrix error" THEN . nl \

      1000 1000 random 1000 1000 random complex (hA) \
      (hA) dup across swap (hA) bend totals - null? \
      IF " complex matrix agrees" ELSE " complex matrix error" \
      THEN . nl \ */
{
   register double *A,*V,x,y;
   register int i=0,j=0,k;
   register int rA,cA;
   int nw=1,TAG;
  
   if(is_sparse(tos)) dense();

   if(tos->typ==MAT) {
      TAG=tos->tag;

      if(is_complex(tos)) nw=2;

      rA=tos->row;
      cA=tos->col;
      if(rA==0) return(drop() && matstk(rA,cA,"_across"));

      if((V=(double *)memget(rA,1))==NULL) {
         return 0;
      }
      A=tos->mat;
      if(nw!=2) { /* real MAT: */
         for(;j<rA;j++) {
            x=0;
            k=j;
            for(i=0;i<cA;i++) {
               x+=(*(A+k));
               k+=rA;
            }
            *(V+j)=x;
         }
      }
      else { /* complex MAT: */
         for(;j<rA;j++) {
            x=0;
            y=0;
            k=j;
            for(i=0;i<cA*nw;i++) {
               x+=*(A+k);
               y+=*(A+k+1);
               k+=rA;
               i++;
            }
            *(V+j)=x;
            j++;
            *(V+j)=y;
         }
      }
      return(
         drop() &&
         push(MAT,(char *)memgetn("_across",7),TAG,0,V,NULL,rA,1,\
            NULL)
      );
   }
   stkerr(" across: ",MATNOT);
   return 0;
}

int and() { return(byterm2(AND)); } /* and (hA hB --- hC) */

int and_() /* and (hA hB --- hC) */
/* Version for parser, does "0<> swap 0<> and." */
{ 
   zerone();
   swap();
   zerone();
   return(byterm2(AND)); 
}

double _and(double x, double y)
{  
   X.x=x; Y.y=y; X.c[0]=(X.c[0] & Y.d[0]); X.c[1]=(X.c[1] & Y.d[1]);
   return X.x;
}

double _atan2(double y, double x) { return(atan2(y,x)); }

int atan2_() { return(byterm2(ATAN2)); } /* atan2 (hY hX --- hY/X) */

int bins() /* bins (hV x1 x2 n --- hB) */ 
/* Putting values of vector V into n bins from x1 - x2. 
      Example: 
         list: 1 1 2 2 3 3 3 4 4 ; 1 5 4 bins 
*/
{
   register double *B,*V,Vk,*X,Xmax,Xmin;
   register int k=0;
   int bins,i,vals;

   if(!intervals()) {
      stkerr(" bins: ","error creating bins");
      return 0;
   }
   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" bins: ",MATNOT2); 
      return 0;
   }
   X=tos->mat;
   bins=MAX(1,tos->row-1);
   Xmin=*X;
   Xmax=*(X+bins);
   if(Xmax<Xmin) {
      stkerr(" bins: ","bins not in ascending order"); 
      return 0;
   }
   if(!matstk(bins,1,"_bins")) {
      stkerr(" bins: ",MEMNOT);
      return 0;
   }
/* stack: (hV hX hB) */
   B=tos->mat;
   memset(B,0,sizeof(double)*tos->row); 

   X=(tos-1)->mat;
   V=(tos-2)->mat;
   vals=(tos-2)->row;
   
   for(;k<vals;k++) {
      Vk=*(V+k);
      if(Vk<=Xmax && Vk>=Xmin) {
         bsearchd(Vk,X,bins,&i);
         *(B+i)+=1;
      }
   }
   return(lop() && lop());
}

int bit() /* bit (hA n --- hB) */
/* State of bit n in all terms of A.  B(i,j)=-1 if bit n of A(i,j) is
   on, 0 otherwise.  Bits are numbered from left to right, starting 
   with 0 or 1 depending upon index base. */
{
   unsigned char mask[8]={128,64,32,16,8,4,2,1};
   unsigned int cols,rows,typ,word;
   unsigned long n;
   double *A,*B,uand;
   register unsigned int k=0,len;
   double b=0;
   union {
      double d;
      unsigned char c[sizeof(double)];
   } u={0},v={0};

   typ=(tos-1)->typ;

   if(!popuint(&n) || !hand()) return 0;

   if(tos->typ!=MAT) {
      stkerr(" bit: ",MATNOT);
      return 0;
   }
   n-=XBASE;
   if(n>63) {
      stkerr(" bit: ",BITOUT);
      return 0;
   }
   word=n>>3; /* divide by 8 */
   n-=(word<<3);
   *(u.c+word)=*(mask+n);

   A=tos->mat;
   rows=tos->row;
   cols=tos->col;

   if((B=(double *)memget0(rows,cols))==NULL) {
      stkerr(" bit: ",MEMNOT);
      return 0;
   }
   len=rows*cols;
   uand=u.d;

   for(;k<len;k++) {  /* sign bit not seen if just test v.d for 0 */
      v.d=_and(uand,*(A+k));
      if(*(u.c+word) & *(v.c+word)) *(B+k)=-1;
   }
   if(typ==NUM) {
      b=*B;
      mallfree((void *)&B);
      return(
         drop() &&
         push(NUM,NULL,NOTAG,b,NULL,NULL,0,0,NULL)
      );
   }
   else
   return(
      drop() &&
      push(MAT,(char *)memgetn("_bit",6),NOTAG,0,B,NULL,rows,cols,NULL)
   );
}

int bitget() /* bitget (hV hN --- hA) */
/* Wed Oct 30 06:46:03 PDT 2013

   Incoming V is a vector of r rows.

   N can be a vector of c rows or a number (in which case N is treated
   as a 1-by-1 matrix). 

   The kth row of N, N(k), contains a bit number for obtaining the bit
   state in all rows of vector V and creating the kth column of outgoing
   matrix A.

   Bits are numbered from left to right, starting with 0 or 1 depend-
   ing upon index base.

   Since V(i) is a 64-bit number, bit numbers specified in the rows of
   N can range from 0 to 63 or 1 to 64.

   Returned matrix A has r rows and c columns.  Term A(i,k)=-1 if bit 
   N(k) in term V(i) is 1, and 0 otherwise. 

   Test case Wed Oct 30 07:28:07 PDT 2013:

      To run, paste the following at the ready prompt:
         syspath "../src/math.c" + "test_bitget___" msource

      Results:
         [tops@kaffia] ready > syspath "../src/math.c" + \
                               "test_bitget___" msource

         1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0
         1 0 0 0 0 0 0 1 1 0 0 0 0 1 0 0 0 0 1
         0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 1
         0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 1 0

         1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19

          Column 1:
          11000000 00000001 00000000 00000000 00000000 00000000 00000...
          10000001 10000100 00100000 00000000 00000000 00000000 00000...
          00100000 00000000 10100000 00000000 00000000 00000000 00000...
          00000010 00100000 01000000 00000000 00000000 00000000 00000...

         1 2 3 4 5 6 7 8 9

         1 1 0 0 0 0 0 0 0
         1 0 0 0 0 0 0 1 1
         0 0 1 0 0 0 0 0 0
         0 0 0 0 0 0 1 0 0

         [tops@kaffia] ready > 

      Discussion:
         A 4-by-19 matrix of zeroes and ones is first shown, and it is
         transformed to a 4-by-1 column of bit flags containing 19 bits
         using bitset().  The binary display shows bits 1 through 19 set
         by bitset().

         Bits 1 through 9 of the bit flags are obtained with bitget()
         as the 4-by-9 matrix shown last.  Comparing this with columns
         1 through 9 of the 4-by-19 matrix shows agreement.
         
         Results are correct by inspection.

      The phrase above with word msource runs this region of this file:
         test_bitget___
            {"
             1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0
             1 0 0 0 0 0 0 1 1 0 0 0 0 1 0 0 0 0 1
             0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 1
             0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 1 0
            "} chop dup nl . nl (hT) 19 matread (hA)
            (hA) 1 over cols : ndx (hA hN) dup bend itext neat nl . nl
            (hA hN) bitset (hV) dup nl .bin nl (hV)
            (hV) 1 9 : ndx (hV hN) dup bend itext neat nl . nl
            (hV hN) bitget (hA) abs mtext neat nl . nl
         halt \ */
{
   unsigned char mask[8]={128,64,32,16,8,4,2,1};
   unsigned int word;
   register int i=0,k=0;
   int c,n,r;
   register double *A,*N,*V;
   double uand,*V0;
   union {
      double d;
      unsigned char c[sizeof(double)];
   } u={0},v={0};

   hand();
   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" bitget: ",MATNOT2);
      return 0;
   }
   V0=(tos-1)->mat;
   r=(tos-1)->row;
   N=tos->mat;
   c=tos->row;
   for(;i<c;i++) {
      if(*(N+i)>63+XBASE) {
         gprintf(" bitget: bit numbers in N cannot exceed %d",63+XBASE);
         nc();
         stkerr("","");
         return 0;
      }
   }
   if(!matstk(r,c,"_A")) {
      stkerr(" bitget: ",MEMNOT);
      return 0;
   }
   A=tos->mat;
   memset(A,0,r*c*sizeof(double));

   for(;k<c;k++) {
      n=*N-XBASE;
      word=n>>3; /* divide by 8 */
      n-=(word<<3);
      *(u.c+word)=*(mask+n);
      uand=u.d;
      V=V0;
      for(i=0;i<r;i++) {
      /* Sign bit is not seen if just test v.d for 0: */
         v.d=_and(uand,*V); 
         if(*(u.c+word) & *(v.c+word)) *A=-1;
         A++;
         V++;
      }
      N++;
   }
   return(lop() && lop());
}

int bitoff() /* bitoff (hA n --- hB) */
/* Setting bit n to 0 in all terms of A.  Bits are numbered from left
   to right, starting with 0 or 1 depending upon index base. */
{
   unsigned char mask[8]={128,64,32,16,8,4,2,1};
   unsigned int cols,rows,word;
   unsigned long n;
   register double *A,*B,uand;
   register unsigned int k=0,len;
   union {
      double d;
      unsigned char c[sizeof(double)];
   } u={0};

   if(!popuint(&n) || !hand()) return 0;

   if(tos->typ!=MAT) {
      stkerr(" bitoff: ",MATNOT);
      return 0;
   }
   n-=XBASE;
   if(n>63) {
      stkerr(" bitoff: ",BITOUT);
      return 0;
   }
   word=n>>3; /* divide by 8 */
   n-=(word<<3);
   *(u.c+word)=*(mask+n);

   A=tos->mat;
   rows=tos->row;
   cols=tos->col;

   if((B=(double *)memget(rows,cols))==NULL) {
      stkerr(" bitoff: ",MEMNOT);
      return 0;
   }
   len=rows*cols;
   uand=_nott(u.d);

   for(;k<len;k++) *(B+k)=_and(uand,*(A+k));

   return(
      drop() &&
      push(MAT,(char *)memgetn("_bitoff",7),NOTAG,0,B,NULL, \
         rows,cols,NULL)
   );
}

int biton() /* biton (hA n --- hB) */
/* Setting bit n to 1 in all terms of A.  Bits are numbered from left
   to right, starting with 0 or 1 depending upon index base. */
{
   unsigned char mask[8]={128,64,32,16,8,4,2,1};
   unsigned int cols,rows,word;
   unsigned long n;
   register double *A,*B,uor;
   register unsigned int k=0,len;
   union {
      double d;
      unsigned char c[sizeof(double)];
   } u={0};

   if(!popuint(&n) || !hand()) return 0;

   if(tos->typ!=MAT) {
      stkerr(" biton: ",MATNOT);
      return 0;
   }
   n-=XBASE;
   if(n>63) {
      stkerr(" biton: ",BITOUT);
      return 0;
   }
   word=n>>3; /* divide by 8 */
   n-=(word<<3);
   *(u.c+word)=*(mask+n);

   A=tos->mat;
   rows=tos->row;
   cols=tos->col;

   if((B=(double *)memget(rows,cols))==NULL) {
      stkerr(" biton: ",MEMNOT);
      return 0;
   }
   len=rows*cols;
   uor=u.d;

   for(;k<len;k++) *(B+k)=_or(uor,*(A+k));

   return(
      drop() &&
      push(MAT,(char *)memgetn("_biton",6),NOTAG,0,B,NULL, \
         rows,cols,NULL)
   );
}

int bitset() /* bitset (hA hN --- hV) */
/* Wed Jul 31 08:59:44 PDT 2013
   Mon Oct 28 05:07:20 PDT 2013.  Rewrite; reference ga_random() in 
   wapp.c.

   Matrix A of r rows and n columns contains values zero and not zero.  

   N can be a vector of n rows or a number (in which case N is treated
   as a 1-by-1 matrix and A must be a column vector).  The kth row of 
   N, N(k), contains a bit number pertaining to column k of A.

   If A(i,k) is not zero, then bit N(k) of V(i) is set to 1.  Since
   V(i) is a 64-bit number, bit numbers (from left to right) specified
   in N can range from 0 to 63 or 1 to 64 depending upon index base.
   
   Test case Mon Oct 28 11:09:47 PDT 2013.

      To run, paste the following at the ready prompt:
         syspath "../src/math.c" + "test_bitset___" msource

      Results:
         [tops@kaffia] ready > syspath "../src/math.c" + \
                               "test_bitset___" msource

         1 1 0 0 0 0 0 0    0 0 0 0 0 0 0 1    0 0 0
         1 0 0 0 0 0 0 1    1 0 0 0 0 1 0 0    0 0 1
         0 0 1 0 0 0 0 0    0 0 0 0 0 0 0 0    1 0 1
         0 0 0 0 0 0 1 0    0 0 1 0 0 0 0 0    0 1 0

         1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19

          Column 1:
          11000000 00000001 00000000 00000000 00000000 00000000 00000...
          10000001 10000100 00100000 00000000 00000000 00000000 00000...
          00100000 00000000 10100000 00000000 00000000 00000000 00000...
          00000010 00100000 01000000 00000000 00000000 00000000 00000...

         [tops@kaffia] ready > 

      Discussion:

         Matrix A has 4 rows and 19 columns containing ones and zeroes,
         shown above; vector N specifies that bit columns 1 through 19
         will be set in output vector V (19 other columns, up to the 
         64th, could have been chosen instead).

         Results showing the bit patterns of the four rows of output
         vector V are correct by inspection.

      The phrase above that runs word msource will source this region
      of this file:
         test_bitset___
            {"
             1 1 0 0 0 0 0 0    0 0 0 0 0 0 0 1    0 0 0
             1 0 0 0 0 0 0 1    1 0 0 0 0 1 0 0    0 0 1
             0 0 1 0 0 0 0 0    0 0 0 0 0 0 0 0    1 0 1
             0 0 0 0 0 0 1 0    0 0 1 0 0 0 0 0    0 1 0
            "} chop dup nl . nl (hT) 19 matread (hA) 
            (hA) 1 over cols : ndx (hA hN) dup bend itext neat nl . nl
            (hA hN) bitset (hV) nl .bin nl
         halt \ */
{
   unsigned char mask[8]={128,64,32,16,8,4,2,1};
   register double *A,*N,*V;
   double *A0;
   register int i=0,k=0;
   int bit,byte,n,r;
   union {
      double d;
      unsigned char c[sizeof(double)];
   } B;

   if(stkdepth()<2) {
      stkerr(" bitset: ",NEEDTWO);
      return 0;
   }
   hand();
   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" bitset: ",MATNOT2);
      return 0;
   }
   A0=A=(tos-1)->mat;
   r=(tos-1)->row;
   n=(tos-1)->col;
   if(tos->row!=n) {
      gprintf(" bitset: %d rows of N must match %d columns of A",\
         n,tos->row);
      nc();
      stkerr("","");
      return 0;
   }
   cop();
   N=tos->mat;
   for(;k<n;k++) {
      if(*(N+k)>63+XBASE) {
         gprintf(" bitset: bit numbers in N cannot exceed %d",63+XBASE);
         nc();
         stkerr("","");
         return 0;
      }
      *(N+k)-=XBASE;
   }
   if(!matstk(r,1,"_V")) {
      stkerr(" bitset: ",MEMNOT);
      return 0;
   }
   V=tos->mat;

   for(;i<r;i++) {
      A=A0+i;
      B.d=0;
      for(k=0;k<n;k++) {
         if(*A) {
            bit=*(N+k);
            byte=bit>>3; /* divide by 8 to get byte in c */
            bit-=byte<<3; /* bit local within byte is mask index */
            *(B.c+byte)|=*(mask+bit); /* turn bit on */
         }
         A+=r;
      }
      *V=B.d;
      V++;
   }
   return(lop() && lop());
}

int brandom() /* brandom (b1 b2 r c --- hT) */
/* Random bytes in range b1 to b2 into volume T of size r rows by c 
   chars. */
{
   int b1,b2,chars,rows;
   register int k=0;
   register double *X;
   register char *T;

   /* ranint (x1 x2 r c --- hInt): */
   char *ranint="random swap other less *f swap +d 0.5 +d integer";

   if(!(popint(&chars) && popint(&rows))) return 0;
   if(!(popint(&b2) && popint(&b1))) return 0;

   k=b1;
   b1=MIN(b1,b2);
   b2=MAX(k,b2);

   pushint(b1);
   pushint(b2);
   pushint(rows);
   pushint(chars);

   pushq2(ranint,strlen(ranint));
   xmain(0); /* (x1 x2 r c --- hInt) */
   X=tos->mat;

   if(!volstk(rows,chars,"_brandom")) return 0;
   T=tos->tex;
   
   for(k=0;k<rows*chars;k++) {
      *T=(char)*X;
      T++;
      X++;
   }
   return(lop());
}

int bsearch1() /* bsearch (hA x --- r f) */
/* Finding x in sorted vector A.  Returns index r to row in A nearest-
   below x; f=true if x is in A (and r points to A), false otherwise. 

   Tue Jun 26 08:09:06 PDT 2012.  Modified so x can be a vector, and 
   if it is, returned r and f are also vectors. 

   Testing:

      Paste the following line at the ready prompt to run the phrases
      below:

         "/opt/tops/tops/src/math.c" "BSEARCH_TEST" msource

      BSEARCH_TEST     

       \ A is a vector of 100 ints:
         1 100 uniform "A" book

       \ X is a vector of 2000 reals and ints:
         0 100 1000 1 ranreal, 0 100 1000 1 ranint pile "X" book

       \ Using NUMs from X in a loop:
         X rows 1st DO A X I pry bsearch park LOOP X rows pilen (hS)

       \ Using vector MAT X: 
         A X bsearch (hr hf) park (hV)

         (hS hV) - null?
         IF " bsearch: NUM and MAT results agree" . nl THEN

      halt */
{
   double *A,*F,*R,*X,x;
   int Arows,found,i=0,r,rows;

   if(tos->typ==NUM) {

      if(!popd(&x)) return 0;

      if(tos->typ!=MAT) {
         stkerr(" bsearch: ",MATNOT);
         return 0;
      }
      found=bsearchd(x,tos->mat,tos->row,&r);

      if(found) found=xTRUE;
      else found=xFALSE;

      return(
         drop() &&
         pushint(r+XBASE) &&
         pushint(found)
      );
   }
   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" bsearch: ",MATNOT2);
      return 0;
   }
   A=(tos-1)->mat;
   Arows=(tos-1)->row;

   X=tos->mat;
   rows=tos->row;

   if(!matstk(rows,1,"_r")) return 0;
   R=tos->mat;

   if(!matstk(rows,1,"_f")) return 0;
   F=tos->mat;

   for(;i<rows;i++) {
      x=*X;
      *F=bsearchd(x,A,Arows,&r)*xTRUE;
      *R=r+XBASE;

      X++;
      R++;
      F++;
   }
   return(rot() && drop() && rot() && drop());
}

int bsearchd(double x, double *xtable, int len, int *delta)
/* Finding offset, delta, to value nearest-below x in sorted xtable.
   If x is also in xtable, returned int is 1 (and delta points to x);
   otherwise it is 0. 

   Names bx and di refer to 8086 registers used in the Author's assem-
   bler code that this function is based upon. */
{
   unsigned register int bx,di=0;

   if(!len) {
      *delta=0;
      return 0;
   }
   if(x>=*(xtable+len-1)) {
      *delta=len-1;
      return(x==*(xtable+len-1));
   }
   if(x<=*xtable) {
      *delta=0;
      return(x==*xtable);
   }
   bx=len>>1;
   while(bx && *(xtable+di+bx)!=x) {
      if(*(xtable+di+bx)>x) bx=bx>>1; /* looking lower */
      else {
         di+=bx; /* looking higher */
         bx=MAX(1,bx>>1); /* not allowing 0 index here */
      }
   }
   *delta=di+bx;
   return(x==*(xtable+di+bx));
}

int byteorder(int **k)
/* Getting the machine's 4-byte integer order: 

   This function returns one of the BYTE_ORDER integers shown below 
   (or 0 if error).  Returned as an argument is the static pointer to
   an integer array listing the offsets to ordered bytes, least sig-
   nificant first.

   BYTE_ORDER integers:

      LITTLE_ENDIAN 1234 
      BIG_ENDIAN    4321
      PDP_ENDIAN    3412

   BYTE_ORDER is given in <sys/machine.h> or <sys/types.h>, depending
   on machine.  To be portable, just figure it out here. 

   An example using byteorder():

      int k,*k1=NULL;
      k=byteorder(&k1);
      printf(" k = %d, offsets: %d,%d,%d,%d\n\r",k, \
         *k1,*(k1+1),*(k1+2),*(k1+3));


      On linux (pc), this will display:
         k = 1234, offsets: 0,1,2,3

      and on aix (risc 6000):
         k = 4321, offsets: 3,2,1,0 */
{
   union { 
      char mask[sizeof(int)];
      int n;
   } m1={{'\1','\0','\0','\0'}},
     m2={{'\0','\0','\0','\1'}},
     m3={{'\0','\0','\1','\0'}};

   const int one=1;

   /* Offsets to most significant byte first: */
   static int k1[4]={0,1,2,3}; /* little endian */
   static int k2[4]={3,2,1,0}; /* big endian */
   static int k3[4]={2,3,0,1}; /* pdp endian */

   if(!(one ^ m1.n)) { if(k) *k=(int *)&k1; return(1234); }
   if(!(one ^ m2.n)) { if(k) *k=(int *)&k2; return(4321); }
   if(!(one ^ m3.n)) { if(k) *k=(int *)&k3; return(3412); }

   stkerr(" byteorder: ",BORDUNK);
   return 0;   
}

int byteorder1() /* endian ( --- n) */
{
   int *k,n;
   n=byteorder(&k);
   return((n!=0) && pushint(n));
}

/* The following do not fix the nott problem:
   static double (*exe)(); (local or global)
   static double x,y;
   static double _nott(double x);
   double _nott(double x) to double _notty(double x)



   Setting y instead to the global variable used by nott gives
   the correct result but why?  This is the same variable whose
   value is returned by exe:
   y=exe(x);
   y=X.x;

*/

int byterm1(int oncase) /* arg: int oncase; stk: (hA --- hB) */
/* Runs a term-by-term function for one stack item. */
{
   char *nam;
   char errname[16]={0};

   double xi,xr,yi,yr;
   register double (*exe)();
   register double *A,*B;

   int len,spars=0,typ;
   register int rA,cA,k=0;

   if(is_sparse(tos)) {
      dense();
      spars=1;
   }
   exe=(fbyterm1+oncase)->fexe; /* function exe addr */
   nam=(fbyterm1+oncase)->fnam; /* function name */

   if((typ=tos->typ)==NUM) {
      if(is_complex(tos)) {
         popdx(&xr,&xi);
         yr=exe(xr);
         yi=exe(xi);
         return(pushdx(yr,yi));
      }
      else {
         popd(&xr);
         yr=exe(xr);
         return(pushd(yr));
      }
   }
   if(typ==MAT) {
      A=tos->mat;
      rA=tos->row;
      cA=tos->col;

      if(!matstk(rA,cA,"_byterm1")) {
         strcat(errname,nam); *errname=' ';
         strcat(errname,": "); stkerr(errname,MEMNOT);
         return 0;
      }
      B=tos->mat;
      tos->tag=(tos-1)->tag;

      for(;k<rA*cA;k++) *(B+k)=exe(*(A+k));

      if(spars) sparse();

      return(lop());
   }
   if(typ==STR) {
      len=tos->col;
      return(
         typstr2vol() && typvol2mat() && 
         byterm1(oncase) &&
         typmat2vol() &&
         pushint(XBASE) && quote() &&
         pushint(XBASE) && pushint(len) && items() && catch()
      );
   }
   strcat(errname,nam); *errname=' ';
   strcat(errname,": "); stkerr(errname,NUMORMATORSTRNOT);
   return 0;
}

/* This version of byterm1 is used only for _nottx, which correctly 
passes a binary string for the phrase 2 3 xor nott.  

_nottx returns an address while _nott returns a value.  

Using byterm1 and _nott, something happens to the value for the 
phrase 2 3 xor nott and the incorrect result is obtained.

Until this problem is understood, nott (and only nott) will use 
byterm1x and nottx, with word nott will be vectored to use it in
word.p.  

Word nottx will run the erroneous version using nott and byterm1.

Here is how they work, with nott giving the correct result as it
uses _nottx and biterm1x (on a little endian machine):

           Tops 2.1
Sat May 27 18:24:00 PDT 2000
[tops@gutter] ready > 2 3 xor nott .bin
 11111111 11111111 11111111 11111111 11111111 11111111 11110111 11111111
[tops@gutter] ready > 2 3 xor nottx .bin
 11111111 11111111 11111111 11111111 11111111 11111111 11111111 11111111
[tops@gutter] ready > 

If on the little endian machine the bits are turned around to the order
in a big endian machine, a correct result is obtained:

[tops@gutter] ready > 2 hand BIG_ENDIAN export8 vol2mat 
[tops@gutter] ready > 3 hand BIG_ENDIAN export8 vol2mat 
[tops@gutter] ready > xor nott .bin
 Column 1:
 11111111 11110111 11111111 11111111 11111111 11111111 11111111 11111111
[tops@gutter] ready > 

The next thing to see will be how this works on a big endian machine.
If both bit positions work, then the gnu version of C in linux is 
suspect.

*/

int byterm1x(int oncase) /* arg: int oncase; stk: (hA --- hB) */
/* Runs a term-by-term function for one stack item. */
{
   register double *(*exe)();
   char *nam;
   stkitem *Amat;
   register double *A,*B;
   double x, *y;
   register int rA,cA,k=0;
   char errname[16]={0};
   int len,typ;

   exe=(fbyterm1x+oncase)->fexe; /* function exe addr */
   nam=(fbyterm1x+oncase)->fnam; /* function name */

   if((typ=tos->typ)==NUM) {
      popd(&x);
      y=exe(x);
      return(push(NUM,NULL,NOTAG,*y,NULL,NULL,0,0,NULL));
   }
   if(typ==MAT) {
      Amat=tos; A=Amat->mat; rA=Amat->row; cA=Amat->col;

      if((B=(double *)memget(rA,cA))==NULL) {
         strcat(errname,nam); *errname=' ';
         strcat(errname,": "); stkerr(errname,MEMNOT);
         return 0;
      }
      for(;k<rA*cA;k++) {
         y=exe(*(A+k));
         *(B+k)=*y;
      }
      return(
         drop() &&
         push(MAT,(char *)memgetn(nam,strlen((char *)nam)),NOTAG,0, \
            B,NULL,rA,cA,NULL)
      );
   }
   if(typ==STR) {
      len=tos->col;
      return(
         typstr2vol() && typvol2mat() &&
         byterm1(oncase) &&
         typmat2vol() &&
         pushint(XBASE) && quote() &&
         pushint(XBASE) && pushint(len) && items() && catch()
      );
   }
   strcat(errname,nam); *errname=' ';
   strcat(errname,": "); stkerr(errname,NUMORMATORSTRNOT);
   return 0;
}

int byterm2(int oncase) /* arg: int oncase; stk: (hA hB --- hC) */
/* Runs a term-by-term function for two stack items. */
{
   stkitem *Amat,*Bmat;

   char *nam;
   char errname[16]={0};

   double xi,xr,yi,yr;
   register double (*exe)();
   register double *A,*B,*C;

   int len,types=0;
   register int rA,cA,k=0;

   exe=(fbyterm2+oncase)->fexe; /* function exe addr */
   nam=(fbyterm2+oncase)->fnam; /* function name */

   switch(tos->typ) {   

      case NUM:
         if((tos-1)->typ==NUM) types=NUM;
         else {
            if((tos-1)->typ==MAT) {
               over();
               dims();
               fill();
               types=MAT;
            }
         }
      break;

      case MAT: 
         if((tos-1)->typ==MAT) types=MAT;
         else {
            if((tos-1)->typ==NUM) {
               swap();
               over();
               dims();
               fill();
               swap();
               types=MAT;
            }
         }
      break;

      case STR: 
         if((tos-1)->typ==STR) types=STR;
      break;

      default:
      break;
   }
   if(types==NUM) {
      if(is_complex(tos)) {
         popdx(&yr,&yi);
         popdx(&xr,&xi);
         yr=exe(xr,yr);
         yi=exe(xi,yi);
         return(pushdx(yr,yi));
      }
      else {
         xr=pop()->real;
         return(
            push(NUM,NULL,NOTAG,exe(pop()->real,xr),NULL,NULL,0,0,NULL)
         );
      }
   }
   if(types==MAT) {
      cmplxmatch();

      Amat=tos-1; Bmat=tos;
      A=Amat->mat; rA=Amat->row; cA=Amat->col;
      B=Bmat->mat;
      if(rA!=Bmat->row || cA!=Bmat->col) {
         stkerr(" byterm2: ",MATSNOTC);
         return 0;
      }
      if(!matstk(rA,cA,"_byterm2")) {
         strcat(errname,nam); *errname=' ';
         strcat(errname,": "); stkerr(errname,MEMNOT);
         return 0;
      }
      C=tos->mat;
      tos->tag=(tos-1)->tag;

      for(;k<rA*cA;k++) *(C+k)=exe(*(A+k),*(B+k));

      return(lop() && lop());
   }
   if(types==STR) {
      len=tos->col;
      if((tos-1)->col!=len) {
         strcat(errname,nam); *errname=' ';
         strcat(errname,": "); stkerr(errname,STKNOTC);
         return 0;
      }
      return(
         typstr2vol() && typvol2mat() && swap() &&
         typstr2vol() && typvol2mat() && swap() &&
         byterm2(oncase) &&
         typmat2vol() &&
         pushint(XBASE) && quote() &&
         pushint(XBASE) && pushint(len) && items() && catch()
      );
   }
   if(!types) {
      if(is_sparse(tos)) {
         dense();
         types=MAT;
      }
      if(is_sparse(tos-1)) {
         swap();
         dense();
         swap();
         types=MAT;
      }
   }
   if(types) return(byterm2(oncase));

   strcat(errname,nam); 
   *errname=' ';
   strcat(errname,": "); 
   stkerr(errname,STKNOT);
   return 0;
}

int closest() /* closest (hXYt hX --- hY) */
{
   return(lerp1(CLOSEST,CLOSEST));
}

int colsort() /* colsort (hA --- hA1) */
/* Sort each column of matrix A into ascending order. */
{
   register double *A;
   int cols,j=0,rows,VQSORTsav;

   if(tos->typ!=MAT) {
      stkerr(" colsort: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;

   cop();
   A=tos->mat;
   
   VQSORTsav=VQSORT;
   for(;j<cols;j++) {
      VQSORT=VQSORTsav;
      qsort1(A,rows,1);
      A+=rows;
   }
   return 1;
}

int colsort1() /* colsort1 (hA n --- hA1) */
/* Sort the last n columns of matrix A into ascending order. */
{
   register double *A;
   int cols,j,n,n0,rows,VQSORTsav;

   if(!popint(&n)) return 0;
   n=MAX(n,0);

   if(tos->typ!=MAT) {
      stkerr(" colsort1: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;

   n=MIN(n,cols);
   n0=cols-n;

   rows=tos->row;
   cols=tos->col;

   cop();
   A=tos->mat+locvec(n0,rows);

   VQSORTsav=VQSORT;
   for(j=n0;j<cols;j++) {
      VQSORT=VQSORTsav;
      qsort1(A,rows,1);
      A+=rows;
   }
   return 1;
}

int colsortAB() /* colsortAB (hA hB f --- hA1 hB1) */
/* Sun Jun 24 03:18:30 PDT 2012

   Sort each column of matrix A, carrying B along for the ride, so the
   reordering of the rows of B matches the reordering of the rows of A.

   Incoming f is true for ascending sort, false for descending.

   Row and column dimensions of A and B must match. */
{
   register double *A,*B,*B1;
   int cols,f,i,j=0,*p,rows;

   if(!popbool(&f)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" colsortAB: ",MATNOT2);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;

   if((tos-1)->row!=rows) {
      stkerr(" colsortAB: ",ROWSNOT);
      return 0;
   }
   if((tos-1)->col!=cols) {
      stkerr(" colsortAB: ",COLSNOT);
      return 0;
   }
   cop(); swap(); 
   cop(); pushq2("_A1",3); naming(); swap();

   A=(tos-1)->mat;
   B=tos->mat;
   if(!matstk(rows,cols,"_B1")) return 0;
   B1=tos->mat;

   for(;j<cols;j++) {
      p=qsort2(A,rows,f);
      if(p==NULL) {
         gprintf(" colsortAB: qsort2 error: null p");
         stkerr("","");
         nc();
         return 0;
      }
      for(i=0;i<rows;i++) *(B1+i)=*(B+*(p+i));
      A+=rows;
      B+=rows;
      B1+=rows;
      mallfree((void *)&p);
   }
   return(lop());
}

int compare() /* compare (hA hB fac ref --- hC) */
/* Compare matrices A and B term-by-term, put discrepancy into C.  
   For these values of ref, discrepancy C is defined as:

      ref=0: C=fac*(A - B)/ABaverage
      ref=1: C=fac*(B - A)/A
      ref=2: C=fac*(A - B)/B

      If A(i) or B(i) is zero, C(i) is zero */
{
   int cols,ref,rows;
   double fac1;
   register int i=0,len;
   register double *A,*B,*C,fac;

   if(!popint(&ref) || !popd(&fac1)) return 0;
   fac=fac1;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" compare: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row || (cols=tos->col)!=(tos-1)->col) {
      stkerr(" compare: ",MATSNOTC);
      return 0;
   }
   if(!matstk(rows,cols,"_compare")) return 0;

   C=tos->mat;
   B=(tos-1)->mat;
   A=(tos-2)->mat;
   len=rows*cols;
  
   switch(ref) {

      case 0:
         fac+=fac; /* double for the 2 due to taking the average */
         for(;i<len;i++) {
            if(*A==0 || *B==0) *C=0;
            else *C=fac*(*A-*B)/(*A+*B);
            A++;
            B++;
            C++;
         }
         break;

      case 1:
         for(;i<len;i++) {
            if(*A==0 || *B==0) *C=0;
            else *C=fac*(*B-*A)/(*A);
            A++;
            B++;
            C++;
         }
         break;

      case 2:
         for(;i<len;i++) {
            if(*A==0 || *B==0) *C=0;
            else *C=fac*(*A-*B)/(*B);
            A++;
            B++;
            C++;
         }
         break;

      default:
         stkerr(" compare: ","ref flag not 0, 1, or 2");
         drop();
         return 0;
   }
   return(lop() && lop());
}

double _cos(double x) { return(cos(x)); }

int cos1() { return(byterm1(COS)); } /* cos (hA --- hA1) */

double _cosh(double x) { return(cosh(x)); }

int cosh1() { return(byterm1(COSH)); } /* cosh (hA --- hA1) */

int cross() /* *cross (hA hB --- hC) */
/* Col j of C contains the cross product of column j of A and B. */
{
   register double *A,*B,*C;
   register int j=0,k=0;
   int rows,cols;

   if(tos->typ!=MAT || tos->typ!=MAT) {
      stkerr(" *cross: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row || (cols=tos->col)!=(tos-1)->col) {
      stkerr(" *cross: ",MATSNOTC);
      return 0;
   }
   if(rows!=3) {
      stkerr(" *cross: ","matrices must have 3 rows");
      return 0;
   }
   if(!matstk(3,cols,"_*cross")) return 0;

   C=tos->mat;
   B=(tos-1)->mat;
   A=(tos-2)->mat;

   for(;j<cols;j++) {
      *(C+k  )=*(A+k+1)*(*(B+k+2)) - *(A+k+2)*(*(B+k+1));
      *(C+k+1)=*(A+k+2)*(*(B+k  )) - *(A+k  )*(*(B+k+2));
      *(C+k+2)=*(A+k  )*(*(B+k+1)) - *(A+k+1)*(*(B+k  ));
      k+=3;
   }
   return(lop() && lop());
}

int delta() /* delta (hA --- hdA) */
/* Change in A values from one row to next.  First value in dA is 0. */
{
   register double *A,*dA;
   int cols,i,j=0,rows;

   if(is_sparse(tos)) dense();

   if(tos->typ!=MAT) {
      stkerr(" delta: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;
   A=tos->mat;

   if(!matstk(rows,cols,"_delta")) return 0;
   dA=tos->mat;
   memcpy(dA,A,rows*cols*sizeof(double));

   for(;j<cols;j++) {
      *dA=0;
      dA++;
      for(i=1;i<rows;i++) {
         *dA-=*A;
         dA++;
         A++;
      }
      A++;
   }
   return(lop());
}

int dotprod() /* *dot (hA hB --- hV) */
/* Row j of vector V contains the dot product of column j of A and B. */
{
   register double *A,*B,*V,d;
   register int i=0,j=0,k=0;
   int rows,cols;

   if(tos->typ!=MAT || tos->typ!=MAT) {
      stkerr(" *dot: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row || (cols=tos->col)!=(tos-1)->col) {
      stkerr(" *dot: ",MATSNOTC);
      return 0;
   }
   if(!matstk(cols,1,"_*dot")) return 0;

   V=tos->mat;
   B=(tos-1)->mat;
   A=(tos-2)->mat;

   for(;j<cols;j++) {
      d=0;
      for(i=0;i<rows;i++) {
         d+=*(A+k)*(*(B+k));
         k++;
      }
      *(V+j)=d;
   }
   return(lop() && lop());
}

double _eq(double x, double y) { return(xTRUE*(x==y)); }

int eq() /* = (hA hB --- hF) */
{ 
   int f;

   if(is_sparse(tos)) dense();
   if(is_sparse(tos-1)) {
      swap();
      dense();
      swap();
   }
   if((tos->typ==NUM || tos->typ==MAT) &&
     ((tos-1)->typ==NUM || (tos-1)->typ==MAT)) return(byterm2(EQ)); 

/* Doing characters: */
   if(!strmatch1()) return 0;

   popint(&f);
   if(f) return(pushint(xFALSE));
   return(pushint(xTRUE));
} 

double _exp(double x) { return(exp(x)); }

int exp1() /* e^ (hA --- hA1) */
{ 
   if(is_complex(tos)) {
      stkerr(" e^: ",REALNOT);
      return 0;
   }
   return(byterm1(EXP)); 
} 

int __f() /* __f (hV n --- hF) */
/* Thu Jun 27 20:39:03 PDT 2013

   Decode rows of vector V that are 0 or -1 flags that have been 
   encoded by powers of 2.

   Incoming n is assumed to be a power of 2. 

   This function works the same as this macro:
      "(hV n --- hF) push abs peek 2 * (n*2) /mod drop"
      " pull (n) /mod lop true * (hf)" + "__f" macro 

   Since n and the values in V are powers of 2, the steps below could
   be done with bit shifting. 

   Example:

      Paste at the ready prompt the following line; results should 
      match the table shown below:
         syspath "../src/math.c" + "__ff_decode" msource
\
      __ff_decode {"

            <<
            "ONES" missing 
            IF "(hV) 1  __f (hf)", "ONES" macro
               "(hV) 2  __f (hf)", "TWOS" macro
               "(hV) 4  __f (hf)", "FOURS" macro
               "(hV) 8  __f (hf)", "EIGHTS" macro
               "(hV) 16 __f (hf)", "SIXTEENS" macro
               "(hV) 32 __f (hf)", "THIRTYTWOS" macro
               "(hV) 64 __f (hf)", "SIXTYFOURS" macro
            THEN
            >>

            f1 = true*(<< list: 0 0 0 0 1 1 1 1 ; >>);
            f2 = true*(<< list: 0 0 1 1 0 0 1 1 ; >>);
            f3 = true*(<< list: 0 1 0 1 0 1 0 1 ; >>);

         // Put f3 into ONES, f2 into TWOS and f1 into FOURS:
            V = f3 + 2*f2 + 4*f1;

            <<
            f1 f2 f3 V 4 parkn 
            V ONES V TWOS V FOURS 4 parkn 
            (hA) itext . nl
         
        {  ------ Before ------     Encoded   ------- Decoded -------
                                              ONES     TWOS     FOURS
           f1       f2       f3        V       f3       f2       f1

           -0       -0       -0       -0       -0        0        0
           -0       -0       -1       -1       -1        0        0
           -0       -1       -0       -2       -0       -1        0
           -0       -1       -1       -3       -1       -1        0
           -1       -0       -0       -4       -0        0       -1
           -1       -0       -1       -5       -1        0       -1
           -1       -1       -0       -6       -0       -1       -1
           -1       -1       -1       -7       -1       -1       -1

        }
      "} eval halt __ff_decode */
{
   long long Q;
   int k=0,r;
   double *F,N,N2,*V,X,Y;

   if((tos-1)->typ!=MAT) {
      stkerr(" __f: ",MATNOT);
      return 0;
   }
   if(!popd(&N)) return 0;
   N2=N*2;

   V=tos->mat;
   r=tos->row;
   if(!matstk(r,1,"_F")) return 0;
   F=tos->mat;

   for(;k<r;k++) {
      X=*V;
      Q=X/N2;
      Y=X-Q*N2;
      Q=Y/N;
      *F=Q;
      V++;
      F++;
   }
   return(lop());
}

int __fsum() /* __fsum (hV hD --- hF) */
/* Sat Jun 29 08:17:09 PDT 2013

   Rows of vector V contain flags 0 or -1 that have been encoded by 
   powers of two.  Decoding V(k) by a power of two is done in __f(),
   and produces 0 or -1.

   Vector D contains a list of powers of two for decoding V.

   For each row of V, sum the result of decoding for all D(j) of D. 

   Example (continuation of example in __f()):

      Paste the six following lines at the ready prompt:

         list: 0 0 0 0 1 1 1 1 ; true * dup "f1" book 4 *   \
         list: 0 0 1 1 0 0 1 1 ; true * dup "f2" book 2 * + \
         list: 0 1 0 1 0 1 0 1 ; true * dup "f3" book 1 * + \        
         "V" book V list: 1 2 4 ; __fsum "Fsum" book        \
         "f1       f2       f3        V      Fsum" 14 indent . nl \
         f1 f2 f3 V Fsum 5 parkn .m nl
\
                       f1       f2       f3        V      Fsum
          Row 1:       -0       -0       -0       -0        0
          Row 2:       -0       -0       -1       -1       -1
          Row 3:       -0       -1       -0       -2       -1
          Row 4:       -0       -1       -1       -3       -2
          Row 5:       -1       -0       -0       -4       -1
          Row 6:       -1       -0       -1       -5       -2
          Row 7:       -1       -1       -0       -6       -2
          Row 8:       -1       -1       -1       -7       -3

      The result in column Fsum equals the row sum f1+f2+f3.
      Results are correct by inspection. */
{
   long long Q;
   int d,j,k=0,r;
   double *D,*F,N,N2,*V,X,Y;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" __fsum: ",MATNOT2);
      return 0;
   }
   D=tos->mat;
   d=tos->row;

   V=(tos-1)->mat;
   r=(tos-1)->row;

   if(!matstk(r,1,"_F")) return 0;
   F=tos->mat;

   for(;k<r;k++) {
      X=*V;
      *F=0;
      for(j=0;j<d;j++) {
         N=*D;
         N2=N*2;
         Q=X/N2;
         Y=X-Q*N2;
         Q=Y/N;
         *F+=Q;
         D++;
      }
      V++;
      F++;
      D=(tos-1)->mat;
   }
   return(lop() && lop());
}

int false() { return(pushint(xFALSE)); } /* false ( --- false) */

int filter1() /* filter (hA x --- hB) */
/* Sets to zero abs A terms less than |x|. */
{
   register double *A,*B,x;
   double x1;
   register int k=0;
   int cols,rows,spars=0;

   if(!popd(&x1)) return 0;
   x=fabs(x1);

   if(is_sparse(tos)) {
      dense();
      spars=1;
   }
   if(tos->typ!=MAT) {
      stkerr(" filter: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;
   if(!matstk(rows,cols,"_filter")) return 0;

   B=tos->mat;
   tos->tag=(tos-1)->tag; /* B gets A's tag */

   A=(tos-1)->mat;
 
   for(;k<rows*cols;k++) {
      if(fabs(*(A+k))<x) *(B+k)=0;
      else *(B+k)=*(A+k);
   }
   if(spars) sparse();
   return lop();
}

int find() /* find (hXt hX --- hf) */
{
   return(lerp1(FIND,FINDX));
}

int ge() { return(byterm2(GE)); } /* >= (hA hB --- hF) */

double _ge(double x, double y) { return(xTRUE*(x>=y)); }

int greg() /* greg (YYmmdd --- D) */ 
/* Days since Gregorian day 0. 

   Given YYmmdd where YY = yy - 1900, compute the Gregorian day.

   Equation from an old TI calculator user's manual:

      D = 365*yy + dd + 31*(mm - 1) + (yy + f)/4
          - 3*[(yy + f)/100 + 1]/4 - L
   where f = -1 (Jan, Feb), f = 0 (Mar - Dec)
         L =  0 (Jan, Feb), L = (4*mm + 23)/10 (Mar - Dec)

   [When using mm1=mm-1: L = (4mm1+27)/10] 

   In RH linux 5.2, see /usr/src/linux-2.0.36/arch/i386/kernel/time.c
   for an equivalent (and simpler) equation attributed to Gauss. */
{
   int fdat[12]={-1,-1,0,0,0,0,0,0,0,0,0,0};
   int Lflag[12]={0,0,1,1,1,1,1,1,1,1,1,1};
   int D,dd,f,L,mm,yy,YYmmdd;

   int rows;
   register int k=0;
   register double *G,*Y;

   if(tos->typ==MAT) {
      Y=tos->mat;
      if(!matstk((rows=tos->row),1,"_greg")) return 0;
      G=tos->mat;
      for(;k<rows;k++) {
         pushint(*(Y+k));
         greg();
         popd((G+k));
      }
      return(lop());
   }
   if(!popint(&YYmmdd)) return 0;

   YYmmdd+=19000000;
   yy=YYmmdd/10000;
   mm=((dd=YYmmdd%10000)/100)-1;
   dd=dd%100;

   L=*(Lflag+mm)*(4*mm+27)/10;

   f=*(fdat+mm)+yy;
/*
   D=365*yy+dd+31*mm+(yy+f)/4-3*(1+(yy+f)/100)/4 - L;
*/
   D=365*yy+dd+31*mm+f/4-3*(1+f/100)/4 - L;

   return(pushint(D));
}

int growth() /* growth (hS hN nU --- hR) */
/* Sat Feb 18 08:42:05 PST 2012

   Incoming column S contains an S-shaped logistic function, an ever-
   increasing function.  Column N holds a set of four increasing growth
   percentage levels (such as the set N[0:3] = [50 100 200 300] per-
   cent). 

   Growth of S is monitored from initial base S(0), and when the growth
   of S increases by N(3)--the largest of set N--say to S(h), growth 
   monitoring of S for N(3) is reset to monitor growth of N(3) levels 
   from new base S(h).  

   Strategy flag U designates the base reset strategy for N[0:2].  If
   U is zero, then the strategy for these lesser levels is the same as
   N(3) described above--each level updates independently.

   For nonzero strategy flag U, the reset strategy for N[0:2] is tied
   to resets of N(3).  All further descriptions and examples are for
   the more complex case of U=1.

   UPDATE Tue Feb 21 12:39:36 PST 2012.  For nonzero strategy flag U,
   the reset of N(0) is tied to the base of N(1) and the reset of N(2)
   is tied to the base of N(3).  The writeup below reflects earlier 
   code where the reset of N(0), N(1) and N(2) were all tied to the 
   base of N(3).

   The growth monitoring of lesser N[0:2], say N(1)<N(3), similarly 
   follows S and ratchets to new steps in 1+N(1)/100 increments.  How-
   ever, if N(3) has reset to monitor its levels from S(h) during the
   current N(1) step, then when N(1) completes its step in progress at
   S(k), monitoring of N(1) is reset to monitor growth not from S(k),
   but from S(h), the earlier level reached by N(3).

   This scheme simplifies the visual interpretation of growth patterns,
   since much of the time they will be moving relative to the same ref-
   erence level, S(h), especially when the values in set N are growth 
   multiples of each other.

   Resets are returned in matrix R, where the four columns of R cor- 
   respond to the four rows of N.  Term R(i, k) contains 1 where S has 
   grown another N(k) percent: from the prior reference point for k to 
   point S(i).  The prior reference point for k<3 is subject to the 
   reset scheme involving N(3) described above.

   Test case (adapted from the example in man logistic):

      Paste these phrases into the top of file work.v and run ww at
      the ready prompt to produce a graph of logistic function S and
      the growth monitoring curves created from R:

        "plot.v" source pgrid_off pgrid

        0.1 200 uniform dup mean - "H" book
        H 0.5 logistic "S" book
 
        list: 50 100 200 300 ; "N" book \ growth steps, percent
        S N 1 (hS hN nU) growth (hR) "R" book

        R dup push cols 1st
        DO S peek I catch looking LOOP pull cols parkn (hS)
        (hS) S park H plot 

      Evaluating the graphical results.  

      This describes the four curves for the four values in N: N(0), 
      N(1), N(2) and N(3):

         N(0): 50%, factor of 1.5, blue
         N(1): 100%, factor of 2, dark green
         N(2): 200%, factor of 3, red
         N(3): 300%, factor of 4, purple 

      Values for curves can be read off using the mouse arrow and click-
      ing the left mouse button.

      To read values more accurately, left clicking two points to de-
      fine the diagonal corners of a zoom-in box, followed by a right
      click, will zoom in; right clicking after zooming in will zoom 
      back out.

      The N(3) curve (purple) has the largest steps, and it defines the
      major resets.  It shows a major step from 0.22 to 0.89, a factor 
      of 4 corresponding to N(3) steps of 300% (steps are taken to the
      point in S that is above the goal, so exact multiples are not 
      expected and they should be slightly higher, never lower).

      Between major resets in N(3), values for the dark green curve 
      (N(1)) are seen to double on each step, as they should for steps
      of 100%.

      Steps of the red curve, N(2), are seen at 0.538 eminating from
      0.175 (a factor of 3), and then at 0.68 eminating from 0.22 (also
      a factor of 3).  The base of 0.22 is due to the major reset by 
      N(3) that caused N(2) to use 0.22 (the base of N(3)) as its base
      when it reset upon reaching 0.538.

      All curves are seen to have stepped as high as they can; addi-
      tional steps in any of them would be above the top of logistic
      curve S.  

      Detailed examination shows the following behavior:

         When N(3) completes a step and resets to S(h), any growth step
         of N(k), k<3, that is in progress will continue until it com-
         pletes that step at S(k)>S(h).  

         Then N(k) will reset and begin measuring its next step from
         earlier S(h), not S(k).

         See description of this behavior above and comments in the 
         loops below.

      Sun Feb 19 20:24:31 PST 2012.

      Just writing the graphical evaluation above uncovered a handful
      of errors in curves that were incorrect but "ok looking."  Other
      runs in real production uncovered endless looping that was fixed
      with small number eps when S(0)=0, and seg faulting when h was
      not constrained (h<m) in the second while() loop.  

      A test was added to verify that S is increasing--there is no use
      processing garbage if it can be helped.

      Is that it?  Never.  Until the next time, it is merely concluded 
      that this function can work correctly with no known errors.  

      Like all functions, it is probably just working correctly with
      unknown errors that are yet to be discovered. */
{
   double eps=1e-6,n,*N,*R,*R0,*S,*S0,Sm,Sn,Sr,*W,*W0;
   int h=0,i=0,j,k=0,r,U;

   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" growth: ",MATNOT2);
      return 0;
   }
   if((tos-1)->row<4) {
      gprintf(" growth: set N requires four values");
      stkerr("","");
      nc();
      return 0;
   }
   if(!popint(&U)) return 0;

   N=tos->mat;
   S0=S=(tos-1)->mat;
   r=(tos-1)->row;

   if(r<2) {
      gprintf(" growth: to grow, S requires two or more rows");
      stkerr("","");
      nc();
      return 0;
   }
   for(;k<r-1;k++) {
      if(*S>*(S+1)) {
         gprintf(" growth: S is not an increasing function");
         stkerr("","");
         nc();
         return 0;
      }
      S++;
   }
   S=S0;
   Sm=*(S0+r-1);

   if(!matstk(r,4,"_W")) return 0; /* work array */
   W0=tos->mat;

   if(!matstk(r,4,"_R")) return 0;
   R0=tos->mat;
   memset(R0,0,r*4*sizeof(double));
   
   k=1;
   while(k<4) { /* Resetting N(1) and N(3), and remembering their
                   reset points. */
      W=W0+locvec(k,r);
      *W=-1;
      n=(1+*(N+k)/100);
      i=h=0;
      while(i<r) {
         S=S0+i;
         R=R0+locvec(k,r)+i;

         Sr=*S;
         Sr=MAX(Sr,eps);
         Sn=Sr*n;

         if(Sn<=Sm) while(*S<Sn && i<r) { i++; S++; R++; }
         else i=r;

         if(i<r && *S>=Sn) {
            *R=1;
            if(U) {
               *(W+h)=i; /* another N(k) reset point */
               h++;
               *(W+h)=-1;
            }
         }
      }
      k+=2;
   }

   k=0;
   while(k<3) { /* Resetting N(0) and N(2) and using reset points
                   of N(1) (for N(0)) and N(3) (for N(2)) if U!=0. */
      W=W0+locvec((1+k),r); /* require parentheses on (1+k) */
      n=(1+*(N+k)/100);
      i=h=0;
      while(i<r) {

         S=S0+i;
         R=R0+locvec(k,r)+i;

      /* Set the appropriate Sr for update strategy U: */
         if(U && *(W+h)!=-1 && i>=(j=*(W+h))) {
            Sr=*(S0+j); /* use the earlier N(k+1) reset point for k */
            h++;
         }
         else Sr=*S; /* use the current N(k) reset point for k */

         Sr=MAX(Sr,eps);
         Sn=Sr*n;

         if(Sn<=Sm) while(*S<Sn && i<r) { i++; S++; R++; }
         else i=r;

         if(i<r && *S>=Sn) *R=1;
      }
      k+=2;
   }
   return(lop() && lop() && lop());
}

int growthW() /* growth (hS hN nU --- hR) */
/* Sat Feb 18 08:42:05 PST 2012

   Incoming column S contains an S-shaped logistic function, an ever-
   increasing function.  Column N holds a set of four increasing growth
   percentage levels (such as the set N[0:3] = [50 100 200 300] per-
   cent). 

   Growth of S is monitored from initial base S(0), and when the growth
   of S increases by N(3)--the largest of set N--say to S(h), growth 
   monitoring of S for N(3) is reset to monitor growth of N(3) levels 
   from new base S(h).  

   Strategy flag U designates the base reset strategy for N[0:2].  

   If U is zero, then the strategy for these lesser levels is the same 
   as N(3) described above--each level updates independently.

   For nonzero strategy flag U, the reset strategy for N[0:2] is tied
   to resets of N(3).  All further descriptions and examples are for
   the more complex case of nonzero U.

   UPDATE Tue Feb 21 17:04:49 PST 2012.  For nonzero strategy flag U,
   the reset of N(k) is tied to the base of N(k+1), k=0:2 to make vis-
   ual interpretation even easier.  The writeup and example below re-
   flect earlier code where the reset of N(0), N(1) and N(2) were all
   tied to the base of N(3).

   The growth monitoring of lesser N[0:2], say N(1)<N(3), follows S 
   similarly to N(3) and ratchets to new steps in 1+N(1)/100 incre-
   ments.  However, if N(3) has reset to monitor its levels from S(h)
   during the current N(1) step, then when N(1) completes its step in
   progress at S(k), monitoring of N(1) is reset to monitor growth not
   from S(k), but from S(h), the earlier level reached by N(3).

   This scheme simplifies the visual interpretation of growth patterns,
   since much of the time they will be moving relative to the same ref-
   erence level, S(h), especially when the values in set N are growth 
   multiples of each other.

   Resets are returned in matrix R, where the four columns of R cor- 
   respond to the four rows of N.  Term R(i, k) contains 1 where S has 
   grown another N(k) percent: from the prior reference point for k to 
   point S(i).  The prior reference point for k<3 is subject to the 
   reset scheme involving N(3) described above.

   Test case (adapted from the example in man logistic):

      Paste these phrases into the top of file work.v and run ww at
      the ready prompt to produce a graph of logistic function S and
      the growth monitoring curves created from R:

        "plot.v" source pgrid_off pgrid

        0.1 200 uniform dup mean - "H" book
        H 0.5 logistic "S" book
 
        list: 50 100 200 300 ; "N" book \ growth steps, percent
        S N 1 (hS hN nU) growth (hR) "R" book

        R dup push cols 1st
        DO S peek I catch looking LOOP pull cols parkn (hS)
        (hS) S park H plot 

      Evaluating the graphical results.  

      This describes the four curves for the four values in N: N(0), 
      N(1), N(2) and N(3):

         N(0): 50%, factor of 1.5, blue
         N(1): 100%, factor of 2, dark green
         N(2): 200%, factor of 3, red
         N(3): 300%, factor of 4, purple 

      Values for curves can be read off using the mouse arrow and click-
      ing the left mouse button.

      To read values more accurately, left clicking two points to de-
      fine the diagonal corners of a zoom-in box, followed by a right
      click, will zoom in; right clicking after zooming in will zoom 
      back out.

      The N(3) curve (purple) has the largest steps, and it defines the
      major resets.  It shows a major step from 0.22 to 0.89, a factor 
      of 4 corresponding to N(3) steps of 300% (steps are taken to the
      point in S that is above the goal, so exact multiples are not 
      expected and they should be slightly higher, never lower).

      Between major resets in N(3), values for the dark green curve 
      (N(1)) are seen to double on each step, as they should for steps
      of 100%.

      Steps of the red curve, N(2), are seen at 0.538 eminating from
      0.175 (a factor of 3), and then at 0.68 eminating from 0.22 (also
      a factor of 3).  The base of 0.22 is due to the major reset by 
      N(3) that caused N(2) to use 0.22 (the base of N(3)) as its base
      when it reset upon reaching 0.538.

      All curves are seen to have stepped as high as they can; addi-
      tional steps in any of them would be above the top of logistic
      curve S.  

      Detailed examination shows the following behavior:

         When N(3) completes a step and resets to S(h), any growth step
         of N(k), k<3, that is in progress will continue until it com-
         pletes that step at S(k)>S(h).  

         Then N(k) will reset and begin measuring its next step from
         earlier S(h), not S(k).

         See description of this behavior above and comments in the 
         loops below.

      Sun Feb 19 20:24:31 PST 2012.

      Just writing the graphical evaluation above uncovered a handful
      of errors in curves that were incorrect but "ok looking."  Other
      runs in real production uncovered endless looping that was fixed
      with small number eps when S(0)=0, and seg faulting when h was
      not constrained (h<m) in the second while() loop (not relevant
      after update Tue Feb 21 17:04:49 PST 2012).  

      A test was added to verify that S is increasing--there is no use
      processing garbage if it can be helped.

      Is that it?  Never.  Until the next time, it is merely concluded 
      that this function can work correctly with no known errors.  

      Like all functions, it is probably just working correctly with
      unknown errors that are yet to be discovered. */
{
   double eps=1e-6,n,*N,*R,*R0,*S,*S0,Sm,Sn,Sr,*W=0,*W0=0;
   int h=0,i=0,j,k=0,r,U;

   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" growth: ",MATNOT2);
      return 0;
   }
   if((tos-1)->row<4) {
      gprintf(" growth: set N requires four values");
      stkerr("","");
      nc();
      return 0;
   }
   if(!popint(&U)) return 0;

   N=tos->mat;
   S0=S=(tos-1)->mat;
   r=(tos-1)->row;

   if(r<2) {
      gprintf(" growth: to grow, S requires two or more rows");
      stkerr("","");
      nc();
      return 0;
   }
   for(;k<r-1;k++) {
      if(*S>*(S+1)) {
         gprintf(" growth: S is not an increasing function");
         stkerr("","");
         nc();
         return 0;
      }
      S++;
   }
   S=S0;
   Sm=*(S0+r-1);

   if(U) {
      if(!matstk(r,3,"_W")) return 0; /* work array */
      W0=W=tos->mat;
   }
   if(!matstk(r,4,"_R")) return 0;
   R0=tos->mat;
   memset(R0,0,r*4*sizeof(double));
   
   k=0;
   while(k<4) {
      n=(1+*(N+k)/100);
      i=h=0;
      if(U && k>0) {
         W=W0+locvec((k-1),r); /* require parentheses on (k-1) */
         *W=-1;
      }
      while(i<r) {
         S=S0+i;
         R=R0+locvec(k,r)+i;

         Sr=*S;
         Sr=MAX(Sr,eps);
         Sn=Sr*n;

         if(Sn<=Sm) while(*S<Sn && i<r) { i++; S++; R++; }
         else i=r;

         if(i<r && *S>=Sn) {
            *R=1;
            if(U && k>0) {
               *(W+h)=i; /* another N(k) reset point */
               h++;
               *(W+h)=-1;
            }
         }
      }
      k++;
   }
   if(!U) return(lop() && lop());

   k=0;
   while(k<3) { /* Reset N(0), N(1) and N(2) reference. */
      n=(1+*(N+k)/100);
      i=h=0;
      W=W0+locvec(k,r);

      while(i<r) {
         S=S0+i;
         R=R0+locvec(k,r)+i;

      /* Set the appropriate Sr: */
         if(*(W+h)!=-1 && i>=(j=*(W+h))) {
            Sr=*(S0+j); /* use earlier N(k+1) reset point for N(k) */
            h++;
         }
         else Sr=*S; /* use the current N(k) reset point for N(k) */

         Sr=MAX(Sr,eps);
         Sn=Sr*n;

         if(Sn<=Sm) while(*S<Sn && i<r) { i++; S++; R++; }
         else i=r;

         if(i<r && *S>=Sn) *R=1;
      }
      k++;
   }
   return(lop() && lop() && lop());
}

int growthX() /* growth (hS hN nU --- hR) */
/* Sat Feb 18 08:42:05 PST 2012

   Incoming column S contains an S-shaped logistic function, an ever-
   increasing function.  Column N holds a set of four increasing growth
   percentage levels (such as the set N[0:3] = [50 100 200 300] per-
   cent). 

   Growth of S is monitored from initial base S(0), and when the growth
   of S increases by N(3)--the largest of set N--say to S(h), growth 
   monitoring of S for N(3) is reset to monitor growth of N(3) levels 
   from new base S(h).  

   Strategy flag U designates the base reset strategy for N[0:2].  If
   U is zero, then the strategy for these lesser levels is the same as
   N(3) described above--each level updates independently.

   For nonzero strategy flag U, the reset strategy for N[0:2] is tied
   to resets of N(3).  All further descriptions and examples are for
   the more complex case of U=1.

   The growth monitoring of lesser N[0:2], say N(1)<N(3), similarly 
   follows S and ratchets to new steps in 1+N(1)/100 increments.  How-
   ever, if N(3) has reset to monitor its levels from S(h) during the
   current N(1) step, then when N(1) completes its step in progress at
   S(k), monitoring of N(1) is reset to monitor growth not from S(k),
   but from S(h), the earlier level reached by N(3).

   This scheme simplifies the visual interpretation of growth patterns,
   since much of the time they will be moving relative to the same ref-
   erence level, S(h), especially when the values in set N are growth 
   multiples of each other.

   Resets are returned in matrix R, where the four columns of R cor- 
   respond to the four rows of N.  Term R(i, k) contains 1 where S has 
   grown another N(k) percent: from the prior reference point for k to 
   point S(i).  The prior reference point for k<3 is subject to the 
   reset scheme involving N(3) described above.

   Test case (adapted from the example in man logistic):

      Paste these phrases into the top of file work.v and run ww at
      the ready prompt to produce a graph of logistic function S and
      the growth monitoring curves created from R:

        "plot.v" source pgrid_off pgrid

        0.1 200 uniform dup mean - "H" book
        H 0.5 logistic "S" book
 
        list: 50 100 200 300 ; "N" book \ growth steps, percent
        S N 1 (hS hN nU) growth (hR) "R" book

        R dup push cols 1st
        DO S peek I catch looking LOOP pull cols parkn (hS)
        (hS) S park H plot 

      Evaluating the graphical results.  

      This describes the four curves for the four values in N: N(0), 
      N(1), N(2) and N(3):

         N(0): 50%, factor of 1.5, blue
         N(1): 100%, factor of 2, dark green
         N(2): 200%, factor of 3, red
         N(3): 300%, factor of 4, purple 

      Values for curves can be read off using the mouse arrow and click-
      ing the left mouse button.

      To read values more accurately, left clicking two points to de-
      fine the diagonal corners of a zoom-in box, followed by a right
      click, will zoom in; right clicking after zooming in will zoom 
      back out.

      The N(3) curve (purple) has the largest steps, and it defines the
      major resets.  It shows a major step from 0.22 to 0.89, a factor 
      of 4 corresponding to N(3) steps of 300% (steps are taken to the
      point in S that is above the goal, so exact multiples are not 
      expected and they should be slightly higher, never lower).

      Between major resets in N(3), values for the dark green curve 
      (N(1)) are seen to double on each step, as they should for steps
      of 100%.

      Steps of the red curve, N(2), are seen at 0.538 eminating from
      0.175 (a factor of 3), and then at 0.68 eminating from 0.22 (also
      a factor of 3).  The base of 0.22 is due to the major reset by 
      N(3) that caused N(2) to use 0.22 (the base of N(3)) as its base
      when it reset upon reaching 0.538.

      All curves are seen to have stepped as high as they can; addi-
      tional steps in any of them would be above the top of logistic
      curve S.  

      Detailed examination shows the following behavior:

         When N(3) completes a step and resets to S(h), any growth step
         of N(k), k<3, that is in progress will continue until it com-
         pletes that step at S(k)>S(h).  

         Then N(k) will reset and begin measuring its next step from
         earlier S(h), not S(k).

         See description of this behavior above and comments in the 
         loops below.

      Sun Feb 19 20:24:31 PST 2012.

      Just writing the graphical evaluation above uncovered a handful
      of errors in curves that were incorrect but "ok looking."  Other
      runs in real production uncovered endless looping that was fixed
      with small number eps when S(0)=0, and seg faulting when h was
      not constrained (h<m) in the second while() loop.  

      A test was added to verify that S is increasing--there is no use
      processing garbage if it can be helped.

      Is that it?  Never.  Until the next time, it is merely concluded 
      that this function can work correctly with no known errors.  

      Like all functions, it is probably just working correctly with
      unknown errors that are yet to be discovered. */
{
   double eps=1e-6,n,*N,*R,*R0,*S,*S0,Sm,Sn,Sr,*W;
   int h,i=0,j,k=0,m=0,r,U;

   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" growth: ",MATNOT2);
      return 0;
   }
   if((tos-1)->row<4) {
      gprintf(" growth: set N requires four values");
      stkerr("","");
      nc();
      return 0;
   }
   if(!popint(&U)) return 0;

   N=tos->mat;
   S0=S=(tos-1)->mat;
   r=(tos-1)->row;

   if(r<2) {
      gprintf(" growth: to grow, S requires two or more rows");
      stkerr("","");
      nc();
      return 0;
   }
   for(;k<r-1;k++) {
      if(*S>*(S+1)) {
         gprintf(" growth: S is not an increasing function");
         stkerr("","");
         nc();
         return 0;
      }
      S++;
   }
   S=S0;
   Sm=*(S0+r-1);

   if(!matstk(r,1,"_W")) return 0; /* work array */
   W=tos->mat;

   if(!matstk(r,4,"_R")) return 0;
   R0=tos->mat;
   memset(R0,0,r*4*sizeof(double));
   
   k=3;
   while(i<r) {
      Sr=MAX(*S,eps);
      Sn=Sr*(1+*(N+k)/100);
      R=R0+locvec(k,r)+i;

      if(Sn<=Sm) while(*S<Sn && i<r) { i++; S++; R++; }
      else i=r;

      if(i<r && *S>=Sn) {
         *R=1;
         *(W+m)=i; /* another N(3) reset point */
         m++;
      }
   }
   k=0;
   while(k<4) {
      n=(1+*(N+k)/100);
      i=h=0;
      while(i<r) {
         S=S0+i;
         R=R0+locvec(k,r)+i;

      /* Set the appropriate Sr for update strategy U: */
         if(U && h<m && i>=(j=*(W+h))) {
            Sr=*(S0+j); /* use the earlier N(3) reset point for k */
            h++;
         }
         else Sr=*S; /* use the current N(k) reset point for k */

         Sr=MAX(Sr,eps);
         Sn=Sr*n;

         if(Sn<=Sm)
            while(*S<Sn && i<r) { i++; S++; R++; }
         else i=r;

         if(i<r && *S>=Sn) *R=1;
      }
      k++;
   }
   return(lop() && lop() && lop());
}

double _fivepow(double x) { return(pow((double)5,x)); }

int fivepow() /* 5^ (hA --- hA1) */
{
   if(is_complex(tos)) {
      stkerr(" 5^: ",REALNOT);
      return 0;
   }
   return(byterm1(FIVEPOW));
}

int gt() { return(byterm2(GT)); } /* > (hA hB --- hF) */

double _gt(double x, double y) 
{ 
   return(xTRUE*(x>y)); 
}

int histogram() /* histogram (hV hC dC --- hH) */
/* Fri Jan 13 20:23:01 PST 2012

   Create histogram H for counts of V in bins of C.

   Returned H has three columns.

   Column 1 of H defines uniform bins of width dC that span the range 
   of values in vector C, from lowest to highest.  For example, if C(m)
   belongs to bin k, then C(m)<=H(k).

   Column 2 of H contains the total counts from V(C) for each bin, the
   basic histogram.  For example, if C(m) belongs to bin k, then V(m)
   is summed into H(k,2) as follows: H(k,2)+=V(m).  If bin k is empty,
   then H(k,2)=0.

   Column 3 of H gives the row number in V that contributed the largest
   element summed into each bin.  For example, if V(m) contributed the
   largest element to bin k, then H(k,3) equals m.  If bin k is empty, 
   then H(k,3)=-1.

   Tue Mar 13 07:04:44 PDT 2012.  Modify bins to be symmetric with re-
   spect to the range of values in C.

   Test case:

      Using these phrases in the top of file work.v:

         list: 100 700 300 400 600  500 ; "V" book
         list: 6.6 6.3 3.3 4.4 2.35 1.1 ; "C" book
         nl
         " Input         C        V" . nl C V park .m nl

         " One-based    H1       H2       H3" nl . nl
         1based V C 1.25 histogram .m nl

         " Zero-based   H1       H2       H3" nl . nl
         0based V C 1.25 histogram .m nl

         halt

   Test case results after modification, Tue Mar 13 07:04:44 PDT 2012:
 
      Running file work.v with command ww:

         [dale@kaffia] /home/dale > tops
                  Tops 3.2.0
         Tue Mar 13 07:30:00 PDT 2012
         [tops@kaffia] ready > ww
 
          Input         C        V
          Row 1:      6.6      100
          Row 2:      6.3      700
          Row 3:      3.3      300
          Row 4:      4.4      400
          Row 5:     2.35      600
          Row 6:      1.1      500
 
          One-based    H1       H2       H3
          Row 1:      0.1        0       -1
          Row 2:     1.35      500        6
          Row 3:      2.6      600        5
          Row 4:     3.85      300        3
          Row 5:      5.1      400        4
          Row 6:     6.35      700        2
          Row 7:      7.6      100        1
 
          Zero-based   H1       H2       H3
          Row 1:      0.1        0       -1
          Row 2:     1.35      500        5
          Row 3:      2.6      600        4
          Row 4:     3.85      300        2
          Row 5:      5.1      400        3
          Row 6:     6.35      700        1
          Row 7:      7.6      100        0
 
         [tops@kaffia] ready >

      These results differ from the earlier ones below due to modifica-
      tion of bins, so some parts of the discussion below no longer
      apply.  Accounting for bin differences, the results above are
      also concluded to be correct.
 
   Test case results Sat Jan 14 16:58:39 PST 2012:

      Running file work.v with command ww:

         [dale@kaffia] /opt/tops/tops/src > tops
                  Tops 3.2.0
         Sat Jan 14 16:58:39 PST 2012
         [tops@kaffia] ready > ww
 
          Input         C        V
          Row 1:      6.6      100
          Row 2:      6.3      700
          Row 3:      3.3      300
          Row 4:      4.4      400
          Row 5:     2.35      600
          Row 6:      1.1      500
 
          One-based    H1       H2       H3
          Row 1:     2.35     1100        5
          Row 2:      3.6      300        3
          Row 3:     4.85      400        4
          Row 4:      6.1        0       -1
          Row 5:     7.35      800        2
 
          Zero-based   H1       H2       H3
          Row 1:     2.35     1100        4
          Row 2:      3.6      300        2
          Row 3:     4.85      400        3
          Row 4:      6.1        0       -1
          Row 5:     7.35      800        1
 
         [tops@kaffia] ready >

      Results are deemed correct by the following inspections:

         Zero-based and one-based results match.  Zero-based offsets
         in H3 are one less than one-based indices, as required; -1 
         values correspond to an empty bin.

         Each H1(k) shows the upper limit for C of bin k.  Bins in H1 
         cover the range of C, 1.1 to 6.6, in steps of 1.25.

         The first bin (Row 1) will contain V for the lowest value of C
         (1.1) and V for other C values ranging up to 1.1+1.25 (2.35).  

         Bin 5, the last bin, covers the range above 6.1, up to 7.35.
         It will contain V for values of C above 6.1 including highest 
         C (6.6).

         The total sum of histogram column H2 is 2600, matching the sum 
         of incoming V.

         Bin 1 must contain V(5) and V(6) because C(5) and C(6) are less
         than or equal to H1(1).  H2(1) holds the sum, 1100.  One-based
         H3(1)=5 defines V(5) (600) as the larger contributor to the 
         sum 1100.

         Bin 4 must be empty because there are no terms in C between
         4.85 and 6.1.  Empty bin 4 is denoted by H2(4)=0 and H3(4)=-1.

         Bin 5 must contain V(1) and V(2) which sum to 800 shown in 
         H2(5).  One-based H3(5)=2, the index for V(2) (700) which is
         the larger contributor. */
{
   double *C,*H1,*H2,*H3,*R,*V;
   double Cbar,Cmax,Cmin,dC,HSUM=0,VMAX=-INF,RMAX=-1;
   int b,d,i,k=0,r;

   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" histogram: ",MATNOT2);
      return 0;
   }
   if((r=(tos-1)->row)!=(tos-2)->row) {
      stkerr(" histogram: ",ROWSNOT);
      return 0;
   }
   if(!popd(&dC)) return 0;

   if(!(dC>0)) {
      gprintf(" histogram: bin width must be greater than zero");
      stkerr("","");
      nc();
      return 0;
   }
   if(!matstk(r,3,"_W")) return 0; /* r-by-3 work matrix W */

   C=tos->mat;
   memcpy(C,(tos-1)->mat,sizeof(double)*r); /* C in W(*,1) */
   lop();

   V=C+locvec(1,r);
   memcpy(V,(tos-1)->mat,sizeof(double)*r); /* V in W(*,2) */
   lop();

   R=C+locvec(2,r);
   pushd(1);
   pushd(r);
   uniform();
   memcpy(R,tos->mat,r*sizeof(double)); /* row offsets in W(*,3) */
   drop();

/* Sort C in W(*,1).  Counts V in W(*,2) and row offsets in W(*,3) go 
   along for the ride: */
   pushint(xTRUE);
   sort1();
   C=tos->mat;      /* sorted C, W(*,1) */
   V=C+locvec(1,r); /* shuffled V, W(*,2) */
   R=C+locvec(2,r); /* shuffled R, W(*,3) */

/* Initialize array H.  Set up bins to be symmetric with respect to
   the average of the largest and smallest terms in C: */
   Cbar=(*C+*(C+r-1))/2;
   d=1+(Cbar-*C)/dC;
   Cmin=Cbar-d*dC;
   Cmax=Cbar+d*dC;
   b=1;
   while((Cmin+b*dC)<=Cmax) b++; /* b is the number of bin lines */
      
   if(!matstk(b,3,"_H")) return 0; /* b-by-3 histogram array H */
   H1=tos->mat;

   i=0;
   while(i<b) { *H1=Cmin+i*dC; i++; H1++; } /* H(*, 1) bin lines */

   H1=tos->mat;
   H2=H1+locvec(1,b);
   H3=H1+locvec(2,b);
   
   memset(H2,0,b*sizeof(double)); /* H(*,2)=0 */

   pushd(-1);
   pushd(b);
   pushd(1);
   fill();
   memcpy(H3,tos->mat,b*sizeof(double)); /* H(*,3)=-1 */
   drop();

/* Sum the r elements of V into the b bins of H: */
   i=0;
   while(i<r) {

      while(i<r && *C<=*H1) {
         HSUM+=*V;
         if(*V>VMAX) {
            VMAX=*V;
            RMAX=*R;
         }
         V++;
         C++;
         R++;
         i++;
      }
      if(HSUM) {
         *H2+=HSUM;
         *H3=RMAX+XBASE;
      }
      HSUM=0;
      VMAX=-INF;
      RMAX=-1;
          
      while(k<b && *C>*H1) {
         H3++;
         H2++;
         H1++;
         k++;
      }
   }
   return(lop()); /* drop work matrix W from stack */
}

int INFpos() { return(pushd(INF)); } /* INF ( --- bigval) */

int INFneg() { return(pushd(-INF)); } /* -INF ( --- -bigval) */

void insrt(double *A, int n, int *rnew, double Anew, double *B, \
   double Bnew)
/* Sat Apr 20 18:55:01 PDT 2013

   Incoming vector A with n rows is already in ascending order sort.

   Determine row rnew where new value Anew is inserted into A so that 
   returned A is still in sort with n+1 rows.

   If B is not NULL, also insert new value Bnew into companion vector 
   B at row rnew.

   The value of returned rnew is the zero-based index where Anew was 
   inserted into A and Bnew was inserted into B. 

   Warning: vectors A and B (unless B==NULL) must have length n+1 or 
   greater before this function is called. 

   For test cases, see funtest() in file wapp.c and the test cases in 
   this file for ranking() and ranking1() which use this function. */
{
   if(B==NULL) {
      if(Anew>=*(A+n-1)) *rnew=n;
      else { 
         if(Anew<=*A) *rnew=0;
         else if(!bsearchd(Anew,A,n,rnew)) *rnew+=1;
         memmove(A+*rnew+1,A+*rnew,(n-*rnew)*sizeof(double));
      }
      *(A+*rnew)=Anew;
      return;
   }
   if(Anew>=*(A+n-1)) *rnew=n;
   else { 
      if(Anew<=*A) *rnew=0;
      else if(!bsearchd(Anew,A,n,rnew)) *rnew+=1;
      memmove(A+*rnew+1,A+*rnew,(n-*rnew)*sizeof(double));
      memmove(B+*rnew+1,B+*rnew,(n-*rnew)*sizeof(double));
   }
   *(A+*rnew)=Anew;
   *(B+*rnew)=Bnew;
}

double _int(double x) { return((long long)x); }

int int2_() { return(byterm1(INT2)); } /* not used (hA --- hI) */
int int4_() { return(byterm1(INT4)); } /* int | ints (hA --- hI) */
int int8_() { return(byterm1(INT8)); } /* int8 (hA --- hI) */

int uint2_() { return(byterm1(UINT2)); } /* not used (hA --- hI) */
int uint4_() { return(byterm1(UINT4)); } /* uint | uints (hA --- hU) */
int uint8_() { return(byterm1(UINT8)); } /* not used (hA --- hI) */

int int2() /* int2 or uint2 (hA --- hT) */
/* 8-byte fp to 2 byte int; works for signed or unsigned */
{
   int f=0,ret=0;

   if(tos->typ==NUM) {
      hand();
      f=1;
   }
   ret=(byteorder1() && export2());
   if(f && ret) 
      ret=(
         ret &&
         pushint(XBASE) &&
         quote()
      );
   return(ret);
}

int int4() /* int4 or uint4 (hA --- hT) */
/* 8-byte fp to 4 byte int; works for signed or unsigned */
{
   int f=0,ret=0;

   if(tos->typ==NUM) {
      hand();
      f=1;
   }
   ret=(byteorder1() && export4());
   if(f && ret) 
      ret=(
         ret &&
         pushint(XBASE) &&
         quote()
      );
   return(ret);
}  

double _int2(double x)
/* Converting 8-byte floating point bit pattern into 2-byte integer
   bit pattern followed by 6 null bytes. */
{
   union {
      double x;
      unsigned short u;
      signed short s;
   } X={0};

   if(x<0) X.s=(signed short)x;
   else X.u=(unsigned short)x;
   return(X.x);
}

double _int4(double x)
/* Converting 8-byte floating point bit pattern into 4-byte integer 
   bit pattern followed by 4 null bytes. */
{  
   union {
      double x;
      int i;
   } X={0};

   if(x<0) X.i=(signed int)x;
   else X.i=(unsigned int)x;
   return(X.x);
}

double _int8x(double x)
/* Converting 8-byte floating point bit pattern into 8-byte integer 
   bit pattern. */
{  
   union {
      double x;
      long long i;
   } X={0};

   X.i=(long long)x;
   return(X.x);
}

int integer() { return(byterm1(INT)); } /* integer (hA --- hA1) */

int intervals() /* intervals (x1 x2 n --- hX) */
{
   int n;
   double x1,x2;
   register int i=1;
   register double delta,*X,X1;

   if(!popint(&n) || !popd(&x2) || !popd(&x1)) return 0;

   if(n<=0) n=1;

   if(!matstk(1+n,1,"_intervals")) {
      stkerr(" intervals: ",MEMNOT);
      return 0;
   }
   X=tos->mat;
   *X=x1;
   *(X+n)=x2;

   delta=(x2-x1)/n;
   X1=x1;

   for(;i<n;i++) *(X+i)=X1+i*delta;

   return 1;
}

int lag() /* lag (hA n -- hA1) */
/* All A rows shifted down by n rows; if n is negative, rows are 
   shifted up. 

   Sun Jan 12 08:16:14 PST 2014.  Initialize the first n+1 values 
   of A1 to A(0) if n>0 or the last n+1 values of A1 to A(rows-1) 
   if n<0. 

   Sun Jan 12 09:02:56 PST 2014.  Testing:

      Paste the following at the ready prompt:

         list: 1 10 thru ; dup reversed park \
         "A" book \
         A dup 3 lag park nl .m nl  \
         A dup -3 lag park nl .m nl

      Running the test case:

         [dale@kaffia] /opt/tops/tops/src > tops
                  Tops 3.2.1
         Sun Jan 12 09:05:17 PST 2014
         [tops@kaffia] ready > list: 1 10 thru ; dup reversed park \
                               "A" book \
                               A dup 3 lag park nl .m nl  \
                               A dup -3 lag park nl .m nl

                       A[1]     A[2]     A1[1]    A1[2]
           Row 1:        1       10        1       10
           Row 2:        2        9        1       10
           Row 3:        3        8        1       10
           Row 4:        4        7        1       10
           Row 5:        5        6        2        9
           Row 6:        6        5        3        8
           Row 7:        7        4        4        7
           Row 8:        8        3        5        6
           Row 9:        9        2        6        5
          Row 10:       10        1        7        4

                       A[1]     A[2]     A1[1]    A1[2]
           Row 1:        1       10        4        7
           Row 2:        2        9        5        6
           Row 3:        3        8        6        5
           Row 4:        4        7        7        4
           Row 5:        5        6        8        3
           Row 6:        6        5        9        2
           Row 7:        7        4       10        1
           Row 8:        8        3       10        1
           Row 9:        9        2       10        1
          Row 10:       10        1       10        1

         [tops@kaffia] ready > bye 
         140 keys
                   Good-bye
         Sun Jan 12 09:05:33 PST 2014
         [dale@kaffia] /opt/tops/tops/src > 

         Results are correct by inspection. */
{
   register double *S,*T;
   double A;
   int cols,i,j=0,len,n,rows;

   if(!popint(&n)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" lag: ",MATNOT);
      return 0;
   }
   if(n==0) return(cop());

   rows=tos->row;
   cols=tos->col;
   S=tos->mat;

   if(!matstk(rows,cols,"_lag")) return 0;
   T=tos->mat;

   if(n>0) {
      n=MIN(n,rows);
      len=(rows-n)*sizeof(double);
      for(;j<cols;j++) {
         memcpy(T+n,S,len);
         for(i=0;i<n;i++) *(T+i)=*S;
         S+=rows;
         T+=rows;
      }
   }
   else {
      n=ABS(n);
      n=MIN(n,rows);
      len=(rows-n)*sizeof(double);
      for(;j<cols;j++) {
         memcpy(T,S+n,len);
         A=*(S+rows-1);
         for(i=rows-1;i>rows-n-1;i--) *(T+i)=A;
         S+=rows;
         T+=rows;
      }
   }
   return(lop());
}

int lagN() /* lagN (hA hN -- hB) */
/* Wed Jan 29 12:46:31 PST 2014

   Output vector B is initialized to vector A.  

   Then row B(k+N(k)) is set to A(k), where N(k)>=0.

   Test case: Paste the following lines at the ready prompt to compare
   output with the output from lag():

      list: 1 2 3 4 5 6 7 ; "A" book \
      2 "n" book n A rows 1 fill "N" book \
      A A n lag A N lagN 3 parkn .m nl

   Running the test case:

      [dale@kaffia] /opt/tops/tops/src > tops
               Tops 3.2.1
      Wed Jan 29 14:04:44 PST 2014
      [tops@kaffia] ready > list: 1 2 3 4 5 6 7 ; "A" book \
      >       2 "n" book n A rows 1 fill "N" book \
      >       A A n lag A N lagN 3 parkn .m nl

                     A    lag(A,n)  lagN(A,N)
       Row 1:        1        1        1
       Row 2:        2        1        2
       Row 3:        3        1        1
       Row 4:        4        2        2
       Row 5:        5        3        3
       Row 6:        6        4        4
       Row 7:        7        5        5

      [tops@kaffia] ready > bye
      120 keys
                Good-bye
      Wed Jan 29 14:04:47 PST 2014
      [dale@kaffia] /opt/tops/tops/src > 

      The first 2 rows differ due to initialization rules; the rest 
      of the rows agree. */
{
   register double *A,*B,*N;
   int k=0,n,rows;

   if(!(tos->typ==MAT && (tos-1)->typ==MAT)) {
      stkerr(" lagN: ",MATNOT2);
      return 0;
   }
   rows=tos->row;
   if((tos-1)->row!=rows) {
      stkerr(" lagN: ",ROWSNOT);
      return 0;
   }
   N=tos->mat;
   A=(tos-1)->mat;

   if(!matstk(rows,1,"_lagN")) return 0;
   B=tos->mat;

   for(;k<rows;k++) *(B+k)=*(A+k);

   for(k=0;k<rows;k++) {
      n=(int)*(N+k);
      if((k+n)<rows) *(B+k+n)=*(A+k);
   }
   return(lop() && lop());
}

int lagr() /* lagr (hA n -- hA1) */
/* Sat Mar  2 17:07:34 PST 2013.  

   All A rows are shifted down by n rows; require n>0.  

   Works the same as lag, then the order of each n row set of data 
   is reversed.

   Testing:

      [tops@kaffia] ready > 1 12 items "A" book 
                            A A 3 lag A 3 lagr 3 parkn .m

                      A   lag(A, n) lagr(A, n)
        Row 1:        1        1        3
        Row 2:        2        2        2
        Row 3:        3        3        1
        Row 4:        4        1        3
        Row 5:        5        2        2
        Row 6:        6        3        1
        Row 7:        7        4        6
        Row 8:        8        5        5
        Row 9:        9        6        4
       Row 10:       10        7        9
       Row 11:       11        8        8
       Row 12:       12        9        7
      [tops@kaffia] ready > */
{
   register double *S,*S0,*T,*T0;
   int cols,j,k=0,n,quo,rem,rows;

   if(tos->typ!=NUM) {
      stkerr(" lagr: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" lagr: ",MATNOT);
      return 0;
   }
   n=tos->real;
   if(!(n>0)) {
      gprintf(" lagr: require n>0");
      stkerr("","");
      nc();
      return 0;
   }
   lag();

   rows=tos->row;
   cols=tos->col;
   S0=tos->mat+rows-n;

   if(n>rows) return 1;

   if(!matstk(rows,cols,"_lagr")) return 0;
   T0=tos->mat+rows-n;

   quo=rows/n;
   rem=rows-quo*n;

   for(;k<quo;k++) {
      S=S0-n*k;
      T=T0-n*k;
      for(j=0;j<cols;j++) {
         memcpyr(T,S,n);
         S+=rows;
         T+=rows;
      }
   }
   if(rem) {
      S=(tos-1)->mat;
      T=tos->mat;
      for(j=0;j<cols;j++) {
         memcpyr(T,S,rem);
      }
   }
   return(lop());
}

int lagV() /* lagV (hV hN -- hA) */
/* Tue Jan 14 15:37:01 PST 2014

   A(k) equals vector V with all rows shifted down by n=N(k) rows.
   The first n+1 values of A(k) are set to V(0). 

   Test case: Paste the following lines at the ready prompt:

      seed0 seedset \
      20000 1 random 1000 * integer "V" book \
      list: 100 200 400 0 -100 ; "N" book \
\
      V N lagV "A" book \
      V N 1st pry lag A 1st catch - null? \
      V N 2nd pry lag A 2nd catch - null? and \
      V N 3rd pry lag A 3rd catch - null? and \
      V N 4th pry lag A 4th catch - null? and \
      V N 5 ndx pry lag A 5 ndx catch - null? and \
      IF " lagV results agree with lag" \
      ELSE " lagV results do not agree with lag" \
      THEN . nl \ */
{
   register double *A,*N,*V;
   double C;
   int cols,i,j=0,len,n,rows;

   if((tos-1)->typ!=MAT && tos->typ!=MAT) {
      stkerr(" lagV: ",MATNOT);
      return 0;
   }
   V=(tos-1)->mat;
   N=tos->mat;

   cols=tos->row;
   rows=(tos-1)->row;

   if(!matstk(rows,cols,"_lagV")) return 0;
   A=tos->mat;

   for(;j<cols;j++) {
      n=*(N+j);
      if(n>0) {
         n=MIN(n,rows);
      /* Wed Jan 15 04:45:59 PST 2014.  Tested and verified.  
         The following three lines are equivalent to expression 
         _lag(V,rows,n,A); (see _lag() in file mmath.c): */
         len=(rows-n)*sizeof(double);
         memcpy(A+n,V,len);
         for(i=0;i<n;i++) *(A+i)=*V;
      }
      else {
      /* Mon Jul 14 21:50:08 PDT 2014.  Added for negative lags, similar
         to the branch in lag(), file math.c: */
         if(n<0) {
            n=MIN(ABS(n),rows);
            len=(rows-n)*sizeof(double);
            memcpy(A,V+n,len);
            C=*(V+rows-1);
            for(i=rows-1;i>rows-n-1;i--) *(A+i)=C;
         }
         else {
            len=rows*sizeof(double);
            memcpy(A,V,len);
         }
      }
      A+=rows;
   }
   return(lop() && lop());
}

int le() { return(byterm2(LE)); } /* <= (hA hB --- hF) */

double _le(double x, double y) 
{ 
   return(xTRUE*(x<=y)); 
}

int lerp(int opt, double *xtable, double *ytable, int lentable,
         double x, double *y) 
/* Performing linear interpolation to find y(x) from given tables:

                 y = y1 + (x - x1)*(y2 - y1)/(x2 - x1)
   where
      xtable holds ascending order x values
      ytable parallels xtable to give y(x), i.e., value ytable(i) 
         goes with value xtable(i)

   and opt cases are as follows:
      INTERP: interpolates y using nearest xtable values that strad-
         dle x
      XTRAP: interpolates y using xtable values that are both less
         than x, causing extrapolation
      CLOSESTX: does not interpolate; uses ytable(i) value that 
         corresponds to xtable(i) that is closest to x
      NEARBELOW: does not interpolate; uses ytable(i) value that 
         corresponds to xtable(i) that is nearest-below x
      FINDX: if x is in xtable, returns int value 1, 0 otherwise (ytable
         is not used in this case)

   Returned int value 1 indicates that x was found in the binary search.
*/
{
   int bump=0,delta,found,len;

   switch(opt) {

      case NEARBELOW: 

      case FINDX: 
         len=lentable;
         break;

      default:
         len=lentable-1;
   }
   found=bsearchd(x,xtable,len,&delta);

   switch(opt) {

      case CLOSESTX:
         bump=(*(xtable+delta+1)-x < x-*(xtable+delta));
         *y=*(ytable+delta+bump);
         return found;

      case NEARBELOW:
         *y=*(ytable+delta);
         return found;

      case XTRAP:
         delta=MAX(0,delta-1);
         break;

      case FINDX:
         return found;
   }
   *y=*(ytable+delta) + (x-*(xtable+delta))*(*(ytable+delta+1)
        - *(ytable+delta))/(*(xtable+delta+1) - *(xtable+delta)); 
   return found;
}

int lerp1(int opt, int srch)
/* Servicing cases of interpolation, each with a stack diagram given by

      (hXYt hX --- hY) 

   where XY table, XYt, contains a column vector of ascending order x
   values in the first column and a matrix of y(x) values in the second
   and subsequent columns.  

   The interpolation function performed on term X(i,j) of incoming ma-
   trix X, using table XYt, produces the value in Y(i,j) of returned
   matrix Y.

   IMPORTANT.  NOTE THIS:
   The number of columns in X is N, producing N columns in Y.  Thus
   table XYt has 1+N columns.

   Cases opt are as follows:
      LERP: Linear interpolation of values in matrix X into matrix Y,
         using table XYt, where column 1 of XYt contains x values and
         columns 2 - N+1 of XYt contains corresponding y(1:N) values 
         for the N columns of X.
 
      LEXT: Linear extrapolation within table, to mimic real time case
         where X is time and the two table x values below X are used.
 
      CLOSEST: No interpolation.  Returns table y corresponding to 
          table x values that are closest to incoming X.
 
      LOOK: No interpolation.  Returns table y corresponding to table x
          values that are equal to or nearest below incoming X.
 
      FIND: Finds X elements in table Xt (there are no Y columns in the
          incoming table), and in the elements of Y returns true if 
          found, 0 if not.

   Cases srch correspond to cases in lerp(), which are:
      INTERP,XTRAP,CLOSEST,NEARBELOW,FINDX

   They happen to parallel the cases of opt: LERP uses INTERP, LEXT
   uses XTRAP, CLOSEST uses CLOSESTX, LOOK uses NEARBELOW and FIND 
   uses FINDX. */
{
   stkitem *XYstk;
   unsigned register int cols,i=0,j=0,lentable,rows;
   register double *X,*Y,*Xt,*Yt;
   double *Y0;
   int xycols;

   if(!hand()) return 0;

   if(!(tos->typ==MAT && (tos-1)->typ==MAT)) {
      stkerr(" lerp1: ",MATNOT2);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;
   
   XYstk=tos-1;
   xycols=XYstk->col;
   if(xycols!=1+cols) {
      switch(opt) {
         default:
            stkerr(" lerp1: ", \
               "Interpolation table is of the form: [X Y]");
            gprintf("        where X is a vector in ascending order");
            nc();
            gprintf("        and Y is matrix of N functions of X.");
            nc();
            return 0;

         case FIND:
            break;
      }
   }
   if((lentable=XYstk->row)<2) {
      if(lentable==0) {
         stkerr(" lerp1: ","XY table is purged");
         return 0;
      }
      switch(opt) {
         default:
            stkerr(" lerp1: ", \
               "Interpolation table [X Y] requires two rows minimum");
            gprintf("        where X is a vector in ascending order");
            nc();
            gprintf("        and Y is matrix of N functions of X.");
            return 0;

         case CLOSEST:
         case LOOK:
         case FIND:
            break;
      }
   }
   if((Y0=(double *)memget(rows,cols))==NULL) {
      stkerr(" lerp1: ",MEMNOT); 
      return 0;
   }
   X=tos->mat;
   Xt=XYstk->mat;
   Yt=Xt+lentable;
   Y=Y0;

   switch(opt) {

      default:
         for(;j<cols;j++) {

            for(i=0;i<rows;i++) {
               lerp(srch,Xt,Yt,lentable,*(X+i),Y+i);
            }
            X+=rows;
            Y+=rows;
            Yt+=lentable;
         }
         break;
    
      case FIND:
         for(;j<cols;j++) {

            for(i=0;i<rows;i++) {
               *(Y+i)=lerp(srch,Xt,Xt,lentable,*(X+i),Y+i);
            }
            X+=rows;
            Y+=rows;
         }
         break;
   }
   return(
      drop2() &&
      push(MAT,(char *)memgetn("_lerp",5),NOTAG,0,Y0,NULL, \
         rows,cols,NULL)
   );
}

int lerp2() /* lerp (hXYt hX --- hY) */
{
   return(lerp1(LERP,INTERP));
}

int lext() /* lext (hXYt hX --- hY) */
{
   return(lerp1(LEXT,XTRAP));
}

double _ln(double x) { return(log(x)); }

int ln() /* ln (hA --- hA1) */
{ 
   if(is_complex(tos)) {
      stkerr(" ln: ",REALNOT);
      return 0;
   }
   return(byterm1(LN)); 
} 

double _log2(double x) 
{ 
   const double oneoverln2=1/log(2);
   return(log(x)*oneoverln2); /* log2(x) = ln(x)/ln(2) */
}

int log2a() /* log2 (hA --- hA1) */
{ 
   if(is_complex(tos)) {
      stkerr(" log2: ",REALNOT);
      return 0;
   }
   return(byterm1(LOG2)); 
} 

double _log5(double x) 
{ 
   const double oneoverln5=1/log(5);
   return(log(x)*oneoverln5); /* log5(x) = ln(x)/ln(5) */
}

int log5() /* log5 (hA --- hA1) */
{
   if(is_complex(tos)) {
      stkerr(" log5: ",REALNOT);
      return 0;
   }
   return(byterm1(LOG5)); 
} 

double _log10(double x) { return(log10(x)); }

int log10_() /* log10 (hA --- hA1) */
{ 
   if(is_complex(tos)) {
      stkerr(" log10: ",REALNOT);
      return 0;
   }
   return(byterm1(LOG10)); 
} 

int logistic_rmax() /* logistic_rmax (hdS --- hR) */
/* Fri Feb 17 07:13:43 PST 2012

   Each column of incoming S is a difference curve made from the dif-
   ference of successive values of an S-shaped logistic or sigmoid 
   function, an ever-increasing growth function.  No values in S should
   be negative.

   Each column of returned R contains a 1 at R(k, *) where dS(k, *) is
   a local maximum, i.e., the highest value seen so far.

   Test case (adapted from the example in man logistic):

      Paste these phrases into the top of file work.v and run ww at
      the ready prompt:

        "plot.v" source pgrid_off pgrid

        100 "n" book \ 100% growth steps

        0.1 80 uniform dup mean - "H" book

        list: 0.3 0.5 1.0 5.0 ; "Beta" book

        Beta dup push rows 1st
        DO H peek I pry logistic delta dup (hdS) \ difference
           (hdS) logistic_rmax (hdS hR) park 
        LOOP (hS1 hS2 ...) pull rows parkn (hS)
        (hS) eview

      Result: since these are ideal logistic functions, they have
      higher and higher rates until the midpoint, so of the 80 rows
      of R, R(1) through R(41) equal 1 and the rest equal 0. */
{
   double *R,*R1,*dS,*dS1,dSmax;
   int c,i,k=0,r;

   if(tos->typ!=MAT) {
      stkerr(" logistic_rmax: ",MATNOT);
      return 0;
   }
   dS1=tos->mat;
   r=tos->row;
   c=tos->col;

   if(!matstk(r,c,"_R")) return 0;
   R1=tos->mat;
   memset(R1,0,r*c*sizeof(double));

   for(;k<c;k++) {
      dS=dS1+locvec(k,r);
      R=R1+locvec(k,r);
      dSmax=-INF;
      for(i=0;i<r;i++) {
         while(*dS<=dSmax && i<r) {
            i++;
            dS++;
            R++;
         }
         if(*dS>dSmax) {
            dSmax=*dS;
            *R=1;
         }
      }
   }
   return(lop());
}

int logistic_steps() /* logistic_steps (hS n% --- hR) */
/* Fri Feb 17 02:54:55 PST 2012

   Each column of incoming S contains a curve like an S-shaped logis-
   tic or sigmoid function, an ever-increasing growth function.

   Each column of returned R contains a 1 at R(k, *) where S(k, *)
   has stepped (grown) another n%. 

   Test case (adapted from the example in man logistic):

      Paste these phrases into the top of file work.v and run ww at
      the ready prompt:

        "plot.v" source pgrid_off pgrid

        100 "n" book \ 100% growth steps

        0.1 80 uniform dup mean - "H" book

        list: 0.3 0.5 1.0 5.0 ; "Beta" book

        Beta dup push rows 1st
        DO H peek I pry logistic dup (hS)
           (hS) dup n logistic_steps (hS hR)
           (hS hS hR) looking  (hS1) swap (hS1 hS) park (hS)
        LOOP (hS1 hS2 ...) pull rows parkn (hS)

        (hS) H plot pause plotclose HALT \ */
{
   double n,*R,*R1,*S,*S1,Sn;
   int c,i,k=0,r;

   if((tos-1)->typ!=MAT) {
      stkerr(" logistic_steps: ",MATNOT);
      return 0;
   }
   if(!popd(&n)) return 0;
   n=1+n/100;

   S1=tos->mat;
   r=tos->row;
   c=tos->col;

   if(!matstk(r,c,"_R")) return 0;
   R1=tos->mat;
   memset(R1,0,r*c*sizeof(double));

   for(;k<c;k++) {
      S=S1+locvec(k,r);
      R=R1+locvec(k,r);
      for(i=0;i<r;i++) {
         Sn=*S*n;
         while(*S<=Sn && i<r) {
            i++;
            S++;
            R++;
         }
         if(i<r && *S>Sn) *R=1;
      }
   }
   return(lop());
}

int look() /* look (hXYt hX --- hY) */
{
   return(lerp1(LOOK,NEARBELOW));
}

int looking() /* looking (hA hR --- hB) */
/* Fri Apr 17 14:51:22 PDT 2009

   Fri May  4 06:42:32 PDT 2012.  Copy incoming R to temporary, since
      first R will be set to nonzero and R might be in the catalog.

   Sun Feb 12 17:39:40 PST 2012.  Set first R to nonzero to match
      original word looking stored in the appendix of file math.v.
      Otherwise, in real time simulations past data can receive
      future values.

   Sun Aug 29 10:47:33 PDT 2010.  Incoming A is not restricted to a 
      vector.  Operations with R are repeated on all columns of A
      to produce columns of B.

   Where R(k) equals 0, set B(k,j)=A(m,j) coinciding with closest
   previous R(m) not equal to 0, m<k.

   This function is intended to work exactly the same as word looking,
   which is stored in the appendix of file math.v. */
{
   register double *A,*A1,*B,*B1,*R,*R1;
   double Am;
   register int i;
   int cols,j=0,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" looking: ",MATNOT2);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row) {
      stkerr(" looking: ",ROWSNOT);
      return 0;
   }
   cop(); /* copy, as R will be changed and might be in the catalog */
   A1=(tos-1)->mat;
   cols=(tos-1)->col;
   R1=tos->mat;
   *R1=1; /* force first R to be nonzero */
   
   if(!matstk(rows,cols,"_B")) return 0;
   B1=tos->mat;

   for(;j<cols;j++) {
      A=A1+locvec(j,rows);
      B=B1+locvec(j,rows);

      Am=*A;
      *B=Am;

      R=R1;
    
      for(i=1;i<rows;i++) {
         A++;
         R++;
         B++; 

         if(*R!=0) Am=*A;
         *B=Am;
      }
   }
   return(lop() && lop());
}

/*
This version of looking() was made part of the lerp1() set of 
routines, but that is very inefficient because it unnecessarily 
uses binary search.

int looking() // looking  (hV hR --- hV1) //
// Where R(k) equals 0, set V1(k)=V(j) coinciding with closest
   previous R(j) not equal to 0, j<k.

   This word forces R(1st)=true.  Otherwise, in real time simula-
   tions past data can receive future values.

   Note: the convention for R is opposite the one for word looking
   in program express. 

   This function is intended to work exactly the same as word looking,
   which is stored in the appendix of file math.v. //
{
   register double *R,*V,*Xk,*Yk;
   double *X;
   register int i,k=0,l;
   int rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" looking: ",MATNOT2);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row) {
      stkerr(" looking: ",ROWSNOT);
      return 0;
   }
   V=(tos-1)->mat;
   R=1+tos->mat;
   
   if(!matstk(rows,2,"_hXY")) return 0;
   X=tos->mat;
   Xk=X;
   
   // Always taking the first value: //
   *Xk=XBASE;
   Xk++;
   k++;

   for(i=1;i<rows;i++) {
      if(*R!=0) {
         *Xk=XBASE+i;
         Xk++;
         k++;
      }
      R++;
   }
// By forcing the first value, this code can't be reached.
   if(Xk==X) {
      return(
         drop() &&
         drop2() &&
         pushint(rows) &&
         pushint(0) &&
         null()
      );
   } //

   Yk=Xk;
   tos->row=k;
   Xk=X;

   for(i=0;i<k;i++) {
      l=*Xk-XBASE;
      *Yk=*(V+l);
      Xk++;
      Yk++;
   }
   lop();
   lop();

   if(!matstk(rows,1,"_hx")) return 0;
   Xk=tos->mat;
   for(i=XBASE;i<rows+XBASE;i++) {
      *Xk=i;
      Xk++;
   }
   return(lerp1(LOOK,NEARBELOW)); // running word look //
}
*/

int looking1() /* looking1 (hA hR --- hB) */
/* Tue Jul 10 04:13:29 PDT 2012

   Works like looking(), but incoming A is restricted to a vector and 
   incoming R can be a matrix:
      where R(k,j) equals 0, set B(k,j)=A(m) coinciding with closest
      previous R(m,j) not equal to 0, m<k. */
{
   register double *A,*A1,*B,*B1,*R,*R1;
   double Am;
   register int i;
   int cols,j=0,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" looking1: ",MATNOT2);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row) {
      stkerr(" looking1: ",ROWSNOT);
      return 0;
   }
   A1=(tos-1)->mat;

   cop(); /* copy, as R will be changed and might be in the catalog */
   R1=tos->mat;
   cols=tos->col;

/* Force first R to be nonzero: */
   for(;j<cols;j++) *(R1+locvec(j,rows))=1;
   
   if(!matstk(rows,cols,"_B")) return 0;
   B1=tos->mat;

   for(j=0;j<cols;j++) {
      A=A1;
      B=B1+locvec(j,rows);
      R=R1+locvec(j,rows);

      Am=*A;
      *B=Am;
    
      for(i=1;i<rows;i++) {
         A++;
         R++;
         B++; 

         if(*R!=0) Am=*A;
         *B=Am;
      }
   }
   return(lop() && lop());
}

/* Obsolete.  Replaced by the version above, which works differently.
int looking1() // looking1  (hV hR r --- hV1) //
// Wed Apr 22 11:54:21 PDT 2009

   Where R(k) equals 0, set V1(k)=V(j) coinciding with closest
   previous R(j) not equal to 0, j<k.

   This word works exactly like word looking, but V1 has only r rows 
   reflecting the last r rows processed.

   This word is used for logical purpose, and has no speed advantage
   over using all rows. //
{
   register double *R,*V,*V1;
   double Vj;
   register int i;
   int r,r0,rows;

   if(!popint(&r)) return 0;
   r=MAX(r,0);

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" looking: ",MATNOT2);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row) {
      stkerr(" looking: ",ROWSNOT);
      return 0;
   }
   r=MIN(r,rows);
   r0=rows-r;

   V=(tos-1)->mat;
   R=tos->mat;
   
   if(!matstk(r,1,"_V1")) return 0;
   V1=tos->mat;

   Vj=*V;
   *V1=Vj;
   if(r0>0) V1--;

   for(i=1;i<rows;i++) {
      V++;
      R++;

      if(*R!=0) Vj=*V;

      if(i>=r0) {
         V1++; 
         *V1=Vj;
      }
   }
   return(lop() && lop());
}
*/

int looking2() /* looking2 (hV hU hR --- hW) */
/* Mon Feb 24 10:36:34 PST 2014

   Where R(k) is equal to 0, W(k) equals U(k) and where R(k) is not 
   equal to 0, W(k) equals V(k).

   This function replaces high level version looking2() in sys/mat.v.

   Testing: paste the following five lines at the ready prompt:

      101 5 items 301 5 items park "V" book \
      201 5 items 401 5 items park "U" book \
      list: 1 0 1 0 1 ; "R" book \
      R V U 3 parkn nl .m nl \
      V U R looking2 nl .m nl 

      Results are correct by inspection and agree with the version of
      looking2() in sys/mat.v:

                       R      V(1)     V(2)     U(1)     U(2)
         Row 1:        1      101      301      201      401
         Row 2:        0      102      302      202      402
         Row 3:        1      103      303      203      403
         Row 4:        0      104      304      204      404
         Row 5:        1      105      305      205      405

                     W(1)     W(2)
         Row 1:      101      301
         Row 2:      202      402
         Row 3:      103      303
         Row 4:      204      404
         Row 5:      105      305 */
{
   register double *R,*U,*V,*W;
   register int k;
   int cols,j=0,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" looking2: ",MATNOT3);
      return 0;
   }
   if((rows=(tos-1)->row)!=(tos-2)->row) {
      stkerr(" looking2: ",ROWSNOT);
      return 0;
   }
   if((cols=(tos-1)->col)!=(tos-2)->col) {
      stkerr(" looking2: ",COLSNOT);
      return 0;
   }
   U=(tos-1)->mat;
   V=(tos-2)->mat;
   
   if(!matstk(rows,cols,"_W")) return 0;
   W=tos->mat;

   for(;j<cols;j++) {
      R=(tos-1)->mat;
      for(k=0;k<rows;k++) {
         if(*R) *W=*V; else *W=*U;
         R++;
         U++;
         V++;
         W++;
      }
   }
   return(lop() && lop() && lop());
}

int lt() { return(byterm2(LT)); } /* < (hA hB --- hF) */

double _lt(double x, double y) { return(xTRUE*(x<y)); }
 
int mag() /* mag (hA --- hV) */
/* Magnitude of column k of A into row k of vector V. */
{
   register double *A,s,*V;
   register int i=0,j=0,k=0;
   int rows,cols;

   if(tos->typ!=MAT) {
      stkerr(" mag: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;

   if(!matstk(cols,1,"_mag")) return 0;

   V=tos->mat;
   A=(tos-1)->mat;

   for(;j<cols;j++) {
      s=0;
      for(i=0;i<rows;i++) {
         s=*(A+k)*(*(A+k))+s;
         k++;
      }
      *(V+j)=sqrt(s);
   }
   return(lop());
}

int mathinit()
{
   if(TRACE) {
      gprintf(" initializing math"); nc();
   }

   INF=_or(1,2);
   UNDEF=-INF;
   VQSORT=0;

/* Term-by-term math operations: */
   fbyterm1[ABS1].fexe=(double (*)()) _abs;
   fbyterm1[ABS1].fnam="_abs";

   fbyterm2[ATAN2].fexe=(double (*)()) _atan2;
   fbyterm2[ATAN2].fnam="_atan2";

   fbyterm1[COS].fexe=(double (*)()) _cos;
   fbyterm1[COS].fnam="_cos";

   fbyterm1[COSH].fexe=(double (*)()) _cosh;
   fbyterm1[COSH].fnam="_cosh";

   fbyterm1[EXP].fexe=(double (*)()) _exp;
   fbyterm1[EXP].fnam="_exp";

   fbyterm1[FIVEPOW].fexe=(double (*)()) _fivepow;
   fbyterm1[FIVEPOW].fnam="_fivepow";

   fbyterm1[INT].fexe=(double (*)()) _int;
   fbyterm1[INT].fnam="_int";

   fbyterm2[MAX1].fexe=(double (*)()) _max;
   fbyterm2[MAX1].fnam="_max";

   fbyterm2[MIN1].fexe=(double (*)()) _min;
   fbyterm2[MIN1].fnam="_min";

   fbyterm1[INT2].fexe=(double (*)()) _int2;
   fbyterm1[INT2].fnam="_int2";

   fbyterm1[INT4].fexe=(double (*)()) _int4;
   fbyterm1[INT4].fnam="_int4";

   fbyterm1[INT8].fexe=(double (*)()) _int8x;
   fbyterm1[INT8].fnam="_int8";

   fbyterm1[LN].fexe=(double (*)()) _ln;
   fbyterm1[LN].fnam="_ln";

   fbyterm1[LOG2].fexe=(double (*)()) _log2;
   fbyterm1[LOG2].fnam="_log2";

   fbyterm1[LOG5].fexe=(double (*)()) _log5;
   fbyterm1[LOG5].fnam="_log5";

   fbyterm1[LOG10].fexe=(double (*)()) _log10;
   fbyterm1[LOG10].fnam="_log10";

   fbyterm1[REAL2].fexe=(double (*)()) _real2;
   fbyterm1[REAL2].fnam="_2real";

   fbyterm1[REAL4].fexe=(double (*)()) _real4;
   fbyterm1[REAL4].fnam="_4real";

   fbyterm1[SIN].fexe=(double (*)()) _sin;
   fbyterm1[SIN].fnam="_sin";

   fbyterm1[SINH].fexe=(double (*)()) _sinh;
   fbyterm1[SINH].fnam="_sinh";

   fbyterm1[TAN].fexe=(double (*)()) _tan;
   fbyterm1[TAN].fnam="_tan";

   fbyterm1[TANH].fexe=(double (*)()) _tanh;
   fbyterm1[TANH].fnam="_tanh";

   fbyterm1[TENPOW].fexe=(double (*)()) _tenpow;
   fbyterm1[TENPOW].fnam="_tenpow";

   fbyterm1[TWOPOW].fexe=(double (*)()) _twopow;
   fbyterm1[TWOPOW].fnam="_twopow";

   fbyterm1[UINT2].fexe=(double (*)()) _int2;
   fbyterm1[UINT2].fnam="_uint2";

   fbyterm1[UINT4].fexe=(double (*)()) _int4;
   fbyterm1[UINT4].fnam="_uint4";

   fbyterm1[UINT8].fexe=(double (*)()) _int8x;
   fbyterm1[UINT8].fnam="_uint8";

   fbyterm1[UREAL2].fexe=(double (*)()) _ureal2;
   fbyterm1[UREAL2].fnam="_u2real";

   fbyterm1[UREAL4].fexe=(double (*)()) _ureal4;
   fbyterm1[UREAL4].fnam="_u4real";

/* Term-by-term bitwise operations: */
   fbyterm2[AND].fexe=(double (*)()) _and;
   fbyterm2[AND].fnam="_and";

   fbyterm2[OR].fexe=(double (*)()) _or;
   fbyterm2[OR].fnam="_or";

   fbyterm2[XOR].fexe=(double (*)()) _xor;
   fbyterm2[XOR].fnam="_xor";

   fbyterm1[NOTT].fexe=(double (*)()) _nott;
   fbyterm1[NOTT].fnam="_nott";

   fbyterm1x[NOTT].fexe=(double* (*)()) _nottx;
   fbyterm1x[NOTT].fnam="_nottx";

/* Term-by-term logical operations: */
   fbyterm1[NOT].fexe=(double (*)()) _not;
   fbyterm1[NOT].fnam="_not";

/* Term-by-term relational operations: */
   fbyterm2[EQ].fexe=(double (*)()) _eq;
   fbyterm2[EQ].fnam="_eq";

   fbyterm2[GE].fexe=(double (*)()) _ge;
   fbyterm2[GE].fnam="_ge";

   fbyterm2[GT].fexe=(double (*)()) _gt;
   fbyterm2[GT].fnam="_gt";

   fbyterm2[LE].fexe=(double (*)()) _le;
   fbyterm2[LE].fnam="_le";

   fbyterm2[LT].fexe=(double (*)()) _lt;
   fbyterm2[LT].fnam="_lt";

   fbyterm2[NE].fexe=(double (*)()) _ne;
   fbyterm2[NE].fnam="_ne";

   fbyterm1[EQ0].fexe=(double (*)()) _zeroeq;
   fbyterm1[EQ0].fnam="_zeroeq";

   fbyterm1[GT0].fexe=(double (*)()) _zerogt;
   fbyterm1[GT0].fnam="_zerogt";

   fbyterm1[LT0].fexe=(double (*)()) _zerolt;
   fbyterm1[LT0].fnam="_zerolt";

   fbyterm1[NE0].fexe=(double (*)()) _zerone;
   fbyterm1[NE0].fnam="_zerone";

   return(sign1(&SIGNBIT));
}

double _max(double x, double y) { return(MAX(x,y)); }

int max1() { return(byterm2(MAX1)); } /* max (hA hB --- hC) */

int max2() /* max1 (hA --- hB) */
/* B(i,1) holds max A(i,J) across all columns j of A; B(i,2) holds J. */
{
   register double *A,Amax,*B;
   register int cmax=0,i=0,j;
   int cols,rows;

   if(tos->typ!=MAT) {
      stkerr(" max1: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   cols=tos->col;

   if(!matstk((rows=tos->row),2,"_max1")) return 0;
   B=tos->mat;

   for(;i<rows;i++) {
      Amax=-INF;
      for(j=0;j<cols;j++) {
         if(*(A+locvec(j,rows)+i)>Amax) {
            Amax=*(A+locvec(j,rows)+i);
            cmax=j;
         }
      }
      *(B+i)=Amax;
      *(B+i+rows)=cmax+XBASE;
   }
   return(lop());
}

int maxcompress() /* maxcompress (hA ht ht1 --- hB) */
/* Tue Jan 10 21:53:14 PST 2012  

   Map A(t) to B(t1), using a max criterion to select which nearby A(t)
   goes into B(t1).

   When rows of t are greater than rows of t1, B is a compressed ver-
   sion of A that preserves its higher values.

   B(1)=A(1) and B(t1)=0 at times in t1 that are greater than last t.

   Test case:

      list: 0 0   497 323 121 167 39  78  7   125 133 ; "A" book
      list: 0 46  106 162 219 283 338 403 456 526 569 ; "t" book
      list: 0 180 360 540 720 ; "t1" book

      [tops@plunger] ready > A t t1 maxcompress .m
       Row 1:        0
       Row 2:      497
       Row 3:      167
       Row 4:      125
       Row 5:        0 */
{
   double *A,Amax,*B,*t,*t1;
   int i=0,k=0,r,r1;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" maxcompress: ",MATNOT3);
      return 0;
   }
   if((r=(tos-1)->row)!=(tos-2)->row) {
      stkerr(" maxcompress: ",ROWSNOT);
      return 0;
   }
   r1=tos->row;

   A=(tos-2)->mat;
   t=(tos-1)->mat;
   t1=tos->mat;

   if(!matstk(r1,1,"_maxcompress")) return 0;
   B=tos->mat;
   memset(B,0,r1*sizeof(double));

   Amax=*B=*A;
   B++;
   t1++;
   k++;

   while(i<r && k<r1) {
      if(*t>=*t1) {
         *B=Amax;
         B++;
         t1++;
         k++;
         Amax=-INF;
      }
      Amax=MAX(Amax,*A);
      A++;
      t++;
      i++;
   }
   return(lop() && lop() && lop());
}

int maxfetch() /* maxfetch (hA --- x i j) */
/* Fetching largest term in matrix. */
{
   register double *A,x;
   register int k=1,kat,len;
   int i,j,rows;

   hand();

   if(is_sparse(tos)) dense();

   if(tos->typ!=MAT) {
      stkerr(" maxfetch: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   rows=tos->row;
   len=rows*tos->col;

   if(!len) {
      stkerr(" maxfetch: ",MATPURG);
      return 0;
   }
   x=*A;
   kat=0;

   for(;k<len;k++) {
      if(*(A+k)>x) {
         x=*(A+k);
         kat=k;
      }
   }
   j=kat/rows;
   i=kat-j*rows;

   return(
      drop() &&
      push(NUM,NULL,NOTAG,x,NULL,NULL,0,0,NULL) &&
      push(NUM,NULL,NOTAG,(double)(i+XBASE),NULL,NULL,0,0,NULL) &&
      push(NUM,NULL,NOTAG,(double)(j+XBASE),NULL,NULL,0,0,NULL)
   );
}

double _min(double x, double y) { return(MIN(x,y)); }

int min1() { return(byterm2(MIN1)); } /* min (hA hB --- hC) */

int min2() /* min1 (hA --- hB) */
/* B(i,1) holds min A(i,J) across all columns j of A; B(i,2) holds J. */
{
   register double *A,Amin,*B;
   register int cmin=0,i=0,j;
   int cols,rows;

   if(tos->typ!=MAT) {
      stkerr(" min1: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   cols=tos->col;

   if(!matstk((rows=tos->row),2,"_min1")) return 0;
   B=tos->mat;

   for(;i<rows;i++) {
      Amin=INF;
      for(j=0;j<cols;j++) {
         if(*(A+locvec(j,rows)+i)<Amin) {
            Amin=*(A+locvec(j,rows)+i);
            cmin=j;
         }
      }
      *(B+i)=Amin;
      *(B+i+rows)=cmin+XBASE;
   }
   return(lop());
}

int minfetch() /* minfetch (hA --- x i j) */
/* Fetching smallest term in matrix. */
{
   register double *A,x;
   register int k=1,kat,len;
   int i,j,rows;

   hand();

   if(is_sparse(tos)) dense();

   if(tos->typ!=MAT) {
      stkerr(" minfetch: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   rows=tos->row;
   len=rows*tos->col;

   if(!len) {
      stkerr(" minfetch: ",MATPURG);
      return 0;
   }
   x=*A;
   kat=0;

   for(;k<len;k++) {
      if(*(A+k)<x) {
         x=*(A+k);
         kat=k;
      }
   }
   j=kat/rows;
   i=kat-j*rows;

   return(
      drop() &&
      push(NUM,NULL,NOTAG,x,NULL,NULL,0,0,NULL) &&
      push(NUM,NULL,NOTAG,(double)(i+XBASE),NULL,NULL,0,0,NULL) &&
      push(NUM,NULL,NOTAG,(double)(j+XBASE),NULL,NULL,0,0,NULL)
   );
}

int minus() /* - (x y --- x-y) or (hA hB --- hC) */
/* Driver for subtraction. 

   For the nine permutations of the three types of stack items for
   subtraction: MAT, NUM, VOL:SPARSE. */
{
   int f=0;

   if(stkdepth()<2) {
      stkerr(" minus: ",NEEDTWO);
      return 0;
   }
   switch((tos-1)->typ) {

      case MAT: /* dense */
         switch(tos->typ) {
            case MAT: /* Have two MATs.  Is one a scalar? */
               if((f=is_scalar(tos-1)) || is_scalar(tos)) {
                  if(f) swap();
                  if(is_complex(tos))
                     pushdx(*tos->mat,*(tos->mat+1));
                  else pushd(*tos->mat);
                  lop();
                  if(f) { /* -dense + scalar */
                     return(
                        swap() &&
                        negate() &&
                        swap() &&
                        plusd() 
                     );
                  }
                  else return(minusd()); /* dense - scalar */
               }
               return(minusm()); /* dense - dense */
            case NUM:
               return(minusd()); /* dense - scalar */
            case VOL:
               return(
                  swap() &&
                  sparse() &&
                  swap() &&
                  negate() &&
                  spadd() /* dense - sparse */
               );
         }
      case NUM: /* scalar */
         switch(tos->typ) {
            case MAT:
               return( 
                  negate() &&
                  swap() &&
                  plusd() /* scalar - dense */
               );
            case NUM:
               return(minusn()); /* scalar - scalar */
            case VOL:
               return(
                  dense() &&
                  negate() &&
                  swap() &&
                  plusd() /* scalar - sparse */
               );
         }
      case VOL: /* sparse */
         switch(tos->typ) {
            case MAT:
               return( 
                  negate() &&
                  sparse() &&
                  spadd() /* sparse - dense */
               );
            case NUM:
               return(
                  swap() &&
                  dense() &&
                  swap() &&
                  minusd() /* sparse - scalar */
               );
            case VOL:
               return(
                  negate() &&
                  spadd() /* sparse - sparse */
               );
         }
   }
   stkerr(" minus: ",STKNOT);
   return 0;
}

int minusd() /* -d (hA d --- hB) B(i,j)=A(i,j)-d */
/* Deduct offset d from all terms in matrix A. */
{
   register double *A,*B;
   double xi,xr;
   register int rA,cA,rB,k=0;
   int nword=1;
  
   if((tos-1)->typ!=MAT) {
      stkerr(" minusd: ",MATNOT); return 0;
   }
   nword+=is_complex(tos);
   if(!popdx(&xr,&xi)) return 0;

   nword=MAX(nword,(k=(1+is_complex(tos))));

   A=tos->mat;
   rA=(tos->row)/k;
   cA=tos->col;

   rB=nword*rA;
   if(!matstk(rB,cA,"_minusd")) return 0;
   B=tos->mat;

   if(nword==2) { /* doing complex because A and/or d is complex */
      set_complex(tos);

      if(k==2) { /* subtracting d from complex A to make complex B */
         for(k=0;k<rB*cA;k+=nword) {
            *B=*A-xr;
            B++;
            A++;
            *B=*A-xi;
            B++;
            A++;
         }
      }
      else { /* subtracting d from real A to make complex B */
         for(k=0;k<rB*cA;k+=nword) {
            *B=*A-xr;
            B++;
            A++;
            *B=-xi;
            B++;
         }
      }
   }
   else for(k=0;k<rA*cA;k++) *(B+k)=*(A+k)-xr;

   return(lop());
}

int minusm() /* - (hA hB --- hC) \ C=A-B */
/* Matrix subtraction. */
{
   stkitem *Amat,*Bmat;
   register double *A,*B,*C;
   register int rA,cA,k=0;
   int TAG;

   if(!cmplxmatch()) return 0;

   Amat=tos-1; Bmat=tos;
   A=Amat->mat; rA=Amat->row; cA=Amat->col;
   B=Bmat->mat; 
   if(rA!=Bmat->row || cA!=Bmat->col) {
      stkerr(" minus: ",MATSNOTC); return 0;
   }
   if((C=(double *)memget(rA,cA))==NULL) {
      stkerr(" minus: ",MEMNOT); return 0;
   }
   TAG=tos->tag;

   for(;k<rA*cA;k++) *(C+k)=*(A+k)-(*(B+k));
   drop2();
   return(push(MAT,(char *)memgetn("_minus",6),TAG,0,C,NULL, \
      rA,cA,NULL));
}

int minusn() /* - (x y --- x-y) */
/* Subtracting numbers. */
{
   double xi,xr,yi,yr;

   if(is_complex(tos) || is_complex(tos-1)) {
      popdx(&yr,&yi);
      popdx(&xr,&xi);
      return(pushdx(xr-yr,xi-yi));
   }
   yr=pop()->real;
   return(push(NUM,NULL,NOTAG,(pop()->real)-yr,NULL,NULL,0,0,NULL));
}

int mod() /* mod (x y --- rem) */
/* Remainder for division of x by y; x and y are real numbers.

   Fri May  3 13:21:54 PDT 2013.  If x is a MAT and y is a NUM, create
   a MAT from y for term-by-term operation. */
{
   register double *X,*Y,*R,r,x,y;
   register int k=0;
   long long q;
   double x1,y1;
   int cols,len,rows;

   if(is_complex(tos) || is_complex(tos-1)) {
      stkerr(" mod: ",REALNOT);
      return 0;
   }
   if(tos->typ==NUM && (tos-1)->typ==NUM) {
      if(!popd(&y1) || !popd(&x1)) return 0;
      q=x1/y1;
      r=x1-q*y1;
      return(pushd(r));
   }
   if(tos->typ==NUM) { /* make NUM y into a MAT */
      over();
      dims();
      fill();
   }
   if(tos->typ==MAT && (tos-1)->typ==MAT) {
      if(tos->row!=(tos-1)->row || tos->col!=(tos-1)->col) {
         stkerr(" mod: ",STKNOTC);
         return 0;
      }
      X=(tos-1)->mat;
      Y=tos->mat;

      rows=tos->row;
      cols=tos->col;

      if(!matstk(rows,cols,"_rem")) return 0;
      R=tos->mat;

      len=rows*cols;
      for(;k<len;k++) {
         y=*(Y+k);
         x=*(X+k);
         q=x/y;
         *(R+k)=x-q*y;
      }
      return(lop() && lop());
   }
   stkerr(" mod: ",STKNOT);
   return 0; 
}

int mod1() /* % (x y --- mod) */
/* Modulus: remainder for division of real or complex numbers. */
{
   return(
      slashby() &&
      dup1s() &&
      integer() &&
      minus()
   );
}

int mpy() /* mpy (hA hB --- hC) \ C=A*B */
/* Real matrix multiplication.  Identical to starm() except no bounds 
   check or mem alloc check. */
{
   char *name="_mpy";
   const int nlen=strlen(name);
   char *p;
   int rA,cA,cB;
   register double *A,*B,*C;

#ifdef LAPACK

   char *TRANSA="N",*TRANSB="N";
   double alpha=1,beta=0;

   A=(tos-1)->mat;
   rA=(tos-1)->row;
   cA=(tos-1)->col;

   B=tos->mat;
   cB=tos->col;

   C=(double *)memget(rA,cB);

   DGEMM(TRANSA, TRANSB, &rA, &cB, &cA, &alpha, A, &rA, B, &cA, \
      &beta, C, &rA);

#else

   stkitem *Amat,*Bmat;
   register double Bj;
   register int i,j,k=0;
   register int iC1,iCn,jB1,jBn,jA1,iA;

   Amat=tos-1; Bmat=tos;
   A=Amat->mat; cA=Amat->col; rA=Amat->row;
   B=Bmat->mat; cB=Bmat->col;

   C=(double *)memget0(rA,cB);

   MPYLOOP

#endif

   drop2();

   p=memgetn(name,nlen);

   return(push(MAT,p,NOTAG,0,C,NULL,rA,cB,NULL));
}

int mpyad() /* mpyad (hD hA hB --- hC) \ C=D+A*B */
/* Real Matrix multiply and add.  No compatibility checks or bounds 
   checks are made. */
{
   char *name="_mpyad";
   const int nlen=strlen(name);
   char *p;
   int rA,cA,cB;
   register double *A,*B,*C;

#ifdef LAPACK

   char *TRANSA="N",*TRANSB="N";
   double alpha=1;

   A=(tos-1)->mat;
   rA=(tos-1)->row;
   cA=(tos-1)->col;

   B=tos->mat;
   cB=tos->col;

   C=(double *)memget(rA,cB);
   memcpy(C,(tos-2)->mat,rA*cB*sizeof(double));

   DGEMM(TRANSA, TRANSB, &rA, &cB, &cA, &alpha, A, &rA, B, &cA, \
      &alpha, C, &rA);

#else

   stkitem *Amat,*Bmat;
   register double Bj;
   register int i,j,k=0;
   register int iC1,iCn,jB1,jBn,jA1,iA;

   Amat=tos-1; Bmat=tos;
   A=Amat->mat; cA=Amat->col; rA=Amat->row;
   B=Bmat->mat; cB=Bmat->col;

   C=(double *)memget(rA,cB);
   memcpy(C,(tos-2)->mat,rA*cB*sizeof(double));

   MPYLOOP

#endif

   drop2();
   drop();

   p=memgetn(name,nlen);

   return(push(MAT,p,NOTAG,0,C,NULL,rA,cB,NULL));
}

void mpyad1(double *D, double *Ain, double *Bin, int rowA, int colA, \
   int colB, double *Cout)
/* Matrix multiply and add: C=D+A*B.  No checks of compatibility or
   bounds are made. */
{
   register double *A,*B,*C,Bj;
   register int rA,cA,cB;
   register int i,j,k=0;
   register int iC1,iCn,jB1,jBn,jA1,iA;

   A=Ain;
   B=Bin;
   C=Cout;

   cA=colA;
   rA=rowA;
   cB=colB;
   memcpy(C,D,rA*cB*sizeof(double));

   MPYLOOP
}

int mpydg() /* mpydg (hA hB --- hC) \ C=diag(A*B) */
/* Column vector C holds the diagonal of A*B, where the number of rows
   of A matches the number of columns of B. */
{
   double *A,*B,*C;
   int nw=1,rA,cA,rB,cB,i=0,j,k;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mpydg: ",MATNOT2);
      return 0;
   }
   cmplxmatch();
   if(is_complex(tos)) nw=2;

   A=(tos-1)->mat;
   rA=(tos-1)->row;
   cA=(tos-1)->col;

   B=tos->mat;
   rB=tos->row;
   cB=tos->col;

   if(rA!=nw*cB || nw*cA!=rB) {
      stkerr(" mpydg: ",MATSNOTC);
      return 0;
   }
   if(!matstk(rA,1,"_mpydg")) return 0;
   C=tos->mat;
   tos->tag=(tos-1)->tag;

   if(nw==1) {

      for(;i<rA;i++) {
         *C=0;
         for(j=0;j<cA;j++) *C+=*(A+locvec(j,rA))*(*(B+j));
         A++;
         B+=rB;
         C++;
      }
   }
   else { /* complex case, nw=2 */

      for(;i<rA;i+=2) {
         *C=0;
         *(C+1)=0;

         for(j=0;j<rB;j+=2) {
          k=locvec(j/2,rA);

         /*
          A = u + iv
          B = x + iy
          C = A*B = (u*x - v*y) + i(u*y + v*x)

          Here, *(A+k)=u, *(B+j)=x: */
          *C+=(*(A+k)*(*(B+j)) - *(A+k+1)*(*(B+j+1))); /* u*x - v*y */
          *(C+1)+=(*(A+k)*(*(B+j+1)) + *(A+k+1)*(*(B+j))); /* u*y+v*x */
         }
         A++;
         A++;
         B+=rB;
         C++;
         C++;
      }
   }
   return(lop() && lop());
}

double _ne(double x, double y) { return(xTRUE*(x!=y)); }

int ne() /* <> (hA hB --- hF) */
{
   int f;

   if(is_sparse(tos)) dense();
   if(is_sparse(tos-1)) {
      swap();
      dense();
      swap();
   }
   if((tos->typ==NUM || tos->typ==MAT) &&
     ((tos-1)->typ==NUM || (tos-1)->typ==MAT)) return(byterm2(NE));

/* Doing characters: */
   if(!strmatch1()) return 0;

   popint(&f);
   if(f==0) return(pushint(xFALSE));
   return(pushint(xTRUE));
}

int negate() /* negate (x --- -x) or (hA --- hB) */
/* Flipping sign of number or signs of all terms in matrix. */
{
   register double *A,*B,v;
   double xi,xr;
   register int k=0,TAG;
   int typ,rA,cA;
   
   if(is_sparse(tos)) return(pushint(-1) && spscale());

   if((typ=tos->typ)==NUM) {
      if(is_complex(tos)) {
            popdx(&xr,&xi);
            if(xr) xr=-xr; /* not setting the sign bit on zero */
            if(xi) xi=-xi;
            return(pushdx(xr,xi));
      }
      else {
         v=pop()->real;
         if(v) v=-v; /* not setting the sign bit on zero */
         return(pushd(v));
      }
   }
   if(typ==MAT) {
      if((B=(double *)memget0((rA=tos->row),(cA=tos->col)))==NULL) {
         stkerr(" negate: ",MEMNOT); return 0;
      }
      A=tos->mat;
      TAG=tos->tag;

      for(;k<rA*cA;k++) {
         if((v=*(A+k))) *(B+k)=-v;
      }
      return(
         drop() &&
         push(MAT,(char *)memgetn("_negate",7),TAG,0,B,NULL,rA,cA,NULL)
      );
   }
   stkerr(" negate: ",NUMORMATNOT);
   return 0;
}

int nit() /* nit (x --- x-1) */
{
   if(tos->typ==NUM) {
      tos->real--;
      tos->nam=NULL; /* wipe out lib name if any */
      return 1;
   }
   pushint(1);
   return(minus());
}

int norm() /* norm (hA --- hB) */
/* Normalize the columns of A to unity magnitude. */
{
   register double a,*A,*B,d;
   register int i=0,j=0,k=0,n=0;
   int rows,cols;

   if(tos->typ!=MAT) {
      stkerr(" norm: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;

   if(!matstk(rows,cols,"_norm")) return 0;

   B=tos->mat;
   A=(tos-1)->mat;

   for(;j<cols;j++) {
      d=0;
      for(i=0;i<rows;i++) {
         a=*(A+k);
         d=d+a*a;
         k++;
      }
      if(d) {
         d=sqrt(1/d);
         for(i=0;i<rows;i++) {
            *(B+n)=*(A+n)*d;
            n++;
         }
      }
      else {
         for(i=0;i<rows;i++) {
            *(B+n)=0;
            n++;
         }
      }
   }
   return(lop());
}

int not() { return(byterm1(NOT)); } /* not (hA --- hB) */

double _not(double x) { return(_xor(x,xTRUE)); }

int nott() { return(byterm1(NOTT)); } /* nott (hA --- hB) */

double _nott(double x)
{  
   X.x=x; 
   X.c[0]=~X.c[0]; 
   X.c[1]=~X.c[1];

   return(X.x);
}

int nottx() { return(byterm1x(NOTT)); } /* nott (hA --- hB) */

double *_nottx(double x)
{  
   X.x=x; 
   X.c[0]=~X.c[0]; 
   X.c[1]=~X.c[1];

   return(&X.x);
}

int nrcrv() /* nrcrv (hV hA --- hV1 hK) */
/* Each column of A holds a curve.  V1(i) holds the value A(i,k) of
   curve k that is nearest to V(i), and K holds the index k. */
{
   double *A,D,*K,*V,*V1;
   int cols,i=0,j,k,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" nrcrv: ",MATNOT2);
      return 0;
   }
   A=tos->mat;
   V=(tos-1)->mat;

   rows=MIN(tos->row,(tos-1)->row);
   cols=tos->col;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=tos->mat;

   if(!matstk(rows,1,"_K")) return 0;
   K=tos->mat;

   for(;i<rows;i++) {
      D=fabs(*A-*V);
      k=0;
      for(j=1;j<cols;j++) {
         if(fabs(*(A+j*rows)-*V)<D) {
            D=fabs(*(A+j*rows)-*V);
            k=j;
         }
      }
      *V1=*(A+k*rows);
      *K=k+XBASE;
      A++;
      V++;
      V1++;
      K++;
   }
   return(rot() && drop() && rot() && drop());
}

int nrcrv2() /* nrcrv2 (hV hA --- hV1 hK) */
/* Sun Aug  5 07:44:23 PDT 2012

   Each column of A holds a curve.  V1(i) holds the value A(i,k) of
   curve k that is the one nearest-above V(i), and K holds the index
   k.  If V(i) is above the highest curve, V(i) equals its value.

   To run the following tests, paste the following at the ready prompt:

      syspath "../src/math.c" + "Testing nrcrv2:" msource

   Testing nrcrv2:
    \ Sun Aug  5 12:53:13 PDT 2012

      list: 0 2 4 ; list: 1 3 5 ; list: 2 4 6 ; 3 parkn "A" book
      list: 2 3 4 ; "V" book

      V A 2dup nrcrv2 (hV hA hV1 hK) 4 parkn mtext (hT)
      "   Case 1: V on each line:" . nl
      "       V       A(1)     A(2)     A(3)     V1        K" . nl
      (hT) . nl    

      list: 1.5 2.5 3.5 ; "V" book
      V A 2dup nrcrv2 (hV hA hV1 hK) 4 parkn mtext (hT)
      "   Case 2: V between each line:" . nl
      "       V       A(1)     A(2)     A(3)     V1        K" . nl
      (hT) . nl

      list: 0 0 0 ; "V" book
      V A 2dup nrcrv2 (hV hA hV1 hK) 4 parkn mtext (hT)
      "   Case 3: V below all lines:" . nl
      "       V       A(1)     A(2)     A(3)     V1        K" . nl
      (hT) . nl

      list: 6 6 6 ; "V" book
      V A 2dup nrcrv2 (hV hA hV1 hK) 4 parkn mtext (hT)
      "   Case 4: V above all lines:" . nl
      "       V       A(1)     A(2)     A(3)     V1        K" . nl
      (hT) . nl

      halt 

   These test results are correct by inspection.  Tests with negative
   values have not been run. */
{
   double *A,D,Dmin,*K,*V,*V1;
   int cols,i=0,j,k,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" nrcrv2: ",MATNOT2);
      return 0;
   }
   A=tos->mat;
   V=(tos-1)->mat;

   rows=MIN(tos->row,(tos-1)->row);
   cols=tos->col;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=tos->mat;

   if(!matstk(rows,1,"_K")) return 0;
   K=tos->mat;

   k=-1;
   for(;i<rows;i++) {
      Dmin=INF;
      for(j=0;j<cols;j++) {
         if((D=(*(A+j*rows)-*V))>=0 && D<=Dmin) {
            Dmin=D;
            k=j;
         }
      }
      if(k==-1) {
         Dmin=*(A+rows);
         k=0;
         for(j=1;j<cols;j++) if(*(A+j*rows)>Dmin) k=j;
      }
      *V1=*(A+k*rows);
      *K=k+XBASE;
      k=-1;
      A++;
      V++;
      V1++;
      K++;
   }
   return(rot() && drop() && rot() && drop());
}

int nzmin1() /* nzmin1 (hA --- hB) */
/* B(i,1) holds non-zero min A(i,J) across all columns j of A; B(i,2) 
   holds J. */
{
   register double *A,Amin,*B,d;
   register int cmin=0,i=0,j;
   int cols,rows;

   if(tos->typ!=MAT) {
      stkerr(" min1: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   cols=tos->col;

   if(!matstk((rows=tos->row),2,"_min1")) return 0;
   B=tos->mat;

   for(;i<rows;i++) {
      Amin=INF;
      cmin=0;
      for(j=0;j<cols;j++) {
         if(*(A+locvec(j,rows)+i)<Amin) {
            if((d=*(A+locvec(j,rows)+i))!=0) {
               Amin=d;
               cmin=j;
            }
         }
      }
      *(B+i)=Amin;
      *(B+i+rows)=cmin+XBASE;
   }
   return(lop());
}

int or() { return(byterm2(OR)); } /* or (hA hB --- hC) */

int or_() /* or (hA hB --- hC) */
/* Version for parser, does "0<> swap 0<> or." */
{ 
   zerone();
   swap();
   zerone();
   return(byterm2(OR)); 
}

double _or(double x, double y)
{  
   X.x=x; Y.y=y; X.c[0]=(X.c[0] | Y.d[0]); X.c[1]=(X.c[1] | Y.d[1]);
   return X.x;
}

int outside(double *x, double xMin, double xMax)
/* If x<xMin, return x=xMin; if x>xMax, return x=xMax. */
{
   if(xMax<*x) { /* xMax<x, so return x=xMax */
      *x=xMax;
      return 1;
   }
   if(*x<xMin) { /* x<xMin, so return x=xMin */
      *x=xMin;
      return 1;
   }
   return 0;
}

int overlay() /* overlay (hA hB --- hC) */
/* C(i,j)=A(i,j) unless A(i,j)=0.  Then C(i,j)=B(i,j). */
{
   double *A,*B,*C;
   int rA,cA,rB,cB,i=0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" overlay: ",MATNOT2);
      return 0;
   }
   A=(tos-1)->mat;
   rA=(tos-1)->row;
   cA=(tos-1)->col;

   B=tos->mat;
   rB=tos->row;
   cB=tos->col;

   if(rA!=rB || cA!=cB) {
      stkerr(" overlay: ",MATSNOTC);
      return 0;
   }
   if(!matstk(rA,cA,"_overlay")) return 0;
   C=tos->mat;

   for(;i<rA*cA;i++) {
      if(*A!=0) *C=*A;
      else *C=*B;
      A++;
      B++;
      C++;
   }
   return(lop() && lop());
}

int partials() /* partials (hA --- hB) */
/* Columns of B contain the partial sum of columns of A: 
           B(i,j) = sum[A(k,j)], k=1,i */
{
   register double *A,*Aj,*B,*Bj;
   register int i,ilim,j=0,jlim;
   int rA,cA;

   if(is_sparse(tos)) dense();

   if(tos->typ!=MAT) {
      stkerr(" partials: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   if((B=(double *)memget0((rA=tos->row),(cA=tos->col)))==NULL) {
      return 0;
   }
   jlim=cA;
   ilim=rA;
   for(;j<jlim;j++) {
      *(B+locvec(j,ilim))=*(A+locvec(j,ilim));
   }
   for(j=0;j<jlim;j++) {

      Aj=A+locvec(j,ilim);
      Bj=B+locvec(j,ilim);

      for(i=1;i<ilim;i++) {
         *(Bj+i)=*(Bj+i-1)+*(Aj+i);
      }
   }
   return(
      drop() &&
      push(MAT,(char *)memgetn("_partials",9),NOTAG,0,B,NULL,rA,cA,NULL)
   );
}

int patint(double x)
/* Converting 8-byte floating point bit pattern into 4-byte integer bit
   pattern. */
{
   union {
      double x;
      int i;
   } X={0};
   X.i=(int)x;
   return(X.i);
}

unsigned long patlong(double x)
/* Converting 8-byte floating point bit pattern into 4-byte unsigned
   long bit pattern. */
{
   union {
      double x;
      unsigned long u;
   } X={0};
   X.u=(unsigned long)x;
   return(X.u);
}

unsigned int patuint(double x)
/* Converting 8-byte floating point bit pattern into 4-byte unsigned
   int bit pattern. */
{  
   union {
      double x; 
      unsigned int u;
   } X={0};
   X.u=(unsigned int)x;
   return(X.u);
}

int plus() /* + (x y --- x+y) */
/* Driver for addition.

   For the nine permutations of the three types of stack items for
   addition: MAT, NUM, VOL:SPARSE. 

   For type STR, perform cat(). */
{
   int f=0;

   if(stkdepth()<2) {
      stkerr(" plus: ",NEEDTWO);
      return 0;
   }
   switch((tos-1)->typ) {

      case MAT: /* dense */
         switch(tos->typ) {
            case MAT: /* Have two MATs.  Is one a scalar? */
               if((f=is_scalar(tos-1)) || is_scalar(tos)) {
                  if(f) swap();
                  if(is_complex(tos)) 
                     pushdx(*tos->mat,*(tos->mat+1));
                  else pushd(*tos->mat);
                  lop();
                  return(plusd()); /* dense + scalar */
               }
               return(plusm()); /* dense + dense */
            case NUM:
               return(plusd()); /* dense + scalar */
            case VOL:
               swap();
               sparse();
               swap();

               return spadd(); /* dense + sparse */
         }
      case NUM: /* scalar */
         switch(tos->typ) {
            case MAT:
               swap();
               return(plusd()); /* scalar + dense */
            case NUM:
               return(plusn()); /* scalar + scalar */
            case VOL:
               dense();
               swap();
               return(plusd()); /* scalar + sparse */
         }
      case VOL: /* sparse */
         if(!is_sparse(tos-1)) return(cat()); /* parking 2 VOLs */

         switch(tos->typ) {
            case MAT:
               sparse();
               return spadd(); /* sparse + dense */
            case NUM:
               swap();
               dense();
               swap();
               return(plusd() && sparse()); /* sparse + scalar */
            case VOL:
               return spadd(); /* sparse + sparse */
         }
      case STR: /* cat two strings */
         return(cat());

   }
   stkerr(" plus: ",STKNOT);
   return 0;
}

int plusd() /* +d (hA d --- hB) B(i,j)=A(i,j)+d */
/* Add offset d to all terms in matrix A. */
{
   register double *A,*B;
   double xi,xr;
   register int rA,cA,rB,k=0;
   int nword=1;
   
   if((tos-1)->typ!=MAT) {
      stkerr(" plusd: ",MATNOT); return 0;
   }
   nword+=is_complex(tos);
   if(!popdx(&xr,&xi)) return 0;

   nword=MAX(nword,(k=(1+is_complex(tos))));

   A=tos->mat; 
   rA=(tos->row)/k;
   cA=tos->col;

   rB=nword*rA;
   if(!matstk(rB,cA,"_plusd")) return 0;
   B=tos->mat;

   if(nword==2) { /* doing complex */
      set_complex(tos);

      if(k==2) { /* adding d to complex A to make complex B */
         for(k=0;k<rB*cA;k+=nword) {
            *B=*A+xr; 
            B++;
            A++;
            *B=*A+xi;
            B++;
            A++;
         }
      }
      else { /* adding d to real A to make complex B */
         for(k=0;k<rB*cA;k+=nword) {
            *B=*A+xr; 
            B++;
            A++;
            *B=xi;
            B++;
         }
      }
   }
   else for(k=0;k<rA*cA;k++) *(B+k)=*(A+k)+xr;

   return(lop());
}

int plusm() /* + (hA hB --- hC) \ C=A+B */
/* Matrix addition. */
{
   stkitem *Amat,*Bmat;
   register double *A,*B,*C;
   register int rA,cA,k=0;
   int TAG, r, c, *A_row_ind, *A_col_ind, *B_row_ind, *B_col_ind;

   if(!cmplxmatch()) return 0;

   Amat=tos-1; Bmat=tos;
   A=Amat->mat; rA=Amat->row; cA=Amat->col;
   B=Bmat->mat;
   if(rA!=Bmat->row || cA!=Bmat->col) {
      stkerr(" plus: ",MATSNOTC); return 0;
   }
   TAG=tos->tag; /* using tag of tos */

   if((C=(double *)memget(rA,cA))==NULL) {
      stkerr(" plus: ",MEMNOT); return 0;
   }

   if (is_index_used(Amat) || is_index_used(Bmat)) {
       /* indexed addition */
       switch (is_index_used(Amat) + 10*is_index_used(Bmat)) {
           case 1:  /* A is indexed, B is not */
               A_row_ind = MAT_ROW_IDX(Amat);
               A_col_ind = MAT_COL_IDX(Amat);
               for (c = 0; c < cA; c++) {
                   for (r = 0; r < rA; r++) {
                       C[r+c*rA] = A[A_row_ind[r] + A_col_ind[c]*rA] + B[r+c*rA];
                   }
               }
               break;
           case 10:  /* B is indexed, A is not */
               B_row_ind = MAT_ROW_IDX(Bmat);
               B_col_ind = MAT_COL_IDX(Bmat);
               for (c = 0; c < cA; c++) {
                   for (r = 0; r < rA; r++) {
                       C[r+c*rA] = B[B_row_ind[r] + B_col_ind[c]*rA] + A[r+c*rA];
                   }
               }
               break;
           case 11:  /* both A and B are indexed */
               A_row_ind = MAT_ROW_IDX(Amat);
               A_col_ind = MAT_COL_IDX(Amat);
               B_row_ind = MAT_ROW_IDX(Bmat);
               B_col_ind = MAT_COL_IDX(Bmat);
               for (c = 0; c < cA; c++) {
                   for (r = 0; r < rA; r++) {
                       C[r+c*rA] = A[A_row_ind[r] + A_col_ind[c]*rA] +
                                   B[B_row_ind[r] + B_col_ind[c]*rA];
                   }
               }
               break;
           default:
               stkerr(" plusm: ","case out of bounds"); 
               return 0;
       }

   } else {
       /* conventional addition */
       for(;k<rA*cA;k++) *(C+k)=*(A+k)+(*(B+k));
   }
   drop2();
   return(push(MAT,(char *)memgetn("_plus",5),TAG,0,C,NULL,rA,cA,NULL));
}

int plusi() /* ( hA hD hR hC --- hB )  indexed matrix addition {{{1 */
/*
 * man entry:  plusi {{{2
 * (hA hD hR hC --- hB) Sum a dense matrix, D, into an existing, possibly larger matrix, A, using the provided row (R) and column (C) indices such that 
  B[R, C] = A[R, C] + D 
This is the dense version of spsum.  A and D must either both be real, or both be complex.
 * category: math::matrix::operator
 * related: plus, spsum
 * 2}}}
 */
{
    stkitem *Amat, *Dmat, *Rmat, *Cmat;
    double  *A, *Row_Ind, *Col_Ind, *D, *B;
    int      nRows_D, nCols_D, nRows_A, nCols_A, i, j;
int DEBUG = 0;

    Amat    = tos-3;
    Dmat    = tos-2;
    Rmat    = tos-1;
    Cmat    = tos; 

    A       = Amat->mat; 
    D       = Dmat->mat;
    Row_Ind = Rmat->mat;
    Col_Ind = Cmat->mat;

    /* type and size check the inputs */
    if (Cmat->typ != MAT || is_complex(Cmat)) {
        stkerr(" plusi [Col_index]: ",MATNOT); 
        return 0;
    }
    if (Rmat->typ != MAT || is_complex(Rmat)) {
        stkerr(" plusi [Row_index]: ",MATNOT); 
        return 0;
    }
    if (Dmat->typ != MAT) {
        stkerr(" plusi [D]: ",MATNOT); 
        return 0;
    }
    if (Amat->typ != MAT) {
        stkerr(" plusi [A]: ",MATNOT); 
        return 0;
    }
    if (is_complex(Amat) != is_complex(Dmat)) {
        stkerr(" plusi : ",MATSNOTC); 
        return 0;
    }

    nRows_D = Dmat->row;
    if (is_complex(Dmat)) nRows_D /= 2;
    nCols_D = Dmat->col;

    /* number of terms in the index vectors must align with 
     * number of rows and columns in [D] 
     */
    if (Rmat->row*Rmat->col != nRows_D ||
        Cmat->row*Cmat->col != nCols_D) {
        stkerr(" plusi : ",MATSNOTC); 
        return 0;
    }

    /* [A] need not really be larger than [D], but it must 
     * contain the indices R,C.
     */
    nRows_A = Amat->row;
    if (is_complex(Amat)) nRows_A /= 2;
    nCols_A = Amat->col;

    for (i = 0; i < nRows_D; i++) {
        if ( (int) Row_Ind[i] <            XBASE ||
             (int) Row_Ind[i] >= nRows_A + XBASE) {
if (DEBUG) gprintf("plusi index error:  %d < %d or >= %d\n",
(int) Row_Ind[i], XBASE, nRows_A + XBASE);

            stkerr(" plusi : ","row indices out of range"); 
            return 0;
        }
    }

    for (i = 0; i < nCols_D; i++) {
        if ( (int) Col_Ind[i] <            XBASE ||
             (int) Col_Ind[i] >= nCols_A + XBASE) {
            stkerr(" plusi : ","column indices out of range"); 
            return 0;
        }
    }

if (DEBUG) {
gprintf("A = %2d x %2d     A = %2d x %2d\n", 
Amat->row, Amat->col, Dmat->row, Dmat->col);
gprintf("R=");
for (i = 0; i < nRows_D; i++) gprintf("%2d ", (int) Row_Ind[i]);
gprintf("\n");
gprintf("C=");
for (i = 0; i < nCols_D; i++) gprintf("%2d ", (int) Col_Ind[i]);
gprintf("\n");
}

    /* finally ready to do some math */

    /* work with a copy of [A] */
    pushstr("4 revn cop 4 revn"); xmain(0);  /* ( A D R C --- _cop D R C ) */
    B = (tos-3)->mat;

    /* recover the original row sizes to account for complex case */
    nRows_A = Amat->row;
    nRows_D = Dmat->row;
    for (j = 0; j < nCols_D; j++) {
        for (i = 0; i < nRows_D; i++) {
if (DEBUG) {
gprintf("B[%3d]=%14.5e += D[%2d]=%14.5e = ",
  ((int) Row_Ind[i]-XBASE) + ((int) Col_Ind[j]-XBASE)*nRows_A  ,
B[((int) Row_Ind[i]-XBASE) + ((int) Col_Ind[j]-XBASE)*nRows_A] ,
  i + j*nRows_D                                                ,
D[i + j*nRows_D] );
}
            B[ ((int) Row_Ind[i]-XBASE) + 
               ((int) Col_Ind[j]-XBASE)*nRows_A ] += D[i + j*nRows_D];
if (DEBUG) {
gprintf("%14.5e\n",
B[((int) Row_Ind[i]-XBASE) + ((int) Col_Ind[j]-XBASE)*nRows_A] );
}
        }
    }

    drop(); /* [D] */
    drop(); /* {R} */
    drop(); /* {C} */

    pushstr("'_plusi' naming"); xmain(0);

    return 1;

} /* 1}}} */

int plusn() /* + (x y --- x+y) */
/* Adding numbers. */
{
   double xi,xr,yi,yr;

   if(is_complex(tos) || is_complex(tos-1)) {
      popdx(&yr,&yi);
      popdx(&xr,&xi);
      return(pushdx(xr+yr,xi+yi));
   }
   return(push(NUM,NULL,NOTAG,(pop()->real)+(pop()->real),NULL,NULL, \
      0,0,NULL));
}

int power() /* pow (hA d --- hB)  ^ (hA d --- hB) */
{
   if(stkdepth()<2) {
      stkerr(" pow: ",NEEDTWO);
      return 0;
   }
   if(is_sparse(tos-1)) {
      swap();
      dense();
      swap();
   }
   if((tos-1)->typ==MAT && ((tos-1)->row==(tos-1)->col)) {
      return(
         pushstr("powmat") &&
         xmain(0) 
      );
   }
   return(power1());
}

int power1() /* ^by (hA d --- hB) B(i,j)=A(i,j)**d */
/* Raises all terms in matrix A to power d.

   Operates as follows when A(i,j) or d are negative or zero:
      If A(i,j) = 0, A(i,j)**d = 0
      If A(i,j) < 0, A(i,j)**d = -|A(i,j)|**d

      Thus 0**(-1) gives 0, not an error
      and -3**(-1) gives -.333333 */ 
{
   int typ;
   register double *A,Ak,*B,*P,*R;
   register int rA,cA,k=0;
   double arg,d,r;
   char *s="_pow";
  
   if(!popd(&d)) return 0;

   if(is_sparse(tos)) dense();

   if((typ=tos->typ)!=MAT && typ!=NUM) {
      stkerr(" ^by: ",NUMORMATNOT); 
      return 0;
   }
   if(is_complex(tos)) {
      hand();
      if(d<0) {
         conj1();

         R=tos->mat;
         rA=(tos->row*tos->col)/2;

         for(;k<rA;k++) {
            r=(*R)*(*R) + (*(R+1))*(*(R+1));

            *R/=r;
            R++;
            *R/=r;
            R++;
         }
         if(typ==NUM) ontop();
         pushd(ABS(d));
         return(power()); /* re-enter this function */
      }
      cmagphase();

      R=(tos-1)->mat;
      P=tos->mat;
      rA=tos->row*tos->col;

      for(;k<rA;k++) {
         r=pow(*R,d);
         arg=d*(*P);

         *R=r*(cos(arg));
         *P=r*(sin(arg));

         R++;
         P++;

      }
      if(typ==NUM) 
         return(dblcmplx() && ontop());

      else 
         return(
            dblcmplx() &&
            pushq2(s,strlen(s)) &&
            naming()
         );
   }
/* Doing real. */
   if(typ==NUM && d==0.5 && tos->real<0) {
      return(
         pushint(0) &&
         pushd(pow(fabs((tos-1)->real),d)) &&
         dblcmplx() &&
         lop()
      );
   }
   hand();

   if((B=(double *)memget((rA=tos->row),(cA=tos->col)))==NULL) {
      stkerr(" ^by: ",MEMNOT); 
      return 0;
   }
   A=tos->mat; 

   if(d<0) {
      for(;k<rA*cA;k++) {
         if((Ak=*(A+k))!=0) {
            if(Ak<0) {
               Ak=ABS(Ak);
               *(B+k)=-pow(Ak,d);
            }
            else {
               *(B+k)=pow(Ak,d);
            }
         }
         else *(B+k)=0;
      }
   }
   else {
      for(;k<rA*cA;k++) {
         if((Ak=*(A+k))!=0) *(B+k)=pow(Ak,d);
         else *(B+k)=0;
      }
   }
   if(typ==NUM)
      return(
         drop() &&
         push(NUM,NULL,NOTAG,*B,NULL,NULL,0,0,NULL)
      );
   else
      return(
         drop() &&
         push(MAT,(char *)memgetn(s,strlen(s)),NOTAG,0,B,NULL,\
            rA,cA,NULL)
      );
}

void qsort1(double *A, int len, int dir)
/* Fri Aug 27 13:02:50 PDT 2010.  Add a call to vqsort().

   Using quick sort algorithm; ascending if dir=1, descending if dir=0.

   Word size of A elements is 8 bytes, so if A is really text the
   sorting is being done on 8 characters interpreted as a double
   number. 

   Sun Aug 29 05:28:26 PDT 2010.  Testing option VQSORT:
      Descending sort:
         1E7 1 random 10000 * integer (hA)
         yes VQSORT
         memprobe trace
         (hA) no sort
         notrace memprobe reversed sorted? .i

         Sourcing the phrases above from work.v:
            [tops@plunger] ready > ww
             0  next word: no
             next word: sort
             qsort1: vqsort ok
             next word: notrace
             0  -1 [0 = no memory leak, -1 = descending sort ok]
            [tops@plunger] ready > 

      Sorting columns:
         5 3 random 64000 * integer (hA)
         yes VQSORT trace (hA) colsort (hB) notrace itext .

         Sourcing the phrases above from work.v; results are correct
         by inspection:
            [tops@plunger] ready > ww
             next word: colsort
             qsort1: vqsort ok
             qsort1: vqsort ok
             qsort1: vqsort ok
             next word: notrace
               37822    24985     4005
               40477    33894    29130
               45503    42125    50515
               47605    45513    57385
               58256    56714    59851
            [tops@plunger] ready > 

   Here is a word for timing vqsort:

      inline: vqtest (hA n --- )
       \ Test vqsort on matrix A.  A must contain integer numbers.
         "N" book
         "A" book

         time push
         N 1st DO A yes sort "B1" book LOOP
         time pull - " vqtest:  qsort ET: " . . nl

         yes VQSORT
         time push
         N 1st DO A yes sort "B2" book LOOP
         time pull - " vqtest: vqsort ET: " . . nl

         B1 B2 - null?
         IF " vqtest: OK"
         ELSE " vqtest: ERROR"
         THEN . nl
      end

      1E7 1 random 10000 * integer "A" book     

   Here is a timing test for matrix A of 1E7 rows; it shows vqsort() is
   about 4 times faster:

      [dale@plunger] /home/dale > tops
               Tops 3.2.0
      Sun Aug 29 05:58:59 PDT 2010
      [tops@plunger] ready > ww
       word vqtest into catalog
 
      [tops@plunger] ready > A 1 vqtest
       vqtest:  qsort ET:  1.8592E+01
       vqtest: vqsort ET:  3.9374E+00
       vqtest: OK

      [tops@plunger] ready > 
*/
{  register int i=0;
   static int *p=NULL;
   int ok=0;
   double *Ak,*B=NULL,*Bi;

/* Fri Aug 27 08:00:29 PDT 2010.  Try very quick sort first: */
   if(VQSORT) {
      VQSORT=0;
      ok=vqsort(A,len);
      if(ok) {
         if(TRACE) {
            gprintf(" qsort1: vqsort ok"); nc();
         }
         if(dir==0) {
            if((B=malloc(1+len*sizeof(double)))==NULL) {
               stkerr(" qsort1 reverse vqsort: ",MEMNOT);
               return;
            }
            memcpy(B,A,len*sizeof(double));
            Bi=B;
            Ak=A+len;
            for(i=0;i<len;i++) {
               Ak--;
               *Ak=*Bi;
               Bi++;
            }
            mallfree((void *)&B);
         }
         return;
      }
   }
   if(dir) qsortup(A,p,0,len-1);
   else qsortdn(A,p,0,len-1);

   mallfree((void *)&p);
   return;
}

int *qsort2(double *A, int len, int dir)
/* Using quick sort algorithm; ascending if dir=1, descending if dir=0.

   Sorts A and returns a mallocked integer pointer to a list of un-
   sorted row numbers (0-based) used as follows: 
      The list index is a row number in sorted A, and the value 
      in the list is a row number in unsorted A.  For example, if 
      element 10 (index 10) in the list contains the value 54, it 
      means the 10th element in sorted A used to be the 54th in 
      unsorted A.

   WARNING: the calling function must free the returned int pointer p.

   Returned p is null if error.

   Word size of A elements is 8 bytes, so if A is really text the
   sorting is being done on 8 characters interpreted as a double
   number. */
{
   register int i=0;
   static int *p=NULL;

   if((p=malloc(1+len*sizeof(int)))==NULL) {
      stkerr(" qsort2: ",MEMNOT);
      return NULL;
   }
   if(len==0) return p;
   for(;i<len;i++) *(p+i)=i;

   if(dir) qsortup(A,p,0,len-1);
   else qsortdn(A,p,0,len-1);

   return p;
}

void qsortdn(double *A, int *p, int left, int right)
/* Quick sort for double vector A, with companion int vector p.

   Adapted from the quicksort invented and named by C.A.R. Hoare and
   presented in:
      Schlict, H., "C: The Complete Reference," Osborne McGraw-Hill,
      1995.
 
   On average, this sort uses n*log(n) comparisions and is much faster
   than order n-squared algorithms like the bubble sort. */
{
   register int i,j,q;
   double x,y;

   i=left;
   j=right;
   x=*(A+(i+j)/2);

   if(p==NULL) {
      do {
         while(*(A+i)>x && i<right) i++;
         while(x>*(A+j) && j>left) j--;

         if(i<=j) {
            y=*(A+j);
            *(A+j)=*(A+i);
            *(A+i)=y;

            i++;
            j--;
         }
      }
      while(i<=j);
   } 
   else {
      do {
         while(*(A+i)>x && i<right) i++;
         while(x>*(A+j) && j>left) j--;

         if(i<=j) {
            y=*(A+j);
            *(A+j)=*(A+i);
            *(A+i)=y;

            q=*(p+j);
            *(p+j)=*(p+i);
            *(p+i)=q;

            i++;
            j--;
         }
      }
      while(i<=j);
   } 
   if(left<j) qsortdn(A,p,left,j);
   if(i<right) qsortdn(A,p,i,right);
}

void qsortup(double *A, int *p, int left, int right)
/* Quick sort for double vector A, with companion int vector p.

   Adapted from the quicksort invented and named by C.A.R. Hoare and
   presented in:
      Schlict, H., "C: The Complete Reference," Osborne McGraw-Hill,
      1995.

   On average, this sort uses n*log(n) comparisions and is much faster
   than order n-squared algorithms like the bubble sort. */
{
   register int i,j,q;
   double x,y;

   i=left;
   j=right;
   x=*(A+(i+j)/2);

   if(p==NULL) {
      do {
         while(*(A+i)<x && i<right) i++;
         while(x<*(A+j) && j>left) j--;

         if(i<=j) {
            y=*(A+i);
            *(A+i)=*(A+j);
            *(A+j)=y;

            i++;
            j--;
         }
      }
      while(i<=j);
   }
   else {
      do {
         while(*(A+i)<x && i<right) i++;
         while(x<*(A+j) && j>left) j--;

         if(i<=j) {
            y=*(A+i);
            *(A+i)=*(A+j);
            *(A+j)=y;

            q=*(p+i);
            *(p+i)=*(p+j);
            *(p+j)=q;

            i++;
            j--;
         }
      }
      while(i<=j);
   }
   if(left<j) qsortup(A,p,left,j);
   if(i<right) qsortup(A,p,i,right);
}

double rand01(double *seed)
/* Unit uniform random number by Lehmer's method:

         X(i) = mod[C * X(i-1) / M], the next seed
         r = X(i)/M, the current random number

   Random number r is never 0 or 1, and the nonrepetition period 
   is M-1.

   Ref: Gillespie, D.T., "Markov Processes," Academic Press, 1992. */
{
   const double M=2147483647,C=16807;

   *seed=fmod(*seed*C,M);
   return(*seed/M);
}

double rand0(double *seed)
/* Random number using Schrage's method to avoid exceeding 32 bits
   in the modulo function 
   Reference: "Numerical Recipes in Fortran," 2nd ed., 1992, p. 270

   In the first 1,000,000 random numbers, this method and Lehmer's 
   method agree on 995,478 of them.  Of the 4522 values that did not 
   agree, the largest difference was 1.1102E-16 (hey, that's what word
   tiny gives for 1 bit) and the smallest was 4.3368E-19. */
{
   register unsigned int k;
   const unsigned int IA=16807,IM=2147483647,IQ=127773,IR=2836;
   const double AM=4.6566128752457969230960E-10;

   k=*seed/IQ;
   *seed=IA*(*seed-k*IQ)-IR*k;
   if(*seed<0) *seed+=IM;
   return(*seed*AM);
}

int random1() /* random (r c --- hA) */
/* Creating r-by-c matrix of random numbers between 0 and 1. */
{
   int rows,cols;
   register int k=0;
   register double *A;

   if(!(popint(&cols) && popint(&rows))) return 0;
    
   if((A=(double *)memget(rows,cols))==NULL) {
      stkerr(" random: ",MEMNOT); return 0;
   }
   for(;k<rows*cols;k++) *(A+k)=rand0(&SEED);

   return(push(MAT,(char *)memgetn("_random",7),NOTAG,0,A,NULL, \
      rows,cols,NULL));
}

int ranking() /* ranking (hV n --- hR hH) */
/* Sat Oct 26 02:51:31 PDT 2013
   Tue Oct 29 04:12:36 PDT 2013.  Add returned matrix H.

   R(k) of vector R, 0<k<=n, gives the ranking of value V(k) relative
   to all prior values of vector V(1):V(k-1).

   R(k) equal to 1 means highest ranking, i.e., that V(k) is the maxi-
   mum of all prior values V(1):V(k-1); R(k) equal to 2 means V(k) is
   next highest, and so on.

   Values in V at rows above n are not ranked, and R(k) for k>n is
   equal to zero.

   Row k of returned matrix H has three columns containing row numbers
   in V (in the current index base) pointing to:
      V[H(k,1)] highest ranked (R[H(k,1)] equals 1)
      V[H(k,2)] median rank
      V[H(k,3)] lowest ranked (for large n, R[H(k,3)] approaches n)

   For rows in V that are not ranked, rows of H contain (n-1)+XBASE
   which points to the highest ranked row of V.

   Test case Tue Oct 29 06:24:04 PDT 2013:

      To run, paste the following at the ready prompt:
         1based syspath "../src/math.c" + "test_ranking___" msource

      Results:
         [tops@kaffia] ready > syspath "../src/math.c" + \
                               "test_ranking___" msource

                        V        R      H(1)     H(2)     H(3)
          Row 1:        5        1        1        1        1
          Row 2:        4        2        1        2        2
          Row 3:        3        3        1        2        3
          Row 4:        2        4        1        3        4
          Row 5:        5        1        5        2        4
          Row 6:      5.1        1        6        2        4
          Row 7:      102        0        6        6        6
          Row 8:      101        0        6        6        6

         [tops@kaffia] ready > 

      Discussion:
         Ranking is done serially as in a real time process where value
         V(k) is next after V(k-1) in a series appearing one by one, and
         not as in a batch process where all values are known and ranked
         at the same time.  When k=n is the latest real time step, R(k)
         is always correct; but R(k) for k<n may no longer be the cor-
         rect rank, since former steps are not reranked.

         In row 1, initial V is ranked 1, the highest.  Then lesser
         values of V in rows 2, 3 and 4 are ranked 2nd, 3rd and 4th as
         they appear.  

         Value 5 appears again in row 5 and is again ranked 1.  

         Then higher value 5.1 appears in row 6 and is ranked 1.

         In row 6, value 5.1 becomes top ranked, and earlier top rank
         given to value 5 in row 5 is not valid; in fact we now know 
         that value 5 is second ranked and therefore second rank of 
         value 4 in row 2 is also not valid anymore; such is the lot of
         real time rankers since what's coming is not known.

         Inspection of the three columns H(1), H(2) and H(3) contain-
         ing row indices for V shows they correctly point to, respec-
         tively, highest rank, median rank and lowest rank.

         Since n=6, the remaining values in rows 7 and 8 are unranked.

         Results are correct by inspection.  For further discussion of
         these results see man ranking, where also a real time case 
         displaying plotted curves is given.

      The phrase above with word msource runs this region of this file:
         test_ranking___
            list: 5 4 3 2 5 5.1 102 101 ; (hV) \
            dup 6 (hV n) ranking (hV hR hH) 3 parkn .m nl
         halt \ */
{
   register double *A,*B,*H,*R,*V;
   int k=0,n,r,rput=0;

   if(!popint(&n)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" ranking: ",MATNOT); 
      return 0;
   }
   V=tos->mat;
   r=tos->row;
   if(n>r) {
      gprintf(" ranking: n cannot be greater than rows of V");
      nc();
      stkerr("","");
      return 0;
   }
   if(!matstk(r,1,"_R")) return 0;
   R=tos->mat;
   memset(R,0,r*sizeof(double));

   if(!matstk(r,3,"_H")) return 0;
   H=tos->mat;
   for(;k<3*r;k++) *(H+k)=(n-1)+XBASE;
   *H=*(H+r)=*(H+2*r)=XBASE;

   if(!matstk(n,1,"_A")) return 0;
   A=tos->mat;

   if(!matstk(n,1,"_B")) return 0;
   B=tos->mat;

   *A=*V;
   *B=0;
   *R=1;
   for(k=1;k<n;k++) {
      insrt(A,k,&rput,*(V+k),B,k);
      *(R+k)=k-rput+1;         /* ranking */
      *(H+k)=*(B+k)+XBASE;     /* highest */
      *(H+k+r)=*(B+k/2)+XBASE; /* median */
      *(H+k+2*r)=*B+XBASE;     /* lowest */
   }
   return(drop2() && rot() && drop());
}

int ranking1() /* ranking1 (hV hP n --- hR hH) */
/* Tue Oct 29 14:41:59 PDT 2013

   R(k) of vector R, 0<k<=n, gives the ranking of value V(k) relative
   to all prior values of vector V(1):V(k-1).

   R(k) equal to 1 means highest ranking, i.e., that V(k) is the maxi-
   mum of all prior values V(1):V(k-1); R(k) equal to 2 means V(k) is
   next highest, and so on.

   Values in V at rows above n are not ranked, and R(k) for k>n is
   equal to zero.

   Vector P of h rows contains percentile levels (0 to 100) of rankings
   to return in the h columns of H.  For example, if P(i)=50, term 
   V[H(k,i)] corresponds to the median rank (the 50 percentile level).
   If P=[100, 50, 0]' then returned H exactly matches H returned by 
   function ranking(), where:
      V[H(*,1)] highest ranked (R[H(k,1)] equals 1)
      V[H(*,2)] median rank
      V[H(*,3)] lowest ranked (for large n, R[H(k,3)] approaches n)

   For rows in V that are not ranked, rows of H contain (n-1)+XBASE
   which points to the highest ranked row of V.

   Test case Tue Oct 29 16:06:24 PDT 2013:

      To run, paste the following at the ready prompt:
         1based syspath "../src/math.c" + "test_ranking1___" msource

      Results:
         [tops@kaffia] ready > syspath "../src/math.c" + \
                               "test_ranking1___" msource

              100       50        0 

                        V        R      H(1)     H(2)     H(3)
          Row 1:        5        1        1        1        1
          Row 2:        4        2        1        2        2
          Row 3:        3        3        1        2        3
          Row 4:        2        4        1        3        4
          Row 5:        5        1        5        2        4
          Row 6:      5.1        1        6        2        4
          Row 7:      102        0        6        6        6
          Row 8:      101        0        6        6        6

         [tops@kaffia] ready > 

      Discussion:
         Incoming P=[100 50 0] to match [highest median lowest] ranks
         for H returned by function ranking().

         These results match the results of function ranking(), which
         are correct by inspection (H corresponds to 1-based indexing).

      The phrase above with word msource runs this region of this file:
         test_ranking1___
            list: 5 4 3 2 5 5.1 102 101 ; (hV) dup \
            list: 100 50 0 ; (hP) dup bend mtext nl . nl \
            6 (hV hP n) ranking1 (hV hR hH) 3 parkn nl .m nl
         halt \ */
{
   register double *A,*B,*H,*P,*R,*V;
   int h,i=0,k=1,n,r,rput=0;

   if(!popint(&n)) return 0;

   hand(); /* P into 1-by-1 MAT if it is a NUM */

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" ranking1: ",MATNOT2); 
      return 0;
   }
/* Fri Nov 15 14:45:38 PST 2013.  Values of P will be changed (divided
   by 100), so use a copy in case P is in the library: */
   cop(); 
   P=tos->mat;
   h=tos->row;
   for(;i<h;i++) *(P+i)/=100;

   V=(tos-1)->mat;
   r=(tos-1)->row;
   if(n>r) {
      gprintf(" ranking1: n cannot be greater than rows of V");
      nc();
      stkerr("","");
      return 0;
   }
   if(!matstk(r,1,"_R")) return 0;
   R=tos->mat;
   memset(R,0,r*sizeof(double));

   if(!matstk(r,h,"_H")) return 0;
   H=tos->mat;
   for(i=0;i<h*r;i++) *(H+i)=(n-1)+XBASE;
   for(i=0;i<h;i++) *(H+i*r)=XBASE;

   if(!matstk(n,1,"_A")) return 0;
   A=tos->mat;

   if(!matstk(n,1,"_B")) return 0;
   B=tos->mat;

   *A=*V;
   *B=0;
   *R=1;
   for(;k<n;k++) {
      insrt(A,k,&rput,*(V+k),B,k);
      *(R+k)=k-rput+1; /* ranking */
      for(i=0;i<h;i++) *(H+k+(i*r))=*(B+(int)(*(P+i)*k))+XBASE;
   }
   return(drop2() && rot() && drop() && rot() && drop());
}

int real2() /* 2real (hI --- hA) */
/* 2-byte signed int to 8-byte fp */
{
   int f=0,ret=0;

   if(tos->typ==STR) {
      hand();
      if(tos->col==2) f=1;
   }
   ret=(byteorder1() && import2());
   if(f && ret) 
      ret=(
         ret &&
         pushint(XBASE) &&
         pry()
      );
   return(ret);
}  

int real4() /* 4real (hI --- hA) */
/* 4-byte signed int to 8-byte fp */
{
   int f=0,ret=0;

   if(tos->typ==STR) {
      hand();
      if(tos->col==4) f=1;
   }
   ret=(byteorder1() && import4());
   if(f && ret) 
      ret=(
         ret &&
         pushint(XBASE) &&
         pry()
      );
   return(ret);
}  

int real2_() { return(byterm1(REAL2)); } /* not used (hI --- hA) */
int real4_() { return(byterm1(REAL4)); } /* not used (hI --- hA) */

double _real2(short x)
/* Converting 2-byte integer bit pattern into 8-byte floating point 
   bit pattern. */
{  
   return((double)x); 
}

double _real4(long x)
/* Converting 4-byte integer bit pattern into 8-byte floating point 
   bit pattern. */
{  
   return((double)x); 
}

double _real4f(float x)
/* Converting 4-byte floating point bit pattern into 8-byte floating 
   point bit pattern. */
{ 
   return((double)x);
}

int rsort(double *A, int n, int *rnew, double Anew, double Aold)
/* Tue Mar  5 20:06:15 PST 2013

   Incoming vector A with n rows is already in ascending order sort.

   Insert new value Anew and remove old value Aold, so returned A is
   still in sort with n rows. 

   Mon Mar 25 10:18:41 PDT 2013
   The value in returned rnew is the zero-based index where Anew was 
   inserted into A. */ 
{
   int rold;

   if(Anew>=*(A+n-1)) *rnew=n;
   else {
      if(Anew<=*A) *rnew=0;
      else {
         if(!bsearchd(Anew,A,n,rnew)) *rnew+=1;
      }
   }
   if(!bsearchd(Aold,A,n,&rold)) {
      gprintf(" rsort: old value %f not found in A",Aold);
      nc();
      stkerr("","");
      return 0;
   }
   if(rold<*rnew) {
      *rnew-=1;
      memmove(A+rold,A+rold+1,(*rnew-rold)*sizeof(double));
   }
   else {
      if(*rnew<rold)
         memmove(A+*rnew+1,A+*rnew,(rold-*rnew)*sizeof(double));
   }
   *(A+*rnew)=Anew;
   return 1;
}

int searching() /* searching (u qU(k) kmax --- k) */
/* Finding k of function U(k) that produces value closest and below u.

   Wed Jan 15 12:07:08 PST 2014.  Phrase U(k) is made into a macro for 
   speed while searching.

   For word or phrase U(k), find nonnegative integer k<=kmax such 
   that U(k)<u<U(k+1). 

   U(k) must be a monotonically increasing and non-negative function
   with stack behavior like a table look-up for offset k: 

      U(k) (k --- u[k]).

   Note that k is a 0-based index.

   Note that a decreasing function can be negatively scaled to put it
   into the increasing form expected here.

   Example: 
      51 "10 *" 100 searching .i 

      Here, U(k)=10*k acts like a table of values: 0,10,20,30,..., for 
      indices k=0,1,2,3,...100.  Searching will find U(5) (50 from the 
      table) to be closest and below the value 51 of u, so k=5 will 
      return on the stack. 

   Test: with debug print uncommented, running the example and obtain-
      ing result k=5:

      [dale@kaffia] /opt/tops/tops/src > tops
               Tops 3.2.1
      Wed Jan 15 12:47:08 PST 2014
      [tops@kaffia] ready > 51 "10 *" 100 searching .i
       searching.  k1 0 k2 25 u1 500
       searching.  k1 0 k2 12 u1 250
       searching.  k1 0 k2 6 u1 120
       searching.  k1 0 k2 3 u1 60
       searching.  k1 3 k2 1 u1 30
       searching.  k1 4 k2 1 u1 40
       searching.  k1 5 k2 1 u1 50
       searching.  k1 5 k2 0 u1 60
       5
      [tops@kaffia] ready > bye
      31 keys
                Good-bye
      Wed Jan 15 12:47:21 PST 2014
      [dale@kaffia] /opt/tops/tops/src > 

   Macro __searching() appears in the catalog after this function runs:

      [dale@kaffia] /opt/tops/tops/src > tops
               Tops 3.2.1
      Wed Jan 15 12:54:42 PST 2014
      [tops@kaffia] ready > whos
       No stack items or words have been added to main library

      [tops@kaffia] ready > 51 "dup *" 90 searching
 
       stack elements:
             0 number: 7
       [1] ok!
      [tops@kaffia] ready > whos
       Words added to the main library:
        Name        Rows Cols Bytes Type  Description
        __searching 4    1    16    PTR   inline
                              16    total

       stack elements:
             0 number: 7
       [1] ok!
      [tops@kaffia] ready > bye
      38 keys
                Good-bye
      Wed Jan 15 12:54:58 PST 2014
      [dale@kaffia] /opt/tops/tops/src > 

   From the Appendix of file sys/cal.v, this high level version of 
   searching() shows the binary search implemented in a BEGIN ... UNTIL
   loop to test the program in about 1999 (cal.v also holds /int(), a
   word no longer in the program):

   define: searching (u qU kmax --- k) \ k of U(k) with value closest
                                       \ below u
{     For executable function U(k), find k such that U(k) < u < U(k+1)
      U(k) must be a monotonically increasing and non-negative function
      Stack behavior of function U(k) is like a table look-up:
         (k --- u[k])
      Example: 51 "10 *" 300 searching \ U(k) = 10*k; finds U(5)
                                       \ to be closest to 51
}     push into U(k), (u) dup peek (kmax) U(k) execute drop
      (u U[kmax]) < not \ equal or above max?
      IF drop pull return THEN 0 pull 2 /int \ Run binary search loop:
      BEGIN 2dup + U(k) main, 3 pick 2dup =
         IF 2drop + lop true
         ELSE > IF 2 /int
                ELSE dup rot + swap 2 /int 1 max
                THEN dup 0= IF + lop true ELSE false THEN
         THEN
      UNTIL
   end */
{
   register int k1=0,k2=0;
   int kmax,typ;
   double u,u1;
   char *name="__searching";
   int catmsg;

   if(!popint(&kmax)) return 0;

   if((typ=tos->typ)!=VOL && typ!=STR) {
      stkerr(" searching: ",STRORVOLNOT);
      return 0;
   }
   if(!(swap() && popd(&u))) return 0;

   catmsg=CATMSG; CATMSG=0;
   pushstr(name);
   pushstr("macro"); xmain(0); /* making U(k) into a macro */
   CATMSG=catmsg;

/* Checking boundary conditions: */
   pushint(kmax); 
   caton(tagnative(name)); exeinline(); /* execute macro */
   popd(&u1);
   if(u>=u1) return(pushint(kmax));

   pushint(k1); 
   caton(tagnative(name)); exeinline(); /* execute macro */
   popd(&u1);
   if(u<=u1) return(pushint(k1));

/* Binary searching with macro U(k).  The structure of code below
   follows bsearchd() in file math.c: */
   k2=kmax>>1;
   while(k2) {

   /* Setting up stack and running function U(k1+k2): */
      pushint(k1+k2); /* set up for U(k1+k2) */
      caton(tagnative(name)); exeinline(); /* execute macro */
      popd(&u1);
      if(u==u1) break; /* checking if u=U(k1+k2) */
      else {
        if(u1>u) k2=k2>>1; /* looking lower */
         else { 
            k1+=k2; /* looking higher */
            k2=MAX(1,k2>>1); /* not allowing 0 here */
         }
      }
    /*gprintf(" searching.  k1 %d k2 %d u1 %d\n",k1,k2,(int)u1);*/
   } 
   return(pushint(k1+k2)); /* returning k */
}

int seed0() /* seed0 ( --- x) */
/* Getting initial random seed. */
{
   return(pushd(SEED0));
}

int seedget() /* seedget ( --- x) */
/* Getting current random seed. */
{
   return(pushd(SEED));
}

int seedset() /* seedset (x --- ) */
/* Setting random seed. */
{
   const double seedmax=2147483646;
   double seed;

   if(!popd(&seed)) return 0;

   if((seed<=0) || (ABS(seed)>seedmax)) {
      stkerr(" seed: ",SEEDBAD);
      return 0;
   }
   SEED=seed;

/* Using system seconds (as returned by time) is handy for setting a
   different seed (as in phrase 'time seedset') but this always gives a
   similar number for the very first random number.  Run rand0() once 
   to get modulo cranked up (but note that 'N seedset setget' will not 
   return N): */
   rand0(&SEED);
   
   return 1;
}

int sign() /* sign ( --- n) */
/* Pushing to stack the number of the bit that contains the sign of
   8-byte floating point numbers (includes current index base). */
{
   return(pushint(XBASE+SIGNBIT));
}

int sign1(int *signbit)
/* Finding the bit that contains the sign of 8-byte fp numbers on 
   this machine. */
{
   double x;
   int b=0,k,ret;
   
   x=_xor(1,-1);
   ret=pushd(x);

   k=XBASE-1;
   while(ret && !b) {
      k++;
      ret=(
         ret &&
         dup1s() &&
         pushint(k) &&
         bit() &&
         popint(&b)
      );
   } 
   *signbit=k-XBASE;
   ret=(
      ret &&
      drop()
   );
   if(!ret) {
      stkerr(" sign1: ",SBITERR);
   }
   return ret;
}

double _sin(double x) { return(sin(x)); }

int sin1() { return(byterm1(SIN)); } /* sin (hA --- hA1) */

double _sinh(double x) { return(sinh(x)); }

int sinh1() { return(byterm1(SINH)); } /* sinh (hA --- hA1) */

int slash() /* / (x y --- x/y) or (hA hB --- hC) */
/* Driver for division.

   For the nine permutations of the three types of stack items for
   division: MAT, NUM, VOL:SPARSE. 

   Matrices (dense and sparse) assume term-by-term division. */
{
   int nw=1;

   if(stkdepth()<2) {
      stkerr(" slash: ",NEEDTWO);
      return 0;
   }
   switch((tos-1)->typ) {

      case MAT: /* dense */
         switch(tos->typ) {
            case MAT: /* Have two MATs.  Is tos a scalar? */
               if(is_scalar(tos)) {
                  if(is_complex(tos)) 
                     pushdx(*tos->mat,*(tos->mat+1));
                  else pushd(*tos->mat);
                  lop();
                  return(slashf()); /* dense / scalar */
               }
               return(slashby()); /* dense / dense */
            case NUM:
               return(slashf()); /* dense / scalar */
            case VOL:
               return slashby(); /* dense / sparse */
         }
      case NUM: /* scalar */
         switch(tos->typ) {
            case MAT:
               if(is_scalar(tos)) { /* 1-by-1 MAT ok */
                  return(
                     ontop() &&
                     slashn() && /* scalar / scalar */
                     hand()
                  );
               }
               else {
                  if(is_complex(tos)) nw=2;
                  return( /*  C(i,j)=NUM/B(i,j) (not for sparse,
                              since some B(i,j)=0) */
                     swap() &&
                     pushint(((tos-1)->row)/nw) &&
                     pushint((tos-2)->col) &&
                     pushq2("fill",4) &&
                     xmain(0) &&
                     swap() &&
                     slashby()
                  );
               }
            case NUM:
               return(slashn()); /* scalar / scalar */
            case VOL:
               if(is_sparse(tos)) dense(); /* into MAT */
               else break;
               if(is_scalar(tos)) { /* 1-by-1 MAT ok */
                  return(
                     ontop() &&
                     slashn() && /* scalar / scalar */
                     hand() &&
                     sparse()
                  );
               }
               break;
         }
      case VOL: /* sparse */
         switch(tos->typ) {
            case MAT:
               return slashby(); /* sparse / dense */
            case NUM:
               pushint(1);
               swap();
               slashn();
               return spscale(); /* sparse / scalar */
            case VOL:
               return slashby(); /* sparse / sparse */
         }
   }
   stkerr(" slash: ",STKNOT);
   return 0;
}

int slashby() /* /by (hA hB --- hC) \ C(i,j)=A(i,j)/B(i,j) */
/* Matrix term-by-term division. */
{
   register double *A,*B,*C,G;
   register int rA,cA,k=0;
   int num=0,spars=0,TAG;

   if(is_sparse(tos)) {
      dense();
      spars=1;
   }
   if(is_sparse(tos-1)) {
      swap();
      dense();
      swap();
      spars=1;
   }
   if(tos->typ==NUM || (tos-1)->typ==NUM) {
      swap();
      hand();
      swap();
      hand();
      num=1;
   }
   if(stkset(" /by: ")!=MAT) {
      stkerr(" /by: ",MATNOT2); return 0;
   }
   if(!cmplxmatch()) return 0;
   TAG=tos->tag;

   A=(tos-1)->mat; 
   rA=(tos-1)->row; 
   cA=(tos-1)->col;

   B=tos->mat;

   if(rA!=tos->row || cA!=tos->col) {
      stkerr(" /by: ",MATSNOTC); 
      return 0;
   }
   if(!matstk(rA,cA,"_/by")) return 0;
   C=tos->mat;
   tos->tag=TAG;

   if(!is_complex(tos)) for(;k<rA*cA;k++) *(C+k)=*(A+k)/(*(B+k));

   else {
/*    C = (a+ib)/(c+id) = Cr + iCi
      where
         G = (c+id)*(c-id) = c*c + d*d
         Cr = (a*c + b*d)/G
         Ci = (b*c - a*d)/G
      and
         a=*A, b=*(A+1)
         c=*B, d=*(B+1)
*/
      for(;k<rA*cA;k++) {
         G=*B*(*B) + *(B+1)*(*(B+1));
         *(C+k)=(*A*(*B) + *(A+1)*(*(B+1)))/G; /* Cr at row k */
         k++;
         *(C+k)=(*(A+1)*(*B) - *A*(*(B+1)))/G; /* Ci at row k+1 */
         A++; A++;
         B++; B++;
      }
   }
   if(spars) sparse();
   else if(num) ontop();

   return(lop() && lop());
}

int slashf() /* /f (hA f --- hB) B(i,j)=A(i,j)/f */
/* Dividing each matrix term by factor f. */
{
   double fr,fi,mag;
 
   if((tos-1)->typ!=MAT) {
      stkerr(" slashf: ",MATNOT); return 0;
   }
   if(is_complex(tos)) {
      if(!popdx(&fr,&fi)) return 0;
      mag=fr*fr + fi*fi;
      pushdx(fr/mag,-fi/mag); /* complex conjugate to stack */
   }
   else {
      if(!popd(&fr)) return 0;
      pushd(1/fr);
   }
   return(starf());
}

int slashmod() /* /mod (x y --- rem quot) */
/* Quotient and remainder for division x by y; x and y are real numbers.

   Fri May  3 13:21:54 PDT 2013.  If x is a MAT and y is a NUM, create
   a MAT from y for term-by-term operation. */
{
   register double *X,*Y,*R,*Q,r,x,y;
   register int k=0;
   long long q;
   double x1,y1;
   int cols,len,rows;

   if(is_complex(tos) || is_complex(tos-1)) {
      stkerr(" /mod: ",REALNOT);
      return 0; 
   }
   if(tos->typ==NUM && (tos-1)->typ==NUM) {
      if(!popd(&y1) || !popd(&x1)) return 0;
      q=x1/y1;
      r=x1-q*y1;
      x=q;
      return(pushd(r) && pushd(x));
   }
   if(tos->typ==NUM) { /* make NUM y into a MAT */
      over();
      dims();
      fill();
   }
   if(is_sparse(tos)) dense();
   if(is_sparse(tos-1)) {
      swap();
      dense();
      swap();
   }
   if(tos->typ==MAT && (tos-1)->typ==MAT) {
      if(tos->row!=(tos-1)->row || tos->col!=(tos-1)->col) {
         stkerr(" /mod: ",STKNOTC);
         return 0; 
      }
      X=(tos-1)->mat;
      Y=tos->mat;

      rows=tos->row;
      cols=tos->col;

      if(!matstk(rows,cols,"_rem")) return 0;
      if(!matstk(rows,cols,"_quot")) return 0;
      R=(tos-1)->mat;
      Q=tos->mat;

      len=rows*cols;
      for(;k<len;k++) {
         y=*(Y+k);
         x=*(X+k);
         q=x/y;
         *(Q+k)=q;
         *(R+k)=x-q*y;
      }
      return(rot() && drop() && rot() && drop());
   }
   stkerr(" /mod: ",STKNOT);
   return 0; 
}

int slashn() /* / (x y --- x/y) */
/* Dividing numbers. */
{
   double mag,xi,xr,y,yi,yr,zi,zr;

   if(is_complex(tos) || is_complex(tos-1)) {
      popdx(&yr,&yi);
      popdx(&xr,&xi);

   /* Complex conjugate of y: */
      mag=yr*yr + yi*yi;
      yr/=mag;
      yi/=-mag;

      zr=xr*yr - xi*yi;
      zi=xr*yi + xi*yr;
      return(pushdx(zr,zi));
   }
   y=pop()->real;
   return(push(NUM,NULL,NOTAG,(pop()->real)/y,NULL,NULL,0,0,NULL));
}

int sort1() /* sort (hA f --- hA1) */
/* Sorting rows of A using the values in the first column.  If A is a
   volume, the first eight characters are used to sort. */
{
   int f,k,*k1,ret=1;
   double *L;
   double L1234[8]={7,6,5,4,3,2,1,0}; /* little endian */
   double L4321[8]={0,1,2,3,4,5,6,7}; /* big endian */
   double L3412[8]={5,4,7,6,1,0,3,2}; /* pdp endian */
   char *s;

   if(!popbool(&f)) return 0;

   if(tos->typ==NUM) return 1;
   if(tos->typ==MAT) {
      return(
         pushint(f) &&
         pushint(XBASE) &&
         sorton()
      );
   }
/* For text, the first 8 characters are used for sorting.  To use the
   numerical sorting function, the first 8 character bytes are ordered 
   so that the leftmost character is at the most significant numerical
   byte, the second character is at the second most significant byte,
   etc.  On different machines, this means taking into account the
   byte order for numerical words, as done below in the creation of 
   vector L. */

   if(tos->typ==STR) hand(); /* STR into VOL */
   if(tos->typ==VOL) {

      if(tos->col<8) { /* widen to 8 chars */
         ret=( /* piling onto phony 0-by-8 blockofblanks widens */
            dup1s() && pushint(0) && pushint(8) && blockofblanks() &&
            pile()
         );
      }
      else ret=dup1s();
      
      /* L will be used to pull chars (with function catch()) so left-
      most char goes to most significant numerical byte, etc.  

                          math byte: 0 1 2 3 4 5 6 7
             little endian char: L = 7 6 5 4 3 2 1 0 (case 1234) 
                big endian char: L = 0 1 2 3 4 5 6 7 (case 4321)
                pdp endian char: L = 5 4 7 6 1 0 3 2 (case 3412)

      L for little endian says char 0 (leftmost) will be math byte 7,
      which is the most significant.

      Big endian has bytes already in the desired order: math and char
      orderings are consistent, perhaps by design. */

      if(!matstk(8,1,"_Lsort")) return 0; /* hL on stack */
      L=tos->mat;

      k=byteorder(&k1); /* byte order for this machine */

      switch(k) { /* making L: */
         case 1234: 
            memcpy(L,L1234,8*sizeof(double));
         break;
         case 4321: 
            memcpy(L,L4321,8*sizeof(double));
         break;
         case 3412: 
            memcpy(L,L3412,8*sizeof(double));
         break;
      }
      /* now L on the stack is set to extract char cols in byteorder */

      ret=( /* stk: hA hA hL */
         ret &&
         pushint(XBASE) && plusd() && /* adding user index base to L */
         catch() && /* extracting char cols according to byteorder() */

         typvol2mat() && /* text typ to num typ (bytes not changed) */
         pushint(XBASE) && /* 1st */
         reach() && /* reach 1st row in matrix = 1st col in volume */
         bend() && /* have column of chars, 8 bytes wide, to sort */

         pushint(XBASE) &&
         over() && 
         rows() &&
         items() && /* list of ascending integers */
         park() && /* parking column of ascending integers */

         pushint(f) &&
         pushint(XBASE) && /* stack for sorton: hA f 1st */
         sorton() &&
         pushint(1+XBASE) &&
         catch() && /* grabbing 2nd column, the resorted integers */
         reach() /* reaching rows in the order of resorted integers */
      );

      if(f) s="_sort";
      else s="_sortrev";

      return(
         ret &&
         pushq2(s,strlen(s)) &&
         naming()
      );
   }
   stkerr(" sort: ",MATORVOLNOT);
   return 0;
}

int sorton() /* sorton (hA f n --- hA1) */
/* Sorting rows of A using the values in the nth column.  Incoming f
   is true for ascending sort, false for descending. */
{
   register double *B;
   register int i=0,rows;
   int n,*p;
   int cols,f,ret;
   char *s;

   if(!popint(&n)) return 0;

   if(!popbool(&f)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" sort: ",MATNOT);
      return 0;
   }
   cols=tos->col;
   if((n-XBASE)>(cols-1)) {
      stkerr(" sort: ",OUTCOL);
      return 0;
   }
   if(cols==1 && n==XBASE) {
      cop(); /* just copy if vec and 1st col */
      qsort1(tos->mat,tos->row,f);
   }
   else {
      ret=(
         dup1s() &&
         pushint(n) &&
         catch() /* column to sort */
      );
      if(!ret) return 0;
      p=qsort2(tos->mat,tos->row,f);

      if(p==NULL) {
         drop();
         return 0;
      }
      if(cols>1) {
         rows=tos->row;
         B=tos->mat;
         for(;i<rows;i++) *(B+i)=(double)(XBASE+*(p+i));
         reach();
      }
      mallfree((void *)&p);
   }
   if(f) s="_sort";
   else s="_sortrev";

   return(
      pushq2(s,strlen(s)) &&
      naming()
   );
}

int sqrt1() /* sqrt (x -- sqrt_x) */
/*
   This function replaces the following word:

   inline: sqrt (x -- sqrt_x) \ sq root of real num or real mat x terms
      [ 0.5 is ^ ] ^ pow ;
*/
{ 
   return(
      pushd(0.5) &&
      power()
   );
}

int star() /* * (x y --- x*y) or (hA hB --- hC) */
/* Driver for multiplication. 

   For the nine permutations of the three types of stack items for
   multiplication: MAT, NUM, VOL:SPARSE. */
{
   int f=0;

   if(stkdepth()<2) {
      stkerr(" star: ",NEEDTWO);
      return 0;
   }
   switch((tos-1)->typ) {

      case MAT: /* dense */
         switch(tos->typ) {
            case MAT: /* Have two MATs.  Is one a scalar? */
               if((f=is_scalar(tos-1)) || is_scalar(tos)) {
                  if(f) swap();
                  if(is_complex(tos)) 
                     pushdx(*tos->mat,*(tos->mat+1));
                  else pushd(*tos->mat);
                  lop();
                  return(starf()); /* dense * scalar */
               }
               return(starm()); /* dense * dense */
            case NUM:
               return(starf()); /* dense * scalar */
            case VOL:
               swap();
               sparse();
               swap();
               return spmult(); /* dense * sparse */
         }
      case NUM: /* scalar */
         switch(tos->typ) {
            case MAT:
               swap();
               return(starf()); /* scalar * dense */
            case NUM:
               return(starn()); /* scalar * scalar */
            case VOL:
               swap();
               return spscale(); /* scalar * sparse */
         }
      case VOL: /* sparse */
         switch(tos->typ) {
            case MAT:
               return spmult(); /* sparse * dense */
            case NUM:
               return spscale(); /* sparse * scalar */
            case VOL:
               return spmult(); /* sparse * sparse */
         }
   }
   stkerr(" star: ",STKNOT);
   return 0;
}

int starby() /* *by (hA hB --- hC) \ C(i,j)=A(i,j)*B(i,j) */
/* Matrix term-by-term multiplication. */
{
   register double *A,*B,*C;
   register int rA,cA,k=0;
   int num=0,complex=0,spars=0;

   if(is_sparse(tos)) {
      dense();
      spars=1;
   }
   if(is_sparse(tos-1)) {
      swap();
      dense();
      spars=1;
      swap();
   }
   if(tos->typ==NUM || (tos-1)->typ==NUM) {
      swap();
      hand();
      swap();
      hand();
      num=1;
   }
   if(stkset(" *by: ")!=MAT) {
      stkerr(" *by: ",MATNOT2); return 0;
   }
   if(!cmplxmatch()) return 0;
   complex=is_complex(tos);

   A=(tos-1)->mat; 
   rA=(tos-1)->row; 
   cA=(tos-1)->col;

   B=tos->mat;

   if(rA!=tos->row || cA!=tos->col) {
      stkerr(" *by: ",MATSNOTC); 
      return 0;
   }
   if(!matstk(rA,cA,"_*by")) return 0;
   C=tos->mat;

   if(complex) {
      set_complex(tos);

/*    C = (a+ib)*(c+id) = Cr + iCi
      where
         Cr = a*c - b*d 
         Ci = b*c + a*d 
      and
         a=*A, b=*(A+1)
         c=*B, d=*(B+1)
*/
      for(;k<rA*cA;k++) {
         *(C+k)=*A*(*B) - *(A+1)*(*(B+1)); /* Cr at row k */
         k++;
         *(C+k)=*(A+1)*(*B) + *A*(*(B+1)); /* Ci at row k+1 */
         A++; A++;
         B++; B++;
      }
   }
   else {
      for(;k<rA*cA;k++) *(C+k)=*(A+k)*(*(B+k));
   }
   if(spars) sparse();
   else if(num) ontop();

   return(lop() && lop());
}

int starf() /* *f (hA f --- hB) B(i,j)=A(i,j)*f */
/* Multiplying matrix A by scale factor f. */
{
   register double *A,*B;
   double xi,xr;
   register int rA,cA,rB,k=0;
   int nword=1;
  
   if((tos-1)->typ!=MAT) {
      stkerr(" starf: ",MATNOT); return 0;
   }
   nword+=is_complex(tos);
   if(!popdx(&xr,&xi)) return 0;

   nword=MAX(nword,(k=(1+is_complex(tos))));

   A=tos->mat;
   rA=(tos->row)/k;
   cA=tos->col;

   rB=nword*rA;
   if(!matstk(rB,cA,"_starf")) return 0;
   B=tos->mat;

   if(nword==2) { /* doing complex */
      set_complex(tos);

      if(k==2) { /* scaling complex A to make complex B */
         for(k=0;k<rB*cA;k+=nword) {
            *B=(*A)*xr - *(A+1)*xi;
            B++;
            *B=(*A)*xi + *(A+1)*xr;
            B++;
            A+=2;
         }
      }
      else { /* scaling real A to make complex B */
         for(k=0;k<rB*cA;k+=nword) {
            *B=(*A)*xr;
            B++;
            *B=(*A)*xi;
            B++;
            A++;
         }
      }
   }
   else for(k=0;k<rA*cA;k++) *(B+k)=*(A+k)*xr;

   return(lop());
}

int starm() /* * (hA hB --- hC) \ C=A*B */
/* Matrix multiplication. */
{
#if defined(LAPACK) || defined(ESSL)
   int ret=0;

   if(!cmplxmatch()) return 0;

 #ifdef LAPACK
   if(is_complex(tos)) ret=(
      
/*       Setting up the stack for complex zgemm1():
            (hA trnA alpha hB trnB hC beta --- hD) */

         lpush() &&                            /* B to local stk */
         pushint(xFALSE) &&                    /* no trnA */
         pushd(1) &&                           /* alpha real */
         pushd(0) &&                           /* alpha imag */
         dblcmplx() &&                         /* into complex scalar */
         lpull() &&                            /* B from local stk */
         pushint(xFALSE) &&                    /* no trnB */
         pushint(0) && pushint(0) && null() && /* purged C */
         pushd(0) &&                           /* beta real */
         pushd(0) &&                           /* beta imag */
         dblcmplx() &&                         /* into complex scalar */

         zgemm1() /* D = A*B + C */
      );

   else ret=(

/*       Setting up the stack for real dgemm1():
            (hA trnA alpha hB trnB hC beta --- hD) */

         lpush() &&                            /* B to local stk */
         pushint(xFALSE) &&                    /* no trnA */
         pushd(1) &&                           /* alpha */
         lpull() &&                            /* B from local stk */
         pushint(xFALSE) &&                    /* no trnB */
         pushint(0) && pushint(0) && null() && /* purged C */
         pushint(0) &&                         /* beta */

         dgemm1() /* D = A*B + C */
      );

   if(!ret) drop(); /* drop null C if error */

   return(ret);

 #endif

 #ifdef ESSL

   return(mpySSL());

 #endif

#else

/* Complex will source mmath.v to obtain *c. */
   char *do_complex="push real-imag pull real-imag *c complex";

   stkitem *Amat,*Bmat;
   register double *A,*B,*C,Bj;
   register int rA,cA,cB;
   register int i,j,k=0;
   register int iC1,iCn,jB1,jBn,jA1,iA;

   if(!cmplxmatch()) return 0;

   if(is_complex(tos)) {
      if(!_exists("*c")) {
         pushstr("mmath.v");
         source();
      }
      return(
         pushstr(do_complex) &&
         xmain(0) 
      );
   }
   Amat=tos-1; 
   Bmat=tos;
   A=Amat->mat; 
   cA=Amat->col; 
   rA=Amat->row;
   B=Bmat->mat; 
   cB=Bmat->col;
   if(cA!=Bmat->row) { 
      stkerr(" starm: ",MATSNOTC); 
      return 0;
   }
   if((C=(double *)memget0(rA,cB))==NULL) {
      stkerr(" starm: ",MEMNOT); 
      return 0;
   }

   MPYLOOP

   drop2();
   return(push(MAT,(char *)memgetn("_star",5),NOTAG,0,C,NULL, \
      rA,cB,NULL));

#endif
}

int starn() /* * (x y --- x*y) */
/* Multiplying two numbers. */
{
   double xi,xr,yi,yr,zi,zr;

   if(is_complex(tos) || is_complex(tos-1)) {
      popdx(&yr,&yi);
      popdx(&xr,&xi);
      zr=xr*yr - xi*yi;
      zi=xr*yi + xi*yr;
      return(pushdx(zr,zi));
   }
   return(push(NUM,NULL,NOTAG,(pop()->real)*(pop()->real),NULL,NULL, \
      0,0,NULL));
}

int stats() /* stats (hA --- hB) */
/* Matrix B of three columns holds minimum, average, maximum taken 
   over the columns of A. */
{
   register double *Aj,Amax,Amean,Amin,d;
   double *A,*B;
   int cols,i=0,j,rows,rows2;

   if(is_sparse(tos)) dense();

   if(tos->typ!=MAT) {
      stkerr(" stats: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   rows2=2*rows;

   cols=tos->col;
   A=tos->mat;

   if(!matstk(rows,3,"_stats")) return 0;
   B=tos->mat;

   for(;i<rows;i++) {
      Aj=A+i;
      Amean=*Aj;
      Amin=Amean;
      Amax=Amean;
      Aj+=rows;
      for(j=1;j<cols;j++) {
         d=*Aj;
         Amax=MAX(Amax,d);
         Amin=MIN(Amin,d);
         Amean+=d;
         Aj+=rows;
      }
      *B=Amin;
      *(B+rows)=Amean/cols;
      *(B+rows2)=Amax;
      B++;
   }
   return(lop());
}

int stats1() /* stats1 (hA --- hB) */
/* Fri Apr 23 12:18:43 PDT 2010

   Matrix B of four columns holds minimum, average, maximum and
   variance taken over the columns (i.e., along each row) of A. 

   Returned "variance" is an unbiased estimate of the variance for
   n independent samples, obtained by dividing by n - 1 for n >= 2:
      v = (Sum[(x(k) - xmean)^2])/(n - 1)

   Reference: Sokolnikoff, I. S., and R. M. Redheffer, "Mathematics
      of Physics and Modern Engineering," 1958, pp. 669-670.

   Check.

      [tops@plunger] ready > >> A = [1 ; 2 ; 4]*[1 , 2 , 3];

      >> .m(A); nl;
       Row 1:        1        2        3
       Row 2:        2        4        6
       Row 3:        4        8       12

      >> .m(stats1(A));
                    min      mean     max      var
       Row 1:        1        2        3        1
       Row 2:        2        4        6        4
       Row 3:        4        8       12       16
      >> 

      Results for the rows of A are correct by inspection. */
{
   register double *Aj,Amax,Amean,Amin,Avar,d;
   double *A,*B,*Bmean,*Bvar;
   int cols,i=0,j,n,rows,rows2;

   if(is_sparse(tos)) dense();

   if(tos->typ!=MAT) {
      stkerr(" stats1: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   rows2=2*rows;

   cols=tos->col;
   A=tos->mat;

   if(!matstk(rows,4,"_stats1")) return 0;
   B=tos->mat;

   for(;i<rows;i++) {
      Aj=A+i;
      Amean=*Aj;
      Amin=Amax=Amean;
      Aj+=rows;
      for(j=1;j<cols;j++) {
         d=*Aj;
         Amax=MAX(Amax,d);
         Amin=MIN(Amin,d);
         Amean+=d;
         Aj+=rows;
      }
      *B=Amin;
      *(B+rows)=Amean/cols;
      *(B+rows2)=Amax;
      B++;
   }
   B=tos->mat;
   Bmean=B+locvec(1,rows);
   Bvar=B+locvec(3,rows);
   n=MAX(1,cols-1);

   for(i=0;i<rows;i++) {
      Aj=A+i;
      Amean=*Bmean;
      Avar=*Aj-Amean;
      Avar*=Avar;
      Aj+=rows;
      for(j=1;j<cols;j++) {
         d=*Aj-Amean;
         Avar+=d*d;
         Aj+=rows;
      }
      *Bvar=Avar/n;
      Bvar++;
      Bmean++;
   }
   return(lop());
}

int stats2() /* stats2 (hA hW --- hB) */
/* Wed Oct 17 04:48:47 PDT 2012

   Same as stats1() except terms in A are weighted by terms in W.

   Matrix B of four columns holds minimum, average, maximum and
   variance taken over the columns (i.e., along each row) of A
   weighted by columns of W.

   Returned "variance" is an unbiased estimate of the variance for
   n independent samples, obtained by dividing by n - 1 for n >= 2:
      v = (Sum[(x(k) - xmean)^2])/(n - 1)

   Reference: Sokolnikoff, I. S., and R. M. Redheffer, "Mathematics
      of Physics and Modern Engineering," 1958, pp. 669-670.

   Check using 3-by-3 matrix A from stats1() (see above).

      [tops@plunger] ready > >> A = [1 ; 2 ; 4]*[1 , 2 , 3];
      >> .m(A); nl; nl;
       Row 1:        1        2        3
       Row 2:        2        4        6
       Row 3:        4        8       12

   // Run stats1() on A cloned 1000 times to make C of 3000 columns:
      >> C = clone(A, 1000);

      >> .m(park(dims(C))); // C is a 3-by-3000 matrix:
       Row 1:        3     3000 

      >> .m(stats1(C)); nl; nl;
                    min      mean     max   var
       Row 1:        1        2        3   0.6669
       Row 2:        2        4        6    2.668
       Row 3:        4        8       12    10.67

   // Run stats2() on 3-by-3 A with 3-by-3 weights W valued at 1000:
      >> W = fill(1000, dims(A));
      >> .m(W); nl; nl;
       Row 1:     1000     1000     1000
       Row 2:     1000     1000     1000
       Row 3:     1000     1000     1000

      >> .m(stats2(A, W));
                    min      mean     max   var
       Row 1:        1        2        3   0.6669
       Row 2:        2        4        6    2.668
       Row 3:        4        8       12    10.67

      Results from stats1() and stats2() agree.  This test case is not
      sufficient because all terms in W are the same.  Other tests with
      different terms in W uncovered a bug where Wj was not incremented
      properly. */
{
   register double *Aj,Amax,Amean,Amin,Avar,d,*Wj;
   double *A,*B,*Bmean,*Bvar,S,*W;
   int cols,i=0,j,rows,rows2;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" stats2: ",MATNOT);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row || (cols=tos->col)!=(tos-1)->col) {
      stkerr(" stats2: ",MATSNOTC);
      return 0;
   }
   rows2=2*rows;
   A=(tos-1)->mat;
   W=tos->mat;

   if(!matstk(rows,4,"_stats2")) return 0;
   B=tos->mat;

   for(;i<rows;i++) {
      Aj=A+i;
      Wj=W+i;
      Amean=*Aj*(S=*Wj);
      Amin=Amax=*Aj;
      Aj+=rows;
      Wj+=rows;
      for(j=1;j<cols;j++) {
         d=*Aj;
         Amax=MAX(Amax,d);
         Amin=MIN(Amin,d);
         Amean+=d*(*Wj);
         S+=*Wj;
         Aj+=rows;
         Wj+=rows;
      }
      *B=Amin;
      *(B+rows)=Amean/MAX(1,S);
      *(B+rows2)=Amax;
      B++;
   }
   B=tos->mat;
   Bmean=B+locvec(1,rows);
   Bvar=B+locvec(3,rows);

   for(i=0;i<rows;i++) {
      Aj=A+i;
      Wj=W+i;
      Amean=*Bmean;
      d=*Aj-Amean;
      Avar=d*d*(S=*Wj);
      Aj+=rows;
      Wj+=rows;
      for(j=1;j<cols;j++) {
         d=*Aj-Amean;
         Avar+=d*d*(*Wj);
         S+=*Wj;
         Aj+=rows;
         Wj+=rows;
      }
      *Bvar=Avar/MAX(1,S-1);
      Bvar++;
      Bmean++;
   }
   return(lop() && lop());
}

double _tan(double x) { return(tan(x)); }

int tan1() { return(byterm1(TAN)); } /* tan (hA --- hA1) */

double _tanh(double x) { return(tanh(x)); }

int tanh1() { return(byterm1(TANH)); } /* tanh (hA --- hA1) */

double _tenpow(double x) { return(pow((double)10,x)); }

int tenpow() /* 10^ (hA --- hA1) */
{
   if(is_complex(tos)) {
      stkerr(" 10^: ",REALNOT);
      return 0;
   }
   return(byterm1(TENPOW)); 
}

int tic() /* tic (x --- x+1) */
{
   if(tos->typ==NUM) {
      tos->real++;
      tos->nam=NULL; /* wipe out lib name if any */
      return 1;
   }
   pushint(1);
   return(plus());
}
  
int totals() /* totals (hA --- hV)  down (hA --- hV) */
/* Returning vector V containing the totals of each column in A.  Row
   (element) n of V holds the total of values in column n of A. */
{
   register double *A,*V,x,y;
   register int i=0,j=0,k=0;
   register int rA,cA;
   int nw=1,TAG;
  
   if(is_sparse(tos)) dense();

   if(tos->typ==MAT) {
      TAG=tos->tag;

      if(is_complex(tos)) nw=2;

      rA=tos->row;
      cA=tos->col;
      if(rA==0) return(drop() && matstk(rA,cA,"_totals"));

      if((V=(double *)memget(cA*nw,1))==NULL) {
         return 0;
      }
      A=tos->mat;
      if(nw!=2) { /* real MAT: */
         for(;j<cA;j++) {
            x=0;
            for(i=0;i<rA;i++) {
               x+=(*(A+k));
               k++;
            }
            *(V+j)=x;
         }
      }
      else { /* complex MAT: */
         for(;j<cA*nw;j++) {
            x=0;
            y=0;
            for(i=0;i<rA;i++) {
               x+=(*(A+k));
               k++;
               y+=(*(A+k));
               k++;
               i++;
            }
            *(V+j)=x;
            j++;
            *(V+j)=y;
         }
      }
      return(
         drop() &&
         push(MAT,(char *)memgetn("_totals",7),TAG,0,V,NULL,cA*nw,1,\
            NULL)
      );
   }
   stkerr(" totals: ",MATNOT);
   return 0;
}

int trandom() /* trandom (rows bytes --- hT) */
/* A block of random bytes in a volume. */
{
   char *name="_trandom";
   int bytes,cols,rows;
   int little=1234;
   int big=4321;

   if(!popint(&bytes) || !popint(&rows)) return 0;

   cols=1+bytes/sizeof(double);

   pushint(rows); pushint(cols); random1();
   pushint(little); export8(); typvol2mat();

   pushint(rows); pushint(cols); random1();
   pushint(big); export8(); typvol2mat();

   xor(); bend(); typmat2vol();
   pushint(XBASE); pushint(bytes); ndx(); crop();

   pushq2(name,strlen(name)); naming();

   return 1;
}

int trandom1() /* trandom1 (rows bytes --- hV) */
/* Thu Mar 13 13:48:53 PDT 2014

   Create rows-by-bytes volume V containing random bits.

   Bits in each row are are numbered from left to right, starting 
   with 1st. 

   Expressions from ga_random() (file src/wapp.c) are used here
   to create each row of 8*bytes random bits. */
{
   char *V;
   int bit,byte,k,n;
   unsigned char mask[8]={128,64,32,16,8,4,2,1};
   int bytes,r=0,rows;

   if(!popint(&bytes)) return 0;
   if(!popint(&rows)) return 0;
   bytes=MAX(0,bytes);
   n=8*bytes;

   if(!volstk(rows,bytes,"_trandom1")) {
      stkerr(" trandom1: ",MEMNOT);
      return 0;
   }
   V=tos->tex;
   memset(V,0,rows*bytes);

   for(;r<rows;r++) {
      for(k=0;k<n;k++) {
         if(rand0(&SEED)>0.5) {
            bit=k;
            byte=bit>>3; /* divide by 8 to get byte in V */
            bit-=byte<<3; /* bit local within byte = mask index */
            *(V+byte)|=*(mask+bit); /* turn bit on */
         }
      }
      V+=bytes;
   }
   return 1;
}

int true() { return(pushint(xTRUE)); } /* true ( --- true) */

double _twopow(double x) { return(pow((double)2,x)); }

int twopow() { return(byterm1(TWOPOW)); } /* 2^ (hA --- hA1) */

int ureal2() /* u2real (hI --- hA) */
/* 2-byte unsigned int to 8-byte fp */
{
   int f=0,ret=0;

   if(tos->typ==STR) {
      hand();
      if(tos->col==2) f=1;
   }
   ret=(byteorder1() && uimport2());
   if(f && ret)
      ret=(
         ret &&
         pushint(XBASE) &&
         pry()
      );
   return(ret);
} 

int ureal4() /* u4real (hI --- hA) */
/* 4-byte unsigned int to 8-byte fp */
{
   int f=0,ret=0;

   if(tos->typ==STR) {
      hand();
      if(tos->col==4) f=1;
   }
   ret=(byteorder1() && uimport4());
   if(f && ret)
      ret=(
         ret &&
         pushint(XBASE) &&
         pry()
      );
   return(ret);
}

int ureal2_() { return(byterm1(UREAL2)); } /* u2real (hI --- hA) */
int ureal4_() { return(byterm1(UREAL4)); } /* u4real (hI --- hA) */

double _ureal2(unsigned short x)
/* Converting 2-byte unsigned integer bit pattern into 8-byte floating
   point bit pattern. */
{  
   return((double)x); 
}

double _ureal4(unsigned long x)
/* Converting 4-byte unsigned integer bit pattern into 8-byte floating 
   point bit pattern. */
{  
   return((double)x); 
}

int urn() /* urn ( --- x) */
{
   return(pushd(rand0(&SEED)));
}

int vqsort(double *A1, int len)
/*
   Fri Aug 27 13:00:07 PDT 2010.

   Very quick ascending order sort where doubles in A1 are integer 
   numbers.

   Returns 1 if A1 has been sorted, 0 if not.

   Restricted to range of 65534 between highest and lowest values in
   A1.  If this restriction is not met, 0 is returned and the calling 
   function can use another method. 

   Tests of this function are shown in the notes for qsort1() in this 
   file. */
{
   double *A;
   double Amax=-INF,Amin=INF;
   unsigned short u,*U,*U1,*V,*V1;
   int k=0,m=0,n,ulen=0,ulim=65535;

   A=A1;
   for(;k<len;k++) {
      if(*A<Amin) Amin=*A;
      if(*A>Amax) Amax=*A;
      A++;
   }
   ulen=(int)(Amax-Amin+1);
   if(ulen>ulim-1) return 0;

   U1=malloc(sizeof(unsigned short)*(1+ulen));
   V1=malloc(sizeof(unsigned short)*(1+ulen));
   if(U1==NULL || V1==NULL) {
      stkerr(" vqsort: ",MEMNOT);
      return 0;
   }
   memset(V1,0,sizeof(unsigned short)*(1+ulen));

   A=A1;
   for(k=0;k<len;k++) {
      u=*A-Amin;
      *(U1+u)=u;
      *(V1+u)+=1;
      A++;
   }
   A=A1;
   U=U1;
   V=V1;
   for(;m<ulen;m++) {
      if((n=*V)) {
         u=*U;
         for(k=0;k<n;k++) {
            *A=u+Amin;
            A++;
         }
      }
      U++;
      V++;
   }
   mallfree((void *)&U1);
   mallfree((void *)&V1);

   return 1;
}

int VQSORTset() /* VQSORT (f --- ) */
/* Sat Aug 28 10:37:05 PDT 2010
   Set flag to yes to enable use of vqsort(); flag is set to no by
   the program after every use of vqsort(). */
{
   int f;

   if(!popint(&f)) return 0;

   VQSORT=(f!=0);
   return 1;
}

int xor() { return(byterm2(XOR)); } /* xor (hA hB --- hC) */

double _xor(double x, double y)
{ 
   X.x=x; Y.y=y; X.c[0]=(X.c[0] ^ Y.d[0]); X.c[1]=(X.c[1] ^ Y.d[1]);
   return X.x;
}

int zeroeq() /* 0= (hA --- hF) */
{ 
   if(is_complex(tos)) {
      return(
         cmplxdbl() &&
         byterm1(EQ0) && /* imaginary part 0= */
         swap() && 
         byterm1(EQ0) && /* real part 0= */
         byterm2(AND)
      );
   }
   return(byterm1(EQ0)); 
} 

double _zeroeq(double x) { return(xTRUE*(x==0)); }

int zerogt() { return(byterm1(GT0)); } /* 0> (hA --- hF) */

double _zerogt(double x) { return(xTRUE*(x>0)); }

int zerolt() { return(byterm1(LT0)); } /* 0< (hA --- hF) */

double _zerolt(double x) { return(xTRUE*(x<0)); }

int zerone() { return(byterm1(NE0)); } /* 0<> (hA --- hF) */

double _zerone(double x) { return(xTRUE*(x!=0)); }
