/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author and copyright holder of loadtx.c:  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
#ifdef DYNLOAD

#ifndef __STRICT_ANSI__
   #define __STRICT_ANSI__
#endif
#define _XOPEN_SOURCE 500 /* snprintf */
#include <stdio.h>

#include <stdlib.h>
#include <string.h>        /* strlen, strncmp */
#include <sys/types.h>
#include <dirent.h>        /* opendir, readdir, closedir      */
#include <dlfcn.h>         /* dlopen, dlerror, dlsym, dlclose */

#include "stk.h"
#include "main.h"
#include "inpo.h"          /* gprintf */
#include "lib.h"

#define StrSize 200
int loadtx();

int loadtx() 
/* Search directories in the dynamic load path for files that end with ".tx"
 * then try to load them and add their functions to the catalog of known words.
 * Assumes the function name is the same as the file name minus the extension.
 */
{
int DEBUG = 0;
    DIR   *dirp;
    struct dirent *dp;
    char  *name = "loadtx ";
    char  *ext  = ".tx";    /* dynamically load files with this extension */
    char  *path;            /* to hold a copy of getenv("LD_LIBRARY_PATH") */
    char  *dir_name;
    char   error_msg[StrSize - 1];
    int    nFound, len, ld_path_len, len_ext, success;

    char               **fn_name;
    void               **fn_handle;
    unsigned long (**fn_symbol)();
if (DEBUG) printf("top of %s LD_LIBRARY_PATH=[%s]\n", 
name, getenv("LD_LIBRARY_PATH"));

    if (!getenv("LD_LIBRARY_PATH")) return 1; /* this environment variable */
                                              /* isn't defined; exit       */
    len_ext = strlen(ext);
    nFound  = 0;

    /* Pass 1:  count the number of .tx files in the search path {{{1
     *
     * Need to make a copy of getenv("LD_LIBRARY_PATH") and work with it
     * because two instances of the line
     *        dir_name = strtok(getenv("LD_LIBRARY_PATH"), ":");
     * (one for each pass) don't seem to work.
     */
    ld_path_len = strlen(getenv("LD_LIBRARY_PATH"));
    if ((path = (char *) malloc((ld_path_len+1) * sizeof(char))) == NULL) {
        stkerr(" loadtx (path): "  ,MEMNOT);
    }
    strncpy(path, getenv("LD_LIBRARY_PATH"), ld_path_len);
    path[ld_path_len] = '\0';
    dir_name  = strtok(path, ":"); 
    /* loop over colon-delimited directory names in the dynamic load path */
    while (dir_name != NULL) {
        dirp    = opendir(dir_name);
        /* loop over all the files in the current directory */
        while (dirp) {
            if ((dp = readdir(dirp)) != NULL) {
                len = strlen(dp->d_name);
                if ((len > 3) &&    /* don't try to load just '.tx' */
                    !strncmp(&dp->d_name[len-len_ext], ext, len_ext)) {
                    ++nFound;
                }
            } else { /* no more files in this directory or cannot read directory */
                break;
            }
        }
        closedir(dirp);
        dir_name = strtok(NULL, ":");
    } 
if (DEBUG) printf("%s pass 1 found %d .tx files\n", name, nFound);
    /* 1}}} */

    if (!nFound) {
        /* didn't find any .tx files to load */
        free(path);
        return 1;
    }

    /* allocate memory for the function names, handles, and symbols {{{1 */
    if ((fn_name   = (char **) malloc(nFound * sizeof(char *))) == NULL) {
        stkerr(" loadtx (fn_name): "  ,MEMNOT);
    }
    if ((fn_handle = (void **) malloc(nFound * sizeof(void *))) == NULL) {
        stkerr(" loadtx (fn_handle): ",MEMNOT);
    }
    if ((fn_symbol = (unsigned long (**)())
                     malloc(nFound * sizeof(unsigned long *))) == NULL) {
        stkerr(" loadtx (fn_symbol): ",MEMNOT);
    }
    /* 1}}} */

    /* Pass 2:  try to load all the .tx files in the search path  {{{1 */
    nFound = 0;
    strncpy(path, getenv("LD_LIBRARY_PATH"), ld_path_len);
    path[ld_path_len] = '\0';
    dir_name = strtok(path, ":");
    while (dir_name != NULL) {

if (DEBUG) printf("%s looking in directory [%s]\n", name, dir_name);
        dirp    = opendir(dir_name);
        /* loop over all the files in the current directory */
        while (dirp) {
            if ((dp = readdir(dirp)) != NULL) {
                len = strlen(dp->d_name);
if (DEBUG) printf("%s checking [%s] len=%d nFound=%d\n", 
name, dp->d_name, len, nFound);
                if ((len > 3) &&    /* don't try to load just '.tx' */
                    !strncmp(&dp->d_name[len-len_ext], ext, len_ext)) {
                    /*
                     * dp->d_name = name of shared object file, eg "hello.tx"
                     * fn_name    = name of function,           eg "hello"
                     */
                    if ((fn_name[nFound] = (char *) malloc(
                                           (len+1) * sizeof(char))) == NULL) {
                        stkerr(" loadtx (fn_name): ",MEMNOT);
                    }
                    fn_handle[nFound] = dlopen(dp->d_name, RTLD_LAZY);
                    strncpy(fn_name[nFound], dp->d_name, 
                            (len = MIN(StrSize, len - len_ext)));
                    *(fn_name[nFound]+len) = '\0'; /* null terminate for catadd() */

                    if (fn_handle[nFound]) {
gprintf("found shared object file [%s]; adding function [%s] to catalog", 
dp->d_name, fn_name[nFound]);
nc();
                        fn_symbol[nFound] = dlsym(fn_handle[nFound], 
                                                  fn_name[nFound]);
                        success = catadd(NATI, tagnative(fn_name[nFound]), fn_symbol[nFound], 
                                         NULL, 0);
/*
                        Need to call this before exiting tops:
                        dlclose(fn_handle[nFound]);
*/
                        if (success) {
                            ++nFound;
                        } else {
                            snprintf(error_msg, StrSize,
                                     "Failed to add dyn function %s to catalog\n",
                                     fn_name[nFound]);
                            stkerr(name, error_msg);
                        }
                    } else {
                        /* found a .tx file but was unable to load it */
                        gprintf("%s Dynamic Load Error:", name);
                        nc();
                        gprintf("%s %s", name, dlerror());
                        nc();
                        nc();
                        gprintf("%s Diagnostics:", name);
                        nc();
                        gprintf("%s Is the file '%s%s' in a directory "
                                "defined by $LD_LIBRARY_PATH?", 
                                name, fn_name[nFound], ext);
                        nc();
                        gprintf("%s If there are undefined symbols, was tops "
                                "built with the -D pragmas that enable "
                                "compilation of the needed functions?",
                                name);
                        nc();
                    }
                }
            } else { /* no more files in this directory or cannot read directory */
                break;
            }
        }
        closedir(dirp);
if (DEBUG) printf("%s: found %d files with %s extension\n", name, nFound, ext);

        dir_name = strtok(NULL, ":");
    } /* 1}}} */
    free(path);
    return 1;
}
#endif
