/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#ifdef LAPACK
/* lapack.c

   D. R. Williamson  November 3, 2001

   Words that run LAPACK subroutines.

----------------------------------------------------------------------*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "lapack.h"
#include "mat.h"
#include "math1.h"
#include "mmath.h"
#include "mem.h"
#include "ordering.h"
#include "sys.h"
#include "tag.h"
#include "tex.h"

/*--------------------------------------------------------------------*/

int dgeev1() /* dgeev (hC --- hLr hLi hAr hAi) {{{1 */
/* Eigenanalysis of real matrix C.  Resulting eigenvalues and eigen-
   vectors are complex. 

      SUBROUTINE DGEEV( JOBVL, JOBVR, N, A, LDA, WR, WI, VL, LDVL, VR,
     $                  LDVR, WORK, LWORK, INFO )
*
*  -- LAPACK driver routine (version 3.0) --
*     Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
*     Courant Institute, Argonne National Lab, and Rice University
*     December 8, 1999
*
*     .. Scalar Arguments ..
      CHARACTER          JOBVL, JOBVR
      INTEGER            INFO, LDA, LDVL, LDVR, LWORK, N
*     ..
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), VL( LDVL, * ), VR( LDVR, * ),
     $                   WI( * ), WORK( * ), WR( * )
*     ..
*
*  Purpose
*  =======
*
*  DGEEV computes for an N-by-N real nonsymmetric matrix A, the
*  eigenvalues and, optionally, the left and/or right eigenvectors.
*
*  The right eigenvector v(j) of A satisfies
*                   A * v(j) = lambda(j) * v(j)
*  where lambda(j) is its eigenvalue.
*  The left eigenvector u(j) of A satisfies
*                u(j)**H * A = lambda(j) * u(j)**H
*  where u(j)**H denotes the conjugate transpose of u(j).
*
*  The computed eigenvectors are normalized to have Euclidean norm
*  equal to 1 and largest component real.
*
*  Arguments

*  =========
*
*  JOBVL   (input) CHARACTER*1
*          = 'N': left eigenvectors of A are not computed;
*          = 'V': left eigenvectors of A are computed.
*
*  JOBVR   (input) CHARACTER*1
*          = 'N': right eigenvectors of A are not computed;
*          = 'V': right eigenvectors of A are computed.
*
*  N       (input) INTEGER
*          The order of the matrix A. N >= 0.
*
*  A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
*          On entry, the N-by-N matrix A.
*          On exit, A has been overwritten.
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,N).
*
*  WR      (output) DOUBLE PRECISION array, dimension (N)
*  WI      (output) DOUBLE PRECISION array, dimension (N)
*          WR and WI contain the real and imaginary parts,
*          respectively, of the computed eigenvalues.  Complex
*          conjugate pairs of eigenvalues appear consecutively
*          with the eigenvalue having the positive imaginary part
*          first.
*
*  VL      (output) DOUBLE PRECISION array, dimension (LDVL,N)
*          If JOBVL = 'V', the left eigenvectors u(j) are stored one
*          after another in the columns of VL, in the same order
*          as their eigenvalues.
*          If JOBVL = 'N', VL is not referenced.
*          If the j-th eigenvalue is real, then u(j) = VL(:,j),
*          the j-th column of VL.
*          If the j-th and (j+1)-st eigenvalues form a complex
*          conjugate pair, then u(j) = VL(:,j) + i*VL(:,j+1) and
*          u(j+1) = VL(:,j) - i*VL(:,j+1).
*
*  LDVL    (input) INTEGER
*          The leading dimension of the array VL.  LDVL >= 1; if
*          JOBVL = 'V', LDVL >= N.
*
*  VR      (output) DOUBLE PRECISION array, dimension (LDVR,N)
*          If JOBVR = 'V', the right eigenvectors v(j) are stored one
*          after another in the columns of VR, in the same order
*          as their eigenvalues.
*          If JOBVR = 'N', VR is not referenced.
*          If the j-th eigenvalue is real, then v(j) = VR(:,j),
*          the j-th column of VR.
*          If the j-th and (j+1)-st eigenvalues form a complex
*          conjugate pair, then v(j) = VR(:,j) + i*VR(:,j+1) and
*          v(j+1) = VR(:,j) - i*VR(:,j+1).
*
*  LDVR    (input) INTEGER
*          The leading dimension of the array VR.  LDVR >= 1; if
*          JOBVR = 'V', LDVR >= N.
*
*  WORK    (workspace/output) DOUBLE PRECISION array, dimension (LWORK)
*          On exit, if INFO = 0, WORK(1) returns the optimal LWORK.
*
*  LWORK   (input) INTEGER
*          The dimension of the array WORK.  LWORK >= max(1,3*N), and
*          if JOBVL = 'V' or JOBVR = 'V', LWORK >= 4*N.  For good
*          performance, LWORK must generally be larger.
*
*          If LWORK = -1, then a workspace query is assumed; the routine
*          only calculates the optimal size of the WORK array, returns
*          this value as the first entry of the WORK array, and no error
*          message related to LWORK is issued by XERBLA.
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value.
*          > 0:  if INFO = i, the QR algorithm failed to compute all the
*                eigenvalues, and no eigenvectors have been computed;
*                elements i+1:N of WR and WI contain eigenvalues which
*                have converged.
*/
{
   double *Ai=NULL, *Ar=NULL, *Aux, *C, *Li, *Lr, *Mag;
   double *lapack_eigvec;
   char    chr_N       = 'N';
   char    chr_V       = 'V';
   double  dummy[1];
   double  double_0    =  0.0;
   double  double_m1   = -1.0;
   int     Aux_size, info;
   int     int_1       =   1;
   int     i=0,j=0,rows;
   double LWORK;

   if(tos->typ!=MAT) {
      stkerr(" dgeev: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   if(rows!=tos->col) {
      stkerr(" dgeev: ",SQUNOT);
      return 0;
   }

/* Workspace size query: */
   Aux_size=-1;
   DGEEV(&chr_N, &chr_V, &rows, tos->mat, &rows, NULL, NULL,
         NULL, &int_1, NULL, &rows, &LWORK, &Aux_size, &info);

   Aux_size=MAX((double)(4*rows),LWORK);

   if(TRACE) { /* show workspace size and return */
      gprintf(" dgeev1: optimal workspace bytes = %d", \
         Aux_size*sizeof(double));
      nc();
      traceoff1(); /* turn off trace */
      return 1;
   }
   cop(); /* C will be overwritten, so use a copy */
   C=tos->mat;

   if(!matstk(rows,1,"_Lr")) return 0;
   Lr=tos->mat;
   if(!matstk(rows,1,"_Li")) return 0;
   Li=tos->mat;

   /* workspace array: */
   if(!matstk(Aux_size,1,"_Aux")) return 0;
   Aux=tos->mat;

/* Computing complex eigenvalues and eigenvectors: */
   if(!matstk(rows,rows,"_Ar")) return 0;
   lapack_eigvec=tos->mat;

   DGEEV(&chr_N, &chr_V, &rows, C, &rows, Lr, Li,
         dummy, &int_1, lapack_eigvec, &rows, Aux, &Aux_size, &info);

   lop();                              /* dropping Aux from the stack */
   pushstr("3 roll drop"); xmain(0); /* dropping C from the stack: */
   
   if(!matstk(rows,rows,"_Ai")) return 0;
   Ai=tos->mat;

/* Store the imaginary portions of the complex conjugates: */
   for (j = 0; j < rows; j++) {
       if (Li[j] == 0.0) {
          /* zero out the imaginary part */
          DSCAL( &rows, &double_0, &Ai[j*rows], &int_1);
       } else {
          /* imag part of eigenvector j */
          DCOPY(&rows,&lapack_eigvec[(j+1)*rows],&int_1, &Ai[j*rows], &int_1);
          j++;
          /* imag part of eigenvector j+1 */
          DCOPY(&rows,&lapack_eigvec[(j  )*rows],&int_1, &Ai[j*rows],&int_1);
          DSCAL(&rows, &double_m1,                       &Ai[j*rows],&int_1);
       }
   }

/* Arrange lapack_eigvec into Ar: */
   Ar=lapack_eigvec;
   for (j = 0; j < rows; j++) {
       if (!(Li[j] == 0.0)) {
          j++;
          /* real part of eigenvector j+1 */
          DCOPY(&rows,&lapack_eigvec[(j-1)*rows],&int_1, &Ar[j*rows],  &int_1);
       }
   }
   if(!matstk(rows,1,"_Mag")) return 0;
   Mag=tos->mat;

   for(i=0;i<rows;i++) {
      *(Mag+i)=pow(*(Lr+i)*(*(Lr+i)) + *(Li+i)*(*(Li+i)),0.5);
   }
   /* get the ascending sort sequence based on eigenvalue magnitudes */
   sort_seq(); /* stk: Lr Li Ar Ai S */

   /* now reorder each part of the eigensolution in-place */
   pushstr("1 pick reorder drop"); xmain(0);    /* Ai */
   pushstr("2 pick reorder drop"); xmain(0);    /* Ar */
   pushstr("3 pick reorder drop"); xmain(0);    /* Li */
   pushstr("4 pick reorder drop"); xmain(0);    /* Lr */

   /* finish off by dropping the sort term hS from the stack */
   drop();

   return(1);
} /* 1}}} */
/*--------------------------------------------------------------------*/

int dgeev2() /* dgeev2 (hC --- hWr hWi hLr hLi hRr hRi) {{{1 */
/* Eigenanalysis of real matrix C.  Resulting eigenvalues and eigen-
   vectors are complex. 
   On return:
      Wr and Wi contain the real and imaginary parts of the eigenvalues.
      Lr and Li contain the real and imaginary parts of the left hand
         eigenvectors.
      Rr and Ri contain the real and imaginary parts of the right hand
         eigenvectors.

      SUBROUTINE DGEEV( JOBVL, JOBVR, N, A, LDA, WR, WI, VL, LDVL, VR,
     $                  LDVR, WORK, LWORK, INFO )
*
*  -- LAPACK driver routine (version 3.0) --
*     Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
*     Courant Institute, Argonne National Lab, and Rice University
*     December 8, 1999
*
*     .. Scalar Arguments ..
      CHARACTER          JOBVL, JOBVR
      INTEGER            INFO, LDA, LDVL, LDVR, LWORK, N
*     ..
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), VL( LDVL, * ), VR( LDVR, * ),
     $                   WI( * ), WORK( * ), WR( * )
*     ..
*
*  Purpose
*  =======
*
*  DGEEV computes for an N-by-N real nonsymmetric matrix A, the
*  eigenvalues and, optionally, the left and/or right eigenvectors.
*
*  The right eigenvector v(j) of A satisfies
*                   A * v(j) = lambda(j) * v(j)
*  where lambda(j) is its eigenvalue.
*  The left eigenvector u(j) of A satisfies
*                u(j)**H * A = lambda(j) * u(j)**H
*  where u(j)**H denotes the conjugate transpose of u(j).
*
*  The computed eigenvectors are normalized to have Euclidean norm
*  equal to 1 and largest component real.
*
*  Arguments

*  =========
*
*  JOBVL   (input) CHARACTER*1
*          = 'N': left eigenvectors of A are not computed;
*          = 'V': left eigenvectors of A are computed.
*
*  JOBVR   (input) CHARACTER*1
*          = 'N': right eigenvectors of A are not computed;
*          = 'V': right eigenvectors of A are computed.
*
*  N       (input) INTEGER
*          The order of the matrix A. N >= 0.
*
*  A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
*          On entry, the N-by-N matrix A.
*          On exit, A has been overwritten.
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,N).
*
*  WR      (output) DOUBLE PRECISION array, dimension (N)
*  WI      (output) DOUBLE PRECISION array, dimension (N)
*          WR and WI contain the real and imaginary parts,
*          respectively, of the computed eigenvalues.  Complex
*          conjugate pairs of eigenvalues appear consecutively
*          with the eigenvalue having the positive imaginary part
*          first.
*
*  VL      (output) DOUBLE PRECISION array, dimension (LDVL,N)
*          If JOBVL = 'V', the left eigenvectors u(j) are stored one
*          after another in the columns of VL, in the same order
*          as their eigenvalues.
*          If JOBVL = 'N', VL is not referenced.
*          If the j-th eigenvalue is real, then u(j) = VL(:,j),
*          the j-th column of VL.
*          If the j-th and (j+1)-st eigenvalues form a complex
*          conjugate pair, then u(j) = VL(:,j) + i*VL(:,j+1) and
*          u(j+1) = VL(:,j) - i*VL(:,j+1).
*
*  LDVL    (input) INTEGER
*          The leading dimension of the array VL.  LDVL >= 1; if
*          JOBVL = 'V', LDVL >= N.
*
*  VR      (output) DOUBLE PRECISION array, dimension (LDVR,N)
*          If JOBVR = 'V', the right eigenvectors v(j) are stored one
*          after another in the columns of VR, in the same order
*          as their eigenvalues.
*          If JOBVR = 'N', VR is not referenced.
*          If the j-th eigenvalue is real, then v(j) = VR(:,j),
*          the j-th column of VR.
*          If the j-th and (j+1)-st eigenvalues form a complex
*          conjugate pair, then v(j) = VR(:,j) + i*VR(:,j+1) and
*          v(j+1) = VR(:,j) - i*VR(:,j+1).
*
*  LDVR    (input) INTEGER
*          The leading dimension of the array VR.  LDVR >= 1; if
*          JOBVR = 'V', LDVR >= N.
*
*  WORK    (workspace/output) DOUBLE PRECISION array, dimension (LWORK)
*          On exit, if INFO = 0, WORK(1) returns the optimal LWORK.
*
*  LWORK   (input) INTEGER
*          The dimension of the array WORK.  LWORK >= max(1,3*N), and
*          if JOBVL = 'V' or JOBVR = 'V', LWORK >= 4*N.  For good
*          performance, LWORK must generally be larger.
*
*          If LWORK = -1, then a workspace query is assumed; the routine
*          only calculates the optimal size of the WORK array, returns
*          this value as the first entry of the WORK array, and no error
*          message related to LWORK is issued by XERBLA.
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value.
*          > 0:  if INFO = i, the QR algorithm failed to compute all the
*                eigenvalues, and no eigenvectors have been computed;
*                elements i+1:N of WR and WI contain eigenvalues which
*                have converged.
*/  
{
   double *Aux,*C,*Li,*Lr,*Mag,*Rr,*Ri,*Vl,*Vr,*Wi,*Wr;
   char    chr_V       = 'V';
   double  double_0    =  0.0;
   double  double_m1   = -1.0;
   int     Aux_size, info;
   int     int_1       =   1;
   int     i=0,j=0,rows;
   double LWORK;

   if(tos->typ!=MAT) {
      stkerr(" dgeev2: ",MATNOT);
      return 0;
   }
   rows=tos->row;
   if(rows!=tos->col) {
      stkerr(" dgeev2: ",SQUNOT);
      return 0;
   }

/* Workspace size query: */
   Aux_size=-1;
   DGEEV(&chr_V, &chr_V, &rows, tos->mat, &rows, NULL, NULL,
         NULL, &rows, NULL, &rows, &LWORK, &Aux_size, &info);

   Aux_size=MAX((double)(4*rows),LWORK);

   if(TRACE) { /* show workspace size and return */
      gprintf(" dgeev2: optimal workspace bytes = %d", \
         Aux_size*sizeof(double));
      nc();
      traceoff1(); /* turn off trace */
      return 1;
   }
   cop(); /* C will be overwritten, so use a copy */
   C=tos->mat;

   if(!matstk(rows,1,"_Wr")) return 0;
   Wr=tos->mat;
   if(!matstk(rows,1,"_Wi")) return 0;
   Wi=tos->mat;

   /* workspace array: */
   if(!matstk(Aux_size,1,"_Aux")) return 0;
   Aux=tos->mat;

/* Computing complex eigenvalues and eigenvectors: */
   if(!matstk(rows,rows,"_Lr")) return 0;
   Vl=tos->mat;

   if(!matstk(rows,rows,"_Rr")) return 0;
   Vr=tos->mat;

   DGEEV(&chr_V, &chr_V, &rows, C, &rows, Wr, Wi,
         Vl, &rows, Vr, &rows, Aux, &Aux_size, &info);

   rot(); drop();                      /* dropping Aux from the stack */
   pushstr("4 roll drop"); xmain(0);   /* dropping C from the stack   */

   lpush(); /* put Vr on the temporary stack */
   
/* Processing the left hand eigenvectors. */
   if(!matstk(rows,rows,"_Li")) return 0;
   Li=tos->mat;

/* Store the imaginary portions of the left hand complex conjugates: */
   for (j = 0; j < rows; j++) {
       if (Wi[j] == 0.0) {
          /* zero out the imaginary part */
          DSCAL( &rows, &double_0, &Li[j*rows], &int_1);
       } else {
          /* imag part of eigenvector j */
          DCOPY(&rows,&Vl[(j+1)*rows],&int_1, &Li[j*rows], &int_1);
          j++;
          /* imag part of eigenvector j+1 */
          DCOPY(&rows,&Vl[(j  )*rows],&int_1, &Li[j*rows],&int_1);
          DSCAL(&rows, &double_m1,            &Li[j*rows],&int_1);
       }
   }

/* Arrange Vl into Lr: */
   Lr=Vl;
   for (j = 0; j < rows; j++) {
       if (!(Wi[j] == 0.0)) {
          j++;
          /* real part of eigenvector j+1 */
          DCOPY(&rows,&Vl[(j-1)*rows],&int_1, &Lr[j*rows],  &int_1);
       }
   }

/* Processing the right hand eigenvectors. */
   lpull(); /* get Vr from the temporary stack */

   if(!matstk(rows,rows,"_Ri")) return 0;
   Ri=tos->mat;

/* Store the imaginary portions of the right hand complex conjugates: */
   for (j = 0; j < rows; j++) {
       if (Wi[j] == 0.0) {
          /* zero out the imaginary part */
          DSCAL( &rows, &double_0, &Ri[j*rows], &int_1);
       } else {
          /* imag part of eigenvector j */
          DCOPY(&rows,&Vr[(j+1)*rows],&int_1, &Ri[j*rows], &int_1);
          j++;
          /* imag part of eigenvector j+1 */
          DCOPY(&rows,&Vr[(j  )*rows],&int_1, &Ri[j*rows],&int_1);
          DSCAL(&rows, &double_m1,            &Ri[j*rows],&int_1);
       }
   }

/* Arrange Vr into Rr: */
   Rr=Vr;
   for (j = 0; j < rows; j++) {
       if (!(Wi[j] == 0.0)) {
          j++;
          /* real part of eigenvector j+1 */
          DCOPY(&rows,&Vr[(j-1)*rows],&int_1, &Rr[j*rows],  &int_1);
       }
   }
   if(!matstk(rows,1,"_Mag")) return 0;
   Mag=tos->mat;

   for(i=0;i<rows;i++) {
      *(Mag+i)=pow(*(Wr+i)*(*(Wr+i)) + *(Wi+i)*(*(Wi+i)),0.5);
   }
   /* get the ascending sort sequence based on eigenvalue magnitudes */
   sort_seq(); /* stk: Wr Wi Lr Li Rr Ri S */

   /* now reorder each part of the eigensolution in-place */
   pushstr("1 pick reorder drop"); xmain(0);    /* Ri */
   pushstr("2 pick reorder drop"); xmain(0);    /* Rr */
   pushstr("3 pick reorder drop"); xmain(0);    /* Li */
   pushstr("4 pick reorder drop"); xmain(0);    /* Lr */
   pushstr("5 pick reorder drop"); xmain(0);    /* Wi */
   pushstr("6 pick reorder drop"); xmain(0);    /* Wr */

   /* finish off by dropping the sort term hS from the stack */
   drop();

   return(1);
} /* 1}}} */
/*--------------------------------------------------------------------*/

int dgemm1() /* dgemm (hA trnA alpha hB trnB hC beta --- hD) {{{1 */
/*  Matrix multiplication and add: A*B+C=D 
    trnA true to use transpose of A
    A is scaled by alpha
    trnB true to use transpose of B
    C is scaled by beta
    Incoming C can be purged or null for: A*B=D

      SUBROUTINE DGEMM ( TRANSA, TRANSB, M, N, K, ALPHA, A, LDA, B, LDB,
     $                   BETA, C, LDC )

*  Purpose
*  =======
*
*  DGEMM  performs one of the matrix-matrix operations
*
*     C := alpha*op( A )*op( B ) + beta*C,
*
*  where  op( X ) is one of
*
*     op( X ) = X   or   op( X ) = X',
*
*  alpha and beta are scalars, and A, B and C are matrices, with op( A )
*  an m by k matrix,  op( B )  a  k by n matrix and  C an m by n matrix.
*
*  Parameters
*  ==========
*
*  TRANSA - CHARACTER*1.
*           On entry, TRANSA specifies the form of op( A ) to be used in
*           the matrix multiplication as follows:
*
*              TRANSA = 'N' or 'n',  op( A ) = A.
*
*              TRANSA = 'T' or 't',  op( A ) = A'.
*
*              TRANSA = 'C' or 'c',  op( A ) = A'.
*
*           Unchanged on exit.
*
*  TRANSB - CHARACTER*1.
*           On entry, TRANSB specifies the form of op( B ) to be used in
*           the matrix multiplication as follows:
*
*              TRANSB = 'N' or 'n',  op( B ) = B.
*
*              TRANSB = 'T' or 't',  op( B ) = B'.
*
*              TRANSB = 'C' or 'c',  op( B ) = B'.
*
*           Unchanged on exit.
*
*  ALPHA  - DOUBLE PRECISION.
*           On entry, ALPHA specifies the scalar alpha.
*           Unchanged on exit.
*
*  A      - DOUBLE PRECISION array of DIMENSION ( LDA, ka ), where ka is
*           k  when  TRANSA = 'N' or 'n',  and is  m  otherwise.
*           Before entry with  TRANSA = 'N' or 'n',  the leading  m by k
*           part of the array  A  must contain the matrix  A,  otherwise
*           the leading  k by m  part of the array  A  must contain  the
*           matrix A.
*           Unchanged on exit.
*
*
*  B      - DOUBLE PRECISION array of DIMENSION ( LDB, kb ), where kb is
*           n  when  TRANSB = 'N' or 'n',  and is  k  otherwise.
*           Before entry with  TRANSB = 'N' or 'n',  the leading  k by n
*           part of the array  B  must contain the matrix  B,  otherwise
*           the leading  n by k  part of the array  B  must contain  the
*           matrix B.
*           Unchanged on exit.
*
*  BETA   - DOUBLE PRECISION.
*           On entry,  BETA  specifies the scalar  beta.  When  BETA  is
*           supplied as zero then C need not be set on input.
*           Unchanged on exit.
*
*  C      - DOUBLE PRECISION array of DIMENSION ( LDC, n ).
*           Before entry, the leading  m by n  part of the array  C must
*           contain the matrix  C,  except when  beta  is zero, in which
*           case C need not be set on entry.
*           On exit, the array  C  is overwritten by the  m by n  matrix
*           ( alpha*op( A )*op( B ) + beta*C ).
*
*  LDC    - INTEGER.
*           On entry, LDC specifies the first dimension of C as declared
*           in  the  calling  (sub)  program.   LDC  must  be  at  least
*           max( 1, m ).
*           Unchanged on exit.
*/
{
/* dgemm (hA trnA alpha hB trnB hC beta --- hD) */

   double *A,*B,*C;
   double alpha,beta;
   int trnA,trnB;

   char TRANSA,TRANSB;
   char *Dname="_D";
   int cA,cB,rA,rB;
   int K,M,N;

   if(!(popd(&beta) && swap() && popbool(&trnB) && rot() && 
        popd(&alpha) && rot() && popbool(&trnA)))
      return 0;

   if(!(tos->typ==MAT && (tos-1)->typ==MAT && (tos-2)->typ==MAT)) {
      stkerr(" dgemm: ",MATNOT3);
      return 0;
   }
   A=(tos-2)->mat;
   rA=(tos-2)->row;
   cA=(tos-2)->col;

   B=(tos-1)->mat;
   rB=(tos-1)->row;
   cB=(tos-1)->col;

   if(!trnA) {
      TRANSA='N';
      M=rA;
      K=cA;
   }
   else {
      TRANSA='T';
      M=cA;
      K=rA;
   }
   if(!trnB) {
      TRANSB='N';
      N=cB;
   }
   else {
      TRANSB='T';
      N=rB;
   }
/* Compatibility tests: */
   if(TRANSA=='N' && TRANSB=='N' && (cA!=rB)) { 
      stkerr(" dgemm: ","columns of A must match rows of B");
      return 0;
   }
   if(TRANSA=='T' && TRANSB=='N' && (rA!=rB)) { 
      stkerr(" dgemm: ","rows of A must match rows of B");
      return 0;
   }
   if(TRANSA=='N' && TRANSB=='T' && (cA!=cB)) { 
      stkerr(" dgemm: ","columns of A must match columns of B");
      return 0;
   }
   if(TRANSA=='T' && TRANSB=='T' && (rA!=cB)) { 
      stkerr(" dgemm: ","rows of A must match columns of B");
      return 0;
   }
   if(tos->row==0 || tos->col==0) {
/*    C is purged, so put an uninitialized MAT on stack to hold the
      results: */
      drop(); 
      if(!matstk(M,N,Dname)) return 0;
      beta=0;
   }
   else {
      if(M!=(tos->row) || N!=(tos->col)) {
         stkerr(" dgemm: ","C is not compatible with A*B");
         return 0;
      }
/*    C is not purged, so it will be overwritten by results.  Make
      a copy: */
      cop();
      pushstr(Dname);
      naming();
   }
   C=tos->mat;

/* Long jump is used below to guard against DGEMM stopping the program 
   if there is a data error.  This approach is useful for any of the 
   Lapack functions; the jump return is called BLASX as shown below, 
   and function exit() will land back here if it is entered. */

   if(!bufup()) return 0; /* Up to next run level, onbuf */

   setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

   if(!*(jmpready+onbuf)) { /* falls through first time */
      *(jmpready+onbuf)=BLASX;

      DGEMM(&TRANSA, &TRANSB, &M, &N, &K, &alpha, A, &rA, B, &rB, \
         &beta, C, &M);
   }
   else { /* comes here on longjmp with error */
      stkerr(" dgemm: ","matrix size error in BLAS DGEMM");
      bufdn();
      return 0;
   }
   bufdn(); /* Down from higher run level */

   return(lop() && lop());
} /* 1}}} */
/*--------------------------------------------------------------------*/

int dgesv1() /* dgesv (hb hA --- hx) {{{1 */
/* Solve A*x=b for x.

   http://www.netlib.org/lapack/double/dgesv.f
      SUBROUTINE DGESV( N, NRHS, A, LDA, IPIV, B, LDB, INFO )
*  Purpose
*  =======
*
*  DGESV computes the solution to a real system of linear equations
*     A * X = B,
*  where A is an N-by-N matrix and X and B are N-by-NRHS matrices.
*
*  The LU decomposition with partial pivoting and row interchanges is
*  used to factor A as
*     A = P * L * U,
*  where P is a permutation matrix, L is unit lower triangular, and U is
*  upper triangular.  The factored form of A is then used to solve the
*  system of equations A * X = B.
*
*  Arguments
*  =========
*
*  N       (input) INTEGER
*          The number of linear equations, i.e., the order of the
*          matrix A.  N >= 0.
*
*  NRHS    (input) INTEGER
*          The number of right hand sides, i.e., the number of columns
*          of the matrix B.  NRHS >= 0.
*
*  A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
*          On entry, the N-by-N coefficient matrix A.
*          On exit, the factors L and U from the factorization
*          A = P*L*U; the unit diagonal elements of L are not stored.
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,N).
*
*  IPIV    (output) INTEGER array, dimension (N)
*          The pivot indices that define the permutation matrix P;
*          row i of the matrix was interchanged with row IPIV(i).
*
*  B       (input/output) DOUBLE PRECISION array, dimension (LDB,NRHS)
*          On entry, the N-by-NRHS matrix of right hand side matrix B.
*          On exit, if INFO = 0, the N-by-NRHS solution matrix X.
*
*  LDB     (input) INTEGER
*          The leading dimension of the array B.  LDB >= max(1,N).
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value
*          > 0:  if INFO = i, U(i,i) is exactly zero.  The factorization
*                has been completed, but the factor U is exactly
*                singular, so the solution could not be computed.
*/
{
   double *A,*b=NULL;
   int cols,info,*PIV,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" dgesv: ",MATSNOT);
      return 0;
   }
   rows=tos->row;
   if(tos->col!=rows) {
      stkerr(" dgesv: ","matrix A is not square");
      return 0;
   }
   if((tos-1)->col>0 && (tos-1)->row!=rows) {
      stkerr(" dgesv: ","matrix b is not compatible with A");
      return 0;
   }
   if((PIV=(int *)malloc(1+rows*sizeof(int)))==NULL) {
      stkerr(" dgesv: ",MEMNOT);
      return 0;
   }
   cop(); /* A will be overwritten, so use a copy */
   A=tos->mat;

   swap(); /* b to top of stack */
   cols=tos->col; /* columns in b */
   if(cols) cop(); /* this copy of b will contain x on return */
   else {
      drop();
      pushint(rows);
      identity();
      cols=rows;
   }
   b=tos->mat;

   DGESV(&rows,&cols,A,&rows,PIV,b,&rows,&info);

   free(PIV);

   if(info) {
      if(info<0) {
         gprintf(" dgesv: illegal argument "); 
         gprintf("%d to DGESV",ABS(info)); nc();
         stkerr("","");
         pushstr("_b"); 
         naming();
         swap(); 
         pushstr("_A"); 
         naming();
         return 0;
      }
      else {
         stkerr(" dgesv: ","factor U is singular; no solution");
         stkerr("","");
         pushstr("_b"); 
         naming();
         swap(); 
         pushstr("_U"); 
         naming();
         return 0;
      }
   }
   return(
      lop() && 
      pushstr("_x") && 
      naming()
   );
}

int dpotrf1() /* dpotrf (hA --- hL) */
/* Return lower triangular matrix L for symmetric matrix A, such that 
   L*L' = A.

      SUBROUTINE DPOTRF( UPLO, N, A, LDA, INFO )
*
*  -- LAPACK routine (version 3.0) --
*     Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
*     Courant Institute, Argonne National Lab, and Rice University
*     March 31, 1993
*
*     .. Scalar Arguments ..
      CHARACTER          UPLO
      INTEGER            INFO, LDA, N
*     ..
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * )
*     ..
*
*  Purpose
*  =======
*
*  DPOTRF computes the Cholesky factorization of a real symmetric
*  positive definite matrix A.
*
*  The factorization has the form
*     A = U**T * U,  if UPLO = 'U', or
*     A = L  * L**T,  if UPLO = 'L',
*  where U is an upper triangular matrix and L is lower triangular.
*
*  This is the block version of the algorithm, calling Level 3 BLAS.
*
*  Arguments
*  =========
*
*  UPLO    (input) CHARACTER*1
*          = 'U':  Upper triangle of A is stored;
*          = 'L':  Lower triangle of A is stored.
*
*  N       (input) INTEGER
*          The order of the matrix A.  N >= 0.
*
*  A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
*          On entry, the symmetric matrix A.  If UPLO = 'U', the leading
*          N-by-N upper triangular part of A contains the upper
*          triangular part of the matrix A, and the strictly lower
*          triangular part of A is not referenced.  If UPLO = 'L', the
*          leading N-by-N lower triangular part of A contains the lower
*          triangular part of the matrix A, and the strictly upper
*          triangular part of A is not referenced.
*
*          On exit, if INFO = 0, the factor U or L from the Cholesky
*          factorization A = U**T*U or A = L*L**T.
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,N).
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value
*          > 0:  if INFO = i, the leading minor of order i is not
*                positive definite, and the factorization could not be
*                completed.
*
*  =====================================================================
*/
{
   double *A;
   int i,info,j=0,n;
   char uplo='L';

   if(!(tos->typ==MAT)) {
      stkerr(" dpotrf: ",MATNOT);
      return 0;
   }
   cop();
   n=tos->row;

   for(;j<n;j++) {
      A=(tos->mat)+locvec(j,n);
      for(i=0;i<j;i++) {
         *A=0;
         A++;
      }
   }
   A=tos->mat; /* on return, this will be L */

/* Long jump is used below to guard against DPOTRF stopping the program
   if there is a data error.  This approach is useful for any of the
   Lapack functions; the jump return is called BLASX as shown below,
   and function exit() will land back here if it is entered. */

   if(!bufup()) return 0; /* Up to next run level, onbuf */

   setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

   if(!*(jmpready+onbuf)) { /* falls through first time */
      *(jmpready+onbuf)=BLASX;

      DPOTRF(&uplo,&n,A,&n,&info);
   }
   else { /* comes here on longjmp with error */
      stkerr(" dpotrf: ","error in LAPACK DPOTRF");
      bufdn();
      return 0;
   }
   bufdn(); /* Down from higher run level */

   if(!info) return 1;

   if(info<0) {
      gprintf(" dpotrf: illegal value for argument %d",abs(info));
   }
   else {
      gprintf(" dpotrf: A is not positive definite at row %d",info);
   }
   stkerr("","");
   nc();
   return 0;
}

int dsygv1() /* dsygv (hM hK --- hZ hw) {{{1 */
/* Eigenvalues and eigenvectors of a real, symmetric eigensystem:

      K*Z=w*M*Z

   where K and M are real and symmetric, and M is positive definite.

      SUBROUTINE DSYGV( ITYPE, JOBZ, UPLO, N, A, LDA, B, LDB, W, WORK,
     $                  LWORK, INFO )
*
*  -- LAPACK driver routine (version 3.0) --
*     Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
*     Courant Institute, Argonne National Lab, and Rice University
*     June 30, 1999
*
*     .. Scalar Arguments ..
      CHARACTER          JOBZ, UPLO
      INTEGER            INFO, ITYPE, LDA, LDB, LWORK, N
*     ..
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), B( LDB, * ), W( * ), WORK( * )
*     ..
*
*  Purpose
*  =======
*
*  DSYGV computes all the eigenvalues, and optionally, the eigenvectors
*  of a real generalized symmetric-definite eigenproblem, of the form
*  A*x=(lambda)*B*x,  A*Bx=(lambda)*x,  or B*A*x=(lambda)*x.
*  Here A and B are assumed to be symmetric and B is also
*  positive definite.
*
*  Arguments
*  =========
*
*  ITYPE   (input) INTEGER
*          Specifies the problem type to be solved:
*          = 1:  A*x = (lambda)*B*x
*          = 2:  A*B*x = (lambda)*x
*          = 3:  B*A*x = (lambda)*x
*
*  JOBZ    (input) CHARACTER*1
*          = 'N':  Compute eigenvalues only;
*          = 'V':  Compute eigenvalues and eigenvectors.
*
*  UPLO    (input) CHARACTER*1
*          = 'U':  Upper triangles of A and B are stored;
*          = 'L':  Lower triangles of A and B are stored.
*
*  N       (input) INTEGER
*          The order of the matrices A and B.  N >= 0.
*
*  A       (input/output) DOUBLE PRECISION array, dimension (LDA, N)
*          On entry, the symmetric matrix A.  If UPLO = 'U', the
*          leading N-by-N upper triangular part of A contains the
*          upper triangular part of the matrix A.  If UPLO = 'L',
*          the leading N-by-N lower triangular part of A contains
*          the lower triangular part of the matrix A.
*
*          On exit, if JOBZ = 'V', then if INFO = 0, A contains the
*          matrix Z of eigenvectors.  The eigenvectors are normalized
*          as follows:
*          if ITYPE = 1 or 2, Z**T*B*Z = I;
*          if ITYPE = 3, Z**T*inv(B)*Z = I.
*          If JOBZ = 'N', then on exit the upper triangle (if UPLO='U')
*          or the lower triangle (if UPLO='L') of A, including the
*          diagonal, is destroyed.
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,N).
*
*  B       (input/output) DOUBLE PRECISION array, dimension (LDB, N)
*          On entry, the symmetric positive definite matrix B.
*          If UPLO = 'U', the leading N-by-N upper triangular part of B
*          contains the upper triangular part of the matrix B.
*          If UPLO = 'L', the leading N-by-N lower triangular part of B
*          contains the lower triangular part of the matrix B.
*
*          On exit, if INFO <= N, the part of B containing the matrix is
*          overwritten by the triangular factor U or L from the Cholesky
*          factorization B = U**T*U or B = L*L**T.
*
*  LDB     (input) INTEGER
*          The leading dimension of the array B.  LDB >= max(1,N).
*
*  W       (output) DOUBLE PRECISION array, dimension (N)
*          If INFO = 0, the eigenvalues in ascending order.
*
*  WORK    (workspace/output) DOUBLE PRECISION array, dimension (LWORK)
*          On exit, if INFO = 0, WORK(1) returns the optimal LWORK.
*
*  LWORK   (input) INTEGER
*          The length of the array WORK.  LWORK >= max(1,3*N-1).
*          For optimal efficiency, LWORK >= (NB+2)*N,
*          where NB is the blocksize for DSYTRD returned by ILAENV.
*
*          If LWORK = -1, then a workspace query is assumed; the routine
*          only calculates the optimal size of the WORK array, returns
*          this value as the first entry of the WORK array, and no error
*          message related to LWORK is issued by XERBLA.
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value
*          > 0:  DPOTRF or DSYEV returned an error code:
*             <= N:  if INFO = i, DSYEV failed to converge;
*                    i off-diagonal elements of an intermediate
*                    tridiagonal form did not converge to zero;
*             > N:   if INFO = N + i, for 1 <= i <= N, then the leading
*                    minor of order i of B is not positive definite.
*                    The factorization of B could not be completed and
*                    no eigenvalues or eigenvectors were computed.
*
*  =====================================================================
* 
   Test case for dsygv.

   Reference:
      Engineering and Scientific Subroutine Library, Version 2,
      Release 2, Second Edition (January 1994), IBM Corporation.

   Data from the Reference, Volume 2, Example 2, p. 792:

      list: -1 1 -1 ; list: 1 1 -1 ; list: -1 -1 1 ; 3 parkn into Kaa
      list:  2 1  0 ; list: 1 2  1 ; list:  0  1 2 ; 3 parkn into Maa

      [tops@clacker] ready > Maa Kaa dsygv

       stack elements:
             0 matrix: _w  3 by 1
             1 matrix: _Z  3 by 3
       [2] ok!
      [tops@clacker] ready > 1E-8 filter .m nl nl 1E-8 filter .m
       Row 1:     -1.5
       Row 2:        0
       Row 3:        2

       Row 1:   -0.866        0        0
       Row 2:   0.5774   0.4082  -0.7071
       Row 3:  -0.2887   0.4082   0.7071 */
{
   double *K,*M,*w,*WORK,WORK_SPACE;
   int info=0,lwork=-1,one=1,rows;
   char *jobz="V",*uplo="L";

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" dsygv: ",MATNOT);
      return 0;
   }
   rows=tos->row;

   if(rows!=tos->col || rows!=(tos-1)->col) {
      stkerr(" dsygv: ",SQUNOT);
      return 0;
   }
   if(rows!=(tos-1)->row) {
      stkerr(" dsygv: ",MATSNOTC);
      return 0;
   }
   if(rows==0) {
      stkerr(" dsygv: ",MATPURG);
      return 0;
   }
   /* Copy M and K so originals stay intact: */
   cop();
   K=tos->mat;
   pushstr("_Z"); /* K will contain modes, Z */
   naming();

   swap();
   cop();
   M=tos->mat; 

   if(!matstk(rows,1,"_w")) return 0;
   w=tos->mat;
   swap();

/* Stack at this point: hK hw hM */

   DSYGV(&one,jobz,uplo,&rows,K,&rows,M,&rows,w,&WORK_SPACE,&lwork,
      &info);
   lwork=MAX(10*rows,WORK_SPACE);
   if(TRACE) {
      gprintf(" dsygv: rows %d, lwork: %d",rows,lwork);
      nc();
   }
   if(!matstk(lwork,1,"_WORK")) return 0;
   WORK=tos->mat;

/* Contents of stack: hK hw hM hWORK */
/* Eigenvectors return in K */

   DSYGV(&one,jobz,uplo,&rows,K,&rows,M,&rows,w,WORK,&lwork,&info);

   if(info) {
      if(info<0) {
         gprintf(" dsygv: illegal argument %d to DSYGV",ABS(info));
         nc();
      }
      else {
         if(info<rows) {
            gprintf( \
               " dsygv: %d off-diagonals did not converge in DSYEV",\
               info);
         }
         else gprintf(" dsygv: M is not positive definite");
      }
      nc();
   /* Return stack to hM hK, although they now contain garbage: */
      drop(); /* hWORK gone */ 
      lop();  /* hw gone */
      swap(); /* hM hK */
      return 0;
   }
   return(drop2()); /* success; dropping M and WORK */
} /* 1}}} */

int zdscal1() /* zdscal (hVr hVi x --- hWr hWi) {{{1 */
/* Scale complex matrix by real constant.
      subroutine  zdscal(n,da,zx,incx)
*/
{
   double x;
   int len,one=1;

   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" zdscal: ",MATNOT2);
      return 0;
   }
   len=2*(tos-1)->row*(tos-1)->col;

   if(!(popd(&x) && dblcmplx())) return 0;


gprintf(" len: %d",len); nc();
return 1;

   ZDSCAL(&len,&x,tos->mat,&one);
   return 1;
} /* 1}}} */

int zgemm1() /* zgemm (hA trnA alpha hB trnB hC beta --- hD) {{{1 */
/*  Complex matrix multiplication and add: A*B+C=D
    trnA true to use transpose of A
    A is scaled by alpha
    trnB true to use transpose of B
    C is scaled by beta
    Incoming C can be purged or null for: A*B=D

      SUBROUTINE ZGEMM ( TRANSA, TRANSB, M, N, K, ALPHA, A, LDA, B, LDB,
     $                   BETA, C, LDC )
*     .. Scalar Arguments ..
      CHARACTER*1        TRANSA, TRANSB
      INTEGER            M, N, K, LDA, LDB, LDC
      COMPLEX*16         ALPHA, BETA
*     .. Array Arguments ..
      COMPLEX*16         A( LDA, * ), B( LDB, * ), C( LDC, * )
*     ..
*
*  Purpose
*  =======
*
*  ZGEMM  performs one of the matrix-matrix operations
*
*     C := alpha*op( A )*op( B ) + beta*C,
*
*  where  op( X ) is one of
*
*     op( X ) = X   or   op( X ) = X'   or   op( X ) = conjg( X' ),
*
*  alpha and beta are scalars, and A, B and C are matrices, with op( A )
*  an m by k matrix,  op( B )  a  k by n matrix and  C an m by n matrix.
*
*  Parameters
*  ==========
*
*  TRANSA - CHARACTER*1.
*           On entry, TRANSA specifies the form of op( A ) to be used in
*           the matrix multiplication as follows:
*
*              TRANSA = 'N' or 'n',  op( A ) = A.
*
*              TRANSA = 'T' or 't',  op( A ) = A'.
*
*              TRANSA = 'C' or 'c',  op( A ) = conjg( A' ).
*
*           Unchanged on exit.
*
*  TRANSB - CHARACTER*1.
*           On entry, TRANSB specifies the form of op( B ) to be used in
*           the matrix multiplication as follows:
*
*              TRANSB = 'N' or 'n',  op( B ) = B.
*
*              TRANSB = 'T' or 't',  op( B ) = B'.
*
*              TRANSB = 'C' or 'c',  op( B ) = conjg( B' ).
*
*           Unchanged on exit.
*
*  M      - INTEGER.
*           On entry,  M  specifies  the number  of rows  of the  matrix
*           op( A )  and of the  matrix  C.  M  must  be at least  zero.
*           Unchanged on exit.
*
*  N      - INTEGER.
*           On entry,  N  specifies the number  of columns of the matrix
*           op( B ) and the number of columns of the matrix C. N must be
*           at least zero.
*           Unchanged on exit.
*
*  K      - INTEGER.
*           On entry,  K  specifies  the number of columns of the matrix
*           op( A ) and the number of rows of the matrix op( B ). K must
*           be at least  zero.
*           Unchanged on exit.
*
*  ALPHA  - COMPLEX*16      .
*           On entry, ALPHA specifies the scalar alpha.
*           Unchanged on exit.
*
*  A      - COMPLEX*16       array of DIMENSION ( LDA, ka ), where ka is
*           k  when  TRANSA = 'N' or 'n',  and is  m  otherwise.
*           Before entry with  TRANSA = 'N' or 'n',  the leading  m by k
*           part of the array  A  must contain the matrix  A,  otherwise
*           the leading  k by m  part of the array  A  must contain  the
*           matrix A.
*           Unchanged on exit.
*
*  LDA    - INTEGER.
*           On entry, LDA specifies the first dimension of A as declared
*           in the calling (sub) program. When  TRANSA = 'N' or 'n' then
*           LDA must be at least  max( 1, m ), otherwise  LDA must be at
*           least  max( 1, k ).
*           Unchanged on exit.
*
*  B      - COMPLEX*16       array of DIMENSION ( LDB, kb ), where kb is
*           n  when  TRANSB = 'N' or 'n',  and is  k  otherwise.
*           Before entry with  TRANSB = 'N' or 'n',  the leading  k by n
*           part of the array  B  must contain the matrix  B,  otherwise
*           the leading  n by k  part of the array  B  must contain  the
*           matrix B.
*           Unchanged on exit.
*
*  LDB    - INTEGER.
*           On entry, LDB specifies the first dimension of B as declared
*           in the calling (sub) program. When  TRANSB = 'N' or 'n' then
*           LDB must be at least  max( 1, k ), otherwise  LDB must be at
*           least  max( 1, n ).
*           Unchanged on exit.
*
*  BETA   - COMPLEX*16      .
*           On entry,  BETA  specifies the scalar  beta.  When  BETA  is
*           supplied as zero then C need not be set on input.
*           Unchanged on exit.
*
*  C      - COMPLEX*16       array of DIMENSION ( LDC, n ).
*           Before entry, the leading  m by n  part of the array  C must
*           contain the matrix  C,  except when  beta  is zero, in which
*           case C need not be set on entry.
*           On exit, the array  C  is overwritten by the  m by n  matrix
*           ( alpha*op( A )*op( B ) + beta*C ).
*
*  LDC    - INTEGER.
*           On entry, LDC specifies the first dimension of C as declared
*           in  the  calling  (sub)  program.   LDC  must  be  at  least
*           max( 1, m ).
*           Unchanged on exit.
*/
{
/* zgemm (hA trnA alpha hB trnB hC beta --- hD) */

   double *A,*B,*C;
   double alpha[2],beta[2];
   int trnA,trnB;

   char TRANSA,TRANSB;
   char *Dname="_D";
   int cA,cB,rA,rB;
   int K,M,N;

   if(!(
      popdx(beta,(beta+1)) && swap() &&
      popbool(&trnB) && rot() &&
      popdx(alpha,(alpha+1)) && rot() &&
      popbool(&trnA))
   ) return 0;

   if(!(tos->typ==MAT && (tos-1)->typ==MAT && (tos-2)->typ==MAT)) {
      stkerr(" zgemm: ",MATNOT3);
      return 0;
   }
   A=(tos-2)->mat;

/* NOTE: For complex matrices A, B, and C, tos->row returns twice the
   actual number of rows. */
   rA=((tos-2)->row)/2;

   cA=(tos-2)->col;

   B=(tos-1)->mat;
   rB=((tos-1)->row)/2;
   cB=(tos-1)->col;

   if(!trnA) {
      TRANSA='N';
      M=rA;
      K=cA;
   }
   else {
      TRANSA='T';
      M=cA;
      K=rA;
   }
   if(!trnB) {
      TRANSB='N';
      N=cB;
   }
   else {
      TRANSB='T';
      N=rB;
   }
/* Compatibility tests: */
   if(TRANSA=='N' && TRANSB=='N' && (cA!=rB)) {
      stkerr(" zgemm: ","columns of A must match rows of B");
      return 0;
   }
   if(TRANSA=='T' && TRANSB=='N' && (rA!=rB)) {
      stkerr(" zgemm: ","rows of A must match rows of B");
      return 0;
   }
   if(TRANSA=='N' && TRANSB=='T' && (cA!=cB)) {
      stkerr(" zgemm: ","columns of A must match columns of B");
      return 0;
   }
   if(TRANSA=='T' && TRANSB=='T' && (rA!=cB)) {
      stkerr(" zgemm: ","rows of A must match columns of B");
      return 0;
   }
   if(tos->row==0 || tos->col==0) {
/*    C is purged, so put an uninitialized complex MAT on stack to hold
      the results: */
      drop();
      if(!matstk(2*M,N,Dname)) return 0;
      set_complex(tos);
      *beta=0;
      *(beta+1)=0;
   }
   else {
      if(M!=((tos->row)/2) || N!=(tos->col)) {
         stkerr(" zgemm: ","C is not compatible with A*B");
         return 0;
      }
/*    C is not purged, so it will be overwritten by results.  Make
      a copy: */
      cop();
      pushstr(Dname);
      naming();
   }
   C=tos->mat;

/* Make this check after compatibility and C has been sorted out: */
   if(!is_complex(tos) || !is_complex(tos-1) || !is_complex(tos-2)) {
      stkerr(" zgemm: ",COMPLEXNOT);
      return 0;
   }
/* Long jump is used below to guard against ZGEMM stopping the program
   if there is a data error.  This approach is useful for any of the
   Lapack functions; the jump return is called BLASX as shown below,
   and function exit() will land back here if it is entered. */

   if(!bufup()) return 0; /* Up to next run level, onbuf */

   setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

   if(!*(jmpready+onbuf)) { /* falls through first time */
      *(jmpready+onbuf)=BLASX;

      ZGEMM(&TRANSA, &TRANSB, &M, &N, &K, alpha, A, &rA, B, &rB, \
         beta, C, &M);
   }
   else { /* comes here on longjmp with error */
      stkerr(" zgemm: ","matrix size error in BLAS ZGEMM");
      bufdn();
      return 0;
   }
   bufdn(); /* Down from higher run level */

   return(lop() && lop());
} /* 1}}} */
/*--------------------------------------------------------------------*/

int zgesv1() /* zgesv (hB hA --- hX) {{{1 */
/* Solve A*X=B for X, where the matrices are complex.

      SUBROUTINE ZGESV( N, NRHS, A, LDA, IPIV, B, LDB, INFO )

*  Purpose
*  =======
*
*  ZGESV computes the solution to a complex system of linear equations
*     A * X = B,
*  where A is an N-by-N matrix and X and B are N-by-NRHS matrices.
*
*  The LU decomposition with partial pivoting and row interchanges is
*  used to factor A as
*     A = P * L * U,
*  where P is a permutation matrix, L is unit lower triangular, and U is
*  upper triangular.  The factored form of A is then used to solve the
*  system of equations A * X = B.
*
*
*  Arguments
*  =========
*
*  N       (input) INTEGER
*          The number of linear equations, i.e., the order of the
*          matrix A.  N >= 0.
*
*  NRHS    (input) INTEGER
*          The number of right hand sides, i.e., the number of columns
*          of the matrix B.  NRHS >= 0.
*
*  A       (input/output) COMPLEX*16 array, dimension (LDA,N)
*          On entry, the N-by-N coefficient matrix A.
*          On exit, the factors L and U from the factorization
*          A = P*L*U; the unit diagonal elements of L are not stored.
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,N).
*
*  IPIV    (output) INTEGER array, dimension (N)
*          The pivot indices that define the permutation matrix P;
*          row i of the matrix was interchanged with row IPIV(i).
*
*  B       (input/output) COMPLEX*16 array, dimension (LDB,NRHS)
*          On entry, the N-by-NRHS matrix of right hand side matrix B.
*          On exit, if INFO = 0, the N-by-NRHS solution matrix X.
*
*  LDB     (input) INTEGER
*          The leading dimension of the array B.  LDB >= max(1,N).
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value
*          > 0:  if INFO = i, U(i,i) is exactly zero.  The factorization
*                has been completed, but the factor U is exactly
*                singular, so the solution could not be computed.
*/
{
   double *A,*b=NULL;
   int cols,info=0,*PIV,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" zgesv: ",MATSNOT);
      return 0;
   }
   if(!is_complex(tos)) {
      stkerr(" zgesv: ","matrix A is not complex");
      return 0;
   }
   rows=(tos->row)/2;
   if(tos->col!=rows) {
      stkerr(" zgesv: ","complex matrix A is not square");
      return 0;
   }
   cop(); /* use a copy since A will be destroyed */
   A=tos->mat;

   swap(); /* b to top of stack */

   if(tos->row==0 || tos->col==0) {
/*    b is purged, so put a complex identity matrix on the stack for
      computing the inverse of A: */
      drop();        /* drop purged b */
      pushint(rows);
      identity();    /* real part of b */
      pushint(rows);
      pushint(rows);
      null();        /* null imaginary part of b */
      dblcmplx();
   }
   else {
      cop(); /* this copy of b will contain x on return */
   }
   cols=tos->col; /* columns in b */
   b=tos->mat;

   if(tos->row!=2*rows) {
      stkerr(" zgesv: ","matrix b is not compatible with A");
      return 0;
   }
   if((PIV=(int *)malloc(1+rows*sizeof(int)))==NULL) {
      stkerr(" zgesv: ",MEMNOT);
      return 0;
   }

   ZGESV(&rows,&cols,A,&rows,PIV,b,&rows,&info);

   free(PIV);

   if(info) {
      if(info<0) {
         gprintf(" zgesv: illegal argument %d to DGESV",ABS(info)); 
         nc();
         stkerr("","");
         pushstr("_b");
         naming();
         swap();
         pushstr("_A");
         naming();
      }
      else {
         gprintf(" zgesv: factor diagonal U is singular at row %d");
         nc();
         stkerr("","");
         pushstr("_b");
         naming();
         swap();
         pushstr("_U");
         naming();
      }
      return 0;
   }
   return(
      lop() &&
      pushstr("_x") &&
      naming()
   );
} /* 1}}} */
/*--------------------------------------------------------------------*/
int zggev1() /* zggev (hA hB --- hAlpha hBeta hVL hVR ) {{{1 */
/* Solves the complex, nonsymmetric, general eigenvalue problem on (A,B).
 * man entry:  zggev {{{2
 * (hA hB --- hAlpha hBeta hVL hVR ) Solve the complex, nonsymmetric, general eigenvalue problem on (A,B).  Alpha and Beta are the numberator and denominator of the eigenvalue array (the numerator and denominator appear separately because the solution can be meaningful even if the denominator is zero).  The left (VL) and right (VR) eigenvectors will satisfy these relationships:
 * A*VR = B*VR*diag(eigval)
 *    and
 * VL'*A = diag(eigval)*VL'*B
where eigval is Alpha ./ Beta, and VL' is the conjugate transpose of VL, 
 * category: math::matrix::operator
 * related: dgeev, dgeev2, ceig, eigchk, modes, lapack, dsygv
 * 2}}}

*  {{{2
      SUBROUTINE ZGGEV( JOBVL, JOBVR, N, A, LDA, B, LDB, ALPHA, BETA,
     $                  VL, LDVL, VR, LDVR, WORK, LWORK, RWORK, INFO )
 
*  ZGGEV computes for a pair of N-by-N complex nonsymmetric matrices
*  (A,B), the generalized eigenvalues, and optionally, the left and/or
*  right generalized eigenvectors.
*
*  A generalized eigenvalue for a pair of matrices (A,B) is a scalar
*  lambda or a ratio alpha/beta = lambda, such that A - lambda*B is
*  singular. It is usually represented as the pair (alpha,beta), as
*  there is a reasonable interpretation for beta=0, and even for both
*  being zero.
*
*  The right generalized eigenvector v(j) corresponding to the
*  generalized eigenvalue lambda(j) of (A,B) satisfies
*
*               A * v(j) = lambda(j) * B * v(j).
*
*  The left generalized eigenvector u(j) corresponding to the
*  generalized eigenvalues lambda(j) of (A,B) satisfies
*
*               u(j)**H * A = lambda(j) * u(j)**H * B
*
*  where u(j)**H is the conjugate-transpose of u(j).
*
*  Arguments
*  =========
*
*  JOBVL   (input) CHARACTER*1
*          = 'N':  do not compute the left generalized eigenvectors;
*          = 'V':  compute the left generalized eigenvectors.
*
*  JOBVR   (input) CHARACTER*1
*          = 'N':  do not compute the right generalized eigenvectors;
*          = 'V':  compute the right generalized eigenvectors.
*
*  N       (input) INTEGER
*          The order of the matrices A, B, VL, and VR.  N >= 0.
*
*  A       (input/output) COMPLEX*16 array, dimension (LDA, N)
*          On entry, the matrix A in the pair (A,B).
*          On exit, A has been overwritten.
*
*  LDA     (input) INTEGER
*          The leading dimension of A.  LDA >= max(1,N).
*
*  B       (input/output) COMPLEX*16 array, dimension (LDB, N)
*          On entry, the matrix B in the pair (A,B).
*          On exit, B has been overwritten.
*
*  LDB     (input) INTEGER
*          The leading dimension of B.  LDB >= max(1,N).
*
*  ALPHA   (output) COMPLEX*16 array, dimension (N)
*  BETA    (output) COMPLEX*16 array, dimension (N)
*          On exit, ALPHA(j)/BETA(j), j=1,...,N, will be the
*          generalized eigenvalues.
*
*          Note: the quotients ALPHA(j)/BETA(j) may easily over- or
*          underflow, and BETA(j) may even be zero.  Thus, the user
*          should avoid naively computing the ratio alpha/beta.
*          However, ALPHA will be always less than and usually
*          comparable with norm(A) in magnitude, and BETA always less
*          than and usually comparable with norm(B).
*
*  VL      (output) COMPLEX*16 array, dimension (LDVL,N)
*          If JOBVL = 'V', the left generalized eigenvectors u(j) are
*          stored one after another in the columns of VL, in the same
*          order as their eigenvalues.
*          Each eigenvector will be scaled so the largest component
*          will have abs(real part) + abs(imag. part) = 1.
*          Not referenced if JOBVL = 'N'.
*
*  LDVL    (input) INTEGER
*          The leading dimension of the matrix VL. LDVL >= 1, and
*          if JOBVL = 'V', LDVL >= N.
*
*  VR      (output) COMPLEX*16 array, dimension (LDVR,N)
*          If JOBVR = 'V', the right generalized eigenvectors v(j) are
*          stored one after another in the columns of VR, in the same
*          order as their eigenvalues.
*          Each eigenvector will be scaled so the largest component
*          will have abs(real part) + abs(imag. part) = 1.
*          Not referenced if JOBVR = 'N'.
*
*  LDVR    (input) INTEGER
*          The leading dimension of the matrix VR. LDVR >= 1, and
*          if JOBVR = 'V', LDVR >= N.
*
*  WORK    (workspace/output) COMPLEX*16 array, dimension (LWORK)
*          On exit, if INFO = 0, WORK(1) returns the optimal LWORK.
*
*  LWORK   (input) INTEGER
*          The dimension of the array WORK.  LWORK >= max(1,2*N).
*          For good performance, LWORK must generally be larger.
*
*          If LWORK = -1, then a workspace query is assumed; the routine
*          only calculates the optimal size of the WORK array, returns
*          this value as the first entry of the WORK array, and no error
*          message related to LWORK is issued by XERBLA.
*
*  RWORK   (workspace/output) DOUBLE PRECISION array, dimension (8*N)
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value.
*          =1,...,N:
*                The QZ iteration failed.  No eigenvectors have been
*                calculated, but ALPHA(j) and BETA(j) should be
*                correct for j=INFO+1,...,N.
*          > N:  =N+1: other then QZ iteration failed in DHGEQZ,
*  2}}}
*/
{
    char    JOBVL = 'V';
    char    JOBVR = 'V';
    int     N, Na, LDA, LDB, LDVL, LDVR, LWORK, info;
    double *A, *B;
    double *ALPHA, *BETA, *VL, *VR, *WORK, *RWORK;
int DEBUG = 0;

    N = tos->row/2;  /* complex matrices appear with 2x number of rows */
    LWORK = 40*N;
    if ((WORK = (double *) malloc(2*LWORK*sizeof(double))) == NULL) {
      stkerr(" no memory for WORK in zggev: ",MEMNOT);
      return 0;
    }
    if ((RWORK = (double *) malloc(8*N*sizeof(double))) == NULL) {
      free(WORK);
      stkerr(" no memory for RWORK in zggev: ",MEMNOT);
      return 0;
    }

    if (tos->typ != MAT) {
       stkerr(" zggev:  [B] ", MATNOT);
       return 0;
    }
    if (!is_complex(tos)) {
       stkerr(" zggev:  [B] ", COMPLEXNOT);
       return 0;
    }
    if (N != tos->col) {
       stkerr(" zggev:  [B] ", SQUNOT);
       return 0;
    }
    if ((tos-1)->typ != MAT) {
       stkerr(" zggev:  [A] ", MATNOT);
       return 0;
    }
    if (!is_complex(tos-1)) {
       stkerr(" zggev:  [A] ", COMPLEXNOT);
       return 0;
    }
    Na = (tos-1)->row/2;
    if (Na != (tos-1)->col) {
       stkerr(" zggev:  [A] ", SQUNOT);
       return 0;
    }
    if (Na != N) {
       stkerr(" zggev:  ", MATSNOT);
       return 0;
    }

    cop();    /* A, B will be overwritten, so use copies */
    swap();
    cop();
    swap();
    A = (tos-1)->mat;
    B = tos->mat;
 
if (DEBUG)
for (Na = 0; Na < 2*N*N; Na += 2) {
gprintf("A[%2d]=% 12.6e,% 12.6e  B[%2d]=% 12.6e,% 12.6e\n",
Na, A[Na],A[Na+1], Na, B[Na],B[Na+1]);
}
    if (!matstk(2*N, 1, "_alpha")) return 0; 
    /* ( A B --- A B alpha ) */
    ALPHA    = tos->mat;
    set_complex(tos);
 
    if (!matstk(2*N, 1, "_beta" )) return 0; 
    /* ( A B --- A B alpha beta ) */
    BETA     = tos->mat;
    set_complex(tos);
 
    if (!matstk(2*N, N, "_VL" )) return 0; 
    /* ( A B --- A B alpha beta VL ) */
    VL       = tos->mat;
    set_complex(tos);
 
    if (!matstk(2*N, N, "_VR" )) return 0; 
    /* ( A B --- A B alpha beta VL VR ) */
    VR       = tos->mat;
    set_complex(tos);

    LDA  = N;
    LDB  = N;
    LDVL = N;
    LDVR = N;
 
    ZGGEV(&JOBVL, &JOBVR, &N, A, &LDA, B, &LDB, ALPHA, BETA,
         VL, &LDVL, VR, &LDVR, WORK, &LWORK, RWORK, &info );
if (DEBUG)
gprintf("optimal LWORK=%le  real LWORK=%d\n", WORK[0], LWORK);

    pushstr("4 roll drop"); xmain(0);    /* drops [B] */
    /* ( A B alpha beta VL VR --- A alpha beta VL VR ) */
    pushstr("4 roll drop"); xmain(0);    /* drops [A] */
    /* ( A B alpha beta VL VR --- alpha beta VL VR ) */

    free(WORK);
    free(RWORK);

if (DEBUG) {
gprintf("after zggev info=%d\n", info);
for (Na = 0; Na < 2*N; Na += 2) {
gprintf("alpha[%2d]=% 12.6e,% 12.6e  beta[%2d]=% 12.6e,% 12.6e\n",
Na, ALPHA[Na],ALPHA[Na+1], Na, BETA[Na],BETA[Na+1]);
}
for (Na = 0; Na < 2*N*N; Na += 2) {
gprintf("VL[%2d]=% 12.6e,% 12.6e  VR[%2d]=% 12.6e,% 12.6e\n",
Na, VL[Na],VL[Na+1], Na, VR[Na],VR[Na+1]);
}
}

    if (info) {
       if        (info < 0) {
          gprintf(" zggev: illegal argument %d to ZGGEV",ABS(info)); 
          nc();
          stkerr("","");
          return 0;
       } else if (info > N) {
          gprintf(" zggev: qz iteration failed; N = %d",
                   ABS(info));
          nc();
          stkerr("","");
          return 0;
       } else {
          gprintf(" zggev: qz iteration failed but first %d ALPHA/BETA are OK",
                   ABS(info));
          nc();
       }
    }
    return 1;
} /* 1}}} */
int dgetri1() /* dgetri (hLU hP --- hinvA) {{{1 */
/* Invert real matrix A.
 * man entry:  dgetri {{{2
 * (hLU hP --- hinvA ) hLU contains the [L] and [U] factors of a matrix [A], and hP is the permutation array as computed by DGETRF.  This routine computes the inverse of [A] from its LU factors and the associated permutation array.
 * category: math::matrix::operator
 * related:  dgetrf, slash, \, dgetri, dgetrs, lapack
 * 2}}}

   {{{2
   http://www.netlib.org/lapack/double/dgetri.f
 
      SUBROUTINE DGETRI( N, A, LDA, IPIV, WORK, LWORK, INFO )
    
*  -- LAPACK routine (version 3.0) --
*     Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
*     Courant Institute, Argonne National Lab, and Rice University
*     June 30, 1999
*
*     Purpose
*     =======
*   
*     DGETRI computes the inverse of a matrix using the LU factorization
*     computed by DGETRF.
*   
*     This method inverts U and then computes inv(A) by solving the system
*     inv(A)*L = inv(U) for inv(A).
*   
*     Arguments
*     =========
*   
*     N       (input) INTEGER
*             The order of the matrix A.  N >= 0.
*   
*     A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
*             On entry, the factors L and U from the factorization
*             A = P*L*U as computed by DGETRF.
*             On exit, if INFO = 0, the inverse of the original matrix A.
*   
*     LDA     (input) INTEGER
*             The leading dimension of the array A.  LDA >= max(1,N).
*   
*     IPIV    (input) INTEGER array, dimension (N)
*             The pivot indices from DGETRF; for 1<=i<=N, row i of the
*             matrix was interchanged with row IPIV(i).
*   
*     WORK    (workspace/output) DOUBLE PRECISION array, dimension (LWORK)
*             On exit, if INFO=0, then WORK(1) returns the optimal LWORK.
*   
*     LWORK   (input) INTEGER
*             The dimension of the array WORK.  LWORK >= max(1,N).
*             For optimal performance LWORK >= N*NB, where NB is
*             the optimal blocksize returned by ILAENV.
*   
*             If LWORK = -1, then a workspace query is assumed; the routine
*             only calculates the optimal size of the WORK array, returns
*             this value as the first entry of the WORK array, and no error
*             message related to LWORK is issued by XERBLA.
*   
*     INFO    (output) INTEGER
*             = 0:  successful exit
*             < 0:  if INFO = -i, the i-th argument had an illegal value
*             > 0:  if INFO = i, U(i,i) is exactly zero; the matrix is
*                   singular and its inverse could not be computed.
*   

*  2}}}
*/
{
    int     N, LDA, LWORK, info, nRows, nCols, i;
    int    *IPIV;
    double *A, *d_IPIV, *WORK;
int DEBUG = 0;

    if ((tos-1)->typ != MAT) {
       stkerr(" dgetri:  [LU] ", MATNOT);
       return 0;
    }
    if (is_complex(tos-1)) {
       stkerr(" dgetri:  [LU] ", "must be real");
       return 0;
    }
    if (!is_factor_lu(tos-1)) {
       stkerr(" dgetri:  [LU] ", "must be an LU factor matrix");
       return 0;
    }
    if (!is_lapack(tos-1)) {
       stkerr(" dgetri:  [LU] ", "must be LU factor from with LAPACK");
       return 0;
    }
    nRows = (tos-1)->row;
    nCols = (tos-1)->col;
    N     = nRows;
    LDA   = N;
    if (nRows != nCols) {
       stkerr(" dgetri:  [LU] ", SQUNOT);
       return 0;
    }

    swap();
    cop();    /* A will be overwritten, use a copy */
    swap();
    A = (tos-1)->mat;

    d_IPIV = tos->mat;
    if (tos->row * tos->col != N) {
       stkerr(" dgetri:  [IPIV] ", COLSNOT);
       return 0;
    }
    /* DGETRI expects an integer, not double precision, permutation array */
    if ((IPIV = (int *) malloc(N*sizeof(int))) == NULL) {
       stkerr(" dgetri: ",MEMNOT);
       return 0;
    }
    d_IPIV = tos->mat;
    for (i = 0; i < N; i++) {
        IPIV[i] = (int) d_IPIV[i];
    }
    drop();  /*  ( hLU  hP --- hLU ) */

    LWORK = 40*N;
    if (!matstk(LWORK, 1, "_Work")) return 0;  /*  ( hLU  --- hLU  h_Work ) */
    WORK = tos->mat;
 
    DGETRI( &N, A, &LDA, IPIV, WORK, &LWORK, &info ); 
    /* ( hLU h_Work --- hinvA h_Work ) */

if (DEBUG)
gprintf("optimal LWORK=%le  real LWORK=%d\n", WORK[0], LWORK);

    drop();  /*  hinvA  h_Work --- hinvA ) */

if (DEBUG) {
gprintf("after dgetri info=%d\n", info);
}

    if (info) {
       if        (info < 0) {
          gprintf(" dgetri: illegal argument %d to DGETRI",ABS(info)); 
          nc();
          stkerr("","");
          return 0;
       } else if (info > N) {
          gprintf(" dgetri: U[%d,%d] is zero; matrix is singular",
                   ABS(info) , ABS(info) );
          nc();
          stkerr("","");
          return 0;
       }
    }
    return 1;
} /* 1}}} */
int dgetrf1() /* dgetrf (hA --- hLU hP) {{{1 */
/* Factor matrix A.
 * man entry:  dgetrf {{{2
 * (hA --- hLU hP) Factor the general real matrix [A].  [A] need not be square.
 * category: math::matrix::operator
 * related:  dgetrs, dgetri, slash, \, solve, lapack
 * 2}}}
   {{{2
   http://www.netlib.org/lapack/double/dgetrf.f
 
      SUBROUTINE DGETRF( M, N, A, LDA, IPIV, INFO )
*
*  -- LAPACK routine (version 3.0) --
*     Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
*     Courant Institute, Argonne National Lab, and Rice University
*     March 31, 1993
*
*     .. Scalar Arguments ..
      INTEGER            INFO, LDA, M, N
*     ..
*     .. Array Arguments ..
      INTEGER            IPIV( * )
      DOUBLE PRECISION   A( LDA, * )
*     ..
*
*  Purpose
*  =======
*
*  DGETRF computes an LU factorization of a general M-by-N matrix A
*  using partial pivoting with row interchanges.
*
*  The factorization has the form
*     A = P * L * U
*  where P is a permutation matrix, L is lower triangular with unit
*  diagonal elements (lower trapezoidal if m > n), and U is upper
*  triangular (upper trapezoidal if m < n).
*
*  This is the right-looking Level 3 BLAS version of the algorithm.
*
*  Arguments
*  =========
*
*  M       (input) INTEGER
*          The number of rows of the matrix A.  M >= 0.
*
*  N       (input) INTEGER
*          The number of columns of the matrix A.  N >= 0.
*
*  A       (input/output) DOUBLE PRECISION array, dimension (LDA,N)
*          On entry, the M-by-N matrix to be factored.
*          On exit, the factors L and U from the factorization
*          A = P*L*U; the unit diagonal elements of L are not stored.
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,M).
*
*  IPIV    (output) INTEGER array, dimension (min(M,N))
*          The pivot indices; for 1 <= i <= min(M,N), row i of the
*          matrix was interchanged with row IPIV(i).
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value
*          > 0:  if INFO = i, U(i,i) is exactly zero. The factorization
*                has been completed, but the factor U is exactly
*                singular, and division by zero will occur if it is used
*                to solve a system of equations.
*  2}}}
*/
{
    int     M, N, LDA, info, nRows, nCols, i, piv_size;
    int    *IPIV;
    double *A, *DPIV;
int DEBUG = 0;

    if (tos->typ != MAT) {
       stkerr(" dgetrf:  [A] ", MATNOT);
       return 0;
    }
    if (is_complex(tos)) {
       stkerr(" dgetrf:  [A] ", "must be real");
       return 0;
    }
    nRows    = tos->row;
    nCols    = tos->col;
    M        = nRows;
    N        = nCols;
    LDA      = N;
    piv_size = MIN(nRows, nCols);

    cop();    /* A will be overwritten, use a copy */
    A = tos->mat;

    if ((IPIV = (int *) malloc (piv_size*sizeof(int))) == NULL) {
       stkerr(" dgetrf: ",MEMNOT);
       return 0;
    }

    /* Allocate memory for the double precision version of the 
     * permutation array before computing the factor to avoid the
     * wasted time of getting the factor only to fail with insufficient
     * memory to store the permutation.
     */
    if (!matstk(piv_size, 1, "_PIV")) return 0;  /*  ( hA  --- hA hPIV ) */
    DPIV = tos->mat;

    DGETRF( &M, &N, A, &LDA, IPIV, &info ); /* on exit A contains LU */
    set_factor_lu(tos-1);
    set_lapack(   tos-1);

if (DEBUG) {
gprintf("after dgetrf info=%d\n", info);
}
    for (i = 0; i < piv_size; i++) {
        DPIV[i] = (double) IPIV[i];
    }
    free(IPIV);

    if (info) {
       if        (info < 0) {
          gprintf(" dgetrf: illegal argument %d to DGETRF",ABS(info)); 
          nc();
          stkerr("","");
          return 0;
       } else {
          gprintf(" dgetrf: U[%d,%d] is zero; matrix is singular",
                   ABS(info) , ABS(info) );
          nc();
          stkerr("","");
          return 0;
       }
    }
    return 1;
} /* 1}}} */
int dgetrs1() /* dgetrs (hLU hP hB --- hX) {{{1 */
/* Perform forward and backward substitution to solve [L][U]{X} = {B}.
 * man entry:  dgetrs {{{2
 * (hLU hP hB --- hX) Solves for {X} in the equation [L][U]{X} = {B} by performing forward and backward substitution using the triangular factors stored in [LU] and the right hand side vector {B}.
 * category: math::matrix::operator
 * related:  dgetri, dgetrf, slash, \, solve, lapack
 * 2}}}
   {{{2
   http://www.netlib.org/lapack/double/dgetrs.f
 
      SUBROUTINE DGETRS( TRANS, N, NRHS, A, LDA, IPIV, B, LDB, INFO )
*
*  -- LAPACK routine (version 3.0) --
*     Univ. of Tennessee, Univ. of California Berkeley, NAG Ltd.,
*     Courant Institute, Argonne National Lab, and Rice University
*     March 31, 1993
*
*     .. Scalar Arguments ..
      CHARACTER          TRANS
      INTEGER            INFO, LDA, LDB, N, NRHS
*     ..
*     .. Array Arguments ..
      INTEGER            IPIV( * )
      DOUBLE PRECISION   A( LDA, * ), B( LDB, * )
*     ..
*
*  Purpose
*  =======
*
*  DGETRS solves a system of linear equations
*     A * X = B  or  A' * X = B
*  with a general N-by-N matrix A using the LU factorization computed
*  by DGETRF.
*
*  Arguments
*  =========
*
*  TRANS   (input) CHARACTER*1
*          Specifies the form of the system of equations:
*          = 'N':  A * X = B  (No transpose)
*          = 'T':  A'* X = B  (Transpose)
*          = 'C':  A'* X = B  (Conjugate transpose = Transpose)
*
*  N       (input) INTEGER
*          The order of the matrix A.  N >= 0.
*
*  NRHS    (input) INTEGER
*          The number of right hand sides, i.e., the number of columns
*          of the matrix B.  NRHS >= 0.
*
*  A       (input) DOUBLE PRECISION array, dimension (LDA,N)
*          The factors L and U from the factorization A = P*L*U
*          as computed by DGETRF.
*
*  LDA     (input) INTEGER
*          The leading dimension of the array A.  LDA >= max(1,N).
*
*  IPIV    (input) INTEGER array, dimension (N)
*          The pivot indices from DGETRF; for 1<=i<=N, row i of the
*          matrix was interchanged with row IPIV(i).
*
*  B       (input/output) DOUBLE PRECISION array, dimension (LDB,NRHS)
*          On entry, the right hand side matrix B.
*          On exit, the solution matrix X.
*
*  LDB     (input) INTEGER
*          The leading dimension of the array B.  LDB >= max(1,N).
*
*  INFO    (output) INTEGER
*          = 0:  successful exit
*          < 0:  if INFO = -i, the i-th argument had an illegal value
*  2}}}
*/
{
    int     N, LDA, LDB, NRHS, info, piv_size, i,
            nRows_B, nCols_B, nRows_LU, nCols_LU;
    int    *IPIV;
    double *A, *B, *DPIV;  /* A contains LU; am using LAPACK notation */
    char    no_transpose = 'N';
int DEBUG = 0;

    /* type check the inputs {{{2 */
    if (tos->typ != MAT) {
       stkerr(" dgetrs:  {B} ", MATNOT);
       return 0;
    }
    if (is_complex(tos)) {
       stkerr(" dgetrs:  {B} ", "must be real");
       return 0;
    }
    if ((tos-1)->typ != MAT) {
       stkerr(" dgetrs:  [P] ", MATNOT);
       return 0;
    }
    if (is_complex(tos-1)) {
       stkerr(" dgetrs:  [P] ", "must be real");
       return 0;
    }
    if ((tos-2)->typ != MAT) {
       stkerr(" dgetrs:  [LU] ", MATNOT);
       return 0;
    }
    if (is_complex(tos-2)) {
       stkerr(" dgetrs:  [LU] ", "must be real");
       return 0;
    }
    if (!is_factor_lu(tos-2)) {
       stkerr(" dgetrs:  [LU] ", "must be an LU factor matrix");
       return 0;
    }
    if (!is_lapack(tos-2)) {
       stkerr(" dgetrs:  [LU] ", "must be LU factor from with LAPACK");
       return 0;
    }

    nRows_B    = tos->row;
    nCols_B    = tos->col;
    piv_size     = (tos-1)->row * (tos-1)->col;
    nRows_LU   = (tos-2)->row;
    nCols_LU   = (tos-2)->col;

    if (nRows_LU != nCols_LU) {
       stkerr(" dgetrs:  [LU] ", SQUNOT);
       return 0;
    }
    if (nRows_LU != nRows_B) {
       stkerr(" dgetrs:  [LU] and {B} ", MATSNOTC);
       return 0;
    }
    if (nRows_LU != piv_size) {
       stkerr(" dgetrs:  [LU] and [P] ", MATSNOTC);
       return 0;
    }

    /* 2}}} */

    N        = nCols_LU;
    LDA      = N;
    LDB      = nRows_B;
    NRHS     = nCols_B;

    cop();    /* B will be overwritten, use a copy */
    B = tos->mat;

    if ((IPIV = (int *) malloc (piv_size*sizeof(int))) == NULL) {
       stkerr(" dgetrs: ",MEMNOT);
       return 0;
    }

    DPIV = (tos-1)->mat;
    for (i = 0; i < piv_size; i++) {
        IPIV[i] = (int) DPIV[i];
    }
    A    = (tos-2)->mat;

    DGETRS(&no_transpose, &N, &NRHS, A, &LDA, IPIV, B, &LDB, &info);
    /* on exit B contains the solution X */
    free(IPIV);

    lop(); /* drops pivot array  */
    lop(); /* drops LU factor    */

if (DEBUG) {
gprintf("after dgetrs info=%d\n", info);
}

    if (info) {
       if        (info < 0) {
          gprintf(" dgetrs: illegal argument %d to DGETRS",ABS(info)); 
          nc();
          stkerr("","");
          return 0;
       } else {
          gprintf(" dgetrs: unexpected positive info value %d\n", info);
          nc();
          stkerr("","");
          return 0;
       }
    }
    return 1;
} /* 1}}} */

#endif
