/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2013  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* ctrl.c  April 1999

Copyright (c) 1999  D. R. Williamson

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "key.h"
#include "lib.h"
#include "mat.h"
#include "mem.h"
#include "sys.h"
#include "term.h"
#include "tex.h"
#include "tag.h"

/*----------------------------------------------------------------------

The following functions modify run level, onbuf:

      Increase       Decrease             Used by

      bufjump()      bufreturn()           source() xmain() local()
                                           exedefn()

      bufup()        bufdn()               various

      bufup1()       bufdn1()              wait() LOCK() run() runt() 

      bufascend()    bufdescend()=bufdn()  word UP

   source(): sourcing words on files
   xmain(): running words given in a VOL on the stack
   local(): running words given in a VOL on the stack, with
      restricted catalog search
   execdefn(): running text in catalog item at oncat->stk->tex
   run(): running inline addresses given in PTR on the stack
   runt(): running with trace the inline addresses given in PTR 
      on the stack

Cases of bufup() use various longjmp returns for a function's task.

Control mode: control mode is defined as the mode during interpretation
when words are being skipped in a branch not to be taken, as when words
in the IF part of an IF ... ELSE ... THEN structure are being skipped to
get to the ELSE part. 

Control mode parameters apply to each run level, and the following are 
defined in ctrl.h:

Control activity when interpreting; this defines control mode: //
char ctrlactive[NBUF]; // ctrlactive=1 when skipping text, like during
                          ELSE ... THEN when flag for IF was true or
                          during IF ... ELSE when flag was false //

char ctrlelse[NBUF]; // count of mid words like ..ELSE.., ..WHILE.. //

enum ctrlret {START=1,ENDSOU,ENDEXE,ENDWAIT, ... (see ctrl.h and
doc/jmptable.doc) 

----------------------------------------------------------------------*/

/* Pointer arrays for text being interpreted, set by bufjump,
   used by bufup:
THESE ARE IN FILE ctrl.h:
  unsigned char *iP[NBUF];     // array of ip (see inpo.h) //
  unsigned char *iPEND[NBUF];  // array of ipend //
  unsigned char *iPRECL[NBUF]; // array of iprecl //
  unsigned char *iPRECT[NBUF]; // array of iprect //
*/

/* Pointer arrays for text during DO loop, set by loopup */
unsigned char *jP[NDBUF];     /* array of ip */
unsigned char *jPEND[NDBUF];  /* array of ipend */
unsigned char *jPRECL[NDBUF]; /* array of iprecl */
unsigned char *jPRECT[NDBUF]; /* array of iprect */
/* Parameters for DO loop: */
int onloop; /* current level in stack of DO loops (0 to NDBUF-1) */
int loopindex[NDBUF];
int loopdelta[NDBUF];

/* Pointer arrays for text during BEGIN loop, set by beginup */
unsigned char *kP[NDBUF];     /* array of ip */
unsigned char *kPEND[NDBUF];  /* array of ipend */
unsigned char *kPRECL[NDBUF]; /* array of iprecl */
unsigned char *kPRECT[NDBUF]; /* array of iprect */
/* Parameters for BEGIN loop: */
int onbegin; /* current level in stack of BEGIN loops (0 to NDBUF-1) */
int loopend[NDBUF];
int whilehit[NDBUF];

#ifdef NET
/* Array for socket file descriptor at each run level: */
   int netsocket[NBUF]; /* array of socket fd */
#endif

char tokenArray[NBUF*(TOKBUF)]; /* tokens for each run level */
char *tokenq_Array[NBUF];

/*--------------------------------------------------------------------*/

int begin() /* BEGIN ( --- ) */
/* Runs a BEGIN ... UNTIL or BEGIN ... WHILE ... REPEAT loop. */
{
/* Note: flag REPEAT = flag UNTIL */
   int ret;

   if(!bufup() || beginup()<0) return 0;

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=UNTIL;
      performc();
   }
   if(!*(ctrlactive+onbuf) && ret==UNTIL) {
      if(*(loopend+onbegin)) {
         bufdn();
         begindn();
         return 1; /* <<-- returning to where begin was called */
      }
      /* doing another loop: */
      ip=*(kP+onbegin);
      iprecl=*(kPRECL+onbegin);
      iprect=*(kPRECT+onbegin);
      ipend=*(kPEND+onbegin);
      performc();
   }
   if(*(ctrlactive+onbuf) && ret==UNTIL) {
      bufdn();
      begindn();
      return 1;
   } 
   if(ret==WHILE && !*(ctrlactive+onbuf-1)) {
      *(ctrlactive+onbuf)=!*(ctrlactive+onbuf);
      performc(); 
   }
   if(ret==ABORT || ret==ERRC) {
      bufdn();
      begindn();
      if(ret==ABORT) longjmp(abortbuf,ABORT);
      return 1;
   }
/* Falling to here from bufunwind in while1(), due to error: */
   *(jmpready+onbuf)=UNTIL;
   *(loopend+onbegin)=1;
   *(ctrlactive+onbuf)=1; 
   performc(); /* limping in control mode from WHILE to REPEAT */
   return 1;
}

void begindn()
{
   onbegin--;
}

int beginup()
{
   if(onbegin==NDBUF-1) {
      stkerr(" beginup: ",MAXDOLEV);
      return -1;
   }
   onbegin++;
   *(kP+onbegin)=ip;
   *(kPRECL+onbegin)=iprecl;
   *(kPRECT+onbegin)=iprect;
   *(kPEND+onbegin)=ipend;
   *(loopend+onbegin)=0;
   *(whilehit+onbegin)=0;

   return onbegin;
}

int brak() /* [ ( --- ) */
/* Runs a phrase between brackets, [ ... ].  This phrase is simply run
   and is not included as part of any definition or inline function 
   that is currently being created. */
{
   int ret;

   if(!bufup()) return 0;

/* Vectoring the search function to caton3() (see catexe()).  This 
   means that booked items in the local and then the main library 
   will be sought: */
   *(catsearch+onbuf)=(unsigned long (*)(char *word))caton3;

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=ENDBRAK;
      perform(); /* returns here if error, like no ] ending */
   }
   if(ret!=ENDBRAK) {
      bufdn(); 
      if(ret==ABORT) longjmp(abortbuf,ABORT);

      stkerr(" brak: ",NEEDENDB);
      return 0;
   }
   bufdn(); 
   return 1;
}

int brakend() /* ] ( --- ) */
/* Jumps to brak; never returns. */
{
   if(*(jmpready+onbuf)==ENDBRAK) {
      longjmp(*(jmpenv+onbuf),ENDBRAK);
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(*(jmpenv+onbuf),ABORT);
   }
   stkerr(" brakend: ",BRAKNOT);
   return 0;
}

void bufabort(int onbuf1)
/* Unwinds jump structures down to onbufto, then jumps to onbufto. */
{
   static int onbufto;

   onbufto=onbuf1;

   if(onbuf<onbufto) {
      gprintf(" bufabort: unwinding to higher level not permitted");
      nc();
      return;
   }
/* An ALARM or TASK will probably never return to tasker(), so 
   turn BUSY off: */ 
   BUSY=0; 

   setjmp(abortbuf); /* <<-- longjmp will land here */

   while(onbufto<onbuf) {

      if(TRACE || WTRACE) {
         gprintf(" bufabort: unwinding from run level %d",onbuf);
         nc();
      }
      *(jmpready+onbuf)=ABORT;
      longjmp(*(jmpenv+onbuf),ABORT);
   }
/* Text pointer must be moved to end (in test/, run halt1 and halt2): */
   ip=ipend; /* text pointer to end; this is one consequence of abort */

   nativewords(*(catlib+onbufto)); /* back to catlib for onbufto */

   longjmp(*(jmpenv+onbufto),*(jmpready+onbufto));
}

int bufascend()
/* Jumps to higher buffer given by number popped from stack.  Retains
   current source text pointers, similar to bufup().  The high level 
   word running this function must call bufdescend() before it returns.
*/
{
   int from,upto,waslib;

   if(!popint(&upto)) return 0;

   if(upto<onbuf) {
      stkerr(" bufascend: ",NOTUP);
      return 0;
   }
   if(upto>=NBUF-1) {
      stkerr(" bufascend: ",MAXRUNLEV);
      return 0;
   }
   if(TRACE) {
      gprintf("---- Up to run level %d; stack depth = %d -----", \
      upto,stkdepth()); nc();
   }
   from=onbuf;
   *(tokenq_Array+from)=tokenq; /* remembering */

#ifdef NET
   *(netsocket+from)=SOCKFD;
#endif

   waslib=*(catlib+onbuf);
   onbuf=upto;

   *(jmpready+onbuf)=0;
   *(jmptype+onbuf)=CTRL;
   *(endbuf+onbuf)=0;
   *(bufdes+onbuf)=from;
   *(ctrlactive+onbuf)=*(ctrlactive+from);
   *(ctrlelse+onbuf)=0;

/* Setting catalog search to default and tag to MAINLIB and catlib
   to same as previous: */
   *(catsearch+onbuf)=(unsigned long (*)(char *word))caton2;
   *(cattag+onbuf)=MAINLIB; /* context() can reset this default */
   *(catlib+onbuf)=waslib; /* same lib as previous */

   token=tokenArray+loclin(onbuf,TOKBUF);
   tokenq=NULL;

#ifdef NET
   SOCKFD=*(netsocket+from);
#endif

   return onbuf;
}

void bufdescend()
/* Resets the source buffer index to the lower level when bufascend()
   was run, and leaves pointers unchanged.  A function calls bufde-
   scend() after bufascend() and before it returns. */
{
   bufdn();
}

void bufdn()
/* Decrements the source buffer index and leaves pointers unchanged.
   A function calls bufdn() after bufup() and before it returns. */
{
   *(jmpready+onbuf)=0;

   onbuf=*(bufdes+onbuf); /* onbuf is decremented */

   token=tokenArray+loclin(onbuf,TOKBUF);
   tokenq=*(tokenq_Array+onbuf); /* get remembered */

#ifdef NET
   SOCKFD=*(netsocket+onbuf);
#endif

   if(TRACE) {
      gprintf("---- Down to run level %d; stack depth = %d",
         onbuf,stkdepth()); 
      gprintf(", jmpready = %d ---",*(jmpready+onbuf));
      nc();
   }
}

void bufdn1()
/* Decrements the source buffer index, onbuf, and restores file 
   pointers.  A function calls bufdn1() after bufup1() and before 
   it returns. 

   Functions bufup1() and bufdn1() work for inlines as bufjump() and
   bufreturn() work for defines. */
{
   bufdn(); /* decrements onbuf */

   ip=*(iP+onbuf);
   iprecl=*(iPRECL+onbuf);
   iprect=*(iPRECT+onbuf);
   ipend=*(iPEND+onbuf);
}

void bufinit()
/* Initializes the run level source and control buffers at start up. */
{ 
   int i=0;

   if(TRACE) {
      gprintf(" initializing buf"); nc();
   }
   for(;i<NBUF;i++)  {
      *(pBUF+i)=NULL; 
      *(jmptype+i)=EXEC;
      *(endbuf+i)=0;
      *(bufdes+i)=0;

      *(ctrlactive+i)=0;
      *(ctrlelse+i)=0;

      *(catsearch+i)=(unsigned long (*)(char *word))caton2;
      *(cattag+i)=MAINLIB; /* context() can reset this default */
      *(catlib+i)=LIB0; 

      *(tokenq_Array+i)=NULL;

#ifdef NET
      *(netsocket+i)=-1;
#endif
   }
   onloop=-1; 
   onbegin=-1; 

   onbuf=0;    /* the lowest run level */
   onlib=LIB0; /* the start up catalog library */

   token=tokenArray+loclin(onbuf,TOKBUF);
   tokenq=NULL;

   bufreturn(); 
}

int bufjump(char *pbuf, unsigned int size)
/* Jumps to next higher source buffer and makes it size bytes with
   pointer pbuf.  A function with a successful call to bufjump() must 
   call bufreturn() before it returns. */
{
   if(TRACE) {
      gprintf("---- Begin run level %d; stack depth = %d -----", \
         1+onbuf,stkdepth()); nc();
   }
   if(onbuf>=NBUF-1) {
      stkerr(" bufjump: ",MAXRUNLEV);
      mallfree((void *)&pbuf);
      return 0;
   }
/* Saving internal pointers of current buffer */
   *(iP+onbuf)=ip;
   *(iPRECL+onbuf)=iprecl;
   *(iPRECT+onbuf)=iprect;
   *(iPEND+onbuf)=ipend;

/* Initializing internal pointers to beginning of new buffer */
   ip=(unsigned char *)pbuf;
   iprecl=ip-1;
   iprect=ip-1;
   ipend=ip+size;

   *(tokenq_Array+onbuf)=tokenq; /* remembering */

#ifdef NET
   *(netsocket+onbuf)=SOCKFD;
#endif

   onbuf++;

/* Defining current (start) pointers in new buffer: */
   *(pBUF+onbuf)=pbuf; 
   *(iP+onbuf)=(unsigned char *)pbuf;
   *(iPRECL+onbuf)=(unsigned char *)pbuf;
   *(iPRECT+onbuf)=(unsigned char *)pbuf;
   *(iPEND+onbuf)=ipend;

/* Setting control parameters to begin a new buffer: */
   *(jmpready+onbuf)=0;
   *(jmptype+onbuf)=EXEC;
   *(endbuf+onbuf)=0;
   *(bufdes+onbuf)=onbuf-1;
   *(ctrlactive+onbuf)=*(ctrlactive+onbuf-1);
   *(ctrlelse+onbuf)=0;

/* Setting catalog search to default and tag to MAINLIB and catlib
   to same as previous: */
   *(catsearch+onbuf)=(unsigned long (*)(char *word))caton2;
   *(cattag+onbuf)=MAINLIB; /* context() can reset this default */
   *(catlib+onbuf)=*(catlib+onbuf-1); 

   token=tokenArray+loclin(onbuf,TOKBUF);
   tokenq=NULL;

#ifdef NET
   SOCKFD=*(netsocket+onbuf-1);
#endif

   return onbuf;
}

void bufreturn()
/* Returns to next lower source buffer and sets pointers to its 
   pointers.  Frees mallocked pointer that was used in bufjump(). */
{
   if(onbuf!=0) {

      if(TRACE) {
         gprintf(\
            "---- Resume run level %d; stack depth = %d, cattag: %s",
            onbuf-1,stkdepth(),*(cattag+onbuf-1)); 
            gprintf(", jmpready = %d ----",*(jmpready+onbuf));
            nc();
      }
      *(jmpready+onbuf)=0;
      mallfree((void *)(pBUF+onbuf));

      onbuf--;

      ip=*(iP+onbuf);
      iprecl=*(iPRECL+onbuf);
      iprect=*(iPRECT+onbuf);
      ipend=*(iPEND+onbuf);

      token=tokenArray+loclin(onbuf,TOKBUF);
      tokenq=*(tokenq_Array+onbuf); /* get remembered */

#ifdef NET
      SOCKFD=*(netsocket+onbuf);
#endif

   }
   else {
      pBUF[0]=0; iP[0]=0; iPEND[0]=0; iPRECL[0]=0; iPRECT[0]=0;
      *(jmpready)=0;
   }
}

int bufunwind(int jmpid1, int jmpid2, int silent)
/* Unwinds higher control structures down to the one with jmpid1, and 
   does longjmp with jmpenv of jmpid1 and return set to jmpid2. */
{
   int i=0,j=0,onbuf2;
   
   if(TRACE) silent=0;

   if(!silent) {
      silent=0;
      nc();
      gprintf(" bufunwind: from level %d",onbuf);
      nc();
      jmptable();
      gprintf(\
         " bufunwind: search down for ret=%d and return with ret=%d",\
         jmpid1,jmpid2);
      nc();
      gprintf("   i: %d, onbuf-i: %d, ret: %d, typ: %d\n",\
         i,onbuf-i,*(jmpready+onbuf-i),*(jmptype+onbuf-i));

      while(*(jmpready+onbuf-i)!=jmpid1 && i<onbuf) {
         i++;
         gprintf("   i: %d, onbuf-i: %d, ret: %d, typ: %d\n",\
            i,onbuf-i,*(jmpready+onbuf-i),*(jmptype+onbuf-i));
      }
   }
   else {
      while(*(jmpready+onbuf-i)!=jmpid1 && i<onbuf) i++;
   }
   if(i<onbuf) {

      if(!silent) {
         if(i) gprintf(\
            " bufunwind: unwinding from level %d to level %d\n",\
            onbuf,onbuf-i);
         else gprintf(" bufunwind: remain at level %d\n",onbuf);
      }
      onbuf2=onbuf;
      while(j<i) {

         if(!silent)
            gprintf(" bufunwind: exit level: %d, typ: %d\n",\
               onbuf2-j,*(jmptype+onbuf2-j));

         if(*(jmptype+onbuf2-j)==CTRL) bufdn(); /* jmptype CTRL=1 */

         else bufreturn(); /* jmptype EXEC */
         j++;
      }
      if(!silent) {
         gprintf(" bufunwind: longjmp to level %d, ret: %d\n",
            onbuf,jmpid2);
         nc();
      }
      longjmp(*(jmpenv+onbuf),jmpid2);
   }
   stkerr(" bufunwind: ",JUMPNOT);
   return 0; 
}

int bufup()
/* Jumps to next higher buffer, but one that retains current source
   text pointers.  A function calling bufup() must call bufdn() before
   it returns. */
{
   if(TRACE) {
      gprintf("---- Up to run level %d; stack depth = %d -----", \
      1+onbuf,stkdepth()); nc();
   }
   if(onbuf>=NBUF-1) {
      stkerr(" bufup: ",MAXRUNLEV);
      return 0;
   }
   *(tokenq_Array+onbuf)=tokenq; /* remembering */

#ifdef NET
   *(netsocket+onbuf)=SOCKFD;
#endif

   onbuf++;

   *(jmpready+onbuf)=0;
   *(jmptype+onbuf)=CTRL;
   *(endbuf+onbuf)=0;
   *(bufdes+onbuf)=onbuf-1;
   *(ctrlactive+onbuf)=*(ctrlactive+onbuf-1);
   *(ctrlelse+onbuf)=0;

/* Setting catalog search to default and tag and catlib to same as 
   previous: */
   *(catsearch+onbuf)=(unsigned long (*)(char *word))caton2;
   *(cattag+onbuf)=*(cattag+onbuf-1);
   *(catlib+onbuf)=*(catlib+onbuf-1); 

   token=tokenArray+loclin(onbuf,TOKBUF);
   tokenq=NULL;

#ifdef NET
   SOCKFD=*(netsocket+onbuf-1);
#endif

   return onbuf;
}

int bufup1()
/* Just like bufup(), but save text file pointers before jumping
   to next run level.  Used by an inline so file pointers can be
   restored upon return from the inline, using bufdn1(). */
{
/* Saving internal pointers of current buffer: */
   *(iP+onbuf)=ip;
   *(iPRECL+onbuf)=iprecl;
   *(iPRECT+onbuf)=iprect;
   *(iPEND+onbuf)=ipend;

   return(bufup());
}

int byeflag() /* END ( --- n) */
{
   return(pushint(BYE));
}

int doexit() /* EXIT ( --- ) */
/* Jumps to dorun; never returns. */
{
   if(*(ctrlactive+onbuf)) return 1; /* ignoring */

   if(*(jmpready+onbuf)==LOOP) longjmp(*(jmpenv+onbuf),EXIT);

   if(*(jmpready+onbuf)==ABORT) longjmp(*(jmpenv+onbuf),ABORT);

   return(bufunwind(LOOP,EXIT,1));
}

int doindex0() /* ( --- I) */
{ 
   if(doindex(0)) return 1;
   stkerr(" I: ",INDEXNOT); 
   return(bufunwind(LOOP,LOOP,1) && 0);
}

int doindex1() /* ( --- J) */
{ 
   if(doindex(1)) return 1;
   stkerr(" J: ",INDEXNOT); 
   return(bufunwind(LOOP,LOOP,1) && 0);
}

int doindex(int offset)
/* Pushes loop index I or J to the stack. */
{
   if(offset>onloop) return 0;
   return(push(NUM,NULL,NOTAG,*(loopindex+onloop-offset)+XBASE,NULL, \
      NULL,0,0,NULL));
}

int dojmp() /* LOOP ( --- ) */
/* Jumps to dorun; never returns. */
{
   if(*(jmpready+onbuf)==LOOP) {
      longjmp(*(jmpenv+onbuf),LOOP);
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(*(jmpenv+onbuf),ABORT);
   }
   stkerr(" LOOP: ",BADLOOP); 
   return 0;
}

int dojmp1() /* +LOOP (n --- ) */
/* Sets loop increment and jumps to dorun; never returns. */
{
   int delta;

   if(*(jmpready+onbuf)==LOOP) {
      if(!*(ctrlactive+onbuf)) {
         if(popint(&delta)) {
            *(loopdelta+onloop)=delta;
         }
         else {
            stkerr(" +LOOP: ",BADLOOP);
            return 0;
         }
      }
      longjmp(*(jmpenv+onbuf),LOOP);
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(*(jmpenv+onbuf),ABORT);
   }
   stkerr(" +LOOP: ",BADLOOP);
   return 0;
}

int doloop() /* DO (n2 n1 --- ) */
{
   return(dorun(1));
}

int doloop1() /* ?DO (n2 n1 --- ) */
{
   return(dorun(0));
}

int done() /* done ( --- ) */
/* Bumps the text pointer of the run level just below this one to the
   end of its text being interpreted, and immediately returns to it.  
   This means the lower run level will stop interpreting and return 
   to the one below it, since its text pointer is at the end. */
{
   if(DONEOFF) { /* skip doing done this time because flag DONEOFF is 
      set to yes; but set it back to no: */
      DONEOFF=0; 
      return 1;
   }
   *(iP+onbuf-1)=*(iPEND+onbuf-1);
   return(return1());
}

int doneoff() /* done_off ( --- ) */
/* Set flag so done does nothing just the very next time it runs. */
{
   DONEOFF=1;
   return 1;
}

int dorun(int eq) 
/* Runs a DO ... LOOP from index n1 to index n2-1 (0-based).
   When eq=1 (word DO), the loop is always traversed once; when 
   eq=0 (word ?DO), the loop is skipped if n1=n2. */
{
   static int limit[NDBUF], slope[NDBUF];
   int n1,n2,ret;

   if(!bufup() || loopup()<0) return 0;

   if(!*(ctrlactive+onbuf)) {
      if(!popint(&n1) || !popint(&n2)) {
         stkerr(" DO: ",NEEDLIM); 
         bufdn(); 
         loopdn();
         return(bufunwind(LOOP,LOOP,1) && 0);
      }
      if(n1<n2 || (n1==n2 && eq)) {
         *(loopindex+onloop)=n1-XBASE; /* 1st 0-based index */
         *(limit+onloop)=n2-1; /* last 0-based index */
         *(loopdelta+onloop)=1; /* default */
         *(slope+onloop)=1;
      }
      else {
         if(n1>n2) {
            *(loopindex+onloop)=n1-XBASE; /* 1st 0-based index */
            *(limit+onloop)=n2-XBASE; /* last 0-based index */
            *(loopdelta+onloop)=-1; /* default */
            *(slope+onloop)=-1;
         } 
         else *(ctrlactive+onbuf)=1; /* n1=n2 and eq=0 */
      }
   }
   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=LOOP;
      performc();
   }
   if(!*(ctrlactive+onbuf) && ret==LOOP) {
      *(loopindex+onloop)+=*(loopdelta+onloop);
      if((*(loopindex+onloop)-*(limit+onloop))*(*(slope+onloop))>0) {
         bufdn();
         loopdn();
         return 1; /* <<-- returning to where dorun was called */
      }
      ip=*(jP+onloop);
      iprecl=*(jPRECL+onloop);
      iprect=*(jPRECT+onloop);
      ipend=*(jPEND+onloop);
      performc();
   }
   if(*(ctrlactive+onbuf) && (ret==LOOP || ret==0)) { 
      bufdn(); 
      loopdn();
      return 1;
   }   
   if(ret==ABORT) {
      bufdn(); 
      loopdn();
      longjmp(abortbuf,ABORT);
   }
   *(ctrlactive+onbuf)=1; /* setting control mode for ret=EXIT */
   performc();
   return 1;
}

int end1() /* ( --- ) */
/* Determines action of word end.  Either makes a jump or returns
   (always with 1) after setting endbuf[onbuf] to 1 if no jump is 
   pending. */
{
   if(*(jmpready+onbuf)==ENDSOU) { /* end source file */
      longjmp(*(jmpenv+onbuf),ENDSOU);
   }
   if(*(jmpready+onbuf)==ENDDEF) { /* end executing a definition */
      longjmp(*(jmpenv+onbuf),ENDDEF);
   }
   if(*(jmpready+onbuf)==ENDPTR) { /* end executing an inline */
      longjmp(*(jmpenv+onbuf),ENDPTR);
   }
   if(*(ctrlactive+onbuf)) return 1; /* skipping active: ignore end */ 

   if(*(jmpready+onbuf)==ENDLIST) { /* end making a list */
      longjmp(*(jmpenv+onbuf),ENDLIST);
   }
   if(*(jmpready+onbuf)==DEFINE) { /* end creating a definition */
      longjmp(*(jmpenv+onbuf),DEFINE);
   }
   if(*(jmpready+onbuf)==ENDBRAK) { /* end bracket mode */
      longjmp(*(jmpenv+onbuf),ENDBRAK);
   }
   if(*(jmpready+onbuf)==THEN) { /* end incomplete IF ... THEN */
      longjmp(*(jmpenv+onbuf),THEN);
   }
   if(*(jmpready+onbuf)==LOOP) { /* incomplete DO ... LOOP */
      longjmp(*(jmpenv+onbuf),LOOP);
   }
   if(*(jmpready+onbuf)==UNTIL) { /* incomplete BEGIN ... UNTIL */
      pushint(xTRUE);
      return until();
   }
   if(!*(jmpready+onbuf)) {
      *(endbuf+onbuf)=1; /* that's it, go home */
      return 1;
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(abortbuf,ABORT);
   }
   if(*(jmpready+onbuf)==VOICE) {
      longjmp(*(jmpenv+onbuf),VOICE);
   }
   if(!KEYS) {
      stkerr(" end: ",PENDING); /* *(endbuf+onbuf) remains 0 */
      gprintf("      for pending longjmp ctrlret = %d", \
         *(jmpready+onbuf)); nc();
   }
   return 1;
}

int erp() /* erp ( --- ) */
/* Report an error to the error monitor. */
{
   reperr();
   return 1;
}

int ersys() /* ersys (qS --- ) */
/* Report a system error at current location in text, with a message. */
{
   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" ersys: ",STRORVOLNOT);
      return 0;
   }
   stkerr(tos->tex,"");
   return(drop());
}

int halt() /* halt ( --- ) */
/* Stops sourcing a file and drops to a lower level. */
{
   _depthMIN=0;

   if(*(jmpready+onbuf)==ENDSOU) {
      ip=ipend; /* jump pointer to end of source file */
      return 1;
   }
   return(return1());
}

int HALT() /* HALT ( --- ) */
/* Runs bufabort() to make a longjmp down to the halting run level. */
{
   int RLHALT;

   RLHALT=rlhalt();
   _depthMIN=0;

   if(onbuf>=RLHALT) {
      clrfs();
      if(_exists("queue_halt")) { /* halt the queueing system */
         pushstr("queue_halt"); 
         xmain(0);
      }
      bufabort(RLHALT); /* longjmp down to halting run level */
   }
   return 1;              
}

int ifbranch() /* IF (f --- ) */
/* Runs an IF ... ELSE ... THEN structure. */
{
   int n1,ret;

   if(!bufup()) return 0;
   if(!*(ctrlactive+onbuf) && !popint1(&n1)) { 
      gprintf(" IF: expect number on stack"); 
      nc();
      stkerr("","");
      bufdn(); 
      return 0;
   }
   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=THEN;
      if(!*(ctrlactive+onbuf) && !n1) {
         *(ctrlactive+onbuf)=1;
      }
      performc(); 
      *(ctrlactive+onbuf)=0; /* problem if here; turn off control */
      performc();            /* and go again */ 
   }
   if(ret==THEN) {
      bufdn();
      return 1; /* <<-- returning to where ifbranch was called */
   }
   if(ret==ELSE && !*(ctrlactive+onbuf-1)) {
      *(ctrlactive+onbuf)=!*(ctrlactive+onbuf);
   }
   if(ret==ABORT) {
      bufdn();
      longjmp(abortbuf,ABORT);
   }
   performc();
   return 1;
}

int ifelse() /* ELSE ( --- ) */
/* Jumps to ifbranch; never returns. */
{
   *(ctrlelse+onbuf)+=1; /* only one allowed */
   if(!(*(ctrlelse+onbuf)>1) && (
       *(ctrlactive+onbuf) || *(jmpready+onbuf)==THEN)) {
      longjmp(*(jmpenv+onbuf),ELSE);
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(*(jmpenv+onbuf),ABORT);
   }
   stkerr(" ELSE: ",BADBRANCH);
   return 0;
}

int ifthen() /* THEN ( --- ) */
/* Jumps to ifbranch; never returns. */
{
   if(*(ctrlactive+onbuf) || *(jmpready+onbuf)==THEN) {
      longjmp(*(jmpenv+onbuf),THEN);
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(*(jmpenv+onbuf),ABORT);
   }
   stkerr(" THEN: ",BADBRANCH);
   return 0;
}

int jmptable() /* jmptable ( --- ) */
/* Display the jmpready flag values at the active run levels.  See
   doc/jmptable.doc. */
{
   int buf;
   char *p,*t;

   gprintf(" Jmp table at lev %d",onbuf);
   nc();
   gprintf("  lev   ret  typ  Lib:lib");
   nc();

/* Showing table up to onbuf, the level where this word was fired: */
   for(buf=onbuf;buf>0;buf--) {
      t=tagged(*(cattag+buf),*(cattag+buf)); /* t is like ,1:wordname */
      p=strtok(t,",:"); /* put colon delimiter at colon, skip 1st: */
      p=(char *)strtok('\0',":");  /* skip 2nd */
      p=(char *)strtok('\0',":"); /* 3rd is local lib name */
      gprintf(\
         "%4d  %4d %4d     %d:%s",buf,*(jmpready+buf),*(jmptype+buf),\
         *(catlib+buf),p);
      nc();
   }
   return 1;
}

int list() /* list: ( --- hA) */
/* Interpretive driver to gather upcoming numbers into a column matrix,
   or upcoming quote-strings into a text row matrix; listfin() finishes
   the job. */
{
   int d,ret;
   static unsigned int d1[NBUF];

   if(!bufup()) return 0;

   if(*(jmpready+onbuf-1)==ENDBRAK) { /* above bracket mode? */

   /* If in bracket mode at onbuf-1, the search function here at onbuf
      uses its search strategy, caton3().  This means that local, and
      then main, booked items will be sought: */
      *(catsearch+onbuf)=(unsigned long (*)(char *word))caton3;
   }
   *(d1+onbuf)=stkdepth();

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=ENDLIST;
      perform();
   }
   if(ret!=ENDLIST) { 
      bufdn(); 
      if(ret==ABORT) longjmp(abortbuf,ABORT);

      stkerr(" list: ",NEEDEND); 
      return 0;
   }
   d=stkdepth()-(*(d1+onbuf));
   ret=(
      pushint(d) &&
      listfin()
   );
   bufdn();
   return(ret);
}

void loopdn()
{
   onloop--;
}

int loopup()
{
   if(onloop==NDBUF-1) {
      stkerr(" loopup: ",MAXDOLEV);
      return -1;
   }
   onloop++;
   *(jP+onloop)=ip;
   *(jPRECL+onloop)=iprecl;
   *(jPRECT+onloop)=iprect;
   *(jPEND+onloop)=ipend;

   return onloop;
}

void performc()
/* Runs the control words when controls are active, skipping over
   most words.  
   Note: this runs only for interpretive processing (DEFN); inlines
   (INLI) do not use controls skipping. */
{
   char *token[NBUF];

   if(!*(ctrlactive+onbuf)) { /* running normally if no control: */
      perform();
      return;
   }
   while(!*(endbuf+onbuf) && (*(token+onbuf)=tokengetc())!=NULL) {

      if(TRACE) {
         gprintf(" ctrl next word: %s",*(token+onbuf)); nc();
      }
      if(*(token+onbuf)==0) {
         token0(); goto bottom; }

      if(strmatch(*(token+onbuf),"IF")) {
         if(!ifbranch()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"THEN")) {
         if(!ifthen()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"ELSE")) {
         if(!ifelse()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"DO")) {
         if(!doloop(1)) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"LOOP")) {
         if(!dojmp()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"?DO")) {
         if(!doloop(0)) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"+LOOP")) {
         if(!dojmp1()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"EXIT")) {
         if(!doexit()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"BEGIN")) {
         if(!begin()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"UNTIL")) {
         if(!until()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"WHILE")) {
         if(!while1()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"REPEAT")) {
         if(!repeat()) report(*(token+onbuf));
         goto bottom;
      }
      if(strmatch(*(token+onbuf),"end") ||
         strmatch(*(token+onbuf),";")) {
         end1(); goto bottom;
      }
      bottom: continue;
   }
   if(!*(endbuf+onbuf)) end1(); /* must close out any jumps */

   if(TRACE) {
      gprintf(" Returning from performc on word %s",*(token+onbuf));
      nc();
   }
}

int repeat() /* REPEAT ( --- ) */
/* Jumps to begin(); never returns. */
{
   if(*(jmpready+onbuf)==REPEAT) {
      longjmp(*(jmpenv+onbuf),REPEAT);
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(*(jmpenv+onbuf),ABORT);
   }
   stkerr(" REPEAT: ",BADLOOP);
   report(token); /* error in repeat() */
   longjmp(*(jmpenv+onbuf),ERRC);

   return 1;
}

void reperr() 
/* Unwind back to console if error report monitor returns true. */

/* Sat Nov  3 12:00:16 PDT 2012.  Fixing a four year old bug.  

   Going below the run level of the interactive prompt and getting 
   bounced out to the machine prompt has been tolerated long enough.

   Word console running the interactive prompt is on run level 2, as 
   the runaway case below shows.  But unwinding continues clear down
   to level 0 and the program simply exits to the machine's command
   prompt, losing the interactive console.

   In the program's early years, this did not happen.  But updates made
   in March 2008 for word HALT() seem to have introduced this error.

   Note that word jmptable has been permanently added to word ureset in
   file usr/key.v.  Word jmptable displays the current table of run 
   levels as the program unwinds to lower levels.  Since jmptable is 
   being run by ureset, it shows up at the highest run level in the
   jump table lists shown below:

      [dale@kaffia] /home/dale > tops
               Tops 3.2.0
      Sat Nov  3 10:24:19 PDT 2012
      [tops@kaffia] ready > 100 1 DO " running away" . nl erp LOOP
       running away
       running away
       running away
       ...
       running away
       running away
       running away
      ureset running: 1based, ok on, private, strict, stkbal, xl, -kon,
        yes catmsg, nohide
       Jmp table at lev 6
        lev   ret  typ  Lib:lib
         6     2    1     0:ureset
         5     1    2     0:DATA__
         4    12    1     0:DATA__
         3     1    2     0:DATA__
         2     2    1     0:console
         1     1    2     0:DATA__
       faulty phrase: 100 1 DO " running away" . nl erp LOOP [Enter]

      ureset running: 1based, ok on, private, strict, stkbal, xl, -kon,
         yes catmsg, nohide
       Jmp table at lev 4
        lev   ret  typ  Lib:lib
         4     2    1     0:ureset
         3     1    2     0:DATA__
         2     2    1     0:console
         1     1    2     0:DATA__
       faulty phrase: console
       runaway detected: HALT on run level 2 Sat Nov  3 10:24:24 PDT 201
      [dale@kaffia] /home/dale >

   The second running of ureset() shown above, the one on run level 4,
   is where the program gets bounced out to the machine prompt.

   Testing showed that word erp(), being called rapidly during run away,
   made a call to this word, reperr(), while reperr() was still running
   from an earlier call.  This second call caused the program to con-
   tinue its run level descent and the program ran all the way to its
   exit point.

   To prevent this, a busy flag was introduced that causes reperr() to
   return if it is still busy from an earlier call.

   With the busy flag, console and tinyc (running script usr/tiny) run
   correctly by returning to their levels of interactive prompt:

   1. Running program tops:

      [dale@kaffia] /opt/tops/tops/src > tops
               Tops 3.2.0
      Sat Nov  3 11:41:22 PDT 2012
      [tops@kaffia] ready > 100 1 DO " running away" . nl erp LOOP
       running away
       running away
       ...
       running away
       running away
      ureset running: 1based, ok on, private, strict, stkbal, xl, -kon,
        yes catmsg, nohide
       Jmp table at lev 6
        lev   ret  typ  Lib:lib
         6     2    1     0:ureset
         5     1    2     0:DATA__
         4    12    1     0:DATA__
         3     1    2     0:DATA__
         2     2    1     0:console
         1     1    2     0:DATA__
       runaway detected: HALT on run level 4 Sat Nov  3 11:41:26 PDT 201
        faulty phrase: 100 1 DO " running away" . nl erp LOOP [Enter]

      [tops@kaffia] ready >
      [tops@kaffia] ready > jmptable \ running interactively
       Jmp table at lev 3
        lev   ret  typ  Lib:lib
         3     1    2     0:DATA__
         2     2    1     0:console
         1     1    2     0:DATA__

      [tops@kaffia] ready > 

   2. Running script tiny:

      [dale@kaffia] /opt/tops/tops/usr > tiny
      % 100 1 DO " running away" . nl erp LOOP
       running away
       running away
       ...
       running away
       running away
      ureset running: 1based, ok on, private, strict, stkbal, xl, -kon,
        yes catmsg, nohide
       Jmp table at lev 7
        lev   ret  typ  Lib:lib
         7     2    1     0:ureset
         6     1    2     0:DATA__
         5    12    1     0:DATA__
         4     1    2     0:DATA__
         3     2    1     0:tinyc
         2    19    1     0:DATA__
         1     1    2     0:DATA__
       runaway detected: HALT on run level 5 Sat Nov  3 11:43:31 PDT 201
       faulty phrase: 100 1 DO " running away" . nl erp LOOP
      % 
      % jmptable \ running interactively
       Jmp table at lev 4
        lev   ret  typ  Lib:lib
         4     1    2     0:DATA__
         3     2    1     0:tinyc
         2    19    1     0:DATA__
         1     1    2     0:DATA__
      % 
*/
{
   static int busy=0;
   char *ureset="ureset";

   if(busy) return;
   busy=1;

   if(repmon(0)) {
      if(_exists(ureset)) {
         pushstr(ureset); 
         xmain(0);
      }
      gprintf(" runaway detected: HALT on run level %d %s",onbuf,
         datetime());
      nc();

      repmon(1); /* reset error report monitor */
      busy=0; /* reset busy here in case report() does not return */

   /* Sun Nov  4 08:07:57 PST 2012.  Function report() will HALT() the 
      program if an interactive console is running, and never return: */
      report(NULL);

   /* Fri Nov  9 14:18:03 PST 2012.  Run HALT() if not keyboard: */
      if(!KEYS) HALT();
   }
   busy=0;
   return;
}

int repmon(int reset)
/* Monitor for reported error runaway.
   Returns 1 if runaway detected.
   This function should be called with reset=0 only when there is 
   an error. */
{
   static int e1;
   const int treset=3; /* reset e1 to errTHRESHOLD after treset secs */

   static time_t tlast=0;
   time_t time1;

   time(&time1);

   if((time1-tlast)>treset || reset) {
      e1=errTHRESHOLD;
      tlast=time1;
      return 0;
   }
   else {
      e1--;
      if(e1<1) return 1;
      else return 0;
   }
}

void report(char *token)
/* Reports fault at current location in text and HALTs if running an
   interactive console. */
{
   unsigned char *addr;
   char *phrase;
   int adv;

   _depthMIN=0;

   addr=MIN(ip,iprect+1);
   if(!addr) return;

   adv=MAX(0,stradv((char *)addr,"\n")-1);
   if((phrase=malloc(adv+1))==NULL) {
      stkerr(" report: ",MEMNOT);
      return;
   }
   memcpy(phrase,(char *)addr,adv); 
   *(phrase+adv)='\0';

   if(token!=NULL && 
      strcmp(token,phrase) /* token and phrase are different */) 
      {
      gprintf(" fault at word: %s",token); nc();
   } 
   if(adv) {
      gprintf(" faulty phrase: "); 
      pushstr(phrase); 
      pushint(0); 
      justify();
      notrailing(); 
      dot(); 
      nc();
   }
   mallfree((void *)&phrase); 

   if(rlconsole()>-1)
      HALT(); /* stop everything if keyboard() (key.c) */
} 

int return1() /* return ( --- ) */
/* Unwinds to a lower level of "running of words" that got us here.  If
   not used inside a word, word return will appear to do nothing--the 
   interpreter simply moves to the next text token.

   The program got to this place by running text--case ENDSOU or ENDDEF,
   shown below in the cases of jmpready returns for "running of words:"

       1 ENDSOU  end of running words from a file:
                    source()
                    xmain()
                    local()
       2 ENDPTR  end of running inlined word in PTR on the stack:
                    run()
                    runt() (run with trace)
       4 ENDDEF  end of running a defined word in a catalog item at
                 oncat->stk->tex:
                    execdefn()

   Case ENDPTR applies to inlined words, and while an inline would not 
   come here to return (inlines use _endp() in exe.c), the inline may 
   be running text, at a level above it, so the text will land here.

   The next step is to drop down, using bufabort(), to the first in-
   stance of one of these three "running of words" levels. */
{
   int jmp,k=1,ret=0,RLHALT;

/* Look below this run level and jump to the first level found from
   the set ENDSOU, ENDPTR and ENDDEF: */

   RLHALT=rlhalt();
   while((onbuf-k)>RLHALT && !ret) {
      if((jmp=*(jmpready+onbuf-k))==ENDPTR ||
                               jmp==ENDDEF ||
                               jmp==ENDSOU) ret=k;
      k++;
   }
   bufabort(onbuf-ret);
   return 1; /* never get here */
}

int return2() /* return2 ( --- ) */
/* This word is useful within inlines that are inside of a word,
   when it is desired to have the word return immediately.

   Word return2 causes the word below the word saying return2 to
   return.  These words demonstrate return2:

   inline: F2 ( --- )
      [ "'F2 up one' . nl return2" "up" inlinex ]
      up "never see this" . nl
   end

   inline: F1 ( --- ) F2 "F1 after F2" . nl end

   This shows that the phrase "never see this" is skipped when word
   F1 runs F2:

      [tops@clacker] ready > F1
      F2 up one
      F1 after F2

   Replacing word return2 in F2 with word return:

   inline: F2 ( --- )
      [ "'F2 up one' . nl return" "up" inlinex ]
      up "never see this" . nl
   end
  
   and rerunning shows normal behavior, where F2 runs to completion.
   All return did in this case was to cause inline up to return,
   which it was going to do anyway.

      [tops@clacker] ready > F1
      F2 up one
      never see this
      F1 after F2
*/
{
   int jmp,k=1,ret=0,RLHALT;

/* Look below this run level and jump to the first level found from
   the set ENDSOU, ENDPTR and ENDDEF: */

   RLHALT=rlhalt();

   while((onbuf-k)>RLHALT && !ret) {
      if((jmp=*(jmpready+onbuf-k))==ENDPTR ||
                               jmp==ENDDEF ||
                               jmp==ENDSOU) ret=k;
      k++;
   }
/* Look for the next one below: */
   k=ret;
   ret=0;
   while((onbuf-k)>RLHALT && !ret) {
      if((jmp=*(jmpready+onbuf-k))==ENDPTR ||
                               jmp==ENDDEF ||
                               jmp==ENDSOU) ret=k;
      k++;
   }
   bufabort(onbuf-ret);
   return 1; /* never get here */
}

int rlconsole()
/* Return the run level where an interactive console word is running,
   or -1 if not found.

   With little effort, this function could be generalized to see if any
   word, not just one named console or tinyc, is currently running.

   Running doesn't just mean in the catalog, but actually on the cur-
   rent run stack somewhere below this function, still executing and
   not yet exited. */
{
   int buf;
   char *p;

/* Checking jmptable down from onbuf, the level where this word was 
   fired: */
   for(buf=onbuf-1;buf>0;buf--) {
      p=*(cattag+buf);
      if(strlen(p)>4) { /* need at least 5 chars for "tinyc" */
         if(strcmp(p,"console")==0 || strcmp(p,"tinyc")==0) return buf;
      }
   }
   return -1;
}

#ifdef KEYBOARD

int rlhalt()
/* Function HALT() will bufabort() to one level above the word that
   is running the interactive prompt. */

/* February 2008.  Rework the way HALT() works.
   Function HALT() will bufabort() to one level above the running
   console.  This word returns the proper level.

   Function rlconsole() will detect the run level of console words 
   console() and tinyc().  When either of these is running, rlhalt()
   returns rlconsole()+1.

   Examples of rlconsole() and rlhalt().  

   1. Running from key.v console() prompt, rlconsole() = 2, so
      rlhalt() = 3:

      [dale@clacker] /opt/tops/tops/src > tops
               Tops 3.0.1
      Sun May 27 11:25:31 PDT 2007
      [tops@clacker] ready > jmptable
       Jmp table at lev 3
        lev   ret  typ  Lib:lib
         3     1    2     0:DATA__  <<<<< want rlhalt() to return 3
         2     2    1     0:console <<<<< this is rlconsole()
         1     1    2     0:DATA__

      [tops@clacker] ready >

   2. Running script tiny that starts the tinyc() console,
      rlconsole() = 3, so rlhalt() = 4:

      % jmptable
       Jmp table at lev 4
        lev   ret  typ  Lib:lib
         4     1    2     0:DATA__  <<<<< want rlhalt() to return 4
         3     2    1     0:tinyc   <<<<< this is rlconsole()
         2    19    1     0:DATA__
         1     1    2     0:DATA__
      %
   */
{
   int lev;
   lev=rlconsole();
   if(lev>-1) return(lev+1); else return 1;
}

#else

int rlhalt()
/* Function HALT() will bufabort() to one level above the lowest and
   the program will exit. */
{
   return 1; /* 1 is one level above the lowest run level */
}
#endif

int runlevel()
/* Pushes current run level to the stack. */
{
   return(pushint(onbuf));
}

int until() /* UNTIL (f --- ) */
/* Jumps to begin(); never returns. */
{
   int f;
  
   if(!*(whilehit+onbegin) && *(jmpready+onbuf)==UNTIL) {
      if(!*(ctrlactive+onbuf)) {
         if(!popint(&f)) {
            stkerr(" UNTIL: ",FLAGNOT);
            longjmp(*(jmpenv+onbuf),ERRC);
         }
         *(loopend+onbegin)=f;
      }
      longjmp(*(jmpenv+onbuf),UNTIL);
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(*(jmpenv+onbuf),ABORT);
   }
   stkerr(" UNTIL: ",BADLOOP);
   report(token); /* error in until() */
   longjmp(*(jmpenv+onbuf),ERRC);

   return 1;
}

int wait1(int f) 
/* Begin or end a wait state.  If f is true, jump to the next run level
   and begin a wait state.  If f is false, end the latest wait state and
   drop to a lower run level.

   A wait state is started by WAIT_BEGIN.

   When it resumes after an indeterminate waiting period, the program
   will pick up again at the word following word WAIT_BEGIN.

   During the waiting period the multitasker continues to run, and data
   received from connected sockets will continue to be read and acted 
   upon.

   End the wait state with WAIT_END.

   Since the program in the wait state only responds to the multitasker
   or to a remote socket connection, it is from one of these that the 
   command WAIT_END must be received.  

   For example this program, connected on socket N to another instance 
   of this program in a wait state, could send the following command 
   that the waiting program would run:

      "WAIT_END" N remoterun

   For wait1() examples, see file sys.v, word WAIT_INIT  and its multi-
   tasker word WAITING; and file net.v where WAIT_INIT is used by words
   remoteack and remoterun1.

   This function is re-entrant, and each subsequent wait state is at a
   higher run level.  Unwinding of wait states begins with the latest
   one and ends with the first one. 

   Sun Jun  2 09:50:47 PDT 2013.  Add WTRACE. */
{
   char ch;
   int any_waiting,k,RLHALT;

   if(f) {
      if(!bufup1()) return 0; /* bufup1() preserves text file pointers,
                                 so text following WAIT is run */
      if(TRACE || WTRACE) {
         gprintf(" wait1: begin wait on run level %d",onbuf);
         nc();
      }
      setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

      if(!*(jmpready+onbuf)) { /* falls through first time */
         *(jmpready+onbuf)=ENDWAIT;

         *(_waiting+onbuf)=1;

         while(*(_waiting+onbuf)) {

            terminal(); /* returns only on a key press, socket In */
            read(In,&ch,1); /* drain a byte from socket In */

            if(TRACE) gprintf("%c",ch); /* display key pressed */
         }
      }
      if(TRACE || WTRACE) {
         gprintf(" wait1: end wait on run level %d",onbuf);
         nc();
      }
      *(_waiting+onbuf)=0;
      bufdn1();

   /* See if there are more _waiting levels below: */
      RLHALT=rlhalt();
      any_waiting=0;
      k=onbuf;
      while(!any_waiting && k>RLHALT) {
         any_waiting=*(_waiting+k);
         k--;
      }
      if(!any_waiting) {
         BUSY=0;
      }
   }
   else {
      if(TRACE || WTRACE) {
         gprintf(\
            " wait1: on run level %d, received flag to end wait",onbuf);
         nc();
      }
   /* Unwind down to highest ENDWAIT level using bufabort(): */
      RLHALT=rlhalt();
      any_waiting=0;
      k=onbuf;
      while(!any_waiting && k>RLHALT) { 
         any_waiting=*(_waiting+k);
         k--;
      }
      if(any_waiting) bufabort(1+k); /* longjmp back to ENDWAIT */
      else {
         BUSY=0;
      }
   }
   return 1;
}

int WAIT_BEGIN() /* WAIT_BEGIN ( --- ) */
/* Start an indeterminate wait period. */
{
   return(wait1(xTRUE));
}

int WAIT_END() /* WAIT_END ( --- ) */
/* End a wait period. */
{
   return(wait1(xFALSE));
}

int while1() /* WHILE (f --- ) */
/* Controls jump in BEGIN ... WHILE ... REPEAT loop. */
{
   int f;

   *(whilehit+onbegin)=1;
   if(*(jmpready+onbuf)==UNTIL) {
      if(!*(ctrlactive+onbuf)) {

         if(!popint(&f)) {
            stkerr(" WHILE: ",FLAGNOT);
            longjmp(*(jmpenv+onbuf),ERRC);
         }
         *(loopend+onbegin)=!f;
         if(*(loopend+onbegin)) {
            longjmp(*(jmpenv+onbuf),WHILE);
         }
      }
      return 1;
   }
   if(*(jmpready+onbuf)==ABORT) {
      longjmp(*(jmpenv+onbuf),ABORT);
   }
   stkerr(" WHILE: ",BADBRANCH);
   report(token); /* error in while1() */
   return(bufunwind(UNTIL,ERRC,0));
}
