\ {{{1 GNU General Public License
{
Program Tops - a stack-based computing environment
Copyright (C) 1999-2010  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} 
}

{  File signal.v  May 2001

   Copyright (c) 2001  D. R. Williamson
   
   References: 

      1. Bendat, J. S., and Piersol, A. G., "Random Data Analysis and 
         Measurement Procedures," 3rd edition, 2000, J. W. Wiley & Sons,
         Inc.

      2. Parks, T. W., and C. S. Rurrus, "Digital Filter Design," 1987,
         J. W. Wiley & Sons, Inc.

      3. Bendat, J. S., and Piersol, A. G., "Engineering Applications
         of Correlation and Spectral Analysis," 1980, J. W. Wiley 
         & Sons, Inc.

      4. Oppenheim, A. V., and R. W. Schafer, "Digital Signal Process-
         ing," 1975, Prentice-Hall, Inc.

\-----------------------------------------------------------------------

   The words in this file are listed below.

   syspath "signal.v" cat asciiload this " inline:" grepr reach dot

   inline: 3octave (hA hF f0 f1 --- hR hF1) \ one-third octave bands
   inline: bands (hX hFc hdF --- hY) \ filter X into bands dF at Fc
   inline: band_table (hF hA --- hXY) \ table for bands
   inline: cheby1 (N Rp Rs f1 f2 S --- ) \ initialize for cheby1 filter
   inline: cheby2 (N Rp Rs f1 f2 S --- ) \ initialize for cheby2 filter
   inline: ellip (N Rp Rs f1 f2 S --- ) \ initialize for elliptic filter
   inline: FFT (hX S N --- hFr hFi) \ fast Fourier transform of X
   inline: fft1 (hX S N --- hFr hFi) \ N-point Fourier transform of X
   inline: hanning (ht --- hH) \ cosine-squared tapering for time list
   inline: highPass (hy S fHz --- hY ht) \ remove frequencies below fHz
   inline: IFFT (hFr hFi S Nt --- hY) \ inverse of FFT from fft1
   inline: ifft1 (hFr hFi S Nt --- hY) \ inverse of FFT from fft1
   inline: lamp (hA ht hf --- hFr hFi) \ complex Fourier transform of A
   inline: lowPass (hy S fHz --- hY ht) \ remove frequencies above fHz
   inline: nsteady (hA n --- hA1) \ steady using n latest at each step
   inline: nwave (hA n --- hA1) \ fluctuation from steady at each step
   inline: pad2 (hA --- hA1) \ pad rows of A to next higher power of 2
   inline: resample (hA2 S2 S1 --- hA1) \ data A2 at S2 into A1 at S1
   inline: resample1 (hA2 S2 S1 --- hA1) \ data A2 at S2 into A1 at S1
   inline: Sfreqs (S N --- hSf) \ N frequencies for sample rate S
   function (U, t) = sine_sweep(f1, f2, dt, oct) 
   inline: three_octave (hA hF f0 f1 -- hR hF1) \ one-third octave bands
   inline: tshift (hS ht dt --- hS1) \ time shift signal S
   inline: tuning (hRef hSig ht --- hSig1) \ tuning Sig to match Ref
   inline: wfft (hX ht t1 N --- hFr hFi) \ windowed N-point FFT at t1
   inline: wPSD (hP S Nf --- hPSD hF) \ PSD using adjacent windows

\  Words that run the Matlab engine:
   inline: mbandpass ( --- hB hA) \ make bandpass filters
   inline: mfft (hX S N --- hFr hFi) \ N-point Fourier transform of X
   inline: mfilter (hB hA hX --- hY) \ filter X with filter [A,B]
   inline: mfreqz (hB hA N --- hHr hHi hW) \ Z-transform freq response
   inline: mifft (hFr hFi S Nt --- hY) \ inverse of FFT from mfft
   inline: mprony (hH nB nA --- hB hA) \ prony filter
   inline: mresample (hA2 S2 S1 --- hA1) \ data A2 at S2 into A1 at S1
   inline: showFilter (Fc dF Nf--- ) \ bandpass filter for Fc, dF
}
\-----------------------------------------------------------------------

   CATMSG push no catmsg

\  Defining FFT and IFFT resources.

   "no" "using_matlab" macro

   "fft" exists? (FFTW?) 

   IF \ using FFTW functions:

      inline: FFT (hX S N --- hFr hFi) \ fast Fourier transform of X
         fft1 ;
      inline: IFFT (hFr hFi S Nt --- hY) \ inverse of FFT from fft1
         ifft1 ;
 
      " using FFTW functions" (qS)
      (qS) this "FFT" "MSG" bank, "IFFT" "MSG" bank

   ELSE "MATLAB" exists? (Matlab?) 
      IF \ else using Matlab functions:

         inline: FFT (hX S N --- hFr hFi) \ fast Fourier transform of X
            mfft ;
         inline: IFFT (hFr hFi S Nt --- hY) \ inverse of FFT from mfft
            mifft ;
         "engOn" missing IF matlab THEN

         " using Matlab FFT functions" (qS)
         (qS) this "FFT" "MSG" bank, (qS) "IFFT" "MSG" bank

         "yes" "using_matlab" macro
      THEN
   THEN

   "FFT" missing
   IF 
      inline: FFT MSG . ;
      inline: IFFT MSG . ;

      " words for FFT are not present"
      (qS) this "FFT" "MSG" bank, (qS) "IFFT" "MSG" bank
      
      pull catmsg 
      CATMSG IF "FFT" "MSG" yank (qS) . nl THEN
      halt
   THEN

   CATMSG IF "FFT" "MSG" yank (qS) . nl THEN

   "*conj" missing IF "mmath.v" source THEN

\-----------------------------------------------------------------------
{
   This demo is run with the phrase:

      "signal.v" "FFT/PSD demo" msource
 
   A curve of two unit-amplitude sine waves at 40 and 400 Hz, plus some
   uniform noise, is created for signal processing.

   The first graph shows the two FFTs of the curve.  The blue curve
   is an ordinary FFT (word fft1) and the green one is an FFT of the 
   data with Hanning filtering (word wfft).

   The second graph shows the original curve overlaid by one recreated 
   from the inverse of the FFT (word mifft).  Blue is the original and 
   green is the recreated--the green curve should match the blue one.

   The last curve shows PSDs of the data created from averaging over
   four adjacent windows, with and without Hanning filtering (word 
   wPSD).

   FFT/PSD demo

      "wfft" missing IF "signal.v" source THEN 
      "wfft" missing IF halt THEN

      "sine" missing IF mmath.v source THEN

      0.001 makes dt     \ time step
      1 dt slash makes S \ sample rate
      8192 into Nt       \ time points in data
      Nt into Nf         \ points in FFT

    \ Sine amplitude and noise level:
      1.0 into A
      1 is N (even with large N, say 10, the sines still come through)

    \ Making sine waves:
      A, 400 (Hz) rad/cyc star, 0 dt Nt sine (y1 t) into t
      A,  40 (Hz) 2pi star, 0 dt Nt sine (y2 t) drop
      (y1 y2) plus (y)

    \ Adding uniform random noise of plus-or-minus N:
      seed0 seedset
      (y) N this negate, them dims ranreal plus (y)

    \ Detrending:
      (y) this trend detrend into y

    \ Making FFTs with and without Hanning filtering:
      "wfft" "HAN" yank push
      yes "wfft" "HAN" bank
      y t 0 Nf wfft "Hi" book, "Hr" book \ Hanning filtered FFT
      y S Nf FFT "Fi" book, "Fr" book    \ ordinary FFT
      pull "wfft" "HAN" bank

    \ The FFT peak at 40 and 400 Hz should be approximately A*T/2, 
    \ where T=Nt*dt; 
    \ for A=1, Nt=8192 and dt=.001, expect peak=4.096 (12.25 dB).

    \ Frequencies that go with the FFTs:
      S Fr rows Sfreqs into f \ frequencies 0 to Nyquist
 
    \ Plotting FFT magnitude (dB) versus frequency, first tossing 
    \ several initial low frequency points: 
      4 ndx f rows 3 less items (hRows) push

      Fr peek reach, Fi peek reach *conj sqrt db20
      Hr peek reach, Hi peek reach cmag db20

      (F H) park f pull reach 

      "MATLAB" exists?
      IF "mplot" missing IF matlab THEN
         mplot mgrid
         "FFT magnitude" "Frequency, Hz" "FFT, dB" mlabel 
         list: 30 500, -40 20 ; maxis_set
      ELSE plot 
      THEN
      pause

    \ Computing the inverse FFT of F:
      Fr Fi S Nt IFFT (hY)

    \ Plotting the original time history and the inverse FFT:
      (hY) y 1st them rows items reach swap (y hY) park 
      t 1st them rows items reach (hyY ht)

      "MATLAB" exists?
      IF mplot mgrid
         list: 0 .5 3 -3 ; maxis_set \ showing the first 0.5 seconds
         "Response" "Time, sec" "Acceleration, g" mlabel
      ELSE plot
      THEN
      pause

    \ Making a PSD of y, averaging over four windows.

       \ Without Hanning filter:
         "wfft" "HAN" yank push
         no "wfft" "HAN" bank
         y, S, Nf four slash, wPSD "Freq" book, "PSDno" book

       \ With Hanning filter:
         yes "wfft" "HAN" bank
         y, S, Nf four slash, wPSD "Freq" book, "PSDyes" book
         pull "wfft" "HAN" bank

    \ Plotting the PSDs:
      
      PSDno PSDyes park, Freq 

      "MATLAB" exists?
      IF mYsemilog mgrid
         list: 30 500, 1e-4 1 ; maxis_set
         PSDno Freq Frms "%4.3f" format "Response PSD, RMS=" swap cat
         "Frequency, Hz" "Acceleration, g^2/Hz" mlabel
      ELSE plot
      THEN

   private halt (end of FFT/PSD demo)

\-----------------------------------------------------------------------

\  FFT of a step function.

\  This line runs this region:
\     "signal.v" "\  FFT of a step function." msource

   "Sfreqs" missing IF "signal.v" source THEN

 \ Making discrete FFT:
   list: 1 1 0      0 ; 1333 refine is A1
   list: 0 1 1.0001 2 ; 1333 refine is t1

   one t1 2nd pry slash is S

   A1 S those rows FFT *conj into H1
   S H1 rows Sfreqs 1E-1 that 1st poke into f1
 
   H1 1st 301 items reach, f1 1st 301 items reach 
   "MATLAB" exists?
   IF "mplot" missing IF matlab THEN mloglog mgrid
   ELSE log10 swap log10 swap plot 
   THEN pause

 \ Making continuous FFT:
   list: 1 1 0      0 ; is A
   list: 0 1 1.0001 2 ; is t
   0.01 500 uniform into f

   A t f lamp *conj 
   these numbad swap        \ 1st is NANQ to be gotten rid of
   (hA) that rake drop (hA) \ rake out NANQ from A
   f rot rake drop (hf)     \ rake out row of NANQ from f

   "mplot" exists? 
   IF mloglog mgrid 
   ELSE log10 swap log10 swap plot pause plotclose
   THEN

   private halt

\-----------------------------------------------------------------------

\  Demonstrate words lowPass and highPass.

\  This line runs this region:
\     "signal.v" "\  Demonstrate words lowPass and highPass." msource

   "highPass" missing IF "highPass" sourceof source THEN
   "sine" missing IF "sine" sourceof source THEN

   5e-4 is dt
   1 5 2pi *  0 dt 10000 sine drop
   1 10 2pi * 0 dt 10000 sine drop
   1 100 2pi * 0 dt 10000 sine push plus plus pull

   "t1" book, "Y1" book
   1 t1 2nd pry slash is S
 
   \ Before high pass takes out 5 Hz:
   Y1 S those rows wPSD
   swap 1st 520 items reach
   swap 1st 520 items reach plot
   pause

   \ After high pass takes out 5 Hz:
   Y1 S 9.9 (Hz) highPass drop "Yhigh" book
   Yhigh S those rows wPSD
   swap 1st 520 items reach 
   swap 1st 520 items reach plot 
   pause

   \ After low pass takes out 100 Hz:
   Yhigh S 10.1 (Hz) lowPass drop "Y10" book
   Y10 S those rows wPSD
   swap 1st 520 items reach
   swap 1st 520 items reach plot
   pause 
   plotclose

   private halt
}
\-----------------------------------------------------------------------

\  Words.

   "logspace" missing IF "math.v" source THEN

   inline: 3octave (hA hF f0 f1 --- hR hF1) \ one-third octave bands
{     RMS of A(F) for one-third octave bands in range f0 to f1.
      Returned R contains one-third octave band RMS sine amplitudes
      and F1 contains the corresponding frequencies.  The last few
      frequencies in F1 may be greater than f1.

      Example:
         "logspace" missing IF "math.v" source THEN
         1000 ones dup 2 * park
         1 1000 999 logspace
         2 800 3octave
}
      "f1" book "f0" book
      "F" book "A" book

      f1 log2 f0 log2 - rounded "n" book
      f0 f0 two n pow * (fmax) n 3 * logspace "F1" book

      depth push
      one A cols null (hA)

      1st "n" book
      BEGIN
         F F1 n pry F1 n tic pry inclusive "M" book
         F M rake "F" book drop
         A M rake "A" book (hA0) sine_rms hand bend
         one n bump
         n F1 rows >=
      UNTIL
      depth pull less pilen

    \ Put small amplitude on null rows:
      (hR) dup nullr dup rev rake (hNull hR)
      (hR1) dup minfetch 2drop (min)
      (min) two / rot plus swap rot tier (hA)

      (hR) "_R" naming
      F1 cop "_F1" naming
   end
   
   inline: bands (hX hFc hdF --- hY) \ filter X into bands dF at Fc
{     Filter data X into bands of frequencies in the range Fc plus-
      or-minus dF/2.

      Parameters for the bandpass filter must have been set up previ-
      ously: see words ellip, cheby1, and cheby2.  Undefined (UDEF) 
      values for Fc and dF are ok when the parameters are set up be-
      cause they are set here as each band is done.

      Here is an example for an ellip bandpass filter:

         Order, Rp, Rs, UDEF (Fc), UDEF (dF), Fs ellip

      Returned Y has one column for each {Fc,dF} row-pair in F.
}
      "ellip" "function" yank into function, function exists? not 
      IF " require word " function cat ersys return THEN

      "dF" book, "Fc" book
      (hX) push, Fc rows 1st
      DO
       \ Making the filter for Fc(i) and dF(i):
         Fc I pry function "Fc" bank
         dF I pry function "dF" bank
         function main (hB hA)

       \ Filtering the data:
         peek (hX) mfilter (hYi)

      LOOP pull (hX) drop, (hY1 ... hYn) Fc rows parkn
      "_Y" naming
   end

   inline: band_table (hF hA --- hXY) \ table for bands
{     Band pairs F(i)={f1,f2} and amplitudes A(i) into XY table.

      F contains f1 in the first column and f2 in the second column.

      Note: creates incorrect table if a pair {f1(i),f2(i)} overlaps
      another pair {f1(j),f2(j)}.

      Example:
         X Fc dF bands (hY) bend max1 1st catch into Ymax    
         Fc dF -, Fc dF +, park (hF) Ymax band_table (hXY)
         (hX) this 2nd catch swap 1st catch plot
}
      [ 0.0001 is eps, "2 / .51 + rounded 2 * 1 max" "2next" inlinex ]
      abs push (hF)
      this maxfetch 2drop (fmax) 2next makes Fs, this 
      (hF) 1st catch (f1) eps +d peek (hA) park, that
      (hF) 2nd catch (f2) eps -d pull (hA) park pile, swap
      (hF) chain these rows one null park pile yes sort
      this 1st catch this one lag = no that 1st poke rake drop
      no no park, swap Fs no park, pile pile "_XY" naming
   end

   inline: cheby1 (N Rp Rs f1 f2 S --- ) \ initialize for cheby1 filter
\     Note: for cheby1 filter, Rs is not used.
\     Example: 3 0.1 UDEF 15 5 2000 cheby1 mbandpass
      ellip, "ellip" "function" yank (qFunc) "Type" 2 rev bank
   end

   inline: cheby2 (N Rp Rs f1 f2 S --- ) \ initialize for cheby2 filter
\     Note: for cheby2 filter, Rp is not used.
\     Example: 3 UDEF 20 15 5 2000 cheby2 mbandpass
      ellip, "ellip" "function" yank (qFunc) "Type" 3 rev bank
   end

   inline: ellip (N Rp Rs f1 f2 S --- ) \ initialize for elliptic filter
\     Example: 4 0.1 10 10 20 2000 ellip mbandpass
      [ "mbandpass" is function ]
      function exists? not 
      IF " require word " function cat ersys return THEN

      1 function "Type" bank
      function "S" bank
      function "f2" bank
      function "f1" bank
      function "RS" bank
      function "RP" bank
      function "N" bank
   end

   inline: fft1 (hX S N --- hFr hFi) \ N-point Fourier transform of X
{     N-point Fourier transform of real data X taken at sample rate S.

      N is forced to be even, and returned Fr and Fi have N/2 rows.
      Pads with zeroes if length of X < N; truncates if X length > N.

      Word Sfreqs provides the frequencies (Hz) that correspond to rows
      of Fr and Fi
}
      [ { Two words for scaling and unscaling are included in this local
          library; they are used by this word, and words mfft, ifft1,
          and mifft: 

          1) Word fft_scale scales the fft and returns just the left
             portion.  PSD calculations will require a factor of two to
             account for the missing portion (for example, see 'Scaling
             the PSD' in word wPSD).

          2) Word fft_unscale removes scaling and recreates the orig-
             inal fft. 
        } 

        {" fft_scale (hFr hFi sfac --- hFr hFi) \ extract half and scale

         { Taking left half and scaling by sfac.  

           Values in Fr at row 0 and row N/2 are real and independent, 
           and must be preserved for accurate inverse fft.

           To keep all terms in N/2 storage, the value at N/2 is saved
           temporarily in row 0 of Fi where the value is always zero.
         } 
         "sfac" book
         1st those rows two slash items push

         (hFi) peek reach that 
         (hFr) peek rows tic ndx reach F *f \ Fr(N/2)
         1st them ram \ Fr(N/2) into Fi(0)
         (hFi) sfac *f "_Fi" naming swap

         (hFr) pull reach sfac *f "_Fr" naming swap

        "} "fft_scale" inlinex

        {" fft_unscale (hFr hFi sfac --- hFr hFi) \ reassemble the fft
        \ This word undoes the work of word fft_scale.
         "sfac" book (hFi) push

         (hFr) this reversed one lag peek 
         (hFi) 1st reach F /f 1st them ram pile \ Fi(0) into Fr(N/2)
         sfac /f "_Yr" naming pull

         (hFi) this reversed one lag negate \ complex conjugate
         one those cols null 1st them ram pile \ zero at Fi(0)

         sfac /f "_Yi" naming 

        "} "fft_unscale" inlinex

        1E-20 "F" book
      ]
      tic two slash integer two star "N" book \ force N to be even

      "S" (1/dt) book

      these rows N <> 
      IF (1st catch) these rows N < 
         IF N those rows less those cols null pile \ padding zeroes
         ELSE 1st N items reach \ truncating elements beyond N
         THEN
      THEN

      (hX) this push cols 1st
      DO peek (hX) I catch fft (hFr hFi) \ word fft scales by 1/N
      LOOP (hFr1 hFi1 hFr2 hFi2 ...)

      pull cols two * parkn 
      these cols two spikes 0= claw (hFr hFi)

    \ Undo 1/N scaling by fft, scale by dt, and pack to left plane
    \ so returned Fr and Fi have N/2 rows:
      N S / (N/S) fft_scale (hFr hFi)
   end

   inline: hanning (ht --- hH) \ cosine-squared tapering for time list
{     Cosine tapered window:

         H(i) = 1 - cos(S(i))*cos(S(i))

      where S(i) = pi*[t(i) - t(1st)]/T
            T = t(last) - t(1st)
}
      [ \ Factor to apply to frequency domain results when using
        \ Hanning tapering:
          8 3 slash sqrt is HanningAdjust
      ]
      this 1st pry -d \ t - t(1st)
      pi that one endmost ontop slash
      (t pi/T) *f cos again *by
      these rows ones swap less
   end

   inline: highPass (hy S fHz --- hY ht) \ remove frequencies below fHz
\    WARNING: Stack diagram has changed; sample rate S instead of ht.
{     Data y at sample rate S.

      If TAPER=yes, performs zero-to-one tapering at the beginning
      0.1% of the data, and one-to-zero tapering on the ending 0.1% of
      the data.  This removes extraneous spikes at the beginning and
      end of the reconstructed data, but means reconstruction will not
      be exact if no frequencies are removed.
}
      [ 0.1 (%) 100 slash is taper
        list: 0, taper, 1 taper less, 1 ; (X)
        list: 0 1 1 0 ; (y) park "winXY" book

        no is TAPER
      ]
      "fHz" book
      "S" book

      (hy) "y" book \ detrending y can give bias to high passed Y(t)
      one S slash y rows uniform (ht) dup push

      TAPER
      IF
\        Tapering the endmost 0.1% at the beginning and end of the data.

\        NOTE: tapering means the inverse FFT in a case where no fre-
\        quencies are removed will not exactly agree with the original 
\        data, i.e., Y will not agree with y.

         (ht) winXY swap (ht) this one endmost ontop /f lerp (hW)
         (hW) y cols clone y *by into y
      ELSE (ht) drop
      THEN

\     FFT:
      y, S, those rows (Nt), FFT "Gi" book, "Gr" book
      S Gr rows Sfreqs makes Fd

      1st Fd fHz bsearch IF nit THEN (n) \ keep n and above
      (n) qdx items Fd rows teeth (hRake) push

      Gr peek rake swap dims null swap peek tier (hGr)
      Gi peek rake swap dims null swap pull tier (hGi)

\     Inverse FFT:
      (hGr hGi) S y rows (Gr Gi S Nt) IFFT (hY) "_Y" naming
      pull "_t" naming (hY ht)
 
\     Release memory of big items in local library:
      purged is y, purged is Fd, purged is Gr, purged is Gi
   end

   inline: ifft1 (hFr hFi S Nt --- hY) \ inverse of FFT from fft1
{     Restores the right half removed, and scaling done, by word fft1.
      Incoming S is sample rate; Nt points in Y are created.

      NOTE: Padding of zeroes may be necessary to create Nt points.
            To put zeroes at the beginning of Y, input negative Nt; 
            otherwise, padded zeroes are at the end of Y.
}
      this abs "Nt" book, (Nt) 0< "0front" book

      those rows two star swap (2N S) slash \ 2N/S

      (hFr hFi sfac) "fft1" "fft_unscale" localrun \ recreate full fft
      (hFr hFi) "Fi" book "Fr" book

      Fr cols 1st
      DO Fr I catch Fi I catch ifft (hYr) LOOP
      Fr cols 
      purged "Fi" book
      purged "Fr" book
      (hY1 hY2 ... hYN N) parkn (hY)

      these rows Nt <> 
      IF these rows Nt <
         IF Nt those rows less, those cols null
            (hYi hNull) 0front IF swap THEN pile
         ELSE 1st Nt items reach
         THEN
      THEN

      (hYr) "_Y" naming
   end

   inline: lamp (hA ht hf --- hFr hFi) \ complex Fourier transform of A
{     At frequencies f, computing the complex Fourier transform of time
      history A. 

      begin man lamp
         X11 not IF " X11 graphics required" . halt THEN

         "lamp" missing IF "signal.v" source THEN
         "lamp" missing IF halt THEN \ no FFT code if still missing

         "sine" missing IF mmath.v source THEN

         "lamp" "HAN" yank (f)

       \ Unit sine at 10 Hz:
         1 10 rad/cyc * 0 0.001 3000 sine "t" book, "A" book
         t one endmost ontop, t 1st pry less "T" book \ sample period

       \ List of frequencies:
         list: 9 10 11 ; 30 refine "_Freq" book

       \ Making PSDs without and with Hanning tapering:

            "lamp" "HAN" yank push
            no  "lamp" "HAN" bank A t _Freq lamp 
            (hFr hFi) *conj 2 T / *f "A_PSD1" book

            yes "lamp" "HAN" bank A t _Freq lamp
            (hFr hFi) *conj 2 T / *f "A_PSD2" book

            A_PSD1 A_PSD2 park _Freq Frms "A_RMS" book
            pull "lamp" "HAN" bank

       \ Displaying results:
         {"
            The green PSD curve uses Hanning cosine tapering of the 
            signal, which reduces side lobes at the expense of wider 
            bandwidth.  But with the side lobe reduction, the Hanning 
            curve gives a more accurate RMS value over a limited 
            frequency range as these RMS results show:
         "} chop vol2str cr sp 72 .out nl
 
         "RMS for 10 Hz sine of unit amplitude will be 0.707 for -INF "
         "to +INF.  Here is the result for 9 to 11 Hz shown above:" 
         pile chop vol2str (qS) nl
         5 spaces . (qS) 72 .out nl
         9 spaces . A_RMS 1st pry "%5.3f" format . " (blue)" . nl
         9 spaces . A_RMS 2nd pry "%5.3f" format . " (green)" .

         A_PSD1 A_PSD2 park, _Freq plot pause plotclose 
        (f) "lamp" "HAN" bank
      end man lamp
}
      [ no is HAN \ initial setting is for no Hanning tapering
      ]
      HAN
      IF rev (hA ht) dup hanning them cols (Acols) clone (hH)
         rot (hH hA) *by swap (hA ht) rot
      THEN 

      (hA ht hf) _lamp (Fr Fi)

      HAN 
      IF (Fr Fi) swap "hanning" "HanningAdjust" yank, dup rev *f rev *f
      ELSE swap
      THEN (Fi Fr) "_Fr" naming swap "_Fi" naming
   end

   inline: lowPass (hy S fHz --- hY ht) \ remove frequencies above fHz
{     Data y at sample rate S.

      If TAPER=yes, performs zero-to-one tapering at the beginning 
      0.1% of the data, and one-to-zero tapering on the ending 0.1% of 
      the data.  This removes extraneous spikes at the beginning and 
      end of the reconstructed data, but means reconstruction will not 
      be exact if no frequencies are removed.
}
      [ 0.1 (%) 100 slash is taper
        list: 0, taper, 1 taper less, 1 ; (X)
        list: 0 1 1 0 ; (y) park "winXY" book

        no is TAPER
      ]
      "fHz" book "S" book
      (hy) "y" book \ detrending y can give bias to low passed Y(t)
      one S slash y rows uniform (ht) dup push

      TAPER
      IF
\        Tapering the endmost 0.1% at the beginning and end of the data.

\        NOTE: tapering means the inverse FFT in a case where no fre-
\        quencies are removed will not exactly agree with the original 
\        data, i.e., Y will not agree with y.

         (ht) winXY swap (ht) this one endmost ontop /f lerp (hW)
         (hW) y cols clone y *by into y

      ELSE (ht) drop
      THEN

\     FFT:
      y S those rows FFT "Gi" book, "Gr" book
      S Gr rows Sfreqs makes Fd

      1st Fd fHz bsearch IF nit THEN (n) \ keep below n
      (n) qdx items Fd rows teeth (hRake) push

      Gr peek rake dims null peek tier (hGr)
      Gi peek rake dims null pull tier (hGi)

\     Inverse FFT:
      (hGr hGi) S y rows (Gr Gi S Nt) IFFT (hY) "_Y" naming
      pull "_t" naming (hY ht)

\     Release memory of big items in local library:
      purged is y, purged is Fd, purged is Gr, purged is Gi
   end

   inline: nsteady (hA n --- hA1) \ steady using n latest at each step
\     Steady signal using n previous values at each step (row) in A.
\     The linear portion of A that goes with the oscillatory portion
\     from word nwave.

\     Note: A n nsteady, A n nwave plus \ gives original A

      depth two less push

      "n" book push

      peek 1st n items reach

      peek rows n tic
      DO peek I tic n less, n items reach
         (hA) this again trend detrend less n ndx reach
      LOOP
      pull drop depth pull less pilen "_nsteady" naming
   end

   inline: nwave (hA n --- hA1) \ fluctuation from steady at each step
\     Oscillatory signal using n previous values at each step (row) in 
\     A.
\     The oscillatory portion of A that goes with the steady portion
\     from word nsteady.

\     Note: A n nsteady, A n nwave plus \ gives original A

      depth two less push

      "n" book push

      n peek cols null

      peek rows n tic
      DO peek I tic n less, n items reach
         (hA) this trend detrend n ndx reach
      LOOP
      pull drop depth pull less pilen "_nwave" naming
   end

   inline: pad2 (hA --- hA1) \ pad rows of A to next higher power of 2
      these rows log2 tic integer 2 swap pow (N)
      (hA N) those rows less those cols null pile \ padding zeroes
   end

   inline: resample (hA2 S2 S1 --- hA1) \ data A2 at S2 into A1 at S1
{     For data A2 at rate S2, resample to create data A1 at rate S1.  

      Each column of A2 is a time history of samples taken at rate S2.

      A2 is first low pass filtered to Nyquist frequency S1/2, and then
      resampled at S1 to produce A1 of the same total duration.

      Maintaining RMS: if flag RMS_SCALE=yes, resampled A1 is scaled to
      have the RMS of A2; the scale factor is reported and should be 
      close to 1 unless significant response has been removed by low-
      pass filtering.

      The default is to not scale to maintain RMS.  Banking yes for 
      flag RMS_SCALE will enable scaling:
         yes "resample" "RMS_SCALE" bank
}
      [ no "RMS_SCALE" book yes "VERBOSE" book ]
      "S1" book "S2" book

\     Save original RMS for later scaling:
      (hA2) its rms into RMS

\     Pad the rows of A2 to a power of two:
      (hA2) its rows S1 S2 slash star even into newR, (hA2) pad2 (hA2)

\     Low pass filtering to S1/2 (Reference 1, page 398):
      (hA2) S2 S1 two slash lowPass (hA1 ht) \ filtering by FFT

\     Decimation:
      one S1 slash those rows S1 S2 slash star uniform (ht1)
      (hA1 ht ht1) lerp1 (hA1) \ resampling at dt1=1/S1

\     Scaling columns to force resampled rms to equal original RMS:
      RMS_SCALE
      IF these cols 1 =
         IF (hA1) RMS that rms (RMS1) slash (hA1 ratio) 
            VERBOSE IF " resample: scale factor:" . dup . nl THEN 
            (hA1 ratio) *f

         ELSE (hA1) RMS that rms (RMS RMS1) /by bend (hA1 hRatio) 
            VERBOSE 
            IF " resample: column scale factors: " . 
               (hRatio) dup "%6.2f" those cols cats format . nl
            THEN
            (hA1 hRatio) those rows repeat *by

         THEN
      THEN

      (hA1) 1st newR items reach "_resample" naming
   end

   inline: resample1 (hA2 S2 S1 --- hA1) \ data A2 at S2 into A1 at S1
{     For data A2 at rate S2, resample to create data A1 at rate S1.  
      Each column of A2 is a sample time history at rate S2.

      A2 is simply decimated from S2 to S1 with no low pass filtering.

      This word is used to hear the difference low pass filtering makes
      in resampling sound (voice) files.  There is a difference: in the
      phrase 'Festival speech synthesis system,' s-sounds become slur-
      red, kind of Daffy-duck like (sufferin' succotash), due to fold-
      ing-in of ficticious data from frequencies above S1.
}
      "S1" book "S2" book
      (hA2) its rms into RMS

\     Decimation:
      (hA) one S2 slash those rows uniform (hA ht)
      one S1 slash those rows S1 S2 slash star uniform (ht1)
      (hA1 ht ht1) lerp1 (hA1) \ resampling at dt1=1/S1

\     Scaling to force low pass rms to equal original RMS:
      (hA1) RMS that rms (RMS1) slash (hA1 ratio)
      " resample scale factor:" . dup . *f

      "_resample1" naming
   end

   inline: Sfreqs (S N --- hSf) \ N frequencies for sample rate S
\     N is the number of points in the fft from word FFT.
\     Sf ranges from 0 to Nyquist frequency, S/2.

\     Generating a set of N frequencies with uniform step df=S/(2*N):
      integer push (S) two slash peek slash pull uniform
   end

{" This word is written in infix.  Gather its text and run eval.

   function (U, t) = sine_sweep(f1, f2, dt, oct) { 
   /* Returned U(t) is a unit amplitude sine sweep from f1 to f2 (Hz)
      at oct octaves per minute with time step dt. */
      { 
        ln2 = ln(2);
        macro("e^", "exp"); // make exp() that will work for infix
      }

      r = oct/60; // octaves per second

   /* The equation for sine sweep frequency as a function of time
      x is:

         f(x) = f1*2^(r*x)                                   (1)

      The angular value to use in the sine function at specific 
      time t is the integral of f(x) from x=0 to x=t subject to
      the initial condition f(0)=f1:

         Integral(f(x), 0 to t) = (f1*2^(r*t) - f1)/(r*ln2)  (2)

      and the total time t2 to reach final frequency f2, obtained 
      from (1), is: */

         t2 = ln(f2/f1)/(r*ln2); // total time

         t = uniform(dt, t2/dt); // time steps from 0 to t2

         U = sin(2*pi*(f1*exp(r*t*ln2) - f1)/(r*ln2)); // amplitude

   /* Note: exp(r*t*ln2) in the expression above is equivalent to 
      2^(r*t) in (2) and is used to take advantage of postfix word
      e^ and avoid looping.  

      Infix cannot parse the function name e^ since ^ is taken as a 
      separate power operator.  Macro exp() created above gets around 
      this by providing a substitute. */
   }

"} eval \ parse infix into an inline and add to library

   inline: three_octave (hA hF f0 f1 -- hR hF1) \ one-third octave bands
\     Version of 3octave for the parser.
      3octave ;

   inline: tshift (hS ht dt --- hS1) \ time shift signal S
\     Positive dt shifts S to higher time, i.e., events happen later.

\     See man tshift for an example.

      "dt" book, this rev, dt +d swap park, dt 0>
      IF this 1st reach dt negate that 1st goose swap
      ELSE this one endmost dt negate that 1st goose
      THEN pile swap lerp
   end

   inline: tuning (hRef hSig ht --- hSig1) \ tuning Sig to match Ref
{     Tuning Sig(t) to minimize rms(Ref-Sig) by changing phase, not
      frequency.

      See man tuning for an example.

      This is done by brute force: taking the best of a number of 
      tries using a Newton-Raphson-like approach on a nonmonotonic 
      function.

      The matrix of dRMS and tau pairs is stored here as A in case
      there is a need to extract them.

      Inline func() computes the change in rms, dRMS.
}     [ "(dt --- func) S t rot (dt) tshift, R less rms ontop RMS -"
        "func()" inlinex
        20 makes tries
      ]
      "t" book, "S" book, "R" book
      t 2nd pry "dt" book, dt "tau" book
      0 is RMS

      tries two null, tries 1st
      DO
         tau func() into dRMS           \ f(tau)
         tau dt + func()                \ f(tau+dt)
         dRMS less dt slash makes slope \ [f(tau+dt) - f(dt)]/dt

         dRMS tau park I them ram

         tau dRMS slope slash
         less makes tau \ tau = tau - f(tau)/slope
      LOOP
      S t rot yes sort, dup "A" book, 1st 2nd fetch tshift
      freed is S, freed is R, freed is t
   end

   inline: wfft (hX ht t1 N --- hFr hFi) \ windowed N-point FFT at t1
{     Computes fft of X(t) for window of N points starting at time t1.

      Turn on Hanning filter by: yes "wfft" "HAN" bank

      Notes: 

         For no Hanning filtering of X, t1=0, and S=t(2)-t(1), this 
         word used as:
            X t 0 N wfft
         produces the same FFT as word fft1 used as: 
            X this trend detrend S N fft1

         Here is a specific example for time history A:
            "load_demo" missing IF "mat.v" source THEN
            load_demo 1st catch into A

            "wfft" missing IF "signal.v" source THEN

            A 1st those rows items plot, pause plotclose
            
            "wfft" "HAN" yank push \ saving current value

            no "wfft" "HAN" bank
            A 1st those rows items 0 them rows wfft cmag (hWFFT)
            A this trend detrend 1 those rows FFT cmag (hFFT)

          \ This comparison should leave -1 (yes) on the stack,
          \ meaning the results are identical:

            (hWFFT hFFT) 100 (%) 0 compareRef null? (yes)

            pull "wfft" "HAN" bank \ putting back saved value
}
      [ "(ht) this 2nd pry (t2), swap 1st pry (t1), less (dt)"
        "dt" inlinex
        no is HAN \ initial setting is to not do cosine tapering
      ]
      xbase push 0based

      (N) "N" book \ FFT points

      one other (1 ht) dt slash "S" book \ sample rate is 1/dt

      that rev (ht t1) bsearch drop into rindex

   \  Using less than N if overflow past end (here 0-based indexing
   \  is handy):
      rindex N items, those (ht) rows (r), those (items) dims fill
      over > rake lop (hList)

      (hX ht hList) HAN
      IF dup push (hList) reach (ht1) hanning (hW) 
         (hX hW) those cols clone (hW1)
         swap (hW1 hX) pull (hList) reach (hX0) *by (hX1)
      ELSE lop reach (hX1)
      THEN

      (hX1) this trend detrend S N FFT

      (hFr hFi) HAN
      IF "hanning" "HanningAdjust" extract (fac) dup rev
         (hFr fac hFi fac) *f rev *f
      ELSE swap
      THEN
      (hFi hFr) "_Fr" naming swap "_Fi" naming

      pull indexbase
   end

   inline: wPSD (hP S Nf --- hPSD hF) \ PSD using adjacent windows
{     PSD of P at sample rate S, using adjacent windows of Nf points.

      This word carries out the steps outlined in Reference 1, page 432:
      Recommended Computational Steps for Ensemble-Averaged Estimates.

      The returned PSD is the average of the summed PSDs for all adja-
      cent windows.

      The fft of each window is computed using cosine-squared tapering
      of the end-points; see words hanning and wfft.

      Hanning tapering can be turned off by banking HAN to no in word
      wfft, as in: no "wfft" "HAN" bank.

      The width of each window is Tw=Nf/S seconds.

      The number of windows is int(T/Tw), where T is the period of
      samples P.

      The uniform windows are centered within period T, and data from
      each end of P that falls outside them is not used.
}
      "wfft" exists? not IF " require word wfft" ersys return THEN

      integer "Nf" book, "S" book, "P" book

    \ Making uniform time points:
         one S slash, P rows uniform makes t \ vector of sample times
         t rows, t 2nd pry star makes T \ period of samples

    \ Making FFT windows.
       \ Width of each window is Nf/S seconds:
         Nf S / is Tw

       \ and the number of windows is int(T/Tw):
         T Tw /mod into nWIND drop \ number of FFT windows

         nWIND 0=
         IF " not enough sample points for specified Nf points" ersys
            freed is P, freed is t, return
         THEN
         Tw nWIND uniform (hWIND) \ nWIND windows of Tw size

       \ Centering the adjacent windows in the sample period:
         T Tw nWIND star less (rem) two slash (t0)
         (hWIND t0) +d (hWIND) "WIND" book \ start times for windows

   \  Averaging PSD=FFT*conj(FFT) over all windows:
         nWIND 1st
         DO P, t, WIND I pry, Nf wfft (hHr hHi) *conj 
            I 1st > IF plus THEN
         LOOP (hPSD)

       \ Freeing space:
         purged "P" purged "t" purged "WIND" book book book

       \ Scaling the PSD:
         (hPSD) two Tw nWIND star slash (scale) \ 2/(nWIND*Tw)
         (scale) *f "_PSD" naming (hPSD)

    \ Frequencies of the PSD:
         (hPSD) S those rows Sfreqs "_F" naming
   end

\-----------------------------------------------------------------------

\  Words using functions from the Matlab signal toolbox.

   using_matlab
   IF
   "engOn" missing IF matlab THEN

   inline: mbandpass ( --- hB hA) \ make bandpass filters
{     Using a Matlab filter function.

      A bandpass filter setup word must be run first, like
      ellip, cheby1, cheby2.
 
      Examples:
         3 0.1  40   15 5 100 ellip  mbandpass
         3 0.1  UDEF 15 5 100 cheby1 mbandpass
         3 UDEF 30   15 5 100 cheby1 mbandpass

      Nth order bandpass digital elliptic filter:
         Rp - decibels of ripple in the pass band
         Rs - decibels down stop band
         Fc - center frequency
         dF - width of pass region
         S - sample rate

      Frequency range is normalized by S/2 (Nyquist):
         f1, f2 - pass band region, f1=Fc-dF/2 < f < f2=Fc+dF/2
         WN = [w1=f1*2/S, w2=f2*2/S]

      Size of returned A and B is 2N+1.
}
      [ UDEF is Type \ of Matlab filter function:
        "[B,A]=ellip(N,RP,RS,WN)"         \ Type 1
        "[B,A]=cheby1(N,RP,WN)" pile      \ Type 2
        "[B,A]=cheby2(N,RS,WN)" pile (hT) \ Type 3

        (hT) chop into FILTERS
      ]
      Type UDEF = IF " filter type not defined" . nl return THEN

      Fc dF 2 / -, Fc dF 2 / + (f1 f2) park (hF)
      (hF) 2 S / (2/S) *f (hW) 
      (hWN) "WN" naming engPut 
      RS hand "RS" naming engPut 
      RP hand "RP" naming engPut 
      N  hand "N"  naming engPut

      Type one three within
      IF FILTERS Type ndx quote strchop engRun
      ELSE " mbandpass: invalid type " . Type .u nl
      THEN
      "B" engGet "A" engGet
   end

   inline: mfft (hX S N --- hFr hFi) \ N-point Fourier transform of X
\     N-point Fourier transform of data X taken at sample rate S.

\     N is forced to be even, and returned arrays have N/2 rows.

\     Pads with zeroes if length of X < N; truncates if X length > N.

\     Uses word fft_scale in local library of word fft1.

\     Using Matlab fft function.
      engOn? not IF " Matlab engine is not on" . nl return THEN

      tic two slash integer two star "N" book \ force N to be even
      one swap (1 S) slash "dt" book

      these rows N <> 
      IF 1st catch these rows N < 
         IF N those rows less those cols null pile \ padding zeroes
         ELSE 1st N items reach \ truncating elements beyond N
         THEN
      THEN

      N hand "N" naming engPut 

      (hX) this push cols 1st
      DO peek (hX) I catch
       \ Running Matlab:
            (hX) "X" naming engPut 
            "F=fft(X,N)" engRun 

       \ Results from Matlab:
            "F" engGet (hFr hFi)
      LOOP (hFr1 hFi1 hFr2 hFi2 ...)

      pull cols two * parkn 
      these cols two spikes 0= claw (hFr hFi)

    \ Scale by dt and pack to left plane
      (hFr hFi) dt "fft1" "fft_scale" localrun
   end

   inline: mfilter (hB hA hX --- hY) \ filter X with filter [A,B]
\     Using Matlab filter function.
      "X" naming engPut 
      "A" naming engPut 
      "B" naming engPut
      "Y=filter(B,A,X)" engRun 
      "Y" engGet
   end

   inline: mfreqz (hB hA N --- hHr hHi hW) \ Z-transform freq response
{     Using Matlab freqz function.
      Returned W holds N frequencies from 0 to pi*(N-1)/N.

      Examples:
         3 0.1 40 15 5 100 mbandpass (hB hA) 512 mfreqz
 
         3 0.001 20 433.8 0.25 2000 cheby2 mbandpass (hB hA) 8192 mfreqz
         "mbandpass" "S" yank 2 / pi / *f rev
         (hHr hHi) cmag swap (hHmag hW) myYsemilog mgrid
}
      hand "N" naming engPut
      "A" naming engPut
      "B" naming engPut
      "[H,W]=freqz(B,A,N)" engRun
      "H" engGet "W" engGet
   end

   inline: mifft (hFr hFi S Nt --- hY) \ inverse of FFT from mfft
{     Restores the right half removed, and scaling done, by word mfft.
      Incoming S is sample rate; Nt points in Y are created.

      NOTE: Padding of zeroes may be necessary to create Nt points.
            To put zeroes at the beginning of Y, input negative Nt; 
            otherwise, padded zeroes are at the end of Y.
}
      [ 1E-6 is eps ]

      this abs "Nt" book, (Nt) 0< "0front" book

      1 swap slash (1/S)
      (hFr hFi sfac) "fft1" "fft_unscale" localrun (hFr hFi)
      (hFr hFi) "Fi" book "Fr" book

      Fr cols 1st
      DO Fr I catch Fi I catch (hFr hFi)
       \ Send matrices to Matlab:
            "Fi" naming engPut
            "Fr" naming engPut

       \ Run Matlab:
            "F=Fr+i*Fi" engRun
            "Y=ifft(F)" engRun

       \ Fetch complex matrix from Matlab:
            "Y" engGet (hYr hYi)
      LOOP (hYr1 hYi1 hYr2 hYi2 ...)

      Fr cols two * 

      purged "Fi" book
      purged "Fr" book

      (hYr1 hYi1 hYr2 hYi2 ... hYrN hYiN N) parkn (hY)

      these cols two spikes 0= claw (hYr hYi)

      (hYr hYi) these rows Nt <> 
      IF these rows Nt <
         IF Nt those rows less, those cols null (hNull)
            dup three roll 0front not
            IF swap (hYr hNull) THEN pile rev
            (hYi hNull) 0front IF swap THEN pile
         ELSE swap 1st Nt items reach, swap 1st Nt items reach
         THEN
      THEN

    \ The complex part should be negligible for a real signal: 
      (hYr hYi) these abs totals ontop, those rows slash, eps >
      IF " mifft: warning: FFT inverse is complex" . nl
      ELSE (hYr hYi) drop (hYr) "_Y" naming
      THEN
   end

   define: mlowPass (hy S fHz --- hY ht) 
\     Remove frequencies that are above fHz, from data y sampled at
\     rate S.

\     Returns filtered Y and a corresponding vector of uniform times. 

\     Using Matlab to make a Chebyshev lowpass filter, and Matlab to
\     do the filtering.

      [ \ Making local words CHEBY1 and FILTER that run Matlab:

      {" CHEBY1 (N R Wn --- hA hB) \ build a filter in Matlab
         { Running this function in Matlab:
             [B,A] = cheby1(N,R,Wn) designs an Nth order lowpass 
             digital Chebyshev filter with R decibals of ripple
             in the bandpass.
           See Matlab: help cheby1.
         }
         (N R Wn) three (items on stack) listn "C" naming engPut
         "[B,A]=cheby1(C(1),C(2),C(3))" engRun
         "A" engGet (hA)
         "B" engGet (hB)
      "} "CHEBY1" inlinex

      {" FILTER (hX hA hB --- hX1) \ filter data X in Matlab
         \ Running this function in Matlab, where A and B are filter
         \ coefficients from a Matlab function like cheby1():
         \    X1=filter(B,A,X)
         \ See Matlab: help filter.

         (hB) "B" naming engPut
         (hA) "A" naming engPut
         (hX) "X" naming engPut
         "X1=filter(B,A,X)" engRun
         "X1" engGet (hX1)
      "} "FILTER" inlinex
      ]
      (S fHz) "fHz" book "S" book (hy)

    \ Make Chebyshev lowpass filter in Matlab (see Matlab help cheby1):
         3             \ Nth order filter
         0.1           \ R db ripple in bandpass
         fHz S / 1 min \ 0<Wn<1 (1 corresponds to S/2)
         (N R Wn) CHEBY1 (hA hB)

    \ Filter y in Matlab:
         (hy hA hB) FILTER (hY) "_Y" naming

    \ Make a uniform time vector:
         1 S / those rows uniform (ht) "_t" naming

      (hY ht)
   end

   inline: mprony (hH nB nA --- hB hA) \ prony filter
\     Using Matlab prony function.
\     Finds a filter with numerator order nB, denominator order nA, and
\     having the impulse response in vector H.

      hand "nA" naming engPut
      hand "nB" naming engPut
      hand "H"  naming engPut
      "[B,A]=prony(H,nB,nA)" engRun
      "B" engGet "A" engGet
   end

   inline: mresample (hA2 S2 S1 --- hA1) \ data A2 at S2 into A1 at S1
{     Data A2 at sample rate S2 is resampled at rate S1.  

      Each column of A2 is a history of data for one channel.

      A2 is first low pass filtered, using a Chebyshev lowpass filter,
      to Nyquist frequency S1/2, and then resampled at S1 to produce A1.

      Maintaining RMS: resampled A1 is scaled to have the RMS of A2;
      the scale factor is reported and should be close to 1 unless
      significant response has been removed by lowpass filtering.
}
      "S1" book "S2" book

\     Save original RMS for later scaling:
      (hA2) its rms into RMS

\     Low pass filtering to S1/2 (Reference 1, page 398):
      (hA2) S2 S1 two slash mlowPass (hA1 ht) \ Chebyshev lowpass filter

\     Decimation:
      one S1 slash those rows S1 S2 slash star uniform (ht1)
      (hA1 ht ht1) lerp1 (hA1) \ resampling at dt1=1/S1

\     Scaling columns to force resampled rms to equal original RMS:
      these cols 1 =
      IF (hA1) RMS that rms (RMS1) slash (hA1 ratio) 
         " mresample: scale factor:" . dup . *f

      ELSE (hA1) RMS that rms (RMS RMS1) /by bend (hA1 hRatio) 
         " mresample: column scale factors: " . 
         (hRatio) dup "%6.2f" those cols cats format .
         (hA1 hRatio) those rows repeat *by

      THEN nl

      (hA1) "_mresample" naming
   end

   inline: showFilter (Fc dF Nf--- ) \ bandpass filter for Fc, dF
\     Parameters in word mbandpass must already be set up, using a 
\     word like ellip, cheby1, or cheby2; see examples in mbandpass.

      "Nf" book
      this "mbandpass" "dF" bank, "dF" book  
      this "mbandpass" "Fc" bank, "Fc" book
      list: Fc dup .05 star less,
            Fc dup .05 star plus, 0.001 1.5
      end into Faxis \ frequency axis plot scale

     \ Computing frequency response:
      mbandpass (hB hA) Nf mfreqz (hHr hHi hW)
      (hW) "mbandpass" "S" yank (samp_freq) 2 / pi / *f (hF) rev

     \ Plotting amplitude:
      (hHr hHi) 2dup cmag 3 pick (hHmag hF) myYsemilog mgrid
      "Frequency Response Amplitude" "Frequency, Hz" "Amplitude" mlabel
      Faxis maxis_set

     \ Plotting phase:
      swap (hF hHi hHr) atan2 deg/rad *f swap (hPhase hF)
      myYsemilog mgrid
      "Frequency Response Phase" "Frequency, Hz" "Phase" mlabel
   end

   THEN

   pull catmsg
   private halt

\-----------------------------------------------------------------------

;  Appendix

\  Note: stack setup for some words may have changed.

\  Matlab example: filtdem.m.

   "cmag" missing IF mmath.v source THEN

\  Making a signal:
      100 is Fs \ sample rate
      1 Fs / Fs uniform is t \ starts at 0; Matlabs starts at 0.01
\     X=sin(2*pi*t*5) + sin(2*pi*t*15) + sin(2*pi*t*30)
      t 2 pi * *f push
      peek 05 *f sin
      peek 15 *f sin
      peek 30 *f sin + +
      into X
      X t mplot mgrid

\  Making a band pass filter for 10-20 Hz:
      8 0.1 40 10 20 Fs mbandpass into A, into B

\  Showing the properties of the filter:
      B A 512 mfreqz (hHr hHi hW) Fs 2 / pi / *f rev
      (hHr hHi) cmag swap (hHmag hW) mplot mgrid

\  Filtering the signal:
      B A X mfilter into X1
      X1 t mplot mgrid

\  Comparing FFTs of original and filtered:
      X 512 mfft cmag (Gx)
      X1 512 mfft cmag (Gx Gx1) park
    \ Taking the left half result:
      1st those rows items reach
      Fs those rows slash those rows uniform mplot mgrid

\  Prony method:
      X 6 6 mprony into A, into B
      B A 512 mfreqz (hHr hHi hW) Fs 2 / pi / *f rev
      (hHr hHi) cmag swap (hHmag hW) mplot mgrid

      X, B A X mfilter (Y), park t mplot mgrid

\-----------------------------------------------------------------------

;  Obsolete or replaced by native words in C:

  _inline: cross-correlation (hX hY d --- hR) \ between X(k), Y(k-d)
{     Cross-correlation, also called covariance, of X and Y, with delay
      d applied to Y.

      Each row in X and Y is a data record, and each column is a step
      in time.  Columns of Y are delayed for d steps and the cross-
      correlation is computed (Ref: Bendat and Piersol, p. 120):
         Rxy(k) = [X'(k) * Y(k-d)]
}
      "d" book
      (hY) these cols "N" book
      swap (hX) N d less endcols         \ endmost d columns of X
      swap (hY) 1st N d less items catch \ d delayed columns of Y
      *by totals
      d one null swap pile
   end

  _inline: detrend (hA hB0 hB1 --- hA1) \ remove linear trend B from A
{     Removing linear trend B from each column of N uniformly spaced
      points in A:
         A1(k,j) = A(k,j) - [B0(j) + k*B1(j)]
      where k=1,N.

      Word trend provides the matrices B0 and B1.

      Example:
         0.001 1000 uniform, pi 100 / 1000 uniform sin plus
         dup dup trend detrend park, 1st those rows items plot
}
      push those rows, repeat less
      once those rows items, those cols clone
      pull those rows, repeat *by less
   end

  _inline: trend (hA --- hB0 hB1) \ linear trend for uniformly spaced A
{     Each column in A is a data record of N uniformly spaced points.

      For A(k,j), k=1,N, the one-row trend is:
         trend(k,j) = B0(j) + k*B1(j)

      Size of matrices B0 and B1 is one row by the number of columns
      in A.

      Reference 1, page 397, equation 11.16.
}
      these rows into N
      these totals bend into sum(Ai)

      once those rows items, those cols clone *by
      totals bend into sum(i*Ai)

      sum(Ai) N dup + tic dup + *f \ times 2*(2N+1)
      sum(i*Ai) 6 *f less          \ less 6*sum(i*Ai)
      N dup nit * /f (hB0)         \ over [N*(N-1)]
      "_B0" naming

      sum(i*Ai) 12 *f              \ times 12
      sum(Ai) N tic 6 * *f less    \ less 6*(N+1)*sum(Ai)
      N N nit N tic * * /f (hB1)   \ over [N*(N-1)*(N+1)]
      "_B1" naming

      freed is sum(Ai), freed is sum(i*Ai)
   end
