/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2011  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#ifdef NET
#ifdef OPENSSL

/* ssl.c  August 2005

Copyright (c) 2005  D. R. Williamson

References:

   1. OpenSSL statement: Implements the Secure Sockets Layer 
      (SSL v2/v3) and Transport Layer Security (TLS v1) protocols, 
      and provides a general purpose cryptography library
      http://www.openssl.org/

   2. OpenSSL source: Jul  5 21:19:24 2005 openssl-0.9.8.tar.gz
      http://www.openssl.org/source/

   3. Rescorla, E., "SSL and TLS: Designing and Building Secure 
      Systems," Addison-Wesley, 2001, ISBN 0-201-61598-3
      http://www.rtfm.com/openssl-examples/

   4. Viega, J., Matt Messier, and Pravir Chandra, "Network Security
      with OpenSSL," O'Reilly Media, Inc., 2002.

Notes:

   Ways of storing session time data:
      Capturing session start up:
         timesys();
         session=SSL_get_session(ssl);
         SSL_SESSION_set_time(session,(long)tos->real);
         drop();
      (But connection time, within clientsockets data gives about
      the same info: clitim[FD_SETSIZE]; // times of connection //)

      Setting session timeout (default is 304 seconds):
         session=SSL_get_session(ssl);
         SSL_SESSION_set_timeout(session,(long)86400);

   Timings (microseconds) for the following SSL functions on machine 
   loopback, Netscape to server (Pentium II 350 MHz):
      BIO_new_socket: 30
      SSL_new: 144
      SSL_set_bio: 8
      SSL_accept_timeo: 195681 [handshaking new session]
      SSL_accept_timeo: 6142 [reuse session]

----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>

#include "main.h"
#include "stk.h"

#include "exe.h"
#include "inpo.h"
#include "mem.h" /* for memprobe() */
#include "net.h"
#include "ssl.h"
#include "sys.h"
#include "tex.h"

extern int NTRACE; /* term.h */
extern int WTRACE; /* term.h */

#undef ALLOW_OLD_VERSIONS
#ifndef ALLOW_OLD_VERSIONS
   #if (OPENSSL_VERSION_NUMBER < 0x00905100L)
      #error "Must use OpenSSL 0.9.6 or later"
   #endif
#endif

/* Certification files: */
#define CA_LIST "root.pem"
#define CLIENT_KEYFILE "client.pem"
#define DHFILE "dh1024.pem"
#define SERVER_KEYFILE "server.pem"

#define PASSWORD "password"

/* Local headers. */
SSL_CTX *initialize_ctx(char *keyfile, char *password, char *cafile);
int load_dh_params(SSL_CTX *ctx,char *file);
static char *passwd_save;
static int pem_passwd_cb(char *buf, int size, int rwflag, void *passwd);

/*--------------------------------------------------------------------*/

int client_SSL()
/* Set up for word CONNECT_SSL to connect an SSL client to an SSL
   server. 

   Called at start up by terminit(), so certification files must be 
   defined even if CONNECT_SSL will not be used. 

   Description of .pem files for client:
      Files root.pem and client.pem from Reference 3 are contained in
      program directory tops/usr and are typical of the types of files
      used by an SSL client.

      Following Reference 3, function initialize_ctx() used below for
      the client expects its CA certificate in CA_LIST, defined above
      to be file root.pem.  The client key and certificate must be in 
      CLIENT_KEYFILE, defined above to be client.pem. 

      The path to these files in the program is called usrpath. */
{
   char *cafile,*keyfile;

   gprintf(" client_SSL: create context and check certificates");
   nc();

/* Build SSL context (following Reference 3). */
   pushstr("usrpath");
   xmain(0);
   pushstr(CLIENT_KEYFILE);
   cat();
   keyfile=tos->tex;

   pushstr("usrpath");
   xmain(0);
   pushstr(CA_LIST);
   cat();
   cafile=tos->tex;

   client_ctx=NULL;
   client_ctx=initialize_ctx(keyfile,PASSWORD,cafile);
   if(!client_ctx) {
      gprintf(" client_SSL: error initializing SSL context");
      nc();
      stkerr("","");
      return 0;
   }
   drop2(); /* keyfile and cafile names off stack */
   return 1;
}

int pem_passwd_cb(char *buf, int size, int rwflag, void *passwd)
/* Receive password to be used during decryption. */
{
   if(passwd) {
   /* Example from man SSL_CTX_set_default_passwd_cb: */
      strncpy(buf,(char *)passwd,size);
      buf[size - 1] = '\0';
      return(strlen(buf));
   }
   else {
      gprintf(" pem_passwd_cb: failed to set decryption password");
      nc();
      stkerr("","");
      return 0;
   }
}

int server_SSL()
/* Set up for word SERVER_SSL to start an SSL server.

   Called at start up by terminit(), so certification files must be 
   defined even if SERVER_SSL will not be used.

   Description of .pem files for server:
      Files root.pem and server.pem from Reference 3 are contained in
      program directory tops/usr and are typical of the types of files
      used by an SSL server.

      Following Reference 3, function initialize_ctx() used below for
      the server expects its CA certificate in CA_LIST, defined above
      to be file root.pem.  The server key and certificate must be in 
      SERVER_KEYFILE, defined above to be server.pem. 

      For the server setting the context within which the session can 
      be reused, DHFILE--defined above to be dh1024.pem--is also given
      in program directory tops/usr.

      The path to these files in the program is called usrpath. */
{
   char *cafile,*dhfile,*keyfile;
   static int s_server_session_id_context=1;

   gprintf(" server_SSL: create context and check certificates");
   nc();

/* Build SSL context (following Reference 3). */
   pushstr("usrpath");
   xmain(0);
   pushstr(SERVER_KEYFILE);
   cat();
   keyfile=tos->tex;

   pushstr("usrpath");
   xmain(0);
   pushstr(CA_LIST);
   cat();
   cafile=tos->tex;

   server_ctx=initialize_ctx(keyfile,PASSWORD,cafile);
   if(!server_ctx) {
      gprintf(" server_SSL: error initializing SSL context");
      nc();
      stkerr("","");
      return 0;
   }
   drop2(); /* keyfile and cafile names off stack */

   pushstr("usrpath");
   xmain(0);
   pushstr(DHFILE);
   cat();

   dhfile=tos->tex;
   if(!load_dh_params(server_ctx,dhfile)) {
      gprintf(" server_SSL: error loading dh params");
      nc();
      stkerr("","");
      return 0;
   }
   gprintf(" Server DH parameters file ok: %s",dhfile);
   nc();
   drop(); /* dhfile name off stack */

   if(!SSL_CTX_set_session_id_context(server_ctx,
      (void*)&s_server_session_id_context,
      sizeof s_server_session_id_context)) {
      gprintf(" server_SSL: error in SSL_CTX_set_session_id_context");
      nc();
      stkerr("","");
      return 0;
   }
   SSL_CTX_set_verify(server_ctx,SSL_VERIFY_NONE,NULL);
   return 1;
}

static void SSL_accept_alarm(int signo)
{
   gprintf(" SSL_accept_alarm: signum %d  %s",signo,datetime());
   nc(); 
   return; /* just interrupt the function */
}

int SSL_accept_timeo(SSL *ssl, int sec)
/* This function just implements an alarm.  Caller must still check for 
   ssl_ret<=0 and call SSL_get_error() */
{
   Sigfunc *sigfunc;
   int ssl_ret=0;

   sigfunc=signal(SIGALRM,SSL_accept_alarm);
   if(alarm(sec)!=0) {
      gprintf(" SSL_accept_timeo: alarm was already set");
      nc();
   }
   ssl_ret=SSL_accept(ssl);

   alarm(0);                /* turn off the alarm */
   signal(SIGALRM,sigfunc); /* restore previous signal handler */

   if(ssl_ret<=0) {
      if(errno==EINTR) {
         errno=ETIMEDOUT;
         gprintf(" SSL_accept_timeo: %d sec alarm timeout",sec);
         nc();
      }
   }
   return(ssl_ret);
}

static void SSL_connect_alarm(int signo)
{
   gprintf(" SSL_connect_alarm: signum %d  %s",signo,datetime());
   nc(); 
   return; /* just interrupt the function */
}

int SSL_connect_timeo(SSL *ssl, int sec)
/* This function just implements an alarm.  Caller must still check for
   ssl_ret<=0 and call SSL_get_error() */
{
   Sigfunc *sigfunc;
   int ssl_ret=0;

   sigfunc=signal(SIGALRM,SSL_connect_alarm);
   if(alarm(sec)!=0) {
      gprintf(" SSL_connect_timeo: alarm was already set");
      nc();
   }
/* memprobe(); gprintf("SSL_connect_timeo call SSL_connect\n");*/

   ssl_ret=SSL_connect(ssl);

/* memprobe(); gprintf("SSL_connect_timeo after SSL_connect\n");*/

   alarm(0);                /* turn off the alarm */
   signal(SIGALRM,sigfunc); /* restore previous signal handler */

   if(ssl_ret<=0) {
      if(errno==EINTR) {
         errno=ETIMEDOUT;
         gprintf(" SSL_connect_timeo: %d sec alarm timeout",sec);
         nc();
      }
   }
   return(ssl_ret);
}

int SSL_ERROR_code() /* SSL_ERROR_code (n --- qS) */ 
/* Sun Apr 10 11:30:13 PDT 2011
   For SSL_ERROR code number n, return its string. 

   These are SSL_ERROR numbers from openssl-0.9.8/ssl/ssl.h:

     0 SSL_ERROR_NONE
     1 SSL_ERROR_SSL
     2 SSL_ERROR_WANT_READ
     3 SSL_ERROR_WANT_WRITE
     4 SSL_ERROR_WANT_X509_LOOKUP
     5 SSL_ERROR_SYSCALL: look at error stack/return value/errno
        Values for errno (for errors in read() and write(), among 
        others) are found in these places:
           GNU/Linux: /usr/include/asm/errno.h
           AIX: /usr/include/errno.h
     6 SSL_ERROR_ZERO_RETURN
     7 SSL_ERROR_WANT_CONNECT
     8 SSL_ERROR_WANT_ACCEPT */
{
   int n;

   if(!popint(&n)) return 0;

   switch(n) {

      case SSL_ERROR_NONE: /* 0 */
         pushstr("SSL_ERROR_NONE"); 
         break;

      case SSL_ERROR_SSL: /* 1 */
         pushstr("SSL_ERROR_SSL"); 
         break;

      case SSL_ERROR_WANT_READ: /* 2 */
         pushstr("SSL_ERROR_WANT_READ"); 
         break;

      case SSL_ERROR_WANT_WRITE: /* 3 */
         pushstr("SSL_ERROR_WANT_WRITE"); 
         break;

      case SSL_ERROR_WANT_X509_LOOKUP: /* 4 */
         pushstr("SSL_ERROR_WANT_X509_LOOKUP"); 
         break;

      case SSL_ERROR_SYSCALL: /* 5 */
         pushstr("SSL_ERROR_SYSCALL; see value of errno"); 
         break;

      case SSL_ERROR_ZERO_RETURN: /* 6 */
         pushstr("SSL_ERROR_ZERO_RETURN"); 
         break;

      case SSL_ERROR_WANT_CONNECT: /* 7 */
         pushstr("SSL_ERROR_WANT_CONNECT"); 
         break;

      case SSL_ERROR_WANT_ACCEPT: /* 8 */
         pushstr("SSL_ERROR_WANT_ACCEPT"); 
         break;

      default:
         pushint(n);
         intstr();
         pushstr(" not found in SSL_ERROR codes"); 
         cat();
         break;
   }
   return 1;
}

int SSL_select(int cn) 
/* Return 1 if socket for client number cn is selectable and there 
   are bytes pending in the SSL buffer. */
{
   if((*(SELECT+cn)>0 &&         /* client socket is selectable */
      *(clientSSL_PENDING+cn)>0) /* SSL bytes are pending */
   ) return 1;
   else return 0;
}

int SSLconnto() /* SSLconnto (sec --- ) */
/* Set the timeout seconds for the handshake between connecting client
   and SERVER_SSL.  Initial value is 10 seconds, and minimum value is
   one second. */
{
   int sec;

   if(!popint(&sec)) return 0;

   SSL_CONNTO=MAX(1,sec);
   return 1;
}

/*----------------------------------------------------------------------

Selected functions from Reference 3, modified slightly for this program,
are in this section. 

OpenSSL Example Programs 20020110    
by Eric Rescorla
January 10, 2002 Edition 
Copyright (C) 2001 RTFM, Inc.       
http://www.rtfm.com/openssl-examples/ 

LICENSE
Copyright (C) 2000-2001 RTFM, Inc.
All Rights Reserved

This package is a series of demonstration programs written by
Eric Rescorla <ekr@rtfm.com> and licensed by RTFM, Inc.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. Neither the name of RTFM, Inc. nor the name of Eric Rescorla may be
   used to endorse or promote products derived from this
   software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE ERIC RESCORLA AND RTFM ``AS IS'' AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
THE POSSIBILITY OF SUCH DAMAGE.
*/

BIO *bio_err=0;
static void sigpipe_handle(int x);

SSL_CTX *initialize_ctx(char *keyfile, char *password, char *cafile)
  {
    SSL_METHOD *meth;
    SSL_CTX *ctx;
   
    if(!bio_err){
      /* Global system initialization*/
      SSL_library_init();
      SSL_load_error_strings();

      /* An error write context */
      bio_err=BIO_new_fp(stderr,BIO_NOCLOSE);
    }

    /* Set up a SIGPIPE handler */
    signal(SIGPIPE,sigpipe_handle);
   
    /* Create our context*/
    meth=SSLv23_method();
    ctx=SSL_CTX_new(meth);

    /* Load our keys and certificates*/
    if(!(SSL_CTX_use_certificate_chain_file(ctx,keyfile))) {
       gprintf(" initialize_ctx: cannot read certificate file %s",
          keyfile);
       nc();
       stkerr("","");
       return 0;
    }
    gprintf(" Certificate file ok: %s",keyfile);
    nc();

    passwd_save=password;
    SSL_CTX_set_default_passwd_cb(ctx,pem_passwd_cb);
    SSL_CTX_set_default_passwd_cb_userdata(ctx, password);

    if(!(SSL_CTX_use_PrivateKey_file(ctx,keyfile,SSL_FILETYPE_PEM))) {
       gprintf(" initialize_ctx: cannot read private key file %s",
          keyfile);
       nc();
       stkerr("","");
       return 0;
    }
    gprintf(" Private key file ok: %s",keyfile);
    nc();

    /* Load the CAs we trust*/
    if(!(SSL_CTX_load_verify_locations(ctx,cafile,0))) {
       gprintf(" initialize_ctx: cannot read CA file %s",cafile);
       nc();
       stkerr("","");
       return 0;
    }
    gprintf(" Certificate authority file ok: %s",cafile);
    nc();

#if (OPENSSL_VERSION_NUMBER < 0x00905100L)
    SSL_CTX_set_verify_depth(ctx,1);
#endif
    return ctx;
  }

int load_dh_params(ctx,file)
  SSL_CTX *ctx;
  char *file;
  {
    DH *ret=0;
    BIO *bio;

    if ((bio=BIO_new_file(file,"r")) == NULL) {
       gprintf(" load_dh_params: could not open DH file %s",file);
       nc();
       stkerr("","");
       return 0;
    }
    ret=PEM_read_bio_DHparams(bio,NULL,NULL,NULL);
    BIO_free(bio);

    if(SSL_CTX_set_tmp_dh(ctx,ret)<0) {
       gprintf(" load_dh_params: could not set server DH parameters");
       nc();
       stkerr("","");
       return 0;
    }
    return 1;
  }

static void sigpipe_handle(int x){
}

/*--------------------------------------------------------------------*/

/* end ssl.c functions */
#endif
#endif
