/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* mem.c  April 1999

Copyright (c) 1999  D. R. Williamson
*/
  
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef __USE_BSD
   #define __USE_BSD
#endif
#ifndef __USE_MISC
   #define __USE_MISC
#endif
#include <unistd.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "inpo.h"
#include "lib.h"
#include "mem.h"
#include "tag.h"
#include "tex.h"
#include "sparse.h"

/* General note: for portability across machines, functions that call
   malloc() allot at least one byte (actually, one size-of-whatever), 
   even for a matrix of zero rows or zero columns.  

   This avoids NULL pointers that might be returned by some machines 
   (like the IBM RISC 6000) when zero bytes are requested. */

static unsigned long _Mem=0; /* shared by memprobe() and mprobe1() */

int BITS() /* BITS ( --- n) */
/* Return address size of the machine in bits (32 or 64) */
{ 
   return(pushint(8*sizeof(long)));
}

int blockofblanks() /* blockofblanks (rows chars --- hT) */
{
   int rows,chars;
   char *blocko;

   if(!popint(&chars) || !popint(&rows)) return 0;

   chars=MAX(0,chars);
   rows=MAX(0,rows);

   if((blocko=(char *)memgetc(rows,chars))!=NULL) {
      return(push(VOL,(char *)memgetn("_blockof",8),NOTAG,0,NULL, \
         blocko,rows,chars,NULL));
   }
   stkerr(" blockofblanks: ",MEMNOT); 
   return 0;
}

int cop() /* cop (hA --- hB) */
/* Making a copy of A that has new mallocked pointers.  But if A is
   not in the catalog, and A is not elsewhere on the stack, this
   function just returns since no copy is necessary. */
{
   int cols,rows;

   if(!stkdepth()) {
      stkerr(" cop: ",EMPTYSTK);
      return 0;
   }
   if(*(tos->cntc)==0 && *(tos->cnt)<2) return 1;

   switch(tos->typ) {
   
      case NUM:
         pushint(0);
         tos->real=(tos-1)->real;
         tos->imag=(tos-1)->imag;
         tos->tag=(tos-1)->tag;
         if(is_catptr(tos)) { /* ptr NUMs have their cat item as tok */
            pushq2((tos-1)->tok,strlen((tos-1)->tok));
            naming();
         }
         return(lop());

      case STR:
         return(pushq2(tos->tex,tos->col) && lop());

      case MAT:
         if(!matstk((rows=tos->row),(cols=tos->col),"_cop")) return 0;

         memcpy(tos->mat,(tos-1)->mat,sizeof(double)*rows*cols);
         tos->tag=(tos-1)->tag;
         return(lop());

      case VOL:     
         if(!volstk((rows=tos->row),(cols=tos->col),"_cop")) return 0;

         memcpy(tos->tex,(tos-1)->tex,rows*cols);
         tos->tag=(tos-1)->tag;
         return(lop());

      case PTR:
         /* Type PTR must be in the catalog (see popptr() discus-
            sion in stk.c).  Don't have a new catalog name at this 
            point, so cannot make new copy.  Its not clear why a 
            copy with new pointers would be needed anyway. */
          stkerr(" cop: ",INLINENOT);
          return 0;
   }
   return 1;
}
   
int fill() /* fill (x r c --- hA) */
/* Filling r-by-c matrix with value x. 

   Sun May 22 07:06:31 PDT 2011.  Check for negative rows and cols. */
{
   int rows,cols,k=0;
   double *A,x,xi,xr;

   if(!(popint(&cols) && popint(&rows))) {
      return 0;
   }
   if(rows<0 || cols<0) {
      gprintf(" fill: rows %d, cols %d cannot be negative",rows,cols); 
      nc();
      stkerr("","");
      return 0;
   }
   if(is_complex(tos)) {
      return(
         popdx(&xr,&xi) &&
         pushd(xr) &&
         pushint(rows) &&
         pushint(cols) &&
         pushd(xi) &&
         pushint(rows) &&
         pushint(cols) &&
         fill() &&
         lpush() &&
         fill() &&
	 lpull() &&
         dblcmplx()
      );
   }
   else if(!popd(&x)) return 0;

   if((A=(double *)memget(rows,cols))==NULL) {
      stkerr(" fill: ",MEMNOT); return 0;
   }
   for(;k<rows*cols;k++) *(A+k)=x;    
   return(push(MAT,(char *)memgetn("_fill",5),NOTAG,0,A,NULL, \
      rows,cols,NULL));
}

int getbrk() /* getbrk ( --- addr) */
/* Return the address the end of data space. */
{
   void *end; /* end of addressable data space, the "break." */

   end=sbrk(0);
   if(end==(void *)-1) return(pushint(-1));
   else return(pushuint((unsigned long)end));
}
   
void mallfree(void **ptr)
/* Free the contents of mallocked ptr, then make the contents NULL. */
{
   if(*ptr) {
      free(*ptr);
      *ptr=NULL;
   }
}

int mallinfo1() /* mallinfo ( --- ) */
/* Displaying the mallinfo structure.

SVID2/XPG mallinfo structure
(The following is for 32 bit machines; on 64 bit machines, the large
 ints like arena and uordblks are unsigned long.)
struct mallinfo {
  int arena;    // total space allocated from system 
  int ordblks;  // number of non-inuse chunks
  int smblks;   // unused -- always zero
  int hblks;    // number of mmapped regions
  int hblkhd;   // total space in mmapped regions 
  int usmblks;  // unused -- always zero 
  int fsmblks;  // unused -- always zero
  int uordblks; // total allocated space
  int fordblks; // total non-inuse space
  int keepcost; // top-most, releasable (via malloc_trim) space */
{
   struct mallinfo M;
   double arena,uordblks,fordblks,hblks,hblkhd,ordblks,keepcost;

   M=mallinfo();
/*
   malloc_stats(); // handy Linux function, but writes only to stderr 
   nc();
*/
   arena=M.arena;       gprintf("  arena bytes: %0.0f\n",arena);
   uordblks=M.uordblks; gprintf(" in use bytes: %0.0f\n",uordblks);
   fordblks=M.fordblks; gprintf("  freed bytes: %0.0f\n",fordblks);
   hblks=M.hblks;       gprintf(" mmap regions: %0.0f\n",hblks);
   hblkhd=M.hblkhd;     gprintf("   mmap bytes: %0.0f\n",hblkhd);
   ordblks=M.ordblks;   gprintf("      ordblks: %0.0f\n",ordblks);
   keepcost=M.keepcost; gprintf("     keepcost: %0.0f\n",keepcost);

   return 1;
}

int matstk(int rows, int cols, char *name) /* ( --- hA) */
/* Push uninitialized matrix rows-by-cols to stack.  Matrix A is 
   stkitem type MAT.  

   Incoming rows and cols are used to define the bytes needed for a 
   real matrix of size rows-by-cols matrix.

   If A is to be a complex MAT (stkitem tag==TAG_COMPLEX), then rows 
   should be doubled to account for the real and imaginary parts that 
   will exist in A.
*/
{
   int c1,r1;
   double *A;

   r1=MAX(0,rows);
   c1=MAX(0,cols);

   if((A=malloc((1+r1*c1)*sizeof(double)))==NULL) {
      stkerr(" matstk: ",MEMNOT);
      return 0;
   }
   return(push(MAT,(char *)memgetn(name,strlen(name)),NOTAG,0,A,NULL, \
      r1,c1,NULL));
}

int matstk_idx(int rows, int cols, char *name) /* ( --- hA) */
/* Does same thing as matstk except the matrix will have additional
   memory allocated to contain row and column indices in the bytes
   following the numerical matrix values.

   The matrix will be tagged as having index arrays, but that the
   indices are not being used.  Once the indices are populated,
   call set_index_used() with the stack item if subsequent words 
   are expected to employ these indices.
*/
{
   double *A;

   rows=MAX(0,rows);
   cols=MAX(0,cols);

   if((A=malloc((1+rows*cols)*sizeof(double) + /* for numeric values */
                  (rows+cols)*sizeof(int)      /* for index terms    */
       ))==NULL) {
      stkerr(" matstk: ",MEMNOT);
      return 0;
   }
   return(push(MAT,(char *)memgetn(name,strlen(name)),TAG_INDEXED,0,A,NULL,
      rows,cols,NULL));
}

int memarena() /* memarena ( --- N) */
/* System arena bytes from mallinfo. */
{
   struct mallinfo M;

   M=mallinfo();
   return(
      pushd(M.arena) &&
      pushq2("_memarena",8) &&
      naming()
   );
}

int memfree() /* memfree (memptr --- ) */
/* Frees memory at memptr obtained from word memput. */
{
   stkitem *A;

/* Free stack item A:*/
   if(!(dup1s() && mempull() && drop())) return 0;

   memcpy(&A,&tos->real,sizeof(long));
   mallfree((void *)&A);

   if(*(tos->cntc)==0) return(drop());

/* Store 0 into cataloged memptr word: */
   pushd(0);

   swap();
   named(); /* full catalog name of memptr word, like mp1,0:EIG1 */
   pushstr(":"); 
   chblank();

   dup1s();
   pushint(2+XBASE); /* 3rd word (library name), like EIG1 */
   word();
   drop();
   
   swap();
   pushint(XBASE); /* 1st word (memptr word name), like mp1 */
   word();
   drop();

   implant(); /* running EIG1.mp1=0; i.e., 0 "EIG1" "mp1" bank */

   return 1;
}

int memfreed() /* memfreed ( --- N) */
/* System freed bytes from mallinfo. */
{
   struct mallinfo M;

   M=mallinfo();
   return(
      pushd(M.fordblks) &&
      pushq2("_memfreed",8) &&
      naming()
   );
}

double *memget(int rows, int cols)
/* Uninitialized rows-by-cols matrix of 8-byte elements. */
{
   double *p;

   rows=MAX(0,rows);
   cols=MAX(0,cols);

   if((p=malloc((1+rows*cols)*sizeof(double)))!=NULL) return p;
   stkerr(" memget: ",MEMNOT);
   return p; 
}

double *memget0(int rows, int cols)
/* Matrix of rows-by-cols 8-byte elements, initialized to zero. */
{
   double *p;

   rows=MAX(0,rows);
   cols=MAX(0,cols);

   if((p=calloc(1,(1+rows*cols)*sizeof(double)))!=NULL) {
      return p;
   }
   stkerr(" memget0: ",MEMNOT);
   return NULL;
}

char *memgetc(int rows, int chars)
/* Character matrix rows-by-chars, with null ending byte. */
{  
   char *p;
   int len=0;

   len=MAX(0,rows*chars);

   if((p=malloc(1+len))!=NULL) {
      if(len) memset(p,' ',len); 
      *(p+len)='\0';
      return p;
   } 
   stkerr(" memgetc: ",MEMNOT);
   return p;
}

char *memgetn(char *name, int len)
/* Character row of len with null ending byte, initialized to name. */
{
   char *p;

   len=MAX(0,len);

   if((p=malloc(1+len))!=NULL) {
      if(len) memcpy(p,name,len); 
      *(p+len)='\0';
      return p;
   } 
   stkerr(" memgetn: ",MEMNOT);
   return p;
}

char *memgetn1(char *name, int len, int len1)
/* Character row of len1, containing name of len bytes followed by
   blanks to total length of len1, with null ending byte. */
{
   char *p;

   len=MAX(0,len);
   len1=MAX(0,len1);

   if((p=malloc(1+len1))!=NULL) {
      if(len) memcpy(p,name,MIN(len,len1)); 
      memset(p+len,' ',MAX(0,len1-len));
      *(p+len1)='\0';
      return p;
   } 
   stkerr(" memgetn1: ",MEMNOT);
   return p;
}

char *memgetnNL(char *name, int rows, int chars, int *len)
/* Allocate character matrix rows-by-chars initialized to name and
   with NL at the end of each row, and null at end of everything. */
{
   char *p0;
   register char *p;
   register int k=0;

   *len=MAX(0,rows*(1+chars));

   if((p0=malloc(1+*len))!=NULL) {

      p=p0;
      *(p+*len)='\0';

      for(;k<rows;k++) {
         memcpy(p,name+loclin(k,chars),chars);
         p+=chars;
         *p='\n';
         p++;
      }
      return p0;
   }
   stkerr(" memgetnNL: ",MEMNOT); 
   return p0;
}

int meminuse() /* meminuse ( --- N) */
/* System in-use bytes from mallinfo. */
{
   struct mallinfo M;

   M=mallinfo();
   return(
      pushd(M.uordblks) &&
      pushq2("_meminuse",9) &&
      naming()
   );
}

int mempeek() /* mempeek (memptr --- hA) */
/* Pushes to stack item A saved in a memptr; A is not freed if it
   drops from the stack (but do not lose memptr). */
{
   stkitem *A;

   if(tos->typ!=NUM) {
      stkerr(" mempeek: ",NUMNOT);
      return 0;
   }
   if(tos->real==0) {
      stkerr(" mempeek: ","zero memptr is invalid");
      return 0;
   }
   memcpy(&A,&tos->real,sizeof(long));
   *(A->cnt)=1;

/* Refer to exestkitem() in exe.c for this portion of code: */
   CNT=A->cnt;
   CNTC=A->cntc;
   CTAG=A->tag;
   CIMAG=A->imag;
   NAM=NULL;
   push(A->typ,A->tok,A->tag,A->real,A->mat,A->tex,A->row,A->col,
      A->ptr);
   CNT=NULL;
   CNTC=NULL;
   CTAG=0;
   CIMAG=0;
   NAM=NULL;

   *(A->cnt)-=1;
   return(lop());
}

int memprobe() /* memprobe ( --- ) */
/* Write to stderr the change in M.uordblks since the last call to
   this word. */
{
   struct mallinfo M;
   long memchange;
   double d;

   M=mallinfo();

   memchange=M.uordblks-_Mem;
   _Mem=M.uordblks;
   
   d=memchange;
   gprintf(" %0.0f ",d); 
   return 1;
}

int memprobe1() /* memprobe1 ( --- delta_mem) */
/* Push to the stack the change in M.uordblks since the last call to
   this word. */
{
   struct mallinfo M;
   long memchange;
   double d;

   M=mallinfo();

   memchange=M.uordblks-_Mem;
   _Mem=M.uordblks;

   d=memchange;
   return(pushd(d));
}

int memptrshow() /* memptrshow (memptr --- memptr) */
/* Show properties of stack item at memptr, keeping memptr on the stack.
   Catalog count value (cntc) shows one instance, but the item is not
   in the catalog (the phony catalog count prevents the stack item from 
   being freed when it is dropped from the stack; see memput()). */
{
   return(dup1s() && mempeek() && props());
}

int mempull() /* mempull (memptr --- hA) */
/* Pushes to stack item A saved in a memptr; A is disconnected from
   memptr and will be freed if it drops from the stack without being
   placed into the catalog. */
{
   stkitem *A;

   if(tos->typ!=NUM) {
      stkerr(" mempull: ",NUMNOT);
      return 0;
   }
   if(tos->real==0) {
      stkerr(" mempull: ","zero memptr is invalid");
      return 0;
   }
   memcpy(&A,&tos->real,sizeof(long));
   *(A->cntc)=0;

/* Refer to exestkitem() in exe.c for this portion of code: */
   CNT=A->cnt;
   CNTC=A->cntc;
   CTAG=A->tag;
   CIMAG=A->imag;
   NAM=NULL;
   push(A->typ,A->tok,A->tag,A->real,A->mat,A->tex,A->row,A->col,
      A->ptr);
   CNT=NULL;
   CNTC=NULL;
   CTAG=0;
   CIMAG=0;
   NAM=NULL;

   *(A->cnt)-=1;
   return(lop());
}

int memput() /* memput (hA --- memptr) */
/* Removes item A from the stack and returns a memptr NUM (not a ptr
   NUM) used to return A to the stack.  Item A is not placed into the
   catalog. */
{
   stkitem *A;
   char *tok;

   if(!cop()) return 0;

   if((tok=(char *)memgetn("memptr",6))==NULL) return 0;
   tos->tok=tok; /* giving memptr a tok name */
   *(tos->cntc)=1; /* catalog count so drop() won't deallocate */

   if((A=malloc(sizeof(stkitem)))==NULL) return 0;
   memcpy(A,tos,sizeof(stkitem));

   pushd(0); /* making memptr */
   memcpy(&tos->real,&A,sizeof(long));
   swap();
   tos--; /* dropping A without using drop(), which would deallocate */

   return(pushq2("_memptr",7) && naming());
}

int null() /* null (r c --- hA) */
{
   int rows,cols;
   double *A;

   if(popint(&cols) && popint(&rows)) {
      if(rows<0 || cols<0) {
         stkerr(" null: ","r or c is negative"); 
         HALT();
         return 0;
      }
      if((A=(double *)memget0(rows,cols))==NULL) {
         stkerr(" null: ",MEMNOT); 
         return 0;
      }
      return(push(MAT,(char *)memgetn("_null",5),NOTAG,0,A,NULL, \
         rows,cols,NULL));
   }
   return 0;
}

int patadd(void *pat, patitem *p, int *loc)
/* Add pat to pattern table p if pat is not already in p.  If loc is
   not NULL, on return loc contains the row offset where pat is located.

   NOTE: memcpy() below copies the full width of pattern bytes from
   incoming pat, so all its bytes need to have been set before calling
   this function to avoid random garbage in the copied row.

   The pattern table is increased by gro rows if it is full. 

   Note: with trace turned on, the pattern table called ptrnum is shown 
   when ptr is run.

   Example: adding ptr NUM for "swap" to ptrnum table:
      [tops@clacker] ready > "swap" trace ptr
       Pattern table properties:
        pattern array address: 8B85948
        bytes width: 4
        rows total: 3
        rows added when grow: 1 <<< forces table increase every time
        rows in use: 3
        contents (hex) in byte order (not endian order):
         Row  1: 2870AA8
         Row  2: C8E2A98
         Row  3: A878A98 <<< this is swap just added
*/
{
   char *P;
   int at;

   if(p->use && patloc(pat,p,&at)) {
      if(loc) *loc=at;
      return 1;
   }
   if(p->use==p->siz) { /* increase table size by gro rows */

      if((P=malloc(1+(p->siz+p->gro)*p->wid))==NULL) {
         stkerr(" patadd: ",MEMNOT);
         return 0;
      }
      memcpy(P,(char *)p->pat,(p->use*p->wid));
      p->siz+=p->gro;

      mallfree((void *)&p->pat);
      p->pat=(char *)P;
   }
/* Copy the full wid of bytes from pat into next row of table: */
   memcpy((char *)p->pat+(p->use*p->wid),(char *)pat,p->wid);
   if(loc) *loc=p->use;
   p->use+=1; /* next row to use */

   return 1;
}

int patadd1(void *pat, patitem *p, int *loc)
/* Add pat to pattern table p even if pat is already in p.  If loc is
   not NULL, on return loc contains the row offset where pat is located.

   NOTE: memcpy() below copies the full width of pattern bytes from
   incoming pat, so all its bytes need to have been set before calling
   this function to avoid random garbage in the copied row.

   The pattern table is increased by gro rows if it is full. 
*/
{
   char *P;
   if(p->use==p->siz) { /* increase table size by gro rows */

      if((P=malloc(1+(p->siz+p->gro)*p->wid))==NULL) {
         stkerr(" patadd: ",MEMNOT);
         return 0;
      }
      memcpy(P,(char *)p->pat,(p->use*p->wid));
      p->siz+=p->gro;

      mallfree((void *)&p->pat);
      p->pat=(char *)P;
   }
/* Copy the full wid of bytes from pat into next row of table: */
   memcpy((char *)p->pat+(p->use*p->wid),(char *)pat,p->wid);
   if(loc) *loc=p->use;
   p->use+=1; /* next row to use */

   return 1;
}

void *patget(int n, patitem *p)
/* Get the memory pointer to the nth pattern of pattern table p.  The
   pattern bytes must be fetched from the returned pointer.  

   The value of n is obtained from function patadd() when a pattern is 
   added to the table.  Require n>0 and n!>p.use.

   Returns a NULL pointer with no message if n is out of bounds.

   Note that bytes are in memory order, not endian order, and a func-
   tion like memcpy() can be used to move them into any type of vari-
   able. 

   Here is an example from function exe() in exe.c, where bytes of a
   catitem pointer are copied from pattern table ptrnum into pointer 
   oncat:
      catitem *c;
      c=(catitem *)patget((int)*(X.i+oplus),ptrnum);
      memcpy(&oncat,c,sizeof(catitem *)); */
{
   if(n<1 || n>p->use) {
      return NULL; /* n is out of bounds */
   }
   return((void *)((char *)p->pat+n*(p->wid)));
}

int patloc(void *pat, patitem *p, int *loc)
/* Find row offset, loc, where pat is located in pattern table p.
   Return 1 if pat is found, 0 if not. */
{
   int len,k=0;
   char *p1,*p2;

   p1=(char *)pat;    /* pattern to find */
   p2=(char *)p->pat; /* table of patterns */
   len=p->wid;        /* length of each pattern */
   
   while(k<p->use && !strnmatch(p1,p2,len)) {
      k++;
      p2+=len;
   }
   if(k<p->use) { /* found */
      *loc=k;
      return 1;
   }
   return 0; /* not found */
}

patitem *patnew(int siz, int gro, int wid)
/* Create a pattern table with siz rows of wid bytes.  Every time the
   table needs more rows, gro rows will be added. 

   The first row (offset 0) is marked as in use, so a row location of 
   zero will not be a valid one.  Function patget() does not fetch rows
   with offsets less than one. */
{
   patitem *p;
   char *P;
   int len;

   siz=MAX(1,siz);
   wid=MAX(1,wid);
   gro=MAX(1,gro);

   if((p=malloc(sizeof(patitem)))==NULL) {
      stkerr(" patnew: ",MEMNOT);
      return NULL;
   }
   if((P=malloc((len=1+siz*wid)))==NULL) {
      stkerr(" patnew: ",MEMNOT);
      return NULL;
   }
   memset(P,0,len); 

   p->pat=(char *)P;
   p->use=1; /* don't use row 0 */
   p->siz=siz;
   p->gro=gro;
   p->wid=wid;

   return p;
}

void patprops(patitem *p)
/* Displaying elements of patitem. */
{
   unsigned char *P;
   int bytes,i,k=0,rows;

   gprintf(" Pattern table properties:");
   nc();
   gprintf("  pattern array address: %X",(P=p->pat));
   nc();
   gprintf("  rows added when grow: %d",p->gro);
   nc();
   gprintf("  rows total: %d",p->siz);
   nc();
   gprintf("  rows in use: %d",(rows=p->use));
   nc();
   gprintf("  bytes width: %d",(bytes=p->wid));
   nc();
   if(rows) {
      gprintf("  contents (hex) in byte order (not endian order):");
      nc();
      for(;k<rows;k++) {
         gprintf("   Row %2d: ",k);
         for(i=0;i<bytes;i++) gprintf("%0X",*(P+i));
         nc();
         P+=bytes;
      }
   }
}

int scalar() /* scalar ( --- hS) */
{
   double *A;

   if((A=(double *)memget0(1,1))==NULL) {
      stkerr(" scalar: ",MEMNOT); 
      return 0;
   }
   return(push(MAT,(char *)memgetn("_scalar",7),NOTAG,0,A,NULL, \
      1,1,NULL));
}

int setbrk() /* setbrk (addr --- ) */
/* Set the end of accessible data space (aka "the break") to addr. */
{
   unsigned long a; /* intptr_t a; is not portable */
   double d;

   if(!popd(&d)) return 0;
   a=d;

   if(brk((void *)a)!=0) {
      stkerr(" setbrk: ", \
         "error setting the end of accessible data space");
      return 0;
   }
   return 1;
}

int strstk(int chars, char *name) /* ( --- qS) */
/* From C function, push uninitialized text string chars long to
   stack. */
{
   char *S;

   chars=MAX(0,chars);

   if((S=malloc(1+chars))==NULL) {
      stkerr(" strstk: ",MEMNOT);
      return 0;
   }
   *(S+chars)='\0';

   return(push(STR,(char *)memgetn(name,strlen(name)),NOTAG,0,NULL,S, \
      1,chars,NULL));
}

int tpurged() /* tpurged (nType --- hPurged) */
/* Return a purged stack item for Type */
{
   int type;

   if(tos->typ!=NUM) {
      stkerr(" tpurged: ",NUMNOT);
      return 0;
   }
   popint(&type);

   switch(type) {

      default:
      case MAT: return(matstk(0,0,"_purged"));
      case NUM: return(pushint(0));
      case VOL: return(volstk(0,0,"_empty"));
      case STR: return(strstk(0,"_empty"));
   }
}

int volstk(int rows, int chars, char *name) /* ( --- hT) */
/* From C function, push uninitialized volume, rows-by-chars, to stack.

   Always allocates an exact multiple of sizeof(double) bytes, so VOLs
   have 8-byte boundaries that are suitable for doubles on machines 
   requiring them, such as Sun workstations. */
{
   char *T;
   unsigned int bytes=1,len;

   rows=MAX(0,rows);
   chars=MAX(0,chars);

   len=rows*chars;

/* Want len+1 bytes to allow for an ending null byte: */
   bytes+=MAX((unsigned int)sizeof(int),(unsigned int)len);

   bytes/=sizeof(double); /* how many doubles */
   bytes++;               /* plus one for case of non-zero mod 8 */
   bytes*=sizeof(double); /* how many bytes */

   if((T=malloc(bytes))==NULL) {
      stkerr(" volstk: ",MEMNOT);
      return 0;
   }
   *(T+len)='\0'; /* null following rows*cols */

   return(push(VOL,(char *)memgetn(name,strlen(name)),NOTAG,0,NULL,T, \
      rows,chars,NULL));
}

int add_idx() /* ( hA hR hC --- hAI ) {{{1 */
/*
 * man entry:  add_idx {{{2
 * ( hA hR hC --- hAI ) Embed row and column vectors {R} and {C} inside matrix [A].
 * category: math::matrix::partitioning, partitioning
 * related: get_idx, +, mtext
 * 2}}}
 */
{
int DEBUG = 0;
    int     nRow, nCol, *row_idx = 0, *col_idx = 0, i, sparse, NPT;
    double *R_vec, *C_vec, *A_in, *A_out;
    char   *name = "_add_idx";
    SparseMatrix Asp_in, Asp_out;
    /* type check the inputs {{{2 */
    if (is_complex(tos-2)) {
        stkerr(" add_idx: ", "complex case not yet implemented");
        return 0;
    }
    if (tos->typ != MAT) {
        stkerr(" add_idx:  column vector ", MATNOT);
        return 0;
    }
    nCol  = tos->row * tos->col;
    C_vec = tos->mat;

    if ((tos-1)->typ != MAT) {
        stkerr(" add_idx:  row vector ", MATNOT);
        return 0;
    }
    nRow  = (tos-1)->row * (tos-1)->col;
    R_vec = (tos-1)->mat;

    sparse = is_sparse(tos-2);
    if (((tos-2)->typ != MAT) && !sparse) {
        stkerr(" add_idx:  [A] ", MATNOT);
        return 0;
    }
    /* 2}}} */

    if (sparse) {    /* sparse input */
        Asp_in = sparse_overlay(tos-2);
        if ((nRow != Asp_in.H[ROWS]) ||
            (nCol != Asp_in.H[COLS])) {
            stkerr(" add_idx:  [A] ", MATSNOTC);
            return 0;
        }
        if (!sparse_stk(Asp_in.H[ROWS]  , /* in  */
                        Asp_in.H[COLS]  , /* in  */
                        Asp_in.H[n_STR] , /* in  number of strings       */
                        Asp_in.H[n_NONZ], /* in  number of nonzero terms */
                        Asp_in.H[COMPLX], /* in  0=real  1=complex       */
                        1               , /* in  1=add internal index    */
                        name            , /* in  */
                        &Asp_out)) {      /* out */
            stkerr(" add_idx: [Asp_out] ",MEMNOT); 
            return 0;
        }
        NPT = NUM_PER_TERM( Asp_in.H[COMPLX] );
        /* note:  cannot simply do memcpy(Asp_out.data, Asp_in.data, ... )
         * because Asp_out has internal index arrays in the middle. 
         */
        memcpy(Asp_out.S_start, Asp_in.S_start, (nCol+1)*sizeof(int));
        memcpy(Asp_out.N_start, Asp_in.N_start, (nCol+1)*sizeof(int));
if (DEBUG) {
for (i = 0; i < (nCol+1); i++) {
gprintf("add_idx: Asp_in.S_start[%2d] = %2d Asp_out.S_start[%2d] = %2d\n",
i, Asp_in.S_start[i], i, Asp_out.S_start[i]);
gprintf("         Asp_in.N_start[%2d] = %2d Asp_out.N_start[%2d] = %2d\n",
i, Asp_in.N_start[i], i, Asp_out.N_start[i]);
}
}
        memcpy(Asp_out.S      , Asp_in.S      , Asp_in.H[n_STR]*
                                                sizeof(str_t));
if (DEBUG) {
for (i = 0; i < (nCol+1); i++) {
gprintf("         Asp_in.S      [%2d] = %2d\n", i, Asp_in.S[i]);
}
}
        memcpy(Asp_out.N      , Asp_in.N      , Asp_in.H[n_NONZ]*
                                                    NPT*sizeof(double));
        for (i = 0; i < nRow; i++) Asp_out.row_idx[i] = (int) R_vec[i];
        for (i = 0; i < nCol; i++) Asp_out.col_idx[i] = (int) C_vec[i];
    } else {         /* dense  input */
        if ((nRow != (tos-2)->row) ||
            (nCol != (tos-2)->col)) {
            stkerr(" add_idx:  [A] ", MATSNOTC);
            return 0;
        }

        A_in = (tos-2)->mat;

        matstk_idx(nRow, nCol, name);
        A_out   = tos->mat;
        row_idx = MAT_ROW_IDX(tos);
        col_idx = MAT_COL_IDX(tos);
        for (i = 0; i < nRow*nCol; i++) A_out[  i] =       A_in[ i];
        for (i = 0; i < nRow; i++)      row_idx[i] = (int) R_vec[i];
        for (i = 0; i < nCol; i++)      col_idx[i] = (int) C_vec[i];
    }
    lop(); /* drop [A] */
    lop(); /* drop {R} */
    lop(); /* drop {C} */
    set_indexed(tos);
    set_index_used(tos);
    return 1;
}
/* 1}}} */
int get_idx() /* ( hA --- hR hC ) {{{1 */
/*
 * man entry:  get_idx {{{2
 * ( hA --- hR hC ) Put on the stack row and column index vectors from [A] which contains embedded indices.  [A] can be dense or sparse.
 * category: math::matrix::partitioning, partitioning
 * related: add_idx, +, mtext
 * 2}}}
 */
{
    int     nRow, nCol, *row_idx, *col_idx, i;
    double *R_vec, *C_vec;
    SparseMatrix A;

    if (tos->typ != MAT && !is_sparse(tos)) {
        stkerr(" get_idx: ", MATNOT);
        return 0;
    }
    if (!is_indexed(tos)) {
        stkerr(" get_idx: ", "[A] is not internally indexed");
        return 0;
    }

    if (is_sparse(tos)) {
        A       = sparse_overlay(tos);
        nRow    = A.H[ROWS];
        nCol    = A.H[COLS];
        row_idx = A.row_idx;
        col_idx = A.col_idx;
    } else {
        nRow    = tos->row;
        nCol    = tos->col;
        row_idx = MAT_ROW_IDX(tos);
        col_idx = MAT_COL_IDX(tos);
    }
    if (!matstk(nRow, 1, "_R_ind")) return 0;
    R_vec   =  tos->mat;
    if (!matstk(nCol, 1, "_C_ind")) return 0;
    C_vec   =  tos->mat;

    for (i = 0; i < nRow; i++) R_vec[i] = (double) row_idx[i];
    for (i = 0; i < nCol; i++) C_vec[i] = (double) col_idx[i];

    pushd(2); roll(); /* ( A R C --- R C A ) */
    drop();           /* ( R C A --- R C )   */
    return 1;
}
/* 1}}} */
