/* File mday.n  June 2006

   Copyright (c) 2006-2011   D. R. Williamson

   Market model each day.

   Notice: a postfix version of this file, mday.nv, is sourced by
   files that use its words, so any time it is modified, it must be 
   parsed and saved, as this example shows:

      [dale@plunger] /opt/mytops/usr > tops
               Tops 3.0.1
      Sat Oct 18 12:27:47 PDT 2008
      [tops@plunger] ready > 'mday.n' parse_save
       parse_save: wrote file mday.nv

      [tops@plunger] ready > 

----------------------------------------------------------------------*/

   indexbase(1);
   if(missing("newday")) source("mday.v");
   if(missing("median")) source("mmath.v");
   if(missing("dCursor1")) source("mplot.v"); // mplot1.v, mplot2.v too

   fbook.path = fbook2.path = fbook4.path = "/tmp1/";
<<

\  Cleaned up Sat May 21 19:42:34 PDT 2011:

\  Structure of the columns in matrix MODEL of word pit().
\  LIB.MODEL = [LIB.H, LIB.L, LIB.C, LIB.V, LIB.OI];

\  Word dayget() (file mfil.v) uses these columns when it runs word 
\  pitget in this file:
   "pit" list: \ columns of LIB.MODEL

      "H" "L" "P" \ high, low, price
      "V" "OI" \ pit volume and open interest

   end struct

\-----------------------------------------------------------------------
>>
   function () = bs_add(n, t, p, id) { // add a record to buy/sell file
   /* On step n, doing transaction t at price p using strategy id, add 
      a record of this information to the buy/sell file.

      The buy/sell text file contains one record per line with these 
      fields separated by a blank (20h):

         SYM DATE CONMO PRICE TRANS IDENT STAMP

      that are defined as follows:

         SYM = symbol for market currently loaded by mdata() (uppercase)
         DATE = date for step n in the form YYYMMDD
         CONMO = symbol of contract month symbol in use (lowercase)
         PRICE = as-quoted price p
         TRANS = transaction given by these symbols:
            cl = close long
            cs = close short
            ol = open long
            os = open short
         IDENT = unique text string to identify decision strategy
         STAMP = Unix GMT time when this line was made (ctime() decodes)

      Incoming step n is converted to DATE, and as-quoted price p on
      step n is assumed to apply for CONMO being tracked.

      If incoming n=0, the latest step (today) is used.

      If incoming p=0, the closing perpetual price on step n is used,
      and converted to an as-quoted price.  If n is today and p=0, the 
      latest price (as from the web) is used.

      If strategy id is an empty string, "*" is used.

      Each record is for one contract, and over time the open and close
      records should balance as positions are opened and closed.

      Added records are simply appended to the file.

      There is no reason to remove records that were correctly added,
      so there is no function to do so; corrections to the file can
      be made with a text editor.

      Whenever libload() creates a market word and loads its library,
      it calls function bs_load() to place the buy/sell record into 
      the library of the word.

      Function bs_posn() also runs bs_load(), in case the buy/sell 
      record has been updated, and computes the running average of 
      all open positions at each step.  

      The purpose of bs_posn() is to create data for graphs, so it 
      uses only DATE and TRANS and assumes price to be the close used 
      in the graph.

      Example:

         Crude light for August, open long now at 7275:

            cl // load database
            bs_add(0, "ol", 7275, ""); // n=0 means today

            This is the record produced; * means no strategy id
            string was specified; q is month August:

               CL 1070706 q 7275 ol * 1183745154

            Here is how to decode the Unix time stamp:

               >> dot(ctime(numerate("1183745154")))
               Fri Jul  6 11:05:54 PDT 2007
   */
      { FILE = mpath + "bs.txt";
        MO = words(lowercase("F G H J K M N Q U V X Z"));
      }
      LIB = mdata.LIB;

      if(LIB=="")
         return(nl(dot(" bs_add: no market is defined")));

      if(rows(LIB.C)==0)
         return(nl(dot(" bs_add: no market is defined")));

      if(n==0) n1 = LIB.STEPS; // latest day, including today
      else n1 = n;

   /* SYM: */
      R = LIB.SYM + " ";

   /* DATE: */
   /* See mfil.v for column struct of array DATA; the last row of DATA 
      corresponds to last night's update, so it is one row short of 
      the arrays in LIB: */
      IF(n1>rows(DATA)); 
         (D, T) = sysdate(date); // using current date
         R = R + intstr(D) + " ";
      ELSE
         datecol = (<< data.date >>);
         R = R + intstr(@DATA[n1, datecol]) + " ";
      THEN

   /* CONMO: */
      conmo = @DATA[min(n1, rows(DATA)), (<< data.delmo >>) ];
      R = R + MO[integer(conmo/100)] + " ";

   /* PRICE: */
      IF(p==0) // use perpetual closing price C[n1] from database
      /* Converting perpetual C[n1] to scaled by removing ROLLDELTA: */
         p1 = @LIB.C[n1] - LIB.ROLLDELTA[n1];
      /* Converting scaled p1 to quoted by unscaling: */
         R = R +
            priceformat1(@main(p1, main("unscale")), LIB.SYM) + " ";
      ELSE R = R + intstr(p) + " ";
      THEN

   /* TRANS: */
      t1 = lowercase(t);
      if(!(t1=="cl" || t1=="cs" || t1=="ol" || t1=="os"))
         return(nl(dot(" bs_add: invalid transaction symbol: " + t)));
      R = R + t1 + " ";

   /* IDENT */
      if(id=="") R = R + "* ";
      else R = R + id + " ";

   /* STAMP: */
      R = R + intstr(time);

   /* Append R to FILE: */
      append(R, FILE);
   }

/* Utilities for function bs_add(): */

   function () = CL(n, id) { // close long on step n at the close
      return(bs_add(n, "cl", 0, id));
   }

   function () = CS(n, id) { // close short on step n at the close
      return(bs_add(n, "cs", 0, id));
   }

   function () = OL(n, id) { // open long on step n at the close
      return(bs_add(n, "ol", 0, id));
   }

   function () = OS(n, id) { // open short on step n at the close
      return(bs_add(n, "os", 0, id));
   }

   function () = bs_load(LIB) { // load buy/sell records into LIB
         
      LIB.OS = LIB.CS = LIB.OL = LIB.CL = "";
         
      FILE = bs_add.FILE;
      if(!filefound(FILE)) return;
      
      T = asciiload(FILE);
      if(!chars(T)) return;
        
      (SYMS, f) = word("", T, ndx(1)); // symbols are at index 1
      IF(!f)
         T = ""; 
         return; 
      THEN
        
      IF(any?((ROWS=grepe(SYMS, LIB.SYM)))) T = chop(T[ROWS]);
      ELSE
         T = "";
         return;
      THEN

      if(any?((ROWS=grepr(T, "os")))) LIB.OS = T[ROWS];
      if(any?((ROWS=grepr(T, "cs")))) LIB.CS = T[ROWS];
      if(any?((ROWS=grepr(T, "ol")))) LIB.OL = T[ROWS];
      if(any?((ROWS=grepr(T, "cl")))) LIB.CL = T[ROWS];
        
      T = "";
   }

   function (NL, PL, GL, NS, PS, GS) = bs_posn() { 
   /* Process positions data from the buy/sell records.

      The average price for long positions and short positions is re-
      turned in PL and PS, respectively.  

      NL and NS contain the number of contracts held on each step in 
      PL and PS, respectively.  

      NS(k)=0 means no contracts were held on step k, so for example, 
      if NS(k-1)=3, there were 3 contracts with average price PS(k-1) 
      closed out on step k-1.  The closing price would be P(k), the 
      price vector in LIB, and since PS contains short positions, the 
      gain would be 3*PS(k-1) - P(k). 

      Returned GL and GS are the gains of the longs and the shorts
      computed as just described.

      Note that the purpose of this word is to create graphs, not to 
      give a detailed accounting.  FOR SIMPLICITY, THE PRICE IN THE 
      BUY/SELL RECORDS IS IGNORED, AND PERPETUAL CLOSING PRICE IS 
      USED, AVOIDING PROBLEMS TRACKING ROLLOVER. */

      LIB = mdata.LIB;

   /* The buy/sell records are loaded by bs_load into arrays OS, CS, 
      OL, CL of LIB: */
      bs_load(LIB);

   /* The list of dates from DATA has rows up to last night's update;
      append a date for today that is one day past the last one: */
      datecol = (<< data.date >>);
      DATES = [DATA[*, datecol] ; 1+DATA[rows(DATA), datecol]];

      (P, f) = word(purged, LIB.OS, 2); // steps when opened short
      if(f) OPN = look([DATES , 1:rows(DATES)], numerate(P));
      else OPN = P; // P=purged when no OS steps

      (P, f) = word(purged, LIB.CS, 2); // steps when closed short
      if(f) CLS = look([DATES , 1:rows(DATES)], numerate(P));
      else CLS = P;

      (NS, PS) = posn(LIB.C, OPN, CLS); // using perpetual price

      (P, f) = word(purged, LIB.OL, 2); // steps when opened long
      if(f) OPN = look([DATES , 1:rows(DATES)], numerate(P));
      else OPN = P;

      (P, f) = word(purged, LIB.CL, 2); // steps when closed long
      if(f) CLS = look([DATES , 1:rows(DATES)], numerate(P));
      else CLS = P;

      (NL, PL) = posn(LIB.C, OPN, CLS); // using perpetual price

   /* Compute the gains of the longs from NL and PL: */
      R1 = (lag(NL, -1)==0 && NL!=0); // true on k-1
      R2 = lag(R1, 1); // true on steps k when longs closed out
      (toss, nc) = rake(NL, R1); // number of contracts
      (toss, pa) = rake(PL, R1); // average price bought
      (toss, p) = rake(LIB.C, R2); // sell price
      Prof = Dollars((nc .* (p - pa)), LIB.SYM);
   /* Running profit over all steps: */
      GL = partials(tier(null(dims(toss)), Prof, R2));

   /* Compute the gains of the shorts from NS and PS: */
      R1 = (lag(NS, -1)==0 && NS!=0); // true on k-1
      R2 = lag(R1, 1); // true on steps k when shorts closed out
      (toss, nc) = rake(NS, R1); // number of contracts
      (toss, pa) = rake(PS, R1); // average price sold
      (toss, p) = rake(LIB.C, R2); // buy price
      Prof = Dollars((nc .* (pa - p)), LIB.SYM);
   /* Running profit over all steps: */
      GS = partials(tier(null(dims(toss)), Prof, R2));

      P = "";
      CLS = DATES = nc = OPN = p = pa = 
      R1 = R2 = toss = Prof = purged;

      return(NL, PL, GL, NS, PS, GS);
   }

   function () = bs_load(LIB) { // load buy/sell records into LIB

      LIB.OS = LIB.CS = LIB.OL = LIB.CL = "";

      FILE = bs_add.FILE;
      if(!filefound(FILE)) return;

      T = asciiload(FILE);
      if(!chars(T)) return;

      (SYMS, f) = word("", T, ndx(1)); // symbols are at index 1
      IF(!f) 
         T = "";
         return;
      THEN

      IF(any?((ROWS=grepe(SYMS, LIB.SYM)))) T = chop(T[ROWS]);
      ELSE
         T = "";
         return;
      THEN

      if(any?((ROWS=grepr(T, "os")))) LIB.OS = T[ROWS];
      if(any?((ROWS=grepr(T, "cs")))) LIB.CS = T[ROWS];
      if(any?((ROWS=grepr(T, "ol")))) LIB.OL = T[ROWS];
      if(any?((ROWS=grepr(T, "cl")))) LIB.CL = T[ROWS];

      T = "";
   }

   function () = _COLOR() { /* Library of colors */
      /* To access the colors in the library of this word, use a
         phrase like one of these:
            C = _COLOR.Blue1; // infix
         or
            "_COLOR" "Green1" yank "C" book \ postfix
      */
      {
        Blue1 = "SkyBlue3";
        Blue2 = "DodgerBlue";
        Blue3 = "RoyalBlue";
        Blue4 = "MediumBlue";

        Bluex1 = "SteelBlue4";
        Bluex2 = "DodgerBlue4";
        Bluex3 = "RoyalBlue4";
        Bluex4 = "SlateBlue4";

        Burlywood1 = Burlywood(75); 
        Burlywood2 = Burlywood(90); 
        Burlywood3 = Burlywood(100);
        Burlywood4 = Burlywood(105);

        BrightBlue = colorvalue(0,0,255);
        BrightGreen = colorvalue(0,200,0); 

        CN1 = "DeepSkyBlue4";
        CP1 = "Aquamarine4";
        C6 = "Red3";

        Coral1 = Coral(-20);
        Coral2 = Coral(60);
        Coral3 = Coral(80);
        Coral4 = Coral(95);

        Green1 = "MediumSeaGreen";
        Green2 = "DarkOliveGreen4";
        Green3 = "Chartreuse4";
        Green4 = Green(100);

        Greenx1 = "Aquamarine4";
        Greenx2 = "OliveDrab4";
        Greenx3 = "Khaki4";
        Greenx4 = main("100 Green1");

        Silver4 = SlvrBlue(100);

        XUP = "Wheat3";  
        XDN = "Bisque3";
      }
   }

   function () = div() { // toggle the number of graphs displayed
      if(func.DISPLAY == 1) func.DISPLAY = 2;
      else func.DISPLAY = 1;
   }

   function (d) = Dollars(p, name) {
   /* Incoming p is a change in the perpetual price, like the values
      that are contained in matrix C and plotted. */
      d = integer((0.5+dprices(p,name)/qtic(name))*qticDollars(name));
   }

   function () = Fixup() { // fix out-of-tolerance high and low data
   /* NEED A REPLACMENT FOR REPRESENTATIVE LOW AND HIGH PRICES, L4
      AND S4, BEFORE THIS FUNCTION CAN BE USED AGAIN. */
      return();

   /* Out-of-tolerance high and low data is still wrong, but it is 
      bounded so the plot is not wrecked. */

      r = rows(main("H"));

   /* Replace H[i] if it exceeds a representative high price: */
      L4 = makePOP.MODEL0[1:r,21]; // goal of group L4, a high price
      H = min1([main("H") , L4])[*,1]; 

   /* Replace L[i] if it exceeds a representative low price: */
      S4 = makePOP.MODEL0[1:r,17]; // goal of group S4, a low price
      L = max1([main("L") , S4])[*,1];

   /* Override arrays H and L in the libraries of words H and L: */
      bank(H, "H", "H");
      bank(L, "L", "L");

      H = L = L4 = S4 = purged;
   }

/*
   function (Lx) = fLx() {
   /* Best-positioned longs in midfield model: Lx = Mx/Kb */
      M = fMx;

      LIB = mdata.LIB;
      R = clone(LIB.ROLLDELTA, cols(M));
      K = repeat(LIB.Kb', rows(M));
      Lx = R + (M - R)./K;
      K = R = M = purged;
   }

   function (Mx) = fMx() {
   /* Midfields.  These are independent variables, of which Lx and Sx
      are functions. */
      LIB = mdata.LIB;
      Mx = LIB.Mx;
   }

   function (Sx) = fSx() {
   /* Best-positioned shorts in midfield model: Sx = Mx/Ks */
      M = fMx;

      LIB = mdata.LIB;
      R = clone(LIB.ROLLDELTA, cols(M));
      K = repeat(LIB.Ks', rows(M));
      Sx = R + (M - R)./K;
      K = R = M = purged;
   }
*/
   function () = func(mkt) {
   /* Word func() is called by all the market words in mday.v, like hg, 
      dj, c, sf, etc. */
      { MKT = ""; 
        DISPLAY = 1; // default
        DID = DISPLAY;
      }
/*    Purgelib causes a lot of work for the system.  Try getting along
      without.  File mrtim.n uses a purged ROLLDELTA to detect a purged
      MKT library.
      if(chars(MKT)) purgelib(mdata.LIB); // empty lib of former mkt
*/
      MKT = mkt;

      mdata(mkt);

      IF(X11) 
         IF(mday.SERVERS) 

            IF(DISPLAY == 1)
               IF(DID != DISPLAY)
                  DID = DISPLAY;
                  remoterun("plotclose", @sockets[1]); // close graph 1
                  remoterun("plotclose", @sockets[2]); // close graph 2
               THEN;
            // Set parameters for the default window of file plot.v,
            // called plotWCB:
            // Window height:
               remoterun("571 plotWCB wcb.h poke", @sockets[1]);
            // Window width:
               remoterun("431 plotWCB wcb.w poke", @sockets[1]);
            // Window title:
               remoterun("'Daily' 'plotWCB' 'title' bank", @sockets[1]);
               SERVERS_RUN.NS = 1;
            ELSE // make graph 1 smaller to accommodate graph 2:
               IF(DID != DISPLAY)
                  DID = DISPLAY;
                  remoterun("plotclose", @sockets[1]); // close graph 1
               THEN;
               remoterun("270 plotWCB wcb.h poke", @sockets[1]);
               SERVERS_RUN.NS = 2;
            THEN;

            off(ok);
            view1();
            if(DISPLAY == 2) view2();
            func_show(mdata.LIB);
            sl;
            //<< 30l >> 
            //<< 90l >> 
            //<< 10l >> 
         ELSE
            if(exists?("console")) console.Kptr = no;
         THEN
      ELSE 
         nl(dot(vol2str(LATEST(mkt)))); 
      THEN;
   }

   function (LIB1) = func_show(LIB) {
   /* Set up graphics for displaying MODEL from LIB. */

      if(mday.SERVERS) ( /* Servers supporting graphic windows: */

         remoterun("xx", @sockets[1]), /* socket[1] displays MODEL */
         remoterun("xx", @sockets[2]), /* socket[2] displays various */

         remoterun("noclip", @sockets[1]),
         remoterun("noclip", @sockets[2]),

      /* Graph for Server 1 is view1() that was run in func()
         before this function was called: */

            remoteput(LIB.MODEL, @sockets[1]),
            remoteput(LIB.MODEL_lines, @sockets[1]),
            remoteput(LIB.MODEL_colors, @sockets[1]),
            remoterun("graphset", @sockets[1]),

         /* Clip for graph 1: */
            (CLIPcols = LIB.MODEL_clip),
         << CLIPcols any
            IF >>
               remoteput(LIB.MODEL[*,CLIPcols], @sockets[1]),
               remoterun("clip", @sockets[1]),
         << THEN >>

         << "SERVERS_RUN" "NS" yank 2 = "func" "DISPLAY" yank 2 = and

            IF >>

            /* Graph for Server 2 is view2() that was run in func()
               before this function was called: */
               remoteput(LIB.MODEL2, @sockets[2]),
               remoteput(LIB.MODEL2_lines, @sockets[2]),
               remoteput(LIB.MODEL2_colors, @sockets[2]),
               remoterun("graphset", @sockets[2]),

            /* Clip for graph 2: */
               (CLIPcols = LIB.MODEL2_clip),
            << CLIPcols any
               IF >>
                  remoteput(LIB.MODEL2[*,CLIPcols], @sockets[2]),
                  remoterun("clip", @sockets[2]),
            << THEN >>

         << THEN >>
      );

   /* Graph for head node (only used when no servers): */
      if(X11) graphset(LIB.MODEL_lines, LIB.MODEL_colors);

      return(LIB);
   }

   function () = Kprompt() { // display data at the prompt
      { << 
        [
         "(nN --- qN) dup 0> IF '+' swap intstr + ELSE intstr THEN"
         "intsigned" macro
        ] >>
      }
      <<
      "func" "MKT" localref exists?
      IF "func" "MKT" yank any?
         IF LATEST "L" book 
            >>
            LIB = mdata.LIB;
            T = [L[1] ; L[2] ; L[3] ; L[4]];
            <<
            T vol2str out 0> IF sp THEN dot
         THEN
      THEN
      >>
   }

   function () = lab(LIB) { // make and graph the lab model
      { nd = mo(0.5); }
      lab2(LIB);
   }

   function () = lab2(LIB) { 
      { nd = lab.nd; }
   /* Assemble the curves of graph 2: */

  //  if(nd - pit.nd) HALT(<< "lab2: inconsistent nd" . nl >>);

   /* Curves of graph 2: */
// OBSOLETE
      LIB.LAB = [LIB.H, LIB.L, LIB.C, pit.MP, pit.MH, pit.ML, 
         pit.PT, pit.RL]; // RL only

      { /* Lines and colors for the columns of matrix LIB.LAB: */

         (lines, colors) = colorset(<<

            "LineSolid" "LS" book

            list:
             \ Prices:
               0 Blue LS
               "SpringGreen2" LS
               "DarkGoldenrod2" LS

             \ Medians:
               30 Coral LS
\              "Orchid3" LS
\              "Maroon3" LS
               70 Plum LS
               100 Magenta LS

             \ Trendlines:
               70 Burlywood LS
               85 Burlywood LS

             \ Stages of lows:
               nd 1st DO 80 Green1 LS LOOP
            end >>
         );
         if(X11) graphset(lines, colors);
      }
   /* Lines and colors for graph: */
      LIB.LAB_lines = lines;
      LIB.LAB_colors = colors;

   /* Clipping: */
      LIB.LAB_clip = [1:3];

   /* Nudging: */
      LIB.NUDGE = 0;
      LIB.PIX = 0;
   }

   function () = libload(LIB) { /* load library LIB */
   /* Incoming string LIB equals regular SYMBOL plus "lib," like 
      SFlib. */

/* 
   A goal here is to get all the data that dataget() has previously 
   stored in main arrays moved into LIB, so there is no data floating 
   around in main meaning that each market can be read only once
   and held in its own local library.  I think this goal has been
   reached (on March 19, 2008), and probably main arrays _DATA, C,
   H, L and whatever else can be begin to be eliminated.

   The main arrays C, H, L, ROLLDELTA and DATE are stored below.

   The use of main name "Name" is being diminished and should be
   eliminated.
*/

   /* Load data from yearly files: */
      <<
      LIB "lib" "" strp (qSYM) \ take the lib off of LIB
      (qSYM) dataget not 
      IF " libload: no data found for " LIB + . nl HALT THEN 
      Fixup
      >>
      LIB.SYM = uppercase(<< "Name" main >>);
/*
DEACTIVATED FOR EVERYDAY USE
      bs_load(LIB); /* load buy/sell record into LIB */
*/

   /* Store some main arrays into LIB; from dataget(), prices have
      been updated from web if any, and rollover recomputed: */
      LIB.C = main("C");
      LIB.H = main("H");
      LIB.L = main("L");

   /* Sat May 21 18:57:34 PDT 2011.  Pit volume and open interest; lag 
      negatively one session to align with date: */
      LIB.V = lag(main("_DATA data.vol catch"), -1);
      LIB.OI = lag(main("_DATA data.int catch"), -1);

      LIB.ROLLDELTA = main("_DATA data.roll catch");
      LIB.DATE = main("_DATA data.date catch");

      LIB.STEPS = rows(LIB.C);
   }

   function (V, PV) = lslevel(P, REF, tREF, f) { // price level PV
   /* From prices P, determine line V when P crosses line tREF, then
      from prices REF determine prices V that are above (f!=0) or
      below (f==0) line V.
   */
      V = looking(tREF, ((P<tREF && lag(P>tREF, 1)) ||
                         (P>tREF && lag(P<tREF, 1))));
      if(f) PV = looking(REF, REF>V);
      else  PV = looking(REF, REF<V);
   }

   function (LIB) = mdata(MKT) {
   /* Make a library word called MKTlib, and load data into it.

      Returned handle LIB is used to access data, as in these
      examples of fetching and storing:
         c = LIB.C      // c is equal to C fetched from library LIB
         LIB.A = myMAT; // myMAT is stored in library LIB as A

         LIB = mdata.LIB; // set local LIB to last loaded; used a lot
   */
      { LIB = ""; }

      LIB = uppercase(MKT) + "lib"; /* LIB has the form MKTlib */
      if(missing(LIB)) library(LIB);

   /* Load data and compute curves: */
      libload(LIB);
    //pit(LIB); // removed Tue May 24 05:32:58 PDT 2011
      if(func.DISPLAY == 2) lab(LIB);

   /* Set up animation: */
      IF(X11)
         zspan(mo(3)); // default span
         z.d = LIB.STEPS;
      THEN

   /* Special function for console cursor: */
      if(exists?("console")) console.Kptr = ptr("Kprompt");

      return(LIB);
   }

   function () = pit(LIB) { // pit model and graph
      { nd = lab.nd; }

/*----------

   /* Price data: */
      H = LIB.H;
      L = LIB.L;
      P = LIB.C;

   /* Lowest high in last nd steps: */
      LH = mmin(H, nd); // lowest high

   /* Highest low in last nd steps: */
      HL = mmax(L, nd); // highest low

   /* There are quirky times when HL is lower than LH; force HL to be 
      max: */

      (LH, HL) = (<< LH HL (hLH hHL) 2dup min rev max >>);   

      HL1 = looking(HL, (L==HL)); // HL when low, L, equals HL
      HL2 = looking(HL1, (delta(HL1)<0 && L==HL1));

      LH1 = looking(LH, (H==LH)); // LH when high, H, equals LH
      LH2 = looking(LH1, (delta(LH1)>0 && H==LH1));

   /* Trace of pit close: */
      { W = 5; } // 5 days
      tP = tr(P, W);

   /* Split P into two regions: */
      P2 = looking(P, P>=tP); // high prices
      tP2 = tr(P2, W);

      P1 = looking(P, P<tP);  // low prices
      tP1 = tr(P1, W);

----------*/

   /* Curves of graph 1: */
      IF(func.DISPLAY == 1);

      /* Sat May 21 19:42:34 PDT 2011.  Modified for pit volume and
         open interest, and some cleanup of unused phrases. */

         TEMP = [LIB.H, LIB.L, LIB.C, LIB.V, LIB.OI]; 

      /* MODEL is fbooked into main, and macro LIB.MODEL will fetch 
         it as if it were stored in LIB: */
      // fbook(TEMP, "MODEL");
         mainbook(TEMP, "MODEL"); // July 2009: use RAM
         macro("'MODEL' main", localref(LIB, "MODEL"));

      ELSE
         .(" pit: this branch for DISPLAY==2 must be updated");
         nl();
         .(" pit: halting");
         nl();
         HALT();
      THEN

   /* Arrays for trading functions: */
/*
      ZLIB.P = LIB.C;
      ZLIB.ROLLDELTA = LIB.ROLLDELTA;
*/

   /* Purge arrays here: */
      H = HL = HL1 = HL2 = L = LH = LH1 = LH2 = P = TEMP = purged; 

      P1 = P2 = tP = tP1 = tP2 = purged;
     
   /* Purge arrays H, L, P, V and OI in LIB that are now contained in
      LIB.MODEL: */
      LIB.H = LIB.L = LIB.C = LIB.V = LIB.OI = purged;

   /* If word tgraph is present, this function if being run for real
      time processing.  Return without doing any graphics: */
      if(exists?("tgraph")) return;

   /* Lines and colors for the columns of matrix LIB.MODEL: */
      (lines, colors) = colorset(<<
         "LineSolid" "LS" book
         list:
          \ LH2, HL2:
           0 Purple LS
           40 Magenta LS

          \ LH1, HL1:
           "#2E37FE" LS \ stained glass (blue)
           "#83F52C" LS \ neon green

          \ Lowest high, LH; Highest low, HL
           70 Purple LS
           100 Magenta LS

          \ High and low:
            60 Blue1 LS
            80 Green1 LS

          \ Price:
            "DarkGoldenrod1" LS

      \ Colors for these are from mrtim.n:
          \tP, S1, S2, S3, B1, B2, B3,

           30 Red LS     \ tP
           60 Green LS \ S1
           -10 Green1 LS \ S2
           60 Coral LS   \ S3

           "#2E37FE" LS \ stained glass blue \ B1
           -10 Blue LS     \ B2
           70 Burlywood LS \ B3

         end >>
      );
      if(X11) graphset(lines, colors);

   /* Lines and colors for graph: */
      LIB.MODEL_lines = lines;
      LIB.MODEL_colors = colors;

   /* Clipping: */
      LIB.MODEL_clip = [5:8];

   /* Nudging: */
      LIB.NUDGE = 0;
      LIB.PIX = 0;
   }
<<
   inline: pitget (qLIB k --- hP) \ pit model data on step k
{     Returned P is a row vector with columns of data for the model
      being created in dayget.

      The data for P comes from the pit model that is made from daily
      data in mday.n, and is now stored as matrix MODEL in the library
      of word LIB.

      The struct for the columns of matrix MODEL is called pit, and is
      shown in file mday.n.  The pit struct names, such as pit.H, are 
      used below to obtain data columns from MODEL.

      This word selects the data from matrix MODEL in the order that 
      corresponds to "Daily pit model" given in the struct of matrix 
      daysget.P given at the top file mrtim.n (later, mobius.n).
}
      "k" book
      (qLIB) "MODEL" yank push \ matrix MODEL from word LIB

         list:
          \ Pit daily: "H" "L" "P" \ high, low, price
            peek k reach pit.H pry
            peek k reach pit.L pry
            peek k reach pit.C pry
            peek k reach pit.OI pry
         end (hP)
         (hP) bend (hP1) \ column into row

         (hP) pull (hLIB.MODEL) drop (hP)
   end
{
   inline: pitput (hD qLIB nR nC --- )
\     Store row vector D into matrix LIB.MODEL, at row R beginning at 
\     column C.  This word is called by dayget, file mfile.v.
      rot (qLIB) "MODEL" yank (hA) dup push \ matrix MODEL from word LIB

      (hD nR nC hA) rev (hD hA nR nC) place

\     Actually, LIB.MODEL is stored in main as MODEL; see word pit.
\     Run fbook again for changed A:
    \ pull (hA) "MODEL" fbook \ fbook again; see word pit
      pull (hA) "MODEL" mainbook \ July 2009: use RAM
   end
}
>>
   function () = trader() { 
   /* Run trades(), then gather the trading results and make a graph. */

      trades();

      LIB = mdata.LIB;

      (NL, PL, GL, NS, PS, GS) = bs_posn();

      Z = null(rows(GL), 1); // line at zero

      LIB.TRADER = [GS, GL, GL+GS, Z];

    { /* Lines and colors for the columns of matrix LIB.TRADER: */

         (lines, colors) = colorset(<<

            "LineSolid" "LS" book

            list:
               40 Blue LS \ GS
               40 Green LS \ GL
               "DarkGoldenrod2" LS \ GS + GL
               "Gray40" LS \ Z
            end >>
         );

         if(X11) graphset(lines, colors);
      }

   /* Lines and colors for graph: */
      LIB.TRADER_lines = lines;
      LIB.TRADER_colors = colors;

   /* Clipping: */
      LIB.TRADER_clip = [1:cols(LIB.TRADER)];
  
   /* Nudging: */
      NUDGE = 0;
      PIX[1:rows(NUDGE)] = -ones(rows(NUDGE));
   }

   function () = trades() {
   /* Makes trades and saves record of buying and selling in file
      mpath/SYM_bs.txt.  

      The specifics of the trading strategy are in words zenter(), 
      zexit() and ztrade(). */

      { Y = 100*year; }

      LIB = mdata.LIB;
      if(chars(LIB)==0) HALT(nl(dot(" Need to load a market")));

      ztrade_init(LIB);

      //ztrade.V = no;
      ztrade.V = yes;

      if(mday.SERVERS) ztrade.V = no;

      File = mpath + LIB.SYM + "_bs.txt";
      deleteif(File);
      bs_add.FILE = File;

      if(LIB.STEPS < Y) ns = 1;
      else ns = LIB.STEPS - Y;

      DO(LIB.STEPS-2, ns)

//      zrollover(LIB, I); // bs_posn() can't handle rollover because
                           // a close and another open in the same
                           // direction are on the same step
         <<
         LIB I zexit  LIB I ztrade 
         LIB I zenter LIB I ztrade 
{
         LIB I zexit any?
         IF LIB I ztrade 
         ELSE LIB I zenter LIB I ztrade
         THEN
}
         >>
      LOOP;
   }

   function (M) = view1() {
   /* Default view for graph 1.
      The function that is run here must set up LIB.MODEL arrays. */
      return(vpit());
      //return(vlab1());
   }

   function (M) = view2() {
   /* Default view for graph 2.
      The function that is run here must set up LIB.MODEL2 arrays. */
      return(vlab2());
      //return(vtrader());
   }

   function (M) = vlab1() {
   /* View of the lab model for graph 1 */

      LIB = mdata.LIB;
      LIB.MODEL = LIB.LAB;
      LIB.MODEL_lines = LIB.LAB_lines;
      LIB.MODEL_colors = LIB.LAB_colors;
      LIB.MODEL_clip = LIB.LAB_clip;

      IF(mday.SERVERS)
         remoterun("'noop' 'pExpose' 'NUDGE' bank", @sockets[1]);
      ELSE
         IF(X11)
            graphset(LIB.MODEL_lines, LIB.MODEL_colors);
            IF(rows(LIB.MODEL_clip))
               clip(LIB.MODEL[*, LIB.MODEL_clip]);
            ELSE
               noclip;
            THEN
         THEN
      THEN;

      return(LIB.MODEL);
   }

   function (M) = vlab2() {
   /* View of the lab model for graph 2 */

      IF(exists?("SERVERS_RUN"));
         if(SERVERS_RUN.NS==1) return(0);
      THEN;

      LIB = mdata.LIB;
      LIB.MODEL2 = LIB.LAB;
      LIB.MODEL2_lines = LIB.LAB_lines;
      LIB.MODEL2_colors = LIB.LAB_colors;
      LIB.MODEL2_clip = LIB.LAB_clip;

      IF(mday.SERVERS)
         remoterun("'noop' 'pExpose' 'NUDGE' bank", @sockets[2]);
         <<
       \ Set up Server 2 to use Nudge_g2() of mplot2.v:
         LIB "NUDGE" yank sockets 2nd pry remoteput
         '"Nudge_g2" "COLS" bank' sockets 2nd pry remoterun

         LIB "PIX" yank sockets 2nd pry remoteput
         '"Nudge_g2" "PIX" bank' sockets 2nd pry remoterun

         "'Nudge_g2' 'pExpose' 'NUDGE' bank"
         sockets 2nd pry remoterun
         >>
      ELSE
         IF(X11)
            graphset(LIB.MODEL2_lines, LIB.MODEL2_colors);
            IF(rows(LIB.MODEL2_clip))
               clip(LIB.MODEL2[*, LIB.MODEL2_clip]);
            ELSE
               noclip;
            THEN
            <<
            "Nudge_g2" exists?
            IF LIB "NUDGE" yank rows 0>
               IF LIB "NUDGE" yank "Nudge_g2" "COLS" bank
                  LIB "PIX" yank "Nudge_g2" "PIX" bank
                  "Nudge_g2" "pExpose" "NUDGE" bank
               ELSE
                  "noop" "pExpose" "NUDGE" bank
               THEN
            THEN
            >>
         THEN
      THEN;

      return(LIB.MODEL2);
   }

   function (M) = vpit() {
   /* View of the pit model for graph 1 */

      LIB = strchop(mdata.LIB);

      IF(mday.SERVERS)
         remoterun("'noop' 'pExpose' 'NUDGE' bank", @sockets[1]);
         <<
         LIB "NUDGE" yank rows 0>
       \ Set up Server 1 to use Nudge_g1() of mplot1.v:
         IF LIB "NUDGE" yank sockets 1st pry remoteput
            '"Nudge_g1" "COLS" bank' sockets 1st pry remoterun

            LIB "PIX" yank sockets 1st pry remoteput
            '"Nudge_g1" "PIX" bank' sockets 1st pry remoterun

            "'Nudge_g1' 'pExpose' 'NUDGE' bank"
            sockets 1st pry remoterun
         THEN
         >>
      ELSE
         IF(X11)
            graphset(LIB.MODEL_lines, LIB.MODEL_colors);
            IF(rows(LIB.MODEL_clip))
               clip(LIB.MODEL[*, LIB.MODEL_clip]);
            ELSE
               noclip;
            THEN
         THEN
      THEN;

      return(LIB.MODEL);
   }

   function (M) = vtrader() {
   /* View of the trader model for graph 2 */

      IF(exists?("SERVERS_RUN"));
         if(SERVERS_RUN.NS==1) return(0);
      THEN;
      
      LIB = mdata.LIB; 
      LIB.MODEL2 = LIB.TRADER;
      LIB.MODEL2_lines = LIB.TRADER_lines;
      LIB.MODEL2_colors = LIB.TRADER_colors;
      LIB.MODEL2_clip = LIB.TRADER_clip;
      
      IF(mday.SERVERS)
         remoterun("'noop' 'pExpose' 'NUDGE' bank", @sockets[2]);
      ELSE  
         IF(X11)
            graphset(LIB.MODEL2_lines, LIB.MODEL2_colors);
            IF(rows(LIB.MODEL2_clip))
               clip(LIB.MODEL2[*, LIB.MODEL2_clip]);
            ELSE
               noclip;
            THEN
         THEN
      THEN;

      return(LIB.MODEL2);
   }

/*--------------------------------------------------------------------*/

/* --------------------- NOT IN USE

   function () = ZLIB() {
   /* The library of this function holds all the arrays and shared
      variables.  Function ztrade_init() loads the arrays. */
      {
      /* Other keys advance one step; these keys do things: */
         CLOSE  = 99;  // key c
         LONG   = 108; // key l
         PROMPT = 112; // key p
         SHORT  = 115; // key s
         QUIT   = 113; // key q
      }
   }

   function (sig) = zenter(LIB, k) {
   /* Generate signal to enter on step k. */
      { never = true;

      /* Signals: */
         LONG  = ZLIB.LONG;
         SHORT = ZLIB.SHORT;

      /* Local arrays: */
         macro("'ZLIB' 'P'  yank", "P");
         macro("'ZLIB' 'RX'  yank", "RX");
      }
      IF(never);
         if(k<10) return(0); // skip initial steps in data
         never = false;
      THEN;

      IF((P[k-1] <= RX[k-2]) && (P[k] > RX[k-1])); 
         return(LONG); // open long if P crossed above 
      THEN;

      IF((P[k-1] >= RX[k-2]) && (P[k] < RX[k-1])); 
         return(SHORT); // open short if P crossed below
      THEN;

      return(0);
   }

   function (sig) = zexit(LIB, k) {
   /* Generate signal to exit on step k. */
      { 
      /* Signal: */
         CLOSE = ZLIB.CLOSE; 

      /* Local arrays: */
         macro("'ZLIB' 'P'  yank", "P");
      }
      if(zenter.never) return(0);

return(0);
      IF(ztrade.nSP > 0);
         IF(P[k] > TM[k-1]); 
            return(CLOSE); // close short if P crossed above
         THEN
/*
         IF(@P[k] > (ztrade.pSsum/ztrade.nSP));
            return(CLOSE); // close short if P crossed above orig price
         THEN
*/
      THEN

      IF(ztrade.nLP > 0);
         IF(P[k] < TM[k-1]); 
            return(CLOSE); // close long if P crossed below
         THEN
/*
         IF(@P[k] < (ztrade.pLsum/ztrade.nLP));
            return(CLOSE); // close long if P crossed below orig price
         THEN
*/
      THEN
      return(0);
   }

   function (KEY) = zfunc(LIB, k) {
   /* Run keyboard trading.  This function is called by functions zk() 
      and zn() that are running animations. */

      { never = true;

        macro(
           "(nN --- qN) dup 0> IF '+' swap intstr + ELSE intstr THEN",
           "intsigned");

     /* Keys that do things: */
        CLOSE  = ZLIB.CLOSE;
        LONG   = ZLIB.LONG;
        PROMPT = ZLIB.PROMPT;
        SHORT  = ZLIB.SHORT;
        QUIT   = ZLIB.QUIT;

     /* Macros that access the arrays in ZLIB: */
        macro(parse("ZLIB.P"), "P");
        macro("'ZLIB' 'dP' yank", "dP");
      }
      on(ok);

      dot(LIB.SYM + " step " + intstr(k));

      IF(never)
         never = false;

         dot(" initializing...");
         ztrade_init(LIB);

         name = LIB.SYM;
      THEN

   /* Display the news: */
      p = priceformat1(prices(P[k], name), name);
      dp = priceformat1(prices(dP[k], name), name);

      if(dP[k]>0) dot(sp, p + " " + dp);
      if(dP[k]<0) dot(sp, p + " " + dp);
      if(dP[k]==0) dot(sp, p + " no chg ");

      IF(ztrade.V) // verbose:

      THEN

   /* Check for rollover on every step: */
      zrollover(LIB, k); 

   /* Show closed trades: */
      IF(ztrade.nLtot>0 || ztrade.nStot>0) // any closed trades?
         nl;
         dot(" Gains");
         dot(" S: " + intstr(ztrade.pSHORT) +
             "(" + intstr(ztrade.nStot) + ")");
         dot(" " + intstr(ztrade.nSgood) + "/" + 
                   intstr(ztrade.nSgood+ztrade.nSbad));
         dot(" L: " + intstr(ztrade.pLONG) +
             "(" + intstr(ztrade.nLtot) + ")");
         dot(" " + intstr(ztrade.nLgood) + "/" + 
                   intstr(ztrade.nLgood+ztrade.nLbad));
      THEN

   /* Show open trades: */
      IF(ztrade.nSP>0)
         p = prices(ztrade.pSsum/ztrade.nSP, name);
         nl; sp; sp;
         dot("SHORT " + intstr(ztrade.nSP) + " at: "
            + priceformat1(p, name));
      THEN

      IF(ztrade.nLP>0)
         p = prices(ztrade.pLsum/ztrade.nLP, name);
         nl; sp; sp;
         dot("LONG " + intstr(ztrade.nLP) + " at: "
            + priceformat1(p, name));
      THEN

   /* Get keyboard input: */
      BEGIN

         KEY = getch();

         IF(KEY==PROMPT) 
            zprompter();
            dot(" continuing");
         THEN

      WHILE(KEY==LONG || KEY==SHORT || KEY==CLOSE || KEY==PROMPT)

         ztrade(KEY, LIB, k);

      REPEAT

      if(KEY==QUIT) KEY=ESC; // q is same as Esc

      return(KEY);
   }

   function () = zk(k0) {
   /* Run animation starting on step k0, stepping every key press. 
      Calls function zfunc(LIB, k) on every step k. */

      LIB = mdata.LIB; // use the current data
      kmax = LIB.STEPS;

      k = k0;
      zd(k);

      zfunc.never = true; // initialize zfunc()

      BEGIN
         z; // show step k
      WHILE(zfunc(LIB, k)!=ESC && k<kmax) // press Esc to halt
         k++;
         nl;
      REPEAT 
      HALT;
   }

   function () = zn(n) {
   /* Run animation for latest n steps, stepping on every key press.
      Calls function zfunc(LIB, k) on every step k. */

      LIB = mdata.LIB; // use the current data
      kmax = LIB.STEPS;

      k = kmax - n + 1;
      zd(k);

      zfunc.never = true; // initialize zfunc()

      BEGIN
         z; // show step k
      WHILE(zfunc(LIB, k)!=ESC && k<kmax) // press Esc to halt
         k++;
         nl;
      REPEAT 
      HALT;
   }

   function () = zrollover(LIB, k) {
   /* Roll contracts when the delivery month changes: */
      { never = true;

        CLOSE = ZLIB.CLOSE;
        LONG  = ZLIB.LONG;
        SHORT = ZLIB.SHORT;

        macro("'ZLIB' 'DELMO' yank", "dDelmo");
      }
      IF(never)
         step = 0; // equals k when do rollover
         never = false;
      THEN

      IF(ztrade.nSP>0 || ztrade.nLP>0) // any open trades?
         V = ztrade.V;
         step = 0;
         IF(dDelmo[k]!=0)
            IF(ztrade.nLP>0)
               nP = @ztrade.nLP;
               IF(V)
                  nl;
                  dot(" Rolling " + intstr(nP) + " long contracts");
               THEN
               ztrade(CLOSE, LIB, k); // close old
               DO(nP,1) ztrade(LONG, LIB, k); LOOP // open new
               step = k;
            THEN
            IF(ztrade.nSP>0)
               nP = @ztrade.nSP;
               IF(V)
                  nl;
                  dot(" Rolling " + intstr(nP) + " short contracts");
               THEN
               ztrade(CLOSE, LIB, k); // close
               DO(nP,1) ztrade(SHORT, LIB, k); LOOP // open new
               step = k;
            THEN
         THEN
      THEN;
   }

   function () = ztrade(KEY, LIB, k) {
   /* Sets pGoal for trades based upon Tgoal(LIB.SYM) (file mrc.v), but
      does nothing further with it.  Functions zenter() and zexit() may
      or may not use pGoal in their strategies. */
      { 
     /* Signals: */
        CLOSE = ZLIB.CLOSE;
        LONG  = ZLIB.LONG;
        SHORT = ZLIB.SHORT;

        macro("'ZLIB' 'ROLLDELTA'  yank", "R");

        V = no; // verbose default

        pGoal = pHigh = pLow = 0;

        nLP = 0;
        pLsum = 0;

        nLtot = 0; // total number of long contracts traded
        pLONG = 0; // running total $ of long trades

        nSP = 0;
        pSsum = 0;

        nStot = 0; // total number of short contracts traded
        pSHORT = 0; // running total $ of short trades

     /* Tracking success rate.  A single "trade" is complete and ready 
        for counting when one or more contracts are closed below. */
        nLgood = 0; // total number of good long trades
        nLbad = 0;  // total number of bad long trades
        nSgood = 0; // total number of good short trades
        nSbad = 0;  // total number of bad short trades

        maxHold = 20;
      }

      if(KEY==0) return;

      name = LIB.SYM;

      C = @LIB.C[k]; // perpetual price

      IF(KEY==CLOSE || KEY==LONG)
         IF(nSP) // close existing shorts:
            DO(nSP, 1) CS(k, "") LOOP; // add to buy-sell record
            pSHORT += (P=@Dollars((pSsum - C*nSP), name));

            if(P>0) nSgood++;
            else nSbad++;

            IF(V) 
               nl; sp; sp;
               dot("Closed " + intstr(nSP) + " short for gain of "
                  + intstr(P));
               (<< " on step " k intstr + . >>);
            THEN

            pGoal = pHigh = pLow = 0;
            pSsum = 0;
            nStot += nSP;
            nSP = 0;
         THEN
      THEN;

      IF(KEY==CLOSE || KEY==SHORT)
         IF(nLP) // close existing longs:
            DO(nLP, 1) CL(k, "") LOOP; // add to buy-sell record
            pLONG += (P=@Dollars((C*nLP - pLsum), name));

            if(P>0) nLgood++;
            else nLbad++;

            IF(V) 
               nl; sp; sp;
               dot("Closed " + intstr(nLP) + " long for gain of "
                  + intstr(P));
               (<< " on step " k intstr + . >>);
            THEN

            pGoal = pHigh = pLow = 0;
            pLsum = 0;
            nLtot += nLP;
            nLP = 0;
         THEN
      THEN;

      IF(KEY==LONG && nLP < maxHold)
         OL(k, ""); // add to buy-sell record
         nLP++; // add long
         pLsum += C;
         IF(nLP==1) 
            pGoal = @(R[k] + (C - R[k])*(1+Tgoal(LIB.SYM)));
            pHigh = C;
         ELSE
            pHigh = max(pHigh, C);
         THEN
         p = prices(pLsum/nLP, name);
         IF(V) 
            nl; sp; sp;
            dot("LONG " + intstr(nLP) + " at: " + 
               priceformat1(p, name));
            (<< " on step " k intstr + . >>);
         THEN
      THEN;

      IF(KEY==SHORT && nSP < maxHold)
         OS(k, ""); // add to buy-sell record
         nSP++; // add short
         pSsum += C;
         IF(nSP==1) 
            pGoal = @(R[k] + (C - R[k])/(1+Tgoal(LIB.SYM)));
            pLow = C;
         ELSE
            pLow = min(pLow, C);
         THEN
         p = prices(pSsum/nSP, name);
         IF(V) 
            nl; sp; sp;
            dot("SHORT " + intstr(nSP) + " at: " + 
               priceformat1(p, name));
            (<< " on step " k intstr + . >>);
         THEN
      THEN;
   }

   function () = ztrade_init(LIB) {

   /* Load model arrays into the library of ZLIB(): */

   // Data arrays are stored into ZLIB by pit().

   /* Array .DELMO is nonzero on steps when delivery month changes: */
      c1 = main("data.delmo");
      ZLIB.DELMO = delta([DATA[*,c1] ; DATA[rows(DATA),c1]]);

   /* Initialize variables: */
      ztrade.nLP = 0;
      ztrade.nLtot = 0;
      ztrade.pLsum = 0;
      ztrade.pLONG = 0;

      ztrade.nSP = 0;
      ztrade.nStot = 0;
      ztrade.pSsum = 0;
      ztrade.pSHORT = 0;

      ztrade.nLgood = 0;
      ztrade.nLbad = 0;
      ztrade.nSgood = 0;
      ztrade.nSbad = 0;

      zrollover.never = true;
      zenter.never = true;
   }
--------------------- NOT IN USE */

/*--------------------------------------------------------------------*/

   IF(mday.SERVERS) 
   <<
      inline: SERVERS_RUN (qS --- )
      \  For this file running graphics commands on two SERVERS.
         [ 2 "NS" book ]
         dup (qS) sockets 1st pry remoterun 
         NS 2 =
         IF (qS) sockets 2nd pry remoterun
         ELSE drop
         THEN
      end

      inline: scut (hP s1 k --- hP)
         swap (k s1)
         dup sockets 1st pry remoteput sockets 2nd pry remoteput
         dup sockets 1st pry remoteput sockets 2nd pry remoteput
         "scut" SERVERS_RUN
      end

      inline: last (hA N --- hA) 
         (N) dup sockets 1st pry remoteput sockets 2nd pry remoteput
         "last" SERVERS_RUN
      end

      inline: splot (s1 k hP --- )
         drop
         swap (k s1)
         dup sockets 1st pry remoteput sockets 2nd pry remoteput
         dup sockets 1st pry remoteput sockets 2nd pry remoteput
         "other splot" SERVERS_RUN
      end
{
   These are other substitutions to override words just loaded by
   mday.v, and cause them instead to be run by this CLIENT to make
   the servers respond.

   When this client runs animations with word z, this substitute for
   client's word z causes the servers to run word z at the same time, 
   so their respective graphs stay in synch:
}  "'z'  SERVERS_RUN" "z"  macro 

   "'z0' SERVERS_RUN" "z0" macro

   "int$ ' zd' + SERVERS_RUN" "zd" macro

   inline: zl (n --- ) \ animate the last n steps
{     For use with servers.  Replaces zl from plot.v that has this
      stack diagram: (hData n --- hData)
      Here, Data is on the stack of the servers, so the stack simply
      holds n as shown above.

      The number of rows of Data on each server is the same.  Get the
      rows of Data from the first server:
}     "dup rows remotefd remoteput" sockets 1st pry remoterun1
      swap (rows n) less tic zd zw
   end

   "int$ ' zspan' + SERVERS_RUN" "zspan" macro

   >>
   THEN

   private halt

/*----------------------------------------------------------------------

   Appendix

   Notes, experimental and obsolete functions.

   Example of using centroid:
      x = [1:16]; 
      fx = LIB.fx[x];

      TMP = LIB.MODEL[*,LIB.Ps][*,x];
      wtPs = (<<
         TMP rows 1st
         DO TMP I reach bend fx centroid @ LOOP TMP rows listn
      >>);

      TMP = LIB.MODEL[*,LIB.Pl][*,x];
      wtPl = (<<
         TMP rows 1st
         DO TMP I reach bend fx centroid @ LOOP TMP rows listn
      >>);

   Example of test for crossing something:
      G2 = (W > Mx[*,nxqtr]);
      G1 = (W < Mx[*,nxqtr]);
      E1 = looking(W,
            fdiff(W, Mx[*,nxqtr], R)<0.005 ||
            (G1 && lag(G2, 1)) || (G2 && lag(G1, 1)));

----------------------------------------------------------------------*/
