/* {{{1 GNU General Public License

sofea - the Stack Operated Finite Element Analysis program
Copyright (C) 2004  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
#include <stdio.h>
#include <stk.h>
#include <sqlite3.h>
#include "fea_assemble.h"
#include "sql.h"

int force_vector() /* ( qModelDB qParnDB part_id --- hFg ) {{{1 */
/*
 * man entry:  force_vector {{{2
 * ( qModelDB qParnDB part_id --- hFg ) Return a global set vector of nodal forces and moments for this partition (or for the entire model if part_id is zero).  The terms are sequenced according to the g-set renumbered dof.  Use the index vector from apply_spc to partition this vector down to the analysis set.
 * category: FEA
 * related: fea_assemble, apply_spc
 * 2}}}
 */
{
int DEBUG = 0;

    char   *name = "_Fg";
    int     i, rc, partition_number, n_nodes, node;
    double *force_vec, Fmag, Mmag;
#define size_T 160
    char    T[size_T+1];
    callback_data sql_data;
    sqlite3 *dbh;
    char query[QUERY_STR_SIZE+1];

if (DEBUG)
gprintf("at top of force_vector\n");
    query[0] = '\0';

    if (!popint(&partition_number)) return 0;
    if (partition_number) {  /* requested only a portion of the total model */
        pushstr("DB_PARTN"); /* add to stack alias for partition db */
        if (!db_open2()) {   /* open model db & attach partition db */
            stkerr(" force_vector: ", "failed to open model and partn dbs");
            return 0;
        }
    } else {
        drop();             /* ignore whatever is in position of partn DB */
        if (!db_open()) {   /* open model db */
            stkerr(" force_vector: ", "failed to open model db");
            return 0;
        }
    }
    dbh = (sqlite3 *) tos->tex; /* ( dbh ) */

    if (partition_number) {
        snprintf(query, QUERY_STR_SIZE,
                 "select count(id) from DB_PARTN.P%02d_canonical_nid ",
                 partition_number);
        n_nodes = sql_scalar_i(dbh, query, "force_vector");

        snprintf(query, QUERY_STR_SIZE, 
                 "select R.new_id, L.Fmag, L.F1, L.F2, L.F3, "
                 "                 L.Mmag, L.M1, L.M2, L.M3 "
                 "from nodal_load L, "
                 "     DB_PARTN.P%02d_canonical_nid C, "
                 "     DB_PARTN.P%02d_renumbered_nid R "
                 "where L.nid = C.nid and R.orig_id = C.id "
                 "FIXME "
                 "           order by R.new_id; ",
                partition_number,
                partition_number);
    } else {  /* do the whole model */
        n_nodes = sql_scalar_i(dbh, "select count(seq_no) from node",
                                    "force_vector");
        snprintf(query, QUERY_STR_SIZE, 
                 "select nid, Fmag, F1, F2, F3, "
                             "Mmag, M1, M2, M3 "
                 "from nodal_load L order by nid;"
                 );
    }
if (DEBUG) gprintf("force_vector query= %s",query);
    sql_data.nRows     = 0;
    sql_data.row_index = 0;
    rc = sql_do_query(dbh, query, &sql_data);  
    /* .i[][0]    = node seq_no (canonical ID)
     * .x[][1]    = force  magnitude
     * .x[][2..4] = components of force  vector
     * .x[][5]    = moment magnitude
     * .x[][6..8] = components of moment vector
     */

if (DEBUG) gprintf("force_vector rc=%d\n", rc);
    if (rc != SQLITE_OK) {
        stkerr(" force_vector: ","failure querying for load");
        stkerr(" query is: ",query);
        return 0;
    }

    if (!matstk(n_nodes*DOF_PER_NODE, 1, name))     return 0;
    force_vec = tos->mat;
    for (i = 0; i < n_nodes*DOF_PER_NODE; i++) { 
        force_vec[i] = 0.0;
    }

if (DEBUG) gprintf("force_vector sql_data.nRows=%d\n", sql_data.nRows);
    for (i = 0; i < sql_data.nRows; i++) { 
        node = sql_data.i[i][0];

        Fmag = sql_data.x[i][1];
        force_vec[node*DOF_PER_NODE + 0] = Fmag * sql_data.x[i][2]; /* Fx */
        force_vec[node*DOF_PER_NODE + 1] = Fmag * sql_data.x[i][3]; /* Fy */
        force_vec[node*DOF_PER_NODE + 2] = Fmag * sql_data.x[i][4]; /* Fz */

        Mmag = sql_data.x[i][5];
        force_vec[node*DOF_PER_NODE + 3] = Mmag * sql_data.x[i][6]; /* Mx */
        force_vec[node*DOF_PER_NODE + 4] = Mmag * sql_data.x[i][7]; /* My */
        force_vec[node*DOF_PER_NODE + 5] = Mmag * sql_data.x[i][8]; /* Mz */
if (DEBUG) {
gprintf("force_vector ren_nid=%4d:  % 9.4e % 9.4e % 9.4e % 9.4e % 9.4e % 9.4e\n", 
node, 
force_vec[node*DOF_PER_NODE + 0],
force_vec[node*DOF_PER_NODE + 1],
force_vec[node*DOF_PER_NODE + 2],
force_vec[node*DOF_PER_NODE + 3],
force_vec[node*DOF_PER_NODE + 4],
force_vec[node*DOF_PER_NODE + 5]);
}
    }

    swap();     /* bring dbh to top so it can be closed */
    db_close();

    return 1;

} /* 1}}} */
