!{{{1 GNU General Public License

! Program Tops - a stack-based computing environment
! Copyright (C) 1999-2005  Dale R. Williamson

! Author: Dale R. Williamson <dale.williamson@prodigy.net>

! This program is free software; you can redistribute it and/or modify
! it under the terms of the GNU General Public License as published by
! the Free Software Foundation; either version 2 of the License, or
! (at your option) any later version.

! This program is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.

! You should have received a copy of the GNU General Public License
! along with this program; if not, write to the Free Software Founda-
! tion,
! Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
! 1}}} */

! /* wapp_f.f  October 2004

! Words for applications and projects

! This file contains Fortran subroutines to accompany functions in 
! wapp.c.
! The C prototypes for these functions are in header file wapp.h

      subroutine ramp(t,r0,r1,xd0,x0,g,h,cx,w2,xdd1,xd1,x1)
c
c         Subroutine to integrate a second order de from 0 to t given:
c            initial vel and disp xd0 and x0 at time 0
c            rhs force r0 at time 0 and r1 at time t
c
c         A closed-form solution to a ramp function defined by r0 and
c         r1 is assumed
c         Required constant matrices g and h are computed ahead of
c         time (see subroutine ramp0) assuming equation of motion
c
c                     xdd + (cx)*xd + (w2)*x = r0 + dr
c         where dr = r1 - r0
c
c         Values of acceleration, velocity and displacement at time t
c         are returned in xdd1, xd1 and x1

c         Use this version for constant time step t; use rampt for
c         variable time step

      implicit double precision(a-z)
      dimension g(2,2),h(2,2)
c
      dr = r1 - r0
      if(w2 .le. 0d0) then
c
c         rigid body equation (assumes if w2 (from K) is zero, so is
c         the term from B):
      x1   = x0 + xd0*t + r0*t*t/2.0d0 + dr*t*t/6.0d0
      xd1  = xd0 + r0*t + dr*t/2.d0
      xdd1 = r1
      return
      endif
c
c         elastic equation:
      xd1  = g(1,1)*xd0 + g(1,2)*x0 + h(1,1)*r0 + h(1,2)*dr
      x1   = g(2,1)*xd0 + g(2,2)*x0 + h(2,1)*r0 + h(2,2)*dr
      xdd1 = r1 - cx*xd1 - w2*x1
c
      return
      end
      subroutine ramp0(t,b,k,g,h,err)
c
c         Subroutine to compute constants for the closed-form
c         integration of a second order differential equation
c         subjected to a ramp function:
c
c                      xdd + b*xd + k*x = R + S*t
c
c         Reference: Williamson, D.R., "Dynamic Analysis of Structures
c         with Nonlinear Connections," February 1983, p. 16
c
c         Error flag err is set as follows:
c           err = 0, normal return
c           err = -1 if incoming k (squared freq) is negative
c           err = -2 if damping is outside underdamped assumption
c      
c         This is companion routine for ramp, to initialize
c         coefficients of integration for case of constant time
c         step t
c
      implicit double precision(a-z)
      integer err,i,j
      dimension g(2,2),h(2,2)
c
      err = 0
      if(k .le. 0d0) then
         if(k .lt. 0d0) then
            err = -1
         else
            do j=1,2
               do i=1,2
                  g(i,j) = 0d0
                  h(i,j) = 0d0
               enddo
            enddo
         endif
         return
      endif
c
      alpha = b/2d0
      ealt = dexp(-alpha*t)
      beta = k - alpha*alpha
      if(beta .lt. 0d0) then
      err = -2
      return
      endif
c
      beta = dsqrt(beta)
      sinbt = dsin(beta*t)
      sinbtb = sinbt/beta
      cosbt = dcos(beta*t)
      cosbtb = cosbt/beta
c
      g(1,1) = ealt*(-alpha*sinbtb + cosbt)
      g(1,2) = -ealt*(alpha*alpha/beta + beta)*sinbt
      g(2,1) = ealt*sinbtb
      g(2,2) = ealt*(alpha*sinbtb + cosbt)
c
      phi = datan2(beta,-alpha)
      sinphi = dsin(phi)
      cosphi = dcos(phi)
      gamma = datan2(-2d0*alpha*beta,(alpha*alpha - beta*beta))
      singam = dsin(gamma)
      cosgam = dcos(gamma)
      ksqrt = dsqrt(k)
      oneovk = 1d0/k
      oneovtk = oneovk/t
c
      h(1,1) = ealt*(-alpha*(cosphi*sinbtb - sinphi*cosbtb)
     *               + cosphi*cosbt + sinphi*sinbt)/ksqrt
      h(1,2) = (ealt*(-alpha*(cosgam*sinbtb - singam*cosbtb)
     *         + cosgam*cosbt + singam*sinbt) + 1d0)*oneovtk 
      h(2,1) = ealt*(cosphi*sinbtb - sinphi*cosbtb)/ksqrt + oneovk
      h(2,2) = ealt*(cosgam*sinbtb - singam*cosbtb)*oneovtk
     *         - b*oneovk*oneovtk + oneovk   
c
      return
      end
