/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* wapp.c  December 1999

Words for applications and projects

See file doc/design.doc, "Adding new native functions," for a descrip-
tion of the procedure for adding compiled words to the program.
 
These are functions for wapp.p, the file included by nativewords()
when the catalog is built at start up.

Prototypes for these functions are in header file wapp.h
*/

#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "mat.h"
#include "math1.h"
#include "mem.h"
#include "mmath.h"
#include "sys.h"
#include "tex.h"
#include "tag.h"
#include "wapp.h"

/*--------------------------------------------------------------------*/

/* Example of new word compilation in file word.h */

int alpha() /* prototype is in wapp.h, pattern "alpha" is in wapp.p */
{
   gprintf(" This is new function alpha\n");
   return 1;
}

int funtest() /* funtest (... --- ...) */
/* A word to test functions.  Put things here to try, then load up
   the stack and say funtest at the ready prompt. */
{
/*
   gprintf(" This is where to put some code to run and test"); nc();
*/
   return 1;
}

#ifdef FUNTEST
/*--------------------------------------------------------------------*/

/* Archive of previous funtest code. */

/* 6. Test function rmnull().
   int len,len1;
   char S[100];
   len1=tos->col;
   memcpy(S,tos->tex,len1);
   drop();
   len=rmnull(S,len1);
   strstk(len, "rm");
   memcpy(tos->tex,S,len);
   return 1;
*/

int funtest1() /* funtest (... --- ...) */
/* A word to test functions.  Put things here to try, then load up
   the stack and say funtest at the ready prompt. */
{
/* 5. Demonstrate reading a 4-byte int table from the library of high 
   level word USET.

   To run this demo, these lines must be sourced first.  Word export4
   turns matrix A into volume T.  Each *row* of T contains a *column*
   of A, but the values of A have been converted to 4-byte integers.
   Thus the bytes in T is equal to 4 times the rows in A.

      "USET" library
      1 10 items (hA) endian export4 (hT) "USET" "G" bank
      3 3  items (hA) endian export4 (hT) "USET" "S" bank
      7 2  items (hA) endian export4 (hT) "USET" "M" bank

   Running this shows the USET library:
      USET wholib

   Running these will execute this demo.  Function extract1(), which
   is the low-level equivalent of yank, gets the desired set from the 
   library of USET.
   
      "M" funtest
      "S" funtest
      "G" funtest
      "X" funtest \ causes error
*/
   char *set_name;
   int k=0,nrows,*rows=NULL;

   set_name=tos->tex;            /* quoted set name incoming on stack */
   extract1("USET",set_name);    /* putting VOL on stack from USET lib*/
   if(tos->typ != VOL) return 0; /* if error, won't have VOL on stk */

   rows=(int *)tos->tex;       /* set int pointer to VOL array */
   nrows=tos->col/sizeof(int); /* bytes of VOL = nrows*sizeof(int) */

/* Note that we can drop T from stack.  It will not be deallocated
   because T is stored in the catalog (in the library of USET). */
   drop(); /* T from stack */ 

/* This shows that the 4 byte integer array from USET lib is now in 
   array rows: */
   gprintf(" Read %d ints for set %s:\n",nrows,set_name);
   for(;k<nrows;k++) gprintf("    row %d\n",*(rows+k));

   drop(); /* drop qSet from stack */
   return 1;
}
/* 4. Looking at some functions in sys/dir.h.

   DIR *dp;
   dp=opendir(tos->tex);
   gprintf(" dp: %d",dp); nc();
   if(dp) closedir(dp);
   return 1;
*/
/* 3. Looking for memory leaks using memprobe().

   fprintf(stderr,"f: ");
   memprobe();
   memprobe();
   pushq2("TEST_STRING",11);
   memprobe();
   drop();
   memprobe();
   fprintf(stderr,":f\n");
   return 1;
*/
/*  2. Verifying action of system function ntohs().

   From /usr/include/asm/byteorder.h:
      extern __inline__ unsigned short int
      __ntohs(unsigned short int x)

   Use int and .hex to display result.  Shows that ntohs() swaps 
   the 2 bytes of a 2-byte number:

      [tops@gutter] ready # 65534 int .hex 65534 funtest int nl .hex
       FE FF 00 00 00 00 00 00
       FF FE 00 00 00 00 00 00
      [tops@gutter] ready # 

   int m,n;
   if(!popint(&n)) return 0;
   m=ntohs(n);
   return(pushint(m)); 
*/
/* 1. Testing pointer to function hAlarm().

   void (*hAlarm)(int);

   hAlarm=(void (*)(int)) handleAlarm1;
   hAlarm(SIGALRM);

   hAlarm=(void (*)()) handleAlarm2;
   hAlarm(999);
   return 1;
*/ 
/*--------------------------------------------------------------------*/
#endif 

/* Functions to demonstrate overloading */

int starA() /* *A (hA hB --- hC) */
{
   gprintf(" running *A");
   return(star());
}

int starB() /* *B (hA hB --- hC) */
{
   gprintf(" running *B");
   return(star());
}

int starC() /* *C (hA hB --- hC) */
{
   gprintf(" running *C");
   return(star());
}
   
/*--------------------------------------------------------------------*/

/* Functions for dynamic analysis */

int centroid() /* centroid (hW hA --- hC) */
/* For weights W (such as moment arms) applied to rows of A, compute 
   centroid of each column of A. */
{
   int cols,j=0,k,rows;
   double *A,*C,S,T,*W,*W0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" centroid: ",MATNOT2);
      return 0;
   }
   if(tos->row!=(tos-1)->row) {
      stkerr(" centroid: ",MATSNOTC);
      return 0;
   }
   A=tos->mat;
   W0=(tos-1)->mat;
   
   rows=tos->row;
   cols=tos->col;
   if(!matstk(cols,1,"_centroid")) return 0;
   C=tos->mat;

   for(;j<cols;j++) {
      S=0;
      T=0;
      W=W0;
      for(k=0;k<rows;k++) {
         S+=*A;
         T+=*A*(*W);
         A++;
         W++;
      }
      if(S) *C=T/S;
      else *C=0;
      C++;
   }
   return(lop() && lop());
}

int cgm() /* cgm (hf hA --- hx) */
/* Solve A*x=f for unknown x using the conjugate gradient method. */
{
   register double *A,*f,*x;
   double *r,*s;
   double a,c,*dnull,eps,eps2=1e-8,rm1;
   register int i,k=0;
   int cols,iterating,j=0,kmax,numiter=0,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" cgm: ",MATNOT2);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row) {
      stkerr(" cgm: ",ROWSNOT);
      return 0;
   }
   if(rows!=tos->col) {
      stkerr(" cgm: ",SQUNOT);
      return 0;
   }
   A=tos->mat;
   f=(tos-1)->mat;
   cols=(tos-1)->col;

   if(!matstk(rows,cols,"_x")) return 0;
   x=tos->mat;
   memset(x,0,rows*cols*sizeof(double));

   if((r=(double *)memget(rows,1))==NULL ||
      (s=(double *)memget(rows,1))==NULL ||
      (dnull=(double *)memget0(rows,1))==NULL) {
      stkerr(" cgm: ",MEMNOT);
      return 0;
   }
   /* for sym A, should converge in 2*rows; here, give it 5*rows */
   kmax=5*rows; /* max iterations */

   while(j<cols && k<kmax) {

      dot1(f,f,rows,&a);
      eps=a*eps2;

      for(i=0;i<rows;i++) *(r+i)=-*(f+i);
      memcpy(s,f,rows*sizeof(double));

      k=0;
      iterating=1;
      while(iterating && k<kmax) {
         numiter++;
         dot1(r,r,rows,&rm1);

         mpyad1(dnull,A,s,rows,rows,1,r);
         dot1(s,r,rows,&a);
         a=rm1/a;

         for(i=0;i<rows;i++) *(x+i)+=*(s+i)*a;

         mpyad1(dnull,A,x,rows,rows,1,r);
         for(i=0;i<rows;i++) *(r+i)-=*(f+i);

         dot1(r,r,rows,&c);
         if(c<eps) iterating=0;
         else {
            c=c/rm1;
            for(i=0;i<rows;i++) *(s+i)=*(s+i)*c - *(r+i);
         }
         k++;
      }
      if(TRACE) {
         gprintf(" cgm: column %d %d iterations",j,numiter); nc();
         numiter=0;
      }
      f+=rows;
      x+=rows;
      j++;
   }
   mallfree((void *)&r);
   mallfree((void *)&s);
   mallfree((void *)&dnull);

   if(k>=kmax) {
      stkerr(" cgm: ","convergence failure");
      drop();
      drop2();
      pushint(0);
      pushint(0);
      null();
      return 0;
   }
   return(lop() && lop());
}

int cmagphase() /* cmagphase (Ar Ai --- hMag hPhase) */
/* Complex matrices into magnitude-phase form.  Phase angle P has units
   of radians. */
{
   register double *Ai,*Ar,*M,*P;
   register int k=0;
   int cols,num=0,rows;
   
   if(is_complex(tos)) {
      cmplxdbl();
   }
   if(stkdepth()>1) {
      if(tos->typ==NUM && (tos-1)->typ==NUM) num=1;
      hand(); swap(); hand(); swap();
   }
   else {
      stkerr(" cmagphase: ",STKNOT);
      return 0;
   }
   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" cmagphase: ",STKNOT);
      return 0;
   }
   Ai=tos->mat;
   Ar=(tos-1)->mat;
   if((rows=tos->row)!=(tos-1)->row || (cols=tos->col)!=(tos-1)->col) {
      stkerr(" cmagphase: ",MATSNOTC);
      return 0;
   }
   if(!matstk(rows,cols,"_mag")) return 0;
   M=tos->mat;
   if(!matstk(rows,cols,"_phase")) return 0;
   P=tos->mat;

   for(;k<rows*cols;k++) {
      *M=sqrt(*Ar*(*Ar) + *Ai*(*Ai));
      *P=atan2(*Ai,*Ar);
      Ai++;
      Ar++;
      M++;
      P++;
   }
   if(num) {
      swap();
      pushd(*tos->mat);
      lop();
      swap();
      pushd(*tos->mat);
      lop();
   }
   return(
      rot() &&
      drop() &&
      rot() &&
      drop()
   );
}

int cmagphase1() /* -cmagphase (R P --- Ar Ai) */
/* Undo complex magnitude-phase, convert back to real-imaginary form
   Incoming P contains phase angles in radians.

   Equivalent to this high level word:
   inline: -cmagphase (R P --- Ar Ai) \ magnitude-phase into real-imag
      this cos other *by "_Ar" naming, rev sin *by "_Ai" naming
   end
*/
{
   register double *Ai,*Ar,*P,*R;
   register int k=0;
   int cols,num=0,rows;
   
   if(stkdepth()>1) {
      if(tos->typ==NUM && (tos-1)->typ==NUM) num=1;
      hand(); swap(); hand(); swap();
   }
   else {
      stkerr(" -cmagphase: ",STKNOT);
      return 0;
   }
   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" -cmagphase: ",STKNOT);
      return 0;
   }
   P=tos->mat;
   R=(tos-1)->mat;
   if((rows=tos->row)!=(tos-1)->row || (cols=tos->col)!=(tos-1)->col) {
      stkerr(" -cmagphase: ",MATSNOTC);
      return 0;
   }
   if(!matstk(rows,cols,"_Ar")) return 0;
   Ar=tos->mat;
   if(!matstk(rows,cols,"_Ai")) return 0;
   Ai=tos->mat;

   for(;k<rows*cols;k++) {
      *Ar=*R*cos(*P);
      *Ai=*R*sin(*P);
      R++;
      P++;
      Ar++;
      Ai++;
   }
   if(num) {
      swap();
      pushd(*tos->mat);
      lop();
      swap();
      pushd(*tos->mat);
      lop();
   }
   return(
      rot() &&
      drop() &&
      rot() &&
      drop()
   );
}

int detrend() /* detrend (hA hB0 hB1 --- hA1) */
/* Removing linear trend B from each column of N uniformly spaced
   points in A:
      A1(k,j) = A(k,j) - B0(j) - k*B1(j) 
   where k=1,N.

   Word trend provides the matrices B0 and B1. */
{
   double *A,*A1,*B0,*B1;
   int cols,j=0,k,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" detrend: ",MATNOT);
      return 0;
   }
   A=(tos-2)->mat;
   rows=(tos-2)->row;
   cols=(tos-2)->col;

   B0=(tos-1)->mat;
   B1=tos->mat;

   if(!matstk(rows,cols,"_A1")) return 0;
   A1=tos->mat;

   for(;j<cols;j++) {
      for(k=1;k<=rows;k++) {
         *A1=*A - (*B0) - k*(*B1);
         A++;
         A1++;
      }
      B0++;
      B1++;
   }
   return(lpush() && drop2() && drop() && lpull());
}

int _lamp() /* _lamp (hA ht hf --- hFr hFi) */
/* At frequencies f, compute the complex Fourier transform of time
   histories in A(t).  Each column in A is a time history, and each
   column in returned Fr and Fi is for the corresponding column of A.

   Adapted from original Fortran function lamp created by John Frye,
   using integration of a piece-wise linear ramp function between 
   time points A(t1) and A(t2). */
{
   double *A,*f,*f0,*Fi,*Fi0,*Fr,*Fr0,*t;
   int cols,nf,nt,rows;
   register int i,j,k=0;
   const double pi2=2*3.14159265358979323846;
   register double A1,A2,ct1,dt,Omeg,slp,st1,t1,tr,ti;

   hand();
   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" _lamp: ",MATNOT);
      return 0;
   }
   A=(tos-2)->mat;
   rows=(tos-2)->row;
   cols=(tos-2)->col;

   t=(tos-1)->mat;
   f0=(tos  )->mat;

   nt=MIN((tos-1)->row,(tos-2)->row);
   nf=tos->row;

   if(!matstk(nf,cols,"_Fr")) return 0;
   Fr0=tos->mat;

   if(!matstk(nf,cols,"_Fi")) return 0;
   Fi0=tos->mat;

   for(;k<cols;k++) { /* for each time history, A(t): */
      f=f0;
      Fr=Fr0;
      Fi=Fi0;

      for(i=0;i<nf;i++) { /* for each frequency, f(i): */
         *Fr=0;
         *Fi=0;
         Omeg=(*f)*(pi2);

         for(j=1;j<nt;j++) { /* integrating over all t(i): */
            t1=*(t+j-1);
            dt=*(t+j)-t1;

            A1=*(A+j-1);
            A2=*(A+j);

            slp=(A2-A1)/dt;
            st1=sin(Omeg*dt);
            ct1=cos(Omeg*dt);

            tr=A2*st1/Omeg-slp/(Omeg*Omeg)*(1-ct1);
            ti=-(A1-A2*ct1)/Omeg-slp/(Omeg*Omeg)*st1;

            st1=sin(Omeg*t1);
            ct1=cos(Omeg*t1);

            *Fr+=((tr*ct1)+(ti*st1));
            *Fi+=((ti*ct1)-(tr*st1));
         }
         Fr++;
         Fi++;
         f++;
      }
      A+=rows;
      Fr0+=nf;
      Fi0+=nf;
   }
   return(
      lpush() && lpush() &&
      drop() && drop2() &&
      lpull() && lpull()
   );
}

int logarea() /* logarea (hP hF --- hA) */
/* Logarithmic trapezoidal integration.  Returned A contains the 
   incremental areas between elements of P and F, discrete points
   of function P(F).  The total area is the sum of all elements
   in A. 

   Assumes F is in ascending order. */
{
   int cols,i,j=0,rows;
   double *A,*F,*P;
   double G;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" logarea: ",MATSNOT);
      return 0;
   }
   P=(tos-1)->mat;

   cols=(tos-1)->col;
   rows=(tos-1)->row;

   if(rows!=tos->row) {
      stkerr(" logarea: ",MATSNOTC);
      return 0;
   }
   if(!matstk(rows,cols,"_A")) return 0;
   A=tos->mat;

   for(;j<cols;j++) {
      *A=0;
      F=(tos-1)->mat;
      for(i=1;i<rows;i++) {
         A++;
         P++;
         F++;
         G=*P*(*F) - *(P-1)*(*(F-1));

         if(G) {
            if(*F>*(F-1))  
               *A=G/( log10(*P/(*(P-1))) / log10((*F)/(*(F-1))) + 1. );
             
            else *A=0;
         }
         else {
            if(*F>*(F-1)) *A=log(*(F)/(*(F-1)))*(*P)*(*F);
            else *A=0;
         }
      }
      P++;
      A++;
   }
   return(lop() && lop());
}

int moveave() /* ma (hA n --- hB) */
/* Simple moving average; B(i,j) is average of A(i-n+1,j) to A(i,j). */
{
   register int i,j=0;
   register double *A,*B,*B1,sum;
   int cols,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" ma: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" ma: ",MATNOT);
      return 0;
   }
   if(!popint(&n)) return 0;

   rows=tos->row;
   n=MIN(n,rows);

   cols=tos->col;
   if(!matstk(rows,cols,"_ma")) return 0;

   B=tos->mat;
   A=(tos-1)->mat;

   for(;j<cols;j++) {
      sum=0;
      for(i=0;i<n;i++) {
         *B=*A;
         sum+=*B;
         A++;
         B++;
      }
      if(n>1) {
         B1=B-1;
         if(n<rows) *B1=sum/n;
         for(;i<rows;i++) {
            *B=*B1+(*A-*(A-n))/n;
            A++;
            B++;
            B1++;
         }
      }
   }
   return(lop());
}

int mmax() /* mmax (hA n --- hB) */
/* Moving max; B(i,j) is max of A(i-n+1,j) to A(i,j). */
{
   register int i,j=0,k1,k2;
   register double *A,*B,X;
   int cols,kX,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" mmax: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" mmax: ",MATNOT);
      return 0;
   }
   if(!popint(&n)) return 0;

   cols=tos->col;
   rows=tos->row;

   n=MIN(n,rows);
   n=MAX(n,1);

   if(!matstk(rows,cols,"_mmax")) return 0;

   for(;j<cols;j++) {
      A=(tos-1)->mat+j*rows;
      B=tos->mat+j*rows;

      k2=1;
      kX=0;
      X=*B=*A;
      
      while(k2<n) {
         if(*(A+k2)>=X) {
            X=*(A+k2);
            kX=k2;
         }
         *(B+k2)=X;
         k2++;
      }
      for(;k2<rows;k2++) {
         if(*(A+k2)>=X) { /* newest is max */
            X=*(A+k2);
            kX=k2;
         }
         else {
            k1=k2-n+1;
            if(kX<k1) { /* max is gone; find new max */
               X=*(A+k1);
               kX=k1;
               for(i=k1+1;i<=k2;i++) {
                  if(*(A+i)>=X) {
                     X=*(A+i);
                     kX=i;
                  }
               }
            }
         }
         *(B+k2)=X;
      }
   }
   return(lop());
}

/* 
   It is amazing what a little rearranging can do.  Words mmax
   and mmin are 100 to 150 times faster than old versions mmaxX 
   and mminX.

      [tops@plunger] ready > ww
      mmax and mmaxX results match
      mmin and mminX results match
      mmax  (microsec):  22376
      mmaxX (microsec):  2504952
      mmin  (microsec):  24064
      mminX (microsec):  3720356

   Here are the phrases that produced the results above:

      seed0 seedset
      20000 1 random 1000 * integer "A" book

      1000 "N" book

      A N mmaxX "B1" book
      A N mmax  "B2" book
      B2 B1 - null? IF "match" ELSE "do not match" THEN
      " mmax and mmaxX results " swap + . nl

      A N mminX "C1" book
      A N mmin  "C2" book
      C2 C1 - null? IF "match" ELSE "do not match" THEN
      " mmin and mminX results " swap + . nl

      time push 10 1 DO A N mmax drop LOOP time pull less
      " mmax  (microsec): " . 1E6 * .i nl

      time push 10 1 DO A N mmaxX drop LOOP time pull less
      " mmaxX (microsec): " . 1E6 * .i nl

      time push 10 1 DO A N mmin drop LOOP time pull less
      " mmin  (microsec): " . 1E6 * .i nl

      time push 10 1 DO A N mminX drop LOOP time pull less
      " mminX (microsec): " . 1E6 * .i nl
*/

/* int mmaxX() */ /* mmax (hA n --- hB) */
/* Moving max; B(i,j) is max of A(i-n+1,j) to A(i,j). */
/*
{
   register int i,j=0,k1,k2;
   register double *A,*B,X;
   int cols,kX,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" mmax: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" mmax: ",MATNOT);
      return 0;
   }
   if(!popint(&n)) return 0;

   cols=tos->col;
   rows=tos->row;

   n=MIN(n,rows);
   n=MAX(n,1);

   if(!matstk(rows,cols,"_mmax")) return 0;

   for(;j<cols;j++) {
      A=(tos-1)->mat+j*rows;
      B=tos->mat+j*rows;

      k2=1;
      kX=0;
      X=*B=*A;
      
      while(k2<n) {
         if(*(A+k2)>X) {
            X=*(A+k2);
            kX=k2;
         }
         *(B+k2)=X;
         k2++;
      }
      for(;k2<rows;k2++) {
         k1=k2-n+1;
         if(kX>=k1 && *(A+k2)>X) {
            X=*(A+k2);
            kX=k2;
         }
         else {
            X=*(A+k1);
            kX=k1;
            for(i=k1+1;i<=k2;i++) {
               if(*(A+i)>X) {
                  X=*(A+i);
                  kX=i;
               }
            }
         }
         *(B+k2)=X;
      }
   }
   return(lop());
}
*/

int mmaxr() /* mmaxr (hA hB0 n --- hB) */
/* Moving max; B(i,j) is max of A(i-n+1,j) to A(i,j). 
   For use in real time, as vector A grows.  B0 is B from a previous 
   call to this function, or to function mmax(), and has number of rows
   equal to A at that time.  Now, additional rows have been added to A,
   and this function computes moving max for these new rows and appends
   them to B0 to create B. 

   Here is some high level code for testing this word, and verifying 
   that the incremental result agrees with batch function mmax():

      seed0 seedset 
      30001 "L" book L 1 random 1000 * integer "A" book
      200 "N" book

      A dup pile N mmax "B" book \ batch values from mmax

      A purged N mmaxr (hB0)                \ first set
      A dup pile swap (hA hB0) N mmaxr (hB) \ appended set
      "Br" book \ incremental values from mmaxr

      Br B - null? \ get true flag when valid
*/
{
   register int i,j=0,k;
   register double *A,*B,max;
   int cols,kmin,n,rows,rows0;

   if(tos->typ!=NUM) {
      stkerr(" mmax: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" mmax: ",MATNOT2);
      return 0;
   }
   if((tos-1)->col<=0) { /* B0 is purged; run mmax() */
      lop(); /* B0 off stack */
      return(mmax());
   }
   if(!popint(&n)) return 0;

   cols=tos->col;
   if((tos-1)->col != cols) {
      stkerr(" mmax: ","columns of A and B0 do not match");
      return 0;
   }
   rows=(tos-1)->row; /* rows of A and B */
   rows0=tos->row;    /* rows of B0 */

   if(rows0>rows) {
      stkerr(" mmax: ","rows of B0 cannot exceed rows of A");
      return 0;
   }
   else {
      if(rows0==rows) return(lop()); /* B = B0 */
   }
   n=MIN(n,rows);
   n=MAX(n,1);

   if(!matstk(rows-rows0,cols,"_mmax")) return 0;
   pile();

   kmin=MAX(0,rows0-n-1);

   for(;j<cols;j++) {
      A=(tos-1)->mat+rows*cols-1 - j*rows;
      B=tos->mat+rows*cols-1 - j*rows;
      k=rows-1;
      for(;k>kmin;k--) {
         max=*A;
         i=1;
         for(;i<n && i<=k;i++) max=MAX(max,*(A-i));
         if(k>=rows0) *B=max;
         A--;
         B--;
      }
      if(k==0) *B=*A;
   }
   return(lop());
}

int mmin() /* mmin (hA n --- hB) */
/* Moving min; B(i,j) is min of A(i-n+1,j) to A(i,j). */
{
   register int i,j=0,k1,k2;
   register double *A,*B,X;
   int cols,kX,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" mmin: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" mmin: ",MATNOT);
      return 0;
   }
   if(!popint(&n)) return 0;

   cols=tos->col;
   rows=tos->row;

   n=MIN(n,rows);
   n=MAX(n,1);

   if(!matstk(rows,cols,"_mmin")) return 0;

   for(;j<cols;j++) {
      A=(tos-1)->mat+j*rows;
      B=tos->mat+j*rows;

      k2=1;
      kX=0;
      X=*B=*A;
      
      while(k2<n) {
         if(*(A+k2)<=X) {
            X=*(A+k2);
            kX=k2;
         }
         *(B+k2)=X;
         k2++;
      }
      for(;k2<rows;k2++) {
         if(*(A+k2)<=X) { /* newest is min */
            X=*(A+k2);
            kX=k2;
         }
         else {
            k1=k2-n+1;
            if(kX<k1) { /* min is gone; find new min */
               X=*(A+k1);
               kX=k1;
               for(i=k1+1;i<=k2;i++) {
                  if(*(A+i)<=X) {
                     X=*(A+i);
                     kX=i;
                  }
               }
            }
         }
         *(B+k2)=X;
      }
   }
   return(lop());
}

/* int mminX() */ /* mmin (hA n --- hB) */
/* Moving min; B(i,j) is min of A(i-n+1,j) to A(i,j). */
/*
{
   register int i,j=0,k1,k2;
   register double *A,*B,X;
   int cols,kX,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" mmin: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" mmin: ",MATNOT);
      return 0;
   }
   if(!popint(&n)) return 0;

   cols=tos->col;
   rows=tos->row;

   n=MIN(n,rows);
   n=MAX(n,1);

   if(!matstk(rows,cols,"_mmin")) return 0;

   for(;j<cols;j++) {
      A=(tos-1)->mat+j*rows;
      B=tos->mat+j*rows;

      k2=1;
      kX=0;
      X=*B=*A;
      
      while(k2<n) {
         if(*(A+k2)<X) {
            X=*(A+k2);
            kX=k2;
         }
         *(B+k2)=X;
         k2++;
      }
      for(;k2<rows;k2++) {
         k1=k2-n+1;
         if(kX>=k1 && *(A+k2)<X) {
            X=*(A+k2);
            kX=k2;
         }
         else {
            X=*(A+k1);
            kX=k1;
            for(i=k1+1;i<=k2;i++) {
               if(*(A+i)<X) {
                  X=*(A+i);
                  kX=i;
               }
            }
         }
         *(B+k2)=X;
      }
   }
   return(lop());
}
*/

int mminr() /* mminr (hA hB0 n --- hB) */
/* Moving min; B(i,j) is min of A(i-n+1,j) to A(i,j). 
   For use in real time, as vector A grows.  B0 is B from a previous 
   call to this function, or to function mmin(), and has number of rows
   equal to A at that time.  Now, additional rows have been added to A,
   and this function computes moving min for these new rows and appends
   them to B0 to create B. 

   Here is some high level code for testing this word, and verifying 
   that the incremental result agrees with batch function mmin():

      seed0 seedset 
      30001 "L" book L 1 random 1000 * integer "A" book
      200 "N" book

      A dup pile N mmin "B" book \ batch values from mmin

      A purged N mminr (hB0)                \ first set
      A dup pile swap (hA hB0) N mminr (hB) \ appended set
      "Br" book \ incremental values from mminr

      Br B - null? \ get true flag when valid
*/
{
   register int i,j=0,k;
   register double *A,*B,min;
   int cols,kmin,n,rows,rows0;

   if(tos->typ!=NUM) {
      stkerr(" mmin: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" mmin: ",MATNOT2);
      return 0;
   }
   if((tos-1)->col<=0) { /* B0 is purged; run mmin() */
      lop(); /* B0 off stack */
      return(mmin());
   }
   if(!popint(&n)) return 0;

   cols=tos->col;
   if((tos-1)->col != cols) {
      stkerr(" mmin: ","columns of A and B0 do not match");
      return 0;
   }
   rows=(tos-1)->row; /* rows of A and B */
   rows0=tos->row;    /* rows of B0 */

   if(rows0>rows) {
      stkerr(" mmin: ","rows of B0 cannot exceed rows of A");
      return 0;
   }
   else {
      if(rows0==rows) return(lop()); /* B = B0 */
   }
   n=MIN(n,rows);
   n=MAX(n,1);

   if(!matstk(rows-rows0,cols,"_mmin")) return 0;
   pile();

   kmin=MAX(0,rows0-n-1);

   for(;j<cols;j++) {
      A=(tos-1)->mat+rows*cols-1 - j*rows;
      B=tos->mat+rows*cols-1 - j*rows;
      k=rows-1;
      for(;k>kmin;k--) {
         min=*A;
         i=1;
         for(;i<n && i<=k;i++) min=MIN(min,*(A-i));
         if(k>=rows0) *B=min;
         A--;
         B--;
      }
      if(k==0) *B=*A;
   }
   return(lop());
}

int movetot() /* mtot (hA n --- hB) */
/* Moving total; B(i,j) is total of A(i-n+1,j) to A(i,j). */
{
   register int i,j=0;
   register double *A,*B,sum;
   int cols,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" mtot: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" mtot: ",MATNOT);
      return 0;
   }
   if(!popint(&n)) return 0;

   rows=tos->row;
   n=MIN(n,rows);

   cols=tos->col;
   if(!matstk(rows,cols,"_mtot")) return 0;

   B=tos->mat;
   A=(tos-1)->mat;

   for(;j<cols;j++) {
      sum=0;
      for(i=0;i<n;i++) {
         *(B+locvec(j,rows)+i)=*(A+locvec(j,rows)+i);
         sum+=*(B+locvec(j,rows)+i);
      }
      if(n<rows) *(B+locvec(j,rows)+i-1)=sum;
      for(;i<rows;i++) {
         *(B+locvec(j,rows)+i)=*(B+locvec(j,rows)+i-1)
         +(*(A+locvec(j,rows)+i)-*(A+locvec(j,rows)+i-n));
      }
   }
   return(lop());
}

int op2mat1() /* op2mat1 (hFile hMap rows cols type endian --- hA) */
/* COMPLEX TYPES NOT CHECKED OUT. */

/* Reading Nastran Output2 style matrix, rows-by-cols, from File.

   Map has been prepared by word op2mat in nas.v, from which the call
   to this function has originated.

   The four columns of Map contain:
      0 - Fortran record number, 1-based, suitable for word fortseek
      1 - record size, bytes
      2 - value of the first four bytes of the record interpreted as
          an integer; in this case, the integer is the starting row
          for a column of data to be read
      3 - the number of file bytes offset to the record, a 0-based
          offset suitable for fseek()

   The values for type are: 1=SP, 2=DP, 3=SP complex, 4=DP complex.
  
   Informal timings for the following matrix:
      PHIU1, 2064-by-1230, 0.8081 density:
         - No endian conversion: 11 sec
         - With endian conversion: 16 sec
         - High level op2map version in Appendix of file.v, that
           prompted writing this low-level version: 105 sec */
{
   int byord,cols,endi,i=0,mrows,nw=1,r0,rows,size=8,type;
   double *A,*Ai,*hFile,*recsize,*recstrt,*rowstrt;
   char *T=NULL;
   FILE *fp;
   long bytes;
   size_t items=1,red;

   if(!popint(&endi)) return 0;
   byord=byteorder(NULL);
 
   if(!popint(&type)) return 0;

/* import() is unnecessary if same endian and DP type: */
   if(byord==endi && (type==2 || type==4)) endi=0;

   if(!popint(&cols)) return 0;
   if(!popint(&rows)) return 0;

/* Double the rows if complex: */
   if(type==3 || type==4) rows+=rows;

/* If SP, double the bytes transferred after import(): */
   if(type==1 || type==3) {
      nw=2;
      size=-4; /* changing flag for import() */
   }
   if((tos-1)->typ!=MAT ||
      !(fp=filehand((tos-1)->mat,(tos-1)->row*(tos-1)->col))) {
      stkerr(" op2mat1: ",FILHNDNOT);
      return 0;
   }
   hFile=(tos-1)->mat;

   mrows=tos->row;
   recsize=(tos->mat)+locvec(1,mrows); /* column 1 of Map */
   rowstrt=(tos->mat)+locvec(2,mrows); /* column 2 of Map */
   recstrt=(tos->mat)+locvec(3,mrows); /* column 3 of Map */
   
   if(!matstk(rows,cols,"_op2mat")) return 0;
   A=tos->mat;
   memset(A,0,rows*cols*sizeof(double));
   if(type==3 || type==4) set_complex(tos);

   if(endi) { /* Using this branch if endian conversion is done. */

      if(!volstk(1,rows*sizeof(double),"_op2char")) return 0;
      T=tos->tex;

      for(;i<mrows;i++) {
         Ai=A;
         if((r0=(int)*(rowstrt+i))>0) { /* r0=1-based start row */
            if((bytes=(long)(*(recsize+i))-sizeof(int))>0) {

               fseek(fp,(long)(*(recstrt+i)+sizeof(int)),SEEK_SET);

               red=fread(T,(size_t)bytes,items,fp); /* read bytes */
               tos->col=bytes; /* 4- or 8-byte data in 1-row VOL */
               dup1s(); /* want T on the stack after import() */

               import(size,endi); /* on return, 8-byte col is on stk */

               Ai+=r0-1; /* bytes go to this 0-based starting row */
               memcpy(Ai,tos->mat,bytes*nw); /* 8-byte col into A */
               drop(); /* 8-byte col off stk */
            }
         }
         else A+=rows;
      }
      drop() ; /* T off stk */
   }
   else {
      for(;i<mrows;i++) {
         Ai=A;
         if((r0=(int)*(rowstrt+i))>0) { /* r0=1-based start row */
            if((bytes=(long)(*(recsize+i))-sizeof(int))>0) {

               fseek(fp,(long)(*(recstrt+i)+sizeof(int)),SEEK_SET);

               Ai+=r0-1; /* bytes go to this 0-based starting row */

               red=fread(Ai,(size_t)bytes,items,fp); /* read bytes */
            }
         }
         else A+=rows;
      }
   }
   fupdate(hFile,fp); /* keeping File structure up-to-date */
   return(
      lop() && /* Map off stk */
      lop()    /* File off stk */
   );
}

int stardircos9() /* *dircos9 (hCba hXa --- hXb) */
/* Rotational transformation.
   Compute Xb=Cba*Xa, where each 9-element column of Cba holds a 
   3-by-3 matrix stacked by columns, and each column of Xa is a 
   3-by-1 matrix. */
{
   register double *Cba,*Xa,*Xb;
   register int cols,j=0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" *dircos9: ",MATNOT2);
      return 0;
   }
   if(tos->row!=3 || (tos-1)->row!=9) {
      stkerr(" *dircos9: ","expecting matrices of 9 and 3 rows");
      return 0;
   }
   if((cols=tos->col)!=(tos-1)->col) {
      stkerr(" *dircos9: ",COLSNOT);
      return 0;
   }
   if(!matstk(3,cols,"_*dircos9")) return 0;

   Xb=tos->mat;
   Xa=(tos-1)->mat;
   Cba=(tos-2)->mat;

   for(;j<cols;j++) {
      *(Xb  )=*(Cba  )*(*(Xa))+*(Cba+3)*(*(Xa+1))+*(Cba+6)*(*(Xa+2));
      *(Xb+1)=*(Cba+1)*(*(Xa))+*(Cba+4)*(*(Xa+1))+*(Cba+7)*(*(Xa+2));
      *(Xb+2)=*(Cba+2)*(*(Xa))+*(Cba+5)*(*(Xa+1))+*(Cba+8)*(*(Xa+2));
      Cba+=9;
      Xa+=3;
      Xb+=3;
   }
   return(lop() && lop());
}

int steady() /* steady (hZ hW hQ hF --- hqr hqi) */
/* Steady-state frequency response solution of uncoupled equations.  

   Solving 2nd order differential equations of the form:

      qdd + (2*Z*W)*qd + (W*W)*q = Q(F)

   for steady-state response to harmonic input Q(F).

   Adapted from the corresponding fortran function of program express.

      function freqresp() ! steady state frequency response
c
c     Stack description:
c            steady (hZ hW hQ hF --- hqr hqi)
c      where
c          hZ is nW vector of critical damping factors
c          hW is nW vector of undamped natural frequencies, rad/sec
c          hQ is nW-by-nF matrix of generalized force amplitudes (real)
c          hF is nF vector of frequencies, rad/sec, that correspond to
c             each column of Q
c
c          hqr is nW-by-nF matrix of response real part
c          hqi is nW-by-nF matrix of response imaginary part
c
c     It is ok if n-by-1 vectors are stored as 1-by-n. */
{
   double *F,*Q,*qi,*qr,*W0,*W,*Z0,*Z;
   register double A,D,Fj,P,R;
   int j=0,nF,nQrows,nW,nZ;
   register int i=0,k=0;
  
   if(stkdepth()>3) {
      i=0;
      while(i<4) { /* making any NUMs into 1-by-1 MATs */
         pushint(3); pushq2("roll",4); xmain(0); /* doing 3 roll */
         hand();
         i++;
      }
   }
   else {
      stkerr(" steady: ",STKNOT);
      return 0;
   }
   F=tos->mat;
   nF=(tos->row)*(tos->col); /* tos->col should be 1 */

   if(is_complex(tos-1)) {
      stkerr(" steady: ","generalized force matrix Q must be real");
      return 0;
   }
   Q=(tos-1)->mat;
   nQrows=(tos-1)->row;

   Z0=(tos-3)->mat;
   nZ=((tos-3)->row)*((tos-2)->col);

   W0=(tos-2)->mat;
   nW=((tos-2)->row)*((tos-2)->col);

   if(nW!=nQrows) {
      stkerr(" steady: ", \
         "rows of matrix Q must match length of vector W");
      return 0;
   }
   if(nF!=(tos-1)->col) { /* nF len and cols of Q must agree */
      stkerr(" steady: ", \
         "columns of matrix Q must match length of vector F");
      return 0;
   }
   if(nZ!=nW) {
      stkerr(" steady: ", \
         "vector Z length and vector W length do not agree");
      return 0;
   }
   if(!matstk(nW,nF,"_qr")) return 0;
   qr=tos->mat;
   if(!matstk(nW,nF,"_qi")) return 0;
   qi=tos->mat;

   for(;j<nF;j++) { /* looping over nF frequencies */
      W=W0;
      Z=Z0;
      Fj=*(F+j);
      for(i=0;i<nW;i++) { /* looping over nW degrees-of-freedom */
         if(*W) {
            R=Fj/(*W);
            D=(1-R*R);
            R*=2*(*Z);
            A=1/(sqrt(D*D+R*R));
            A*=(*(Q+k))/(*W*(*W));
            P=atan2(R,D); /* phase ang P from Q to q; q always lags Q */
            *(qr+k)=A*cos(P);  /* component along real axis */
            *(qi+k)=-A*sin(P); /* component along imaginary axis */
         }
         else { /* rigid body motion: */
            *(qr+k)=-(*(Q+k))/(Fj*Fj);
            *(qi+k)=0;
         }
         W++;
         Z++;
         k++;
      }
   }
   return( /* (hZ hW hQ hF hqr hqi) dropping F, Q, W, and Z */
      lpush() &&
      lpush() &&
      drop2() &&
      drop2() &&
      lpull() &&
      lpull()
   );
}

int tr() /* tr (hSig n --- hTr) */
/* The trace Tr at step k depends upon signal Sig at step k and the 
   trace at the previous step, in the recursive equation:

      Tr(k) = Tr(k-1) + [Sig(k) - Tr(k-1)]/(n+1)

   To get the recursion started, an initial value is assumed:

      Tr(0)=Sig(0)

   The equation for Tr(k) can be written into a "difference" equation
   form with decay constant A and signal forcing coefficient B:

      Tr(k) = A*Tr(k-1) + B*S(k)

   where 

      A = 1 - 1/(1+n) = n/(1+n)
      B = 1/(1+n)

   Term A*Tr(k-1) represents an exponential decay, and Tr may be called
   an exponential trace. */
{
   register double A,B,*Sig,*Tr;
   register int i=1,j=0;
   int cols,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" tr: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" tr: ",MATNOT);
      return 0;
   }
   popint(&n);

   rows=tos->row;
   cols=tos->col;
   Sig=tos->mat;

   if(!matstk(rows,cols,"_Tr")) return 0;
   Tr=tos->mat;

   B=1.0/(n+1);
   A=n*B;

   for(;j<cols;j++) {
      *Tr=*Sig;
      Tr++;
      Sig++;
      for(i=1;i<rows;i++) {
         *Tr=*(Tr-1)*A + *Sig*B;
         Tr++;
         Sig++;
      }
   }
   return(lop());
}

int tr1() /* tr1 (hSig Tr0 n --- hTr) */
/* The trace Tr at step k depends upon signal Sig at step k and the 
   trace at the previous step, in the recursive equation:

      Tr(k) = Tr(k-1) + [Sig(k) - Tr(k-1)]/(n+1)

   This word computes Tr for the last r rows in Sig, so returned Tr
   has only r rows.

   The equation for Tr(k) can be written into a "difference" equation
   form with decay constant A and signal forcing coefficient B:

      Tr(k) = A*Tr(k-1) + B*S(k)

   where 

      A = 1 - 1/(1+n) = n/(1+n)
      B = 1/(1+n)

   Incoming Tr0 is the initial value for Tr needed to start the recur-
   sion, when k=0.  In real time processing, Tr0 is the last Tr value
   from the last set of data processed, and Sig is a new set of data:

      Tr(-1) = Tr0

   so the first value in returned Tr is:

      Tr(0) = A*Tr0 + B*S(0)

   Note that Tr0 is a row vector if Sig contains more than one column.

   Term A*Tr(k-1) represents an exponential decay, and Tr may be called
   an exponential trace. 

   This test shows usage and verifies matching results with tr() batch
   processing:

      if(missing("sine")) source("math.v");
   
      (S, t) = sine(<< 1 10 2pi * 0 0.001 100 >>); // sine curve
      L = uniform(.01, 100); // straight line
      A = [S, L];

      n = 20;
      trA = tr(A, n); // batch 

      plot([A, trA], t);
 
      B = A[rows(A)/2:rows(A)];
      trB = tr1(B, trA[rows(A)/2-1], n); // second half using tr1

      N = trA[rows(A)/2:rows(A)] - trB; // compare tr1 with tr batch

      if(null?(N)) dot("OK, N is null"); // expect exactly zero
      else dot(" ERROR."); */
{
   register double A,B,*Sig,*Tr0,*Tr;
   register int i=1,j=0;
   int cols,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" tr1: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" tr1: ",MATNOT2);
      return 0;
   }
   popint(&n);
   hand();

   if(tos->col!=(tos-1)->col) {
      stkerr(" tr1: ",MATSNOTC);
      return 0;
   }
   cols=tos->col;
   rows=(tos-1)->row;

   Tr0=tos->mat;
   Sig=(tos-1)->mat;

   if(!matstk(rows,cols,"_Tr")) return 0;
   Tr=tos->mat;

   B=1.0/(n+1);
   A=n*B;

   for(;j<cols;j++) {
      *Tr=*Tr0*A + *Sig*B;
      Tr0++;
      Tr++;
      Sig++;
      for(i=1;i<rows;i++) {
         *Tr=*(Tr-1)*A + *Sig*B;
         Tr++;
         Sig++;
      }
   }
   return(lop() && lop());
}

int trend() /* trend (hA --- hB0 hB1) */
/* Linear trend for data in A uniformly spaced.

   This Least-squares solution is subject to illconditioning, so double 
   precision is important.

   Each column in A is a data record of N uniformly spaced points.

   For A(k,j), k=1,N, the one-row trend is:
      trend(k,j) = B0(j) + k*B1(j)

   Size of matrices B0 and B1 is one row by the number of columns
   in A.

   Reference:

      Bendat, J. S., and Piersol, A. G., "Random Data Analysis and
      Measurement Procedures," 3rd edition, 2000, J. W. Wiley & Sons,
      Inc., page 397, equation 11.16. */
{
   double *A,*B0,*B1,sumA,sumiA;
   double drows,N1,N2,N3,N4;
   int cols,i,j=0,rows;

   if(tos->typ!=MAT) {
      stkerr(" trend: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   rows=tos->row;
   cols=tos->col;

   drows=rows; /* needs to be double or N4 calc won't work */
   N1=(2*drows+1)*2;
   N2=drows*(drows-1);
   N3=(drows+1)*6;
   N4=drows*(drows-1)*(drows+1); /* very large when rows is > 50000 */

   if(!matstk(1,cols,"_B0")) return 0;
   B0=tos->mat;

   if(!matstk(1,cols,"_B1")) return 0;
   B1=tos->mat;

   for(;j<cols;j++) {
      sumA=0;
      sumiA=0;
      for(i=1;i<=rows;i++) {
         sumA+=*A;
         sumiA+=i*(*A);
         A++;
      }
      *B0=(sumA*N1 - 6*(sumiA))/N2;
      *B1=(sumiA*12 - sumA*N3)/N4;
      B0++;
      B1++;
   }
   return(rot() && drop());
}

/*--------------------------------------------------------------------*/

/* Back-propagation network.
   Functions for netbp.v */

enum OutCols {value,grad,utype,decay,bias,dbias,input,error};

int logistic() /* logistic (hH nBeta --- hFexp) */
/* For values in array H, compute the logistic function given by:

      Fexp(H) = 1/[1 + exp(-2*Beta*H)]
*/
{
   double Beta,*Fexp,*H;
   unsigned int len,k=0;

   if(tos->typ!=NUM) {
      stkerr(" logistic: ",NUMNOT);
      return 0;
   }

   if((tos-1)->typ!=MAT && (tos-1)->typ!=NUM) {
      stkerr(" logistic: ",NUMORMATNOT);
      return 0;
   }
   popd(&Beta);
   Beta*=-2.0;

   hand();
   H=tos->mat;

   if(!matstk(tos->row,tos->col,"_Fexp")) return 0;
   Fexp=tos->mat;
   len=tos->row*tos->col;

   for(;k<len;k++) {
      *(Fexp)=1.0/(1.0 + exp(Beta*(*H)));
      Fexp++;
      H++;
   }
   return(lop());
}
   
int net_activate() /* (hIn hOut --- ) */
/* Compute the output, Out, from activation of units in a layer.  

   Matrix Out is a cataloged item, so it is not deallocated when its
   handle is dropped from the stack.  In general, matrix In is not
   cataloged, so its handle is kept on the stack until it is no longer
   needed. 

   In addition to receiving computed outputs, matrix Out contains pro-
   perties of the units needed to compute their outputs.  The columns 
   of Out are defined by enum OutCols. */
{
   double *Grad,*Decay,*Bias,*Utype;
   double A,*In,*Out;

   int i=0,units;

   In=(tos-1)->mat;
   units=tos->row;

   Out=tos->mat;
   Grad=Out+locvec(grad,units);
   Decay=Out+locvec(decay,units);
   Bias=Out+locvec(bias,units);
   Utype=Out+locvec(utype,units);
   drop();

   for(;i<units;i++) { 

      if(*Utype<2) { /* exp sigmoid function, type 0 or 1: */

         /* for 0 to +1 sigmoid range: */
         A=-(*Decay)*(*In-*Bias);           /* -a*(In-b) */
         *Out=1./(1+exp(A));                /* 1/[1 + e^(-a*(In-b))] */
         *Grad=(*Decay)*(1. - *Out)*(*Out); /* a*[1 - f]*f */

         if(*(Utype)) { /* adjusting for -1 to +1 sigmoid range: */
            *(Out)*=2; *(Out)-=1;           /* 2*f - 1 */
            *(Grad)*=2;                     /* 2*f' */
         }
      }
      else { /* linear function, type 2: */
         *Out=(*In-*Bias);
         *Grad=1.;
      }
      In++;
      Out++;
      Grad++;
      Decay++;
      Bias++;
      Utype++;
   }
   return(drop()); /* dropping In from stack */
}

int net_step() /* stepping (hGoal hIn qNet --- ) */
/* Stepping the back-propagation network called Net.  If Goal is purged,
   then just forward pass is done.

   This function is intended to work inside a loop, and so it does not 
   check if Net is a valid network word, or even if qNet is type STR.  
   Word 'net?' can be used to do this ahead of time, outside the loop.

   Uses extract1() to fetch items from the local library of Net. 

   Uses exe1() to fire ptrs that are elements in vectors of ptrs; the
   vectors of ptrs are fetched from the local library of Net.

   In general, after a matrix address is obtained from a MAT that is
   obtained from a ptr list, the ptr list and the MAT itself can be 
   dropped from the stack since they are both in the catalog (in this 
   case, in Net's local library) and therefore will not be deallocated.

   Example usage of vectors of ptrs.  

      Matrix handle hOut is extracted from the local library of word 
      Net using function extract1() (which is a speedy version of
      extract(), the native function for word extract):

         extract1(Net,"Out");
         hOut=tos->mat;
         drop(); // will not deallocate because Out is in lib of Net //

      If hOut is a vector of ptrs, exe1() will fire one of its elements.
      The following puts onto the stack the handle to the matrix of out-
      put units for the 3rd layer, by firing the ptr to the matrix at 
      the 3rd element of hOut:

         exe1(*(hOut+2)); 

   How to develop and debug a function like this one, which really is
   an application:

      A function like this one is developed step by step, so get ready 
      to do cycles of recompile-run-edit.

      Review notes in doc/design.doc: "Adding new native functions"
      and "Easing the pain of debugging."

      Make a simple demo case that can be run interactively.  File
      usr/netbp.v, a companion to sys/netbp.v, has example cases.

      Run the recompiled program interactively from the ready prompt.  

      This function runs straight through from beginning to end, so
      simply insert "return 1;" at the spot of interest, recompile,
      run the demo from the ready prompt and review results on the
      stack when return occurs.
      
      Make changes, perhaps insert gprintf() or fprintf() lines to 
      display function variables, move "return 1;" and repeat the 
      cycle. */ 
{
/* DEBUG=1 compiles the writing of unit inputs and unit errors to 
   columns of Out, which are displayed by word netprops. */
   #define DEBUG 1

   char Net[33];

   int i,j,k,layers,punits,train=0,units;

   double A,Alpha,Eta;
   double *Out,*W;
   double *hdW,*hOut,*hW; 
   double *Atmp,*Bias,*dW,*dBias,*E,*Ek,*Grad,*In,*NetGoal,*NetIn;

   #if DEBUG
      double *Esav,*Insav;
   #endif

   strncpy(Net,tos->tex,32);
   drop();

/* Setting up network input, and network target output. */

   if(tos->typ!=MAT) {
      stkerr(" stepping: ",MATNOT);
      return 0;
   }
   extract1(Net,"NetIn");
   NetIn=tos->mat;
   units=tos->row;

/* Storing incoming stack item In into local library NetIn (no bounds 
   checking): */
   memcpy(NetIn,(tos-1)->mat,sizeof(double)*units);
   drop2();

   if(tos->typ!=MAT) {
      stkerr(" stepping: ",MATNOT);
      return 0;
   }
   if(tos->row) { /* If hGoal is not purged, set up for training: */
      extract1(Net,"NetGoal");
      NetGoal=tos->mat;
      punits=tos->row;
      /* Storing incoming stack item Goal into local library NetGoal 
         (no bounds checking): */
      memcpy(NetGoal,(tos-1)->mat,sizeof(double)*punits);
      drop2();
      train=1;
   }
   else drop();

/* Fetching vectors of ptrs from local library of Net. */

   extract1(Net,"Out");
   hOut=tos->mat;
   layers=tos->row;
   drop();

   extract1(Net,"W");
   hW=tos->mat;
   drop();

/* Forward pass. */

   /* In this implementation, the input layer (1st layer) applies 
      activation (weights and biases in an activation function) to 
      the network input.  It is subject to training, and does not 
      simply pass the network input to the 2nd layer. */

/* Input to 1st layer is the weighted network input: */
   if(!matstk(units,1,"_In")) return 0;
   In=tos->mat;

   exe1(*hW); /* hW(input_layer) weights */
   W=tos->mat;
   drop();

   for(i=0;i<units;i++) { /* In(i)=W(i)*NetIn(i) */
      *(In+i)=(*W)*(*NetIn); /* weighted input */
      W++;
      NetIn++;
   }

/* Output from the 1st layer comes from driving the weighted inputs 
   through the activation function: */
   exe1(*hOut); /* hOut(input_layer) */

   #if DEBUG
      Insav=(tos->mat)+locvec(input,units);
      memcpy(Insav,In,sizeof(double)*units);
   #endif

   net_activate(); /* output from the units of the input layer */

/* Forward pass through the 2nd and following layers: */
   for(k=1;k<layers;k++) {

   /* Input to layer k is the weight matrix times the output from 
      the previous layer: In(k)=W(k)*Out(k-1) */

      exe1(*(hW+k));     /* hW(k) on stk */
      exe1(*(hOut+k-1)); /* hOut(k-1) on stk */
      tos->col=1;        /* 1st col of Out holds values for mpy */

      mpy();             /* hIn(k)=W(k)*Out(k-1) on stk */

      #if DEBUG
         exe1(*(hOut+k)); 
         Insav=(tos->mat)+locvec(input,tos->row); 
         drop();
         memcpy(Insav,tos->mat,sizeof(double)*tos->row);
      #endif

      exe1(*(hOut+k));   /* hOut(k) on stk */

   /* Output from layer k comes from driving the inputs to layer k
      through the activation function: */
      net_activate();
   }
   if(!train) return 1;

/* Back-propagation. */

/* Fetching vectors of ptrs for training, from local library of Net. */
   extract1(Net,"dW");
   hdW=tos->mat;
   drop();

   extract1(Net,"NetGoal");
   NetGoal=tos->mat;
   drop();

/* Fetching learning parameters from the local library of Net. */
   extract1(Net,"Alpha");
   popd(&Alpha);

   extract1(Net,"Eta");
   popd(&Eta);

   exe1(*(hOut+layers-1)); /* hOut(output) */
   Out=tos->mat;
   units=tos->row;
   drop();

   Grad=Out+locvec(grad,units);
   Bias=Out+locvec(bias,units);
   dBias=Out+locvec(dbias,units);

   #if DEBUG
      Esav=Out+locvec(error,units); 
   #endif

   matstk(units,1,"_E");
   E=tos->mat;

   /* Errors in the output layer: 
         E(output) = Grad(output)*(NetGoal - Out(output)) */

   for(i=0;i<units;i++) {
      *(E+i)=*Grad*(*NetGoal - *Out);
      Grad++;
      NetGoal++;
      Out++;
   }

   #if DEBUG
      memcpy(Esav,E,sizeof(double)*units);
   #endif
   
   /* Change in output layer weights due to output layer errors:
         dW(output,k) = Eta*E(output)*Out(k) + Alpha*dW(output,k)~

      Change in output layer bias values due to output layer errors:
         dBias(output) = Eta*E(output) + Alpha*dBias(output)~

      where Out(k) is output from kth unit in the previous layer. 
      and ~ denotes previous change. */

   exe1(*(hdW+layers-1));
   dW=tos->mat;  
   drop();

   exe1(*(hOut+layers-2)); 
   Out=tos->mat;
   punits=tos->row;
   drop();

   for(j=0;j<punits;j++) { /* changes for output layer weights: */
      A=Eta*(*(Out+j));
      for(i=0;i<units;i++) {
         *dW=(*(E+i))*A + Alpha*(*dW);
         dW++;
      }
   }
   for(i=0;i<units;i++) { /* changes for output layer unit biases: */
      *dBias=Eta*(*(E+i)) + Alpha*(*dBias);
      dBias++;
   }
/* Errors in layers below the output layer: 
      E(k) = Grad(k)*(W(k+1,k)'*E(k+1) */

   for(k=layers-2;k>-1;k--) { /* top of back-prop loop */

      E=tos->mat; /* E(k+1), errors in next-higher layer ride on stk */

      exe1(*(hOut+k)); /* hOut(k) */
      Out=tos->mat;
      units=tos->row;
      Grad=Out+locvec(grad,units);
      Bias=Out+locvec(bias,units);
      dBias=Out+locvec(dbias,units);

      #if DEBUG
         Esav=Out+locvec(error,units);
      #endif

      drop(); /* dropping hOut(k) */

      if(!matstk(units,1,"_Ek")) return 0;
      Ek=tos->mat; /* vector for errors, Ek, onto stk */

      exe1(*(hW+k+1));  
      bend(); /* W'(k+1) onto stk */

      rot(); /* stk: Ek W' E */

      mpy(); /* E(k) = W'(k)*E(k+1) */
      Atmp=tos->mat;

      for(i=0;i<units;i++) { /* this layer's errors */
         *(Ek+i)=(*Grad)*(*Atmp); /* E(k) = Grad(k)*W'(k)*E(k+1) */
         Grad++;
         Atmp++;
      }
      drop(); /* dropping Atmp */

      #if DEBUG
         memcpy(Esav,Ek,sizeof(double)*units);
      #endif

      /* Change in layer k weights due to layer k errors:
            dW(k) = Eta*E(k)*Out(k-1) + Alpha*dW(k)~

         Change in layer k bias values due to layer k errors:
            dBias(k) = Eta*E(k) + Alpha*dBias(k)~

         where Out(k-1) is output from the previous layer
         and ~ denotes previous change. */

      exe1(*(hdW+k));
      dW=tos->mat;
      drop();

      if(k) { /* weight updates for an intermediate layer */
         exe1(*(hOut+k-1)); /* output of previous layer to stk */
         Out=tos->mat;
         punits=tos->row;
         drop();
      
         for(j=0;j<punits;j++) { /* updates for weight matrix: */
            A=*(Out+j)*Eta;
            for(i=0;i<units;i++) { 
               *dW=(*(Ek+i))*A + Alpha*(*dW);
               dW++;
            }
         }
      }
      else { /* weight updates for the input layer */
         extract1(Net,"NetIn"); /* network input to stk */
         Out=tos->mat;
         punits=tos->row;
         drop();

         for(i=0;i<punits;i++) { /* updates for weight vector: */
            *dW=Eta*(*(Ek+i))*(*(Out+i)) + Alpha*(*dW);
            dW++;
         }
      }
      for(i=0;i<units;i++) { /* updates for layer bias vector: */
         *dBias=Eta*(*(Ek+i)) + Alpha*(*dBias);
         dBias++;
      }
   } /* bottom of back-prop loop */

   drop(); /* Ek off stack */

/* Applying updates to all biases and weights. */

   for(k=0;k<layers;k++) {
      
      exe1(*(hOut+k));
      Out=tos->mat;
      units=tos->row; /* all terms in vector */
      drop();

      Bias=Out+locvec(bias,units);
      dBias=Out+locvec(dbias,units);

      for(i=0;i<units;i++) {
         *Bias-=*dBias; /* applying negative change because dBias has 
            been computed in the same form as dW for weights, which are
            used positively.  But Bias is used negatively, as -b in 
            net_activate. */
         Bias++;
         dBias++;
      }
      exe1(*(hdW+k));
      dW=tos->mat;
      drop();

      exe1(*(hW+k));
      W=tos->mat;
      units=tos->row*tos->col; /* all terms in matrix */
      drop();

      for(i=0;i<units;i++) {
         *W+=*dW;
         W++;
         dW++;
      }
   }
   return 1;

   #undef DEBUG
}

/* End of functions for netbp.v */

/*--------------------------------------------------------------------*/

/* Functions for linear prediction.

   Reference: 
      Press, W. H., et al, "Numerical Recipes in C," Cambridge 
      University Press, second edition, 1999; chapter 13.6. */

int lpcoef() /* lpcoef (hA m --- hC hMsq) */
/* Each column in returned C contains m linear prediction coefficients 
   for the corresponding column of data in A.  

   Equal spacing between points in the rows of A is assumed.

   If the values in A are called y, and coefficients in C are called d,
   then the estimation equation for point k is:

      yk = [Sum(j=1,m)dj*y(k-j)] + xk  

   This word determines coefficients d, and returns them in matrix C.  
   The evaluation of yk using the equation above is done by word lpeval.

   The mean square discrepancy is <x^2>, and is returned in Msq.  Each 
   row in Msq goes with the corresponding column of A, and each holds 
   the mean square discrepancy of the estimated residual error as the
   coefficients are constructed; see the Reference, p. 568.

   Word lpeval uses C and A (only the last m values) to extend A by a
   number of data points into the future. */
{
   double *A,*C,den,*Msq,num,p,*wk1,*wk2,*wkm;
   int cols,i,j=0,k,m,rows;

   if((tos-1)->typ!=MAT) {
      stkerr(" lpcoef: ",MATNOT);
      return 0;
   }
   if(!popint(&m)) return 0;

   A=tos->mat;
   rows=tos->row;
   cols=tos->col;

   if(!matstk(m,cols,"_C")) return 0;
   C=tos->mat;

   if(!matstk(cols,1,"_Msq")) return 0;
   Msq=tos->mat;

   if(!matstk(rows,1,"_wk1")) return 0;
   wk1=tos->mat;

   if(!matstk(rows,1,"_wk2")) return 0;
   wk2=tos->mat;

   if(!matstk(m,1,"_wkm")) return 0;
   wkm=tos->mat;

   for(;j<cols;j++) { 

   /* The following is based on function memcof() in the Reference. */

      p=0;
      for(i=0;i<rows;i++) p+=*(A+i) * (*(A+i));

      *Msq=p/rows;

      memcpy(wk1,A,rows*sizeof(double)); /* wk1(i)=A(i) */
      memcpy(wk2,A+1,(rows-1)*sizeof(double)); /* wk2(i)=A(i+1) */

      for(k=0;k<m;k++) {
         num=0;
         den=0;
         for(i=0;i<(rows-1-k);i++) {
            num+=*(wk1+i) * (*(wk2+i));
            den+=*(wk1+i)*(*(wk1+i)) + *(wk2+i)*(*(wk2+i));
         }
         if(!den) *(C+k)=0; /* do not divide by zero */
         else *(C+k)=2*num/den;

         *Msq*=(1 - *(C+k)*(*(C+k)));

         for(i=0;i<k;i++) 
            *(C+i)=*(wkm+i) - *(C+k)*(*(wkm+k-i-1));

         if(k+1<m) {
            memcpy(wkm,C,(k+1)*sizeof(double)); /* wkm(i)=C(i) */
            for(i=0;i<(rows-k-2);i++) {
               *(wk1+i)-=*(wkm+k) * (*(wk2+i));
               *(wk2+i)=*(wk2+i+1) - *(wkm+k) * (*(wk1+i+1));
            }
         }
      }
      A+=rows;
      C+=m;
      Msq++;
   }
   return(drop2() && drop() && rot() && drop());
}

int lpeval() /* lpeval (hA hC n --- hP) */
/* Evaluate the data series in each column of A, using linear prediction
   coefficients in the corresponding column of C, for n points beyond 
   the end (last row) of A.  

   Equal spacing between points in the rows of A is assumed.

   Evaluation of points n is made using 

         yn = [Sum(j=1,m)dj*y(n-j)] + xn  

   and points are returned in matrix P.  Residual xn is assumed to be 
   zero.

   Coefficients d contained in incoming matrix C may have been computed
   by word lpcoef. */
{
   double *A,*C,*P,sum,*wrk;
   int cols,i,j=0,k,m,n,rows;

   if(tos->typ!=NUM) {
      stkerr(" lpeval: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" lpeval: ",MATNOT2);
      return 0;
   }
   if((cols=(tos-1)->col)!=(tos-2)->col) {
      stkerr(" lpeval: ",MATSNOTC);
      return 0;
   }
   m=(tos-1)->row; /* rows (coefficients) in C */
   rows=(tos-2)->row; /* rows (past data) in A */

   if(rows<m) {
      stkerr(" lpeval: ","rows in A must equal or exceed rows in C");
      return 0;
   }
   popint(&n);
   n=MAX(n,0);

   C=tos->mat;
   A=(tos-1)->mat;

   if(!matstk(m,1,"_wrk")) return 0;
   wrk=tos->mat;

   if(!matstk(n,cols,"_P")) return 0;
   P=tos->mat;

   for(;j<cols;j++) {

   /* The following is based on function predic() in the Reference. */

      for(i=0;i<m;i++) *(wrk+i)=*(A+rows-i-1); /* ith thru mth */

      for(i=0;i<n;i++) {
         sum=0;

         /* dot product of wrk (imagine a row) and coefficients C 
         (imagine a column): */
         for(k=0;k<m;k++) sum+=*(wrk+k) * *(C+k); /* sum across row i */

         for(k=(m-1);k>0;k--) *(wrk+k)=*(wrk+k-1); /* shift wrk right */

         *(P+i)=*wrk=sum; /* P(k) is sum, and so is first term in wrk */
      }
      A+=rows;
      C+=m;
      P+=n;
   }
   return(lop() && lop() && lop());
}

/*--------------------------------------------------------------------*/

/* Functions for bdf.v: processing bulk data files */

int cylvec() /* cylvec (hRTZ --- hXYZ) */
/* R, Theta, Z cylindrical coordinates transformed into rectangular
   X, Y, Z.  The unit of angles in incoming Theta is degrees. 

   RTZ is a 3-by-N matrix, containing coordinates for N points. */
{
   register double *RTZ,*XYZ;
   register int j=0;
   int cols;

   if(tos->typ!=MAT) {
      stkerr(" cylvec: ",MATNOT);
      return 0;
   }
   if(tos->row!=3) {
      stkerr(" cylvec: ","expect matrix of 3 rows");
      return 0;
   }
   if(!matstk(3,(cols=tos->col),"_cylvec")) return 0;

   XYZ=tos->mat;
   RTZ=(tos-1)->mat;

   for(;j<cols;j++) {
      *(XYZ  )=*(RTZ)*(cos(*(RTZ+1)*RADPERDEG));
      *(XYZ+1)=*(RTZ)*(sin(*(RTZ+1)*RADPERDEG));
      *(XYZ+2)=*(RTZ)*(*(RTZ+2));
      XYZ+=3;
      RTZ+=3;
   }
   return(lop());
}

int render() /* render (hT --- hT1) */
/* Rendering electronic bulk data cards of field width 8 and where 
   double fields may be used to represent numbers.

   Each row of incoming volume T is a series of 9-field cards, each
   field 8 bytes wide, and continuation cards extending left to right
   as might be obtained from word cards, file bdf.v.  

   Saying something like "'PSHELL' cards render" will run this func-
   tion to produce volume T1 on the stack, ready to extract patterns
   and numbers from the appropriate fields of PSHELL cards.

   The first field in each incoming card of 72 bytes is an 8-byte 
   identifier, followed by ascii data in either 8 fields of 8 bytes
   (single field) or 4 fields of 16 bytes (double field).  The first-
   field identifier contains * somewhere in its pattern when the card
   type is double field.

   This function returns volume T1 having rows with two 8-byte parts 
   (side by side) for each original data field:

      The first part contains 8 bytes of the original text, right
      justified for proper sorting of patterns that might represent 
      integer numbers.

      The second part contains an 8-byte floating point number as 
      follows, extracted from the text of the original field:

         the actual number if the text can be converted

         INF (000000000000F07Fh little endian machine) if the
            text cannot be converted

         -INF (000000000000F0FFh little endian) if the text
            field is blank (all 20h).

   The last 8 bytes of double field text will be missing in the first 
   part returned, but double field text is assumed to be used only to 
   represent numbers--and all 64 floating point bits of numbers are 
   contained in the second, binary part.

   The beginning 8-byte identifier field in each card of T is not re-
   turned in T1.

   An example showing output from this function is given after its
   code. */
{
   char *T,*T1;
   register char *t,*t1;
   int i=0,j=0;
   register int k=0,m=0,n=0;

   double xblank,x;
   int cards,chars,chars1,rows;

   char buf[32],id[16],*name="_render";

   if(tos->typ!=VOL) { /* top of stack (tos) type must be VOL */
      stkerr(" render: ",VOLNOT);
      return 0;
   }
   chars=tos->col; /* chars is tos->col when tos->typ is VOL */
   rows=tos->row;
 
   if(!rows || !chars) {
      return(
         volstk(rows,chars,name) &&
         lop() 
      );
   }
   T=tos->tex; /* pointer to incoming text */

   cards=MAX(1,(chars/72)); /* assumes chars is int multiple of 72 */
   chars1=64*cards*2; /* upper bound for T1: all fields of 8 into 16 */

   if(!volstk(rows,chars1,name)) return 0;
   T1=tos->tex; /* T1 is now on top of stack */
   memset(T1,' ',rows*chars1); /* initial T1 to all blanks */

   *(id+8)='\0';
   xblank=-INF;

   for(;i<rows;i++) {
      t=T+loclin(i,chars);
      t1=T1+loclin(i,chars1);

      for(j=0;j<cards;j++) {
         memcpy(id,t,8);
         t+=8;

         if(strchr(id,'*')) {
            for(k=0;k<4;k++) { /* doing double fields */
               memcpy(buf,t,16);
               t+=16;

               *(buf+16)='\0';
               n=strspn(buf," "); /* n is offset to 1st char */

               if(n<16) {

                  m=15;
                  while(isspace((int)*(buf+m))) m--; 
                  m++; /* m is len of buf */
                  *(buf+m)='\0'; 

                  m=MIN(8,(m-n)); /* m is len to copy, <= 8 */
                  t1+=8;
                  memcpy(t1-m,buf+n,m); /* text right justified */

                  if(number((buf+n),&x)) memcpy(t1,(char *)&x,8);

                  else memcpy(t1,(char *)&INF,8);
               }
               else { /* field is blank */
                  t1+=8;
                  memcpy(t1,(char *)&xblank,8);
               }
               t1+=8;
            }
         }
         else {
            for(k=0;k<8;k++) { /* doing single fields */
               memcpy(buf,t,8);
               t+=8;

               *(buf+8)='\0';
               n=strspn(buf," "); /* n is offset to 1st char */

               if(n<8) {

                  m=7;
                  while(isspace((int)*(buf+m))) m--;
                  m++; /* m is len of buf */
                  *(buf+m)='\0';

                  m=MIN(8,(m-n)); /* m is length, <= 8 */
                  t1+=8;
                  memcpy(t1-m,buf+n,m); /* text right justified */

                  if(number((buf+n),&x)) memcpy(t1,(char *)&x,8);

                  else memcpy(t1,(char *)&INF,8);
               }
               else { /* field is blank */
                  t1+=8;
                  memcpy(t1,(char *)&xblank,8);
               }
               t1+=8;
            }
         }
      }
   }
   return(lop());
}
/* 
An example of word render.

Note: this example is best viewed in a window set to character width 72
or 80.

For these 80-character width bulk data cards (two single field, one 
double, and lines ignored with $ in the first column):

$2345678123456781234567812345678123456781234567812345678123456781234567812345678
JOINT     1981    1100    x8.5    -6d-1   41.9    1000

$23456780123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef12345678
JOINT*          10010   0               7.345678901234565.34567890123456JOINT  1
*OINT  11.345678901234561

JOINT     1951    1100    88.5    -62.0   41.9    1000

here is an xray of volume T1 from word render, verifying right justified
text for the first 8 bytes and double precision floating point numbers
(in little endian byte order from host gutter) for the second: 

[tops@gutter] ready > "se.bdf" bdf
 file mapping complete

[tops@gutter] ready > "joint" cards render, INF xray .
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  20 20 20 20 31 39 38 31 00 00 00 00 00 F4 9E 40      1981.......@
   2  20 20 20 20 31 31 30 30 00 00 00 00 00 30 91 40      1100.....0.@
   4  20 20 20 20 78 38 2E 35 00 00 00 00 00 00 F0 7F      x8.5........
   6  20 20 20 2D 36 64 2D 31 33 33 33 33 33 33 E3 BF     -6d-1333333..
   8  20 20 20 20 34 31 2E 39 33 33 33 33 33 F3 44 40      41.933333.D@
  10  20 20 20 20 31 30 30 30 00 00 00 00 00 40 8F 40      1000.....@.@
  12  20 20 20 20 20 20 20 20 00 00 00 00 00 00 F0 FF          ........
...
  30  20 20 20 20 20 20 20 20 00 00 00 00 00 00 F0 FF          ........
  32  20 20 20 31 30 30 31 30 00 00 00 00 00 8D C3 40     10010.......@
  34  20 20 20 20 20 20 20 30 00 00 00 00 00 00 00 00         0........
  36  37 2E 33 34 35 36 37 38 EC E0 5E A6 F9 61 1D 40  7.345678..^..a.@
  38  35 2E 33 34 35 36 37 38 EC E0 5E A6 F9 61 15 40  5.345678..^..a.@
  40  31 2E 33 34 35 36 37 38 AF 83 7B 99 E6 87 F5 3F  1.345678..{....?
  42  20 20 20 20 20 20 20 31 00 00 00 00 00 00 F0 3F         1.......?
  44  20 20 20 20 20 20 20 20 00 00 00 00 00 00 F0 FF          ........
  46  20 20 20 20 20 20 20 20 00 00 00 00 00 00 F0 FF          ........
  48  20 20 20 20 20 20 20 20 20 20 20 20 20 20 20 20
...
  62  20 20 20 20 20 20 20 20 20 20 20 20 20 20 20 20
  64  20 20 20 20 31 39 35 31 00 00 00 00 00 7C 9E 40      1951.....|.@
  66  20 20 20 20 31 31 30 30 00 00 00 00 00 30 91 40      1100.....0.@
  68  20 20 20 20 38 38 2E 35 00 00 00 00 00 20 56 40      88.5..... V@
  70  20 20 20 2D 36 32 2E 30 00 00 00 00 00 00 4F C0     -62.0......O.
  72  20 20 20 20 34 31 2E 39 33 33 33 33 33 F3 44 40      41.933333.D@
  74  20 20 20 20 31 30 30 30 00 00 00 00 00 40 8F 40      1000.....@.@
  76  20 20 20 20 20 20 20 20 00 00 00 00 00 00 F0 FF          ........
...

The following agrees with the hex pattern in line 36, and verifies that
all 16 characters for 7.3456... in the double field card of JOINT 10010
are accounted for:

[tops@gutter] ready > 7.34567890123456 .hex
 EC E0 5E A6 F9 61 1D 40
[tops@gutter] ready > 

Later error checking will reject the x8.5 in line 4, since x8.5 could
not be converted to a number and in JOINT cards it is the value of the
X coordinate.  The INF in the second 8-byte part of the card, shown 
again below, exposes bad numbers like this to a rake of INFs, to be 
easily culled out:

   4  20 20 20 20 78 38 2E 35 00 00 00 00 00 00 F0 7F      x8.5........
*/

int sphvec() /* sphvec (hRTP --- hXYZ) */
/* R, Theta, Phi spherical coordinates transformed into rectangular
   X, Y, Z.  The unit of angles in Theta and Phi is degrees. */
{
   register double *RTP,sinP,*XYZ;
   register int j=0;
   int cols;
   const double radperdeg=0.017453292519943295;

   if(tos->typ!=MAT) {
      stkerr(" sphvec: ",MATNOT);
      return 0;
   }
   if(tos->row!=3) {
      stkerr(" sphvec: ","expect matrix of 3 rows");
      return 0;
   }
   if(!matstk(3,(cols=tos->col),"_sphvec")) return 0;

   XYZ=tos->mat;
   RTP=(tos-1)->mat;

   for(;j<cols;j++) {
      sinP=sin(*(RTP+1)*radperdeg);
      *(XYZ  )=*(RTP)*sinP*cos(*(RTP+2)*radperdeg);
      *(XYZ+1)=*(RTP)*sinP*sin(*(RTP+2)*radperdeg);
      *(XYZ+2)=*(RTP)*cos(*(RTP+1)*radperdeg);
      XYZ+=3;
      RTP+=3;
   }
   return(lop());
}

/* End of functions for bdf.v */

/*--------------------------------------------------------------------*/

/* Functions for genetic algorithm */

int ga_bit() /* ga_bit (hV hR --- hB) */
/* Fetch the bits in V that are listed in R.  B(i)=-1 if R(i) is on,
   and 0 otherwise.
   Bits are numbered from left to right, starting with 1st. */
{
   char *V;
   double *B,*R;
   int bit,byte,k=0,maxbit,rows;
   unsigned char mask[8]={128,64,32,16,8,4,2,1};

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" ga_bit: ",NUMORMATNOT);
      return 0;
   }
   if((tos-1)->typ!=VOL && (tos-1)->typ!=STR) {
      stkerr(" ga_bit: ",STRORVOLNOT);
      return 0;
   }
   hand();
   R=tos->mat;

   V=(tos-1)->tex;
   maxbit=(tos-1)->col<<3; /* multiply by 8 */

   if(!matstk((rows=tos->row),1,"_bit")) return 0;
   B=tos->mat;

   for(;k<rows;k++) {
      bit=*(R+k) - XBASE;
      if(bit<0 || 1+bit>maxbit) {
         stkerr(" ga_bit: ","bit index out of range");
         return 0;
      }
      byte=bit>>3; /* divide by 8 to get byte in V */
      bit-=byte<<3; /* bit local within byte = mask index */

      if(*(V+byte) & *(mask+bit)) *(B+k)=xTRUE;
      else *(B+k)=xFALSE;
   }
   return(lop() && lop());
}

int ga_bitoff() /* ga_bitoff (hV hR --- hV1) */
/* Turn off the bits in V that are listed in R.
   Bits are numbered from left to right, starting with 1st. */
{
   char *V1;
   double *R;
   int bit,byte,k=0,maxbit,rows;
   unsigned char mask[8]={128,64,32,16,8,4,2,1};

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" ga_bitoff: ",NUMORMATNOT);
      return 0;
   }
   if((tos-1)->typ!=VOL && (tos-1)->typ!=STR) {
      stkerr(" ga_bitoff: ",STRORVOLNOT);
      return 0;
   }
   hand();
   R=tos->mat;
   rows=tos->row;

   swap();
   cop();
   V1=tos->tex;
   maxbit=tos->col<<3; /* multiply by 8 */

   for(;k<rows;k++) {
      bit=*(R+k) - XBASE;
      if(bit<0 || 1+bit>maxbit) {
         stkerr(" ga_bitoff: ","bit index out of range");
         return 0;
      }
      byte=bit>>3; /* divide by 8 to get byte in V */
      bit-=byte<<3; /* bit local within byte = mask index */

      *(V1+byte)=*(V1+byte) | *(mask+bit); /* turn bit on */
      *(V1+byte)=*(V1+byte) ^ *(mask+bit); /* xor bit the mask's 1 */

   }
   return(lop());
}

int ga_biton() /* ga_biton (hV hR --- hV1) */
/* Turn on the bits in V that are listed in R.
   Bits are numbered from left to right, starting with 1st. */
{
   char *V1;
   double *R;
   int bit,byte,k=0,maxbit,rows;
   unsigned char mask[8]={128,64,32,16,8,4,2,1};

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" ga_biton: ",NUMORMATNOT);
      return 0;
   }
   if((tos-1)->typ!=VOL && (tos-1)->typ!=STR) {
      stkerr(" ga_biton: ",STRORVOLNOT);
      return 0;
   }
   hand();
   R=tos->mat;
   rows=tos->row;

   swap();
   cop();
   V1=tos->tex;
   maxbit=tos->col<<3; /* multiply by 8 */

   for(;k<rows;k++) {
      bit=*(R+k) - XBASE;
      if(bit<0 || 1+bit>maxbit) {
         stkerr(" ga_biton: ","bit index out of range");
         return 0;
      }
      byte=bit>>3; /* divide by 8 to get byte in V */
      bit-=byte<<3; /* bit local within byte = mask index */

      *(V1+byte)=*(V1+byte) | *(mask+bit); /* turn bit on */
   }
   return(lop());
}

int ga_crossover() /* ga_crossover (hA hB nbit --- hC hD) */
/* Cut A and B at nbit and join the head of A to the tail of B to make 
   C, and the head of B to the tail of A to make D.  Bit nbit goes with
   the head segment.

   Bits are numbered from left to right, starting with 1st. 

   If nbit = 1st-1, then C=B and D=A.
   If nbit = Abits = Bbits, then C=A and D=B.

\  Demo (works for 0based and 1based, BIG_ENDIAN and LITTLE_ENDIAN):

      32 ga_random 1st 32 items ga_biton "A" book
      4 nulls "B" book
      A .bin nl B .bin nl

      20 12 ndx DO A B I ga_crossover .bin nl LOOP nl nl \
      20 12 ndx DO .bin nl LOOP nl

\  Bounding cases:
      A B 0  ndx ga_crossover .bin nl .bin
      A B 32 ndx ga_crossover .bin nl .bin
*/
{
   char *A,*B,*C,*D;
   int byte=0,bytes,nbit;

/* Bit masks */
   unsigned char mask[8]={128,192,224,240,248,252,254,255};
/* nask is not(mask): */
   unsigned char nask[8]={127, 63, 31, 15,  7,  3,  1,  0};

/* Looking at some bit masks:
      [tops@rugger] ready > 240 int4 1st orderbyte catch .bin
       11110000
      [tops@rugger] ready > 240 int4 1st orderbyte catch nott .bin
       00001111
      [tops@rugger] ready > 15 int4 1st orderbyte catch .bin      
       00001111
      [tops@rugger] ready > 
*/
   if(!popint(&nbit)) return 0;
   nbit-=XBASE; /* can range from -1 to 8*(tos->col)-1 */

   if(tos->typ!=VOL && tos->typ!=STR) {
      stkerr(" ga_crossover: ",STRORVOLNOT);
      return 0;
   }
   if((tos-1)->typ!=VOL && (tos-1)->typ!=STR) {
      stkerr(" ga_crossover: ",STRORVOLNOT);
      return 0;
   }
   bytes=tos->col;

   if(bytes!=(tos-1)->col) {
      stkerr(" ga_crossover: ","strings are not the same length");
      return 0;
   }
   if(nbit<-1 || nbit>(bytes<<3)-1) {
      stkerr(" ga_crossover: ","crossover point is outside strings");
      return 0;
   }
   B=tos->tex;
   A=(tos-1)->tex;

   if(!ga_strstk(bytes,"_C")) return 0;
   C=tos->tex;

   if(!ga_strstk(bytes,"_D")) return 0;
   D=tos->tex;

   if(nbit>0) byte=nbit>>3; /* divide by 8 to get byte in V */

   memcpy(C,A,byte); /* head of A to C */
   memcpy(C+byte+1,B+byte+1,bytes-byte-1); /* tail of B to C */

   memcpy(D,B,byte); /* head of B to D */
   memcpy(D+byte+1,A+byte+1,bytes-byte-1); /* tail of A to D */

/* Create byte at the head/tail intersection: */
   nbit-=byte<<3; /* bit local within byte = index for mask and nask */

   if(nbit>-1) {
      *(C+byte)=(*(A+byte) & *(mask+nbit)) | (*(B+byte) & *(nask+nbit));
      *(D+byte)=(*(B+byte) & *(mask+nbit)) | (*(A+byte) & *(nask+nbit));
   }
   else {
      *(C+byte)=*(B+byte);
      *(D+byte)=*(A+byte);
   }
   return(rot() && drop() && rot() && drop());
}

int ga_fill() /* ga_fill (hR --- hV) */
/* Create a 1-row volume of bits where bit k is 0 if R(k)=0, and 1
   otherwise.  Bits are numbered from left to right, starting with 
   1st, and 1st bit corresponds to R(1st). */
{
   char *V;
   double *R;
   int b,bit,byte,h,k=0,n;
   unsigned char mask[8]={128,64,32,16,8,4,2,1};

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" ga_fill: ",NUMORMATNOT);
      return 0;
   }
   hand();
   R=tos->mat;
   n=tos->row;

   h=(((b=n>>3)<<3)!=n);

   if(!ga_strstk(h+b,"_V")) return 0;
   V=tos->tex;
   memset(V,0,tos->col);

   for(;k<n;k++) {
      if(*R) {
         bit=k;

         byte=bit>>3; /* divide by 8 to get byte in V */
         bit-=byte<<3; /* bit local within byte = mask index */

         *(V+byte)=*(V+byte) | *(mask+bit); /* turn bit on */
      }
      R++;
   }
   return(lop());
}

int ga_random() /* ga_random (n --- hV) */
/* Create a 1-row volume of n random bits.
   Bits are numbered from left to right, starting with 1st. */
{
   char *V;
   int b,bit,byte,h,k=0,n;
   unsigned char mask[8]={128,64,32,16,8,4,2,1};

   if(!popint(&n)) return 0;

   h=(((b=n>>3)<<3)!=n);

   if(!ga_strstk(h+b,"_V")) return 0;
   V=tos->tex;
   memset(V,0,tos->col);

   for(;k<n;k++) {
      if(rand0(&SEED)>0.5) {
         bit=k;

         byte=bit>>3; /* divide by 8 to get byte in V */
         bit-=byte<<3; /* bit local within byte = mask index */

         *(V+byte)=*(V+byte) | *(mask+bit); /* turn bit on */
      }
   }
   return 1;
}

int ga_strstk(int bytes, char *name) 
/* Push uninitialized string to stack, stored as a 1-row VOL that is 
   bytes long. */
{
   char *S;

   bytes=MAX(0,bytes);

   if((S=malloc(1+bytes))==NULL) {
      stkerr(" ga_strstk: ",MEMNOT);
      return 0;
   }
   *(S+bytes)='\0';

   return(push(VOL,(char *)memgetn(name,strlen(name)),NOTAG,0,NULL,S, \
      1,bytes,NULL));
}

int ga_toggle() /* ga_toggle (hV hR --- hV1) */
/* Toggle the bits in V that are listed in R.
   Bits are numbered from left to right, starting with 1st. */
{
   char *V1;
   double *R;
   int bit,byte,k=0,maxbit,rows;
   unsigned char mask[8]={128,64,32,16,8,4,2,1};

   if(tos->typ!=MAT && tos->typ!=NUM) {
      stkerr(" ga_biton: ",NUMORMATNOT);
      return 0;
   }
   if((tos-1)->typ!=VOL && (tos-1)->typ!=STR) {
      stkerr(" ga_biton: ",STRORVOLNOT);
      return 0;
   }
   hand();
   R=tos->mat;
   rows=tos->row;

   swap();
   cop();
   V1=tos->tex;
   maxbit=tos->col<<3; /* multiply by 8 */

   for(;k<rows;k++) {
      bit=*(R+k) - XBASE;
      if(bit<0 || 1+bit>maxbit) {
         stkerr(" ga_biton: ","bit index out of range");
         return 0;
      }
      byte=bit>>3; /* divide by 8 to get byte in V */
      bit-=byte<<3; /* bit local within byte = mask index */

      *(V1+byte)=*(V1+byte) ^ *(mask+bit); /* xor with the mask's 1 */
   }
   return(lop());
}

/*--------------------------------------------------------------------*/

/* Functions for mday.n and other related files. */

int cnearest() /* cnearest (hC hB --- hR hS) */
/* For value C(k) in vector C, find from the columns of matrix B, 
   R(k) = B(k,m) nearest above C(k) and S(k) = B(k,n) nearest below 
   C(k).  If no B(k,j) is above C(k) then R(k) = C(k); if no B(k,j) 
   is below C(k), then S(k) = C(k). */
{
   double *B,Bk,*C,Ck,dR,dS,*R,*S;
   int cols,rows,j=0,jR,jS,k=0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" cnearest: ","expect two matrices on the stack");
      return 0;
   }
   C=(tos-1)->mat;
   rows=(tos-1)->row;

   if(rows!=tos->row) {
      stkerr(" cnearest: ",MATSNOTC);
      return 0;
   }
   B=tos->mat;
   cols=tos->col;

   if(!matstk(rows,1,"_R")) return 0;
   R=tos->mat;

   if(!matstk(rows,1,"_S")) return 0;
   S=tos->mat;

   for(;k<rows;k++) {
      Ck=*C;
      dR=dS=INF;
      jR=jS=-1;
      for(j=0;j<cols;j++) {
         Bk=*(B+locvec(j,rows)+k);
         if(Bk>Ck && (Bk-Ck)<dR) {
            jR=j;
            dR=Bk-Ck;
         }
         if(Bk<Ck && (Ck-Bk)<dS) {
            jS=j;
            dS=Ck-Bk;
         }
      }
      if(jR>-1) *R=*(B+k+locvec(jR,rows));
      else *R=Ck;
      
      if(jS>-1) *S=*(B+k+locvec(jS,rows));
      else *S=Ck;

      C++;
      R++;
      S++;
   }
   return(rot() && drop() && rot() && drop());
}

int fdiff() /* fdiff (hC1 hC2 hR --- hD) */
/* Fractional difference between C1 and C2, using

      D(k) = (Cx(k) - Cy(k))/(Cy(k) - R(k))

   where Cx(k) and Cy(k) are the greater and lesser, respectively,
   of C1(k) and C2(k).  When multiplied by 100, D is a percentage. 

   R is the adjustment for rollover. */
{
   double *C1,*C2,*D,*R;
   int k=0,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" fdiff: ","expect three matrices on the stack");
      return 0;
   }
   R=tos->mat;
   rows=tos->row;

   C2=(tos-1)->mat;
   C1=(tos-2)->mat;

   if(!matstk(rows,1,"_D")) return 0;
   D=tos->mat;

   for(;k<rows;k++) {
      if(*C1>*C2) *D=(*C1-*C2)/(*C2-*R);
      else *D=(*C2-*C1)/(*C1-*R);

      C1++;
      C2++;
      D++;
      R++;
   }
   return(lpush() && drop2() && drop() && lpull());
}

double fdiff1(double C1, double C2, double R) 
/* Does what fdiff() does within its loop. */
{
   if(C1>C2) return (C1-C2)/(C2-R);
   else return (C2-C1)/(C1-R);
}

int midfields() /* midfields (hP hR hE --- hM) */
/* This function is equivalent to:
      (L, S, Lc, Sc) = pop_dist(P, R, E);
      M = ((L + S)/2)'; */
{
   return(
      pop_dist() && /* (hL hS hLc hSc) */
      drop2() &&
      plusm() &&
      pushd(0.5) &&
      starf() &&
      bend()
   );
}

int msr() /* msr (hMx hSx hLx --- hS hB1 hB2 hNXS hR hA1 hA2 hNXR) */
/* Levels of support and resistance provided by the midfields, Mx,
   and their extreme values Sx (highest) and Lx (lowest):

      S = support = level of lowest-goal Mx that does not change in
      down move, and B1, B2 = target levels below and above S

      R = resistance = level of lowest-goal Mx that does not change
      in up move, and A1, A2 = target levels below and above R

   This function fetches values from the arrays Mx, Sx and Lx, and
   makes no calculations. */
{
   double *A1,*A2,*B1,*B2,half=0.5,*Lx,*Mx,*M1,*NXR,*NXS,\
      *R,*S,*Sx,ten=10;
   int i,i_res=0,i_sup=0,k=1,nm,steps;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" msr: ","expect three matrices on the stack");
      return 0;
   }
   bend();
   Lx=tos->mat;

   swap();
   bend();
   Sx=tos->mat;

   rot();
   bend();
   dup1s();
   M1=tos->mat; /* Mx with all floating point digits */

   swap();
   pushd(ten);
   starf();
   pushd(half);
   plusd();
   integer();
   Mx=tos->mat; /* Mx*10 rounded to integer for equality comparisons */

   steps=tos->col;
   nm=tos->row;

   if(!matstk(steps,1,"_NXR")) return 0;
   NXR=tos->mat;
   lpush();

   if(!matstk(steps,1,"_A2")) return 0;
   A2=tos->mat;
   lpush();

   if(!matstk(steps,1,"_A1")) return 0;
   A1=tos->mat;
   lpush();

   if(!matstk(steps,1,"_R")) return 0;
   R=tos->mat;
   lpush();

   if(!matstk(steps,1,"_NXS")) return 0;
   NXS=tos->mat;
   lpush();

   if(!matstk(steps,1,"_B2")) return 0;
   B2=tos->mat;
   lpush();

   if(!matstk(steps,1,"_B1")) return 0;
   B1=tos->mat;
   lpush();

   if(!matstk(steps,1,"_S")) return 0;
   S=tos->mat;
   lpush();

   *A1=*A2=*B1=*B2=*R=*S=*M1;
   *NXR=*NXS=i_res+XBASE;

   Mx+=nm;
   M1+=nm;
   Lx+=nm;
   Sx+=nm;

   R++;
   S++;
   A1++;
   A2++;
   B1++;
   B2++;
   NXR++;
   NXS++;

   for(;k<steps;k++) {
      i=1;
      if(*Mx>*(Mx-nm)) { /* Mx+0 up: doing R */
         while(i<nm-1 && *(Mx+i)!=*(Mx-nm+i)) i++;
         if(i>nm-1) i_res=nm-1;
         else i_res=i;
      }
      else {
         if(*Mx<*(Mx-nm)) { /* Mx+0 down: doing S */
            while(i<nm-1 && *(Mx+i)!=*(Mx-nm+i)) i++;
            if(i>nm-1) i_sup=nm-1;
            else i_sup=i;
         }
      }
      *NXS=i_sup+XBASE;
      *S=*(M1+i_sup);
      *B1=*(Lx+i_sup);
      *B2=*(Sx+i_sup);

      *NXR=i_res+XBASE;
      *R=*(M1+i_res);
      *A1=*(Lx+i_res);
      *A2=*(Sx+i_res);

      Mx+=nm;
      M1+=nm;
      Lx+=nm;
      Sx+=nm;

      R++;
      S++;
      A1++;
      A2++;
      B1++;
      B2++;
      NXR++;
      NXS++;
   }
   return(drop2() && drop2() && lpull() && lpull() &&
      lpull() && lpull() && lpull() && lpull() && lpull() && lpull()
   );
}

int pickb() /* pickb (hP hR hE hL hN nD --- hB) */
/* Bottom picker. 

   Pick bottoms for load P(t) and population distributions L(x,t) corre-
   sponding to goal levels E(x), using goals E(N).  The maximum number 
   of steps to backtrack is D.

   On step k where the bottom B(k) has become invalid, B(k) is set
   equal to P(k).  This means that "true" in the test for equality, 
   B(k)==P(k), will indicate that B(k) is not a bottom level. */
{
   double *B,d,*E,*F,*G,*H,*L,*L1,*L2,*L3,*L4,*N,*P,*R;
   int D,erows,hit,k=0,n1,n2,n3,n4,nrows=4,steps,t;

   if(!popint(&D)) return 0;

   for(;k<5;k++) {
      if((tos-k)->typ!=MAT) {
         stkerr(" pickb: ","expect five matrices on the stack");
         return 0;
      }
   }
   steps=(tos-1)->row; /* rows of P, R, L */
   erows=(tos-2)->row; /* rows of E and cols of L */

   N=tos->mat;
   if(tos->row!=nrows) {
      gprintf(" pickb: expect %d rows for matrix N",nrows);
      nc();
      stkerr("","");
      return 0;
   }
   L=(tos-1)->mat;
   E=(tos-2)->mat;
   R=(tos-3)->mat;
   P=(tos-4)->mat;
   
   if(!matstk(steps,1,"_F")) return 0; 
   F=tos->mat;

   if(!matstk(steps,1,"_G")) return 0; 
   G=tos->mat;

   if(!matstk(steps,1,"_H")) return 0; 
   H=tos->mat;

   if(!matstk(steps,1,"_B")) return 0; 
   B=tos->mat;
   
   n1=(int)(*N-XBASE);
   n2=(int)(*(N+1)-XBASE);
   n3=(int)(*(N+2)-XBASE);
   n4=(int)(*(N+3)-XBASE);

   L1=L+locvec(n1,steps);
   L2=L+locvec(n2,steps);

   L3=L+locvec(n3,steps);
   L4=L+locvec(n4,steps);

   d=(*E)*0.1; /* a fraction of the first element of E */

/* Compare L1 and L2 for fractional diffs less than d: */
   for(t=0;t<steps;t++) *(F+t)=fdiff1(*(L1+t),*(L2+t),*(R+t))<d; 

/* Compare L3 and L4 for fractional diffs greater than d: */
   for(t=0;t<steps;t++) *(G+t)=fdiff1(*(L3+t),*(L4+t),*(R+t))>d; 

   for(t=0;t<D;t++) {
      *(B+t)=*(P+t);
      *(H+t)=0;
   }
   for(t=D;t<steps;t++) {
      if((int)*(G+t) && !(int)*(G+t-1)) {
         k=1;
         hit=0;
         while(k<D+1 && !hit) {
            if((int)*(F+t-k)) hit=k;
            k++;
         }
         if(hit) {
            *(B+t)=*(L2+t-hit);
            *(H+t)=1; /* a bottom pick is in effect */
         }
         else {
            if((int)*(H+t-1)) {
               if(*(B+t-1)<*(P+t)) { /* B is still a bottom */
                  *(B+t)=*(B+t-1);
                  *(H+t)=1; /* still in effect */
               }
               else {
                  *(B+t)=*(P+t);
                  *(H+t)=0; 
               }
            }
            else {
               *(B+t)=*(P+t);
               *(H+t)=0;
            }
         }
      }
      else {
         if((int)*(H+t-1)) {
            if(*(B+t-1)<*(P+t)) { /* B is still a bottom */
               *(B+t)=*(B+t-1);
               *(H+t)=1; /* still in effect */
            }
            else {
               *(B+t)=*(P+t);
               *(H+t)=0;
            }
         }
         else {
            *(B+t)=*(P+t);
            *(H+t)=0;
         }
      }
   }
   return(
      lpush() && drop2() && drop2() && drop2() && drop2() &&
      lpull()
   );
}

int pickt() /* pickt (hP hR hE hS hN nD --- hT) */
/* Top picker. 

   Pick tops for load P(t) and population distributions S(x,t) corre-
   sponding to goal levels E(x), using goals E(N).  The maximum num-
   ber of steps to backtrack is D. 

   On step k where the top T(k) has become invalid, T(k) is set equal 
   to P(k), and "true" in the test T(k)==P(k) will mean that T(k) is 
   not a top level. */
{
   double d,*E,*F,*G,*H,*N,*P,*R,*S,*S1,*S2,*S3,*S4,*T;
   int D,erows,hit,k=0,n1,n2,n3,n4,nrows=4,steps,t;

   if(!popint(&D)) return 0;

   for(;k<5;k++) {
      if((tos-k)->typ!=MAT) {
         stkerr(" pickt: ","expect five matrices on the stack");
         return 0;
      }
   }
   steps=(tos-1)->row; /* rows of P, R, S */
   erows=(tos-2)->row; /* rows of E and cols of S */

   N=tos->mat;
   if(tos->row!=nrows) {
      gprintf(" pickt: expect %d rows for matrix N",nrows);
      nc();
      stkerr("","");
      return 0;
   }
   S=(tos-1)->mat;
   E=(tos-2)->mat;
   R=(tos-3)->mat;
   P=(tos-4)->mat;
   
   if(!matstk(steps,1,"_F")) return 0; 
   F=tos->mat;

   if(!matstk(steps,1,"_G")) return 0; 
   G=tos->mat;

   if(!matstk(steps,1,"_H")) return 0;
   H=tos->mat;

   if(!matstk(steps,1,"_T")) return 0; 
   T=tos->mat;
   
   n1=(int)(*N-XBASE);
   n2=(int)(*(N+1)-XBASE);
   n3=(int)(*(N+2)-XBASE);
   n4=(int)(*(N+3)-XBASE);

   S1=S+locvec(n1,steps);
   S2=S+locvec(n2,steps);

   S3=S+locvec(n3,steps);
   S4=S+locvec(n4,steps);

   d=(*E)*0.1; /* a fraction of the first element of E */

/* Compare S1 and S2 for fractional diffs less than d: */
   for(t=0;t<steps;t++) *(F+t)=fdiff1(*(S1+t),*(S2+t),*(R+t))<d; 

/* Compare S3 and S4 for fractional diffs greater than d: */
   for(t=0;t<steps;t++) *(G+t)=fdiff1(*(S3+t),*(S4+t),*(R+t))>d; 

   for(t=0;t<D;t++) {
      *(T+t)=*(P+t);
      *(H+t)=0;
   }
   for(t=D;t<steps;t++) {
      if((int)*(G+t) && !(int)*(G+t-1)) {
         k=1;
         hit=0;
         while(k<D+1 && !hit) {
            if((int)*(F+t-k)) hit=k;
            k++;
         }
         if(hit) {
            *(T+t)=*(S2+t-hit);
            *(H+t)=1; /* a top pick is in effect */
         }
         else {
            if((int)*(H+t-1)) {
               if(*(T+t-1)>*(P+t)) { /* T is still a top */
                  *(T+t)=*(T+t-1);
                  *(H+t)=1; /* still in effect */
               }
               else {
                  *(T+t)=*(P+t);
                  *(H+t)=0;
               }
            }
            else {
               *(T+t)=*(P+t);
               *(H+t)=0;
            }
         }
      }
      else {
         if((int)*(H+t-1)) {
            if(*(T+t-1)>*(P+t)) { /* T is still a top */
               *(T+t)=*(T+t-1);
               *(H+t)=1; /* still in effect */
            }
            else {
               *(T+t)=*(P+t);
               *(H+t)=0;
            }
         }
         else {
            *(T+t)=*(P+t);
            *(H+t)=0;
         }
      }
   }
   return(
      lpush() && drop2() && drop2() && drop2() && drop2() &&
      lpull()
   );
}

int pop_dist() /* pop_dist (hP hR hE --- hLx hSx hLc hSc) */
/* At each step (row) in load P(t), compute the population distribu-
   tions L(x,t) and S(x,t) for the model that result from goal dis-
   tributions E(x).  Then compute the centroids Lc(t) and Sc(t) of
   the distributions L(x,t) and S(x,t).

   R(t) is the term for rollover that must be subtracted from P(t), and
   related numbers, before doing math that involves scaling.  After 
   scaling, R(t) is added back.

   Below is a high level function that does what this function does on
   each step.  Calling it in a loop (over values of e taken from E) for
   it's Lbar and Sbar for all e of E produces results identical this 
   function's Lc and Sc.  

   function (L, Lbar, S, Sbar) = pop_distribution(M, R, e) {
   // L and S distributions for M(e).
      Incoming M is a column, R is a number and e is a column having
      rows that correspond to the rows of M, that is, row k of e goes
      with row k of M.

      These match the equations in midfield(), wapp.c: //
      L = R + (M - R) ./ (1 + e/2);
      S = R + (M - R) .* (1 + e)/(1 + e/2);

   // Compute the centroids of distributions L and S.  Note that the
      centroid of M is just the average of centroids Lbar and Sbar
      computed below: //
      Lbar = @centroid(L, e);
      Sbar = @centroid(S, e);

      return(L, Lbar, S, Sbar);
   } 

   This high level function takes 7.7 seconds to do what the function 
   here does in 0.031 seconds (about 248 times slower).

   But the high level function is written in just four lines and pro-
   vides a means of validating the approach, graphing population dis-
   tributions and finally debugging the low level code written here. */
{
   double D,*E,*E0,e1,e2,*Lc,*Lx,*Lx0,*Mx,*Mx0,*Mx1,*Mx10;
   double NL,NS,*P,*R,*Sc,*Sx,*Sx0;
   int erows,t,x,steps;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" pop_dist: ","expect three matrices on the stack");
      return 0;
   }
   E0=tos->mat;
   erows=tos->row;

   R=(tos-1)->mat;

   P=(tos-2)->mat;
   steps=(tos-2)->row; /* rows of P (and of R) */
 
   if(!matstk(steps,1,"_Sc")) return 0;
   Sc=tos->mat;
   lpush(); /* Sc on temp stack */

/* If there is a "return 0" on one of the next steps, there will
   be items left on the temp stack, but that is a smaller problem
   than insufficient memory. */
   if(!matstk(steps,1,"_Lc")) return 0; 
   Lc=tos->mat;
   lpush(); /* Lc on temp stack */

   if(!matstk(erows,steps,"_Sx")) return 0;
   Sx0=tos->mat;
   lpush(); /* Sx on temp stack */

   if(!matstk(erows,steps,"_Lx")) return 0;
   Lx0=tos->mat;
   lpush(); /* Lx on temp stack */

   if(!matstk(erows,1,"_Mx")) return 0;
   Mx0=tos->mat;

   if(!matstk(erows,1,"_Mx1")) return 0;
   Mx10=tos->mat;

/* Values at the initial point: */
   D=NS=NL=0;
   E=E0;
   Lx=Lx0;
   Mx=Mx0;
   Sx=Sx0;
   for(x=0;x<erows;x++) {
      e1=1+*E;
      e2=1+*E/2;
      *Mx=*P;
      *Lx=*R + (*Mx-*R)/e2;
      *Sx=*R + (*Mx-*R)*e1/e2;

      D+=1;
      NL+=*Lx;
      NS+=*Sx;

      E++;
      Lx++;
      Mx++;
      Sx++;
   } 
   *Lc=NL/D;
   *Sc=NS/D;

   Lc++;
   Sc++;
   P++;
   R++;

/* Values after each step; values of Lx and Sx, used below in logical 
   comparisons with P, are before they are updated and correspond to
   the previous point: */
   for(t=1;t<steps;t++) { 
      D=NS=NL=0;
      E=E0;
      Mx=Mx0;
      Mx1=Mx10;
      memcpy(Mx1,Mx,erows*sizeof(double));

      Lx=Lx0;
      Lx0+=erows;
      memcpy(Lx0,Lx,erows*sizeof(double));
      Lx=Lx0;

      Sx=Sx0;
      Sx0+=erows;
      memcpy(Sx0,Sx,erows*sizeof(double));
      Sx=Sx0;

   /* In this model, Mx is the independent variable being computed.  
      At any step, Mx is sufficient to compute Lx and Sx by these 
      equations:
         *Lx=*R + (*Mx-*R)/e2;
         *Sx=*R + (*Mx-*R)*e1/e2; */

      for(x=0;x<erows;x++) { 
         e1=1+*E;
         e2=1+*E/2;
         if(*P>*Sx) {
            *Sx=*P;
            *Mx=*R + (*P-*R)*e2/e1;
            *Lx=*R + (*Mx-*R)/e2;
         }
         else {
            if(*P<*Lx) {
               *Lx=*P;
               *Mx=*R + (*P-*R)*e2;
               *Sx=*R + (*Mx-*R)*e1/e2;
            }
            else {
            /* Use previous value of Mx: */
               *Mx=*Mx1;
               *Lx=*R + (*Mx-*R)/e2;
               *Sx=*R + (*Mx-*R)*e1/e2;
            }
         }
      /* Running sums for centroids, like centroid() in file wapp.c: */
         D+=1;
         NL+=*Lx;
         NS+=*Sx;

         E++;
         Lx++;
         Mx++;
         Mx1++;
         Sx++;
      } 
   /* Centroids: */
      *Lc=NL/D; 
      *Sc=NS/D;

      Lc++;
      Sc++;
      P++;
      R++;
   }
   return(drop2() && drop2() && drop() && 
      lpull() && lpull() /* Lx Sx distributions */ &&
      lpull() && lpull() /* Lc Sc centroids     */ ); 
}

int posn() /* posn (hC hIn hOut --- hN hC1) */ 
/* Array C contains contract prices on sequential steps, N contains
   the total number on each step.

   List In contains steps when contracts are opened, and list Out
   contains steps when contracts are closed.  There may be duplicates
   in these lists as more than one contract is opened or closed on the
   same step; and at the final step there may be contracts in In that 
   are still open, and so have no counterpart in Out.

   Returns vector N containing the total number of open contracts on 
   each step, and C1 containing the average price of open contracts on 
   each step.  

   On step k where there are no open contracts, N(k)==0 and C1(k) is 
   equal to C(k) for plotting.  If N(k-1)>0 and N(k)==0, contracts of 
   k-1 were closed on k at C(k).

   If list In is empty, null N and C1 = C are returned immediately. 

   Lists In and Out are treated like stacks.  Top of stack pointers 
   are pIn and pOut for In and Out, respectively.  Stack pointers are 
   initially zero, pointing to the beginning of the lists. 

   For In, depth of stack is nPos (initially zero) and max depth is 
   its length, nIn.  As step k is advanced and more values of In are 
   encountered that equal k, nPos (stack depth) grows.

   List Out controls the advancing of the top of stack pointer of In
   and decrementing nPos, the depth of stack of In.  The tos pointer 
   of Out is incremented until it equals max depth of Out, length nOut,
   meaning the Out stack is empty and the process can end. */
{
   int i,k=0,m,nC,nIn,nOut,nPos=0,pIn=0,pOut=0;
   double *C,*C1,*In,*N,*Out,sum;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" posn: ","expect three matrices on the stack");
      return 0;
   }
   pushint(XBASE);
   minus(); /* remove XBASE from the offsets in In */

   swap(); /* (hC hOut hIn) */ 

   pushint(XBASE);
   minus(); /* remove XBASE from the offsets in Out */

   In=tos->mat;
   nIn=tos->row;

   Out=(tos-1)->mat;
   nOut=(tos-1)->row;

   if(nIn<nOut) {
      stkerr(" posn: ","list In cannot be shorter than list Out");
      return 0;
   }
   if(nIn<1) return( /* list In is empty */
      drop2() && 
      cop()   && /* C1 is a copy of C */
      dup1s() &&
      dims()  &&
      null()  && /* null N */
      swap()     /* hN hC1 */
   ); 

   C=(tos-2)->mat;
   nC=(tos-2)->row;

   if(!matstk(nC,1,"_N")) return 0;
   N=tos->mat;
   memset(N,0,nC*sizeof(double)); /* initialize N to 0 */

   if(!matstk(nC,1,"_C1")) return 0;
   C1=tos->mat;
   memcpy(C1,C,nC*sizeof(double)); /* initialize C1 to C */

   for(;k<nC;k++) {
      if((int)*(In+pIn+nPos)==k) { /* k equals an offset in In */
      /* Opening positions: */
         i=0;
         while((pIn+i)<nIn && (int)*(In+pIn+i)<=k) i++;
         nPos=i; /* new In stack depth */
      }
      if(nPos) {
         sum=0;
         for(i=0;i<nPos;i++) {
            m=(int)*(In+pIn+i);
            sum+=*(C+m);
         }
         *(C1+k)=sum/nPos; /* average of open contracts */
         *(N+k)=nPos;
      }
      else {
         *(C1+k)=*(C+k);
         *(N+k)=0;
      }
      if(nPos>0 && nOut>0 && pOut<nOut) {
         if((int)*(Out+pOut)==k) { /* k equals an offset in Out */
         /* Closing positions: */
            while(pOut<nOut && (int)*(Out+pOut)==k) {
               pIn++; /* increment In top of stack pointer */
               nPos--; /* decrement In stack depth */
               pOut++; /* increment Out top of stack pointer */
            }
         }
         if(nPos) {
            sum=0;
            for(i=0;i<nPos;i++) {
               m=(int)*(In+pIn+i);
               sum+=*(C+m);
            }
            *(C1+k)=sum/nPos; /* average of open contracts */
            *(N+k)=nPos;
         }
         else {
            *(C1+k)=*(C+k);
            *(N+k)=0;
         }
      }
   }
   return(lpush() && lpush() && 
          drop2() && drop() && /* (hC hIn hOut) off stk */
          lpull() && lpull()
   );
}

int rolldelta() /* rolldelta (hK hC hChg --- hDelta) */
/* Used by word data() in mfil.v. */
{
   register double *C,*Chg,*Delta,dsum=0,*K;
   register int k;
   int rows;

   rows=tos->row;

   if(!matstk(rows,1,"_delta")) return 0;

   Delta=(tos)->mat;
   Chg=(tos-1)->mat;
   C=(tos-2)->mat;
   K=(tos-3)->mat;

   *(Delta+rows-1)=dsum;

   for(k=rows-2;k>-1;k--) {

      if(*(K+k)!=*(K+k+1)) dsum=dsum + *(C+k+1) - *(C+k) - *(Chg+k+1);

      *(Delta+k)=dsum;
   }
   return(lop() && lop() && lop());
}

int stage_origin() /* stage_origin (hV hP n --- hA) */
/* The n curves returned in A track the origin of levels in curve V 
   that began in curve P.  V(k) exists at P(m), where k-(n-1)<m<=k. 

   Example: P contains L, and V is ML.  ML(k) can be traced back to
   a previous level in L, say L(m), within the last n steps.

   This function is only valid when running in real time. */ 
{
   int i,j,k=0,m,n,rows;
   double *A,*A1,*P,*P0,*V;

   if(!popint(&n)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" stage_origin: ",MATNOT2);
      return 0;
   }
   if(tos->row!=(tos-1)->row) {
      stkerr(" stage_origin: "," rows of V and P must be equal");
      return 0;
   }
   rows=tos->row;
   if(rows<n) {
      stkerr(" stage_origin: n cannot exceed the rows of incoming V",\
         "");
      return 0;
   }
   P=P0=tos->mat;
   V=(tos-1)->mat;

   if(!matstk(rows,n,"_A")) return 0;
   A=tos->mat;

/* Initialize all columns of A to column V: */
   for(i=0;i<n;i++) memcpy(A+locvec(i,rows),V,sizeof(double)*rows);

   P+=rows-1;
   V+=rows-1;

   j=0;
   for(k=rows-1;k>rows-1-n;k--) {
      i=0;
      m=-1;
      while(i<n && m<0) {
         if(*(P-i)==*V) m=i;
         i++;
      }
      if(m<0) {
         gprintf(" stage_origin: match not found for row %d",k);
         nc();
         stkerr("","");
         return 0;
      }
   /* Set A(k-m,j):A(k-1,j) equal to V(k): */
      A1=A+locvec(j,rows)+k-m;
      for(i=0;i<m+1;i++) *(A1+i)=*V;

   /* Set all prior A in this column to prior P: */
      A1=A+locvec(j,rows);
      memcpy(A1,P0,sizeof(double)*(k-m));

      V--;
      P--;
      j++;
   }
   return(lop() && lop());
}

int stages() /* stages (hV n --- hA) */
/* Successive values of incoming vector V make a curve.  The columns
   of output matrix A are curves showing stages of the n previous 
   values of V. */
{ 
   int i,k=0,n,quot,rem,rows;
   double *A,*V;

   if(!popint(&n)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" stages: ",MATNOT);
      return 0;
   }
   V=tos->mat;
   rows=tos->row;
   if(rows<n) {
      stkerr(" stages: n cannot exceed the rows of incoming V","");
      return 0;
   }
/* Build transposed A: */
   if(!matstk(n,rows,"_A")) return 0;
   A=tos->mat;

/* The first n columns of A match the first n rows of V: */
   for(;k<n;k++) {
      memcpy(A,V,sizeof(double)*n);
      A+=n;
   }
   quot=(rows-n)/n;
   rem=(rows-n)-quot*n;

   for(k=0;k<quot;k++) {
      for(i=0;i<n;i++) {
         memcpy(A,A-n,sizeof(double)*n);
         *(A+i)=*(V+n+i); /* replace oldest */
         A+=n;
      }
      V+=n;
   }
   if(rem) {
      for(i=0;i<rem;i++) {
         memcpy(A,A-n,sizeof(double)*n);
         *(A+i)=*(V+n+i); /* replace oldest */
         A+=n;
      }
   }
   return(bend() && lop());
}

int _states1() /* _states1 (hP hD --- hS) */
/* Wed Apr 22 19:07:40 PDT 2009

   Special purpose function used by high level word states(), written
   for faster processing.

   Combines the work of state_vec() and nine calls (when D has nine
   columns) to looking().

   From P and D, compute states like state_vec().  Then loop over P and
   states and compute state curves, S with the logic of looking(). 

   Test problem:
      seed0 seedset

      7 1 random "P" book
      7 5 random bend colsort bend "D" book

    \ Create reference S using state_vec and looking:
      P D state_vec "ST" book
      P ST 1 = looking
      P ST 2 = looking
      P ST dup 3 = swap 4 = or looking \ middle states collapse
      P ST 5 = looking
      P ST 6 = looking
      5 parkn "S" book

    \ Create S using _states1 and compare with reference S:
      P D _states1 S - null? */
{
   int c2,cols,i=0,j,len,mid,rows;
   double *D,*P,*R,*S;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" _states1: ",MATNOT2);
      return 0;
   }
   rows=tos->row;
   cols=tos->col;
   mid=cols/2; /* middle states collapse */
   P=(tos-1)->mat;

   bend(); /* transpose D */
   pushd(-INF); 
   pushint(rows);
   clone();
   swap();
   pushd(INF); 
   pushint(rows);
   clone();
   pushint(3);
   pilen(); /* -INF;D;+INF */
   D=tos->mat;
   c2=cols+2; /* rows in D' on stack */

   over(); /* P to tos */
   bend();
   pushint(cols);
   repeat1(); /* initial S' on tos, all rows set to P */
   S=tos->mat;

   if(!matstk(cols,1,"_R")) return 0;
   len=(cols)*sizeof(double);
   R=tos->mat;

   for(i=1;i<rows;i++) {
      memcpy(R,S,len);
      P++;
      D+=c2;
      bsearchd(*P,D,c2,&j); 
      if(j>mid) j-=1; /* collapse the middle states */
      *(R+j)=*P;
      S+=cols;
      memcpy(S,R,len);
   }
   return(drop() && lop() && lop() && bend());
}

int state_vec() /* state_vec (hP hD --- hS) */
/* Fri Apr 17 14:51:22 PDT 2009

   For vector P and the vectors in the columns of D, return a vector 
   of state ID for each row of P.  

   Rows of matrix D are assumed to be sorted lowest to highest from
   left to right.

   State IDs returned in S range from 1 to nC+1 where nC is the number 
   of columns in D. 

   This function does the work of the following high level infix when
   D has 5 columns (minus is used because true flags are -1):

      S = null(rows(P), 1);
      S -= 1*(S==0 && P<D[*, 1]); // < D1
      S -= 2*(S==0 && P<D[*, 2]); // < D2
      S -= 3*(S==0 && P<D[*, 3]); // < D3
      S -= 4*(S==0 && P<D[*, 4]); // < D4
      S -= 5*(S==0 && P<D[*, 5]); // < D5
      S -= 6*(S==0);              // < INF */ 
{ 
   int cols,i,j=0,rows;
   double *D,*P,*S;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" state_vec: ",MATNOT2);
      return 0;
   }
   if(tos->row!=(tos-1)->row) {
      stkerr(" state_vec: ",MATSNOTC);
      return 0;
   }
   D=tos->mat;
   P=(tos-1)->mat;

   rows=tos->row;
   cols=tos->col;

   if(!matstk(rows,1,"_S")) return 0;
   S=tos->mat;
   memset(S,0,rows*sizeof(double));

   for(;j<cols;j++) {
      for(i=0;i<rows;i++) {
         if(*S==0 && *P<*D) *S=j+1;
         D++;
         S++;
         P++;
      }
      S=tos->mat;
      P=(tos-2)->mat;
   }
   S=tos->mat;
   for(i=0;i<rows;i++) {
      if(*S==0) *S=cols+1;
      S++;
   }
   return(lop() && lop());
}

int state_vec1() /* state_vec1 (hP hD hS0 r --- hS) */
/* Tue Apr 21 19:59:57 PDT 2009

   For vector P and the vectors in the columns of D, return a vector 
   of state ID for each row of P.  

   Rows of matrix D are assumed to be sorted lowest to highest from
   left to right.

   State IDs returned in S range from 1 to nC+1 where nC is the number 
   of columns in D. 

   This function is identical to state_vec() except that it operates
   only on the last r rows of P and D, and uses the values in S0 for
   the initial rows of returned S. 

   Using state_vec to verify this function:

      seed0 seedset

      10 1 random "P" book
      10 5 random bend colsort bend "D" book

      P D state_vec "ST" book
      P D ST 4 state_vec1 "ST1" book

      ST ST1 - null? (f) */
{ 
   int cols,i,j=0,r,r0,rows;
   double *D,*P,*S;

   if(!popint(&r)) return 0;
   r=MAX(r,0);

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" state_vec1: ",MATNOT3);
      return 0;
   }
   if(tos->row!=(tos-1)->row || tos->row!=(tos-2)->row) {
      stkerr(" state_vec: ",MATSNOTC);
      return 0;
   }
   rows=tos->row;
   cols=(tos-1)->col;

   r=MIN(r,rows);
   r0=rows-r;

   cop();
   S=tos->mat+r0;
   memset(S,0,r*sizeof(double));

   for(;j<cols;j++) {
      S=tos->mat+r0;
      P=(tos-2)->mat+r0;
      D=(tos-1)->mat+locvec(rows,j)+r0;
      for(i=0;i<r;i++) {
         if(*S==0 && *P<*D) *S=j+1;
         D++;
         S++;
         P++;
      }
   }
   S=tos->mat+r0;
   for(i=0;i<r;i++) {
      if(*S==0) *S=cols+1;
      S++;
   }
   return(lop() && lop());
}

/* End of functions for mday.n */

/*--------------------------------------------------------------------*/

/* Transient analysis */

int uncoupled() /* uncoupled (hB hK hP hqd0 hq0 step --- hqdd hqd hq) */
/*
c     Perform transient response analysis
c
c     Adapted from program Express subroutines transresp.f, integuc.f.
c
c     Integrate an uncoupled set of equations of the form:
c
c                      qdd + B*qd + K*q = P
c
c     using constant step and subject to initial conditions qd0, q0.
c
c     Incoming B and K are diagonal matrices stored as column vectors
c     of length m, P is an m by nt matrix of loads for nt time points,
c     where the time points range in value from 0 to step*(nt - 1)
c
c     Stack description:
c            (hB hK hP hqd0 hq0 step --- hqdd hqd hq)
c     where
c        step is time step between columns of p, sec
c        q0, qd0 are initial displacements, velocities
c        P is matrix of time varying loads
c        K, B are stiffness and damping for unity mass
*/
{
   double *B,*B0,*g,*g0,*h,*h0,*K,*K0,*P,*P0,*q,*q0,*qd,*qd0,*qdd,step;
   int err=1,i=0,j=1,ndof,nstep;

   if(!popd(&step)) return 0;

   B0=(tos-4)->mat;
   K0=(tos-3)->mat;
   P=(tos-2)->mat;

/* Get sizes from P: */
   ndof=(tos-2)->row;
   nstep=(tos-2)->col;

   lpush(); /* q0 to temp stack */
   lpush(); /* qd0 to temp stack */

   if(!matstk(ndof,nstep,"_qdd")) return 0;
   qdd=tos->mat;

   if(!matstk(ndof,nstep,"_qd")) return 0;
   qd=tos->mat;
   lpull(); /* qd0 */
   memcpy(qd,tos->mat,ndof*sizeof(double)); /* qd0 to 1st col */
   drop();

   if(!matstk(ndof,nstep,"_q")) return 0;
   q=tos->mat;
   lpull(); /* q0 */
   memcpy(q,tos->mat,ndof*sizeof(double)); /* q0 to 1st col */
   drop();

   if(!matstk(4,ndof,"_g")) return 0; /* working space, 4 rows/dof */
   g0=tos->mat;

   if(!matstk(4,ndof,"_h")) return 0; /* working space, 4 rows/dof */
   h0=tos->mat;

/* Compute initial accelerations and constants of integration: */
   for(;i<ndof;i++) {
      *(qdd+i)=*(P+i) - *(K0+i)*(*(q+i)) - *(B0+i)*(*(qd+i));
      RAMP0(&step,(B0+i),(K0+i),(g0+locvec(i,4)),(h0+locvec(i,4)),&err);
      if(err) {
         gprintf(" uncoupled: bad B or K data for mode %d",i);
         nc();
         stkerr("","");
         return 0;
      }
   }
   P0=P;
   P+=ndof;

   qdd+=ndof;

   qd0=qd;
   qd+=ndof;
  
   q0=q;
   q+=ndof;
  
/* Perform the integration (time points 2 thru nstep): */
   for(;j<nstep;j++) { /* loop over time steps */
      g=g0;
      h=h0;

      B=B0;
      K=K0;

      for(i=0;i<ndof;i++) { /* loop over dofs */
         RAMP(&step,P0,P,qd0,q0,g,h,B,K,qdd,qd,q);

         P0++;
         P++;

         g+=4;
         h+=4;

         B++;
         K++;

         qdd++;

         qd0++;
         qd++;

         q0++;
         q++;
      }
   }
   return(drop2() && /* g and h off stack */
          lpush() && lpush() && lpush() && /* q, qd, qdd on temp stk */
          drop() && drop2() && /* B, K, and P off stack */
          lpull() && lpull() && lpull() /* qd, qd, q on stk */
   );
}
/*--------------------------------------------------------------------*/

/* end wapp.c functions */
