/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* vmo.c  December 2000

   Copyright (c) 2000   D. R. Williamson

   Functions that use mgetty and vgetty for voice modem.

   The Makefile for tops will use the library and object files of the
   mgetty distribution (which includes vgetty) once it has been instal-
   led.  
   Sample scripts in usr/voice can be edited to create the 'include' 
   and 'library' files required.  Be sure and read the scripts for any 
   changes that need to be made to mgetty functions beforehand.

   Source for mgetty download:
      vgetty-maintainer@alphanet.ch

   The November 2002 distribution of mgetty:
      mgetty1.1.29-Nov25.tar.gz 983945 (bytes)

   Documentation:
      mgetty and vgetty man pages
      http://alpha.greenie.net/mgetty/index.html
      http://www.leo.org/~doering/mgetty/index.html
*/

#ifdef VMO

#include <stdio.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "lib.h"
#include "mem.h"
#include "term.h"
#include "tex.h"

#include "vmo.h"

#include <ugly.h>
#include <conf_mg.h>
#include <mg_utmp.h>
#include <version.h>

/*--------------------------------------------------------------------*/

/* 1. Only these definitions from mgetty.c are needed: */

   char    * Device;                    /* device to use */
   char    * DevID;                     /* device name withouth '/'s */

/* 2. A change required to file locks.c in the mgetty distribution.

   In .../mgetty/mgetty-1.1.29/locks.c, making an array static: 

   The following global array in locks.c causes seg fault when 
   makelock() is run:

      char     lock[MAXLINE+1];             // name of the lockfile //

   Before running script usr/voice/vlib to create the libraries, make
   it static and rerun 'make' in the Mgetty directory of locks.c to 
   create a new locks.o file:

      static     char     lock[MAXLINE+1];  // name of the lockfile //
*/

/* This completes the link to mgetty and vgetty. */

/*--------------------------------------------------------------------*/

#define MAXVM 255
char vm_device_reg[MAXVM+1]={0};

#define OFFSEC 86400
#define STOP_PLAYING 0x3000

/* Local status variables for stopping voice modem actions: */
static int vstop_dialing;

void setdog(int n)  
/* Setting the mgetty watchdog timer. */
{
   cvd.watchdog_timeout.d.i=n; 
   reset_watchdog();
   voice_check_events();
}

int vfprops() /* vfprops (hFile --- ) */
/* Properties for raw modem data file. */
{
   FILE *fd;
   rmd_header header;
   char name[5];
   int bits,compression,speed;

   if(!(fd=filehand(tos->mat,tos->row*tos->col))) {
      stkerr(" vfprops: ",FILHNDNOT);
      return 0;
   }
   fseek(fd,0,SEEK_SET); /* rewind */

   if (fread(&header, sizeof(rmd_header), 1, fd) != 1) {
      stkerr(" vfprops: ","could not read voice file header");
      return 0;
   }
   dup1s(); 
   named();
   notag();
   gprintf(" Properties of voice file handle %s",tos->tex); nc();
   drop();
   
   gprintf("    Modem: %s",header.voice_modem_type); nc(); 

   memcpy(name,header.magic,4);
   *(name+4)='\0';
   gprintf("    Type: %s",name); nc();

   compression = ntohs(header.compression);
   speed = ntohs(header.speed);
   bits = header.bits;
   gprintf("    Compression method: %d",compression); nc();
   gprintf("    Speed: %d",speed); nc();
   gprintf("    Bits: %d",bits); nc();

   fseek(fd,0,SEEK_SET); /* rewind */
   return(drop());
}

int vmcommand() /* vmcommand (qC qE --- f) */
/* Sending command C with expected answer E directly to the voice 
   modem. 

   Note: May return false (0) for commands that return values.

      For example sending this command produces erroneous false on
      the stack:

         [tops@gutter] ready # 'AT+VIT?' 'OK' vmcommand

          stack elements:
                0 number: 0
          [1] ok!

      The log file for this command implies that the mgetty function
      is looking at the returned value, 060, and not at the OK that
      follows:

         12/28 15:18:51  vmcommand: Sending command: AT+VIT?
         12/28 15:18:51   voice command: 'AT+VIT?' -> 'OK'
         12/28 15:18:51    vm: AT+VIT?
         12/28 15:18:51    ZyXEL 1496: 060
         12/28 15:18:51  vm: Invalid modem answer
         12/28 15:18:51    ZyXEL 1496: OK
*/
{
   if(*vm_device_reg==0) {
      stkerr(" vmcommand: ","no device registered");
      return 0;
   }
   if(tos->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" vmcommand: ",STRSNOT2);
      return 0;
   }
   lprintf(L_MESG,"%s: sending command: %s, %s","vmcommand", \
      (tos-1)->tex,tos->tex);

   if (voice_command((tos-1)->tex,tos->tex)!=VMA_FAIL) pushint(xTRUE);
   else pushint(xFALSE);

   return(lop() && lop());
}

int vmcompression() /* vmcompression (comp speed bits --- f) */
/* Set voice modem compression. */
{
   char *vm="vmcompression";
   int bits,compression,speed;

   if(*vm_device_reg==0) {
      stkerr(" vmcompression: ","no device registered");
      return 0;
   }
   if(!popint(&bits)) return 0;
   if(!popint(&speed)) return 0;
   if(!popint(&compression)) return 0;

   if(voice_modem->set_compression(&compression,&speed,&bits)== OK) {
     lprintf(L_NOISE, "%s: compression method 0x%04x,speed %d,bits %d",\
     vm,compression,speed,bits);
     pushint(xTRUE);

     if(1 || TRACE) {
        gprintf(" vmcompression: comp %d, speed %d, bits %d",\
           compression,speed,bits);
        nc();
     }
   }
   else { 
     gprintf(" vmcompression: illegal comp %d, speed %d, bits %d",\
        compression,speed,bits);
     nc();
     lprintf(L_NOISE, \
        "%s: illegal compression method 0x%04x,speed %d,bits %d",\
        vm,compression,speed,bits);
     pushint(xFALSE); 
   }
   return 1;
}  

int vmdiagnostics() /* vmdiagnostics ( --- ) */
/* Run voice modem diagnostic tests.
   Adapted from mgetty-1.1.22/voice/vm/main.c. */
{
   char *test_command[] = {"ATI", "ATI1", "ATI2", "ATI3",
      "ATI4", "ATI5", "ATI6", "ATI7", "ATI8", "ATI9",
      "AT+FMI?", "AT+FMM?", "AT+FMR?", NULL};
   char buffer[VOICE_BUF_LEN];
   TIO tio;
   int first,i;

   if(*vm_device_reg==0) {
      stkerr(" vmdiagnostics: ","no device registered");
      return 0;
   }
   gprintf("*\n* Diagnostics for device /dev/%s\n*\n",vm_device_reg);
   gprintf("* vgetty %s\n", vgetty_version);

   tio_get(voice_fd, &tio);
   gprintf("* port speed is set to %d baud.\n*\n\n",
      tio_get_speed(&tio));

   for (i = 0; test_command[i] != NULL; i++) {
      gprintf("%-7s --> ", test_command[i]);

      voice_command(test_command[i], "");
      first = TRUE;

      do {
         do { 
            if (voice_read(buffer) != OK) {
               gprintf("could not read from modem");
               return 0;
            }
         }
         while (strlen(buffer) == 0);

            if (first) first = FALSE;
            else gprintf("%-7s     ", " ");

            gprintf("%s\n", buffer);
      }
      while((voice_analyze(buffer,"OK|ERROR",TRUE) & VMA_USER)==0);
   }
   return 1;
}

int vmdial() /* vmdial (qS --- f) */
/* Dial the number in string S. */
{
   char *vm="vmdial";
   int ret;

   if(*vm_device_reg==0) {
      stkerr(" vmdial: ","no device registered");
      return 0;
   }
   if(tos->typ!=STR) {
      stkerr(" vmdial: ",STRNOT);
      return 0;
   }
   strchop();

   lprintf(L_MESG,"%s: dialing %s",vm,tos->tex);

   vstop_dialing=FALSE;

   if(!bufup()) return 0;

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=VOICE;
 
      if(voice_dial((void *)tos->tex)==OK && vstop_dialing==FALSE) {
         lprintf(L_MESG,"%s returning for: %s",vm,tos->tex);
         pushint(xTRUE);
      }
      else { 
         lprintf(L_MESG,"%s: failed",vm);
         pushint(xFALSE);
      }
   }
   else {
      lprintf(L_MESG,"%s: timed out",vm,tos->tex);
      pushint(xFALSE);
   }
   bufdn();
   lop();

   vstop_dialing=FALSE;

   if(ret==ABORT) longjmp(abortbuf,ABORT);
   return 1;
}

int vmevent(int event, event_data data)
/* Send the event to the high level event handler.

   Remember, this function is returning to an mgetty function, not this
   program's perform() function.

   See mgetty event.h for a list of event messages (also sys/vmo.v). */
{
   int f=0,ret=0;

   if(event==SIGNAL_SIGINT) {
      ret=voice_stop_current_action();
      voice_modem_state=IDLE;
      return(ret);
   }
   if(!volstk(1,sizeof(event_data),"_data")) return(UNKNOWN_EVENT);
   memcpy(tos->tex,&data,tos->col);

   pushint(voice_modem_state);
   pushint(event);

/* Running this program's high level event handler: 
      inline: VMEVENT (hData state event --- f) */

   pushq2("VMEVENT",7); main1();

   if(popint(&f)) {

      switch(f) {
         case -1: /* yes, its OK */
            return(OK);

         case 0: /* let the mgetty handler take care of this one */
            return(UNKNOWN_EVENT);

         case 1: /* run voice_stop_current_action() and return OK */
            voice_stop_current_action();
            voice_modem_state=IDLE;
            return(OK);

         case 2: /* run voice_stop_playing() and return OK */
            if(voice_modem_state==PLAYING) {
               voice_stop_playing();
               voice_modem_state=IDLE;
            }
            return(OK);

         case 3: /* run voice_stop_recording() and return OK */
            if(voice_modem_state==RECORDING) {
               voice_stop_recording();
               voice_modem_state=IDLE;
            }
            return(OK);

         default:
            return(UNKNOWN_EVENT);
      }
   }
   else return(UNKNOWN_EVENT);
}

int vmflow() /* vmflow (flow --- f) */
/*  Set modem data flow control. */
{
   TIO tio;
   int flow;

   if(tos->typ!=NUM) {
      stkerr(" vmflow: ",NUMNOT);
      return 0;
   }
   popint(&flow);

   lprintf(L_MESG,"%s: setting flow","vmflow");
   tio_get(voice_fd, &tio);

   if(flow==2) {
      if ((voice_command(voice_modem->hardflow_cmnd,
         voice_modem->hardflow_answr) & VMA_USER) != VMA_USER) {
         pushint(xFALSE);
         return 1;
      }
      tio_set_flow_control(voice_fd, &tio, FLOW_HARD | FLOW_XON_OUT);
   }
   else {
      if ((voice_command(voice_modem->softflow_cmnd,
         voice_modem->softflow_answr) & VMA_USER) != VMA_USER) {
         pushint(xFALSE);
         return 1;
      }
      tio_set_flow_control(voice_fd, &tio, FLOW_XON_OUT);
   }
   tio_set(voice_fd, &tio);
   pushint(xTRUE);
   return 1;
}

int vmlog() /* vmlog (qS --- ) */ 
/* Log a message into the voice modem log from a high level word.
   Note: logging level set in voice.conf must be 4 or higher. 

   Example:
      "vmdial: " "Dialing " "82-XXX-YYY-ZZZZ" cat cat vmlog 
   will appear in the voice modem log as
      12/28 23:26:49  vmdial: Dialing 82-XXX-YYY-ZZZZ
*/
{
   if(*vm_device_reg==0) {
      stkerr(" vmlog: ","no device registered");
      return 0;
   }
   lprintf(L_MESG,tos->tex);
   return(drop());
}

int vmopen(char *device) 
/* Open a voice modem device.
   Adapted from mgetty-1.1.22/voice/vm/main.c.
    
   Here is conf string definition from config.h:
      #define conf_set_string( cp, s ) { 
           (cp)->d.p = (s); (cp)->flags = C_OVERRIDE; }
*/
{
   int result = OK;
   int tries = 0;
   conf_set_string(&cvd.voice_devices, device);
   lprintf( L_MESG, "vmopen: opening device %s",cvd.voice_devices.d.p);

   voice_modem = &no_modem;
   rom_release = 0;

   while (((result = voice_open_device()) != OK) && ((++tries) <
      cvd.max_tries.d.i)) sleep(cvd.retry_delay.d.i);
   if(result!=OK) {
      lprintf( L_MESG, "vmopen: voice open failed");
      return 0;
   }
   voice_mode_on();
   lprintf( L_MESG, "vmopen: voice mode on");
   return 1;
}

int vmplay() /* vmplay (hFile secs --- f) */
/* Play raw modem data (rmd) formatted File. */
{
   FILE *fd;
   rmd_header header;
   int bits,ret,secs,speed;
   char *vm="vmplay";

   if(*vm_device_reg==0) {
      stkerr(" vmplay: ","no device registered");
      return 0;
   }
   if(!popint(&secs)) return 0;

   if(tos->typ!=MAT ||
      !(fd=filehand(tos->mat,tos->row*tos->col))) {
      stkerr(" vmplay: ",FILHNDNOT);
      drop();
      pushint(xFALSE);
      return 0;
   }
   drop();
   if (fread(&header, sizeof(rmd_header), 1, fd) != 1) {
      stkerr(" vmplay: ","could not read voice file header");
      pushint(xFALSE);
      return 0;
   }
   speed = ntohs(header.speed);
   bits = header.bits;

   if(!bufup()) {
      pushint(xFALSE);
      return 0;
   }
   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=VOICE;

      setdog(secs);

      if(TRACE) {
         gprintf(" vmplay: call start_play_file"); nc();
      }
      if(voice_modem->start_play_file!=NULL) {
         lprintf(L_MESG,"%s: start_play_file",vm);
         if(voice_modem->start_play_file()!=OK) {
            stkerr(" vmplay: ","start_play_file command failed");
            bufdn();
            pushint(xFALSE);
            setdog(OFFSEC);
            return 0;
         }
      }
      if(1 || TRACE) {
         gprintf(" vmplay: call play_file: speed %d, bits %d", \
            speed,bits); nc();
      }
      lprintf(L_MESG,"%s: play_file",vm);
      if(voice_modem->play_file(fd, speed * bits)!=OK) {
         stkerr(" vmplay: ","play_file command failed");
         bufdn();
         pushint(xFALSE);
         setdog(OFFSEC);
         return 0;
      }
      if(voice_modem->stop_play_file!=NULL) {

         vmevent(STOP_PLAYING,(event_data)NULL);

         if(1 || TRACE) {
            gprintf(" vmplay: call stop_play_file"); nc();
         }
         lprintf(L_MESG,"%s: stop_play_file",vm);
         if(voice_modem->stop_play_file()!=OK) {
            stkerr(" vmplay: ","stop_play_file command failed");
            bufdn();
            pushint(xFALSE);
            setdog(OFFSEC);
            return 0;
         }
      }
      pushint(xTRUE);
   }
   else { /* longjmp error */
      gprintf(" vmplay: long jump return\n");
      pushint(xFALSE);
   }
   bufdn();
   setdog(OFFSEC);

   if(ret==ABORT) longjmp(abortbuf,ABORT);

   return 1;
}

int vmrecord() /* vmrecord (hFile secs comp speed bits --- f) */
/* Record a raw modem data (rmd) formatted File. */
{
   FILE *fd;
   rmd_header header;
   int bits,compression=0,ret,secs,speed;

   if(*vm_device_reg==0) {
      stkerr(" vmrecord: ","no device registered");
      return 0;
   }
   if(!(
      dup1s() && popint(&bits) &&
      over() && popint(&speed) &&
      other() && popint(&compression)
   )) return 0;
   
   if(!vmcompression()) {
      drop2(); /* drop secs and flag from vmcompression() */
      pushint(xFALSE);
      return 0; 
   } 
   drop(); /* drop flag from vmcompression() */

   if(!popint(&secs)) return 0;

   if(tos->typ!=MAT ||
      !(fd=filehand(tos->mat,tos->row*tos->col))) {
      stkerr(" vmrecord: ",FILHNDNOT);
      drop();
      pushint(xFALSE);
      return 0;
   }
   drop(); /* ok to drop hFile because it is in the catalog */

   memset(&header, 0x00, sizeof(rmd_header));
   sprintf(header.magic, "%s", "RMD1");
   sprintf(header.voice_modem_type, "%s", voice_modem_rmd_name);
   header.compression = ntohs(compression);
   header.speed = ntohs(speed);
   header.bits = bits;

   if(fwrite(&header, sizeof(rmd_header), 1, fd) != 1) {
      stkerr(" vmrecord: ","could not write file header");
      pushint(xFALSE);
      return 0;
   }
   if(!bufup()) {
      pushint(xFALSE);
      return 0;
   }
   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=VOICE;

      setdog(secs);

      if(TRACE) {
         gprintf(" vmrecord: call start_record_file"); nc();
      }
      voice_modem->record_file(fd,speed*bits);
      pushint(xTRUE);
   }
   else { /* longjmp error */
      gprintf(" vmrecord: long jump return\n");
      pushint(xFALSE);
   }
   bufdn();
   setdog(OFFSEC);
   if(ret==ABORT) longjmp(abortbuf,ABORT);

   return 1;
}

int vmregister() /* vmregister (qDev --- f) */
/* Set up logging, read configuration file and open the voice modem 
   device.  The configuration file is probably:
      /usr/local/etc/mgetty+sendfax/voice.conf.

   Log file is /var/log/vm.log (same as mgetty vm program). 

   Opens the voice modem device denoted by Dev (like ttyS3). 

   For non-root to register, these files must allow non-owner writing:
      /var/log/vm.log 
      /var/log/mgetty.unknown
   and /var/lock must allow non-owner writing for
      /var/lock/LCK..ttySx,
   the lock file. */
{
   int fopen=0,ftry=0,len=0;

   if(tos->typ!=STR) {
      stkerr(" vmregister: ",STRNOT);
      pushint(xFALSE);
      return 0;
   }
   strchop();

   if(*vm_device_reg!=0) vmunregister();
   memcpy(vm_device_reg,tos->tex,len=(MIN(tos->col,MAXVM)));
   *(vm_device_reg+len)='\0';
   drop();

   check_system();
   voice_config("vm", ""); /* read the config file; vm names log */
   voice_register_event_handler(vmevent);

   if(!bufup()) return 0;

   setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

   fopen=0;

   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=VOICE;

      ftry=1;
      if(TRACE) {
         gprintf(" vmregister: calling vmopen, try %d",ftry); 
         nc();
      }
      fopen=vmopen(vm_device_reg);
   }
   else {
      if(!fopen && ftry<2) {
         ftry=2;
         if(TRACE) {
            gprintf(" vmregister: calling vmopen, try %d",ftry); 
            nc();
         }
         fopen=vmopen(vm_device_reg); /* try again */
      }
   }
   bufdn();

   if(!fopen) {
      lprintf( L_MESG, "vmregister: vmopen could not open device %s",\
         vm_device_reg);
      stkerr("","");
      gprintf(" vmregister: could not open device %s",vm_device_reg); 
      nc();
      *vm_device_reg='\0';
      pushint(xFALSE);
      return 0;
   }
   gprintf(" Device %s is open, using log file %s", \
         vm_device_reg,cvd.voice_shell_log.d.p); nc();
   lprintf( L_MESG, "vmregister: Device %s is open, using log file %s",\
         vm_device_reg,cvd.voice_shell_log.d.p); 
   pushint(xTRUE);

   lprintf( L_MESG, "vmregister: returning");
   return 1;
}

int vmstate() /* vmstate ( --- n) */
/* Returns -1 if no voice device is registered.

   These are execution states in handlers:
      0 DIALING
      1 IDLE
      2 INITIALIZING
      3 OFF_LINE
      4 ON_LINE
      5 PLAYING
      6 RECORDING
      7 WAITING
*/
{
   if(*vm_device_reg==0) return(pushint(-1));
   return(pushint(voice_modem_state));
}

int vmstop() /* vmstop ( --- ) */
/* Stop the current voice modem action. */
{
   voice_stop_current_action();
   voice_modem_state=IDLE;
   return 1;
}

int vmstruct() /* vmstruct ( --- ) */
/* Displaying elements of voice_modem struct (see hardware.h). */
{
   if(*vm_device_reg==0) {
      stkerr(" vmstruct: ","no device registered");
      return 0;
   }
   gprintf(" name: %s",voice_modem->name); nc();
   gprintf(" rmd_name: %s",voice_modem->rmd_name); nc();
   gprintf(" pick_phone_cmnd: %s, pick_phone_answr: %s", \
    voice_modem->pick_phone_cmnd, voice_modem->pick_phone_answr); nc();
   gprintf(" beep_cmnd: %s, beep_answr: %s", \
    voice_modem->beep_cmnd, voice_modem->beep_answr); nc();
   gprintf(" beep_timeunit: %d",voice_modem->beep_timeunit); nc();
   gprintf(" hardflow_cmnd: %s, hardflow_answr: %s", \
    voice_modem->hardflow_cmnd, voice_modem->hardflow_answr); nc();
   gprintf(" softflow_cmnd: %s, softflow_answr: %s", \
    voice_modem->softflow_cmnd, voice_modem->softflow_answr); nc();
   gprintf(" start_play_cmnd: %s, start_play_answr: %s", \
    voice_modem->start_play_cmnd, voice_modem->start_play_answr); nc();
   gprintf(" reset_play_cmnd: %02Xh %02Xh", \
     *voice_modem->reset_play_cmnd, *(voice_modem->reset_play_cmnd+1));
     nc();
   gprintf(" intr_play_cmnd: %02Xh %02Xh, intr_play_answr: %s", \
    *voice_modem->intr_play_cmnd, *(voice_modem->intr_play_cmnd+1), \
    voice_modem->intr_play_answr); nc();
   gprintf(" stop_play_cmnd: %02Xh %02Xh, stop_play_answr: %s", \
    *voice_modem->stop_play_cmnd, *(voice_modem->stop_play_cmnd+1), \
    voice_modem->stop_play_answr); nc();
   gprintf(" start_rec_cmnd: %s, start_rec_answr: %s", \
    voice_modem->start_rec_cmnd, voice_modem->start_rec_answr); nc();
   gprintf(" stop_rec_cmnd: %02Xh %02Xh, stop_rec_answr: %s", \
    *voice_modem->stop_rec_cmnd, *(voice_modem->stop_rec_cmnd+1), \
    voice_modem->stop_rec_answr); nc();
   gprintf(" switch_mode_cmnd: %s, switch_mode_answr: %s", \
    voice_modem->switch_mode_cmnd,voice_modem->switch_mode_answr); nc();
   gprintf(" ask_mode_cmnd: %s, ask_mode_answr: %s", \
    voice_modem->ask_mode_cmnd,voice_modem->ask_mode_answr); nc();
   gprintf(" voice_mode_id: %s",voice_modem->voice_mode_id); nc();
   gprintf(" play_dtmf_cmd: %s, play_dtmf_answr: %s", \
    voice_modem->play_dtmf_cmd,voice_modem->play_dtmf_answr); nc();
   gprintf(" play_dtmf_extra: %s",voice_modem->play_dtmf_extra); nc();
   
   return 1;
}

int vmunregister() /* vmunregister ( --- ) */
/* Unregister and close the voice modem device. */
{
   if(*vm_device_reg==0) return 1;

   *vm_device_reg='\0';

   voice_unregister_event_handler();
   voice_set_device(NO_DEVICE);
   voice_mode_off();
   voice_close_device();

   return 1;
}

int vmwatch() /* vmwatch (n --- ) */
/* Setting mgetty watchdog timer to n seconds in the future. */
{
   int t;

   if(tos->typ!=NUM) {
      stkerr(" vmwatch: ",NUMNOT);
      return 0;
   }
   popint(&t);
   setdog(t);
   return 1;
}
/*--------------------------------------------------------------------*/

#endif
