/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author and copyright holder of tag.c:  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
#define _XOPEN_SOURCE 500 /* snprintf (features.h) */
#include <stdio.h>
#include <string.h> /* strncmp */
#include "main.h"
#include "exe.h"
#include "ctrl.h"
#include "inpo.h"   /* gprintf, nc */
#include "tex.h"    /* strchop */
#include "stk.h"
#include "tag.h"

void dump_tag   (stkitem *x) { /* {{{1 */
    if (!(*x).tag) {
        gprintf("tag is NULL");
    } else {
        gprintf("tag bits set:  ");
        if (is_sparse    (x)) gprintf("[sparse]");
        if (is_complex   (x)) gprintf("[complex]");
        if (is_symm      (x)) gprintf("[symmetric]");
        if (is_low_tri   (x)) gprintf("[lower triangular]");
        if (is_up_tri    (x)) gprintf("[upper triangular]");
        if (is_factor_lu (x)) gprintf("[LU factor]");
        if (is_factor_l  (x)) gprintf("[L factor]");
        if (is_factor_u  (x)) gprintf("[U factor]");
        if (is_lapack    (x)) gprintf("[LAPACK]");
        if (is_mumps     (x)) gprintf("[MUMPS]");
        if (is_umfpack   (x)) gprintf("[UMFPACK]");
        if (is_spooles   (x)) gprintf("[SPOOLES]");
        if (is_symbolic  (x)) gprintf("[symbolic factor]");
        if (is_oo_core   (x)) gprintf("[stored out of core]");
        if (is_indexed   (x)) {
            if (is_index_used(x)) {
                              gprintf("[internally indexed]");
            } else {
                              gprintf("[internal index not used]");
            }
        }
        if (is_catptr    (x)) gprintf("[pointer to word in catalog]");
        if (is_dbh       (x)) gprintf("[SQLite database handle]");
        if (is_fileh     (x)) gprintf("[file handle]");
        if (is_port      (x)) gprintf("[port]");
        if (is_socket    (x)) gprintf("[socket]");
        if (is_client    (x)) gprintf("[client]");
    }
    nc();
} /* 1}}} */

int   is_sparse    (stkitem *x){ return  (*x).tag &  TAG_SPARSE      ? 1 : 0; }
void set_sparse    (stkitem *x){         (*x).tag |= TAG_SPARSE             ; }
void clr_sparse    (stkitem *x){         (*x).tag |= TAG_SPARSE             ; 
                                         (*x).tag ^= TAG_SPARSE             ; }

int   is_complex   (stkitem *x){ return  (*x).tag &  TAG_COMPLEX     ? 1 : 0; }
void set_complex   (stkitem *x){         (*x).tag |= TAG_COMPLEX            ; }
void clr_complex   (stkitem *x){         (*x).tag |= TAG_COMPLEX            ;
                                         (*x).tag ^= TAG_COMPLEX            ; }

int   is_symm      (stkitem *x){ return  (*x).tag &  TAG_SYMMETRIC   ? 1 : 0; }
void set_symm      (stkitem *x){         (*x).tag |= TAG_SYMMETRIC          ; }
void clr_symm      (stkitem *x){         (*x).tag |= TAG_SYMMETRIC          ;
                                         (*x).tag ^= TAG_SYMMETRIC          ; }

int   is_low_tri   (stkitem *x){ return  (*x).tag &  TAG_LOWER_TRI   ? 1 : 0; }
void set_low_tri   (stkitem *x){         (*x).tag |= TAG_LOWER_TRI          ; }
void clr_low_tri   (stkitem *x){         (*x).tag |= TAG_LOWER_TRI          ;
                                         (*x).tag ^= TAG_LOWER_TRI          ; }

int   is_up_tri    (stkitem *x){ return  (*x).tag &  TAG_UPPER_TRI   ? 1 : 0; }
void set_up_tri    (stkitem *x){         (*x).tag |= TAG_UPPER_TRI          ; }
void clr_up_tri    (stkitem *x){         (*x).tag |= TAG_UPPER_TRI          ;
                                         (*x).tag ^= TAG_UPPER_TRI          ; }

int   is_factor_lu (stkitem *x){ return  (*x).tag &  TAG_FACTOR_LU   ? 1 : 0; }
void set_factor_lu (stkitem *x){         (*x).tag |= TAG_FACTOR_LU          ; }
void clr_factor_lu (stkitem *x){         (*x).tag |= TAG_FACTOR_LU          ;
                                         (*x).tag ^= TAG_FACTOR_LU          ; }

int   is_factor_l  (stkitem *x){ return  (*x).tag &  TAG_FACTOR_L    ? 1 : 0; }
void set_factor_l  (stkitem *x){         (*x).tag |= TAG_FACTOR_L           ; }
void clr_factor_l  (stkitem *x){         (*x).tag |= TAG_FACTOR_L           ;
                                         (*x).tag ^= TAG_FACTOR_L           ; }

int   is_factor_u  (stkitem *x){ return  (*x).tag &  TAG_FACTOR_U    ? 1 : 0; }
void set_factor_u  (stkitem *x){         (*x).tag |= TAG_FACTOR_U           ; }
void clr_factor_u  (stkitem *x){         (*x).tag |= TAG_FACTOR_U           ;
                                         (*x).tag ^= TAG_FACTOR_U           ; }

int   is_symbolic  (stkitem *x){ return  (*x).tag &  TAG_SYMBOLIC    ? 1 : 0; }
void set_symbolic  (stkitem *x){         (*x).tag |= TAG_SYMBOLIC           ; }
void clr_symbolic  (stkitem *x){         (*x).tag |= TAG_SYMBOLIC           ;
                                         (*x).tag ^= TAG_SYMBOLIC           ; }

int   is_lapack    (stkitem *x){ return ((*x).tag & TAG_10_12) == TAG_LAPACK
                                                                    ? 1 : 0; }
void set_lapack    (stkitem *x){                     clr_lapack(x);
                                         (*x).tag |= TAG_LAPACK             ; }
void clr_lapack    (stkitem *x){         (*x).tag |= TAG_10_12              ;
                                         (*x).tag ^= TAG_10_12              ; }

int   is_mumps     (stkitem *x){ return ((*x).tag & TAG_10_12) == TAG_MUMPS
                                                                    ? 1 : 0; }
void set_mumps     (stkitem *x){                     clr_mumps(x);
                                         (*x).tag |= TAG_MUMPS              ; }
void clr_mumps     (stkitem *x){         (*x).tag |= TAG_10_12              ;
                                         (*x).tag ^= TAG_10_12              ; }

int   is_umfpack   (stkitem *x){ return ((*x).tag & TAG_10_12) == TAG_UMFPACK
                                                                    ? 1 : 0; }
void set_umfpack   (stkitem *x){                     clr_umfpack(x);
                                         (*x).tag |= TAG_UMFPACK            ; }
void clr_umfpack   (stkitem *x){         (*x).tag |= TAG_10_12              ;
                                         (*x).tag ^= TAG_10_12              ; }

int   is_spooles   (stkitem *x){ return ((*x).tag & TAG_10_12) == TAG_SPOOLES
                                                                    ? 1 : 0; }
void set_spooles   (stkitem *x){                     clr_spooles(x);
                                         (*x).tag |= TAG_SPOOLES            ; }
void clr_spooles   (stkitem *x){         (*x).tag |= TAG_10_12              ;
                                         (*x).tag ^= TAG_10_12              ; }

int   is_oo_core   (stkitem *x){ return  (*x).tag &  TAG_OUT_OF_CORE ? 1 : 0; }
void set_oo_core   (stkitem *x){         (*x).tag |= TAG_OUT_OF_CORE        ; }
void clr_oo_core   (stkitem *x){         (*x).tag |= TAG_OUT_OF_CORE        ;
                                         (*x).tag ^= TAG_OUT_OF_CORE        ; }

int   is_indexed   (stkitem *x){ return  (*x).tag &  TAG_INDEXED     ? 1 : 0; }
void set_indexed   (stkitem *x){         (*x).tag |= TAG_INDEXED            ; }
void clr_indexed   (stkitem *x){         (*x).tag |= TAG_INDEXED            ;
                                         (*x).tag ^= TAG_INDEXED            ; }

int   is_index_used(stkitem *x){ return  (*x).tag &  TAG_USE_INDEX   ? 1 : 0; }
void set_index_used(stkitem *x){         (*x).tag |= TAG_USE_INDEX          ; }
void clr_index_used(stkitem *x){         (*x).tag |= TAG_USE_INDEX          ;
                                         (*x).tag ^= TAG_USE_INDEX          ; }

int   is_catptr    (stkitem *x){ return ((*x).tag & TAG_15_18) == TAG_CATPTR
                                                                    ? 1 : 0; }
void set_catptr    (stkitem *x){                     clr_catptr(x);
                                         (*x).tag |= TAG_CATPTR             ; }
void clr_catptr    (stkitem *x){         (*x).tag |= TAG_15_18              ;
                                         (*x).tag ^= TAG_15_18              ; }

int   is_fileh     (stkitem *x){ return ((*x).tag & TAG_15_18) == TAG_FILEH 
                                                                    ? 1 : 0; }
void set_fileh     (stkitem *x){                     clr_fileh(x);
                                         (*x).tag |= TAG_FILEH              ; }
void clr_fileh     (stkitem *x){         (*x).tag |= TAG_15_18              ;
                                         (*x).tag ^= TAG_15_18              ; }

int   is_dbh       (stkitem *x){ return ((*x).tag & TAG_15_18) == TAG_DBH
                                                                    ? 1 : 0; }
void set_dbh       (stkitem *x){                     clr_dbh(x);
                                         (*x).tag |= TAG_DBH                ; }
void clr_dbh       (stkitem *x){         (*x).tag |= TAG_15_18              ;
                                         (*x).tag ^= TAG_15_18              ; }

int   is_port      (stkitem *x){ return ((*x).tag & TAG_15_18) == TAG_PORT
                                                                    ? 1 : 0; }
void set_port      (stkitem *x){                     clr_port(x);
                                         (*x).tag |= TAG_PORT               ; }
void clr_port      (stkitem *x){         (*x).tag |= TAG_15_18              ;
                                         (*x).tag ^= TAG_15_18              ; }

int   is_socket    (stkitem *x){ return ((*x).tag & TAG_15_18) == TAG_SOCKET
                                                                    ? 1 : 0; }
void set_socket    (stkitem *x){                     clr_socket(x);
                                         (*x).tag |= TAG_SOCKET             ; }
void clr_socket    (stkitem *x){         (*x).tag |= TAG_15_18              ;
                                         (*x).tag ^= TAG_15_18              ; }

int   is_client    (stkitem *x){ return ((*x).tag & TAG_15_18) == TAG_CLIENT
                                                                    ? 1 : 0; }
void set_client    (stkitem *x){                     clr_client(x);
                                         (*x).tag |= TAG_CLIENT             ; }
void clr_client    (stkitem *x){         (*x).tag |= TAG_15_18              ;
                                         (*x).tag ^= TAG_15_18              ; }

/* C functions */
int  get_set_tag(                 /* {{{1 */
                 int   get_set,   /* in  0 => get,  1 => set,  -1 => clr */
                 char *tag,       /* in */ 
                 int   tag_len,   /* in */
                 int  *is_set)    /* out */
    /*
     * called by words get_tag() and set_tag()
     */
{
#define size_T 160
    char    T[size_T+1];

    int     R = 0;  /* the result of is_() calls */

    if (get_set == 1) {

      if        (!strncmp(tag, "sparse"     ,tag_len)) { set_sparse(   tos-1);
      } else if (!strncmp(tag, "complex"    ,tag_len)) { set_complex(  tos-1);
      } else if (!strncmp(tag, "symmetric"  ,tag_len)) { set_symm(     tos-1);
      } else if (!strncmp(tag, "lower_tri"  ,tag_len)) { set_low_tri(  tos-1);
      } else if (!strncmp(tag, "upper_tri"  ,tag_len)) { set_up_tri(   tos-1);
      } else if (!strncmp(tag, "factor_lu"  ,tag_len)) { set_factor_lu(tos-1);
      } else if (!strncmp(tag, "factor_l"   ,tag_len)) { set_factor_l( tos-1);
      } else if (!strncmp(tag, "factor_u"   ,tag_len)) { set_factor_u( tos-1);
      } else if (!strncmp(tag, "out_of_core",tag_len)) { set_oo_core(  tos-1);
      } else if (!strncmp(tag, "lapack"     ,tag_len)) { set_lapack(   tos-1);
      } else if (!strncmp(tag, "symbolic"   ,tag_len)) { set_symbolic( tos-1);
      } else if (!strncmp(tag, "mumps"      ,tag_len)) { set_mumps(    tos-1);
      } else if (!strncmp(tag, "umfpack"    ,tag_len)) { set_umfpack(  tos-1);
      } else if (!strncmp(tag, "spooles"    ,tag_len)) { set_spooles(  tos-1);

      } else if (!strncmp(tag, "catptr"     ,tag_len)) { set_catptr(   tos-1);
      } else if (!strncmp(tag, "fileh"      ,tag_len)) { set_fileh(    tos-1);
      } else if (!strncmp(tag, "dbh"        ,tag_len)) { set_dbh(      tos-1);
      } else if (!strncmp(tag, "port"       ,tag_len)) { set_port(     tos-1);
      } else if (!strncmp(tag, "socket"     ,tag_len)) { set_socket(   tos-1);
      } else if (!strncmp(tag, "client"     ,tag_len)) { set_client(   tos-1);
      } else {
          snprintf(T, size_T, "'%s' is not a known tag type.  See src/tag.h",
                   tos->tex);
          stkerr(" set_tag: ", T);
          return 0;
      }

    } else if (get_set == -1) {

      if        (!strncmp(tag, "sparse"     ,tag_len)) { clr_sparse(   tos-1);
      } else if (!strncmp(tag, "complex"    ,tag_len)) { clr_complex(  tos-1);
      } else if (!strncmp(tag, "symmetric"  ,tag_len)) { clr_symm(     tos-1);
      } else if (!strncmp(tag, "lower_tri"  ,tag_len)) { clr_low_tri(  tos-1);
      } else if (!strncmp(tag, "upper_tri"  ,tag_len)) { clr_up_tri(   tos-1);
      } else if (!strncmp(tag, "factor_lu"  ,tag_len)) { clr_factor_lu(tos-1);
      } else if (!strncmp(tag, "factor_l"   ,tag_len)) { clr_factor_l( tos-1);
      } else if (!strncmp(tag, "factor_u"   ,tag_len)) { clr_factor_u( tos-1);
      } else if (!strncmp(tag, "out_of_core",tag_len)) { clr_oo_core(  tos-1);
      } else if (!strncmp(tag, "lapack"     ,tag_len)) { clr_lapack(   tos-1);
      } else if (!strncmp(tag, "symbolic"   ,tag_len)) { clr_symbolic( tos-1);
      } else if (!strncmp(tag, "mumps"      ,tag_len)) { clr_mumps(    tos-1);
      } else if (!strncmp(tag, "umfpack"    ,tag_len)) { clr_umfpack(  tos-1);
      } else if (!strncmp(tag, "spooles"    ,tag_len)) { clr_spooles(  tos-1);
      } else if (!strncmp(tag, "catptr"     ,tag_len)) { clr_catptr(   tos-1);
      } else if (!strncmp(tag, "fileh"      ,tag_len)) { clr_fileh(    tos-1);
      } else if (!strncmp(tag, "dbh"        ,tag_len)) { clr_dbh(      tos-1);
      } else if (!strncmp(tag, "port"       ,tag_len)) { clr_port(     tos-1);
      } else if (!strncmp(tag, "socket"     ,tag_len)) { clr_socket(   tos-1);
      } else if (!strncmp(tag, "client"     ,tag_len)) { clr_client(   tos-1);
      } else {
          snprintf(T, size_T, "'%s' is not a known tag type.  See src/tag.h",
                   tos->tex);
          stkerr(" clr_tag: ", T);
          return 0;
      }

    } else {

      if        (!strncmp(tag, "sparse"     ,tag_len)) {R=is_sparse(   tos-1);
      } else if (!strncmp(tag, "complex"    ,tag_len)) {R=is_complex(  tos-1);
      } else if (!strncmp(tag, "symmetric"  ,tag_len)) {R=is_symm(     tos-1);
      } else if (!strncmp(tag, "lower_tri"  ,tag_len)) {R=is_low_tri(  tos-1);
      } else if (!strncmp(tag, "upper_tri"  ,tag_len)) {R=is_up_tri(   tos-1);
      } else if (!strncmp(tag, "factor_lu"  ,tag_len)) {R=is_factor_lu(tos-1);
      } else if (!strncmp(tag, "factor_l"   ,tag_len)) {R=is_factor_l( tos-1);
      } else if (!strncmp(tag, "factor_u"   ,tag_len)) {R=is_factor_u( tos-1);
      } else if (!strncmp(tag, "out_of_core",tag_len)) {R=is_oo_core(  tos-1);
      } else if (!strncmp(tag, "lapack"     ,tag_len)) {R=is_lapack(   tos-1);
      } else if (!strncmp(tag, "symbolic"   ,tag_len)) {R=is_symbolic( tos-1);
      } else if (!strncmp(tag, "mumps"      ,tag_len)) {R=is_mumps(    tos-1);
      } else if (!strncmp(tag, "umfpack"    ,tag_len)) {R=is_umfpack(  tos-1);
      } else if (!strncmp(tag, "spooles"    ,tag_len)) {R=is_spooles(  tos-1);
      } else if (!strncmp(tag, "catptr"     ,tag_len)) {R=is_catptr(   tos-1);
      } else if (!strncmp(tag, "fileh"      ,tag_len)) {R=is_fileh(    tos-1);
      } else if (!strncmp(tag, "dbh"        ,tag_len)) {R=is_dbh(      tos-1);
      } else if (!strncmp(tag, "port"       ,tag_len)) {R=is_port(     tos-1);
      } else if (!strncmp(tag, "socket"     ,tag_len)) {R=is_socket(   tos-1);
      } else if (!strncmp(tag, "client"     ,tag_len)) {R=is_client(   tos-1);
      } else {
          snprintf(T, size_T, "'%s' is not a known tag type.  See src/tag.h",
                   tos->tex);
          stkerr(" get_tag: ", T);
          return 0;
      }
      *is_set = R;

    }

    return 1;
} /* 1}}} */
void int2bin(char bits[65], long long tag)  /* {{{1 */
    /*
     * return a string of .'s and 1's representing the bits of tag
     */
{
    int  i, nBits = 64;

    for (i = 0; i < nBits; i++) {
        if (tag % 2) bits[(nBits-1)-i] = '1';
        else         bits[(nBits-1)-i] = '.';
        tag >>= 1;
    }
    bits[nBits] = '\0';

    return;
} /* 1}}} */

/* words */
int  is_sparse_word()   /* is_sparse (hA --- f) {{{1 */
    /*
     * man entry:  is_sparse {{{2
     * (hA --- f) Puts 0 on the stack if the item on the stack is not a sparse matrix, -1 if it is a sparse matrix.
     * category: math::matrix::sparse, stack operator
     * related:  dense, sparse, is_complex, show_tag
     * 2}}}
     */
{
    if (!is_sparse(tos))
        pushint( 0);
    else
        pushint(-1);

    lop();

    return 1;
} /* 1}}} */
int  show_tag()  /* show_tag (hA ---  ) {{{1 */
    /*
     * man entry:  show_tag {{{2
     * (A --- ) Prints descriptions of the tag settings for A.
     * category: stack operator
     * related:  is_sparse, is_complex
     * 2}}}
     */
{
    dump_tag(tos);
    return drop();
} /* 1}}} */
int  set_tag()   /* set_tag  (hA qTag --- hA ) {{{1 */
    /*
     * man entry:  set_tag {{{2
     * (hA qTag --- hA) Changes the tag of A by setting the tag defined by the string qTag.  qTag can have values of the TAG_ definition names found in src/tag.h, in lower case, without the leading "TAG_" prefix.  Sample values for qTag are 'sparse', 'complex', 'symmetric', et cetera.
     * category: stack operator
     * related:  show_tag, get_tag, clr_tag
     * 2}}}
     */
{
    int junk = 0, result;

    if (tos->typ !=STR) {
        stkerr(" tag_set: ",STRSNOT);
        return 0;
    }
    strchop();
    result = get_set_tag(1, tos->tex, tos->col, &junk);
    if (result) {  /* only drop the item if the result was successful */
        drop(); 
    }
    return result;

} /* 1}}} */
int  clr_tag()   /* clr_tag  (hA qTag --- hA ) {{{1 */
    /*
     * man entry:  clr_tag {{{2
     * (hA qTag --- hA) Changes the tag of A by clearing, ie, unsetting, the tag defined by the string qTag.  qTag can have values of the TAG_ definition names found in src/tag.h, in lower case, without the leading "TAG_" prefix.  Sample values for qTag are 'sparse', 'complex', 'symmetric', et cetera.
     * category: stack operator
     * related:  show_tag, get_tag, set_tag
     * 2}}}
     */
{
    int junk = 0, result;

    if (tos->typ !=STR) {
        stkerr(" tag_set: ",STRSNOT);
        return 0;
    }
    strchop();
    result = get_set_tag(-1, tos->tex, tos->col, &junk);
    if (result) {  /* only drop the item if the result was successful */
        drop(); 
    }
    return result;

} /* 1}}} */
int  get_tag()   /* get_tag  (hA qTag --- f ) {{{1 */
    /*
     * man entry:  get_tag {{{2
     * (hA qTag --- f) Returns -1 if the tag of A defined by the string qTag is set; returns 0 otherwise.  qTag can have values of the TAG_ definition names found in src/tag.h, in lower case, without the leading "TAG_" prefix.  Sample values for qTag are 'sparse', 'complex', 'symmetric', et cetera.
     * category: stack operator
     * related:  show_tag, set_tag, clr_tag
     * 2}}}
     */
{
    int is_set = 0, result;

    if (tos->typ !=STR) {
        stkerr(" tag_set: ",STRSNOT);
        return 0;
    }
    strchop();
    result = get_set_tag(0, tos->tex, tos->col, &is_set);
    if (is_set) {
        pushint(xTRUE);
    } else {
        pushint(xFALSE);
    }
    if (result) {  /* only drop the item if the result was successful */
        lop(); 
        lop(); 
    }
    return result;

} /* 1}}} */
