/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author and copyright holder of sparse.c: Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
/* Contents   {{{1
   sparse.c  - Words for creating and manipulating sparse matrices.
      sparse       - creates a sparse matrix from a dense  matrix
      dense        - creates a dense  matrix from a sparse matrix
      is_sparse    - puts -1 on stack if tos is a sparse matrix; 0 otherwise
      speye        - create sparse identity matrix (name from matlab)
      spbend       - transpose sparse matrix
      sprand       - create sparse random   matrix (name from matlab)
      spdiag       - extracts diagonal from sparse matrix
      spy          - display pattern of non-zeros  (name from matlab)
      spdump       - print contents of sparse matrix arrays
      sperror      - perform tests on sparse matrix to see if indices are OK
      spnull       - create a null sparse matrix
      spscale      - multiply a sparse matrix by a scalar
      spadd        - add two sparse matrices
      spones       - replace existing numerical values with (real-only) 1.0
      spreal_imag  - extract separate Re,Im matrices from a complex sparse mat
      spparkn      - append n sparse matrices together into one sparse mat
      spconvert    - make sparse matrix from i,j,vr[,vi] in text file
      sp2ijv       - create matrix of i,j,vr[,vi] from sparse matrix
      sppilen      - vertically stack n sparse matrices together into one mat
      spcatch      - extract columns from a sparse matrix
      spreach      - extract rows    from a sparse matrix
      spcomb       - column-wise partition a sparse matrix into two matrices
      spmesh       - combine columns of two matrices according to partition

   Albert Danial  (pre-CVS:  Oct, Nov 2001)
 1}}} */
/* headers {{{1 */

#ifndef __STRICT_ANSI__
   #define __STRICT_ANSI__
#endif
#define _XOPEN_SOURCE 500 /* snprintf */
#include <stdio.h>   /* FILE */

#include <stdlib.h>  /* random, RAND_MAX */
#include <string.h>  /* memcpy */
#include <math.h>    /* fabs   */

#include "stk.h"
#include "math1.h"   /* SEED0 */
#include "main.h"
#include "exe.h"     /* xmain */
#include "tag.h"
#include "sparse.h"
#if defined(UMFPACK)
#include <umfpack.h>
#include "spsolve.h"
#endif

#include "ctrl.h"  /* halt */
#include "inpo.h"  /* gprintf, nl */
#include "mem.h"   /* volstk */


#ifndef MAX
#define  MAX(a,b)              ((a) >  (b)) ? (a)  : (b)
#endif
#ifndef MIN
#define  MIN(a,b)              ((a) <  (b)) ? (a)  : (b)
#endif
#define  INDEX_MAX(ia,a, ib,b) ((a) >  (b)) ? (ia) : (ib)
#define  INDEX_MIN(ia,a, ib,b) ((a) <  (b)) ? (ia) : (ib)
#define  TRUE      1
#define  FALSE     0
/*                 ((1<<31) - 1) gives overflow warning in sqrt()  */
#define  MAXINT    ((1<<30) - 1)
#define  MAXDOUBLE 1.0e+300
#ifdef BLAS
#ifdef LAPACK
#ifdef FORT_UDSC
#define DAXPY daxpy_
#define DCOPY dcopy_
#define DSCAL dscal_
#define DDOT  ddot_ 
#else
#define DAXPY daxpy
#define DCOPY dcopy
#define DSCAL dscal
#define DDOT  ddot
#endif
#endif
double   double_0    =  0.0;
double   double_1    =  1.0;
double   double_m1   = -1.0;
int      int_1       =   1;
int      int_2       =   2;
#endif

/* headers 1}}} */

/* words */
int  sparse()      /* sparse (hA --- hA_sp) {{{1 */
/* 
 *  Stack has a dense matrix; this converts it to sparse form.
 * man entry:  sparse {{{2
 * (hA --- hA_sp)  Create a sparse matrix from a dense matrix.  The storage scheme is efficient even if the dense matrix has no zeros.
 * category: math::matrix::sparse
 * related:  dense, spbend, sprand, sperror, is_sparse, spdump, speye, spdiag, spy, spnull, spadd, spmult, splu, spfbs, spscale, spones, spreal_imag, spparkn, spconvert, sp2ijv, sppilen, spcatch, spreach, spcomb, sprake, spmesh, spsubmat, partition, spinfo, spsum, spflip_sym 
 * 2}}}
 */
{
int DEBUG = 0;
    SparseMatrix m;

    char   *name = "_sparse";
    int     ncols, nrows, c, r, n_str_this_col, prev_nonzero,
            iS, iN, indexed,
            lower_tri     = 1,
            upper_tri     = 1,
            curr_str_len  = 0,
            num_size      = 0,
            nstr          = 0,
            cmplx         = 0;
    double *dense_A;

    if (tos->typ != MAT) {
        stkerr(" sparse: ", MATNOT);
        return 0;
    }
    indexed = is_indexed(tos); 
    if (is_complex(tos)) 
        cmplx   = 1;
    nrows   = tos->row;
    if (cmplx)
        nrows /= 2;
    ncols   = tos->col;
    dense_A = tos->mat;

if (DEBUG)
printf("top of sparse for %d x %d\n", nrows, ncols);
    /* Pass 1: determine the number of strings and non-zero values. */
    for (c = 0; c < ncols; c++) {
        prev_nonzero   = 0;  /* 1 == previous term in column was non-zero */
        n_str_this_col = 0;
        if (cmplx) {
            for (r = 0; r < nrows; r++) {
                if ((fabs(dense_A[2*r+c*2*nrows]  ) > SPARSE_ZERO_THRESH) ||
                    (fabs(dense_A[2*r+c*2*nrows+1]) > SPARSE_ZERO_THRESH)) {
                    ++num_size;
                    if (prev_nonzero) { /* (r,c) in same string as (r-1,c) */
                        ++curr_str_len;
                    } else {            /* (r,c) begins a new string */
                        ++nstr;
                        ++n_str_this_col;
                        curr_str_len = 1;
                    }
                    if (r < c) lower_tri = 0;
                    if (r > c) upper_tri = 0;
                    prev_nonzero = 1;
                } else {
                    prev_nonzero = 0;
                }
            }
        } else {
            for (r = 0; r < nrows; r++) {
                if (fabs(dense_A[r+c*nrows]) > SPARSE_ZERO_THRESH) {
                    ++num_size;
                    if (prev_nonzero) { /* (r,c) in same string as (r-1,c) */
                        ++curr_str_len;
                    } else {            /* (r,c) begins a new string */
                        ++nstr;
                        ++n_str_this_col;
                        curr_str_len = 1;
                    }
                    if (r < c) lower_tri = 0;
                    if (r > c) upper_tri = 0;
                    prev_nonzero = 1;
                } else {
                    prev_nonzero = 0;
                }
            }
        }
if (DEBUG)
printf("sparse Pass 1 col=%d  nstr=%d num_size=%d\n", c, nstr, num_size);
    }
    if (!sparse_stk(nrows      , /* in  */ 
                    ncols      , /* in  */
                    nstr       , /* in  number of strings       */
                    num_size   , /* in  number of nonzero terms */
                    cmplx      , /* in  0=real  1=complex       */
                    indexed    , /* in  0=no    1=yes           */
                    name       , /* in  */
                   &m)) {        /* out */
        return 0;
    }
    if (lower_tri) set_low_tri(tos);
    if (upper_tri) set_up_tri( tos);
    if (indexed) {
        set_indexed(tos);
        /* copy indices from the dense matrix to the sparse matrix */
        memcpy(m.row_idx, MAT_ROW_IDX(tos-1), nrows*sizeof(int));
        memcpy(m.col_idx, MAT_COL_IDX(tos-1), ncols*sizeof(int));
    }

    /* Pass 2: populate the sparse pointer and data arrays */
    iS       =-1;
    iN       = 0;
    for (c = 0; c < ncols; c++) {
if (DEBUG)
printf("d c=%d m.S_start[c]=%d\n", c, m.S_start[c]);
        stringify_vector(nrows,             /* in     number of rows */
                         c,                 /* in     column index */
                        &dense_A[c*nrows*NUM_PER_TERM(cmplx)],  
                                            /* in     the dense vector */
                         cmplx,             /* in     0=real   1=complex */
                        &m,                 /* in/out */
                        &iS,                /* in/out index to S */
                        &iN);               /* in/out index to N */
if (DEBUG) {
printf("sparse: ncols=%d  iS[%d]=%d   iN[%d]=%d\n", ncols, c,iS, c,iN);
printf("sparse: m.S[%d].start_row=%d m.S[%d].len=%d m.S[%d].N_idx=%d\n",
iS, m.S[iS].start_row, iS, m.S[iS].len, iS, m.S[iS].N_idx);
}
    }

if (DEBUG) dump_sp_detail(m);
    lop(); /* drop dense matrix from stack */

    return 1;
} /* 1}}} */
int  spdump()      /* spdump (hA_sp --- hA_sp) {{{1 */
/*
 * man entry:  spdump {{{2
 * (hA_sp --- hA_sp) Intended only for debugging sparse code; prints entire sparse matrix data structure.
 * category: math::matrix::sparse
 * related:  dense, spbend, spadd, sprand, spy, spconvert, spdiag, spinfo
 * 2}}}
 */
{
    if (!is_sparse(tos)) {
        stkerr(" spdump:", "tos not sparse, nothing done");
        return 0;
    } else {
        f_sp_dump("sp dump:", tos);
    }
    return 1;
} /* 1}}} */
int  dense()       /* dense (hA_sp --- hA) {{{1 */
/* 
 * man entry:  dense {{{2
 * (hA_sp --- hA) Stack has a sparse matrix; this converts it to dense form.
 * category: math::matrix::sparse
 * related: sparse, spbend, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
int DEBUG = 0;
    char   *name = "_dense";
    SparseMatrix m;

    int     n_ptr, c, r, s, i, end_row, nrows, indexed;
    double *dense_A;

    if (!is_sparse(tos)) {
        stkerr(" dense: ",SPARSENOT);
        return 0;
    }
    indexed = is_indexed(tos);

    m = sparse_overlay(tos);
if (DEBUG) gprintf("dense nrows=%d  ncols=%d\n", m.H[ROWS], m.H[COLS]);

    /* Allocate a nrows x ncols dense matrix and put it on the stack */
    /* A dense complex matrix is seen internally as a matrix of size */
    /* 2*nrows x ncols                                               */
    nrows = NUM_PER_TERM(m.H[COMPLX]) * m.H[ROWS];
    if (indexed) {
        if (!matstk_idx(nrows, m.H[COLS], name)) return 0;
        /* copy indices from the sparse matrix to the dense matrix */
        memcpy(MAT_ROW_IDX(tos), m.row_idx, m.H[ROWS]*sizeof(int));
        memcpy(MAT_COL_IDX(tos), m.col_idx, m.H[COLS]*sizeof(int));
    } else {
        if (!matstk(    nrows, m.H[COLS], name)) return 0;
    }
    dense_A = tos->mat;
    if (m.H[COMPLX])
        set_complex(tos);
    for (i = 0; i < nrows*m.H[COLS]; i++) { dense_A[i] = (double) 0.0; }

    /* populate the dense matrix with data from the sparse matrix */
    n_ptr = 0;
    for (c = 0; c < m.H[COLS]; c++) {
        for (s = m.S_start[c]; s < m.S_start[c+1]; s++) {
            end_row = m.S[s].start_row + m.S[s].len - 1;
            for (r = m.S[s].start_row; r <= end_row; r++) {
                if (m.H[COMPLX]) {
                    dense_A[2*(r + c*m.H[ROWS])  ] = m.N[n_ptr++];
                    dense_A[2*(r + c*m.H[ROWS])+1] = m.N[n_ptr++];
                }
                else
                    dense_A[r + c*m.H[ROWS]] = m.N[n_ptr++];
            }
        }
    }

    lop(); /* drop sparse matrix from stack */
    return 1;
} /* 1}}} */
int  speye()       /* speye (n --- hI_sp) {{{1 */
/* 
 *   Puts an n x n sparse identity matrix on the stack.
 * man entry:  speye {{{2
 * (n --- hI_sp) Puts an n x n sparse identity matrix on the stack.
 * category: math::matrix::sparse
 * related: dense, sparse, spbend, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
    return f_speye(0);
} /* 1}}} */
int  spbend()      /* spbend (hA_sp --- hA_sp') {{{1 */
/* 
 *   Transpose a sparse matrix.  
 *      Memory requirements:  3*nNZ for working arrays + size of tranposed mat.
 *      Complexity:           O(nNZ*log(nNZ))
 * man entry:  spbend {{{2
 * (hA_sp --- hA_sp') Transposes a sparse matrix.  Memory requirements:  3*nNZ for working arrays + size of tranposed mat.  Complexity:  O(nNZ*log(nNZ))
 * category: math::matrix::sparse
 * related: dense, sparse, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
int DEBUG = 0;
    int          i, c, transpose, cmplx, indexed;
    IJVMatrix    IJV;
    SparseMatrix m, B;
    char   *name = "_spbend";

    if (!is_sparse(tos)) {
        stkerr(" spbend: ",SPARSENOT); 
        return 0;
    }
if (DEBUG) spdump();
    m       = sparse_overlay(tos);
    cmplx   = m.H[COMPLX];
    indexed = is_indexed(tos);

    if (!m.H[n_NONZ]) {  /* transpose null matrix and return {{{2 */
        /* Transpose a null matrix.  This is more involved than it
         * sounds because we still have to allocate memory for the
         * column pointers S_start[] and N_start[]. 
         */
        if (!sparse_stk(m.H[COLS]  , /* in (number of rows)         */ 
                        m.H[ROWS]  , /* in (number of columns)      */
                        0          , /* in  number of strings       */
                        0          , /* in  number of nonzero terms */
                        0          , /* in  0=real  1=complex       */
                        indexed    , /* in  0=no    1=yes           */
                        name       , /* in  */
                       &B)) {        /* out */
            return 0;
        }
        for (c = 0; c <= m.H[ROWS]; c++) {
            B.S_start[c] = 0;
            B.N_start[c] = 0;
        }
        if (indexed) {
            /* copy indices from the input matrix; swap row/col */
            memcpy(B.row_idx, m.col_idx, m.H[COLS]*sizeof(int));
            memcpy(B.col_idx, m.row_idx, m.H[ROWS]*sizeof(int));
        }
        lop(); /* drop original sparse matrix from stack */
        return 1;
    } /* 2}}} */

    /* Traverse the indexing information and populate an integer matrix
       IJV, with a column for each non-zero term
       Example:  input matrix = [  0   1.7  -2.4  0  ]
                                [ 0.3   0     0  6.6 ]
                                [ 2.8  3.1    0   0  ]
                                [ 2.7   0     0   0  ]
          then
            IJV = [ 1   2   3   0   2    0   1 ]      <- row indices
                  [ 0   0   0   1   1    2   3 ]      <- col indices
                  [0.3 2.8 2.7 1.7 3.1 -2.4 6.6]
          
     */

if (DEBUG) gprintf("spbend BEFORE sort\n");
    f_sp2ijv(m, indexed, &IJV);

    /* Now sort IJV in order of ascending row value (ie, item in the
     * first row of each column).  Where row values are equal, do
     * secondary sort on column values.
       using the example above, expect the sorted IJV to be
            IJV = [ 0    0   1   1   2   2   3 ]      <- row indices
                  [ 1    2   0   3   0   1   0 ]      <- col indices
                  [1.7 -2.4 0.3 6.6 2.8 3.1 2.7]
     */
    qsort(IJV.d, m.H[n_NONZ], sizeof(ijv), &comp_IJV_rc);
if (DEBUG) gprintf("spbend AFTER  sort\n");
if (DEBUG)
for (i = 0; i < m.H[n_NONZ]; i++) {
gprintf("IJV.d[%2d] r,c,v= %3d %3d % 12.6e % 12.6e\n",
i, IJV.d[i].row, IJV.d[i].col, IJV.d[i].Re, IJV.d[i].Im);}

    /* 
     * Put a sparse matrix on the stack using the given IJV pattern.
     */
    transpose = TRUE;
    if (!f_ijv2sp("_spbend", IJV, indexed, transpose)) return 0;

    /* 
     * For complex transpose, return Hermetian, ie complex conjugate.
     */
    B = sparse_overlay(tos);
    if (B.H[COMPLX])
        for (i = 0; i < B.H[n_NONZ]; i++)
            B.N[2*i + 1] = -B.N[2*i + 1];

    if (indexed) {
        /* copy indices from the input matrix; swap row/col */
        memcpy(B.row_idx, m.col_idx, m.H[COLS]*sizeof(int));
        memcpy(B.col_idx, m.row_idx, m.H[ROWS]*sizeof(int));
    }
    lop(); /* drop original sparse matrix from stack */

    return 1;
} /* 1}}} */
int  sprand()      /* {{{1 sprand (r c rho --- hA_sp) */
/*  
 *   Creates a r x c sparse matrix with a density of rho, 
 *   where 0.0 <= rho <= 1.0
 * man entry:  sprand {{{2
 * (r c rho --- hA_sp) Creates a r x c sparse matrix with a density of rho, where 0.0 <= rho <= 1.0
 * category: math::matrix::sparse
 * related: dense, sparse, spadd, spbend, spy, spconvert, spdiag
 * 2}}}
 */
{
int DEBUG = 0;
    char   *name = "_sprand";
    int    *shuffled_list, ncols, nrows, i, num_size,
            status    = 1;   /* 1 = good    0 = bad */
    IJVMatrix IJV;
    double  rho /* , *N */ ;

    if (!popd(  &rho))   return 0;
    if (!popint(&ncols)) return 0;
    if (!popint(&nrows)) return 0;
if (DEBUG)
gprintf("sprand Rows=%d  Cols=%d  density=%le\n", nrows, ncols, rho);

    if ((ncols <= 0) || (nrows <= 0)) {
        stkerr(" sprand: ",SEEDBAD);
        return 0;
    }
    rho = fabs(rho);
    if (rho > 1.0) {
        rho = 1.0;
    }

    /* order the product R*C*rho to avoid overflow */
    num_size = (int) (nrows * (ncols * rho) + 0.5); /* round to nearest int */
    if (!num_size) {
        /* density is so low that there are no terms at all; call spnull *
         * to put a null matrix on the stack                             */
        pushd(nrows);
        pushd(ncols);
        spnull();
        return 1;
    }

    if (!malloc_IJVMatrix(&IJV, nrows, ncols, num_size, 0)) {
        stkerr(" sprand: (IJV) ",MEMNOT);
        return 0;
    }

    if ((shuffled_list = (int *) malloc(num_size * sizeof(int))) == NULL) {
        stkerr(" sprand: (shuffled_list) ",MEMNOT);
        free_IJVMatrix(IJV);
        return 0;
    }
    /* The array shuffled_list[0..(num_size-1)] will contain integers on
     * the range 0..(nrows*ncols-1).  Each integer i in the array
     * represents a unique row, column index pair such that
     *    row_index = i % nrows
     *    col_index = i / nrows
     */
    for (i = 0; i < num_size; i++) {
        shuffled_list[i] = i;
    }
    random_subset(nrows * ncols, num_size,  shuffled_list);
if (DEBUG) {
gprintf("shuffled_list: ");
for (i = 0; i < num_size; i++) { printf("%3d ", shuffled_list[i]); }
gprintf("\n");
}

    for (i = 0; i < num_size; i++) {
        IJV.d[i].row = shuffled_list[i] % nrows;                           /*R*/
        IJV.d[i].col = (int) ((double) shuffled_list[i] / (double) nrows); /*C*/
        IJV.d[i].Re  = rand0(&SEED);
        IJV.d[i].Im  = 0.0;
    }
if (DEBUG) {
gprintf("before sort:\n");
for (i = 0; i < num_size; i++) { 
    printf("shuff[i]=%2d r=%2d c=%2d  n[%2d]=%le\n", 
    shuffled_list[i], IJV.d[i].row, IJV.d[i].col, i, IJV.d[i].Re); 
}
}
    free(shuffled_list);

    qsort(IJV.d, num_size, sizeof(ijv), &comp_IJV_cr);
if (DEBUG) {
gprintf("after sort:\n");
for (i = 0; i < num_size; i++) { 
    printf("r=%2d c=%2d  n[%2d]=%le\n", 
    IJV.d[i].row, IJV.d[i].col, i, IJV.d[i].Re); 
}
}

    /* 
     * Put a sparse matrix on the stack using the given IJV pattern.
     */
    status = f_ijv2sp(name ,  /* in  */
                      IJV  ,  /* in  (memory freed)            */
                      0    ,  /* in  0 = no internal indexing  */
                      0    ); /* in  0 = don't transpose       */
    return status;
} /* 1}}} */
int  sperror()     /* sperror (hA_sp --- hA_sp e ) {{{1 */
/*
 * man entry:  sperror {{{2
 * (hA_sp --- hA_sp e) Performs a series of sanity checks on the sparse matrix data structure.  If all checks pass, puts a 0 on the stack.  Nonzero values are error codes; check the source code for their meaning.
 * category: math::matrix::sparse
 * related: dense, sparse, spadd, spbend, spy, spconvert
 * 2}}}
 */
{   int err = -1;
    if (!is_sparse(tos)) {
        printf("tos not sparse, nothing done\n");
    } else {
        err = f_sp_error(tos);
    }
    pushint(err);
    return 1;
} /* 1}}} */
int  spdiag()      /* spdiag (hA_sp --- hD) {{{1 */
/* 
 *   Extract diagonal from a sparse matrix.
 * man entry:  spdiag {{{2
 * (hA_sp/hD --- hD/hA_sp) If the input is a sparse matrix, extract diagonal from it and put it on the stack as a (dense) vector.  If the input is a dense matrix with either one row or one column, create a square sparse matrix and insert the terms of the dense vector into the diagonal terms of the sparse matrix. If [A] is a sparse factor matrix, returns the diagonal of of the upper triangle (the lower triangle is presumed to have unit diagonal terms).  One useful application of the latter is to determine the Sturm count of the factor of the dynamic matrix [K-sM].
 * FIXME:  does not work for complex input
 * category: math::matrix::sparse
 * related: dense, sparse, spadd, spbend, spy, spconvert, splu, diag
 * 2}}}
 */
{
    char   *name_dense  = "_diag";
    char   *name_sparse = "_spdiag";
    int     i, c, s, r, nDiag, done, n_ptr, end_row, diag_ptr,
            mode = 0, status = 0, cmplx = 0, indexed = 0;
    double *diag = 0;
#if defined(UMFPACK)
    void   *Numeric;
    int     error;
#endif
    char    T[ERR_MSG_SIZE+1];
    SparseMatrix m, D;

    if (is_sparse(tos)) {
        mode = 1; /* an ordinary sparse matrix, pull out diagonal */
    } else if (is_factor_lu(tos) && is_umfpack(tos)) {
        mode = 2; /* an LU factor from UMFPACK */
    } else if (tos->typ == MAT && (tos->row == 1 || tos->col == 1)) {
        mode = 3; /* a dense vector; make a square sparse matrix with these
                     terms on the diagonal */
    } else {
        stkerr(" spdiag: ","input is not sparse, not a dense vector, and "
                           "not a sparse LU factor");
        return 0;
    }
    cmplx   = is_complex(tos);
    indexed = is_indexed(tos);
    switch (mode) {
    case 1: /* extract diagonal from a sparse matrix */
        m = sparse_overlay(tos);

        nDiag = MIN(m.H[ROWS], m.H[COLS]);

        /* allocate a vector to contain the diagonal; put it on the stack */
        if (cmplx) nDiag *= 2;
        if (indexed) {
            if (!matstk_idx(nDiag, 1, name_dense)) return 0;
            memcpy(MAT_ROW_IDX(tos), m.row_idx, nDiag*sizeof(int));
        } else {
            if (!matstk(    nDiag, 1, name_dense)) return 0;
        }
        if (cmplx) set_complex(tos);
        diag = tos->mat;
        for (i = 0; i < nDiag; i++) { diag[i] = (double) 0.0; }

        /* populate the dense matrix with data from the sparse matrix */
        n_ptr    = 0;
        diag_ptr = 0;
        for (c = 0; c < m.H[COLS]; c++) {
            n_ptr = m.N_start[c];
            done  = FALSE;
            for (s = m.S_start[c]; s < m.S_start[c+1]; s++) {
                end_row = m.S[s].start_row + m.S[s].len - 1;
                for (r = m.S[s].start_row; r <= end_row; r++) {
                    if (r == c) {
                        diag[diag_ptr++] = m.N[n_ptr];
                        if (cmplx) diag[diag_ptr++] = m.N[n_ptr+1];
                        done    = TRUE;
                        break;
                    }
                    ++n_ptr;
                }
                if (done) 
                    break;
            }
            if (c >= m.H[ROWS]) {
                break;
            }
        }
        status = 1;
        break;
#if defined(UMFPACK)
    case 2:
        error = umfpack_numeric_from_tos(&Numeric);
        if (error < 0) {
            snprintf(T, ERR_MSG_SIZE, 
                     " umfpack_numeric_from_tos error %d ", error);
            stkerr(" spdiag:  ", T);
            return 0;
        }
        nDiag = (int) tos->real; /* ref: comment in umfpack_numeric_to_tos */

        /* allocate a vector to contain the diagonal; put it on the stack */
        if (!matstk(nDiag, 1, name_dense)) return 0;
        diag = tos->mat;
        for (i = 0; i < nDiag; i++) { diag[i] = (double) 0.0; }

        status = umfpack_di_get_numeric(NULL, NULL, NULL, NULL, NULL, NULL,
                                        NULL, NULL, diag, NULL, NULL, Numeric);
        if (status == UMFPACK_OK) {
            status = 1;
        }
        break;
#endif
    case 3:  /* create a square sparse matrix and put in its diagonal
                terms the values from the dense vector */
        r = tos->row;
        if (cmplx) r /= 2;
        c = MAX(r, tos->col);
        pushint(c);
        f_speye(indexed);  /* sparse identity matrix */
        if (cmplx) {
            dup1s();
            spadd_cx();    /* ( D I_sp I_sp --- D I_sp_cmplx ) */
        }
        D = sparse_overlay(tos);

        /* overlay the dense vector onto the numeric terms of the sparse diag */
        memcpy(D.N, (tos-1)->mat, (tos-1)->row*(tos-1)->col * sizeof(double) );
        if (indexed) {
            memcpy(D.row_idx, MAT_ROW_IDX(tos-1), r*sizeof(int));
            memcpy(D.col_idx, MAT_COL_IDX(tos-1), c*sizeof(int));
        }
        pushstr(name_sparse); naming();
        status = 1;
        break;

    default:
        snprintf(T, ERR_MSG_SIZE, " unknown case %d ", mode);
        stkerr(" spdiag:  ", T);
        return 0;
    }

    lop(); /* drop sparse matrix from stack */
    return status;
} /* 1}}} */
int  spy()         /* spy (hA_sp --- ) {{{1 */
/* 
 *   Stack has a sparse matrix; this prints the pattern of non-zeros
 *   as x's on a field of .'s.
 * man entry:  spy {{{2
 * (hA_sp --- ) Stack has a sparse matrix; this word prints the pattern of non-zero's as x's on a field of .'s.  For example, 
 * > 4 speye spy
 * gives
 * x...
 * .x..
 * ..x.
 * ...x
 * category: math::matrix::sparse
 * related: dense, sparse, spadd, spbend, spy, spconvert, spdiag, spinfo
 * 2}}}
 */ 
{
    int     j, c, s, n_str;
    SparseMatrix m;

    if (!is_sparse(tos)) {
        stkerr(" spdiag: ",SPARSENOT);
        return 0;
    }

    spbend(); /* first transpose the matrix */

    m = sparse_overlay(tos);

    for (c = 0; c < m.H[COLS]; c++) {
        n_str = m.S_start[c+1] - m.S_start[c]; /* # strings this column */
        if (n_str) {
            /*
             *  Initial zeros before the first nonzero term in the column.
             */
            for (j = 0; j < m.S[ m.S_start[c] ].start_row; j++) {
                gprintf(".");
            }

            for (s = 0; s < n_str; s++) {
                /*
                 *  Zeros between strings.
                 */
                if (s > 0) {
                    for (j = m.S[ m.S_start[c] + s-1 ].start_row +
                             m.S[ m.S_start[c] + s-1 ].len ; 
                         j < m.S[ m.S_start[c] + s ].start_row; 
                         j++) {
                        gprintf(".");
                    }
                }
                /*
                 *  Non-Zeros within a string.
                 */
                for (j = 0; 
                     j < m.S[ m.S_start[c] + s ].len;
                     j++) {
                    gprintf("x");
                }
            }
            /*
             *  Trailing zeros after the last nonzero term in the column.
             */
            s = m.S_start[c+1]-1; /* index of the last string this column */
            for (j = m.S[s].start_row + m.S[s].len; 
                 j < m.H[ROWS]; 
                 j++) {
                gprintf(".");
            }
        } else {  /* entire column is null */
            for (j = 0; j < m.H[ROWS]; j++) {
                gprintf(".");
            }
        }
        nc();
    }
    drop();

    return 1;
} /* 1}}} */
int  spnull()      /* spnull (r c --- hN_sp) {{{1 */
/* 
 * man entry:  spnull {{{2
 * (r c --- hN_sp) Creates a null r x c sparse matrix.
 * category: math::matrix::sparse
 * related: dense, sparse, spadd, spbend, spy, spconvert, spdiag
 * 2}}}
 */ 
{
    return f_spnull(0);
} /* 1}}} */
int  get_fp_scalar(double  *Re,    /* out {{{1 */
                   double  *Im     /* out */
                  )
    /* Pull a "floating point" scalar off the stack.  In addition to
     * a regular (real) number, a scalar may also be a 1x1 real or
     * complex, dense or sparse matrix.  The imaginary component will
     * be non-zero only if the input is a 1x1 dense or sparse complex
     * matrix.
     */
{
int DEBUG = 0;
    double      *m1x1;
    SparseMatrix m;
    int          cmplx = 0;

    *Im = 0.0;
    if         (tos->typ == NUM)  { /* got a true scalar                */
        popd(Re);
if (DEBUG)
printf("get_fp_scalar Re=%e\n", *Re);
        return 1;
    } else if ((tos->typ == MAT) &&  !is_complex(tos)) {
        if ((tos->row != 1) || (tos->col != 1)) return 0;
        /* "scalar" is a 1x1 dense real matrix    */
        m1x1 = tos->mat;
       *Re   = m1x1[0];
        drop();
        return 1;
    } else if ((tos->typ == MAT) &&  !is_complex(tos)) {
        if ((tos->row != 2) || (tos->col != 1)) return 0;
        /* "scalar" is a 1x1 dense complex matrix */
        cmplx = 1;
        m1x1  = tos->mat;
       *Re    = m1x1[0];
       *Im    = m1x1[1];
        drop();
        return 1;
    } else if (!is_sparse(tos)) {
        /* "scalar" is a sparse matrix */
        m = sparse_overlay(tos);
        /* here= add code:
         *   is m 1x1?      
         *   is m complex?
         */
        return 0;
    }
    stkerr(" get_fp_scalar: ",NUMNOT); 
    return 0;
} /* 1}}} */
int  spscale()     /* spscale (hA_sp f --- hAf_sp) {{{1 */
/* 
 * man entry:  spscale {{{2
 * (hA_sp f --- hAf_sp) Multipy a sparse matrix by a scalar; a 1x1 real matrix or a 1x1 complex matrix are valid scalars.
 * category: math::matrix::sparse
 * related: dense, sparse, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
int DEBUG = 0;
    char        *name = "_Af";
    int          i, indexed, cmplx = 0;
    SparseMatrix m, m_new;
    double       Re, Im = 0.0;

    if (!get_fp_scalar(&Re, &Im)) {
        stkerr(" spscale: ",NUMNOT); 
        return 0;
    } 
if (DEBUG)
printf("spscale got Re=%e Im=%e\n", Re, Im);

    if (!is_sparse(tos)) {
        stkerr(" spscale: ",SPARSENOT); 
        return 0;
    }
    indexed = is_indexed(tos);
    cop();
    m = sparse_overlay(tos);

    if (m.H[n_NONZ] && !m.H[COMPLX] && !cmplx) { 
        /* input matrix and scale factor both real (most common case?) */
        for (i = 0; i < m.H[n_NONZ]; i++) 
            m.N[i] *= Re;
    } else if (m.H[n_NONZ] && m.H[COMPLX]) { 
        /* input matrix is complex; doesn't matter what scalar is */
        for (i = 0; i < 2*m.H[n_NONZ]; i += 2) {
            m.N[i]   = Re * m.N[i]   - Im * m.N[i+1];
            m.N[i+1] = Re * m.N[i+1] + Im * m.N[i]  ;
        }
    } else if (m.H[n_NONZ] && !m.H[COMPLX] &&  cmplx) { 
        /* input matrix is real, scale factor is complex {{{2 */
        /* need to allocate memory for a bigger matrix   */
        if (!sparse_stk(m.H[ROWS]  , /* in  */ 
                        m.H[COLS]  , /* in  */
                        m.H[n_STR] , /* in  number of strings       */
                        m.H[n_NONZ], /* in  number of nonzero terms */
                        1          , /* in  0=real  1=complex       */
                        indexed    , /* in  0=no    1=yes           */
                        name       , /* in  */
                       &m_new)) {        /* out */
            return 0;
        }

if (DEBUG)
printf("spscale: 3\n");
        sp_copy_index_data(m.H[COLS],        /* in */
                           m.H[n_STR],       /* in */
                           m,                /* in */
                           m_new);           /* out */
        /* finally, do the math */
if (DEBUG)
printf("spscale: 4\n");
        for (i = 0; i < m.H[n_NONZ]; i++) {
            m_new.N[2*i]   = Re * m.N[i];
            m_new.N[2*i+1] = Im * m.N[i];
        }
if (DEBUG)
printf("spscale: 5\n");
        lop(); /* drop the original real matrix from the stack 2}}} */
        set_complex(tos);
if (DEBUG)
printf("spscale: 6\n");
    } else if ((Re == ((double) 0.0)) && 
               (Im == ((double) 0.0)) &&  
               m.H[n_NONZ]) {
        /* Input matrix is non-null but scale */
        /* factor is zero:  make null matrix. */
        pushint(m.H[ROWS]);
        pushint(m.H[COLS]);
        return f_spnull(indexed);
    } /* else matrix was null, do nothing */

if (DEBUG) {
printf("spscale: 7\n");
spdump();
}
    return 1;

} /* 1}}} */
int  spadd()       /* spadd (hA_sp hB_sp --- hC_sp) {{{1 */
/* 
 * man entry:  spadd {{{2
 * (hA_sp hB_sp --- hC_sp) Add two sparse matrices.  Either or both inputs may be real or complex.
 * category: math::matrix::sparse
 * related: dense, sparse, spsum, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
    SparseMatrix A, B;

    if (is_indexed(tos) && is_indexed(tos-1) && !f_indices_match(tos, tos-1)) {
        return spadd_ijv();
    }
        
    A = sparse_overlay(tos);
    B = sparse_overlay(tos-1);

    if        (!A.H[COMPLX] && !B.H[COMPLX]) {  /* real    + real    */
        return spadd_rr();
    } else if ( A.H[COMPLX] &&  B.H[COMPLX]) {  /* complex + complex */
        return spadd_cc();
    } else if ( A.H[COMPLX] && !B.H[COMPLX]) {  /* complex + real    */
        swap();
        return spadd_rc();
    } else {                                    /* real    + complex */
        return spadd_rc();
    }

} /* 1}}} */
int  spones()      /* spones (hA_sp --- hO_sp) {{{1 */
/* 
 * man entry:  spones {{{2
 * (hA_sp --- hO_sp) Replaces the non-zero terms of the sparse matrix on the stack with 1.0.  The output is real-only, even if the input is complex.
 * category: math::matrix::sparse
 * related: dense, sparse, spbend, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
    char   *name = "_O";
    int     i, indexed;
    SparseMatrix m_input, m_ones;

    if (!is_sparse(tos)) {
        stkerr(" spones:  ", SPARSENOT); 
        return 0;
    }
    m_input = sparse_overlay(tos);

    indexed = is_indexed(tos);
    if (!sparse_stk(m_input.H[ROWS]  , /* in  */ 
                    m_input.H[COLS]  , /* in  */
                    m_input.H[n_STR] , /* in  number of strings       */
                    m_input.H[n_NONZ], /* in  number of nonzero terms */
                    0                , /* in  0=real  1=complex       */
                    indexed          , /* in  0=no    1=yes           */
                    name             , /* in  */
                   &m_ones)) {         /* out */
        return 0;
    }

    /*
     *  Indexing data is identical to input...
     */
    sp_copy_index_data(m_input.H[COLS] ,      /* in */
                       m_input.H[n_STR],      /* in */
                       m_input         ,      /* in */
                       m_ones);               /* out */
    /*
     *  ...but numerical values are 1.0
     */
    for (i = 0; i < m_ones.H[n_NONZ]; i++) {
        m_ones.N[i] = (double) 1.0;
    }
    if (indexed) {
        memcpy(m_ones.row_idx, m_input.row_idx, m_input.H[ROWS]*sizeof(int));
        memcpy(m_ones.col_idx, m_input.col_idx, m_input.H[COLS]*sizeof(int));
    }
    lop();
    return 1;
} /* 1}}} */
int  spreal_imag() /* spreal_imag (hC_sp --- hRe_sp hIm_sp) {{{1 */
/* 
 * man entry:  spreal_imag {{{2
 * (hC_sp --- hRe_sp hIm_sp) Takes a sparse complex matrix and extracts the real and imaginary terms into two separate (real-only) sparse matrices.
 * category: math::complex, math::matrix::sparse
 * related: dense, sparse, spbend, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
int DEBUG = 0;
    char *name[2] = { "_Re", "_Im"};
    int   r, c, s, s_main, i, r_ptr, n_ptr, end_row, n_str, longest_str,
         *str_ind, *str_len, RI, *ind, err, is, row_offset,
          N_ptr, S_ptr, n_total_str, n_total_nonz, indexed;

    SparseMatrix m_in, m[2];

    if (!is_sparse(tos)) {
        stkerr(" spreal_imag:  ", SPARSENOT); 
        return 0;
    }
    m_in    = sparse_overlay(tos);
    indexed = is_indexed(tos);

    if (!m_in.H[COMPLX]) {
        stkerr(" spreal_imag:  ", COMPLEXNOT); 
        return 0;
    }
if (DEBUG) f_sp_dump("top of spreal_imag input is:", tos);

    /* Allocate working arrays to keep track of row indices {{{2
     * that correspond to non-zero terms in Re and Im matrices
     */ 
    if ((ind = (int *) malloc( m_in.H[ROWS]*sizeof(int) )) == NULL) {
        stkerr(" spreal_imag: (ind) ", MEMNOT); 
        return 0;
    }
    /* str_ind[], str_len[], ind[] count over strings
     * in a column so their max length is (rows+1)/2 
     */
    if ((str_ind = (int *) malloc(((m_in.H[ROWS]+1)/2)*sizeof(int) )) == NULL) {
        stkerr(" spreal_imag: (str_ind) ", MEMNOT); 
        free(ind);
        return 0;
    }
    if ((str_len = (int *) malloc(((m_in.H[ROWS]+1)/2)*sizeof(int) )) == NULL) {
        stkerr(" spreal_imag: (str_len) ", MEMNOT); 
        free(str_ind);
        free(ind);
        return 0;
    }
    /* 2}}} */

    for (RI = 0; RI < 2; RI++) {  /* 0 does real part;  1 does imag */

    /*  Pass 1:  determine # of strings and non-zeros {{{2
     *           in the output matrices
     */
    n_total_str   = 0;
    n_total_nonz  = 0;
    for (c = 0; c < m_in.H[COLS]; c++) {
if (DEBUG)
printf("\nPass 1 spreal_imag %s column %d\n", name[RI], c);
        r_ptr = 0;  /* index to non-zero rows in Re or Im */
        for (s = m_in.S_start[c]; s < m_in.S_start[c+1]; s++) {
            end_row = m_in.S[s].start_row + m_in.S[s].len - 1;
            n_ptr = m_in.S[s].N_idx;
            i     = 0;
            for (r = m_in.S[s].start_row; r <= end_row; r++) {
if (DEBUG) {
printf("  looking at m_in.N[%2d + %2d + %2d]= % 12.6le\n", 
n_ptr, 2*i, RI, m_in.N[ n_ptr + 2*i + RI]);
}
                if (fabs(m_in.N[ n_ptr + 2*i + RI]) > SPARSE_ZERO_THRESH) {
if (DEBUG) {
printf("       got non zero, setting ind[%2d] = %d\n", r_ptr, r);
}
                    ind[r_ptr++] = r;
                } 
                ++i;
            }
        }
        strings_in_list(r_ptr,          /* in  */
                        ind,            /* in  */
                       &n_str,          /* out */
                        str_ind,        /* out (not used here) */
                        str_len,        /* out */
                       &longest_str);   /* out */
        n_total_str  += n_str;
        n_total_nonz += r_ptr;

if (DEBUG) {
printf("  ind[] = ");
for (r = 0; r < r_ptr; r++) { printf(" %2d", ind[r]); } printf("\n");
for (r = 0; r < n_str; r++) { 
printf("  str_ind[%d]=%2d str_len[%d]=%2d\n", r,str_ind[r], r,str_len[r]); } 
}
    }
if (DEBUG) {
printf("spreal_imag end of Pass 1 %s: n_str=%d\n", name[RI], n_total_str);
}
    /* 2}}} */

    /* allocate sparse output matrix and put it on the stack {{{2 */
    if (!sparse_stk(m_in.H[ROWS], /* in  */ 
                    m_in.H[COLS], /* in  */
                    n_total_str , /* in  number of strings       */
                    n_total_nonz, /* in  number of nonzero terms */
                    0           , /* in  0=real  1=complex       */
                    indexed     , /* in  0=no    1=yes           */
                    name[RI]    , /* in  */
                   &m[RI])) {          /* out */
        free(str_len);
        free(str_ind);
        free(ind);
        return 0;
    }
    /* 2}}} */

    /*  Pass 2:  Populate Re[] and Im[] output matrices {{{2
     */
    m[RI].S_start[0] = 0;
    m[RI].N_start[0] = 0;
    s_main           = 0;
    N_ptr            = 0;
    S_ptr            = 0;
    for (c = 0; c < m_in.H[COLS]; c++) {
if (DEBUG)
printf("\nPass 2 %s col %d\n", name[RI], c);
        r_ptr = 0;  /* index to non-zero rows in Re or Im */
        /* Fill ind[] and ind[] with the row indices of
         * non-zero real and imaginary terms...
         */
        for (s = m_in.S_start[c]; s < m_in.S_start[c+1]; s++) {
            end_row = m_in.S[s].start_row + m_in.S[s].len - 1;
            n_ptr   = m_in.S[s].N_idx;
            i       = 0;
            for (r = m_in.S[s].start_row; r <= end_row; r++) {
                if (fabs(m_in.N[ n_ptr + 2*i + RI]) > SPARSE_ZERO_THRESH) {
                    ind[  r_ptr] = r;
                    ++r_ptr;
                } 
                ++i;
            }
        }
        /* ...then find clumps of consecutive terms in ind[]
         * and save this string information in str_ind[] and str_len[].
         */
        strings_in_list(r_ptr,          /* in  # non-zero real terms */
                        ind,            /* in  list of real term row indices */
                       &n_str,          /* out length of str_ind, str_len    */
                        str_ind,        /* out */
                        str_len,        /* out */
                       &longest_str);   /* out */
if (DEBUG) {
printf("  ind[] = ");
for (r = 0; r < r_ptr; r++) { printf(" %2d", ind[r]); } printf("\n");
}
        m[RI].S_start[c+1] = m[RI].S_start[c] + n_str;
        m[RI].N_start[c+1] = m[RI].N_start[c] + r_ptr;
        s_main  = m_in.S_start[c];
        end_row = m_in.S[s_main].start_row + m_in.S[s_main].len - 1;
        n_ptr   = m_in.S[s_main].N_idx;

        for (s = 0; s < n_str; s++) {
            /* s counts over real-only or imag-only strings in this column
             * of either R or I;
             * s_main counts over combined Re,Im strings in the entire
             * input matrix.
             * Here need to locate corresponding string indices between 
             * input and output matrices.
             */
            while ((m_in.S[s_main].start_row  /* str start row in complex mat */
                       < 
                    ind[str_ind[s]])
                       &&
                   (end_row                   /* str end row in complex mat */
                       <
                   (ind[str_ind[s]]+str_len[s]-1)) /* str end row in R|I mat */
                  ) {
if (DEBUG) {
printf("  s=%d/s_main=%d misaligned:\n", s, s_main);
printf("  m_in.S[%2d].start_row         =%d\n", 
s_main, m_in.S[s_main].start_row);
printf("  ind[str_ind[%2d] (=%2d)]       =%d\n", 
s, str_ind[s], ind[str_ind[s]]);
printf("  end_row                      =%d\n", end_row);
printf("  ind[str_ind[%d]]+str_len[%d]-1 =%d\n", 
s,s,ind[str_ind[s]]+str_len[s]-1);
}
                ++s_main;
                end_row = m_in.S[s_main].start_row + m_in.S[s_main].len - 1;
                n_ptr   = m_in.S[s_main].N_idx;
            }
            m[RI].S[S_ptr].start_row = ind[ str_ind[s] ];
            m[RI].S[S_ptr].len       = str_len[s];
            m[RI].S[S_ptr].N_idx     = N_ptr;
            ++S_ptr;
if (DEBUG) {
printf("  m.S[s_main=%2d].start_row=%2d   .len=%2d   .N_idx=%2d\n",
s_main, ind[ str_ind[s] ], str_len[s], N_ptr);
} 
            row_offset = ind[str_ind[s]] - m_in.S[s_main].start_row;
if (DEBUG) {
printf("  str_ind[%d]=%2d str_len[%d]=%2d\n", s,str_ind[s], s,str_len[s]); 
} 
            for (is = 0; is < str_len[s]; is++) {
if (DEBUG) {
printf("  m.N[%2d] = m_in.N[%2d+2*(%2d+%2d)+%2d (=%2d)] = % 12.6le\n",
N_ptr, m_in.S[s_main].N_idx, is, row_offset, RI,
m_in.S[s_main].N_idx + 2*(is + row_offset) + RI,
m_in.N[ m_in.S[s_main].N_idx + 2*(is + row_offset) + RI]);
}
                m[RI].N[ N_ptr++ ] = 
                    m_in.N[ m_in.S[s_main].N_idx + 2*(is + row_offset) + RI];
            }
        }
    }
if (DEBUG) {
printf("spreal_imag end of Pass 2  %s: n_str=%d\n", name[RI], n_total_str);
}
    /* 2}}} */
    }

    free(str_len);
    free(str_ind);
    free(ind);

    err = f_sp_error(tos);
if (DEBUG) {
printf("end of spreal_imag Im error = %d\n", err);
if (err) {
    f_sp_dump("sp dump of Im:", tos);
    return 0;
}
err = f_sp_error(tos-1);
printf("end of spreal_imag Re error = %d\n", err);
if (err) {
    f_sp_dump("sp dump of Re:", tos-1);
    return 0;
}
}
    rot(); drop(); /* dropping input matrix from stack */
    return 1;
} /* 1}}} */
int  spparkn()     /* spparkn (hA1_sp ... hAn_sp n --- hA_sp) {{{1 */
/* 
 *   Append n sparse matrices into a single sparse matrix.
 * man entry:  spparkn {{{2
 * (hA1_sp ... hAn_sp n --- hA_sp) Concatenate columns of n sparse matrices into a single sparse matrix.  Each matrix must have the same number of rows and be of the same type (ie, all must be real or all must be complex).  The columns of the output matrix will be ordered [A1..An].   Does nothing if n < 2.  This is the sparse version of parkn.
 * category: math::matrix::partitioning, math::matrix::sparse
 * related: parkn, sppilen, pilen
 * 2}}}
 */ 
{
int DEBUG = 0;
    int     c, n, s, z, i, j, error, NPT, n_str, n_nonz, n_col,
            S_offset, N_offset;
    SparseMatrix *m_in, m_out;
    char   *name = "_spparkn";

    if (!popint(&n)) return 0;
    if (n < 2)       return 1;

    if ((m_in = (SparseMatrix *) malloc( n *sizeof(SparseMatrix) )) == NULL) {
        stkerr(" spparkn: (m_in) ", MEMNOT); 
        return 0;
    }

    for (i = 0; i < n; i++) {
        if (!is_sparse(tos-i)) {
            stkerr(" spparkn:  input matrix ",SPARSENOT); 
            free(m_in);
            return 0;
        }
        m_in[i] = sparse_overlay(tos-(n-1-i));
        /* m_in[0]   holds hA1_sp, the last matrix on the stack.  This
         *           matrix will be the left-most matrix in the output.
         * m_in[n-1] holds the matrix currently at top of the stack
         *           (hAn_sp).  It will be the right-most part of the 
         *           output matrix.
         */
if (DEBUG) {
printf("spparkn i=%2d nR=%2d nC=%2d\n", i, m_in[i].H[ROWS], m_in[i].H[COLS]);
}
    }

    error  = 0;
    n_str  = m_in[0].H[n_STR];
    n_col  = m_in[0].H[COLS];
    n_nonz = m_in[0].H[n_NONZ]; 
    NPT    = NUM_PER_TERM( m_in[0].H[COMPLX] );
    /*
     *  Do the input matrices have same number of rows & have same type?
     */
    for (i = 1; i < n; i++) {
        if (m_in[i].H[ROWS]   != m_in[0].H[ROWS]) {
            stkerr(" spparkn:  input matrix ",ROWSNOT); 
            error = 1;
        }
        if (m_in[i].H[COMPLX] != m_in[0].H[COMPLX]) {
            stkerr(" spparkn:  input matrix ",MATCHNOT); 
            error = 1;
        }
        if (error) {
            free(m_in);
            return 0;
        }
        n_str  += m_in[i].H[n_STR];
        n_nonz += m_in[i].H[n_NONZ];
        n_col  += m_in[i].H[COLS];
    }

    /*   allocate memory for the sum, put it on the stack. */
    if (!sparse_stk(m_in[0].H[ROWS]   , /* in  */ 
                    n_col             , /* in  */
                    n_str             , /* in  number of strings       */
                    n_nonz            , /* in  number of nonzero terms */
                    m_in[0].H[COMPLX] , /* in  0=real  1=complex       */
                    0                 , /* in  lose indexing data      */
                    name              , /* in  */
                   &m_out)) {           /* out */
        free(m_in);
        return 0;
    }

    /* populate the output matrix */
    c        = 0;    /* output column  index */
    s        = 0;    /* output string  index */
    z        = 0;    /* output numeric index */
    S_offset = 0;
    N_offset = 0;
    m_out.S_start[0] = 0;
    m_out.N_start[0] = 0;
    for (i = 0; i < n; i++) { 
        /* i counts over the input matrices */
        for (j = 0; j < m_in[i].H[COLS]; j++) { 
            /* j counts over columns of the ith input matrix */
            m_out.S_start[c] = S_offset + m_in[i].S_start[j];
            m_out.N_start[c] = N_offset + m_in[i].N_start[j];
            ++c;
        }
        for (j = 0; j < m_in[i].H[n_STR]; j++) {
            /* j counts over strings of the ith input matrix */
            m_out.S[s].start_row =            m_in[i].S[j].start_row;
            m_out.S[s].len       =            m_in[i].S[j].len;
            m_out.S[s].N_idx     = N_offset + m_in[i].S[j].N_idx;
            ++s;
        }
        for (j = 0; j < NPT*m_in[i].H[n_NONZ]; j++) {
            /* j counts over nonzeros in the ith input matrix */
            m_out.N[z] = m_in[i].N[j];
            ++z;
        }
        S_offset +=       m_in[i].H[n_STR];
        N_offset += NPT * m_in[i].H[n_NONZ];
    }
    m_out.S_start[n_col] = S_offset;
    m_out.N_start[n_col] = N_offset;

    /* drop input matrices from the stack */
    for (i = 0; i < n; i++)   
        lop();

    free(m_in);
    return 1;
} /* 1}}} */
int  spconvert()   /* spconvert (hIJV --- hA_sp) {{{1 */
/* 
 * man entry:  spconvert {{{2
 * spconvert (hIJV --- hA_sp) Stack has a dense matrix containing either three or four columns.  spconvert uses the first two columns as row (r) and column (c) indices of the sparse matrix it will return; the third column contains the numeric value for the (r,c) positions indicated by the first two columns.  A fourth column, if present, contains imaginary value at (r,c) and will yield a complex sparse matrix on the stack.  The dimensions of the resulting sparse matrix are determined by the largest values encountered in the first two columns of the input matrix.  The input matrix can have r,c entries in any order.
 * Example 1:  list:  4 4 0 ; bend spconvert
 *   If xbase is 1, will leave on the stack a null 4 x 4 sparse matrix.
 * Example 2:  
 *   list:  1  1  4.4  0   ; bend
 *   list:  2  2  5.5  0.1 ; bend
 *   list:  6  4  0.0  0   ; bend
 *   list:  3  3  6.6  0.2 ; bend 4 pilen spconvert
 * If xbase is 1, puts on the stack this matrix: (here expanded to dense)
 * [ 4.4,0      0        0        0 ]
 * [   0      5.5,0.1    0        0 ]
 * [   0        0      6.6,0.2    0 ]
 * [   0        0        0        0 ]
 * [   0        0        0        0 ]
 * [   0        0        0        0 ]
 * This word does the opposite of sp2ijv which takes a sparse matrix and creates an IJV matrix.
 * category: math::matrix::sparse
 * related: xbase, sp2ijv, sparse, spbend, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
int DEBUG = 0;
    char   *name = "_spconvert";
    int     num_size, nrows, ncols, i, status,
            nnz, got_nonzero, cmplx;
    double *dense_IJV;
    IJVMatrix    IJV;
    SparseMatrix m;

    if (tos->typ != MAT) {
        stkerr(" spconvert: ", MATNOT);
        return 0;
    }
    if (is_complex(tos)) {
        stkerr(" spconvert: ", "cannot use complex input");
        return 0;
    }
    num_size  = tos->row;
    if ((tos->col != 3) && (tos->col != 4)) {
        stkerr(" spconvert: ", "input must have 3 or 4 columns");
        return 0;
    }
    cmplx     = tos->col - 3;  /* 0 = real     1 = complex */
    dense_IJV = tos->mat;

if (DEBUG) gprintf("top of spconvert input is %d x %d (cmplx=%d)\n", 
num_size, tos->col, cmplx);

    /* Pass 1:  scan the first two columns to find out the */
    /* # of rows, # columns, and # of nonzero terms        */
    nrows = 0;
    ncols = 0;
    nnz   = 0;
    for (i = 0; i < num_size; i++) {
        nrows        = MAX(nrows, (int) dense_IJV[           i]);
        ncols        = MAX(ncols, (int) dense_IJV[num_size + i]);
        got_nonzero  = 0;
        if (cmplx) {
            if ((fabs(dense_IJV[2*num_size + i]) > SPARSE_ZERO_THRESH) ||
                (fabs(dense_IJV[3*num_size + i]) > SPARSE_ZERO_THRESH)) {
                ++nnz;
            }
        } else {
            if (fabs(dense_IJV[2*num_size + i]) > SPARSE_ZERO_THRESH) {
                ++nnz;
            }
        }
    }
    nrows += (1 - XBASE);
    ncols += (1 - XBASE);
if (DEBUG) gprintf("spconvert end of pass 1 found %d x %d\n", nrows, ncols);

    if (!nnz) {
        /* density is so low that there are no terms at all; call spnull *
         * to put a null matrix on the stack                             */
        drop(); /* drop dense matrix from stack */
        pushd(nrows);
        pushd(ncols);
        spnull();
        if (cmplx) {
            m = sparse_overlay(tos);
            m.H[COMPLX] = cmplx;
        }
        return 1;
    }

    if (!malloc_IJVMatrix(&IJV, nrows, ncols, nnz, 0)) {
        stkerr(" spconvert: (IJV) ",MEMNOT);
        return 0;
    }
    IJV.cmplx = cmplx;

    /* Pass 2:  copy the nonzero numeric terms */
    nnz = 0;
    for (i = 0; i < num_size; i++) {
        got_nonzero  = 0;
        if (cmplx) {
            if ((fabs(dense_IJV[2*num_size + i]) > SPARSE_ZERO_THRESH) ||
                (fabs(dense_IJV[3*num_size + i]) > SPARSE_ZERO_THRESH)) {
                IJV.d[nnz].Re = dense_IJV[2*num_size + i];
                IJV.d[nnz].Im = dense_IJV[3*num_size + i];
                got_nonzero = 1;
            }
        } else {
            if (fabs(dense_IJV[2*num_size + i]) > SPARSE_ZERO_THRESH) {
                IJV.d[nnz].Re = dense_IJV[2*num_size + i];
                IJV.d[nnz].Im = 0.0;
                got_nonzero = 1;
            }
        }
        if (got_nonzero) {
            IJV.d[nnz].row = (int) dense_IJV[           i] - XBASE;
            IJV.d[nnz].col = (int) dense_IJV[num_size + i] - XBASE;
if (DEBUG) gprintf("spconvert pass 2 i=%2d  %2d, %2d, %e\n",
nnz, IJV.d[nnz].row, IJV.d[nnz].col, IJV.d[nnz].Re);
            ++nnz;
        }
    }
    drop(); /* drop dense matrix from stack */

    qsort(IJV.d, nnz, sizeof(ijv), &comp_IJV_cr);

    /* 
     * Put a sparse matrix on the stack using the given IJV pattern.
     */
    status = f_ijv2sp(name     ,  /* in  */
                      IJV      ,  /* in  (memory freed)            */
                      0        ,  /* in  0 = no internal indexing  */
                      0        ); /* in  0 = don't transpose       */
    return status;
} /* 1}}} */
int  sp2ijv()      /* sp2ijv (hA_sp --- hIJV) {{{1 */
/* 
 * man entry:  sp2ijv {{{2
 * (hA_sp --- hIJV) Takes a sparse matrix and creates a dense matrix with three (if input is real) or four (if input is complex) columns:  row indices, column indices and numerical value(s) for each nonzero in the input matrix.  Each nonzero in the input matrix therefore yields one row in the output matrix.  Additionally, if the last row or the last column in the sparse matrix is null, the output matrix will contain an extra row with zero values simply to convey the full size of the input.
 * This word does the opposite of spconvert.
 * category: math::matrix::sparse
 * related: sparse, spbend, spadd, sprand, spy, spconvert, spdiag, spinfo
 * 2}}}
 */ 
{
int DEBUG = 0;
    char   *name = "_ijv";
    SparseMatrix m;
    IJVMatrix    IJV;

    int     c, cmplx, nrows, ncols, nnz, found_max_i = 0, found_max_j = 0,
            append_last_zero = 0, nOutCols = 0, nOutRows = 0, indexed = 0;
    double *d_IJV;

    if (!is_sparse(tos)) {
        stkerr(" sp2ijv: ",SPARSENOT);
        return 0;
    }
    indexed = is_indexed(tos);

    m = sparse_overlay(tos);
    cmplx = m.H[COMPLX];
    nrows = m.H[ROWS];
    ncols = m.H[COLS];
    nnz   = m.H[n_NONZ];

    if (!malloc_IJVMatrix(&IJV, nrows, ncols, nnz, indexed)) {
        stkerr(" sp2ijv:  ", MEMNOT); 
        return 0;
    }
    f_sp2ijv(m, indexed, &IJV);
    /* scan the index terms of IJV to see if the output needs
     * a place-holder n+1 null term
     */
    found_max_i = 0;
    found_max_j = 0;
    for (c = 0; c < nnz; c++) {
        if (IJV.d[c].row == (nrows - 1)) found_max_i = 1;
        if (IJV.d[c].col == (ncols - 1)) found_max_j = 1;
        if (found_max_i && found_max_j) break;
    }
    if (found_max_i && found_max_j) {
        append_last_zero = 0;
    } else {
        append_last_zero = 1;
    }

if (DEBUG) {
for (c = 0; c < m.H[n_NONZ]; c++) {
gprintf("sp2ijv  r=%3d  c=%3d   v=% 12.6e", 
IJV.d[c].row, IJV.d[c].col, IJV.d[c].Re);
if (cmplx) gprintf(",% 12.6e",  IJV.d[c].Im);
gprintf("\n");
}
gprintf("----------------------------\n");
}

    qsort(IJV.d, m.H[n_NONZ], sizeof(ijv), &comp_IJV_rc);

if (DEBUG) {
for (c = 0; c < m.H[n_NONZ]; c++) {
gprintf("sp2ijv  r=%3d  c=%3d   v=% 12.6e", 
IJV.d[c].row, IJV.d[c].col, IJV.d[c].Re);
if (cmplx) gprintf(",% 12.6e",  IJV.d[c].Im);
gprintf("\n");
}
}
    nOutRows = m.H[n_NONZ] + append_last_zero;
    nOutCols = 3 + cmplx;
if (DEBUG) gprintf("sp2ijv calling matstk with %d x %d\n", nOutRows, nOutCols);
    if (!matstk(nOutRows, nOutCols, name)) {
        stkerr(" sp2ijv: ", MEMNOT); 
        return 0;
    }
    d_IJV = tos->mat;

    for (c = 0; c < m.H[n_NONZ]; c++) {
        d_IJV[c]              = (double) (IJV.d[c].row + XBASE);
        d_IJV[c +   nOutRows] = (double) (IJV.d[c].col + XBASE);
        d_IJV[c + 2*nOutRows] =           IJV.d[c].Re;
        if (cmplx)
            d_IJV[c + 3*nOutRows] =       IJV.d[c].Im;
    }
    c = m.H[n_NONZ];

    if (append_last_zero) {
        d_IJV[c]              = (double) nrows - (1 - XBASE);
        d_IJV[c +   nOutRows] = (double) ncols - (1 - XBASE);
        d_IJV[c + 2*nOutRows] = 0.0;
        if (cmplx)
            d_IJV[c + 3*nOutRows] = 0.0;
    }
    free_IJVMatrix(IJV);

    lop(); /* drop sparse matrix from stack */
    return 1;
} /* 1}}} */
int  sppilen()     /* sppilen (hA1_sp ... hAn_sp n --- hA_sp) {{{1 */
/* 
 *   Append n sparse matrices into a single sparse matrix.
 * man entry:  sppilen {{{2
 * (hA1_sp ... hAn_sp n --- hA_sp) Vertically stacks n sparse matrices into a single sparse matrix.  Each matrix must have the same number of columns and be of the same type (ie, all must be real or all must be complex).  The rows of the output matrix will be ordered [A1..An]'.   Does nothing if n < 2.  This is the sparse version of pilen.
 * category: math::matrix::sparse, math::matrix::partitioning
 * related: pilen, spparkn, parkn
 * 2}}}
 */ 
{
int DEBUG = 0;
    int     c, n, s, i, j, NPT, n_rows, error, prev_end_last,
            s_ptr, n_ptr, row_offset, curr_str_len, n_str_this_col, 
            n_nonz_this_col, end_row, n_str, n_nonz;
    SparseMatrix *m_in, m_out;
    char   *name = "_sppilen";

    if (!popint(&n)) return 0;
    if (n < 2)       return 1;

    if ((m_in = (SparseMatrix *) malloc( n *sizeof(SparseMatrix) )) == NULL) {
        stkerr(" sppilen: (m_in) ", MEMNOT); 
        return 0;
    }

    for (i = 0; i < n; i++) {
        if (!is_sparse(tos-i)) {
            stkerr(" sppilen:  input matrix ",SPARSENOT); 
            free(m_in);
            return 0;
        }

/* drw 2004-06-02.  So upcoming m_in[i] will not seg,fault, as in
           99, 10 1 random sparse, 2 sppilen */
        if (!is_sparse(tos-(n-1-i))) {
            stkerr(" sppilen:  input matrix ",SPARSENOT);
            free(m_in);
            return 0;
        }

        m_in[i] = sparse_overlay(tos-(n-1-i));
        /* m_in[0]   holds hA1_sp, the last matrix on the stack.  This
         *           matrix will be the top-most matrix in the output.
         * m_in[n-1] holds the matrix currently at top of the stack
         *           (hAn_sp).  It will be the lowest part of the 
         *           output matrix.
         */
if (DEBUG) {
printf("sppilen i=%2d nR=%2d nC=%2d\n", i, m_in[i].H[ROWS], m_in[i].H[COLS]);
}
    }

    NPT    = NUM_PER_TERM( m_in[0].H[COMPLX] );
    n_rows = m_in[0].H[ROWS];
    n_str  = 0;
    n_nonz = 0;
    error  = 0;
    /*  Do input matrices have same number of columns & have same type? {{{2
     *  If any of the matrices are complex, convert null matrix header
     *  type to complex as well.
     */
    for (i = 1; i < n; i++) {
        if (m_in[i].H[COLS]   != m_in[0].H[COLS]) {
            stkerr(" sppilen:  input matrix ",COLSNOT); 
            error = 1;
        }
        if (m_in[i].H[COMPLX] != m_in[0].H[COMPLX]) {
            stkerr(" sppilen:  input matrix ",MATCHNOT); 
            error = 1;
        }
        if (error) {
            free(m_in);
            return 0;
        }
        n_rows += m_in[i].H[ROWS];
    } /* 2}}} */
if (DEBUG) printf("sppilen total n_rows=%3d\n", n_rows);

    /* Pass 1:  traverse the input matrices' string structures  {{{2
                to determine how many strings and non-zeros there will
                be in the output matrix 
     */
    for (c = 0; c < m_in[0].H[COLS]; c++) {   /* loop over columns          */
        prev_end_last  = 0;  /* 1 means previous matrix had a non-zero term */
                             /*   in its last row                           */
        curr_str_len   = 0;
        n_str_this_col = 0;
        for (i = 0; i < n; i++) {             /* loop over input matrices   */
            end_row = -2; /* in case there are no terms in this column */
            for (s = m_in[i].S_start[c]; s < m_in[i].S_start[c+1]; s++) {
                end_row = m_in[i].S[s].start_row + m_in[i].S[s].len - 1;
                if (!s && !m_in[i].S[s].start_row && prev_end_last) {
                    /* This is the first string in this column, its
                       first row is row zero, and the previous matrix had
                       a non-zero in its last row; this string therefore
                       continues from the previous matrix.  Do nothing. */
                    curr_str_len += m_in[i].S[s].len;
                } else {
                    curr_str_len  = m_in[i].S[s].len;
                    ++n_str;
                    ++n_str_this_col;
                }
                n_nonz       += m_in[i].S[s].len;
            }
            if (end_row == (m_in[i].H[ROWS]-1)) {
                prev_end_last = 1;
            } else {
                prev_end_last = 0;
            }
        }
    }
    /* 2}}} */
if (DEBUG) {
printf("sppilen end Pass 1:  n_str=%3d n_nonz=%3d\n", n_str, n_nonz);
}

    /*   allocate memory for the output, put it on the stack {{{2 */
    if (!sparse_stk(n_rows            , /* in  */ 
                    m_in[0].H[COLS]   , /* in  */
                    n_str             , /* in  number of strings       */
                    n_nonz            , /* in  number of nonzero terms */
                    m_in[0].H[COMPLX] , /* in  0=real  1=complex       */
                    0                 , /* in  0=no    1=yes           */
                    name              , /* in  */
                   &m_out)) {           /* out */
        free(m_in);
        return 0;
    }
    /* 2}}} */

    /* Pass 2:  assemble the output matrix {{{2 */
    s_ptr            = -1;
    n_ptr            = 0;
    m_out.S_start[0] = 0;
    m_out.N_start[0] = 0;
    m_out.S[0].len   = 0;
    for (c = 0; c < m_in[0].H[COLS]; c++) {   /* loop over columns          */
        n_str_this_col  = 0;
        n_nonz_this_col = 0;
        row_offset      = 0;
        prev_end_last =  0; /* 1 == previous matrix had a non-zero term  */
                            /*      in its last row                      */
        for (i = 0; i < n; i++) {             /* loop over input matrices   */

            end_row       = -2; /* in case there are no terms in this column */
            for (s = m_in[i].S_start[c]; s < m_in[i].S_start[c+1]; s++) {
                end_row = m_in[i].S[s].start_row + m_in[i].S[s].len - 1;
                if (!s && !m_in[i].S[s].start_row && prev_end_last) {
                    /* This is the first string in this column, its
                       first row is row zero, and the previous matrix had
                       a non-zero in its last row; this string therefore
                       continues from the previous matrix.  Do nothing. */
                    m_out.S[s_ptr].len      += m_in[i].S[s].len;
if (DEBUG) {
printf("sppilen P2: +m_out.S[%d].len = %d\n", s_ptr, m_out.S[s_ptr].len);
}
                } else {
                    ++s_ptr;
                    m_out.S[s_ptr].start_row = row_offset +
                                               m_in[i].S[s].start_row;
                    m_out.S[s_ptr].len       = m_in[i].S[s].len;
                    m_out.S[s_ptr].N_idx     = n_ptr;
if (DEBUG) {
printf("sppilen P2:  m_out.S[%d].len = %d\n", s_ptr, m_out.S[s_ptr].len);
}
                    ++n_str_this_col;
                }
                for (j = 0; j < NPT*m_in[i].S[s].len; j++) { /* numeric terms */
                    m_out.N[ n_ptr + j ] = m_in[i].N[ m_in[i].S[s].N_idx + j ];
if (DEBUG) {
printf("sppilen P2: m_out.N[%d+%d] = % 12.6le\n", n_ptr, j, m_out.N[n_ptr+j]);
}
                }
                n_nonz_this_col +=       m_in[i].S[s].len;
                n_ptr           += NPT * m_in[i].S[s].len;
            }
            if (end_row == (m_in[i].H[ROWS]-1)) {
                prev_end_last = 1;
            } else {
                prev_end_last = 0;
            }
            row_offset += m_in[i].H[ROWS];
        }
if (DEBUG) {
printf("sppilen P2: c=%2d  n_str_this_col=%2d n_nonz_this_col=%2d\n", 
c, n_str_this_col, n_nonz_this_col);
}
        m_out.S_start[c+1] = m_out.S_start[c] +       n_str_this_col;
        m_out.N_start[c+1] = m_out.N_start[c] + NPT * n_nonz_this_col;
    }
    /* 2}}} */

    /* drop input matrices from the stack */
    for (i = 0; i < n; i++)   
        lop();

    free(m_in);
    return 1;
} /* 1}}} */
int  spcatch()     /* spcatch (hA_sp c | hA_sp hCols --- hB_sp) {{{1 */
/* 
 *   Extract columns from a sparse matrix.
 * man entry:  spcatch {{{2
 * (hA_sp c | hA_sp hCols --- hB_sp) Extract columns from a sparse matrix.  Use either c, a single column index, or hCols, a vector of column indices to identify the columns of [A] to extract to create [B].  The entries in hCols may be repeated.  For example:
 * > 3 speye ( 3x3 [I] ) list: 3 2 1 ; spcatch
 * will swap the first and third columns of the 3x3 identity matrix leaving a sparse version of
 *  [ 0 0 1 ]
 *  [ 0 1 0 ]
 *  [ 1 0 0 ]
 * on the stack.
 * category: math::matrix::sparse, math::matrix::partitioning
 * related: catch, dense, sparse, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
int DEBUG = 0;
    int       nCols_out, NPT, nStr_out, nNonz_out, i, j, c, s,
              s_ptr, n_ptr, nstr, nnz, indexed;
    char      T[ERR_MSG_SIZE+1];
    SparseMatrix A, B;
    char     *name = "_spcatch";
    double   *List, L;

    /* type check the inputs {{{2 */
    if        (tos->typ == NUM) {   /* a single column defined by c */
        L         = pop()->real;
        pushd(L);        /* keeps stack size consistent with hCols case */
        List      = &L;  /* so that List[0] = L */
        nCols_out = 1;
    } else if (tos->typ == MAT) {   /* a list of columns defined by hCol */
        nCols_out = MAX(tos->row, tos->col);
        List      = tos->mat;
        if (is_complex(tos)) {
            stkerr(" spcatch:  Row definition vector should not be complex ", 
                     LISTNOT);
            return 0;
        }
        if (tos->row != 1 && tos->col != 1) {
            stkerr(" spcatch:  Row definition vector is a matrix ", 
                     STKNOT);
            return 0;
        }
    } else {
        stkerr(" spcatch: ", MATNOT);
        return 0;
    }

    if (!is_sparse(tos-1)) {
        stkerr(" spcatch: ",SPARSENOT); 
        return 0;
    }
    /* 2}}} */
    A       = sparse_overlay(tos-1);
    indexed = is_indexed(tos-1);
    NPT     = NUM_PER_TERM( A.H[COMPLX] );

    /* Pass 1:  figure out how much space needed for [B] {{{2 */
    nStr_out  = 0;
    nNonz_out = 0;
    for (i = 0; i < nCols_out; i++) {
        c = ((int) List[i]) - XBASE;
        /* is List[i] a valid column index? */
        if (c >= A.H[COLS]) {
            snprintf(T, ERR_MSG_SIZE, 
                     " spcatch: List[%d] defines column index %d which is > %d = number of columns in [A] ", 
                     i+XBASE, c+XBASE, A.H[COLS]);
            stkerr(T, ""); 
            return 0;
        }
        if (c < 0) {
            snprintf(T, ERR_MSG_SIZE, 
                     " spcatch: List[%d] = %d; must be >= %d ",
                     i+XBASE, c+XBASE, XBASE);
            stkerr(T, ""); 
            return 0;
        }

        nstr = A.S_start[c+1] - A.S_start[c];
if (DEBUG) {
printf("spcatch P1: i=%2d c=%2d nstr=%2d\n", i, c, nstr);
}
        nnz  = 0;
        if (nstr) {
            nnz = A.N_start[c+1] - A.N_start[c];
if (DEBUG) {
printf("spcatch P1: s counts %d -> %d  nnz=%d\n",
A.S_start[c], A.S_start[c]+nstr, nnz);
}
        }
        nStr_out  += nstr;
        nNonz_out += nnz;
    }
    nNonz_out /= NPT;
    /* 2}}} */

    /* allocate space for [B], put it on the stack {{{2 */
    if (!sparse_stk(A.H[ROWS]  , /* in  */ 
                    nCols_out  , /* in  */
                    nStr_out   , /* in  number of strings       */
                    nNonz_out  , /* in  number of nonzero terms */
                    A.H[COMPLX], /* in  0=real  1=complex       */
                    indexed    , /* in  0=no    1=yes           */
                    name       , /* in  */
                   &B)) {        /* out */
        return 0;
    }
    /* 2}}} */

    /* Pass 2:  copy columns of [A] into [B] {{{2 */
    s_ptr = 0;
    n_ptr = 0;
    for (i = 0; i < nCols_out; i++) {
        c = ((int) List[i]) - XBASE;
        if (indexed) {
            B.col_idx[i] = A.col_idx[c];
        }
        B.S_start[i] = s_ptr;
        B.N_start[i] = n_ptr;
        nstr   = A.S_start[c+1] - A.S_start[c];
        if (nstr) {
            for (s = A.S_start[c]; s < A.S_start[c]+nstr; s++) {
                B.S[s_ptr].len       = A.S[s].len;
                B.S[s_ptr].start_row = A.S[s].start_row;
                B.S[s_ptr].N_idx     = n_ptr;
                ++s_ptr;
                for (j = A.S[s].N_idx; 
                     j < A.S[s].N_idx + NPT * A.S[s].len;
                     j++) {
                    B.N[n_ptr++] = A.N[j];
                }
            }
        }
    }
    if (indexed) {
        memcpy(B.row_idx, A.row_idx, A.H[ROWS]*sizeof(int));
    }
    B.S_start[nCols_out] = s_ptr;
    B.N_start[nCols_out] = n_ptr;
    /* 2}}} */

    lop(); lop(); /* drop original list and sparse matrix from stack */

    return 1;
} /* 1}}} */
int  spreach()     /* spreach (hA_sp r | hA_sp hRows --- hB_sp) {{{1 */
/* 
 *   Extract rows from a sparse matrix.
 * man entry:  spreach {{{2
 * (hA_sp r | hA_sp hRows --- hB_sp) Extract rows from a sparse matrix.  Use either r, a single row index, or hRows, a vector of row indices to identify the rows of [A] to extract to create [B].  The entries in hRows may be repeated.  The current algorithm transposes the input then calls spcatch to operate on columns because sparse row access is cumbersome.  Example:
 * > 3 speye ( 3x3 [I] ) list: 3 1 ; spreach
 * pulls the third and first rows of the 3x3 identity matrix leaving a sparse version of
 *  [ 0 0 1 ]
 *  [ 1 0 0 ]
 * on the stack.
 * category: math::matrix::sparse, math::matrix::partitioning
 * related: reach, dense, sparse, spadd, sprand, spy, spconvert, spdiag
 * 2}}}
 */ 
{
    double   *List, L;

    /* type check the inputs {{{2 */
    if        (tos->typ == NUM) {   /* a single column defined by c */
        L         = pop()->real;
        pushd(L);        /* keeps stack size consistent with hCols case */
        List      = &L;  /* so that List[0] = L */
    } else if (tos->typ == MAT) {   /* a list of columns defined by hCol */
        List      = tos->mat;
        if (is_complex(tos)) {
            stkerr(" spreach:  Row definition vector should not be complex ", 
                     LISTNOT);
            return 0;
        }
        if (tos->row != 1 && tos->col != 1) {
            stkerr(" spreach:  Row definition vector is a matrix ", 
                     STKNOT);
            return 0;
        }
    } else {
        stkerr(" spreach: ", MATNOT);
        return 0;
    }

    if (!is_sparse(tos-1)) {
        stkerr(" spreach: ",SPARSENOT); 
        return 0;
    }
    /* 2}}} */

    swap();     /* bring [A] to top of stack */
    spbend();   /* transpose [A] */
    swap();     /* put [A'] below the rake vector */
    spcatch();  /* extract columns of [A'] */
    spbend();   /* transpose the columns of [A'] so they are rows of [A] */

    return 1;
} /* 1}}} */
int  spcomb()      /* spcomb (hA_sp c01 --- hA0_sp hA1_sp) {{{1 */
/* 
 *   Extract columns from a sparse matrix.
 * man entry:  spcomb {{{2
 * (hA_sp c01 --- hA0_sp hA1_sp) partition columns of sparse matrix A into sparse matrix A0 where column vector c01 has zeroes, and sparse matrix A1 where c01 has nonzeroes
 * category: math::matrix::sparse, math::matrix::partitioning
 * related: comb, sprake, rake, dense, sparse, spadd, sprand, spy, spconvert, spdiag, partition
 * 2}}}
 */ 
{
int DEBUG = 0;
    int       i, C_size, n0, n1, *P0_list = 0, *P1_list = 0;
    double   *C, *P0_copy, *P1_copy;
    SparseMatrix A;

    /* type check the inputs {{{2 */
    if (tos->typ == MAT) {   /* a list of columns defined by hCol */
        C = tos->mat;
        if (is_complex(tos)) {
            stkerr(" spcomb:  Comb definition vector should not be complex ", 
                     LISTNOT);
            return 0;
        }
        if (tos->row != 1 && tos->col != 1) {
            stkerr(" spcomb:  Comb definition vector is a matrix ", 
                     STKNOT);
            return 0;
        }
        C_size = MAX(tos->row, tos->col);
    } else {
        stkerr(" spcomb: ", MATNOT);
        return 0;
    }

    if (!is_sparse(tos-1)) {
        stkerr(" spcomb: ",SPARSENOT); 
        return 0;
    }
    A = sparse_overlay(tos-1);
    if (C_size != A.H[COLS]) {
        stkerr(" spcomb: ",COLSNOTV); 
        return 0;
    }
    /* 2}}} */

    /* create partition lists to be used with spcatch */
    n0 = 0; /* number of columns in the 0 partition */
    n1 = 0; /* number of columns in the 1 partition */
    for (i = 0; i < C_size; i++) {
        if (C[i]) ++n1;
        else      ++n0;
    }
if (DEBUG)
printf("spcomb: n0=%d        n1=%d\n", n0, n1);
    if (n0)
        if ((P0_list  = (int *) malloc(n0 * sizeof(int))) == NULL) {
            stkerr(" spcomb P0: ",MEMNOT);
            return 0;
        }
    if (n1)
        if ((P1_list  = (int *) malloc(n1 * sizeof(int))) == NULL) {
            stkerr(" spcomb P1: ",MEMNOT);
            free(P0_list);
            return 0;
        }
    n0 = 0;
    n1 = 0;
    for (i = 0; i < C_size; i++)
        if (C[i]) P1_list[n1++] = i + XBASE;
        else      P0_list[n0++] = i + XBASE;

    drop();     /* no longer need the comb vector  ( A c01 --- A ) */
    dup1s();    /* make a copy of the input matrix ( A A --- A A ) */
    if (n0) {
        if (!matstk(n0, 1, "P0")) {
            free(P1_list);
            free(P0_list);
            return 0;
        }
        P0_copy = tos->mat;     /* ( A A --- A A P0_copy ) */
        for (i = 0; i < n0; i++)
            P0_copy[i] = (double) P0_list[i];
        free(P0_list);
if (DEBUG) {
for (i = 0; i < n0; i++){
printf("spcomb: P0_copy[%d]=%12.4le\n", i, P0_copy[i]);
}
}
        spcatch(); /* ( A A P0_copy --- A A0 ) */
        swap();    /* ( A A0 --- A0 A )        */
    } else {
        /* the 0 partition is empty; put a purged column in for A0 */
        drop();                                    /* ( A A  --- A    ) */
        if (!matstk(A.H[ROWS], 0, "A0")) return 0; /* ( A    --- A A0 ) */
        sparse(); /* dwil 2004-5-40 */
        swap();                                    /* ( A A0 --- A0 A ) */
    }
    /* stack now has ( A0 A ) */
    if (n1) {
        if (!matstk(n1, 1, "P1")) {
            free(P1_list);
            free(P0_list);
            return 0;
        }
        P1_copy = tos->mat;     /* ( A0 A --- A0 A P1_copy ) */
        for (i = 0; i < n1; i++)
            P1_copy[i] = (double) P1_list[i];
        free(P1_list);
if (DEBUG) {
for (i = 0; i < n1; i++){
printf("spcomb: P1_copy[%d]=%12.4le\n", i, P1_copy[i]);
}
}
        spcatch(); /* ( A0 A P1_copy --- A0 A1 ) */
    } else {
        /* the 1 partition is empty; put a purged column in for A1 */
        drop();                                    /* ( A0 A  --- A0    ) */
        if (!matstk(A.H[ROWS], 0, "A1")) return 0; /* ( A0    --- A0 A1 ) */
        sparse(); /* dwil 2004-5-40 */
    }

    return 1;
} /* 1}}} */
int  sprake()      /* sprake (hA_sp r01 --- hA0_sp hA1_sp) {{{1 */
/* 
 *   Extract rows from a sparse matrix.
 * man entry:  sprake {{{2
 * (hA_sp r01 --- hA0_sp hA1_sp) Partition rows of sparse matrix A into sparse matrix A0 where vector r01 has zeroes, and sparse matrix A1 where r01 has nonzeroes.  Uses spcomb and spbend.
 * category: math::matrix::sparse, math::matrix::partitioning
 * related: rake, spcomb, comb, dense, sparse, spadd, sprand, spy, spconvert, spdiag, partition
 * 2}}}
 */ 
{
    int R_size;
    SparseMatrix A;

    /* type check the inputs {{{2 */
    if (tos->typ == MAT) {   /* a list of rows defined by hRow */
        if (is_complex(tos)) {
            stkerr(" sprake:  Rake definition vector should not be complex ", 
                     LISTNOT);
            return 0;
        }
        if (tos->row != 1 && tos->col != 1) {
            stkerr(" sprake:  Rake definition vector is a matrix ", 
                     STKNOT);
            return 0;
        }
        R_size = MAX(tos->row, tos->col);
    } else {
        stkerr(" sprake: ", MATNOT);
        return 0;
    }

    if (!is_sparse(tos-1)) {
        stkerr(" sprake: ",SPARSENOT); 
        return 0;
    }
    A = sparse_overlay(tos-1);
    if (R_size != A.H[ROWS]) {
        stkerr(" sprake: ","row size does not match partition vector"); 
        return 0;
    }
    /* 2}}} */

    swap();      /* ( hA_sp r01 --- r01 hA_sp ) */
    spbend();    /* ( r01 hA_sp  --- r01 hA_sp' ) */
    swap();      /* ( r01 hA_sp' --- hA_sp' r01 ) */
    spcomb();    /* ( hA_sp' r01 ---  hA0_sp' hA1_sp' ) */
    spbend();    /* ( hA0_sp' hA1_sp' ---  hA0_sp' hA1_sp ) */
    swap();      /* ( hA0_sp' hA1_sp  ---  hA1_sp  hA0_sp' ) */
    spbend();    /* ( hA1_sp  hA0_sp' ---  hA1_sp  hA0_sp ) */
    swap();      /* ( hA1_sp  hA0_sp  ---  hA0_sp hA1_sp  ) */

    return 1;
} /* 1}}} */
int  spmesh()      /* spmesh (hA0_sp hA1_sp c01 --- hA_sp) {{{1 */
/* 
 *   Extract rows from a sparse matrix.
 * man entry:  spmesh {{{2
 * (hA0_sp hA1_sp c01 --- hA_sp) merge sparse matrices A0:A1 into left:right of A using column comb c01
 * category: math::matrix::sparse, math::matrix::partitioning
 * related: mesh, spcomb, comb, sprake, rake, spsubmat, submat, dense, sparse
 * 2}}}
 */ 
{
    char     *name = "_spmesh";
    int       i, P_size, indexed,
              c_0 = 0, c_1 = 0, c = 0;
    double   *P;
    SparseMatrix m, A0, A1;

    /* type check the inputs {{{2 */
    if (tos->typ == MAT) {   /* a list of columns defined by hCol */
        P = tos->mat;
        if (is_complex(tos)) {
            stkerr(" spmesh:  partition vector should not be complex ", 
                     LISTNOT);
            return 0;
        }
        if (tos->row != 1 && tos->col != 1) {
            stkerr(" spmesh:  partition vector is a matrix ", 
                     STKNOT);
            return 0;
        }
        P_size = MAX(tos->row, tos->col);
        P      = tos->mat;
    } else {
        stkerr(" spmesh: ", MATNOT);
        return 0;
    }

    if (!is_sparse(tos-1)) {
        stkerr(" spmesh: A1 ",SPARSENOT); 
        return 0;
    }
    if (!is_sparse(tos-2)) {
        stkerr(" spmesh: A0 ",SPARSENOT); 
        return 0;
    }
    indexed = is_indexed(tos-1) && is_indexed(tos-2);

    A1 = sparse_overlay(tos-1);
    A0 = sparse_overlay(tos-2);

    if (A0.H[ROWS] != A1.H[ROWS]) {
        stkerr(" spmesh: ", ROWSNOT); 
        return 0;
    }
    if (A0.H[COMPLX] != A1.H[COMPLX]) {
        stkerr(" spmesh: ","A0 and A1 must both be real or both be complex"); 
        return 0;
    }

    for (i = 0; i < P_size; i++)
        if (P[i])
            ++c_1;
        else 
            ++c_0;
    if (c_0 != A0.H[COLS]) {
        stkerr(" spmesh: ", "size of 0-partition not equal to # cols in A0"); 
        return 0;
    }
    if (c_1 != A1.H[COLS]) {
        stkerr(" spmesh: ", "size of 1-partition not equal to # cols in A1"); 
        return 0;
    }
    if (P_size != (A0.H[COLS] +  A1.H[COLS])) {
        stkerr(" spmesh: ", COLSNOTV); 
        return 0;
    }
    /* 2}}} */

    /* allocate memory for the combined matrix {{{2 */
    if (!sparse_stk(A0.H[ROWS]                 , /* in  */ 
                    A0.H[COLS]   + A1.H[COLS]  , /* in  */
                    A0.H[n_STR ] + A1.H[n_STR] , /* in  # strings       */
                    A0.H[n_NONZ] + A1.H[n_NONZ], /* in  # nonzero terms */
                    A0.H[COMPLX]               , /* in  0=real 1=complex*/
                    indexed                    , /* in  0=no    1=yes   */
                    name                       , /* in  */
                   &m)) {                        /* out */
        return 0;
    }
    /* 2}}} */

    c_0 = 0; c_1 = 0; c = 0;
    for (i = 0; i < P_size; i++)
        if (P[i])
            sp_copy_column(c_1++, c++, A1, m);
        else
            sp_copy_column(c_0++, c++, A0, m);

    lop(); /* drop partition vector */
    lop(); /* drop A1               */
    lop(); /* drop A0               */

    return 1;
} /* 1}}} */
int  spmult()      /* spmult (hA_sp hB|hB_sp --- hC) {{{1 */
/* 
 * man entry:  spmult {{{2
 * (hA_sp hB|hB_sp --- hC) Multiply sparse and dense, or sparse and sparse matrices.  Either or both inputs may be real or complex.
 * category: math::matrix::sparse
 * related: *, star, dense, sparse, spadd
 * 2}}}
 */ 
{
int DEBUG = 0;
    SparseMatrix A, B;
    int          A_sparse = 0, B_sparse = 0, A_cmplx = 0, B_cmplx = 0; 

if (DEBUG) printf("top of spmult\n");
        
    if (((tos-1)->typ != MAT) && !is_sparse(tos-1)) {
        stkerr(" spmult [A] is not dense or sparse matrix: ", MATNOT);
        return 0;
    }

    if ((tos->typ != MAT) && !is_sparse(tos)) {
        stkerr(" spmult [B] is not dense or sparse matrix: ", MATNOT);
        return 0;
    }

    if (is_sparse(tos-1)) {
        A_sparse = 1;
        A = sparse_overlay(tos-1);
        if (A.H[COMPLX]) {
            A_cmplx  = 1;
        } else {
            A_cmplx  = 0;
        }
    } else {
        A_sparse = 0;
        if (is_complex(tos-1)) {
            A_cmplx = 1;
        } else {
            A_cmplx = 0;
        }
    }

    if (is_sparse(tos)) {
        B_sparse = 1;
        B = sparse_overlay(tos);
        if (B.H[COMPLX]) {
            B_cmplx  = 1;
        } else {
            B_cmplx  = 0;
        }
    } else {
        B_sparse = 0;
        if (is_complex(tos)) {
            B_cmplx = 1;
        } else {
            B_cmplx = 0;
        }
    }

if (DEBUG) {
printf("A sparse=%d A complex=%d;  B sparse=%d B complex=%d\n", 
A_sparse, A_cmplx, B_sparse, B_cmplx);
}

    if        ( A_sparse && !B_sparse) {    /* SPARSE x DENSE */

        if        (!A_cmplx && !B_cmplx) {  /* sparse real    x dense real    */
            return spmult_sd_rr();
        } else if ( A_cmplx &&  B_cmplx) {  /* sparse complex x dense complex */
            return spmult_sd_cc();
        } else if ( A_cmplx && !B_cmplx) {  /* sparse complex x dense real    */
            return spmult_sd_cr();
        } else {                            /* sparse complex x dense real    */
            return spmult_sd_rc();
        }

    } else if (A_sparse &&  B_sparse) {    /* SPARSE x SPARSE */

        if        (!A_cmplx && !B_cmplx) {  /* sparse real    x sparse real    */
            return spmult_ss_rr();
        } else if ( A_cmplx &&  B_cmplx) {  /* sparse complex x sparse complex */
            return spmult_ss_cc();
        } else if ( A_cmplx && !B_cmplx) {  /* sparse complex x sparse real    */
            return spmult_ss_cr();
        } else {                            /* sparse complex x sparse real    */
            return spmult_ss_rc();
        }

    } else {
        stkerr(" spmult code path for dense x sparse not implemented ", MATNOT);
        return 0;
    }

} /* 1}}} */
int  spinfo()      /* spinfo (hA_sp --- ) {{{1 */
/* 
 * man entry:  spinfo {{{2
 * (hA_sp --- ) Print information about the sparse matrix:  statistics about string sizes, storage efficiency, null columns, numeric data.  Maximum and minimum values for complex terms are magnitudes of the terms.  Diagonal max/min values are printed for square matrices.
 * category: math::matrix::sparse
 * related: spdump, sparse, spbend, sp2ijv, spy
 * 2}}}
 */ 
{
int DEBUG = 0;
    SparseMatrix m;
    int    n, s, c, nonZs, nStr, cmplx = 0, n_null_col = 0, square = 0,
           first_row, last_row, N_ptr, offset;
    double mag;
    char  *type_r               = "real    ";
    char  *type_c               = "complex ";

    int    max_terms_per_col    = 0;
    int    min_terms_per_col    = MAXINT;
    double ave_terms_per_col    = 0.0;

    int    max_strings_per_col  = 0;
    int    min_strings_per_col  = MAXINT;
    double ave_strings_per_col  = 0.0;

    int    max_string_length    = 0;
    int    min_string_length    = MAXINT;
    double ave_string_length    = 0.0;

    double max_num_value        = -MAXDOUBLE;
    double min_num_value        =  MAXDOUBLE;
    double ave_num_value        = 0.0;

    double max_diag_value       = -MAXDOUBLE;
    double min_diag_value       =  MAXDOUBLE;
    double ave_diag_value       = 0.0;

    double max_abs_num_value    = -MAXDOUBLE;
    double min_abs_num_value    =  MAXDOUBLE;
    double ave_abs_num_value    = 0.0;

if (DEBUG) printf("top of spinfo\n");

    if (!is_sparse(tos)) {
        stkerr(" spinfo: ", SPARSENOT);
        return 0;
    }
    m = sparse_overlay(tos);

    gprintf("%d x %d ", m.H[ROWS], m.H[COLS]);
    if (m.H[ROWS] == m.H[COLS])
        square = 1;
    if (m.H[COMPLX]) {
        gprintf("%s", type_c);
        cmplx  = 1;
    } else {
        gprintf("%s", type_r);
    }
    gprintf("sparse matrix");
    nc();

    for (c = 0; c < m.H[COLS]; c++) {
        nonZs = m.N_start[c+1] - m.N_start[c];
        nStr  = m.S_start[c+1] - m.S_start[c];
        if (!nonZs) 
            ++n_null_col;
        if (m.H[COMPLX])
            nonZs /= 2;
        max_terms_per_col   = MAX(max_terms_per_col,   nonZs);
        max_strings_per_col = MAX(max_strings_per_col, nStr);

        min_terms_per_col   = MIN(min_terms_per_col,   nonZs);
        min_strings_per_col = MIN(min_strings_per_col, nStr);

        ave_terms_per_col  += nonZs;

        for (s = m.S_start[c]; s < m.S_start[c+1]; s++) {
            max_string_length    = MAX(max_string_length, m.S[s].len);
            min_string_length    = MIN(min_string_length, m.S[s].len);
            ave_string_length   += m.S[s].len;
            /* look for the diagonal term */
            first_row = m.S[s].start_row;
            last_row  = first_row + m.S[s].len - 1;
            N_ptr     = m.S[s].N_idx;
            if ( square          &&
                (c >= first_row) && 
                (c <= last_row )) {
                offset   = c - first_row;
                n        = N_ptr + offset;
if (DEBUG) printf("spinfo: c=%d first_row=%d last_row=%d  off=%d, N=%e\n",
c,first_row,last_row,offset,m.N[n]);
                if (m.H[COMPLX]) {
                    mag = sqrt( m.N[n]*m.N[n] + m.N[n+1]*m.N[n+1] );
                    max_diag_value      = MAX(max_diag_value,  mag);
                    min_diag_value      = MIN(min_diag_value,  mag);
                    ave_diag_value     +=                      mag ;
                } else {
                    max_diag_value      = MAX(max_diag_value, m.N[n]);
                    min_diag_value      = MIN(min_diag_value, m.N[n]);
                    ave_diag_value     +=                     m.N[n] ;
                }
            }
        }
    }
    ave_string_length /= m.H[n_STR];
    if (square)
        ave_diag_value /= m.H[ROWS];

    gprintf("# strings          = %9d ", m.H[n_STR]    );
    gprintf("          # indexing bytes   = %9d" ,
         m.H[DATA_SIZE] - m.H[n_NONZ]*NUM_PER_TERM(cmplx)*BPD);
    nc();
    gprintf("# nonzero terms    = %9d ", m.H[n_NONZ]   );
    gprintf("          # numeric bytes    = %9d" , 
         m.H[n_NONZ]*NUM_PER_TERM(cmplx)*BPD);
    nc();
    gprintf("# null columns     = %9d ", n_null_col    );
    gprintf("          total bytes        = %9d" , m.H[DATA_SIZE]);
    nc();
    gprintf("density            = %12.8e %%",
      100.0 * ((double) m.H[n_NONZ] /  m.H[ROWS]) / m.H[COLS]);
    gprintf("    storage efficiency =   %6.4f %%",
      100.0 * ((double) m.H[n_NONZ]*NUM_PER_TERM(cmplx)*BPD) / m.H[DATA_SIZE]); 
    nc();

    ave_terms_per_col  /= m.H[COLS];
    ave_strings_per_col = ((double) m.H[n_STR]) / m.H[COLS];

    if (m.H[n_NONZ]) {
        max_num_value      =      m.N[0];
        min_num_value      =      m.N[0];
        ave_num_value      =      m.N[0];

        max_abs_num_value  = fabs(m.N[0]);
        min_abs_num_value  = fabs(m.N[0]);
        ave_abs_num_value  = fabs(m.N[0]);
        if (m.H[COMPLX]) {
            mag = sqrt( m.N[0]*m.N[0] + m.N[1]*m.N[1] );
            max_num_value  = mag;
            min_num_value  = mag;
            ave_num_value  = mag;
        }
    }
    for (n = 0; n < NUM_PER_TERM(cmplx)*m.H[n_NONZ]; n += NUM_PER_TERM(cmplx)) {
        /* loop over numeric data */

        if (m.H[COMPLX]) {
            mag = sqrt( m.N[n]*m.N[n] + m.N[n+1]*m.N[n+1] );
            max_num_value      = MAX(max_num_value,  mag);
            min_num_value      = MIN(min_num_value,  mag);
            ave_num_value     +=                     mag ;
        } else {
            max_num_value      = MAX(max_num_value, m.N[n]);
            min_num_value      = MIN(min_num_value, m.N[n]);
            ave_num_value     +=                    m.N[n] ;

            max_abs_num_value  = MAX(max_abs_num_value, fabs(m.N[n]));
            min_abs_num_value  = MIN(min_abs_num_value, fabs(m.N[n]));
            ave_abs_num_value +=                        fabs(m.N[n]) ;
        }
        ave_num_value     /= m.H[n_NONZ];
        ave_abs_num_value /= m.H[n_NONZ];
    }

    gprintf("%12s %11s %12s %12s", " ", "Max", "Min", "Ave");     nc();
    gprintf("terms per col    = %12d "    , max_terms_per_col  );
    gprintf(                   "%12d "    , min_terms_per_col  );
    gprintf(                   "%12.2f"   , ave_terms_per_col  ); nc();
    gprintf("strings per col  = %12d "    , max_strings_per_col);
    gprintf(                   "%12d "    , min_strings_per_col);
    gprintf(                   "%12.2f"   , ave_strings_per_col); nc();
    gprintf("string length    = %12d "    , max_string_length  );
    gprintf(                   "%12d "    , min_string_length  );
    gprintf(                   "%12.2f"   , ave_string_length  ); nc();
    gprintf("numeric value    = % 12.4e " , max_num_value      );
    gprintf(                   "% 12.4e " , min_num_value      );
    gprintf(                   "% 12.4e"  , ave_num_value      ); nc();
    if (!cmplx) {
    gprintf("abs(numeric value) % 12.4e " , max_abs_num_value  );
    gprintf(                   "% 12.4e " , min_abs_num_value  );
    gprintf(                   "% 12.4e"  , ave_abs_num_value  ); nc();
    }
    if (square) {
    gprintf("diagonal value     % 12.4e " , max_diag_value     );
    gprintf(                   "% 12.4e " , min_diag_value     );
    gprintf(                   "% 12.4e"  , ave_diag_value     ); nc();
    }

    drop();
    return 1;

} /* 1}}} */
int  spsum()       /* spsum (hA_sp hD hR hC lower --- hB_sp) {{{1 */
/* 
 * man entry:  spsum {{{2
 * (hA_sp hD hR hC lower --- hB_sp) Sum a small dense matrix, D, into an existing sparse matrix, A, using the provided row (R) and column (C) indices such that
          A[R, C] = A[R, C] + D
If the scalar 'lower' is non-zero, only those terms in D which are on the diagonal or in the lower triangle of A will be added into A.  If 'lower' is zero all terms of D will be added in.
Note 1:  The terms A[R,C] must already exist in the sparse storage structure of A.
Note 2:  The dense version of this word is plusi.
 * category: math::matrix::sparse, math::matrix::operator
 * related: sparse, spadd, plusi
 * 2}}}
 */ 
{
int DEBUG = 0;  /* 1 = normal debug;   2 also prints D contents */
    SparseMatrix A;
    int     lower, s, ir, ic, row, col, found_it, offset, nC, nR,
            N_ptr, first_row, last_row,
            A_cmplx = 0, D_cmplx = 0, nRows_D, nCols_D;
    double *D, *R, *C;
    char    T[ERR_MSG_SIZE+1];

if (DEBUG) gprintf("top of spsum\n");

    pushstr("5 revn cop 5 revn"); xmain(0);  /* make a copy of A */

    /* type check the inputs {{{2 */
    if (!popint(&lower)) return 0;

    if (tos->typ != MAT) {
        stkerr(" spsum: C ", MATNOT);
        return 0;
    }
    if (is_complex(tos)) {
        stkerr(" spsum: ", "column index matrix must not be complex");
        return 0;
    }
    nC = tos->row*tos->col;
    C  = tos->mat;
if (DEBUG) {
gprintf("spsum lower=%d\n", lower);
gprintf("spsum col idx: ");
for (ic=0; ic<nC; ic++) gprintf("%2d)%3d ", ic, (int) C[ic]);
gprintf("\n");
}

    if ((tos-1)->typ != MAT) {
        stkerr(" spsum: R ", MATNOT);
        return 0;
    }
    if (is_complex(tos-1)) {
        stkerr(" spsum: ", "row index matrix must not be complex");
        return 0;
    }
    nR = (tos-1)->row*(tos-1)->col;
    R  = (tos-1)->mat;
if (DEBUG) {
gprintf("spsum row idx: ");
for (ir=0; ir<nR; ir++) gprintf("%2d)%3d ", ir, (int) R[ir]);
gprintf("\n");
}

    if ((tos-2)->typ != MAT) {
        stkerr(" spsum:  D ", MATNOT);
        return 0;
    }
    nRows_D = (tos-2)->row;
    nCols_D = (tos-2)->col;
    if (is_complex(tos-2)) {
        D_cmplx  = 1;
        nRows_D /= 2;
    }
    D  = (tos-2)->mat;
if (DEBUG > 1) {
gprintf("spsum D (%d x %d):\n", nRows_D, nCols_D);
for (ic = 0; ic < nCols_D*nRows_D; ic++) gprintf("D[%4d] = % e\n", ic, D[ic]);
}

    if (!is_sparse(tos-3)) {
        stkerr(" spsum:  A must be sparse ", SPARSENOT);
        return 0;
    }
    A = sparse_overlay(tos-3);

    if (A.H[COMPLX])
        A_cmplx  = 1;

    if (A_cmplx != D_cmplx) {
        stkerr(" spsum: ", "A and D must both be complex, or both be real");
        return 0;
    }
    if (nR != nRows_D) {
        stkerr(" spsum: ", " number of rows in D does not match size of R");
        return 0;
    }
    if (nC != nCols_D) {
        stkerr(" spsum: ", " number of cols in D does not match size of C");
        snprintf(T, ERR_MSG_SIZE, 
                 " number of columns in D (%d) does not match size of column index vector C (%d)", 
                 nCols_D, nC);
        stkerr(" spsum: ", T);
        return 0;
    }
    /* 2}}} */

    for (ic = 0; ic < nC; ic++) {
        col = ((int) C[ic]) - XBASE;
        if (col < 0 || col >= A.H[COLS]) {
            snprintf(T, ERR_MSG_SIZE, 
                     " column index of C[%d]=%d is not in range of %d to %d (# columns in [A] ", 
                     ic, col+XBASE, XBASE, A.H[COLS] - (1 - XBASE));
            stkerr(" spsum: ", T);
            return 0;
        }
        for (ir = 0; ir < nR; ir++) {
            row = ((int) R[ir]) - XBASE;
            if (row < 0 || row >= A.H[ROWS]) {
                snprintf(T, ERR_MSG_SIZE, 
                         " row index of R[%d]=%d is not in range of %d to %d (# rows in [A] ", 
                         ir, row+XBASE, XBASE, A.H[ROWS] - (1 - XBASE));
                stkerr(" spsum: ", T);
                return 0;
            }
            if (!lower || (row >= col)) { /* skip this if lower and row<col */

                /* find the string containing this row */
                found_it = 0;
                for (s = A.S_start[col]; s < A.S_start[col+1]; s++) {
                    first_row = A.S[s].start_row;
                    last_row  = first_row + A.S[s].len - 1;
                    N_ptr     = A.S[s].N_idx;
                    if (row >= first_row && row <= last_row) {
                        offset   = row - first_row;
                        found_it = 1;
if (DEBUG) gprintf("spsum: A.N[(%d+%d)=%d]=%e", 
N_ptr, offset, N_ptr + offset, A.N[N_ptr+offset]);
                        A.N[N_ptr + offset] += D[ir + ic*nRows_D];
if (DEBUG) gprintf(" += D[(%d + %d*%d)=%d]=%e\n", 
ir, ic, nRows_D, ir + ic*nRows_D, D[ ir + ic*nRows_D]); 
                        break;
                    }
                }
                if (!found_it) {
if (DEBUG) gprintf(" spsum: col_idx[ic=%d]=%d  row_idx[ir=%d]=%d\n", 
ic, C[ic], ir, R[ir]);
                    stkerr(" spsum: ", "index out of bounds");
                    return 0;
                }

            }
else {
if (DEBUG) gprintf("row %d, col %d above diagonal, skipping\n", row, col);
}
        }
    }

    drop();
    drop();
    drop();
    return 1;

} /* 1}}} */
int  spflip_sym()  /* spflip_sym (hL_sp --- hLLt_sp ) {{{1 */
/* 
 * man entry:  spflip_sym {{{2
 * (hL_sp --- hLLt_sp ) Creates a symmetric sparse matrix from a sparse lower triangular matrix ([L]) by copying the transpose of the lower triangle ([Lt]) to the upper triangle.  The input matrix must not have terms above the diagonal.  This word is useful for preparing a finite element matrix for sparse solution because currently available sparse solvers need the complete matrix.  If [L] is internally indexed, the row indices are copied to [LLt]'s row and column indices.  [L]'s column indices are not used.
 * category: math::matrix::sparse
 * related: sparse, splu, spfbs, spbend, sp2ijv, tril, spmult_sym_low_sd_rr
 * 2}}}
 */ 
{
int DEBUG = 0;
    SparseMatrix L, LLt;
    IJVMatrix    IJV;
    char   *name = "_LLt";
    int     LLt_nnz, first_row, last_row, n_ptr, i, r, c, s,
            transpose, indexed, cmplx, status;

if (DEBUG) printf("top of spflip_sym\n");

    if (!is_sparse(tos)) {
        stkerr(" spflip_sym: ", SPARSENOT);
        return 0;
    }
    /* can't do this check unless more tag setting is done in other words
    if (!is_low_tri(tos)) {
        stkerr(" spflip_sym: ", "input is not lower triangular" );
        return 0;
    }
    */
    L = sparse_overlay(tos);
    if (!L.H[n_NONZ]) {
        /* input matrix is null, don't have to do anything */ 
        return 1;
    }

    if (L.H[ROWS] != L.H[COLS]) {
        stkerr(" spflip_sym: ", SQUNOT);
        return 0;
    }
    cmplx   = 0;
    if (L.H[COMPLX]) {
        stkerr(" spflip_sym: ", "complex option not coded");
        return 0;
    }
    indexed = is_indexed(tos);

    /* Count the number of nonzeros in the full matrix.  If there are no
     * zeros on the diagonal the total number of nonzeros would simply be
     * 2 x non-zeros in the lower triangle (incl. diagonal) minus the
     * number of terms on the diagonal, ie 2*L.H[n_NONZ] - L.H[COLS] 
     */
    LLt_nnz = 2*L.H[n_NONZ];
    for (c = 0; c < L.H[COLS]; c++) {
        if (L.S_start[c] < L.S_start[c+1]) { /* not a null column */
            first_row = L.S[ L.S_start[c] ].start_row;
if (DEBUG) gprintf("spflip_sym before col %2d  LLt_nnz=%d", c, LLt_nnz);
            if (first_row < c) {
                stkerr(" spflip_sym: [L] ","has terms in upper triangle");
                return 0;
            } else if (first_row == c) {
                /* the diagonal term is non-zero; subtract one from the count */
                --LLt_nnz;
            }
        }
if (DEBUG) gprintf("; after LLt_nnz=%d\n", LLt_nnz);
    }
if (DEBUG)
gprintf("spflip_sym rows/cols=%d  non-zeros L =%d non-zeros LLt =%d\n",
L.H[COLS], L.H[n_NONZ], LLt_nnz);

    /*
    if ((IJV  = (int *) malloc(3 * LLt_nnz * sizeof(int))) == NULL) {
        stkerr(" spflip_sym: (IJV) ",MEMNOT);
        return 0;
    }
    if ((IJV = (ijv *) malloc((1+LLt_nnz) * sizeof(ijv))) == NULL) {
        stkerr(" spflip_sym: (IJV) ",MEMNOT);
        return 0;
    }
    */
    if (!malloc_IJVMatrix(&IJV, L.H[ROWS], L.H[COLS], LLt_nnz, indexed)) {
        stkerr(" spflip_sym: (IJV) ",MEMNOT);
        return 0;
    }
    IJV.cmplx = L.H[COMPLX];

    i = 0;
    for (c = 0; c < L.H[COLS]; c++) {

        for (s = L.S_start[c]; s < L.S_start[c+1]; s++) {
            first_row = L.S[s].start_row;
            last_row  = first_row + L.S[s].len - 1;

            n_ptr = 0;
            for (r = first_row; r <= last_row; r++) {
                IJV.d[i].row = r;
                IJV.d[i].col = c;
                IJV.d[i].Re  = L.N[L.S[s].N_idx + n_ptr];
                i++;
                if (r > c) {  
                    /* add a corresponding term for the upper triangle */
if (DEBUG) gprintf("spflip_sym VG r,c= %2d,%2d;  i=%2d\n", r, c, i);
                    IJV.d[i].row = c;
                    IJV.d[i].col = r;
                    IJV.d[i].Re  = L.N[L.S[s].N_idx + n_ptr];/* same value */
                    i++;
                }
                ++n_ptr;
            }
        }
    }
if (DEBUG) {
gprintf("spflip_sym before sort\n");
for (i = 0; i < LLt_nnz; i++) {
gprintf("IJV[%3d] = %3d %3d % 12.6e\n", 
i, IJV.d[i].row, IJV.d[i].col, IJV.d[i].Re); 
} 
} 

    qsort(IJV.d, LLt_nnz, sizeof(ijv), &comp_IJV_cr);

if (DEBUG) {
gprintf("spflip_sym after  sort\n");
for (i = 0; i < LLt_nnz; i++) {
gprintf("IJV[%3d] = %3d %3d % 12.6e\n", 
i, IJV.d[i].row, IJV.d[i].col, IJV.d[i].Re); 
}
}
    /* 
     * Put a sparse matrix on the stack using the given IJV pattern.
     */
    transpose = FALSE;
    status    = f_ijv2sp(name      ,  /* in  */
                         IJV       ,  /* in  (memory freed)               */
                         indexed   ,  /* in  1 make room for int. indices */
                         transpose ); /* in  1 = treat rows as cols       */
    LLt = L; /* meaningless, but shuts up GCC about LLt possibly uninitialized*/
    if (indexed) {
        LLt = sparse_overlay(tos);
        memcpy(LLt.row_idx, L.row_idx, L.H[ROWS]*sizeof(int));
        memcpy(LLt.col_idx, L.row_idx, L.H[ROWS]*sizeof(int));
    }
    lop();    /* [L] */

    set_symm(tos);
    clr_low_tri(tos);

    return status;

} /* 1}}} */
int  spinflate()   /* spinflate (hA hRind hCind R C --- hB) {{{1 */
/* 
 * man entry:  spinflate {{{2
 * (hA_sp hRind hCind R C --- hB_sp) Creates a sparse matrix [B] by inserting terms of the sparse matrix [A] into [B] according to the row and column index vectors {Rind} and {Cind} such that, using matlab notation, B=spnull(R,C); B(Rind:Cind) = A;    R and C are scalars which define the maximum row and maximum column in [B].
Example:
  "math.v" source    # to get ranint
  11 seedset 
  3 4 .5  sprand is A
  7 is R
  8 is C
  1 R 1 3 ranint is Rind
  1 C 1 4 ranint is Cind
produces
  [A] = 
  [ 0.4675   0.4723        0        0 ]
  [      0        0   0.2185   0.1406 ]
  [      0   0.2816   0.7136        0 ]
  R=7,  C=8
  {Rind} = [3 5 6]
  {Cind} = [4 1 2 6]
The output of 'A Rind Cind R C spinflate' will be
  [      0        0   0       0    0       0     0    0 ]
  [      0        0   0       0    0       0     0    0 ]
  [ 0.4723        0   0  0.4675    0       0     0    0 ]
  [      0        0   0       0    0       0     0    0 ]
  [      0   0.2185   0       0    0  0.1406     0    0 ]
  [ 0.2816   0.7136   0       0    0       0     0    0 ]
  [      0        0   0       0    0       0     0    0 ]
 * category: math::matrix::sparse, math::matrix::partitioning
 * related: sparse, sp2ijv, spsubmat, sppartition, inflate
 * 2}}}
 */ 
{
int DEBUG = 0;
    SparseMatrix A;
    int     nColsA, nColsB, nRowsA, nRowsB, cmplex, R_size, C_size, n_nnz, i,
            n_nnz_out, n_col_out, got_max_row_ind = 0, got_max_col_ind = 0;
    double *R_ind, *C_ind, *in_IJV, *out_IJV;
    char    T[ERR_MSG_SIZE+1];

if (DEBUG) printf("top of spinflate\n");

    /* type- and bounds-check the input {{{2 */
    if (!popint(&nColsB)) return 0;
    if (!popint(&nRowsB)) return 0;
    if (tos->typ != MAT) {
        stkerr(" spinflate:  Cind",MATNOT);
        return 0;
    }
    if ((tos-1)->typ != MAT) {
        stkerr(" spinflate:  Rind",MATNOT);
        return 0;
    }
    if (!is_sparse(tos-2)) {
        stkerr(" spinflate: A", SPARSENOT);
        return 0;
    }
    A = sparse_overlay(tos-2);
    nRowsA = A.H[ROWS];
    nColsA = A.H[COLS];
    cmplex = is_complex(tos-2);
if (DEBUG) gprintf("spinflate [B]: R=%d  C=%d [A]: R=%d  C=%d\n", nRowsB, nColsB, nRowsA, nColsA);

    C_ind  = (tos-0)->mat;
    C_size = (tos-0)->row*(tos-0)->col;
if (DEBUG) gprintf("spinflate {Cind} size=%d\n", C_size);
    if (C_size != nColsA) {
        stkerr(" spinflate: ", COLSNOTV);
        return 0;
    }
    for (i = 0; i < C_size; i++) {
        if ((int) C_ind[i] < XBASE || 
            (int) C_ind[i] >= (nColsB + XBASE)) {
            snprintf(T, ERR_MSG_SIZE, 
                     " Cind[%d]=%d (out of bounds)", i+XBASE, (int) C_ind[i]);
            stkerr(" spinflate: ", T);
            return 0;
        }
    }

    R_ind  = (tos-1)->mat;
    R_size = (tos-1)->row*(tos-1)->col;
if (DEBUG) gprintf("spinflate {Rind} size=%d\n", R_size);
    if (R_size != nRowsA) {
        stkerr(" spinflate: ", ROWSNOTV);
        return 0;
    }
    for (i = 0; i < R_size; i++) {
if (DEBUG) gprintf("spinflate Rind[%2d]=%2d MAX=%2d\n", 
i, (int) R_ind[i], nRowsB - (1 - XBASE) );
        if ((int) R_ind[i] < XBASE || 
            (int) R_ind[i] >= (nRowsB + XBASE)) {
            snprintf(T, ERR_MSG_SIZE, 
                     " Rind[%d]=%d (out of bounds)", i+XBASE, (int) R_ind[i]);
            stkerr(" spinflate: ", T);
            return 0;
        }
    }
    /* 2}}} */

    rot();     /* ( A Rind Cind --- Rind Cind A ) */

    /* convert [A] to IJV form then remap its indices  */
    if (!sp2ijv()) return 0; /* ( Rind Cind A --- Rind Cind in_IJV ) */
    in_IJV = tos->mat;
    n_nnz  = tos->row;
    /* scan remapped indices to determine whether or not a
     * place-holder term at R,C is needed
     */
    for (i = 0; i < n_nnz; i++) {
if (DEBUG) gprintf("spinflate:  new row %2d, new col %2d; MAXROW %2d, MAXCOL %2d\n",
(int) R_ind[ (int) in_IJV[i      ]-XBASE], 
(int) C_ind[ (int) in_IJV[i+n_nnz]-XBASE], 
nRowsB-(1-XBASE), 
nColsB-(1-XBASE));

        if        ( got_max_row_ind && 
                    (int) C_ind[ (int) in_IJV[i+n_nnz]-XBASE] == 
                          nColsB-(1-XBASE)) {
            got_max_col_ind = 1;
            break;
        } else if ( got_max_col_ind && 
                    (int) R_ind[ (int) in_IJV[i      ]-XBASE] == 
                          nRowsB-(1-XBASE)) {
            got_max_row_ind = 1;
            break;
        } else if ((int) R_ind[ (int) in_IJV[i      ]-XBASE] == 
                         nRowsB-(1-XBASE)) {
            got_max_row_ind = 1;
        } else if ((int) C_ind[ (int) in_IJV[i+n_nnz]-XBASE] == 
                         nColsB-(1-XBASE)) {
            got_max_col_ind = 1;
        }
    }

    n_nnz_out = n_nnz;   /* number of rows in out_IJV */
    n_col_out = 3;       /* number of cols in out_IJV */
if (DEBUG) gprintf("spinflate: got_max_row_ind=%d, got_max_col_ind=%d\n", 
got_max_row_ind, got_max_col_ind); 
    if (!got_max_row_ind || !got_max_col_ind) {
        /* The last numeric term in [B] (at row R, column C) is empty.
         * Need to add a placeholder zero to contain the max row and
         * column values for spconvert.
         */
        ++n_nnz_out;
if (DEBUG) gprintf("spinflate: need to add placeholder null term\n");
    }

    if (cmplex)
        ++n_col_out;

    if (!matstk(n_nnz_out, n_col_out, "_IJV")) return 0;
    /* ( Rind Cind in_IJV --- Rind Cind in_IJV out_IJV ) */
    out_IJV = tos->mat;

    for (i = 0; i < n_nnz; i++) {
if (DEBUG) { 
gprintf("i=%2d:  %2d  %2d  %e  ", 
         i, (int) in_IJV[i        ], 
            (int) in_IJV[i+1*n_nnz], 
                  in_IJV[i+2*n_nnz]);
if (cmplex)
    gprintf("%e  ", in_IJV[i+3*n_nnz]);
gprintf("%2d  %2d\n", 
        (int) R_ind[ (int) in_IJV[i        ] - XBASE], 
        (int) C_ind[ (int) in_IJV[i + n_nnz] - XBASE] );
}

        out_IJV[i            ] = (double) R_ind[ (int) in_IJV[i        ]-XBASE];
        out_IJV[i+1*n_nnz_out] = (double) C_ind[ (int) in_IJV[i + n_nnz]-XBASE];
        out_IJV[i+2*n_nnz_out] = in_IJV[i + 2*n_nnz];
        if (cmplex)
            out_IJV[i+3*n_nnz_out] = in_IJV[i + 3*n_nnz];
    }

    if (n_nnz_out > n_nnz) {
        out_IJV[  n_nnz_out - 1] = (double) nRowsB;
        out_IJV[2*n_nnz_out - 1] = (double) nColsB;
        out_IJV[3*n_nnz_out - 1] = 0.0;
        if (cmplex)
            out_IJV[4*n_nnz_out - 1] = 0.0;
if (DEBUG) gprintf("spinflate: added placeholder null term for %d,%d\n", 
nRowsB, nColsB);
    }
    if (!spconvert()) return 0;
    /* ( Rind Cind in_IJV out_IJV --- Rind Cind in_IJV B ) */
    lop();  /* drop  in_IJV */
    lop();  /* drop  Cind   */
    lop();  /* drop  Rind   */

    return 1;

} /* 1}}} */
int  index_align() /* index_align (hX hY --- hx_Map hy_Map) {{{1 */
/* man entry:  index_align {{{2
 * (hX hY --- hx_Map hy_Map) Given two arrays of integers, {X} and {Y}, returns {x_Map} and {y_Map} which are XBASE-based index arrays that show where terms of {Y} fit in {X} and vice versa.  The terms of {y_Map} satisfy Y[i] = X[ y_Map[i] ] for all y_Map[i] which are >= XBASE, and the terms of {x_Map} satisfy X[i] = Y[ x_Map[i] ] for all x_Map[i] >= XBASE.  If x_Map[i] is negative, then Y[i] does not appear in {X} and similarly if y_Map[i] < XBASE, then X[i] does not appear in {Y}.  {X} and {Y} may contain duplicate terms.
Example:
  1based
  list: 84 39 78 79 76 14 55 ;
  list: 78 14 39             ;
  index_align
  swap .i nl .i nl
produces
 Column 1:
      -1      3      1     -1     -1      2     -1
 Column 1:
       3      6      2
 * category: math::matrix::partitioning
 * related: rake, claw, submat, index_merge
 * 2}}} */ 
{
int DEBUG = 0;
    int     i, error_code, X_size, Y_size, *X, *Y, *ix_Map, *iy_Map;
    double *A, *B, *dx_Map, *dy_Map;
    char    T[ERR_MSG_SIZE+1];

if (DEBUG) gprintf("top of index_align\n");

    /* type- and bounds-check the input {{{2 */
    if (tos->typ != MAT) {
        stkerr(" index_align:  B",MATNOT);
        return 0;
    }
    if ((tos-1)->typ != MAT) {
        stkerr(" index_align:  A",MATNOT);
        return 0;
    }
    if (is_complex(tos) || is_complex(tos-1)) {
        stkerr(" index_align: ", "inputs must be real");
        return 0;
    }

    B      = (tos-0)->mat;
    Y_size = (tos-0)->row*(tos-0)->col;
    A      = (tos-1)->mat;
    X_size = (tos-1)->row*(tos-1)->col;
    /* 2}}} */

    if (!matstk(X_size, 1, "_xMap")) return 0;
    dx_Map = tos->mat;
    if (!matstk(Y_size, 1, "_yMap")) return 0;
    dy_Map = tos->mat;

    if ((X = (int *) malloc(X_size*sizeof(int))) == NULL) {
        stkerr(" index_align:  X ", MEMNOT);
        return 0;
    }
    for (i = 0; i < X_size; i++) X[i] = (int) A[i];

    if ((Y = (int *) malloc(Y_size*sizeof(int))) == NULL) {
        stkerr(" index_align:  Y ", MEMNOT);
        return 0;
    }
    for (i = 0; i < Y_size; i++) Y[i] = (int) B[i];

    if ((ix_Map = (int *) malloc(2*X_size*sizeof(int))) == NULL) {
        stkerr(" index_align:  X_map ", MEMNOT);
        return 0;
    }
    if ((iy_Map = (int *) malloc(2*Y_size*sizeof(int))) == NULL) {
        stkerr(" index_align:  Y_map ", MEMNOT);
        return 0;
    }
    for (i = 0; i < Y_size; i++) Y[i] = (int) B[i];

    if (!f_index_align(X_size, X, Y_size, Y, ix_Map, iy_Map, &error_code)) {
        switch (error_code) {
            case -1:
                strncpy(T, "malloc X failed", ERR_MSG_SIZE);
                break;
            case -2:
                strncpy(T, "malloc Y failed", ERR_MSG_SIZE);
                break;
            default:
                snprintf(T, ERR_MSG_SIZE, "unexpected error code %d",
                        error_code);
        }
        rev(); lop(); /* drops B */
        rev(); lop(); /* drops A */
        stkerr(" index_align:  ", T );
        return 0;
    }

    for (i = 0; i < X_size; i++) {
        if (ix_Map[i] < 0) {
            dx_Map[i] = (double) ix_Map[i];
        } else {
            dx_Map[i] = (double) ix_Map[i] + XBASE;
        }
    }
    for (i = 0; i < Y_size; i++) {
        if (iy_Map[i] < 0) {
            dy_Map[i] = (double) iy_Map[i];
        } else {
            dy_Map[i] = (double) iy_Map[i] + XBASE;
        }
    }

    free(X);
    free(Y);
    free(ix_Map);
    free(iy_Map);

    rev(); lop(); /* drops B */
    rev(); lop(); /* drops A */

    return 1;
} /* 1}}} */
int  index_merge() /* index_merge (hX hY --- hy_Map hZ) {{{1 */
/* man entry:  index_merge {{{2
 * (hX hY --- hy_Map hZ) Given two arrays of integers, {X} and {Y}, returns {Z}, an array containing the union of {X} and {Y}.  The first X_size terms of {Z} are copies of {X} and the remaining terms are those {Y} terms which are not in {X}.  Also returns {y_Map}, an XBASE-based index array that show where terms of {Y}--if any--fit in {Z}.  The mapping array satisfies Y[i] = Z[ y_Map[i] ].  (Note there is no need for an {x_Map} array.)
Example:
  1based
  list: 84 39 78 79 76 14 55 ;
  list: 78 14  5 39          ;
  index_merge
  .i nl .i nl
produces
 Column 1:
      84     39     78     79     76     14     55      5
 Column 1:
       3      6      8      2
 * category: math::matrix::partitioning
 * related: rake, claw, submat, index_align
 * 2}}} */ 
{
int DEBUG = 0;
    int     i, error_code, X_size, Y_size, Z_size, 
           *X, *Y, *Z, *ix_Map, *iy_Map;
    double *A, *B, *dy_Map, *dZ;
    char    T[ERR_MSG_SIZE+1];

if (DEBUG) gprintf("top of index_merge\n");

    /* type- and bounds-check the input {{{2 */
    if (tos->typ != MAT) {
        stkerr(" index_merge:  B",MATNOT);
        return 0;
    }
    if ((tos-1)->typ != MAT) {
        stkerr(" index_merge:  A",MATNOT);
        return 0;
    }
    if (is_complex(tos) || is_complex(tos-1)) {
        stkerr(" index_merge: ", "inputs must be real");
        return 0;
    }

    B      = (tos-0)->mat;
    Y_size = (tos-0)->row*(tos-0)->col;
    A      = (tos-1)->mat;
    X_size = (tos-1)->row*(tos-1)->col;
    /* 2}}} */

    if (!matstk(Y_size, 1, "_yMap")) return 0;
    dy_Map = tos->mat;

    if ((X = (int *) malloc(X_size*sizeof(int))) == NULL) {
        stkerr(" index_merge:  X ", MEMNOT);
        return 0;
    }
    for (i = 0; i < X_size; i++) X[i] = (int) A[i];

    if ((Y = (int *) malloc(Y_size*sizeof(int))) == NULL) {
        stkerr(" index_merge:  Y ", MEMNOT);
        return 0;
    }
    for (i = 0; i < Y_size; i++) Y[i] = (int) B[i];

    if ((Z = (int *) malloc((X_size + Y_size)*sizeof(int))) == NULL) {
        stkerr(" index_merge:  Z ", MEMNOT);
        return 0;
    }
    Z_size = X_size + Y_size; /* worst case */
    for (i = 0; i < Z_size; i++) Z[i] = 0;

    if ((ix_Map = (int *) malloc(2*X_size*sizeof(int))) == NULL) {
        stkerr(" index_merge:  X_map ", MEMNOT);
        return 0;
    }
    if ((iy_Map = (int *) malloc(2*Y_size*sizeof(int))) == NULL) {
        stkerr(" index_merge:  Y_map ", MEMNOT);
        return 0;
    }
    for (i = 0; i < Y_size; i++) Y[i] = (int) B[i];

    if (f_index_merge(X_size, X, 
                      Y_size, Y, 
                      iy_Map, 
                     &Z_size, Z,   /* Z_size comes back as correct value */
                     &error_code) < 0) {
        switch (error_code) {
            case -1:
                strncpy(T, "malloc X failed", ERR_MSG_SIZE);
                break;
            case -2:
                strncpy(T, "malloc Y failed", ERR_MSG_SIZE);
                break;
            default:
                snprintf(T, ERR_MSG_SIZE, "unexpected error code %d",
                        error_code);
        }
        rev(); lop(); /* drops Y */
        rev(); lop(); /* drops X */
        stkerr(" index_merge:  ", T );
        return 0;
    }
    if (!matstk(Z_size, 1, "_Z")) return 0;
    dZ = tos->mat;

    for (i = 0; i < Y_size; i++) dy_Map[i] = (double) iy_Map[i] + XBASE;
    for (i = 0; i < Z_size; i++)     dZ[i] = (double) Z[i];

    free(X);
    free(Y);
    free(Z);
    free(ix_Map);
    free(iy_Map);

    rev(); lop(); /* drops Y */
    rev(); lop(); /* drops X */

    return 1;
} /* 1}}} */
int  spadd_ijv()   /* spadd_ijv (hA hB --- hC) {{{1 */
/* man entry:  spadd_ijv {{{2
 * spadd_ijv (hA hB --- hC) Return the sum of two matrices sparse matrices.  If both inputs are internally indexed, [B] is reordered to align with [A] before the sum is computed.
 * category: math::matrix::partitioning, math::matrix::operator
 * related: spsum, add, spadd
 * 2}}} */ 
{
int DEBUG = 0;
    int          A_indexed, B_indexed, C_indexed;
    IJVMatrix    A_ijv    , B_ijv    , C_ijv    ;
    SparseMatrix A_sp     , B_sp     ;

if (DEBUG) gprintf("top of ijv_add\n");

    if (!is_sparse(tos) || !is_sparse(tos-1)) {
        stkerr(" ijv_add:  "," both inputs must be sparse");
    }

    B_sp      = sparse_overlay(tos);
    B_indexed = is_indexed(    tos);

    A_sp      = sparse_overlay(tos-1);
    A_indexed = is_indexed(    tos-1);

    C_indexed = (A_indexed && B_indexed);

    if (!f_sp2ijv(A_sp, is_indexed(tos-1), &A_ijv)) {
        stkerr(" spadd_ijv:  ",MEMNOT);
        return 0;
    }
if (DEBUG) {
gprintf("spadd_ijv: A\n");
f_ijv_dump(A_ijv);
}

    if (!f_sp2ijv(B_sp, is_indexed(tos-0), &B_ijv)) {
        stkerr(" spadd_ijv:  ",MEMNOT);
        return 0;
    }
if (DEBUG) {
gprintf("spadd_ijv: B\n");
f_ijv_dump(B_ijv);
}

    if (!f_ijv_add(A_ijv, &B_ijv, &C_ijv)) {
        stkerr(" spadd_ijv:  ",MEMNOT);
        return 0;
    }
if (DEBUG) {
gprintf("spadd_ijv: C\n");
f_ijv_dump(C_ijv);
}

    if (!f_ijv2sp("_C", C_ijv, C_indexed, 0)) {  /* puts [C] on stack */
        stkerr(" spadd_ijv:  ",MEMNOT);
        return 0;
    }

    lop();
    lop();

    return 1;
} /* 1}}} */

/* internal C functions */
int sparse_stk(int   nRows      , /* {{{1 */ 
               int   nCols      , /* in   */
               int   nStr       , /* in  number of strings       */
               int   nNz_terms  , /* in  number of nonzero terms */
               int   cmplx      , /* in     0=real  1=complex    */
               int   indexed    , /* in     0=no    1=yes        */
               char *name       , /* in   */
               SparseMatrix *m)   /* out  */
/* put a sparse matrix on the stack */
{
int DEBUG = 0;
          
    int  ptr_H, ptr_S_start, ptr_N_start, ptr_S, ptr_row_idx, 
         ptr_col_idx, ptr_N, size,
         success = 1;        

if (DEBUG) {
gprintf("sparse_stk inputs:\n");
gprintf("nRows=%d nCols=%d nStr=%d nNz_terms=%d cmplx=%d indexed=%d name=[%s]\n",
nRows, nCols, nStr, nNz_terms, cmplx, indexed, name);
}
    sp_data_offsets( nRows      ,
                     nCols      ,  /* in */
                     nStr       ,  /* in */
                     nNz_terms  ,  /* in */
                     cmplx      ,  /* in */
                     indexed    ,  /* in */
                    &ptr_H      ,  /* out int offset to H[]       */
                    &ptr_S_start,  /* out int offset to S_start[] */
                    &ptr_N_start,  /* out int offset to N_start[] */
                    &ptr_S      ,  /* out int offset to S[]       */
                    &ptr_row_idx,  /* out int offset to row_idx[] */
                    &ptr_col_idx,  /* out int offset to col_idx[] */
                    &ptr_N      ,  /* out int offset to N[]       */
                    &size);        /* out # bytes taken by matrix */

if (DEBUG) {
gprintf("sparse_stk outputs:\n");
gprintf("ptr_H=%d ptr_S_start=%d, ptr_N_start=%d ptr_S=%d ptr_row_idx=%d\n",
ptr_H, ptr_S_start, ptr_N_start, ptr_S, ptr_row_idx);
gprintf("ptr_col_idx=%d ptr_N=%d size=%d\n",
ptr_col_idx, ptr_N, size);
}
    /* allocate a VOL to store the sparse matrix and put it on the stack */
    if (!volstk(1, size, name)) return 0;

    if (cmplx)
        set_complex(tos);
    if (indexed)
        set_indexed(tos);

if (DEBUG) printf("sparse_stk address of tos->tex is %ld\n", (long) tos->tex);
    sp_set_header(tos         ,       /* in/out */
                  m           ,       /* out */
                  ptr_H       ,       /* in  */
                  ptr_S_start ,       /* in  */
                  ptr_N_start ,       /* in  */
                  ptr_S       ,       /* in  */
                  ptr_row_idx ,       /* in  */
                  ptr_col_idx ,       /* in  */
                  ptr_N       ,       /* in  */
                  size        ,       /* in  */
                  nRows       ,       /* in  */
                  nCols       ,       /* in  */
                  cmplx       ,       /* in  */
                  nStr        ,       /* in  */
                  nNz_terms);         /* in  */
    return success;
} /* 1}}} */
void sp_data_offsets(int  nRows       , /* {{{1 */ 
                     int  nCols       ,
                     int  nStr        , /* in  number of strings       */
                     int  nNz         , /* in  number of nonzero terms */
                     int  cmplx       , /* in     0=real  1=complex    */
                     int  indexed     , /* in     0=no    1=yes        */
                     int *ptr_H       , /* out int offset to H[]       */
                     int *ptr_S_start , /* out int offset to S_start[] */
                     int *ptr_N_start , /* out int offset to S_start[] */
                     int *ptr_S       , /* out int offset to S[]       */
                     int *ptr_row_idx , /* out int offset to row_idx[] */
                     int *ptr_col_idx , /* out int offset to col_idx[] */
                     int *ptr_N       , /* out int offset to N[]       */
                     int *size)         /* out # bytes taken by matrix */
{
int DEBUG = 0;

    *ptr_H       =                SPARSE_MAGIC_SIZE;
    *ptr_S_start = *ptr_H       + SPARSE_HDR_SIZE;
    *ptr_N_start = *ptr_S_start +     (nCols + 1);
    *ptr_S       = *ptr_N_start +     (nCols + 1);

    if (indexed) {
        *ptr_row_idx = *ptr_S   +            nStr*sizeof(str_t)/sizeof(int);
        *ptr_col_idx = *ptr_row_idx + nRows;
        *ptr_N       = *ptr_col_idx + nCols;
    } else {
        *ptr_row_idx = 0;
        *ptr_col_idx = 0;
        *ptr_N       = *ptr_S   +            nStr*sizeof(str_t)/sizeof(int);
    }


    if (*ptr_N % 2) /* force N[] to start on address which is multiple of 8 */
        ++(*ptr_N); /* see code with align_to_8 in sparse_overlay() */

    *size        = (*ptr_N)*sizeof(int) + 
                   NUM_PER_TERM(cmplx)*nNz*sizeof(double);

if (DEBUG) {
gprintf("sp_data_offsets:\n");
gprintf("H       : %d\n", *ptr_H);
gprintf("S_start : %d\n", *ptr_S_start);
gprintf("N_start : %d\n", *ptr_N_start);
gprintf("S       : %d\n", *ptr_S);
gprintf("row_idx : %d\n", *ptr_row_idx);
gprintf("col_idx : %d\n", *ptr_col_idx);
gprintf("N       : %d\n", *ptr_N);
gprintf("size    : %d\n", *size);
}

} /* 1}}} */
SparseMatrix sparse_overlay(stkitem *X) /* {{{1 */
{
    SparseMatrix m;
    int          align_to_8 = 0; /* Sun + gcc; HP need doubles to start
                                  * at an address which is an even multiple
                                  * of 8.  x86 Linux, AIX don't care.
                                  */

    if (0) {  /* 0 -> for traceback info with gdb
               * 1 -> protect against null pointers and exit cleanly
               */
        if (!X->tex) {
            stkerr(" sparse_overlay: ", 
                   "got null pointer instead of sparse matrix");
            halt(); /* return 0; won't work because this function 
                     * must return a SparseMatrix type
                     */
        }
    }
    m.data       = X->tex;
    m.magic      = (int *) m.data;
    if ((m.magic[0] != SPARSE_MAGIC_0) ||
        (m.magic[1] != SPARSE_MAGIC_1) ||
        (m.magic[2] != SPARSE_MAGIC_2) ||
        (m.magic[3] != SPARSE_MAGIC_3)) {
        stkerr(" sparse_overlay: ", SPARSENOT);
        halt();
    }
    m.H       = &m.magic[   SPARSE_MAGIC_SIZE      ];
    m.S_start = &m.H[       SPARSE_HDR_SIZE        ];
    m.N_start = &m.S_start[ m.H[COLS] + 1          ];
    m.S       = (str_t *)  &m.N_start[m.H[COLS] + 1];
    if (is_indexed(X)) {
        m.row_idx = (int *) &m.S[m.H[n_STR]];
        m.col_idx =         &m.row_idx[m.H[ROWS]];
        /* see corresponding size change in sp_data_offsets() */
        if ((m.H[n_STR] + m.H[ROWS] + m.H[COLS]) % 2) align_to_8 = 4;
        m.N       = (double *) ( (long) &m.col_idx[m.H[COLS]] + align_to_8 );
    } else {
        m.row_idx = 0;
        m.col_idx = 0;
        /* see corresponding size change in sp_data_offsets() */
        if (m.H[n_STR] % 2) align_to_8 = 4;
        m.N       = (double *) ( (long) &m.S[m.H[n_STR]] + align_to_8 );
    }

    return m;
} /* 1}}} */
void sp_set_header(stkitem      *X  , /* {{{1 */
                   SparseMatrix *m  ,
                   int   ptr_H      ,  /* in  int offset to H[]       */
                   int   ptr_S_start,  /* in  int offset to S_start[] */
                   int   ptr_N_start,  /* in  int offset to N_start[] */
                   int   ptr_S      ,  /* in  int offset to S[]       */
                   int   ptr_row_idx,  /* in  int offset to row_idx[] */
                   int   ptr_col_idx,  /* in  int offset to col_idx[] */
                   int   ptr_N      ,  /* in  int offset to N[]       */
                   int   size       ,  /* in  # bytes taken by matrix */
                   int   nRows      ,  /* in */
                   int   nCols      ,  /* in */
                   int   cmplx      ,  /* in */
                   int   nStr       ,  /* in */
                   int   nNz)          /* in */
/*
 *  Initialize the sparse matrix header and everything else known about
 *  the matrix.
 */
{
int DEBUG = 0;

if (DEBUG) {
gprintf("sp_set_header ptr_H      =%9d\n", ptr_H);
gprintf("sp_set_header ptr_S_start=%9d\n", ptr_S_start);
gprintf("sp_set_header ptr_N_start=%9d\n", ptr_N_start);
gprintf("sp_set_header ptr_S      =%9d\n", ptr_S);
gprintf("sp_set_header ptr_N      =%9d\n", ptr_N);
gprintf("sp_set_header size       =%9d\n", size );
}
    (*m).data               = X->tex;
    (*m).magic              = (int    *)   (*m).data;
    (*m).H                  = (int    *) &((*m).magic[ptr_H      ]);
    (*m).S_start            = (int    *) &((*m).magic[ptr_S_start]);
    (*m).N_start            = (int    *) &((*m).magic[ptr_N_start]);
    (*m).S                  = (str_t  *) &((*m).magic[ptr_S      ]);
    (*m).row_idx            = (int    *) &((*m).magic[ptr_row_idx]);
    (*m).col_idx            = (int    *) &((*m).magic[ptr_col_idx]);
    (*m).N                  = (double *) &((*m).magic[ptr_N      ]);

    (*m).magic[0]           = SPARSE_MAGIC_0; 
    (*m).magic[1]           = SPARSE_MAGIC_1; 
    (*m).magic[2]           = SPARSE_MAGIC_2; 
    (*m).magic[3]           = SPARSE_MAGIC_3; 
    (*m).H[ROWS]            = nRows;
    (*m).H[COLS]            = nCols; 
    (*m).H[COMPLX]          = cmplx; 
    (*m).H[n_STR]           = nStr; 
    (*m).H[n_NONZ]          = nNz; 
    (*m).H[DATA_SIZE]       = size;
    set_sparse(X);
    if (cmplx)
        set_complex(X);

    (*m).S_start[0]         = 0;
    (*m).N_start[0]         = 0;
    if (nNz) {
        (*m).S[0].start_row = 0;
        (*m).S[0].len       = 0;
        (*m).S[0].N_idx     = 0;
    }
if (DEBUG) {
printf("sp_set_header addresses:\n");
printf("  m.data    %ld %lx\n", 
(long) (*m).data    , (long) (*m).data       );
printf("  m.magic   %ld %lx  %ld bytes\n", 
(long) (*m).magic   , (long) (*m).magic  , (long) (*m).H       - (long) (*m).magic  );
printf("  m.H       %ld %lx  %ld bytes\n", 
(long) (*m).H       , (long) (*m).H      , (long) (*m).S_start - (long) (*m).H      );
printf("  m.S_start %ld %lx  %ld bytes\n", 
(long) (*m).S_start , (long) (*m).S_start, (long) (*m).N_start - (long) (*m).S_start);
printf("  m.N_start %ld %lx  %ld bytes\n", 
(long) (*m).N_start , (long) (*m).N_start, (long) (*m).S       - (long) (*m).N_start);
printf("  m.S       %ld %lx  %ld bytes\n", 
(long) (*m).S       , (long) (*m).S      , (long) (*m).N       - (long) (*m).S      );
printf("  m.N       %ld %lx  %ld bytes; mod(8)=%ld\n", 
(long) (*m).N, (long) (*m).N, (long) &((*m).data[size]) - (long) (*m).N, 
((long) (*m).N) % 8);
printf("  addr m.N[0]=%ld %lx\n", (long) &(*m).N[0], (long) &(*m).N[0]);
}

} /* 1}}} */
void stringify_vector(  /* {{{1 */
                      int           nrows,  /* in     number of rows */
                      int           c,      /* in     column index */
                      double       *vector, /* in     the dense vector */
                      int           cmplx,  /* in     0=real  1=complex */
                      SparseMatrix *m,      /* in/out */
                      int          *iS,     /* in/out index to S */
                      int          *iN)     /* in/out index to N */
/*
 *  Input is a (dense) numeric array.  
 *  Output is sparse string pointer and data arrays.
 */
{
int DEBUG = 0;
    int r, got_nonzero, NPT;

    int prev_nonzero    =  0; /* 1 -> previous term in column was non-zero */
    int n_str_this_col  =  0;
    int n_term_this_col =  0;

if (DEBUG) {
printf("stringify_vector: iS=%d   iN=%d\n", *iS, *iN);
printf("stringify_vector: (*m).S_start[%d]=%d  m.N[0]=%e\n", 
c, (*m).S_start[c], (*m).N[0]);
}
    (*m).S_start[c]  = *iS + 1;
    (*m).N_start[c]  = *iN;
    NPT              = NUM_PER_TERM(cmplx);  /* 1 for real;  2 for complex */
    for (r = 0; r < NPT*nrows; r += NPT) {
if (DEBUG) {
printf("stringify_vector: r=%d   iN=%d,  %10.5e", r, *iN, vector[r]);
if (cmplx) printf(", %10.5e\n", vector[r+1]);
printf("\n"); 
}
        got_nonzero = (!cmplx && (fabs(vector[r]) > SPARSE_ZERO_THRESH)) ||
                      ( cmplx && ((fabs(vector[r])   > SPARSE_ZERO_THRESH) ||
                                  (fabs(vector[r+1]) > SPARSE_ZERO_THRESH)));
if (DEBUG)
printf("r=%d got_nonzero=%d\n", r, got_nonzero);
        if (got_nonzero) {
            if (prev_nonzero) { /* term r in same string as r-1  */
                ++((*m).S[*iS].len);   /* str length */
if (DEBUG)
printf("stringify_vector: string %d grew to %d\n", *iS, (*m).S[*iS].len);
            } else {            /* term r begins a new string    */
                ++(*iS);
                (*m).S[*iS].len       =   1;       /* str length */
                (*m).S[*iS].start_row =   r/NPT;   /* start row  */
                (*m).S[*iS].N_idx     = *iN;       /* index into N[] */
                ++n_str_this_col;
            }
            if (cmplx) {
                (*m).N[*iN]           = vector[r];
                (*m).N[*iN+1]         = vector[r+1];
            } else {
                (*m).N[*iN]           = vector[r];
            }
if (DEBUG) {
printf("stringify_vector: (*m).N[%d] = %e ", (*iN), vector[r]);
if (cmplx) {
printf(" (*m).N[%d] = %e \n", *iN+1, vector[r+1]);
}
printf("\n");
}
            *iN += NPT;
            ++n_term_this_col;
            prev_nonzero = 1;
        } else {
            prev_nonzero = 0;
        }
    }
    (*m).S_start[c+1] = (*m).S_start[c] + n_str_this_col;
    (*m).N_start[c+1] = (*m).N_start[c] + n_term_this_col*NPT;
if (DEBUG) {
printf("end stringify_vector column %d, got %d strings, %d terms\n", 
c, n_str_this_col, n_term_this_col);
printf("  m.S_start[%2d]=%3d  m.N_start[%2d]=%3d\n",
c, (*m).S_start[c], c, (*m).N_start[c]);
printf("  m.S_start[%2d]=%3d  m.N_start[%2d]=%3d\n",
c+1, (*m).S_start[c+1], c+1, (*m).N_start[c+1]);
for (r = 0; r < n_str_this_col; r++) {
printf("    m.S[%2d].start_row=%3d\n", 
(*m).S_start[c]+r, (*m).S[(*m).S_start[c]+r].start_row);
printf("    m.S[%2d].len      =%3d\n", 
(*m).S_start[c]+r, (*m).S[(*m).S_start[c]+r].len      );
printf("    m.S[%2d].N_idx    =%3d\n", 
(*m).S_start[c]+r, (*m).S[(*m).S_start[c]+r].N_idx    );
}
printf("<- stringify_vector\n");
}
} /* 1}}} */
void test_random_subset() {  /* {{{1 */
    int i, j, rand_list[10];
    for (i = 0; i <= 10; i++) {
        random_subset(10, 10, rand_list);
        for (j = 0; j < 10; j++) {
            printf(" %3d", rand_list[j]);
        }
        printf("\n");
    }
} /* 1}}} */
int  random_subset(  /* {{{1 */
                   int  max_value,
                   int  N        ,
                   int *list)
/*
 * Populate list[] with N unique integers on 0..(max value-1).
 * N must be <= max_value.
 */
{
    int i, j, k, *ignore_list = 0, real_N = 0,
        backwards = 0, list_has_duplicates = 1;
int DEBUG = 0;
                                                                                
    if ( N >= max_value ) { /* special case; want the entire list */
        for (i = 0; i < N; i++) {
            list[i] = i;
        }
        return 1;
    }
                                                                                
    if ( (2*N)/max_value >= 1 ) {
        /* Requested more than half the of the range; faster to
         * solve the problem backwards--find numbers which won't
         * be returned.
         */
if (DEBUG) gprintf("asking for more than half; doing inverse\n");
        backwards   = 1;
        real_N      = N;
        N           = max_value - N;
        ignore_list = (int *) malloc( N*sizeof(int));
    }
                                                                                
    for (i = 0; i < N; i++) {
        list[i] = (int) ((double) max_value*rand()/
                        ((double) RAND_MAX+1.0));
    }
                                                                                
    qsort(list, N, sizeof(int), &int_compare);
if (DEBUG) gprint_list("pass 1", N, list);

    while (list_has_duplicates) {
        list_has_duplicates = 0;
        for (i = 0; i < N-1; i++) {
            if (list[i] == list[i+1]) {
                list_has_duplicates = 1;
                list[i+1] = (int) ((double) max_value*rand()/
                                  ((double) RAND_MAX+1.0));
if (DEBUG) gprint_list("pass 2", N, list);
            }
        }
        if (list_has_duplicates) {
            qsort(list, N, sizeof(int), &int_compare);
if (DEBUG) gprint_list("pass 3", N, list);
        }
    }
                                                                                
    if (backwards) {
        /* Populate list[] with all integers on 0..(max_value-1)
         * except for those integers in ignore_list[].
         */
        memcpy(ignore_list, list, N*sizeof(int));
if (DEBUG) gprint_list("ignore", N, ignore_list);
if (DEBUG) gprintf("real_N=%d   N=%d\n", real_N, N);
        j = 0;
        k = 0;
        for (i = 0; i < max_value; i++) {
            if ((j < N) && (i == ignore_list[j])) {
                j++;
            } else {
                list[k++] = i;
                if (k == real_N) {  /* got all the terms, exit */
                    break;
                }
            }
        }
        free(ignore_list);
    }
    return 1;
} /* 1}}} */
void dump_sp_detail(SparseMatrix m) /* {{{1 */
{
    int c, s, r, n, end_row;

    printf("         ---- dump ----\n");
    printf("H[rows]= %5d %ld\n",   (int)  m.H[ROWS]     ,(long) &m.H[ROWS]     );
    printf("H[cols]= %5d %ld\n",   (int)  m.H[COLS]     ,(long) &m.H[COLS]     );
    printf("H[nStr]= %5d %ld\n",   (int)  m.H[n_STR]    ,(long) &m.H[n_STR]    );
    printf("H[nNnz]= %5d %ld\n",   (int)  m.H[n_NONZ]   ,(long) &m.H[n_NONZ]   );
    printf("H[cmpl]= %5d %ld\n",   (int)  m.H[COMPLX]   ,(long) &m.H[COMPLX]   );
    printf("H[size]= %5d %ld\n",   (int)  m.H[DATA_SIZE],(long) &m.H[DATA_SIZE]);
    printf("S_start         %ld\n",(long)  m.S_start);
    printf("N_start         %ld\n",(long)  m.N_start);
    printf("S               %ld\n",(long)  m.S);
    printf("S[0]            %ld\n",(long) &m.S[0]);
    printf("S[0].start_row  %ld\n",(long) &m.S[0].start_row);
    printf("S[0].len        %ld\n",(long) &m.S[0].len);
    printf("S[0].N_idx      %ld\n",(long) &m.S[0].N_idx);
    printf("N[0]            %ld\n",(long)  m.N);
    printf("N[%5d]        %ld\n", m.H[n_NONZ]-1, (long) &m.N[m.H[n_NONZ]-1]);
    printf("N[%5d]-start  %ld\n", m.H[n_NONZ]-1, 
                                 (long) &m.N[m.H[n_NONZ]-1] - (long) m.data);

    for (c = 0; c < m.H[COLS]; c++) {
        printf("Column %3d\n", c);
        printf("m.S_start[%d]=%d  A=%ld  m.S_start[%d]=%d  A=%ld\n",
                c,  m.S_start[c],   (long) &m.S_start[c],
                c+1,m.S_start[c+1], (long) &m.S_start[c+1]);
        printf("m.N_start[%d]=%d  A=%ld  m.N_start[%d]=%d  A=%ld\n",
                c,  m.N_start[c],   (long) &m.N_start[c],
                c+1,m.N_start[c+1], (long) &m.N_start[c+1]);
        n = 0;
        for (s = m.S_start[c]; s < m.S_start[c+1]; s++) {
            end_row = m.S[s].start_row + m.S[s].len - 1;
            for (r = m.S[s].start_row; r <= end_row; r++) {
                printf("R %3d N[%3d]=% 12.4f addr=%ld", 
                          r  
                        ,      m.N_start[c] + n  
                        , m.N[ m.N_start[c] + n ]
                        , (long) &m.N[ m.N_start[c] + n ]
                          );
                n++;
                if (m.H[COMPLX]) {
                    printf(", N[%3d]=% 12.4f addr=%ld", 
                                   m.N_start[c] + n  
                            , m.N[ m.N_start[c] + n ]
                            , (long) &m.N[ m.N_start[c] + n ]
                              );
                    n++;
                }
                printf("\n"); 
                if (r >= m.H[ROWS]) break;
            }
            if (s >= m.H[n_STR]) break;
        }
    }
} /* 1}}} */
void f_sp_dump(const char *title, stkitem *X) /* {{{1 */
{
    int c, nonZs, cmplx;
    SparseMatrix m;

    m     = sparse_overlay(X);
    cmplx = m.H[COMPLX];

    gprintf("%s", title);                           nc();
    gprintf("m.H[ROWS]      = %d", m.H[ROWS]     ); nc();
    gprintf("m.H[COLS]      = %d", m.H[COLS]     ); nc();
    gprintf("m.H[n_STR]     = %d", m.H[n_STR]    ); nc();
    gprintf("m.H[n_NONZ]    = %d", m.H[n_NONZ]   ); nc();
    gprintf("m.H[COMPLX]    = %d", m.H[COMPLX]   ); nc();
    gprintf("m.H[DATA_SIZE] = %d", m.H[DATA_SIZE]); nc();
    for (c = 0; c < m.H[COLS] + 1; c++) {  /* loop over columns + 1 */
        gprintf("m.S_start[%2d]=%2d  m.N_start[%2d]=%2d", c, m.S_start[c], 
                                                          c, m.N_start[c]); 
        if (c < m.H[COLS]) {
            nonZs = m.N_start[c+1] - m.N_start[c];
            if (m.H[COMPLX])
                nonZs /= 2;
            gprintf("    col %3d has %3d strings, %3d nonzeros",
                         c, m.S_start[c+1] - m.S_start[c], nonZs);
        }
        nc();
    }
    for (c = 0; c < m.H[n_STR]; c++) {     /* loop over string data */
        gprintf("m.S[%2d].start_row=%2d   len=%2d   N_idx=%2d   addr=%d %x", 
                c, m.S[c].start_row, m.S[c].len, m.S[c].N_idx, 
                &m.S[c].len, &m.S[c].len);
        nc();
    }
    if (is_indexed(X)) {
        for (c = 0; c < m.H[ROWS]; c++) {     /* loop over row indices */
            gprintf("m.row_idx[%2d]=%2d  addr=%d %x", 
                    c, m.row_idx[c], &m.row_idx[c], &m.row_idx[c] );
            nc();
        }
        for (c = 0; c < m.H[COLS]; c++) {     /* loop over col indices */
            gprintf("m.col_idx[%2d]=%2d  addr=%d %x", 
                    c, m.col_idx[c], &m.col_idx[c], &m.col_idx[c] );
            nc();
        }
    }
    for (c = 0; c < NUM_PER_TERM(cmplx)*m.H[n_NONZ]; c += NUM_PER_TERM(cmplx)) {
        /* loop over numeric data */
        if (m.H[COMPLX]) {
            gprintf("m.N[%2d]=% 23.16le,% 23.16le addr=%d %x", 
                      c,m.N[c],m.N[c+1], &m.N[c], &m.N[c]);
        } else 
            gprintf("m.N[%2d]=% 23.16le addr=%d %x", 
                      c, m.N[c], &m.N[c], &m.N[c]);
        nc();
    }
    return;
} /* 1}}} */
int  f_str_overlap(str_t A,              /* in       {{{1 */ 
                   str_t B,              /* in                                    */
                   int   allow_adjacent, /* in Specifies treatment of adjoining 
                                               strings.  For example,
                                                   .....xxxxxx..
                                                   ..xxx........
                                               1 => yes, this overlaps
                                               0 => no , not an overlap           */
                   int  *A_offset,       /* out Offset to A.start_row to start of 
                                                overlap.  0 if no overlap.        */
                   int  *B_offset,       /* out Offset to B.start_row to start of 
                                                overlap.  0 if no overlap.        */
                   int  *length  )       /* out Length of overlap.                */
{
int DEBUG = 0;
    int type = 0, A_end_row, B_end_row;

    /* Classify all the possible relative positions of two strings.  Returns
     * 0 if the strings do not overlap in any way (depends on value of
     * 'allow_adjacent'), otherwise returns the overlap type:
     *
     * type = 1:
     *      A's first row is less than or equal to B's first row and A's last
     *      row is between B's first and last row:
     *            A      ....xxxxxx........ 
     *            B      ......xxxxxxxx....   
     *      (This case holds for A and B starting and ending at the same locations.)
     *
     * type = 2:
     *      A's first row is between B's first and last row, and A's last row
     *      is greater than or equal to B's last row.
     *            A      ........xxxxxxxx.. 
     *            B      ......xxxxxxxx....   
     *
     * type = 3:
     *      A is contained entirely within B:
     *            A      .......xxx........ 
     *            B      ......xxxxxxxx....   
     *
     * type = 4:
     *      B is contained entirely within A:
     *            A      ......xxxxxxxx....  
     *            B      ........xxxx...... 
     *
     * type = 5 (a possible overlap only if allow_adjacent == 1):
     *      A ends one position before B begins:
     *            A      ...xxxxx..........  
     *            B      ........xxxx...... 
     *
     * type = 6 (a possible overlap only if allow_adjacent == 1):
     *      B ends one position before A begins:
     *            A      ........xxxxxx....
     *            B      ......xx..........  
     */

    *A_offset = 0;
    *B_offset = 0;
    *length   = 0;
    A_end_row = A.start_row + A.len - 1;
    B_end_row = B.start_row + B.len - 1;
    allow_adjacent = ( allow_adjacent ? 1 : 0 );
if (DEBUG) {
gprintf("f_str_overlap input A str (%d,%d), B str (%d,%d)\n",
A.start_row, A_end_row, B.start_row, B_end_row);
}

    /* First check the most likely case:  no overlap at all */
    if ((A.start_row > (B_end_row + allow_adjacent)) || 
        (B.start_row > (A_end_row + allow_adjacent))) {
        type      = 0;
if (DEBUG) gprintf("f_str_overlap type %d\n", type);

    } else if ((A.start_row <= B.start_row) && (A_end_row <= B_end_row) &&
               (A_end_row   >= B.start_row)) {
        type      = 1;
        *A_offset = B.start_row - A.start_row;
        *length   = A_end_row   - B.start_row + 1;
if (DEBUG) gprintf("f_str_overlap type %d Aoff=%d len=%d\n",type,*A_offset,*length);

    } else if ((B.start_row <= A.start_row) && (B_end_row <= A_end_row) &&
               (B_end_row   >= A.start_row)) {
        type      = 2;
        *B_offset = A.start_row - B.start_row;
        *length   = B_end_row   - A.start_row + 1;
if (DEBUG) gprintf("f_str_overlap type %d Boff=%d len=%d\n",type,*B_offset,*length);

    } else if ((A.start_row >= B.start_row) && (A_end_row <= B_end_row)) {
        type      = 3;
        *B_offset = A.start_row - B.start_row;
        *length   = A.len;
if (DEBUG) gprintf("f_str_overlap type %d Boff=%d len=%d\n",type,*B_offset,*length);

    } else if ((B.start_row >= A.start_row) && (B_end_row <= A_end_row)) {
        type      = 4;
        *A_offset = B.start_row - A.start_row;
        *length   = B.len;
if (DEBUG) gprintf("f_str_overlap type %d Aoff=%d len=%d\n",type,*A_offset,*length);

    } else if (allow_adjacent && ((A_end_row + 1) == B.start_row)) {
        type      = 5;
if (DEBUG) gprintf("f_str_overlap type %d\n", type);

    } else if (allow_adjacent && ((B_end_row + 1) == A.start_row)) {
        type      = 6;
if (DEBUG) gprintf("f_str_overlap type %d\n", type);

    }
    return type;
} /* 1}}} */
int  f_col_overlap(int    n_str_A   ,    /* in  number of strings in {A}  {{{1 */
                   str_t *A         ,    /* in  array of {A} strings              */
                   int    n_str_B   ,    /* in  number of strings in {B}          */ 
                   str_t *B         ,    /* in  array of {A} strings              */
                   int    allow_adjacent,/* in  [see f_str_overlap()]             */
                   /*   --------  arguments below are outputs which are  -------- */
                   /*   --------  valid only if there is an overlap      -------- */
                   int  *A_str_idx  ,    /* in/out index to the active string in  */ 
                                         /*        {A}.  Caller must initialize.  */
                   int  *B_str_idx  ,    /* in/out index to the active string in  */
                                         /*        {B}.  Caller must initialize.  */
                   int  *A_offset   ,    /* out offset to A[A_str_idx].start_row  */
                                         /*     to the beginning of the overlap   */
                   int  *B_offset   ,    /* out offset to B[B_str_idx].start_row  */
                                         /*     to the beginning of the overlap   */
                   int  *overlap_len     /* out length of the overlap             */
                  ) {
    /* Inputs are string indexing information for two sparse vectors, A and B. 
     * Returns 1 if A and B have at least one nonzero at the
     * same index.  If terms of A and B do not overlap, returns 0.
     */
int DEBUG = 0;
    int B_end_row, overlap;

    while ((*A_str_idx < n_str_A) && (*B_str_idx < n_str_B)) {
if (DEBUG) gprintf("f_col_overlap A str %d (%d,%d) B str %d (%d,%d)\n",
*A_str_idx, A[*A_str_idx].start_row, A[*A_str_idx].start_row+A[*A_str_idx].len-1,
*B_str_idx, B[*B_str_idx].start_row, B[*B_str_idx].start_row+B[*B_str_idx].len-1);
        overlap = f_str_overlap(A[*A_str_idx], B[*B_str_idx], allow_adjacent, 
                                A_offset,      B_offset, overlap_len);
        if (overlap) {
            return overlap;
        } else {
            B_end_row   = B[*B_str_idx].start_row + B[*B_str_idx].len - 1;
            if (A[*A_str_idx].start_row > B_end_row) {
                ++(*B_str_idx);
            } else {
                ++(*A_str_idx);
            }
        }
    }
    return 0;
} /* 1}}} */
int  f_sp_error(stkitem *X) /* sanity check for sparse matrices {{{1 */
{   /* returns 0 if it looks good, >0 error code otherwise */
    int          c, s, end_row, NPT, err = 0;
#define TvSize 200
    char        *T, Tv[TvSize - 1];
    SparseMatrix m;

    m   = sparse_overlay(X);
    NPT = NUM_PER_TERM(m.H[COMPLX]);
    Tv[0] = 0;

    T = "(undefined)";
    if (         m.H[ROWS]     <= 0) {
        err = 1;
        T   = "header row count number of rows in header is <= 0";
    }
    if (!err && (m.H[ROWS]     >  1.0e8)) {
        err =  2;
        T   = "header row count is unusually large";
    }
    if (!err && (m.H[COLS]     <= 0)) {
        err =  3;
        T   = "header column count is <= 0";
    }
    if (!err && (m.H[COLS]     >  1.0e8)) {
        err =  4;
        T   = "header column count is unusually large";
    }
    if (!err && (m.H[n_STR]    <  0)) {
        err =  5;
        T   = "header string count is negative";
    }
    if (!err && (m.H[n_STR]    >  (m.H[ROWS]+1)*m.H[COLS]/2 + 1)) {
        err =  6;
        T   = "header string count exceeds half nRows x nCols";
    }
    if (!err && (m.H[n_NONZ]   <  0)) {
        err = 13;
        T   = "header non-zero term count is negative";
    }
    if (!err && (m.H[n_NONZ]   >  m.H[ROWS]*m.H[COLS])) {
        err = 14;
        T   = "header non-zero term count exceeds nRows x nCols";
    }
    if (!err && (m.S_start[0])) {
        err = 15;
        T   = "first pointer into S[] is non-zero";
    }
    if (!err && (m.N_start[0])) {                              
        err = 16;
        T   = "first pointer into N[] is non-zero";
    }
    for (c = 0; c < m.H[COLS]; c++) {
        if (!err && (m.S_start[c] < 0))                        {
            err = 17+100*c;
            snprintf(Tv, TvSize, 
                     "pointer to first string in col %d is negative", c);
        }
        if (!err && (m.S_start[c] > m.H[n_STR]))               {
            err = 18+100*c;
            snprintf(Tv, TvSize,
                     "pointer to first string in col %d > number of strings in the matrix", c);
        }
        if (!err && (m.N_start[c] > m.N_start[c+1]))           {
            err = 19+100*c;
            snprintf(Tv, TvSize,
                     "pointer to first numeric term in col %d > pointer to first numeric term in col %d", c, c+1);
        }
        if (!err && (m.N_start[c] > NPT*m.H[n_NONZ]))          {
            err = 20+100*c;
            snprintf(Tv, TvSize,
                     "pointer to first numeric term in col %d > number of nonzeros in the matrix", c);
        }
        if (!err && !(m.S_start[c+1] - m.S_start[c]) &&
                     (m.N_start[c+1] - m.N_start[c])) {
            err = 21+100*c;
            snprintf(Tv, TvSize,
                     "S_start[%d:%d] says no strings in col %d N_Start[%d:%d] says %d nonzeros", c+1, c, c, c+1, c, m.N_start[c+1] - m.N_start[c]);
        }
/*
        if (!err && ((m.S_start[c+1] - m.S_start[c]) > m.H[max_STR_PER_COL])) {
            err = 22+100*c;
            snprintf(Tv, TvSize,
                     "number of strings in col %d > number of strings in the matrix", c);
        }
 */
        if (!err && ((m.N_start[c+1] - m.N_start[c]) > NPT*m.H[ROWS])) {
            err = 23+100*c;
            snprintf(Tv, TvSize,
                     "number of nonzero terms in col %d > than number of nonzeros in the matrix", c);
        }
        if (!err && ((m.N_start[c+1] - m.N_start[c]) < 
                     (m.S_start[c+1] - m.S_start[c])*NPT)) {
            err = 24+100*c;
            snprintf(Tv, TvSize,
                     "number of strings in col %d > number of nonzeros in col %d", c, c);
        }
        if ((m.N_start[c+1] - m.N_start[c])  &&
            (m.N_start[c] != m.S[ m.S_start[c] ].N_idx)) {
            err = 34+100*c;
            snprintf(Tv, TvSize,
                 "N_start[%d]=%d != S[ S_start[%d]=%d ].N_idx=%d", 
                 c, m.N_start[c], c, m.S_start[c], m.S[ m.S_start[c] ].N_idx);
        }
        for (s = m.S_start[c]; s < m.S_start[c+1]; s++) {
            end_row = m.S[s].start_row + m.S[s].len - 1;
            if (!err && (end_row >= m.H[ROWS])) {
                err = 25+100*c;
                snprintf(Tv, TvSize,
                     "end row of string %d, col %d exceeds nRows", s, c);
            }
            if (!err && ((s    > m.S_start[c])   && 
                        ((s+1) < m.S_start[c+1]) &&
                         m.S[s+1].start_row <= m.S[s].start_row)) {
                err = 26+100*c;
                snprintf(Tv, TvSize,
                     "start row of string %d is less than start row of string %d (col %d)", s+1, s, c);
            }
            if (!err && (m.S[s].N_idx < 0)) {
                err = 27+100*c;
                snprintf(Tv, TvSize,
                     "N pointer for string %d is negative (col %d)", s, c);
            }
            if (!err && (m.S[s].N_idx > NPT*m.H[n_NONZ])) {
                err = 28+100*c;
                snprintf(Tv, TvSize,
                     "N pointer for string %d exceeds nonzeros in matrix (col %d)", s, c);
            }
            if (!err && (((s+1) < m.S_start[c+1]) &&
                         (m.S[s+1].N_idx <= m.S[s].N_idx))) {
                err = 29+100*c;
                snprintf(Tv, TvSize,
                     "N pointer for string %d is less than N pointer for string %d (col %d)", s+1, s, c);
            }
            if (!err && (((s+1) < m.S_start[c+1]) &&
                (m.S[s+1].N_idx - m.S[s].N_idx) != NPT*m.S[s].len)) {
                err = 30+100*c;
                snprintf(Tv, TvSize,
                     "difference of N pointers for strings %d and %d does not equal length of string %d (col %d)", s, s+1, s, c);
            }
            if (!err && ((s    >= m.S_start[c])   && 
                        ((s+1) <  m.S_start[c+1]) &&
                         m.S[s+1].start_row <= end_row)) {
                err = 31+100*c;
                snprintf(Tv, TvSize,
                     "start row of string %d is less than end row of string %d (col %d)", s+1, s, c);
            }
            if (m.S[s].N_idx % NPT) {
                err = 35+100*c;
                snprintf(Tv, TvSize,
                     "string %d has N_idx=%d which is not an even multiple of NPT=%d", 
                     s, m.S[s].N_idx, NPT);
            }
        }
    }

    for (c = 0; c < NPT*m.H[n_NONZ]; c += NPT) {   /* loop over numeric data */
        if (m.H[COMPLX]) {
            if (!err && ((fabs(m.N[c])   < SPARSE_ZERO_THRESH) &&
                (fabs(m.N[c+1]) < SPARSE_ZERO_THRESH))) {
                err = 32+100*c;
                snprintf(Tv, TvSize,
                     "complex numeric term at offsets %d,%d are below sparse numeric threshold", c, c+1);
            }
        } else {
            if (!err && (fabs(m.N[c]) < SPARSE_ZERO_THRESH)) {
                err = 33+100*c;
                snprintf(Tv, TvSize,
                     "numeric term at offset %d is below sparse numeric threshold", c);
            }
        }
    }

    if (err) {
        nc();
        if (err <= 16) 
            gprintf("sperror %2d: %s", err,       T);
        else
            gprintf("sperror %2d: %s", err % 100, Tv);
        nc();
    }
    return err;
} /* 1}}} */
int  f_speye(int indexed)  /* {{{1 */
/* 
 *   Puts an n x n sparse identity matrix on the stack.
 */ 
{
int DEBUG = 0;
    char   *name = "_I";
    int     i, n, nrows, ncols, nstr, num_size;
    SparseMatrix m;

    if (!popint(&n)) return 0;

    nrows         = n;
    ncols         = n;
    nstr          = n;
    num_size      = n;
    if (!sparse_stk(nrows      , /* in  */ 
                    ncols      , /* in  */
                    nstr       , /* in  number of strings       */
                    num_size   , /* in  number of nonzero terms */
                    0          , /* in  0=real  1=complex       */
                    indexed    , /* in  0=no    1=yes           */
                    name       , /* in  */
                   &m)) {        /* out */
        return 0;
    }
if (DEBUG)
printf("speye m.N[0] addr=%ld %lx\n", (long) &m.N[0], (long) &m.N[0]);

    for (i = 0; i < n; i++) {
        m.S_start[i] = i; m.S_start[i+1] = i+1;
        m.N_start[i] = i; m.N_start[i+1] = i+1;

        m.S[i].start_row = i;
        m.S[i].len       = 1;
        m.S[i].N_idx     = i;
if (DEBUG)
printf("speye m.N[%d] addr=%ld %lx\n", i, (long) &m.N[i], (long) &m.N[i]);
        m.N[i]           = (double) 1.0;
    }
    set_symm(   tos); /* The identity matrix is symmetric, */
    set_low_tri(tos); /* lower triangular,                 */
    set_up_tri( tos); /* and upper triangular.             */
    return 1;
} /* 1}}} */
int  f_spnull(int indexed) /* {{{1 */
/* Put a null sparse matrix on the stack.  */ 
{
    char   *name = "_N";
    int     i, nrows, ncols, nstr, num_size;
    SparseMatrix m;

    if (!popint(&ncols)) return 0;
    if (!popint(&nrows)) return 0;

    nstr          = 0;
    num_size      = 0;
    if (!sparse_stk(nrows      , /* in  */ 
                    ncols      , /* in  */
                    nstr       , /* in  number of strings       */
                    num_size   , /* in  number of nonzero terms */
                    0          , /* in  0=real  1=complex       */
                    indexed    , /* in  0=no    1=yes           */
                    name       , /* in  */
                   &m)) {        /* out */
        return 0;
    }

    for (i = 0; i < ncols; i++) {
        m.S_start[i] = 0; m.S_start[i+1] = 0;
        m.N_start[i] = 0; m.N_start[i+1] = 0;
    }
    return 1;
} /* 1}}} */
int  comp_IJV_cr(const void *a, const void *b ) { /* {{{1 */
    /* 
     *   Sort on column indices first, then row indices within each column.
     */ 
    const ijv *A = a;
    const ijv *B = b;
    if ( (*A).col == (*B).col) {   /* column indices are equal */
        return (*A).row - (*B).row;      /* sort on row    index */
    } else {
        return (*A).col - (*B).col;      /* sort on column index */
    }
} /* 1}}} */
int  comp_IJV_rc(const void *a, const void *b ) { /* {{{1 */
    /* 
     *   Sort on row indices first, then column indices within each row.
     */ 
    const ijv *A = a;
    const ijv *B = b;
    if ( (*A).row == (*B).row) {   /* row indices are equal */
        return (*A).col - (*B).col;      /* sort on column index */
    } else {
        return (*A).row - (*B).row;      /* sort on row    index */
    }
} /* 1}}} */
int  comp_int_pair(const void *a, const void *b ) { /* {{{1 */
    /* sort based on the value in the first column: */
    return *(int *) a - *(int *) b;
    /* sort based on the value in the second column:
    return *(((int *)a) + 1) - *(((int *)b) + 1);
    */
} /* 1}}} */
void get_sort_sequence( int  N ,  /* {{{1 */
                        int *In,
                        int *Out) {
    /*
     * get_sort_sequence(N  : 3
     *                   In : [3, 4, 1]
     *                   Out: [2, 0, 1])
     *
     * because In[ Out[0]= 2 ] = 1
     *         In[ Out[1]= 0 ] = 3
     *         In[ Out[2]= 1 ] = 4
     *
    */
    int i;

    /* fill the working array */
    for (i = 0; i < N; i++) {
        Out[2*i  ] = In[i];
        Out[2*i+1] =    i ;
    }

    /* sort the working array */
    qsort(Out, N, 2*sizeof(int), &comp_int_pair);

    /* overwrite terms 0..N-1 with the sort order */
    for (i = 0; i < N; i++) {
        Out[i] = Out[2*i+1];
    }
} /* 1}}} */
int  get_sort_sequence2(int  N ,  /* {{{1 */
                        int *In,
                        int *Out2) {
    /*
     * get_sort_sequence2(N  : 3
     *                   in  [3, 4, 1]
     *                   out [1, 2, 0])
     *
     * because In[0]=3 = belongs in the Out[0]=1-th sorted position
     *         In[1]=4 =                Out[1]=2-th                
     *         In[2]=1 =                Out[2]=0-th                
     *
     * Returns 0 if unable to malloc memory here, otherwise 1 is success.
    */

     int *Out;

     if ((Out = (int *) malloc(2*N*sizeof(int))) == NULL) {
         return 0;
     }

     get_sort_sequence(N, In , Out );
     get_sort_sequence(N, Out, Out2);

     free(Out);
     return 1;

} /* 1}}} */
void sp_copy_index_data( /* {{{1 */
                        int ncols,            /* in  */
                        int nstr,             /* in  */
                        SparseMatrix m_in,    /* in  */
                        SparseMatrix m_out)   /* out */
{
    int i;
    for (i = 0; i < ncols + 1; i++) {
        m_out.S_start[i] = m_in.S_start[i];
        m_out.N_start[i] = m_in.N_start[i];
    }
    for (i = 0; i < nstr; i++) {
        m_out.S[i] = m_in.S[i];
    }
} /* 1}}} */
void sp_copy_column( /* {{{1 */
                    int col_index_in,     /* in  */
                    int col_index_out,    /* in  */
                    SparseMatrix m_in,    /* in  */
                    SparseMatrix m_out)   /* out */
{
    int i, s_in_ptr, s_out_ptr, n_in_ptr, n_out_ptr, 
        n_str, n_nnz = 0;

    n_str = m_in.S_start[col_index_in + 1] - m_in.S_start[col_index_in];
    if (n_str) {
        n_nnz     = m_in.N_start[col_index_in + 1] - m_in.N_start[col_index_in];
        s_in_ptr  = m_in.S_start[col_index_in];
        s_out_ptr = m_out.S_start[col_index_out];
        n_in_ptr  = m_in.N_start[col_index_in];
        n_out_ptr = m_out.N_start[col_index_out];
        for (i = 0; i < n_str; i++) {
            m_out.S[s_out_ptr + i].len       = m_in.S[s_in_ptr + i].len;
            m_out.S[s_out_ptr + i].start_row = m_in.S[s_in_ptr + i].start_row;
            m_out.S[s_out_ptr + i].N_idx     = n_out_ptr + i;
        }
        /* n_nnz counts over doubles so works for both real and complex */
        for (i = 0; i < n_nnz; i++)
            m_out.N[n_out_ptr + i] = m_in.N[n_in_ptr + i];
    }
    m_out.S_start[col_index_out + 1] = m_out.S_start[col_index_out] + n_str;
    m_out.N_start[col_index_out + 1] = m_out.N_start[col_index_out] + n_nnz;
} /* 1}}} */
void strings_in_list( /* {{{1 */
                     int  n_terms,        /* in  length of list[]            */
                     int *list,           /* in                              */
                     int *n_str,          /* out number of strings in list[] */
                     int *start_ind,      /* out index of 1st string terms   */
                     int *str_len,        /* out length of each string       */
                     int *max_length)     /* out longest string in list[]    */
     /*  list[] contains non-negative integers in ascending order, eg:
      *     n_terms = 7
      *     list[0..6]  =  2, 5, 6, 7, 10, 12, 13
      *  This routine counts the number of blocks of consecutive terms
      *  and also returns the length of the longest string.  In the example
      *  the strings are   (2)  (5,6,7)  (10)  (12,13)
      *  so:
      *      n_str                   = 4 
      *      start_ind[0..(n_str-1)] =  0, 1, 4, 5
      *      str_len[  0..(n_str-1)] =  1, 3, 1, 2
      *      max_length              = 3
      */
{
int DEBUG = 0;
    int i, length_current;

    *n_str         = 0;
    *max_length    = 0;
    length_current = 0;
    if (n_terms > 0) {
        length_current    = 1;
        start_ind[*n_str] = 0;
        str_len[  *n_str] = length_current;
        ++(*n_str);
    } else {
        return;
    }
    for (i = 1; i < n_terms; i++) {
        if (list[i] != (list[i-1] + 1)) {
            length_current    = 1;
            start_ind[*n_str] = i;
            str_len[  *n_str] = length_current;
if (DEBUG) {
printf("list[%d]=%d  list[%d]=%d  nstr=%d   len=%d\n", 
i, list[i], i-1, list[i-1], *n_str, str_len[*n_str]);
}
            ++(*n_str);
        } else {
            ++length_current;
            str_len[*n_str-1] = length_current;
if (DEBUG) {
printf("           nstr=%d   len=%d\n", *n_str-1, str_len[*n_str-1]);
}
        }
        *max_length = MAX(*max_length, length_current);
    }
} /* 1}}} */
void spmult_ss_str_count(SparseMatrix A,      /* {{{1 */
                         SparseMatrix B,
                         int         *nStr,
                         int         *nNnz)
/* 
 *   Multiply two sparse matrices :    [C] = [A][B]
 */ 
{
int DEBUG = 0;
    int          c, r, s, j, end_row,
                 nRows_B = 0, nCols_B = 0, n_ptr;

if (DEBUG) printf("top of spmult_ss_str_count\n");


if (DEBUG)
printf("A cols=%d  B rows=%d\n", A.H[COLS], nRows_B);
    if (A.H[COLS] != B.H[ROWS]) {
        stkerr(" spmult: ",MATSNOTC); 
        return;
    }

if (DEBUG) {
printf("ss_str_count A %2d x %2d\n", A.H[ROWS], A.H[COLS]);
printf("ss_str_count B %2d x %2d\n", nRows_B,   nCols_B);
}
    n_ptr = 0;
    for (c = 0; c < A.H[COLS]; c++) {
        for (s = A.S_start[c]; s < A.S_start[c+1]; s++) {
            end_row = A.S[s].start_row + A.S[s].len - 1;
            n_ptr   = A.S[s].N_idx;
if (DEBUG) printf("column %d string %d  rows %d:%d (len=%d)  nptr=%d\n",
c, s, A.S[s].start_row, end_row, A.S[s].len, n_ptr);
            for (r = A.S[s].start_row; r <= end_row; r++) {
                for (j = 0; j < nCols_B; j++) {
                }
            }
        }
    }

    return;
}
/* 1}}} */
int  int_compare(const void *a, const void *b) { /* {{{1 */
        return *(int *) a - * (int *) b;
} /* 1}}} */
void gprint_list(char *title, int N, int *list) { /* {{{1 */
    int i;
    gprintf("%8s: ", title);
    for (i = 0; i < N; i++) {
        gprintf(" %3d", list[i]);
    }
    gprintf("\n");
} /* 1}}} */
int  f_index_align(int  X_size,  /* in  {{{1*/
                   int *X     ,  /* in  */
                   int  Y_size,  /* in  */
                   int *Y     ,  /* in  */
                   int *x_Map ,  /* out array, X_size terms */
                   int *y_Map ,  /* out array, Y_size terms */
                   int *err_cod) /* out < 0 means malloc problem; 1 = OK */
/* {{{2
 * Given two arrays of integers, {X} and {Y}, returns {x_Map} and {y_Map} which are 0-based index arrays that show where terms of {Y} fit in {X} and vice versa.  The terms of {y_Map} satisfy Y[i] = X[ y_Map[i] ] for all y_Map[i] which are >= 0, and the terms of {x_Map} satisfy X[i] = Y[ x_Map[i] ].  If x_Map[i] is negative, then Y[i] does not appear in {X} and similarly if y_Map[i] < 0, then X[i] does not appear in {Y}.  {X} and {Y} may contain duplicate terms.
Example:
  X    = { 84 39 78 79 76 14 55 }
  Y    = { 78 14 39 }
produces
 x_Map = { -1  2  0 -1 -1  1 -1 }
 y_Map = {  2  5  1 }
 *
 *  err_cod return value:
 *     0: no error
 *    -1: malloc X failed
 *    -2: malloc Y failed
 * 2}}} */ 
{
int DEBUG = 0;
    int  xi, yi, still_looking, *X_order, *Y_order;

if (DEBUG) gprintf("top of f_index_align\n");

    *err_cod = 0;  /* no error */

    if ((X_order = (int *) malloc(2*X_size*sizeof(int))) == NULL) {
        *err_cod = -1;
if (DEBUG) gprintf("f_index_align malloc X failed\n");
        return 0;
    }

    if ((Y_order = (int *) malloc(2*Y_size*sizeof(int))) == NULL) {
        *err_cod = -2;
if (DEBUG) gprintf("f_index_align malloc Y failed\n");
        return 0;
    }

    get_sort_sequence(X_size, X, X_order);
    get_sort_sequence(Y_size, Y, Y_order);
                                                                                
if (DEBUG) {
for (xi = 0; xi < X_size; xi++) 
gprintf("f_index_align X %2d. %5d %5d\n", xi, X[xi], X_order[xi]);
gprintf("f_index_align \n");
for (yi = 0; yi < Y_size; yi++)
gprintf("f_index_align Y %2d. %5d %5d\n", yi, Y[yi], Y_order[yi]);
gprintf("in sort order: \n");
for (xi = 0; xi < X_size; xi++) 
gprintf("f_index_align X %2d. %5d %5d\n", xi, X[X_order[xi]], X_order[xi]);
for (yi = 0; yi < Y_size; yi++)
gprintf("f_index_align Y %2d. %5d %5d\n", yi, Y[Y_order[yi]], Y_order[yi]);
}
    /* initialize the maps with all mismatches */
    for (xi = 0; xi < X_size; xi++) x_Map[xi] = -1;
    for (yi = 0; yi < Y_size; yi++) y_Map[yi] = -1;
                                                                                
    xi            = 0;
    yi            = 0;
    still_looking = 1;
    while (still_looking) {
if (DEBUG) gprintf("f_index_align:  yi=%d  xi=%d ", yi, xi);
        if ((xi >= X_size) || (yi >= Y_size)) {
if (DEBUG) gprintf("f_index_align end of loop");
            still_looking = 0;
        } else if (Y[Y_order[yi]] == X[X_order[xi]]) {
if (DEBUG) gprintf("f_index_align Y[Y_order[yi=%d]=%d]=%d == X[X_order[xi=%d]=%d]=%d",
yi, Y_order[yi], Y[Y_order[yi]],
xi, X_order[xi], X[X_order[xi]]);
if (DEBUG) gprintf("\n");
            x_Map[ X_order[xi] ] = Y_order[yi];
            y_Map[ Y_order[yi] ] = X_order[xi];
if (DEBUG) gprintf("       x_Map[X_order[xi=%d]=%d] = Y_order[yi=%d] = %d",
xi, X_order[xi], yi, Y_order[yi]);
if (DEBUG) gprintf("\n");
if (DEBUG) gprintf("       y_Map[Y_order[yi=%d]=%d] = X_order[xi=%d] = %d",
yi, Y_order[yi], xi, X_order[xi]);
            ++yi;
            /* don't increment xi in case Y has duplicates and the  *
             * Y term after this one matches the current X term     */
        } else if (Y[Y_order[yi]] <  X[X_order[xi]]) {
if (DEBUG) gprintf("f_index_align Y[Y_order[yi=%d]=%d]=%d <  X[X_order[xi=%d]=%d]=%d",
yi, Y_order[yi], Y[Y_order[yi]], xi, X_order[xi], X[X_order[xi]]);
            ++yi;
        } else {
if (DEBUG) gprintf("f_index_align Y[Y_order[yi=%d]=%d]=%d >  X[X_order[xi=%d]=%d]=%d",
yi, Y_order[yi], Y[Y_order[yi]], xi, X_order[xi], X[X_order[xi]]);
            ++xi;
        }
if (DEBUG) gprintf("\n");
    }

    free(X_order);
    free(Y_order);

    return 1;
} /* 1}}} */
int  f_index_merge(int  X_size,  /* in  {{{1*/
                   int *X     ,  /* in  */
                   int  Y_size,  /* in  */
                   int *Y     ,  /* in  */
                   int *y_Map ,  /* out array, Y_size terms */
                   int *Z_size,  /* out # terms in Z ( <= X_size+Y_size) */
                   int *Z     ,  /* out array, (max X_size+Y_size terms) */
                   int *err_cod) /* out < 0 means malloc problem; 1 = OK */
/* {{{2
 * Given two arrays of integers, {X} and {Y}, returns {Z}, an array containing the union of {X} and {Y}.  The first X_size terms of {Z} are copies of {X} and the remaining terms are those {Y} terms which are not in {X}.  Also returns {y_Map}, a 0-based index array that show where terms of {Y}--if any--fit in {Z}.  The mapping array satisfies Y[i] = Z[ y_Map[i] ].  (Note there is no need for an {x_Map} array.)
Example:
  X    = { 84 39 78 79 76 14 55 }
  Y    = { 78 14 39 40 }
produces
  Z    = { 84 39 78 79 76 14 55 40 }
 y_Map = {  2  5  1  6 }
 *
 *  err_cod return value:
 *     0: no error
 *    -1: malloc X failed (in f_index_align)
 *    -2: malloc Y failed (in f_index_align)
 *    -3: malloc X failed (here)
 * 2}}} */ 
{
int DEBUG = 0;
    int i, iy = 0, iz = 0, *x_Map;

    if ((x_Map = (int *) malloc(X_size*sizeof(int))) == NULL) {
        *err_cod = -3;
        return 0;
    }

if (DEBUG) gprintf("top of f_index_merge\n");
    f_index_align(X_size,   /* in  */
                  X     ,   /* in  */
                  Y_size,   /* in  */
                  Y     ,   /* in  */
                  x_Map ,   /* out */
                  y_Map ,   /* out */
                  err_cod); /* out */
    if (*err_cod < 0) {
        free(x_Map);
        return 0;
    }

    for (i = 0; i < X_size; i++) Z[i] = X[i];
    *Z_size = X_size;
    iz      = X_size;

    while (iy < Y_size) {
if (DEBUG) gprintf("index_merge:  iz=%d  y_Map[%2d]=%2d\n", iz, iy, y_Map[iy]);
        if (y_Map[iy] < 0) {
            Z[iz]     = Y[iy];
            y_Map[iy] = iz;
if (DEBUG) gprintf("              y_Map[%2d]=iz=%2d\n", iy, y_Map[iy]);
            ++iz;
        }
        ++iy;
    }
    *Z_size = iz;
if (DEBUG) {
gprintf("index_merge:  Z_size=%d, Z=[", *Z_size);
for (i = 0; i < *Z_size; i++) gprintf(" %2d", Z[i]);
gprintf(" ]\n");
}
    free(x_Map);
    return 1;
} /* 1}}} */
int  f_indices_match(stkitem *A,  /* in  {{{1 */
                     stkitem *B)  /* in  */
/*
 * Returns 1 if both internal row and column index arrays of A match 
 * those of B.  Either A or B can be sparse or dense.
 */ 
{
int DEBUG = 0;
    int  i, nRows_A, nCols_A, nRows_B, nCols_B, 
        *row_idx_A, *col_idx_A, *row_idx_B, *col_idx_B;
    SparseMatrix Asp, Bsp;
if (DEBUG) gprintf("top of f_indices_match\n");

    if (is_sparse(A)) {
        Asp        = sparse_overlay(A);
        nRows_A    = Asp.H[ROWS];
        nCols_A    = Asp.H[COLS];
         row_idx_A = Asp.row_idx;
         col_idx_A = Asp.col_idx;
    } else {
        nRows_A    = A->row;
        nCols_A    = A->col;
        if (is_complex(A)) nCols_A /= 2;
         row_idx_A = MAT_ROW_IDX(A);
         col_idx_A = MAT_COL_IDX(A);
    }

    if (is_sparse(B)) {
        Bsp        = sparse_overlay(B);
        nRows_B    = Bsp.H[ROWS];
        nCols_B    = Bsp.H[COLS];
         row_idx_B = Bsp.row_idx;
         col_idx_B = Bsp.col_idx;
    } else {
        nRows_B    = B->row;
        nCols_B    = B->col;
        if (is_complex(B)) nCols_B /= 2;
         row_idx_B = MAT_ROW_IDX(B);
         col_idx_B = MAT_COL_IDX(B);
    }

    if ((nRows_A != nRows_B) || (nCols_A != nCols_B)) {
        return 0;
    }
    for (i = 0; i < nRows_A; i++) {
        if (row_idx_A[i] != row_idx_B[i]) {
            return 0;
        }
    }
    for (i = 0; i < nCols_A; i++) {
        if (col_idx_A[i] != col_idx_B[i]) {
            return 0;
        }
    }

    return 1;
} /* 1}}} */
int  f_sp2ijv(SparseMatrix m      , /* in {{{1 */
              int          indexed, /* in  0=no  1=yes, internally indexed */
              IJVMatrix   *IJV    ) /* out */
/* returns 1 if successful,
 *         0 if unable to malloc space for IJV
 */
{
int DEBUG = 0;

    int     r, c, i, s, end_row, n_ptr;

    if (!malloc_IJVMatrix(IJV, m.H[ROWS], m.H[COLS], m.H[n_NONZ], indexed)) {
        return 0;
    }
    (*IJV).cmplx  = m.H[COMPLX];
    if (indexed) {
        memcpy((*IJV).row_idx, m.row_idx, m.H[ROWS]*sizeof(int));
        memcpy((*IJV).col_idx, m.col_idx, m.H[COLS]*sizeof(int));
    }

if (DEBUG) gprintf("f_sp2ijv In: nrows=%d ncols=%d\n", 
m.H[ROWS], m.H[COLS]);

    n_ptr = 0;
    i     = 0;
    for (c = 0; c < m.H[COLS]; c++) {
        for (s = m.S_start[c]; s < m.S_start[c+1]; s++) {
            end_row = m.S[s].start_row + m.S[s].len - 1;
if (DEBUG) gprintf("sp2ijv c=%2d s=%2d start row=%2d end_row=%2d\n",
c, s, m.S[s].start_row, end_row);
            for (r = m.S[s].start_row; r <= end_row; r++) {
                (*IJV).d[i].row = r;
                (*IJV).d[i].col = c;
                (*IJV).d[i].Re  = m.N[n_ptr++];
if (DEBUG) gprintf("sp2ijv IJV[%2d].row=%2d IJV[].col=%2d IJV[].Re=% 12.4le",
i, (*IJV).d[i].row, (*IJV).d[i].col, (*IJV).d[i].Re);
                if (m.H[COMPLX]) (*IJV).d[i].Im = m.N[n_ptr++];
                else             (*IJV).d[i].Im = 0.0;
if (DEBUG) gprintf(" IJV[].Im=% 12.4le", (*IJV).d[i].Im);
if (DEBUG) gprintf("\n");
                ++i;
            }
        }
    }

    return 1;
} /* 1}}} */
int  f_ijv2sp(char      *name       , /* in {{{1  */
              IJVMatrix  IJV        , /* in/out (memory freed on return)  */
              int        indexed    , /* in  1 make room for int. indices */
              int        transpose)   /* in  1 = treat rows as cols       */
/*
 *  Put on the stack a sparse matrix represented by the contents of IJV.
 *  Returns 1 if successful, 0 otherwise (because of insufficient memory).
 *  Frees memory associated with IJV so IJV is useless after a call to
 *  this function.
 */
{
int DEBUG = 0;
    int     c, i, b_nstr, nrows, ncols,
            prev_col, prev_row, s_ptr, n_ptr, curr_str_len,
            n_str_this_col, curr_col, curr_row, n_term_this_col, NPT;
    SparseMatrix m;

if (DEBUG)
gprintf("top of f_ijv2sp\n");
    if (transpose) {
        ncols       = IJV.nRows;
        nrows       = IJV.nCols;
        prev_col    = -1;
        prev_row    = IJV.d[0].col;
    } else {
        nrows       = IJV.nRows;
        ncols       = IJV.nCols;
        prev_col    = -1;
        prev_row    = IJV.d[0].row;
    }
    b_nstr          = 0;
    curr_str_len    = 0;
    n_str_this_col  = 0;
    NPT             = NUM_PER_TERM(IJV.cmplx);
if (DEBUG) gprintf("f_ijv2sp NPT=%d   nTerms=%d\n", NPT, IJV.nTerms);
    for (i = 0; i < IJV.nTerms; i++) {
if (DEBUG) gprintf("f_ijv2sp IJV.d[%2d] -> row=%2d  col=%2d\n",
i, IJV.d[i].row, IJV.d[i].col);
        if (transpose) {
            curr_col = IJV.d[i].row;
            curr_row = IJV.d[i].col;
        } else {
            curr_row = IJV.d[i].row;
            curr_col = IJV.d[i].col;
        }
        if (curr_col > (prev_col+1)) {
            /* gap in column sequence; have jumped null columns */
            prev_col = curr_col - 1;
        }
if (DEBUG)
gprintf("\ni=%d prev_col=%3d curr_col=%3d curr_row=%3d\n", 
i, prev_col, curr_col, curr_row);
        if (curr_col == prev_col) {  /* in same column as before */
if (DEBUG)
gprintf("B. same column\n");
            if (curr_row == (prev_row + 1)) {  /* in same string as before */
                ++curr_str_len;
if (DEBUG)
gprintf("C. same string new length = %d\n", curr_str_len);
            } else {                /* gap in row sequence; start new str */
                ++b_nstr;
                ++n_str_this_col;
                curr_str_len = 1;
if (DEBUG)
gprintf("D. new string count  = %2d  total = %d\n", n_str_this_col, b_nstr);
            }
        } else { /* must be in a new non-null column */
            ++b_nstr;
            curr_str_len   = 1;
            n_str_this_col = 1;
if (DEBUG)
gprintf("E. in a new column\n");
        }
        prev_col = curr_col;
        prev_row = curr_row;
    }
if (DEBUG) {
gprintf("end of Pass 1 in f_ijv2sp:  %d x %d  cmplx=%d\n", 
nrows, ncols, IJV.cmplx);
gprintf("   indexed        =%d\n", IJV.indexed);
gprintf("   b_nstr         =%d\n", b_nstr);
gprintf("   nTerms         =%d\n", IJV.nTerms);
}
    if (!sparse_stk(nrows      , /* in  */ 
                    ncols      , /* in  */
                    b_nstr     , /* in  number of strings       */
                    IJV.nTerms , /* in  number of nonzero terms */
                    IJV.cmplx  , /* in  0=real  1=complex       */
                    IJV.indexed, /* in  0=no    1=yes           */
                    name       , /* in  */
                   &m)) {        /* out */
        return 0;
    }
    if (IJV.indexed) {
        memcpy(m.row_idx, IJV.row_idx, nrows*sizeof(int));
        memcpy(m.col_idx, IJV.col_idx, ncols*sizeof(int));
    }

    /* Traverse IJV a second time, this time populating the b_ pointer
     * and numeric data arrays.
     */
    prev_col        = -1;
    prev_row        =  0;
    n_str_this_col  =  0;
    n_term_this_col =  0;
    s_ptr           = -1;
    n_ptr           =  0;
    curr_col        = ncols; /* to skip past trailing loop when IJV.nTerms=0 */

    /* populate iS & iN with null matrix values in case IJV.nTerms == 0 */
    for (c = 0; c < ncols; c++) {
        m.S_start[c] = 0;
        m.N_start[c] = 0;
    }
    m.S[0].start_row = 0;
    m.S[0].len       = 0;
    m.S[0].N_idx     = 0;

    for (i = 0; i < IJV.nTerms; i++) {
        if (transpose) {
            curr_col = IJV.d[i].row;
            curr_row = IJV.d[i].col;
        } else {
            curr_row = IJV.d[i].row;
            curr_col = IJV.d[i].col;
        }
if (DEBUG)
gprintf("\n1. i=%d prev_col=%3d curr_col=%3d   prev_row=%3d curr_row=%3d\n", 
i, prev_col, curr_col, prev_row, curr_row);
        if (curr_col > (prev_col+1)) {
            /* gap in column sequence; have jumped null columns */
            if (s_ptr == -1) {
                /* very first column was null, cannot assign s_ptr */
                for (c = prev_col+1; c < curr_col; c++) {
                    m.S_start[c] = 0;
                    m.N_start[c] = 0;
                }
            } else {
                for (c = prev_col+1; c < curr_col; c++) {
                    m.S_start[c] = s_ptr +   1;
                    m.N_start[c] = n_ptr + NPT;
if (DEBUG)
gprintf("2. col seq. gap; m.S_start[c=%d]=%d m.N_start[c=%d]=%d\n", 
c, m.S_start[c], c, m.N_start[c]);
                }
            }
            prev_col = curr_col - 1;
        }
        if (curr_col == (prev_col+1)) {    /* starting a new, non-null column */
            ++s_ptr;
            m.S_start[curr_col]   = s_ptr;
            m.N_start[curr_col]   = i*NPT;
            m.S[s_ptr].start_row  = curr_row;
            m.S[s_ptr].len        = 1;
            m.S[s_ptr].N_idx      = i*NPT;
            n_str_this_col        = 1;
            n_term_this_col       = 1;
if (DEBUG) {
gprintf("3. start of new nonnull column; assigning string # %d\n", s_ptr);
gprintf("   m.S_start[%d]     = %d\n", curr_col,   m.S_start[curr_col]);
gprintf("   m.N_start[%d]     = %d\n", curr_col,   m.N_start[curr_col]);
gprintf("   m.S[%d].start_row = %d\n", s_ptr,    m.S[s_ptr].start_row);
gprintf("   m.S[%d].len       = %d\n", s_ptr,    m.S[s_ptr].len);
gprintf("   m.S[%d].N_idx     = %d\n", s_ptr,    m.S[s_ptr].N_idx);
}
        } else { /* can only be curr_col == prev_col; same column as before */
if (DEBUG)
gprintf("4. same column\n");
            ++n_term_this_col;
            if (curr_row == (prev_row+1)) {
                ++m.S[ s_ptr ].len;
if (DEBUG) {
gprintf("5. same string m.S[%d].len = %d\n", s_ptr, m.S[ s_ptr ].len);
}
            } else {                /* gap in row sequence; start new str */
                ++s_ptr;
                m.S[ s_ptr ].len       = 1;
                m.S[ s_ptr ].N_idx     = i*NPT;
                m.S[ s_ptr ].start_row = curr_row;
if (DEBUG) {
gprintf("6. same column, new string, assigning to sting # %d\n", s_ptr);
gprintf("   m.S[%d].start_row = %d\n", s_ptr,    m.S[s_ptr].start_row);
gprintf("   m.S[%d].len       = %d\n", s_ptr,    m.S[s_ptr].len);
gprintf("   m.S[%d].N_idx     = %d\n", s_ptr,    m.S[s_ptr].N_idx);
}
                ++n_str_this_col;
            }
        }
        if (IJV.cmplx) {
            m.N[2*i]    = IJV.d[i].Re;
            m.N[2*i+1]  = IJV.d[i].Im;
        } else {
            m.N[i]      = IJV.d[i].Re;
        }
        n_ptr                 = i*NPT;
        prev_col              = curr_col;
        prev_row              = curr_row;
        m.S_start[curr_col+1] = m.S_start[curr_col] + n_str_this_col;
        m.N_start[curr_col+1] = m.N_start[curr_col] + 
                                n_term_this_col*NPT;
if (DEBUG)
gprintf("7. m.S_start[%d]=%d m.N_start[%d]=%d\n", 
curr_col+1, m.S_start[curr_col+1], curr_col+1, m.N_start[curr_col+1]);
    }

    /* fill iS and iN for trailing null columns */
    for (c = curr_col+2; c <= ncols; c++) {
        m.S_start[c] = m.S_start[c-1];
        m.N_start[c] = m.N_start[c-1];
if (DEBUG)
gprintf("trailing nulls:  m.S_start[c=%d]=%d  m.N_start[c=%d]=%d\n", 
c, m.S_start[c], c, m.N_start[c]);
    }
if (DEBUG)
gprintf("6. num_size=%d    b_nstr=%d\n", IJV.nTerms, b_nstr);
    m.S_start[ncols] = b_nstr;
    m.N_start[ncols] = IJV.nTerms*NPT;

if (DEBUG) {
if (f_sp_error(tos)) {
f_sp_dump(name, tos);
gprintf("end of f_ijv2sp   sp_error = %d\n", f_sp_error(tos));
}
}
    free_IJVMatrix(IJV);
    return 1;
} /* 1}}} */
int  f_ijv_reorder(IJVMatrix *IJV    , /* in/out {{{1  */
                   int       *row_map, /* in  length should be == IJV.nRows */
                   int       *col_map) /* in  length should be == IJV.nCols */
/*
 *  Replace IJV.d[].row indices with row_map[], and
 *          IJV.d[].col indices with col_map[]
 *  then rearrange the position of each .d[] ijv term so that they
 *  appear in sorted order of row_map, col_map.
 */
{
int DEBUG = 0;
    int i, *canonical_row_map, *canonical_col_map;
                                                                                
    if ((canonical_row_map = (int *) malloc(2*(*IJV).nRows*sizeof(int)))
         == NULL) {
        return -3;
    }
    if ((canonical_col_map = (int *) malloc(2*(*IJV).nCols*sizeof(int)))
         == NULL) {
        free(canonical_row_map);
        return -4;
    }

    if (!get_sort_sequence2((*IJV).nRows, row_map, canonical_row_map)) {
        return -5;
    }
    if (!get_sort_sequence2((*IJV).nCols, col_map, canonical_col_map)) {
        return -6;
    }
                                                                                
if (DEBUG) {
gprint_list("f_ijv_reorder           row_map = ",
(*IJV).nRows,           row_map);
gprint_list("f_ijv_reorder canonical_row_map = ",
(*IJV).nRows, canonical_row_map);
gprint_list("f_ijv_reorder           col_map = ",
(*IJV).nCols,           col_map);
gprint_list("f_ijv_reorder canonical_col_map = ",
(*IJV).nCols, canonical_col_map);
}
                                                                                
    for (i = 0; i < (*IJV).nTerms; i++) {
if (DEBUG)
gprintf("f_ijv_reorder 1/ IJV[%2d].row=%2d -> %2d;  .col=%2d -> %2d\n",
i, (*IJV).d[i].row, canonical_row_map[ (*IJV).d[i].row ],
   (*IJV).d[i].col, canonical_col_map[ (*IJV).d[i].col ]);
        (*IJV).d[i].row = canonical_row_map[ (*IJV).d[i].row ];
        (*IJV).d[i].col = canonical_col_map[ (*IJV).d[i].col ];
                                                                                
    }
if (DEBUG) {
gprintf("f_ijv_reorder before sort:\n");
f_ijv_dump(*IJV);
}
    qsort((*IJV).d, (*IJV).nTerms, sizeof(ijv), &comp_IJV_cr);
if (DEBUG) {
gprintf("f_ijv_reorder after  sort:\n");
f_ijv_dump(*IJV);
}

    get_sort_sequence((*IJV).nRows, row_map, canonical_row_map);
    get_sort_sequence((*IJV).nCols, col_map, canonical_col_map);
                                                                                
    for (i = 0; i < (*IJV).nTerms; i++) {
if (DEBUG)
gprintf("f_ijv_reorder 2/ IJV[%2d].row=%2d -> %2d;  .col=%2d -> %2d\n",
i, (*IJV).d[i].row, row_map[ (*IJV).d[i].row ],
   (*IJV).d[i].col, col_map[ (*IJV).d[i].col ]);
        (*IJV).d[i].row = row_map[ canonical_row_map[ (*IJV).d[i].row ] ];
        (*IJV).d[i].col = col_map[ canonical_col_map[ (*IJV).d[i].col ] ];
                                                                                
    }
    free(canonical_row_map);
    free(canonical_col_map);
    return 1;

} /* 1}}} */
int  f_ijv_add(IJVMatrix  A ,  /* in      [A] {{{1                      */
               IJVMatrix *B ,  /* in/out  [B] (might be reordered)      */
               IJVMatrix *C )  /* out [C] (malloc'ed here) = [A] + [B]  */
/*
 *  Add two sparse matrices represented in IJV form.
 */
{
int DEBUG = 0;
    int i, iA, iB, iC, nC_Rows, nC_Cols, err_cod,
        nC_terms, C_indexed,
       *C_row_map = 0, *C_col_map = 0, *merged_row_idx = 0, *merged_col_idx = 0;

if (DEBUG) gprintf("top of f_ijv_add\n");

    if (A.indexed && (*B).indexed) {
        /* reorder [B] so that its terms align with [A] {{{2 */
        C_indexed = 1;

        /* set worst-case index array sizes for [C] */
        nC_Rows = A.nRows + (*B).nRows;
        nC_Cols = A.nCols + (*B).nCols;

        if ((C_row_map      = (int *) malloc( nC_Rows*sizeof(int) )) == NULL) {
            return -5;
        }
        if ((merged_row_idx = (int *) malloc( nC_Rows*sizeof(int) )) == NULL) {
            return -6;
        }

        if (f_index_merge(A.nRows       ,  /* in  */
                          A.row_idx     ,  /* in  */
                         (*B).nRows     ,  /* in  */
                         (*B).row_idx   ,  /* in  */
                          C_row_map     ,  /* out */
                         &nC_Rows       ,  /* out the actual size */
                          merged_row_idx,  /* out */
                         &err_cod) < 0)    /* out */
        {
            free(C_row_map);
            free(merged_row_idx);
            return err_cod;
        }
if (DEBUG) gprint_list("f_ijv_add merged_row_idx=", nC_Rows, merged_row_idx);
if (DEBUG) gprint_list("f_ijv_add B_row_idx     =", (*B).nRows, (*B).row_idx  );
if (DEBUG) gprint_list("f_ijv_add C_row_map     =", (*B).nRows, C_row_map  );

        if ((C_col_map      = (int *) malloc( nC_Cols*sizeof(int) )) == NULL) {
            free(C_row_map);
            free(merged_row_idx);
            return -7;
        }
        if ((merged_col_idx = (int *) malloc( nC_Cols*sizeof(int) )) == NULL) {
            free(C_col_map);
            free(C_row_map);
            free(merged_row_idx);
            return -8;
        }

        if (f_index_merge(A.nCols       ,  /* in  */
                          A.col_idx     ,  /* in  */
                         (*B).nCols     ,  /* in  */
                         (*B).col_idx   ,  /* in  */
                          C_col_map     ,  /* out */
                         &nC_Cols       ,  /* out the actual size */
                          merged_col_idx,  /* out */
                         &err_cod) < 0)    /* out */
        {
            free(C_row_map);
            free(C_col_map);
            free(merged_row_idx);
            free(merged_col_idx);
            return err_cod;
        }
if (DEBUG) gprint_list("f_ijv_add merged_col_idx=", nC_Cols, merged_col_idx);
if (DEBUG) gprint_list("f_ijv_add B_col_idx     =", (*B).nCols, (*B).col_idx  );
if (DEBUG) gprint_list("f_ijv_add C_col_map     =", (*B).nCols, C_col_map  );

        if (f_ijv_reorder(B, C_row_map, C_col_map) < 0) {
            free(merged_row_idx);
            free(merged_col_idx);
            return -9;
        }
        /* 2}}} */
    } else {
        C_indexed = 0;
        /* without internal indices [A] and [B] must be the same size */
        if (A.nRows != (*B).nRows ||
            A.nCols != (*B).nCols) {
            return -11;
        }
        nC_Rows = A.nRows;
        nC_Cols = A.nCols;
    }
if (DEBUG) gprintf("f_ijv_add  nC_Rows=%d,  nC_Cols=%d\n", nC_Rows, nC_Cols);

    /* Pass 1:  figure out how many terms [C] will have {{{2 */
    iA  = 0;
    iB  = 0;
    nC_terms = 0;
    while (iA < A.nTerms && iB < (*B).nTerms) {
        if        (A.d[iA].row == (*B).d[iB].row &&
                   A.d[iA].col == (*B).d[iB].col) {
            /* terms align, produce one term in [C] */
            ++iA;
            ++iB;
if (DEBUG) gprintf("f_ijv_add  P1  Ar=Br,Ac=Bc iA=%d  iB=%d\n", iA, iB);
        } else if (A.d[iA].col == (*B).d[iB].col) {
            /* am in the same column */
            if (A.d[iA].row < (*B).d[iB].row) {
                ++iA;
if (DEBUG) gprintf("f_ijv_add  P1  Ar<Br,Ac=Bc iA=%d  iB=%d\n", iA, iB);
            } else {
                ++iB;
if (DEBUG) gprintf("f_ijv_add  P1  Ar>Br,Ac=Bc iA=%d  iB=%d\n", iA, iB);
            }
        } else if (A.d[iA].col <  (*B).d[iB].col) {
            ++iA;
if (DEBUG) gprintf("f_ijv_add  P1        Ac<Bc iA=%d  iB=%d\n", iA, iB);
        } else {
            ++iB;
if (DEBUG) gprintf("f_ijv_add  P1        Ac>Bc iA=%d  iB=%d\n", iA, iB);
        }
        ++nC_terms;
if (DEBUG) gprintf("f_ijv_add  P1        end loop nC_terms=%d\n", nC_terms);
    }
    /* one of the two RHS matrices may still have some terms left */
    nC_terms += (A.nTerms - iA) + ((*B).nTerms - iB);
if (DEBUG) gprintf("f_ijv_add  end P1 nC_terms=%d\n", nC_terms);
    /* 2}}} */

    /* now malloc the [C] matrix */
    if (!malloc_IJVMatrix(C, nC_Rows, nC_Cols, nC_terms, C_indexed)) {
        return -10;
    }
    (*C).cmplx = (A.cmplx && (*B).cmplx);
    if (C_indexed) {
        memcpy((*C).row_idx, merged_row_idx, nC_Rows*sizeof(int));
        memcpy((*C).col_idx, merged_col_idx, nC_Cols*sizeof(int));
        free(merged_row_idx);
        free(merged_col_idx);
    }

    /* Pass 2:  sum the terms {{{2 */
    iA = 0;
    iB = 0;
    iC = 0;
    while (iA < A.nTerms && iB < (*B).nTerms) {
if (DEBUG) gprintf("f_ijv_add P2 top iA=%d,iB=%d   A(%2d,%2d)   B(%2d,%2d)\n",
iA, iB, A.d[iA].row, A.d[iA].col, (*B).d[iB].row, (*B).d[iB].col);
        if       ((A.d[iA].row == (*B).d[iB].row) &&
                  (A.d[iA].col == (*B).d[iB].col)) {
            /* terms align, produce one term in [C] */
            (*C).d[iC].Re  = A.d[iA].Re + (*B).d[iB].Re;
            (*C).d[iC].Im  = A.d[iA].Im + (*B).d[iB].Im;
            (*C).d[iC].row = A.d[iA].row;
            (*C).d[iC].col = A.d[iA].col;
if (DEBUG) gprintf("f_ijv_add  P2  Ar=Br,Ac=Bc iA=%d  iB=%d\n", iA, iB);
            ++iA;
            ++iB;
        } else if (A.d[iA].col == (*B).d[iB].col) {
            /* am in the same column */
            if (A.d[iA].row < (*B).d[iB].row) {
                (*C).d[iC].Re  = A.d[iA].Re    ;
                (*C).d[iC].Im  = A.d[iA].Im    ;
                (*C).d[iC].row = A.d[iA].row   ;
                (*C).d[iC].col = A.d[iA].col   ;
if (DEBUG) gprintf("f_ijv_add  P2  Ar<Br,Ac=Bc iA=%d  iB=%d\n", iA, iB);
                ++iA;
            } else {
                (*C).d[iC].Re  = (*B).d[iB].Re ;
                (*C).d[iC].Im  = (*B).d[iB].Im ;
                (*C).d[iC].row = (*B).d[iB].row;
                (*C).d[iC].col = (*B).d[iB].col;
if (DEBUG) gprintf("f_ijv_add  P2  Ar>Br,Ac=Bc iA=%d  iB=%d C_row_map[%d]=%d\n",
iA, iB, (*B).d[iB].row, C_row_map[ (*B).d[iB].row ]);
                ++iB;
            }
        } else if (A.d[iA].col <  (*B).d[iB].col) {
            (*C).d[iC].Re  = A.d[iA].Re    ;
            (*C).d[iC].Im  = A.d[iA].Im    ;
            (*C).d[iC].row = A.d[iA].row   ;
            (*C).d[iC].col = A.d[iA].col   ;
if (DEBUG) gprintf("f_ijv_add  P2        Ac<Bc iA=%d  iB=%d\n", iA, iB);
            ++iA;
        } else {
            (*C).d[iC].Re  = (*B).d[iB].Re ;
            (*C).d[iC].Im  = (*B).d[iB].Im ;
            (*C).d[iC].row = (*B).d[iB].row;
            (*C).d[iC].col = (*B).d[iB].col;
if (DEBUG) gprintf("f_ijv_add  P2        Ac>Bc iA=%d  iB=%d C_col_map[%d]=%d\n",
iA, iB, (*B).d[iB].col, C_col_map[ (*B).d[iB].col ]);
            ++iB;
        }
        ++iC;
    }
    /* one of the two RHS matrices may still have some terms left */
    for (i = iA; i < A.nTerms; i++) {
        (*C).d[iC].Re  = A.d[i].Re    ;
        (*C).d[iC].Im  = A.d[i].Im    ;
        (*C).d[iC].row = A.d[i].row   ;
        (*C).d[iC].col = A.d[i].col   ;
        ++iC;
    }
    for (i = iB; i < (*B).nTerms; i++) {
        (*C).d[iC].Re  = (*B).d[i].Re ;
        (*C).d[iC].Im  = (*B).d[i].Im ;
        (*C).d[iC].row = (*B).d[i].row;
        (*C).d[iC].col = (*B).d[i].col;
if (DEBUG) gprintf("f_ijv_add  P2  B leftovers:  iB=%d,iC=%d   (*B).d[].row=%d  (*B).d[].col=%d\n",
iB, iC, (*B).d[iB].row, (*B).d[iB].col);
        ++iC;
    }
    /*
    if (C_indexed) {
        for (i = iC; i < nC_Terms; i++) {
            (*C).d[iC].Re  = (*B).d[i].Re ;
            (*C).d[iC].Im  = (*B).d[i].Im ;
            (*C).d[iC].row = (*B).d[i].row;
            (*C).d[iC].col = (*B).d[i].col;
            ++iB;
        }
    }
    */
    /* 2}}} */
    if (C_indexed) {
        free(C_row_map);
        free(C_col_map);
    }

    return 1;
} /* 1}}} */
void f_ijv_dump(IJVMatrix  IJV) /* {{{1  */
{
    int i;
    gprintf("ijv_dump: %d x %d  nTerms=%d  cmplx=%d  indexed=%d\n",
            IJV.nRows, IJV.nCols, IJV.nTerms, IJV.cmplx, IJV.indexed);
    for (i = 0; i < IJV.nTerms; i++) {
        gprintf("ijv_dump:  .d[%3d]:   (%3d,%3d) % 12.6e,% 12.6e",
                i, IJV.d[i].row,
                   IJV.d[i].col,
                   IJV.d[i].Re ,
                   IJV.d[i].Im );
        if (IJV.indexed) {
            gprintf(" [%3d,%3d]",
                IJV.row_idx[ IJV.d[i].row ],
                IJV.col_idx[ IJV.d[i].col ]);
        }
        gprintf("\n");
    }
} /* 1}}} */
int  malloc_IJVMatrix(IJVMatrix *IJV   ,   /* out {{{1 */
                      int        nRows ,   /* in */
                      int        nCols ,   /* in */
                      int        n_Nonz,   /* in */
                      int        indexed)  /* in */
{
    (*IJV).nRows   = nRows;
    (*IJV).nCols   = nCols;
    (*IJV).cmplx   = 0;
    (*IJV).nTerms  = n_Nonz;
    (*IJV).indexed = indexed;
    if (((*IJV).d = (ijv *) malloc(n_Nonz * sizeof(ijv))) == NULL) 
    {
        return 0;
    }
    if (indexed) {
        if (((*IJV).row_idx = (int *) malloc(nRows * sizeof(int))) == NULL)
        {
            free( (*IJV).d );
            return 0;
        }
        if (((*IJV).col_idx = (int *) malloc(nCols * sizeof(int))) == NULL)
        {
            free( (*IJV).row_idx );
            return 0;
        }
    } else {
        (*IJV).row_idx = 0;
        (*IJV).col_idx = 0;
    }
    return 1;
} /* 1}}} */
void free_IJVMatrix(IJVMatrix IJV)  /* {{{1 */
{
    free(IJV.d);
    if (IJV.indexed) {
        free( IJV.row_idx );
        free( IJV.col_idx );
    }

} /* 1}}} */
