/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#ifdef NET
/* net.c  December 2001

Copyright (c) 2001  D. R. Williamson

Words for networks.

References:

   1. Stevens, W. R., "UNIX Network Programming Volume 1, Networking
      APIs: Sockets and XTI," second edition, Prentice Hall, 1998.

   2. Stevens, W. R., "UNIX Network Programming Volume 2, Inter-
      process Communications," second edition, Prentice Hall, 1999.

   3. Unix Socket FAQ: http://www.developerweb.net/forum
      Search on Maholski for comments on how to check for readable
      socket, and to determine how many bytes are waiting to be read.

   4. D. J. Bernstein, ucspi-tcp 0.88, beta, Copyright 2000
         ucspi-tcp home page:
            http://cr.yp.to/ucspi-tcp.html
         Installation instructions:
            http://cr.yp.to/ucspi-tcp/install.html

Notes:

   Source code for all examples in References 1 and 2 can be obtained
   from: http://www.kohala.com/start/unpv12e.html

   Values for errno (for errors in read() and write(), among others)
   are found in these places:
      GNU/Linux: /usr/include/asm/errno.h
      AIX: /usr/include/errno.h

   Values for signum (for signal handlers) are found here:
      GNU/Linux: /usr/include/bits/signum.h

   /etc/rc.d/init.d/network queries /sbin/linuxconf like this:
      [user@clacker] /opt/tops # /sbin/linuxconf --hint netdev
      DEV_UP=""
      DEV_DOWN=""
      DEV_RECONF=""
      DEV_RECONF_ALIASES=""
      IPV4ROUTING="false"
      DEV_RECONF_ROUTES=""
      [user@clacker] /opt/tops #
*/
#define _XOPEN_SOURCE 500 /* snprintf (features.h) */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/ioctl.h>
#ifndef FIONREAD /* for Sun */
   #include <sys/filio.h>
#endif

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "lib.h"
#include "mat.h"
#include "math1.h"
#include "mem.h"
#if OPENSSL
   #include "ssl.h"
#endif
#include "sys.h"
#include "term.h"
#include "tex.h"

/* This network transfer rate is used to set the alarm in readn1().  It
   can be queried by word NETRATE and set by word netrate. */
static int NET_RATE=512; /* network transfer rate in bytes per second */

/* This connection time out can be queried by word CONNTO and set by
   word connto. */
static int CONNTO=10; /* time out (seconds) for making a connection */

enum drainflags {
   DRAIN_INVALID=-1,
   SOCKET_CLOSE=0,
   VOL_RUN, VOL1_RUN2,
   DRAIN_VOL, DRAIN_STR, DRAIN_MAT, DRAIN_NUM,
   GET_TOS}; /* this one must be last */

enum foreign_types {HTTP=0,TERM};

int ERRALRM; /* flag set by connect_alarm() */
int aset; /* former alarm() setting (checked for, and expect, zero) */

/*--------------------------------------------------------------------*/

int Accept(int fd, struct sockaddr *sa, socklen_t *salenptr)
/* Wrapper function for accept(), from Reference 1. */
{
   int n;

   again:
   if ( (n = accept(fd, sa, salenptr)) < 0) {
#ifdef  EPROTO
     if (errno == EPROTO || errno == ECONNABORTED) goto again;
#else
     if (errno == ECONNABORTED) goto again;
#endif
   }
   return(n);
}

int cog() /* COG (f --- ) */
/* Run COG with f true to make SERVER cog.

   When COG is true, the running server will cog bytes from a FOREIGN 
   or NEWCLI client socket by turning off SELECT for the socket and then
   reading the number of bytes specified by READ_F>0 (or fewer if that
   is all there is).

   Any remaining bytes will remain in the socket, readable but unread, 
   until SELECT for the socket is turned on.  Then the cycle is re-
   peated. */
{
   int f;
   if(!popint(&f)) return 0;
   COG=(f!=0);

   return 1;
}

int CONNECT() /* CONNECT (qIPaddr nPort ptrRun --- nSocket) */
/* This function makes a basic connection to another machine running as
   a remote server on Port at IPaddr.

   Number nSocket returned is >0 if successful, -1 otherwise.

   If incoming number ptrRun is -1, this end of the connection is type
   LOCLI (local client), meaning this connection is to another instance
   of this program running at the other end as a server (type LOSERV,
   local server), and ptrRun is used no further.

   If incoming number ptrRun is not -1, this end of the connection is
   type NATIVE, meaning the connection is to a server that is not this
   program running.

   When the connection is type NATIVE, meaning bytes will be received
   from a remote server that is not this program, number ptrRun is the
   ptr to a catalog word that will be run whenever bytes from the re-
   mote server are received in Socket.

   For type NATIVE and ptrRun=0, bytes from Socket exactly as received
   from the server will appear in a volume on the stack.

   Word NISTdelta in net.v is an example of a word that connects as
   type NATIVE with ptrRun=0, and then receives bytes arriving on the
   stack from a daytime server at the National Institute of Standards
   and Technology until the connection is closed by receiving zero num-
   ber of bytes.

   To make more than one attempt at a connection, TRY is set to 2.  The
   second try often works, so this appears to be a good strategy.  Each
   try must get a new socket file descriptor since the one for the un-
   successful attempt is no longer valid. */
{
   int CONN=NATIVE,k=0,PORT,ret=-1,sockfd=-1,TRY=2;
   double ptrRun;
   struct sockaddr_in servaddr;

   if(tos->typ!=NUM || (tos-1)->typ!=NUM || (tos-2)->typ!=STR) {
      stkerr(" CONNECT: ",STKNOT);
      return 0;
   }
   popd(&ptrRun); /* ptr to word to run when drainf() receives */
   popint(&PORT);

   /* Running _IPhost() to resolve a name into its IP decimal notation
      has been commented out because it can cause large delays if DNS 
      lookup is slow or not working.  _IPhost is also a high level 
      word, so it can be run beforehand in cases that need to resolve 
      DNS before running CONNECT.
      _IPhost(); */ /* empty VOL if cannot resolve to IP address */

   strchop();
   if(!(tos->col>0)) {
      stkerr(" CONNECT: ","host not found");
      drop(); /* qIPaddr off stack */
      pushint(-1);
      return 0;
   }
/* This loop for more than one attempt to connect must include socket(),
   since if connect_timeo() fails the descriptor sockfd is closed and
   is no longer usable (Reference 1, p. 254).  Just looping over con-
   nect_timeo() would be ineffective without another call to socket()
   to get a new sockfd to try. */

   if(WTRACE) {
      gprintf(" CONNECT: trying %s:%d",tos->tex,PORT);
      nc();
   }

   while(k<TRY) {
      sockfd=socket(AF_INET,SOCK_STREAM,0); /* create a TCP socket */

      if(!(sockfd<0)) {
         memset(&servaddr,0,sizeof(servaddr));
         servaddr.sin_family=AF_INET;

         if(PORT<1) PORT=SERV_PORT;
         servaddr.sin_port=htons(PORT);

         inet_pton(AF_INET,tos->tex,&servaddr.sin_addr);

      /* If connect_timeo() fails, sockfd is closed. */
         ret=connect_timeo(sockfd,(const struct sockaddr *)&servaddr,
            sizeof(servaddr),CONNTO);
      }
      else {
         if(!(k<TRY)) {
            stkerr(" CONNECT: ","error creating TCP socket");
            drop(); /* qIPaddr off stack */
            pushint(-1);
            return 0;
         }
      }
      if(ret==0) break; /* 0=OK */

      sockfd=-1;

   /* Wait CONNTO seconds and then try again; do not use high level
      word idle here. */
      sleep(CONNTO);
      k++;
   }
   if(ret<0) {
      gprintf(" CONNECT: could not connect to %s:%d",tos->tex,PORT);
      gprintf(" in %d attempts",k);
      nc();
/*
      Display the message above "could not connect," but do not call
      stkerr(), so word www_open (file net.v) can test for connection
      without generating stkerr's fault report.
      stkerr("","");  \\ do not call
*/
      drop(); /* qIPaddr off stack */
      pushint(-1);
      return 0;
   }
   if(ptrRun==-1) CONN=LOCLI; /* change from NATIVE to LOCLI */

   pushstr("getlogin spaced _host +");
   xmain(0); /* login+hostname */

#ifdef OPENSSL
   if(!clientmake(sockfd,PORT,NULL,CONN,ptrRun,(tos-1)->tex,tos->tex)) {
#else
   if(!clientmake(sockfd,PORT,CONN,ptrRun,(tos-1)->tex,tos->tex)) {
#endif
      drop2(); /* qIPaddr and login+hostname off stack */
      pushint(-1);
      return 0;
   }
   return(
      drop2() && /* qIPaddr and login+hostname off stack */
      pushint(sockfd) /* return socket number on stack */
   );
}

static void connect_alarm(int signo)
/* From Reference 1. */
/* This alarm is used by connect_timeo(), readn1() and writen1(). */
{
   ERRALRM=1;
   gprintf(" connect_alarm: signum %d  %s",signo,datetime());
   nc(); 
   return; /* just interrupt the function */
}

int connect_timeo(int sockfd, const struct sockaddr *saptr, \
   socklen_t salen, int nsec)
/* Adapted from connect_timeo(), Reference 1; returns 0 OK, -1 error. */
{
        Sigfunc *sigfunc;
        int             n;

        ERRALRM=0;
        sigfunc = signal(SIGALRM, connect_alarm);
        if((aset=alarm(nsec))!=0) {
           gprintf(
              " connect_timeo: using alarm that was already set to %d",\
              aset);
           nc();
        }
        if((n=connect(sockfd, (struct sockaddr *) saptr, salen))<0) {
        /* Close this socket so the system can reuse its number: */
           close(sockfd);

           if (errno == EINTR) {
              errno = ETIMEDOUT;
              gprintf(" connect_timeo: %d sec alarm timeout  %s",\
                 nsec,datetime());
              nc();
           }
           else {
              gprintf(" connect_timeo: errno %d  %s",errno,datetime());
              nc();
           }
           gprintf(" connect_timeo: closing socket %d",sockfd);
           nc();
        }
        alarm(0);                 /* turn off the alarm */
        ERRALRM=0;
        signal(SIGALRM, sigfunc); /* restore previous signal handler */

        return(n);
}

int connect1() /* connect (nSocket qIPaddr nPort ptrRun --- f) */
/* Connect Socket to IPaddr(Port), and use word at ptrRun to handle
   returned bytes.  Returned flag f is true if connection succeeded, 
   false if it failed. */
{
   int CONN=NATIVE,PORT,ret=-1,sockfd;
   double ptrRun;
   struct sockaddr_in servaddr;

   if(tos->typ!=NUM || (tos-1)->typ!=NUM || (tos-2)->typ!=STR ||
      (tos-3)->typ!=NUM) {
      stkerr(" connect: ",STKNOT);
      return 0;
   }
   popd(&ptrRun); /* ptr to word to run when drainf() receives */

   popint(&PORT);
   if(PORT<1) PORT=SERV_PORT;

   strchop();
   if(!(tos->col>0)) {
      stkerr(" connect: ","host not found");
      drop2(); /* qIPaddr and nSocket off stack */
      pushint(0); 
      return 0;
   }
   swap();
   popint(&sockfd);
   if(sockfd<0) {
      stkerr(" connect: ","invalid socket descriptor");
      drop(); /* qIPaddr off stack */
      pushint(0); 
      return 0;
   }
   if(WTRACE) {
      gprintf(" connect: trying %s:%d",tos->tex,PORT);
      nc();
   }
   memset(&servaddr,0,sizeof(servaddr));
   servaddr.sin_family=AF_INET;
   servaddr.sin_port=htons(PORT);
   inet_pton(AF_INET,tos->tex,&servaddr.sin_addr);

/* If connect_timeo() fails, incoming sockfd is closed and no longer 
   valid. */
   ret=connect_timeo(sockfd,(const struct sockaddr *)&servaddr,
      sizeof(servaddr),CONNTO);
 
   if(ret<0) {
      gprintf(" connect: could not connect to %s:%d",tos->tex,PORT);
      nc();

      drop(); /* qIPaddr off stack */
      pushint(0);
      return 0;
   }
   if(ptrRun==-1) CONN=LOCLI; /* change from NATIVE to LOCLI */

   pushstr("getlogin spaced _host +");
   xmain(0); /* login+hostname */

#ifdef OPENSSL
   if(!clientmake(sockfd,PORT,NULL,CONN,ptrRun,(tos-1)->tex,tos->tex)) {
#else
   if(!clientmake(sockfd,PORT,CONN,ptrRun,(tos-1)->tex,tos->tex)) {
#endif
      drop2(); /* qIPaddr and login+hostname off stack */
      pushint(0);
      return 0;
   }
   return(
      drop2() &&  /* qIPaddr and login+hostname off stack */
      pushint(-1) /* return true flag */
   );
}

int connto() /* connto (sec --- ) */
/* Set the timeout seconds to make a connection.  This will reduce, but
   cannot extend, the timeout period beyond the kernel's time out which
   is about 75 seconds. */
{
   int sec;

   if(!popint(&sec)) return 0;

   CONNTO=MAX(1,sec);
   return 1;
}

int connto_val() /* CONNTO ( --- sec) */
/* Current connection timeout seconds. */
{
   return(pushint(CONNTO));
}

int _client_open(int sockfd)
/* Return 1 if Socket has a client attached to it, 0 otherwise. */
{
   int cn;

   if(sockfd<0 || sockfd>FD_SETSIZE-1) return 0;

   cn=*(sockCLI+sockfd); /* cn=-1 if no client index */
   if(cn<0) return 0;

   return((*(client+cn)==sockfd)); /* socket=*(client+cn) */
}

int client_open() /* client_open (nSocket --- f) */
/* Return f true if Socket has a client attached to it. */
{
   int cn,sockfd;

   if(tos->typ!=NUM) {
      stkerr(" client_open: ",NUMNOT);
      return 0;
   }
   if(!popint(&sockfd)) return 0;

   if(sockfd<0 || sockfd>FD_SETSIZE-1) {
      stkerr(" client_open: ","socket number out of range");
      pushint(xFALSE);
      return 0;
   }
   cn=*(sockCLI+sockfd); /* cn=-1 if no client index */
   if(cn<0) return(pushint(xFALSE));
  
   return(pushint(xTRUE*(*(client+cn)!=-1))); /* socket=*(client+cn) */
}

int clientindex() /* clientindex (nSocket --- k) */
/* Returned k is -1 if nSocket is not a client's socket */
{
   int cn,n;

   if(!popint(&n)) return 0;

   if(n<0 || n>FD_SETSIZE) {
      stkerr(" clientindex: ","socket number out of range");
      pushint(-1);
      return 0;
   }
   cn=*(sockCLI+n);
   if(cn<0) return(pushint(-1));

   return(pushint(cn+XBASE));
}

int clientIPs() /* clientIPs ( --- hT) */
{
   int cn=0,n=1;
   char *name="_clientIPs";

   pushint(0);
   pushint(0);
   blockofblanks();

   for(;cn<FD_SETSIZE;cn++) {
      if(*(client+cn)>-1) {
         pushstr(*(clientIP+cn));
         n++;
      }
   }
   return(
      pushint(n) &&
      pilen() &&
      pushstr(name) &&
      naming()
   );
}

int clientLOGIN_set() /* clientLOGIN_set (hT k --- ) */
/* Redefine the text at index k in the clientLOGIN table. */
{
   int k;

   popint(&k); 
   if(k<0) {
      stkerr(" clientLOGIN_set: ","socket is not open");
      return 0;
   }
   k-=XBASE;

   if(*(clientLOGIN+k)) mallfree((void *)(clientLOGIN+k));
   *(clientLOGIN+k)=memgetn(tos->tex,tos->col); 

   return(drop());
}

int clientLOGINs() /* clientLOGINs ( --- hT) */
{
   int cn=0,n=1;
   char *name="_clientLOGINs";

   pushint(0);
   pushint(0);
   blockofblanks();

   for(;cn<FD_SETSIZE;cn++) {
      if(*(client+cn)>-1) {
         pushstr(*(clientLOGIN+cn));
         n++;
      }
   }
   return(
      pushint(n) &&
      pilen() &&
      pushstr(name) &&
      naming()
   );
}

int clients() /* clients ( --- ) */
/* Display a list of the clients, local and remote.

     type=-1 NEWCLI: initially for new clients connected to local server
     type=0 LOCLI: local client, connected to this prog remote server
     type=1 NATIVE: local client, connected to remote foreign server
     type=2 LOSERV: remote client this prog, connected to local server
     type=3 FOREIGN: remote client foreign, connected to local server

     Sample output.  Using word remoteprompt to check the clients on a 
     a busy machine (fortycoupe).  The client on socket 6 is us, con-
     nected by remoteprompt.

       [dale@fortycoupe] /home/dale > tops
                Tops 3.0.1
       Sun Jun  8 21:24:11 PDT 2008
       [tops@fortycoupe] ready > IPloop 9879 CLIENT remoteprompt
       tops@socket3 > clients
        Server local is listening on port 9879
        Clients:
         socket 2, port  9884, conn C>S, 71.106.248.129 dale plunger
         socket 3, port  9881, conn C>S, XX.XXX.8.119 dale rilefile
         socket 5, port  9879, conn C>S, YYY.YY.148.191 dale topsdog
         socket 6, port  1489, conn S<C, 127.0.0.1 LOGIN dale fortycoupe
       tops@socket3 > 

     Port 9879 shown for socket 5 is obviously on the other machine 
     (topsdog) because port 9879 on this machine is the server lis-
     tening port.  In fact, all the ports shown for connection C>S 
     are the remote machine's listening port. */
{
   int i=0;

   if(SERVER) {
#if OPENSSL
      if(SECURE)
         gprintf(" Secure server local is listening on port %d",PORT);
      else
#endif
         gprintf(" Server local is listening on port %d",PORT);
      nc();
   }
   else {
      gprintf(" Server local is off");
      nc();
   }
   while(i<clindx+1 && *(client+i)==-1) i++;

   if(i>clindx) {
      gprintf(" No clients");
      nc();
      return 1;
   }
   gprintf(" Clients:"); nc();
   for(i=0;i<clindx+1;i++) {
      if(*(client+i)>-1) {
#if OPENSSL
         if(*(clientSSL+i)!=NULL)
            gprintf("  socket %ds, port %5d,",*(client+i),*(clport+i));
         else
#endif
            gprintf("  socket %d, port %5d,",*(client+i),*(clport+i));

         if(       *(contyp+i)==LOCLI) gprintf(" conn C>S,");
         else if( *(contyp+i)==NATIVE) gprintf(" conn C>F,");
         else if( *(contyp+i)==LOSERV) gprintf(" conn S<C,");
         else if(*(contyp+i)==FOREIGN) gprintf(" conn S<F,");

         gprintf(" %s",*(clientIP+i));
         gprintf(" %s",*(clientLOGIN+i));

         nc();
      }
   }
   return 1;
}

int clientsockets() /* clientsockets ( --- hA) */
/* Putting on the stack the matrix that defines client sockets.
      Column         Contents
        1      client socket descriptor
        2      port number
        3      connection type
        4      run ptr (used by connection type = NATIVE) 
        5      cls ptr */
{
   int cols=5,i;
   double *A;
   char *name="_clientsockets";

   if(!matstk(clindx+1,cols,"_")) return 0;
   A=tos->mat;

   for(i=0;i<clindx+1;i++) *(A+i)=*(client+i); /* socket descriptor */
   A+=clindx+1;

   for(i=0;i<clindx+1;i++) *(A+i)=*(clport+i); /* port number */
   A+=clindx+1;

   for(i=0;i<clindx+1;i++) *(A+i)=*(contyp+i); /* connection type */
   A+=clindx+1;

   for(i=0;i<clindx+1;i++) *(A+i)=*(cliptr+i); /* run ptr */
   A+=clindx+1;

   for(i=0;i<clindx+1;i++) *(A+i)=*(clsptr+i); /* cls ptr */

   return(
   /* Running this phrase: dup 1st catch 0> rake lop */
      dup1s() &&
      pushint(XBASE) &&
      catch() &&
      zerogt() &&
      rake() &&
      lop() &&

      pushstr(name) &&
      naming()
   );
}

int clienttimesoff() /* clienttimesoff ( --- hA) */
/* Times of client disconnections. */
{
   int i=0,n=0,t;
   char *name="_clienttimesoff";

   for(;i<clindx+1;i++) {
      if((t=*(clitimoff+i))>-1) {
         pushint(t); /* time of disconnection */
         n++;
      }
   }
   if(n) {
      pushint(n);
      listfin();
      pushstr(name);
      naming();
   }
   else matstk(0,0,name);
   return 1;
}

int clienttimeson() /* clienttimeson ( --- hA) */
/* Times of client connections.  The first column of A contains the
   times and the second column contains the socket number. */
{
   int i=0,n=0,t;
   char *name="_clienttimeson";

   for(;i<clindx+1;i++) {
      if((t=*(clitim+i))>-1) {
         pushint(t); /* time of connection */
         n++;
      }
   }
   if(n) {
      pushint(n);
      listfin();
      for(i=0;i<clindx+1;i++) {
         if(*(clitim+i)>-1) {
            pushint(*(client+i)); /* socket number */
         }
      }
      pushint(n);
      listfin();
      park();
      pushstr(name);
      naming();
   }
   else matstk(0,0,name);
   return 1;
}

int drain(int sockfd)
/* Drain sockfd that is connected to another instance of this program.

   Drain runflag and a VOL from sockfd, and perform operations depend-
   ing upon the value of runflag.

   This function is reentrant. */
{
#define CLOSE 0
   ssize_t n;
   int bytes=0,err=0,n1,runflag;
/* int infix_save; */

   if(WTRACE) {
      timeprobe();
      gprintf("microsec delta: drain entering");
      nc();
   }
   if((err=readable_timeo(sockfd,R_WAIT,uR_WAIT,&bytes))<1) {
      if(err==0)
         gprintf(\
            " drain: socket %d readable wait time has passed",\
            sockfd);
      else
         gprintf(\
            " drain: select error in readable_timeo for socket %d",\
            sockfd);
      nc();
   /* stkerr("",""); */
      return CLOSE;
   }
/* Reading run flag: */
   n=read4n(sockfd,(unsigned int *)&runflag);

   if(runflag<SOCKET_CLOSE || runflag>GET_TOS) runflag=DRAIN_INVALID;
   if(n<0) n=0;

   if(WTRACE) {
      if(runflag==DRAIN_INVALID) {
         gprintf(\
            " drain: %d bytes in socket %d, runflag=DRAIN_INVALID",\
            n,sockfd);
      }
      else {
         gprintf(" drain: %d bytes in socket %d, runflag is %d",\
            n,sockfd,runflag);
      }
      nc();
   }
   if(!n || runflag==SOCKET_CLOSE) { /* remote closing the connection */
      if(WTRACE) {
         gprintf(\
            " drain: runflag is 0 or 0 bytes from socket %d",\
            sockfd);
         nc();
      }
      return CLOSE;
   }
   if(runflag==DRAIN_INVALID) {
      gprintf(" drain: ignore invalid runflag from socket %d",sockfd);
      nc();
      pushint(sockfd);
      if(sflush()) {
         if(WTRACE && tos->row) { /* hex xray of what flushed out */
            pushstr("dup INF xray . nl"); 
            xmain(0);
         }
         return(drop());
      }
      else return CLOSE;
   }
/*
February 2008

Previously, DRAIN_INVALID runflags caused the offending socket to be
closed.  

But sending a tar file through a socket and running xtar() (which runs 
Unix tar through shell()) on the remote end caused the bytes shown be-
low to be sent back through the socket.  Unix tar was complaining about
a time mismatch (5 seconds), certainly possible between machines.  

Closing the socket for such a complaint is not warranted.

These bytes were not sent through any of the program's socket writing 
functions, but they got here anyway.  The only right thing to do is to 
just ignore them.

So now invalid runflags are ignored and the socket is drained with 
sflush().  The comment above will be printed in the log file.  

The hexdump below shows an example of what came out from the time mis-
match in xtar().  For file 1080205_SM.bin, one of the files being ex-
tracted by /bin/tar, the message in the bytes below reads:

   /bin/tar: 1080205_SM.bin: time stamp 2008-02-05 00:11:58 is 5s 
   in the future

 72525 microsec delta: drain entering
 readn1: 4 bytes from socket 2
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  2F 62 69 6E 00 00 00 00 00 00 00 00 00 00 00 00  /bin............
 1312 microsec delta: end readn1
 drain: 4 bytes in socket 2, runflag=DRAIN_INVALID
 drain: ignore invalid runflag from socket 2
 readn1: 6 bytes from socket 2
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  2F 74 61 72 3A 20 00 00 00 00 00 00 00 00 00 00  /tar: ..........
 1363 microsec delta: end readn1
 sflush: read 6 bytes from socket 2
 39480 microsec delta: drain entering
 readn1: 4 bytes from socket 2
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  31 30 38 30 00 00 00 00 00 00 00 00 00 00 00 00  1080............
 1278 microsec delta: end readn1
 drain: 4 bytes in socket 2, runflag=DRAIN_INVALID
 drain: ignore invalid runflag from socket 2
 readn1: 64 bytes from socket 2
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  32 30 35 5F 53 4D 2E 62 69 6E 3A 20 74 69 6D 65  205_SM.bin: time
   2  20 73 74 61 6D 70 20 32 30 30 38 2D 30 32 2D 30   stamp 2008-02-0
   4  35 20 30 30 3A 31 31 3A 35 38 20 69 73 20 35 20  5 00:11:58 is 5
   6  73 20 69 6E 20 74 68 65 20 66 75 74 75 72 65 0A  s in the future.
 1515 microsec delta: end readn1
 sflush: read 64 bytes from socket 2
*/

/* Reading bytes of a VOL or STR written by netvolwrite(): */
   if(WTRACE) {
      gprintf(" drain: reading socket %d",sockfd);
      nc();
   }
   pushint(sockfd);
   netvolread(); /* put VOL or STR on stack */
   anyq();       /* check for empty item on stack */
   popint((int*)&n1);
   if(n1==xFALSE) { /* read error: stack item is empty */
      gprintf(" drain: error reading VOL or STR");
      nc();
      return CLOSE;
   }
   switch(runflag) {

      case GET_TOS:
         drop(); /* dropping STR that accompanies this request */
         if(!stkdepth()) pushstr(""); /* put something on empty stk */
         remotefd();
         remoteput();
      break;

      case DRAIN_VOL:
         return(
            pushstr("_Tsocket") &&
            pushint(sockfd) &&
            intstr() &&
            cat() &&
            naming()
         );
      break;

      case DRAIN_STR:
         pushint(XBASE);
         quote();
      break;

      case DRAIN_MAT:
         import8n();
         return(
            pushstr("_Asocket") &&
            pushint(sockfd) &&
            intstr() &&
            cat() &&
            naming()
         );
      break;

      case DRAIN_NUM:
         import8n();
         ontop();
      break;

      case VOL1_RUN2:
      /* Reading 2nd VOL bytes written by netvolwrite(): */
         if(WTRACE) {
            gprintf(" drain: reading second VOL from socket %d", \
               sockfd);
            nc();
         }
         pushint(sockfd);
         netvolread(); /* put VOL on stack */
         anyq();       /* check for empty VOL on stack */
         popint((int*)&n1);
         if(n1==xFALSE) { /* read error: VOL is empty */
            gprintf(" drain: error reading 2nd VOL");
            nc();
            return CLOSE;
         }
      case VOL_RUN:
      /* Run the tos VOL.  This function may be entered again. */
         if(WTRACE) {
            gprintf(" drain: running tos VOL from socket %d",sockfd);
            nl_prompt();
         }
         if(NTRACE || (WTRACE && KEYS)) {
         /* Showing first line of text to run: */
            gprintf(" drain: VOL first line: ");
            pushstr("this 1st quote strchop . nl");
            xmain(0);
         }
      /* The parsing branch below works great in certain cases.  But
         in general it does not.  For example, test/cluster fails.

         While it is possible that this receiving machine is running 
         infix, the sending machine may have sent a postfix phrase.  

         File clu.v is full of postfix communications between cluster 
         nodes.  Sending such postfix phrases to the parser in the 
         if() branch below will cause problems (for example, single 
         quotes used for postfix strings will look like misplaced 
         transpose operators to the parser).

         Communications in postfix are more efficient and are native
         to the program.  The rule must be that only postfix is sent 
         from machine to machine, so senders will need to parse their 
         infix messages into postfix before sending them.

         An analogy in networking is that numbers sent across the net-
         work are always converted to big endian first. */

      /* Do not parse here; postfix is required from the sender:
         if(INFIX) { 
            if(KEYS) parse1();
            else parse();
         }
      */
         xmain(0); /* run postfix text in VOL on stack */

         if(WTRACE) nl_prompt();
      break;

      default:
         gprintf(" drain: invalid runflag: %d",runflag);
         nc();
      /* stkerr("",""); */
         return CLOSE;
      break;
   }
   return 1;
#undef CLOSE
}

int drainf(int sockfd, int type, double ptrRun)
/* Drain sockfd that is connected to type FOREIGN or NEWCLI.

   This function receives bytes from foreign clients or servers.

   When connecting this program to another instance of this program,
   word CLIENT runs CONNECT() on the client end which connects to this
   program running SERVER on the other.

   On the CLIENT end, the connection is set to LOCLI or NATIVE (in
   CONNECT()).

   On the SERVER end, the connection is initially set to NEWCLI in
   terminal().

   Right after a connection, when this function on SERVER is entered 
   the very first time, type will still be NEWCLI as set in terminal().
   In this case, bytes read will be tested for string "LOGIN."

   If they match, the connection type is set to LOSERV, and SERVER will
   be responding to another instance of this program.  And if the con-
   nection type is LOSERV, drain() will be used hereafter.

   Otherwise, the connection type is set to FOREIGN, and bytes from the
   foreign client always come through here and are placed on the stack.
   Then, if ptrRun is not zero, the ptr it contains is executed.

   Word clientsockets can be run to show connection types. */
{
#define CLOSE 0
#define DMAX_BYTES 4096
#define WHILE_WAIT 0
#define uWHILE_WAIT 10000

   char CRLF[5]={0x0D,0x0A,0x0D,0x0A,0};
   char login[5],*T=NULL;
   int cn,k=0,len=0,len1=0,LIMIT=0,n=0,p[2],ret=0,server_type;
   int NO_BYTES_SO_FAR=1;

/* Number SERVE_F.SERVICE is in file net.v: */
   char *SERVICE="'SERVE_F' 'SERVICE' yank dup 0<> \
                  IF tok ELSE drop '' THEN";

   if(WTRACE) {
      gprintf(" drainf: socket %d type %d entering",sockfd,type);
      nc();
   }
   cn=*(sockCLI+sockfd); /* offset in client arrays */
   if(cn<0) {
      gprintf(" drainf: invalid client array offset for socket %d",
         sockfd);
      nc();
   /* stkerr("",""); */
      return CLOSE; /* return and close socket */
   }
   ret=readable_timeo(sockfd,WHILE_WAIT,uWHILE_WAIT,&len);

/* Limit bytes if not loopback client: */
   if(SERVER && strncmp("127.0.0.1",*(clientIP+cn),9)) {
      if(len>DMAX_BYTES && WTRACE) {
         gprintf(" drainf: %d bytes from socket %d limited to %d",\
            len,sockfd,DMAX_BYTES);
         nc();
      }
      LIMIT=1;
      len=MIN(DMAX_BYTES,len); 
   }
   if(ret>0) {
      if(len==0) { /* ret=1 and len=0 when sender has closed */
         if(WTRACE) {
            gprintf(" drainf: zero bytes, closing socket %d",sockfd);
            nc();
         }
         return CLOSE; /* socket is closed */
      }
      if(!strstk(len,"_foreign")) {
         stkerr(" drainf: ",MEMNOT);
         return CLOSE;
      }
      T=tos->tex;

      n=readn1(sockfd,T,len);
      if(n<0) { /* n<0 when read error, alarm or SSL error */
         if(WTRACE) {
            gprintf(\
               " drainf: readn1() error, closing socket %d",sockfd);
            nc();
         }
         drop();       /* T off stack */
         return CLOSE; /* socket is closed */
      }
      tos->col=MIN(n,len); /* set T length to match n */

      if(n) NO_BYTES_SO_FAR=0; /* have some bytes, so flag is false */
   }
   else n=0;

   if(WTRACE) {
      memcpy(p,&ptrRun,sizeof(double));
      gprintf(\
         " drainf: %d bytes from socket %d type %d ptrRun 0x%0X 0x%0X",\
         n,sockfd,type,*(p),*(p+1));
      nc();
   }
   if(type!=NEWCLI) {

   /* If here, the program is either a CLIENT (type==NATIVE, connected
      to a foreign server) or a SERVER (with client type==FOREIGN) re-
      ceiving bytes from a foreign source. */

      if(type==NATIVE) { /* CLIENT */
      /* Client receiving bytes from foreign server. */
         if(ptrRun) { 
         /* NATIVE, running ptrRun defined when CONNECT was run */

            len=tos->col; /* socket bytes read into T so far */

         /* Top of while-loop for CLIENT NATIVE */
            while(1) { 
               ret=readable_timeo(sockfd,WHILE_WAIT,uWHILE_WAIT,&len1);
               if(ret<0) { /* timeout or select error */
                  gprintf(" drainf: timeout waiting to read socket %d",
                     sockfd);
                  nc();
                  T=tos->tex;
                  break; /* exit this loop */
               }
               len+=len1;
               if(!(LIMIT && len>DMAX_BYTES)) {

               /* Put new STR on stack to receive bytes: */
                  if(!strstk(MAX(len1,1+strlen(CRLF)),"_f")) {
                     stkerr(" drainf: ",MEMNOT);
                     drop();       /* drop T from stack */
                     return CLOSE; /* return and close socket */
                  }
               /* Read len1 bytes into STR on stack: */
                  n=readn1(sockfd,tos->tex,len1); /* read len1 bytes */
                  if(n<1) {
                     if(WTRACE) {
                        gprintf(\
                           " drainf: exit while-loop for socket %d",
                           sockfd);
                        nc();
                     }
                     drop();
                     T=tos->tex;
                     break; /* exit this loop with T */
                  }
                  if(WTRACE) {
                     gprintf(\
                        " drainf: while-loop: %d bytes from socket %d",
                        n,sockfd);
                     nc();
                  }
                  tos->col=n; /* length for cat() */
                  cat(); /* concatenate this with T already on stack */
                  T=tos->tex; /* T+STR */
               }
               else {
               /* This is a SERVER running as a client to a foreign
                  server.  This is usually not done because bytes
                  are limited by DMAX_BYTES. */
                  gprintf(" drainf: excessive bytes to SERVER/client");
                  gprintf(": %d; halt reading socket %d",len,sockfd);
                  nc();
                  T=tos->tex;
                  break; /* exit this loop with T */
               }
            }
         /* Now have T on the stack, ready for ptrRun. */

            if(WTRACE) {
               gprintf(" drainf: %d total bytes from socket %d", \
                  tos->col,sockfd);
               nc();
            }
            if(tos->col==0) {
               if(WTRACE) {
                  gprintf(" drainf: closing socket %d",sockfd);
                  nc();
               }
               drop(); /* STR T off stack */
               return CLOSE;
            }
            if(WTRACE) {
               memcpy(p,&ptrRun,sizeof(double));
               gprintf(" drainf: socket %d running ptr 0x%0X 0x%0X",
                  sockfd,*(p),*(p+1));
               nc();
               timeprobe();
               gprintf("microsec delta: end drainf");
               nc();
            }
            return(
               pushint(sockfd) &&
               pushd(ptrRun) &&
               exe() /* stk: qT nSocket ptrRun */
            );
         }
         else {
            if(NO_BYTES_SO_FAR) {
               if(WTRACE) {
                  gprintf(" drainf: zero bytes, closing socket %d",
                     sockfd);
                  nc();
               }
               drop(); /* STR T off stack */
               return CLOSE;
            }
            return 1; /* returning with qT on the stack */
         }
      }
      else { /* SERVER */
         if(NO_BYTES_SO_FAR) {
            if(WTRACE) {
               gprintf(" drainf: closing socket %d",sockfd);
               nc();
            }
            drop(); /* STR T off stack */
            return CLOSE;
         }
      /* Server receiving bytes from foreign client.
         Word SERVE_F in net.v was set up when SERVER was created,
         and defines the type of this server (see enum foreign_types
         above): */
         extract1("SERVE_F","TYPE"); /* server type to stack */
         popint(&server_type);

         switch(server_type) {
            case HTTP:
            return 1; /* returning with T on stack */

            default:
            case TERM:
            /* Request T on the stack is processed by word SERVE_F.
               Word SERVE_F has this stack diagram: (qT nSocket --- ) */
               return(
                  pushint(sockfd) &&    /* push socket number */
                  pushstr("SERVE_F") && /* stk: (qT nSocket) */
                  xmain(0)              /* processing T */
               );
            break;
         }
      }
   }
   else {
   /* If here, the program is a SERVER receiving data for the first
      time from a new client, NEWCLI. */

      if(NO_BYTES_SO_FAR) {
         if(WTRACE) {
            gprintf(" drainf: closing socket %d",sockfd);
            nc();
         }
         if(T) drop(); /* STR T off stack */
         return CLOSE;
      }
   /* Determine the type of remote client.  

      A remote client that is this program makes this server a type
      LOSERV in relation to it.  This is determined by the characters 
      LOGIN in the first 5 bytes of T.  

      Other clients will make this server type FOREIGN in relation to 
      them. */

      memcpy(login,T,MIN(5,tos->col)); /* look at first 5 bytes of T */

      if(!strncmp(login,"LOGIN",5)) {
      /* Client type=LOSERV, and all future socket interaction will be
         through function drain(), not this one. */

         *(contyp+cn)=LOSERV; /* this type from now on */

         conn_alarm(); /* shut off alarm for NEWCLI */

      /* Put T from client into clientLOGIN string: */
         if(*(clientLOGIN+cn)) mallfree((void *)(clientLOGIN+cn));
         *(clientLOGIN+cn)=memgetn(T,tos->col); /* Login id */

         if(WTRACE) {
            gprintf(" drainf: socket %d set to type %d, LOSERV", \
               sockfd,LOSERV);
            nc();
         }
      /* This turned out to not be a foreign client.  Remove the socket
         entry from the table in word SERVE_F placed by clientmake(): */
         pushint(sockfd);
         extract1("SERVE_F","REM"); /* running word SERVE_F.REM */

      /* Verify client IP address using word SERVER_ALLOW (net.v),
         where the table of allowed IP addresses is kept: */
         pushstr(*(clientIP+cn));
         pushstr("SERVER_ALLOW");
         xmain(0);
         popint(&k); /* close connection if flag k is false */
         if(k==xFALSE) {
            datesys();
            dot();
            gprintf(" drainf: LOSERV connection from %s refused",\
               *(clientIP+cn));
            nl_prompt();
            drop(); /* dropping T */
            return CLOSE;
         }
         if(WTRACE) {
            gprintf(" drainf: LOSERV client %s",tos->tex);
            nc();
         }
      /* Drop client T (already placed into clientLOGIN above), and 
         make a server version of T to return to client: */
         drop(); 
         pushstr("getlogin spaced _host +"); /* T from this server */
         xmain(0);

      /* This acknowledgement back to client is important.  It appears
         to stabilize the connection, and without it, test/cluster fails
         regularly, about once every three runs. */
         pushint(sockfd);
         remoteput(); /* send ACK T to client (word CLIENT, net.v) */

      /* Inform this server of the new connection: */
         return(
            pushint(sockfd) &&
            pushstr("new_conn") &&
            xmain(0)
         );
      }
      else {
      /* NEWCLI will become client type=FOREIGN. */ 
         *(contyp+cn)=FOREIGN; /* this type from now on */

         conn_alarm(); /* shut off the alarm for NEWCLI */

         if(WTRACE) {
            gprintf(" drainf: socket %d set to type %d, FOREIGN", \
               sockfd,FOREIGN);
            nc();
         }
      /* Get the first line from T on the stack by running this phrase:
            dup textget 1st quote strchop
         Word clients will show the extracted line in its display. */
         dup1s();
         textget0();
         pushint(XBASE);
         quote();
         strchop();

         if(*(clientLOGIN+cn)) mallfree((void *)(clientLOGIN+cn));
         *(clientLOGIN+cn)=memgetn(tos->tex,tos->col); /* Login id */
         drop();

      /* The remaining bytes will be read from the socket and appended
         to T, then T will be processed by high level word SERVE_F.

         Word SERVE_F in net.v was set up when SERVER was created, and 
         defines the type of this server (enum foreign_types above): */
         extract1("SERVE_F","TYPE"); /* type serving foreign clients */
         popint(&server_type);
         if(WTRACE) {
            gprintf(" drainf: server type is %d",server_type);
            nc();
         }
         if(server_type==TERM) goto noWHILE;

      /* Finish getting bytes from FOREIGN client by adding to T riding
         on the stack.  

         The upcoming while-loop runs until the ending pattern CRLF={0D
         0A 0D 0A} is received in the last 4 bytes of T, as from an HTTP
         client, or until DMAX_BYTES have been read: */

         len=tos->col; /* socket bytes read into T so far */

      /* Top of while-loop for SERVER type HTTP */
         while(strncmp(T+len-4,CRLF,4)) {

            ret=readable_timeo(sockfd,WHILE_WAIT,uWHILE_WAIT,&len1);
            if(ret<0) { /* timeout or select error */
               gprintf(" drainf: timeout waiting to read socket %d",
                  sockfd);
               nc();
               pushstr(CRLF);
               cat();
               T=tos->tex;
               break; /* exit this loop with T+CRLF */
            }
            len+=len1;
            if(!(LIMIT && len>DMAX_BYTES)) {

            /* Put new STR on stack to receive bytes: */
               if(!strstk(MAX(len1,1+strlen(CRLF)),"_f")) {
                  stkerr(" drainf: ",MEMNOT);
                  drop();       /* drop T from stack */
                  return CLOSE; /* return and close socket */
               }
            /* Read len1 bytes into STR on stack: */
               n=readn1(sockfd,tos->tex,len1); /* read len1 bytes */

               if(n<1) {
                  if(WTRACE) {
                     gprintf(\
                     " drainf: while-loop %d bytes from socket %d",\
                     n,sockfd);
                     nc();
                  }
                  drop();
                  pushstr(CRLF);
                  cat();
                  T=tos->tex;
                  break; /* exit this loop with T+CRLF */
               }
               tos->col=n; /* length for cat() */
               cat(); /* concatenate this with T already on stack */
               T=tos->tex; /* T+STR */
            }
            else {
               gprintf(" drainf: excessive bytes to SERVER: %d",len);
               gprintf("; halt reading socket %d",sockfd);
               nc();
               pushstr(CRLF);
               cat();
               T=tos->tex;
               break; /* exit this loop with T+CRLF */
            }
         }
      /* Now Have T+CRLF on the stack, ready for SERVE_F. */

         noWHILE:

         switch(server_type) {
            case HTTP:
               if(test_socket_open(sockfd)) {
               /* Request T on the stack is processed by word SERVE_F.
                  Word SERVE_F has this stack diagram: 
                     (qT nSocket --- ) */
                  pushint(sockfd);    /* push socket number */
                  pushstr("SERVE_F"); /* stk: (qT nSocket) */
                  xmain(0);           /* process SERVER request in T */
               }
               else drop(); /* T off stack */

               pushint(sockfd);
               return(sclose()); /* close connection after request */
            break;

            default:
            case TERM:
            /* Word clients will show this in clientLOGIN display: */
               pushstr("TERM ");
               datesys();
            /* chopping out extra spaces: */
                  words();
                  typvol2str();
                  strchop();
               cat();

               if(*(clientLOGIN+cn)) mallfree((void *)(clientLOGIN+cn));
               *(clientLOGIN+cn)=memgetn(tos->tex,tos->col);
               drop();

            /* Request T on the stack is processed by word SERVE_F.
               Word SERVE_F has this stack diagram: (qT nSocket --- ) */
               if(WTRACE) {
                  timeprobe();
                  gprintf("microsec delta: end drainf");
                  nc();

               /* Show the name of the word serving this client: */
                  gprintf(" drainf: socket %d running SERVE_F",sockfd);
                  pushstr(SERVICE);
                  xmain(0);
                  if(tos->col) gprintf(" SERVICE: %s",tos->tex);
                  nc();
                  drop();
               }
               return(
                  pushint(sockfd) &&    /* push socket number */
                  pushstr("SERVE_F") && /* stk: (qT nSocket) */
                  xmain(0)              /* process T */
               );
            break;
         }
      }
   }
#undef CLOSE
#undef DMAX_BYTES
#undef WHILE_WAIT
#undef uWHILE_WAIT
}
/*
An example of the while-loop in drainf().

This shows the drainf() while-loop working in a case where the bytes of
a SERVER request do not all come through the socket at once.  The first
batch is read by readn1() at the top of drainf().  Then the while-loop
in drainf() runs, reading bytes from the socket until the pattern 0D 0A
0D 0A is received (CR LF CR LF):

[root@clacker] /home/dale # tops
         Tops 2.4.3
Thu Feb 17 00:37:46 PST 2005
[tops@clacker] ready # "" 80 SERVER ntrace

[tops@clacker] ready #
Thu Feb 17 00:38:00 PST 2005 SERVER: connection from 127.0.0.1
[tops@clacker] ready #  SERVER: new client on socket 6

Here is the first batch of bytes from readn1() at the top of drainf():

 readn1: 26 bytes received on socket 6
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  47 45 54 20 2F 69 6E 64 65 78 2E 68 74 6D 6C 20  GET /index.html
   2  48 54 54 50 2F 31 2E 30 0D 0A 00 00 00 00 00 00  HTTP/1.0........
 drainf: 26 bytes from socket 6, type -1
 drainf: socket 6 set to type 3

Now beginning the while-loop and reading more bytes until the pattern
0D 0A 0D 0A is received:

 readn1: 24 bytes received on socket 6
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  55 73 65 72 2D 41 67 65 6E 74 3A 20 74 6F 70 73  User-Agent: tops
   2  2F 32 2E 34 2E 33 0D 0A 00 00 00 00 00 00 00 00  /2.4.3..........
 readn1: 17 bytes received on socket 6
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  48 6F 73 74 3A 20 31 32 37 2E 30 2E 30 2E 31 0D  Host: 127.0.0.1.
   2  0A 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  ................
 readn1: 13 bytes received on socket 6
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  41 63 63 65 70 74 3A 20 2A 2F 2A 0D 0A 00 00 00  Accept: x/x.....
 readn1: 24 bytes received on socket 6
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  43 6F 6E 6E 65 63 74 69 6F 6E 3A 20 4B 65 65 70  Connection: Keep
   2  2D 41 6C 69 76 65 0D 0A 00 00 00 00 00 00 00 00  -Alive..........
 readn1: 2 bytes received on socket 6
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  0D 0A 00 00 00 00 00 00 00 00 00 00 00 00 00 00  ................

Ending the while-loop.  The last two readn1() inputs shown above have
resulted in the four ending bytes: 0D 0A 0D 0A

 clientclose: closing socket 6 on flag 3: Thu Feb 17 00:38:00 PST 2005
 clientclose: closing socket 6, port 32823, conn S<F
[tops@clacker] ready #

This shows the same request a little while later, when all the bytes
were received by readn1() at the top of drainf().  In this case, the 
while-loop never ran because the ending pattern 0D 0A 0D 0A had already
been received:

[tops@clacker] ready #
Thu Feb 17 00:54:29 PST 2005 SERVER: connection from 127.0.0.1
[tops@clacker] ready #  SERVER: new client on socket 6
 readn1: 106 bytes received on socket 6
       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F  0123456789ABCDEF
   0  47 45 54 20 2F 69 6E 64 65 78 2E 68 74 6D 6C 20  GET /index.html
   2  48 54 54 50 2F 31 2E 30 0D 0A 55 73 65 72 2D 41  HTTP/1.0..User-A
   4  67 65 6E 74 3A 20 74 6F 70 73 2F 32 2E 34 2E 33  gent: tops/2.4.3
   6  0D 0A 48 6F 73 74 3A 20 31 32 37 2E 30 2E 30 2E  ..Host: 127.0.0.
   8  31 0D 0A 41 63 63 65 70 74 3A 20 2A 2F 2A 0D 0A  1..Accept: x/x..
  10  43 6F 6E 6E 65 63 74 69 6F 6E 3A 20 4B 65 65 70  Connection: Keep
  12  2D 41 6C 69 76 65 0D 0A 0D 0A 00 00 00 00 00 00  -Alive..........
 drainf: 106 bytes from socket 6, type -1
 drainf: socket 6 set to type 3
 clientclose: closing socket 6 on flag 3: Thu Feb 17 00:54:29 PST 2005
 clientclose: closing socket 6, port 32839, conn S<F
[tops@clacker] ready #
*/

int export8n() /* export8n (hA --- hT) */
/* Convert 8-byte floating point numbers of matrix A into a volume of
   8-byte floating point number patterns for transfer over a network. */
{
   pushint(NET_ENDIAN);
   return(export8());
}

int import8n() /* import8n (hT --- hA) */
/* Convert 8-byte floating point number patterns in volume T, of network
   endian type, into 8-byte floating point numbers of matrix A in the
   machine's endian type. */
{
   pushint(NET_ENDIAN);
   return(import8());
}

int _IPhost() /* _IPhost (qHost --- hT) */
/* Table of IP addresses for Host. */
{
   struct hostent *hp;
   struct in_addr **pptr;
   int i,k=0;
   unsigned char c;
   char *name="_IPhost",*p;

   if(tos->typ!=STR) {
      stkerr(" _IPhost: ",STRNOT);
      return 0;
   }
   strchop();
   if((hp=gethostbyname(tos->tex))==NULL) {
      drop();
      strstk(0,"_empty"); /* empty STR if error */
      return 0;
   }
   drop(); /* qHost off stack */

   pptr=(struct in_addr **)hp->h_addr_list;

   for(;*pptr!=NULL;pptr++) {
      pushq2("",0); /* initial empty string for cat() */

      for(i=0;i<sizeof(struct in_addr);i++) {
         if(i) {
            pushstr(".");
            cat(); /* append a dot */
         }
         c=*((char *)*pptr+i); /* unsigned char value at ith byte */
         p=mprintf("%d",c);    /* format number to string */
         pushstr(p);           /* number string to stack */
         cat();                /* concatenate with the rest */

         mallfree((void *)&p);
      }
      strchop();
      k++;
   }
   if(k) {
      pushint(k);
      pilen();
   }
   else { /* empty VOL to stack: */
      volstk(0,0,"_empty");
   }
   return(
      pushstr(name) &&
      naming()
   );
}

/*
unsigned int ip4_scan(char *s, char ip[4])
// Stores each number separated by dots in an IP address as a one-byte
   character in 4-byte character array ip.

   From Reference 4, ip4_scan.c. //
{
  unsigned int i;
  unsigned int len;
  unsigned long u;

  len = 0;

  i = scan_ulong(s,&u);
  if (!i) return 0;
  ip[0] = u; s += i; len += i;
  if (*s != '.') return 0;
  ++s; ++len;

  i = scan_ulong(s,&u);
  if (!i) return 0;
  ip[1] = u; s += i; len += i;
  if (*s != '.') return 0;
  ++s; ++len;

  i = scan_ulong(s,&u);
  if (!i) return 0;
  ip[2] = u; s += i; len += i;
  if (*s != '.') return 0;
  ++s; ++len;

  i = scan_ulong(s,&u);
  if (!i) return 0;
  ip[3] = u; s += i; len += i;

  return len;
}
*/

int netendian() /* netendian (endian --- ) */
/* Set NET_ENDIAN that is used for 8-byte floating point numbers in net-
   work transfers made by this program.

   The default, NET_ENDIAN_DEF (file net.h), can be set to the endian
   of the most-used machines.  For a beowulf cluster of Intel proces-
   sors, this might be little endian to minimize unnecessary network
   conversions.

   This word allows NET_ENDIAN to be set to something other than default
   NET_ENDIAN_DEF, but it must be run on every connected machine (word
   cluster_run, clu.v, is handy for this).

   WARNING: NET_ENDIAN must be the same on all connected machines.
   WARNING: NET_ENDIAN must be the same on all connected machines.
   WARNING: NET_ENDIAN must be the same on all connected machines. */
{
   int net_endian=0;

   if(tos->typ!=NUM) {
      stkerr(" netendian: ",NUMNOT);
      return 0;
   }
   popint(&net_endian);
   if(net_endian) NET_ENDIAN=net_endian;
   else NET_ENDIAN=NET_ENDIAN_DEF;
   return 1;
}

int netrate() /* netrate (bytes/sec --- ) */
/* Set the network transfer rate used for the network read alarm. */
{
   int bps;

   if(!popint(&bps)) return 0;

   NET_RATE=MAX(1,bps);
   return 1;
}

int netrate_val() /* NETRATE ( --- bytes/sec) */
/* Current network transfer rate used to set the network read alarm. */
{
   return(pushint(NET_RATE));
}

int netvolread() /* netvolread (nSocket --- hT) */
/* From network Socket, read bytes written by netvolwrite() and build
   a volume T on the stack.

   The following, written by netvolwrite(), is read from Socket:
      len: total VOL bytes, equal to its rows times chars-across
      rows: number of VOL rows
      bytes: all len bytes of the VOL

   Returned T on stack is a VOL with number of rows equal to rows read
   from Socket, and chars across equal to len read from Socket divided
   by rows.

   If error, T will be an empty volume. */
{
#define VOL_ABORT volstk(0,0,"_empty");

   int n,sockfd;
   unsigned int len=0,rows=0;
   char *T;

   if(tos->typ!=NUM) {
      stkerr(" netvolread: ",NUMNOT);
      return 0;
   }
   popint(&sockfd);

/* Reading bytes written by netvolwrite() */
   n=read4n(sockfd,&len); /* len=rows*chars */

   n=read4n(sockfd,&rows); /* rows */
   if(rows<=0 || rows>MAXROWS) {
      gprintf(" netvolread error: invalid rows = %u on socket %d",
         rows,sockfd);
      nc();
      VOL_ABORT
      return 0;
   }
   if(!volstk(rows,len/rows,"_Tsocket")) {
      stkerr(" netvolread: ",MEMNOT);
      VOL_ABORT
      return 0;
   }
/* Reading data into VOL on stack: */
   T=tos->tex;
   n=readn1(sockfd,T,len);
   if(n!=len) {
      gprintf(" netvolread: read %d bytes, expected %d from socket %d",\
         n,len,sockfd);
      nc();
      VOL_ABORT
      return 0;
   }
   return 1;
#undef VOL_ABORT
}

int netvolwrite() /* netvolwrite (hT nSocket --- ) */
/* Write volume T to network Socket.
   The following is written to Socket:

      len: total VOL bytes, equal to its rows times chars-across
      rows: number of VOL rows
      bytes: all len bytes of the VOL

   Function netvolread() is later used to read T from the other end
   of Socket (where nSocket is probably different). */
{
   int cn,count=0,len,rows,sockfd;
   char *T;

   if(tos->typ!=NUM) {
      stkerr(" netvolwrite: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" netvolwrite: ",STRORVOLNOT);
      return 0;
   }
   popint(&sockfd);

   if(sockfd<0 || sockfd>FD_SETSIZE-1 || 
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
      stkerr(" netvolwrite: ","invalid client socket descriptor");
      return 0;
   }
   T=tos->tex;
   rows=tos->row;
   len=rows*tos->col; /* total VOL bytes */

   if(len) {

   /* Sockets are first-in, first-out (FIFO, like a pipe, not LIFO like
      a stack): */
      count=write4n(sockfd,len);   /* write len of VOL */
      if(count==4) 
         count+=write4n(sockfd,rows); /* write rows of VOL */
      if(count==8) 
         count+=writen1(sockfd,T,len); /* write all len bytes of VOL */

      if(count!=8+len) {
         gprintf(" netvolwrite: %d bytes out of %d to socket %d",\
            count,8+len,sockfd);
         nc();
         stkerr("","");
         drop();
         return 0;
      }
      if(WTRACE) {
         gprintf(" netvolwrite: all %d bytes to socket %d",\
            len,sockfd);
         nc();
      }
      return(drop());
   }
   else {
      gprintf(" netvolwrite: cannot write zero length to socket %d", \
         sockfd);
      nc();
      stkerr("","");
      drop();
      return 0;
   }
}

int ptrCls() /* ptrCls (nSocket --- ptr) */
/* Fetch ptrCls used in clientclose() when Socket closes.
   The stack diagram of word(ptr) is: (nSocket --- ) */
{
   int k;

   if(tos->typ!=NUM) {
      stkerr(" ptrCls: ",NUMNOT);
      return 0;
   }
   clientindex();
   popint(&k);

   if(k<0) {
      stkerr(" ptrCls: ","socket has no client");
      return 0;
   }
   return(pushd(*(clsptr+k-XBASE)));
}

int ptrCls_upd() /* ptrCls_upd (ptr nSocket --- ) */
/* Update ptrCls used in clientclose() when Socket closes.
   The stack diagram of word(ptr) is: (nSocket --- ) */
{
   int k;
   double ptr;

   if(tos->typ!=NUM || (tos-1)->typ!=NUM) {
      stkerr(" ptrCls_upd: ",NUMS2NOT);
      return 0;
   }
   clientindex();
   popint(&k);

   popd(&ptr);

   if(k<0) {
      stkerr(" ptrCls_upd: ","socket has no client");
      return 0;
   }
   *(clsptr+k-XBASE)=ptr;
   return 1;
}

int ptrRun() /* ptrRun (nSocket --- ptr) */
/* Fetch ptrRun used in drainf() by a NATIVE client. 
   The stack diagram of word(ptr) is: (qT nSocket --- ) */
{
   int k;

   if(tos->typ!=NUM) {
      stkerr(" ptrRun: ",NUMNOT);
      return 0;
   }
   clientindex();
   popint(&k);

   if(k<0) {
      stkerr(" ptrRun: ","socket has no client");
      return 0;
   }
   return(pushd(*(cliptr+k-XBASE)));
}

int ptrRun_upd() /* ptrRun_upd (ptr nSocket --- ) */
/* Update ptrRun used in drainf() by a NATIVE client.
   The stack diagram of word(ptr) is: (qT nSocket --- ) */
{
   int k;
   double ptr;

   if(tos->typ!=NUM || (tos-1)->typ!=NUM) {
      stkerr(" ptrRun_upd: ",NUMS2NOT);
      return 0;
   }
   clientindex();
   popint(&k);

   popd(&ptr);

   if(k<0) {
      stkerr(" ptrRun_upd: ","socket has no client");
      return 0;
   }
   *(cliptr+k-XBASE)=ptr;
   return 1;
}

ssize_t read4n(int fd, unsigned int *n)
/* Reads long (4-byte) integer n from fd and converts it from network
   byte order (probably big endian) to host byte order. */
{
   int count;
   unsigned int n1;

   if((count=readn1(fd,&n1,sizeof(int)))<sizeof(int)) return count;

   *n=ntohl(n1);
   return count;
}

/*
unsigned int scan_ulong(register char *s, register unsigned long *u)
// From Reference 4, ip4_scan.c. //
{
  register unsigned int pos = 0;
  register unsigned long result = 0;
  register unsigned long c;
  while ((c = (unsigned long) (unsigned char) (s[pos] - '0')) < 10) {
    result = result * 10 + c;
    ++pos;
  }
  *u = result;
  return pos;
}
*/

int read_f() /* READ_F (n --- ) */
/* In readn1(), set to n maximum number of pending bytes to read each 
   time the server reads a FOREIGN or NEWCLI socket.  

   If the number of pending bytes is greater than n, drainf() will con-
   tinue to call readn1() and return n (or fewer) bytes to the SERVE_F
   handler until all bytes from the socket have been read or until the 
   socket has been closed. */
{
   int n;
   if(!popint(&n)) return 0;
   READ_F=MAX(0,n);

   return 1;
}

int readable_timeo(int fd, time_t sec, time_t usec, int *bytes)
/* Blocks in select() for up to sec+usec seconds, waiting for socket 
   fd to become readable.

   Returns >0 if socket is readable, or 0 if timed out, or <0 if error
   in select().

   If socket is readable, bytes is the number to read. */
{
   fd_set rset;
   struct timeval tv;
   int ret;
#ifdef OPENSSL
   int cn;
#endif

   if(fd<0 || fd>FD_SETSIZE-1) {
      stkerr(" readable_timeo: ","invalid client socket descriptor");
      return -1;
   }
#ifdef OPENSSL
/* If pending bytes are in the SSL buffer, return byte count as if 
   they were still in the socket.  

   Function readn1() will get them when SSL_read() is run. 

   Word socket_readable uses this function: (nSocket --- nBytes).
   Bytes it returns will reflect pending bytes in SSL buffer or
   bytes in the socket. */

   cn=*(sockCLI+fd);
   if(cn>-1 && (*bytes=*(clientSSL_PENDING+cn))>0) return 1; 
#endif

   FD_ZERO(&rset);
   FD_SET(fd,&rset);

   tv.tv_sec=sec;
   tv.tv_usec=usec;

/* Select returns:
      >0 if descriptor is readable
      =0 if timeout
      <0 if select error */
   ret=select(fd+1,&rset,(fd_set *)NULL,(fd_set *)NULL,&tv);

   if(ret>0) ioctl(fd,FIONREAD,bytes);
   else *bytes=0;

   return(ret);
}

ssize_t readn1(int fd, void *vptr, size_t len)
/* Read up to len bytes from file descriptor fd into vptr.

   Returns length of bytes actually read, which may be less than len.

   Sets an alarm based upon NET_RATE and len bytes to read.
   Returns -1 if error, probably an interrupt by the alarm. */
{
   Sigfunc *sigfunc;
   int cn,fssl=0,n=0,nr=0,nsec=0,rc=0;
   const int nsec_min=20; /* minimum alarm seconds */
   char *ptr;
   time_t time1=0,time2=0;

#if OPENSSL
   SSL *ssl=NULL;
#endif

   if(len==0) return(len);
   time(&time1);
   cn=*(sockCLI+fd);

   if(READ_F) {
      if(cn>-1 && (*(contyp+cn)==FOREIGN || *(contyp+cn)==NEWCLI)) {
         if(COG) {
            if(!*(SELECT+cn)) return(0); /* 0 bytes if SELECT(cn)=0 */

            else { /* clear the FD_SET bit for SELECT(cn) */
               pushint(fd); /* running the phrase: fd no SELECT */
               pushint(xFALSE);
               setselect();
            }
         }
         nr=MIN(len,READ_F); /* limit nr bytes to read from foreign */
      }
      else nr=len;
   }
   else { 
      if(cn>-1 && (*(contyp+cn)==FOREIGN || *(contyp+cn)==NEWCLI))
         if(!*(SELECT+cn)) return(0); /* 0 bytes if SELECT(cn)=0 */
      nr=len;
   }
   ERRALRM=0;
   sigfunc=signal(SIGALRM,connect_alarm); /* save signal handler */
   nsec=MAX(nsec_min,nr/NET_RATE);
   if((aset=alarm(nsec))!=0) {
      gprintf(" readn1: using alarm that was already set to %d",aset);
      nc();
   }
#if OPENSSL
   if(cn>-1 && (ssl=*(clientSSL+cn))) {

      fssl=1;
      n=SSL_read(ssl,vptr,nr); /* n is number of valid decrypted bytes,
                                  which may be less than nr */

      if(SSL_get_error(ssl,n)!=SSL_ERROR_NONE) {
          gprintf(" readn1: error reading secure socket %d",fd);
          nc();
      }
      *(clientSSL_PENDING+cn)=SSL_pending(ssl);
   }
   else {
#endif
   ptr=vptr;
   while(n<nr) {
      if((rc=read(fd,ptr,nr-n))<1) {
         if(rc<0) n=-1; /* probable interrupt from alarm; must look
                           at errno set by read() to learn more */
         if(1 || WTRACE) {
            if(errno==ECONNRESET) {
               gprintf(
                  " readn1: socket %d errno %d",fd,errno); 
               gprintf(" connection reset by peer");
            }
            else {
               gprintf(
                  " readn1: probable alarm interrupt, errno: %d",errno);
            }
            nc();
         }
         break; /* n will be 0 (initial value) or -1 */
      }
      ptr+=rc;
      n+=rc;
   }

#if OPENSSL
   }
#endif

   alarm(0);                /* turn off the alarm */
   ERRALRM=0;
   signal(SIGALRM,sigfunc); /* restore signal handler */

   if(WTRACE && n!=len) {
      gprintf(" readn1: read %d of requested %d bytes in socket %d",\
         MAX(n,0),len,fd);
      nc();

      time(&time2);
      if((float)(time2-time1)>(float)0.9*nsec) {
         gprintf(" readn1: probable timeout of %d second alarm",nsec);
         nc();
      }
   }
   if(WTRACE) {
      if(n>-1) {
         if(fssl) 
            gprintf(" readn1: %d secure bytes from socket %d",n,fd);
         else
            gprintf(" readn1: %d bytes from socket %d",n,fd);
         nc();
         if(NTRACE && n>0) {
            pushq2(vptr,n);
            pushstr("512 xray"); /* xray up to 512 bytes */
            xmain(0);
            dot();
            nc();
         }
      }
      timeprobe();
      gprintf("microsec delta: end readn1");
      nc();
   }
   return(n);
}

int remotefd() /* remotefd ( --- nSocket) */
/* Put on stack the file descriptor, SOCKFD, of the socket at the cur-
   rent run level which has been passed up from the level below.

   Functions in ctrl.c (like bufjump() and bufreturn()) take care of
   setting the currently active socket file descriptor, SOCKFD, as
   run levels are raised and lowered.

   A remote connection exists if SOCKFD is 0 or greater, and current
   words are considered to be running "remotely."

   If there is no connection, SOCKFD is -1 and the words are not run-
   ning "remotely."

   When words are running remotely, those in the program that mimic
   Unix command line functions (certain words in sys.v that do not fit
   the program's ready-aim-fire model) will not run correctly because
   keys are not really being hit. */
{
   return(pushint(SOCKFD));
}

int remoteget() /* remoteget (nSocket --- hT)
                   stack on remote: (hT --- ) */
/* Gets stack item T from the stack of the remote client that is con-
   nected to Socket.  If remote stack is empty, T is an empty VOL. */
{
   int cn,runflag=GET_TOS,sockfd;

   if(tos->typ!=NUM) {
      stkerr(" remoteget: ",NUMNOT);
      return 0;
   }
   sockfd=tos->real;

   if(sockfd<0 || sockfd>FD_SETSIZE-1 || 
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
      stkerr(" remoteget: ","invalid client socket descriptor");
      return 0;
   }
   return(
      pushstr("G") &&                 /* dummy byte */
      swap() &&
      (write4n(sockfd,runflag)==4) && /* flag remote to get tos */
      netvolwrite() &&                /* write dummy byte to socket */
      block(sockfd,MAXBLOCK)
   );
}

int remoteput() /* remoteput (hT nSocket --- )
                   stack on remote: ( --- hT) */
{
   int cn,len,runflag=DRAIN_INVALID,sockfd;

   if(stkdepth()<2) {
      stkerr(" remoteput: ",NEEDTWO);
      return 0;
   }
   if(tos->typ!=NUM) {
      stkerr(" remoteput: ",NUMNOT);
      return 0;
   }
   sockfd=tos->real;

   if(sockfd<0 || sockfd>FD_SETSIZE-1 || 
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
      stkerr(" remoteput: ","invalid client socket descriptor");
      return 0;
   }
   switch((tos-1)->typ) {

      case STR:
         swap();
         hand();
         swap();
         runflag=DRAIN_STR;

      case VOL:
         if(runflag==DRAIN_INVALID) runflag=DRAIN_VOL;
         sockfd=tos->real;

         len=(tos-1)->row*(tos-1)->col;
         if(len) {
            if(write4n(sockfd,runflag)==4) /* flag remote to read */
               return(netvolwrite());      /* write T to socket */
            else {
               drop();
               return 0;
            }
         }
         else
         /* Function netvolread() to be used by the client puts an
            empty VOL on the stack as an error indicator, so we can't
            send an empty VOL across the network.  
            Instead, run a phrase on the remote client to make it put 
            an empty VOL on its own stack: */
            return(
               lop() &&
               pushstr("VOL tpurged") &&
               swap() &&
               remoterun()
            );
      break;

      case NUM:
         swap();
         hand();
         swap();
         runflag=DRAIN_NUM;

      case MAT:
         if(runflag==DRAIN_INVALID) runflag=DRAIN_MAT;
         popint(&sockfd);

         len=tos->row*tos->col;
         if(len) {
            if((write4n(sockfd,runflag)==4) && /* flag remote to read */
               export8n() &&          /* MAT to network endian VOL*/
               pushint(sockfd))

               return(netvolwrite()); /* write T to socket */
            else {
               drop();
               return 0;
            }
         }
         else
         /* Function netvolread() to be used by the client puts an
            empty VOL on the stack as an error indicator, so we can't
            send a purged MAT (posing as an empty VOL) across the net-
            work.  Instead, run phrase "0 0 null" on the remote client
            to make it put a purged MAT on its own stack: */
            return(
               drop() &&
               pushstr("0 0 null") &&
               pushint(sockfd) &&
               remoterun()
            );
      break;

      default:
         gprintf(" remoteput: invalid for network, stack type: %d",\
            (tos-1)->typ);
         nc();
         stkerr("","");
         return 0;
      break;
   }
   return 1;
}

int remoteputf() /* remoteputf (hT nSocket --- ) */
/* Write T to Socket connected to foreign client or server. */
{
   char CRLF[3]={0x0D,0x0A,0};
   int open=0,cn,count,len,sockfd;

   if(tos->typ!=NUM) {
      stkerr(" remoteputf: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" remoteputf: ",STRORVOLNOT);
      return 0;
   }
   popint(&sockfd);

   if(sockfd<0 || sockfd>FD_SETSIZE-1 || 
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
      gprintf(" remoteputf: %d is an invalid client socket descriptor",
         sockfd);
      nc();
      stkerr("","");
      return 0;
   }
   open=_client_open(sockfd);

   len=tos->row*tos->col;
   if(!len) {
      drop();
      len=strlen(CRLF);
      pushstr(CRLF); /* send CRLF if empty T */
   }
   if(WTRACE) {
      gprintf(" remoteputf: writing %d bytes to socket %d",\
         len,sockfd);
      nc();
      if(NTRACE) {
         dup1s();
         pushstr("512 xray");
         xmain(0);
         dot();
         nc();
      }
      if(!open) {
         gprintf(" remoteputf: client on socket %d is closed, ",sockfd);
         gprintf("no bytes written");
         nc();
      }
   }
   if(open) count=writen1(sockfd,tos->tex,len);
   else count=len; /* client is closed, really never wrote any */

/* Ignore count=-1.  It happens a lot when closing foreign. */
   if(count!=len && count!=-1) { 
      gprintf(" remoteputf: to socket %d wrote %d bytes out of %d",\
         sockfd,count,len);
      nc();
      stkerr("","");
      drop();
      return 0;
   }
   if(WTRACE) {
      timeprobe();
      gprintf("microsec delta: end remoteputf");
      nc();
   }
   return(drop());
}

int remoteputf0() /* remoteputf0 (hT nSocket --- ) */
/* Write T to Socket connected to foreign client or server,
   followed by a null byte. */
{
   char CRLF[3]={0x0D,0x0A,0};
   int open=0,cn,count,len,sockfd;

   if(tos->typ!=NUM) {
      stkerr(" remoteputf0: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR && (tos-1)->typ!=VOL) {
      stkerr(" remoteputf0: ",STRORVOLNOT);
      return 0;
   }
   popint(&sockfd);

   if(sockfd<0 || sockfd>FD_SETSIZE-1 ||
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
      gprintf(" remoteputf0: %d is an invalid client socket descriptor",
         sockfd);
      nc();
      stkerr("","");
      return 0;
   }
   open=_client_open(sockfd);

   len=tos->row*tos->col;
   if(!len) {
      drop();
      len=strlen(CRLF);
      pushstr(CRLF); /* send CRLF if empty T */
   }
   if(WTRACE) {
      gprintf(" remoteputf0: writing %d bytes to socket %d",\
         len,sockfd);
      nc();
      if(NTRACE) {
         dup1s();
         pushstr("512 xray");
         xmain(0);
         dot();
         nc();
      }
      if(!open) {
         gprintf(" remoteputf0: client on socket %d is closed, ",
            sockfd);
         gprintf("no bytes written");
         nc();
      }
   }
   if(open) /* Using the fact that tos->tex (VOL or STR) always has
      an ending null at len+1: */
      count=writen1(sockfd,tos->tex,len+1)-1;
   else count=len; /* client is closed, really never wrote any */

/* Ignore count=-1.  It happens a lot when closing foreign. */
   if(count!=len && count!=-1) {
      gprintf(" remoteputf0: to socket %d wrote %d bytes out of %d",\
         sockfd,count,len);
      nc();
      stkerr("","");
      drop();
      return 0;
   }
   if(WTRACE) {
      timeprobe();
      gprintf("microsec delta: end remoteputf0");
      nc();
   }
   return(drop());
}

int remoteputfb() /* remoteputfb (hT nSocket nSec --- ) */
/* Write T to Socket connected to foreign client or server, and
   block for a response for up to Sec seconds. */
{
   char CRLF[3]={0x0D,0x0A,0};
   double sec;
   int cn,count,len,sockfd;

   if(!popd(&sec)) return 0;
   if(!popint(&sockfd)) return 0;

   sec=MAX(0,sec);

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" remoteputfb: ",STRORVOLNOT);
      return 0;
   }
   if(sockfd<0 || sockfd>FD_SETSIZE-1 || 
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
      stkerr(" remoteputfb: ","invalid client socket descriptor");
      return 0;
   }
   len=tos->row*tos->col;

   if(WTRACE) {
      gprintf(" remoteputfb: writing %d bytes to socket %d",\
         len,sockfd);
      nc();
      if(NTRACE) {
         dup1s();
         pushstr("512 xray");
         xmain(0);
         dot();
         nc();
      }
   }
   if(!len) {
      drop();
      len=strlen(CRLF);
      pushstr(CRLF); /* send CRLF if empty T */
   }
   LOCK_PEND=1;
   count=writen1(sockfd,tos->tex,len);

/* Ignore count=-1.  It happens a lot when closing foreign. */
   if(count!=len && count!=-1) {
      gprintf(" remoteputfb: to socket %d wrote %d bytes out of %d",\
         sockfd,count,len);
      nc();
      stkerr("","");
      drop();
      return 0;
   }
   return(
      drop() &&
      block(sockfd,sec)
   );
}

int remoteputmat() /* remoteputmat (hA nSocket --- );
                      stack on remote: ( --- hA) */
/* Convert MAT A to network endian VOL and put it in Socket that con-
   nects to remote.  The remote will later read VOL from its end of
   the connection, convert it to MAT A of its endian, and put A on its
   stack.

   Function remoteput() can be used instead of this function.  This
   function is being kept to use for validation of remoterun2() and
   case VOL1_RUN2 in drain(). */
{
   int len,ret=1;

   if(tos->typ!=NUM) {
      stkerr(" remoteputmat: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT && (tos-1)->typ!=NUM) {
      stkerr(" remoteputmat: ",NUMORMATNOT);
      ret=0;
      matstk(0,0,"_purged"); /* send a purged matrix */
      swap();
   }
   swap(); /* Stack: (nSocket hA) */
   hand(); /* if A is NUM, this turns it into 1-by-1 MAT handle */

   len=tos->row*tos->col;
   if(len)
      return(
         export8n() && /* MAT to network endian VOL, T1 */
         pushstr("import8n") && /* client T2 command to run */
         rot() && /* stack: hT1 hT2 nSocket */
         remoterun2() &&
         ret
      );
   else
   /* Function netvolread() to be used by the client puts an empty VOL
      on the stack as an error indicator, so we can't send a purged MAT
      (posing as an empty VOL) across the network.  Instead, run phrase
      "0 0 null" on the remote client to make it put a purged MAT on its
      own stack: */
      return(
         drop() &&
         pushstr("0 0 null") &&
         swap() &&
         remoterun() &&
         ret
      );
}

int remoterun() /* remoterun (hT nSocket --- ) or (hT hSocket --- ) */
/* Write T to Socket with runflag=VOL_RUN.  When the remote runs
   drain(), runflag=VOL_RUN will cause T to appear on its stack and
   it will run T.

   Broadcast: If hSocket (MAT), it is taken to be a list of socket
   numbers, and T will be run on all remotes. */
{
   int cn,k=0,n=0,sockfd=0;
   double *S;
   const int runflag=VOL_RUN; /* flag the remote to run T */

   if(stkdepth()<2) {
      stkerr(" remoterun: ",NEEDTWO);
      return 0;
   }
   if(tos->typ==NUM) {

      sockfd=tos->real;

      if(sockfd<0 || sockfd>FD_SETSIZE-1 || 
         (cn=*(sockCLI+sockfd))>clindx || cn<0) {
         stkerr(" remoterun: ","invalid client socket descriptor");
         drop2(); /* drop (hT nSocket) */
         return 0;
      }
      if(REMOTERUN_DELAY) {
         return(
         /* Send flag to read and run T: */
            (write4n(sockfd,runflag)==4) && 

            netvolwrite() && /* write T to socket */

            pushstr("UNLOCK") &&
            pushd(REMOTERUN_DELAY) &&
            LOCK1()
         );
      }
      else {
         return(
         /* Send flag to read and run T: */
            (write4n(sockfd,runflag)==4) && 

            netvolwrite() /* write T to socket */
         );
      }
   }
   if(tos->typ==MAT) {
      S=tos->mat;
      n=tos->row;

   /* First check all the sockets: */
      while(k<n) {
         sockfd=*S;

         if(sockfd<0 || sockfd>FD_SETSIZE-1 || 
            (cn=*(sockCLI+sockfd))>clindx || cn<0) {
            stkerr(" remoterun: ","invalid client socket descriptor");
            drop2(); /* drop (hT nSocket) */
            return 0;
         }
         k++;
         S++;
      }
      S=tos->mat;
      lpush(); /* put hSocket on temp stack */

   /* There is less checking on success when this MAT loop is run,
      compared to the NUM branch above.  But above it has checked 
      all the sockets beforehand. */

      k=0;
      while(k<n) {
         sockfd=*S;

         write4n(sockfd,runflag); /* send flag to read T socket from
                                     socket and then run T */
         dup1s();                 /* dup T */
         pushint(sockfd);         /* nSocket to stack */
         netvolwrite();           /* write T to socket */

         if(REMOTERUN_DELAY) {
            pushstr("UNLOCK");
            pushd(REMOTERUN_DELAY);
            LOCK1();
         }
         k++;
         S++;
      }
      return(drop() &&     /* drop T */
         lpull() && drop() /* hSocket from temp stack and drop */
      );
   }
   stkerr(" remoterun: ",NUMORMATNOT);
   return 0;
}

int remoterun2() /* remoterun2 (hT1 hT2 nSocket --- ) */
/* Write VOLs T1 and T2 to Socket.  When the remote runs drain(), T1
   and T2 will appear on its stack and it will run T2.

   Functions remoteprompter() and remoteputmat() use this function. */
{
   int cn,sockfd;
   const int runflag=VOL1_RUN2 ; /* flag for remote to run T2 */

   if(stkdepth()<2) {
      stkerr(" remoterun2: ",NEEDTWO);
      return 0;
   }
   if(tos->typ!=NUM) {
      stkerr(" remoterun2: ",NUMNOT);
      return 0;
   }
   sockfd=tos->real;

   if(sockfd<0 || sockfd>FD_SETSIZE-1 || 
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
      stkerr(" remoterun2: ","invalid client socket descriptor");
      return 0;
   }
   if(write4n(sockfd,runflag)!=4) {
      gprintf(" remoterun2: error writing runflag to socket %d",sockfd);
      nc();
      stkerr("","");
      drop2();
      drop();
      return 0;
   }
   rot();
   over(); /* (T2 nSocket T1 nSocket) on stack */
/* Sockets are first-in, first-out (FIFO, like a pipe, not LIFO like
   a stack): */
   if(!netvolwrite()) { /* write T1 to socket following flag */
      gprintf(" remoterun2: error writing T1 to socket %d",sockfd);
      nc();
      stkerr("","");
      drop2();
      return 0;
   }
   if(!netvolwrite()) { /* write T2 to socket following T1 */
      gprintf(" remoterun2: error writing T2 to socket %d",sockfd);
      nc();
      stkerr("","");
      return 0;
   }
   return 1;
}

int remoterun_DELAY() /* remoterun_DELAY ( --- d) */
/* The current number of seconds to delay between each socket write in
   word remoterun. */
{
   return(pushd(REMOTERUN_DELAY));
}

int remoterun_delay() /* remoterun_delay (d --- ) */
/* Set the number of seconds (or fraction of second) to delay between
   each socket write in word remoterun. */
{
   double d;

   if(!popd(&d)) return 0;

   REMOTERUN_DELAY=MAX(0,d);
   return 1;
}

int sclose() /* sclose (nSocket --- ) */
/* Close file descriptor, or list of file descriptors, nSocket. */
{
   int k=0,ret=1,sockfd;
   double *A;

   if(tos->typ!=NUM && tos->typ!=MAT) {
      stkerr(" sclose: ",NUMORMATNOT);
      return 0;
   }
   if(tos->typ==NUM) {
      popint(&sockfd);
      return(clientclose(sockfd,0));
   }
   A=tos->mat;
   for(;k<tos->row;k++) {
      sockfd=*A;
      ret=(ret && clientclose(sockfd,0));
      A++;
   }
   return(ret && drop());
}

int sflush() /* sflush (nSocket --- hT) */
/* Empty all pending bytes from Socket and return them in T.  Socket 
   is a socket number or a list of socket numbers.  Flushed bytes from 
   a socket that had any occupy a row of T; empty sockets do not con-
   tribute to T. */
{
   int bytes=0,k=0,readf_save,ret,sockfd,total,usec=2;
   double *A;

   if(tos->typ!=NUM && tos->typ!=MAT) {
      stkerr(" sflush: ",NUMORMATNOT);
      return 0;
   }
   readf_save=READ_F;
   READ_F=0;

   hand();
   A=tos->mat; /* array of socket numbers */

/* VOL to receive flushed bytes: */
   if(!volstk(0,0,"_empty")) {
      READ_F=readf_save;
      return 0;
   }
   for(;k<(tos-1)->row;k++) {
      sockfd=*A;
      total=0;
      ret=1;
      while(ret>0) {
         ret=readable_timeo(sockfd,0,usec,&bytes);
         if(ret>0) {
            if(!volstk(1,bytes,"_")) { /* temp VOL on stk for bytes */
               READ_F=readf_save;
               return 0;
            }
            ret=readn1(sockfd,tos->tex,bytes);
            pilev();
            total+=bytes;
         }
      }
      if(WTRACE && total) {
         gprintf(" sflush: %d bytes from socket %d",total,sockfd);
         nc();
      }
      A++;
   }
   READ_F=readf_save;
   return(lop());
}

Sigfunc *signal(int signo, Sigfunc *func)
/* From Reference 1. */
{
        struct sigaction        act, oact;

        act.sa_handler = func;
        sigemptyset(&act.sa_mask);
        act.sa_flags = 0;
        if (signo == SIGALRM) {
#ifdef  SA_INTERRUPT
                act.sa_flags |= SA_INTERRUPT; /* SunOS 4.x */
#endif
        } else {
#ifdef  SA_RESTART
                act.sa_flags |= SA_RESTART;   /* SVR4, 44BSD */
#endif
        }
        if (sigaction(signo, &act, &oact) < 0) return(SIG_ERR);

        else return(oact.sa_handler);
}

char *sock_ntop(const struct sockaddr *sa, socklen_t salen)
/* From Reference 1. */
{
   char portstr[7];
   static char str[128]; /* Unix domain is largest */

   switch (sa->sa_family) {
      case AF_INET: {
         struct sockaddr_in      *sin = (struct sockaddr_in *) sa;
         if (inet_ntop(AF_INET, &sin->sin_addr, str, sizeof(str)) == NULL)
         return(NULL);
         if (ntohs(sin->sin_port) != 0) {
            snprintf(portstr, sizeof(portstr), ".%d", ntohs(sin->sin_port));
            strcat(str, portstr);
         }
         return(str);
      }
/* end sock_ntop */

#ifdef IPV6
      case AF_INET6: {
         struct sockaddr_in6     *sin6 = (struct sockaddr_in6 *) sa;
         if (inet_ntop(AF_INET6, &sin6->sin6_addr, str, sizeof(str)) == NULL)
         return(NULL);
         if (ntohs(sin6->sin6_port) != 0) {
            snprintf(portstr, sizeof(portstr), ".%d", ntohs(sin6->sin6_port));
            strcat(str, portstr);
         }
         return(str);
      }
#endif

#ifdef AF_UNIX
      case AF_UNIX: {
         struct sockaddr_un      *unp = (struct sockaddr_un *) sa;
      /* OK to have no pathname bound to the socket: happens on
         every connect() unless client calls bind() first. */
         if (unp->sun_path[0] == 0)
            strcpy(str, "(no pathname bound)");
         else
            snprintf(str, sizeof(str), "%s", unp->sun_path);
         return(str);
      }
#endif

#ifdef HAVE_SOCKADDR_DL_STRUCT
      case AF_LINK: {
         struct sockaddr_dl      *sdl = (struct sockaddr_dl *) sa;
         if (sdl->sdl_nlen > 0)
            snprintf(str, sizeof(str), "%*s", sdl->sdl_nlen, &sdl->sdl_data[0]);
         else snprintf(str, sizeof(str), "AF_LINK, index=%d", sdl->sdl_index);
         return(str);
      }
#endif
      default:
         snprintf(str, sizeof(str), "sock_ntop: unknown AF_xxx: %d, len %d",
         sa->sa_family, salen);
         return(str);
   }
   return (NULL);
}

int socket1() /* socket ( --- nFD) */
/* Get a TCP socket file descriptor.  Use word sclose to close it. */
{
   return(pushint(
      socket(AF_INET,SOCK_STREAM,0)) /* create a TCP socket */
   );
}

int test_socket_openw(int sock)
/* Test for open socket prior to writing.  Do not run this function if
   there are known readable bytes in sock.  They will be lost. */

/* Adapted from test_socket_open() of GNU Wget and test_socket_open()
   of file term.c.

   See examples in file term.c under: int server1(). */
{
  fd_set check_set;
  struct timeval to;
  int ret=0;

  if(sock<0 || sock>FD_SETSIZE-1) {
    return 0;
  }
/* Select() will return a false indication of closed if there are bytes
   to read.  This test is assumed to be made before running write(), so
   flush the socket before testing it.  Bytes flushed are lost: */
   pushint(sock);
   sflush();
   drop();

/* Check if we still have a valid (non-EOF) connection.  From Andrew
   Maholski's code in the Unix Socket FAQ. */

  FD_ZERO (&check_set);
  FD_SET (sock, &check_set);

  /* Wait one microsecond */
  to.tv_sec = 0;
  to.tv_usec = 1;

/* If we get a timeout, then that means still connected */
  ret=(select (sock + 1, &check_set, NULL, NULL, &to) == 0);

  return(ret);
}

int test_socket_readable() /* socket_readable (nSocket --- nBytes) */
/* Return number of bytes waiting if Socket is readable (may be
   zero).  If error, returns negative. 

   See examples in term.c under: int server1() */
{
   int bytes,ret,sockfd;

   if(tos->typ!=NUM) {
      stkerr(" socket_readable: ",NUMNOT);
      return 0;
   }
   popint(&sockfd);
   ret=readable_timeo(sockfd,0,2,&bytes);

   if(ret<0) return(pushint(ret));
   else return(pushint(bytes));
}

int test_socket_writable() /* socket_writable (nSocket --- f) */
/* Return true flag is Socket is writable. 

   See examples in term.c under: int server1() */
{
   int sockfd;

   if(tos->typ!=NUM) {
      stkerr(" socket_write: ",NUMNOT);
      return 0;
   }
   popint(&sockfd);
   return(pushint(xTRUE*(writable_timeo(sockfd,0,2)>0)));
}

int writable_timeo(int fd, time_t sec, time_t usec)
/* Blocks in select() for up to sec+usec seconds, waiting for socket 
   fd to become writable.
   
   Returns >0 if socket is writable, or 0 if timed out, or <0 if error
   in select(). */
{  
   fd_set wset;
   struct timeval tv;

   if(fd<0 || fd>FD_SETSIZE-1) {
      stkerr(" writable_timeo: ","invalid client socket descriptor");
      return -1;
   }
   FD_ZERO(&wset);
   FD_SET(fd,&wset);
   
   tv.tv_sec=sec;
   tv.tv_usec=usec;

/* Select returns:
      >0 if descriptor is writable
      =0 if timeout
      <0 if select error */
   return(select(fd+1,(fd_set *)NULL,&wset,(fd_set *)NULL,&tv));
}

ssize_t write4n(int fd, unsigned int h)
/* Converts integer h from host byte order to network byte order
   (probably big endian) and writes it to fd. */
{
   int n;
   n=htonl(h);
   return(writen1(fd,&n,sizeof(int)));
}

ssize_t writen(int fd, const void *vptr, size_t n)
/* Write "n" bytes to a descriptor. */
/* Adapted from writen() of Reference 1 */

/* This function is used only in one place, in term.c.  Writen1() has
   more more features and does OPENSSL. */
{
        size_t          nleft;
        ssize_t         nwritten;
        const char      *ptr;

        ptr = vptr;
        nleft = n;

        while (nleft>0 && test_socket_open(fd)) {
           if ( (nwritten = write(fd, ptr, nleft)) <= 0) {
              if (errno == EINTR) {
                 nwritten = 0; /* and call write() again if open */
                 if(!test_socket_open(fd)) return(-1);
              }
              else return(-1);     /* error */
           }
           nleft -= nwritten;
           ptr   += nwritten;
        }
        return(n);
}

ssize_t writen1(int fd, const void *vptr, size_t len)
/* From writen() of Reference 1 (also see writen() in this file), with 
   alarm and SSL added.

   Sets an alarm based upon NET_RATE and len bytes to write.

   Returns -1 if error, probably an interrupt by the alarm or the client
   has closed fd if it is a socket. */
{
   size_t nleft=0;
   ssize_t nwritten=-1;
   const char *ptr;

   Sigfunc *sigfunc;
   int nsec=0,OK=1,test=1;
   const int nsec_min=3; /* minimum alarm seconds */
   time_t time1=0,time2=0;
#if OPENSSL
   int cn;
   SSL *ssl=NULL;
#endif

   time(&time1);

#if OPENSSL
   cn=*(sockCLI+fd);
   if(cn>-1 && (ssl=*(clientSSL+cn)) && (*(contyp+cn)==FOREIGN ||
     *(contyp+cn)==NEWCLI)) {

      nwritten=SSL_write(ssl,vptr,len); 
                                   
      if(SSL_get_error(ssl,nwritten)!=SSL_ERROR_NONE) {
          gprintf(" writen1: error writing to secure socket %d",fd);
          nc();
      }
   }
   else {
#endif

   ptr = vptr;
   nleft = len;
/*
   Use the following to test looping with a buffer of 64 bytes:
      if((nwritten=write(fd, ptr, MIN(nleft,64))) <= 0) {

   When things are good, a 1 byte buffer should work:
      if((nwritten=write(fd, ptr, MIN(nleft,1))) <= 0) {

   But a 1 byte buffer will not work when connecting with CLIENT
   because drainf() expects LOGIN all in one string.  Here is an
   example that needs 18 bytes (the string "LOGIN dale plunger") 
   to send LOGIN in one write, and an 18 byte buffer hardcoded 
   here was verified to work:

      [tops@plunger] ready > "CLIENT" "LOGIN" yank
       stack elements:
             0 string: LOGIN dale plunger  18 characters
       [1] ok!
      [tops@plunger] ready > 
*/
   if(test_socket_openw(fd)) {

   /* Set the following alarm after test_socket_openw() since it calls 
      readn1() which also uses alarm() and then turns it off: */
      ERRALRM=0;
      sigfunc=signal(SIGALRM,connect_alarm); /* save signal handler */
      nsec=MAX(nsec_min,len/NET_RATE);
      if((aset=alarm(nsec))!=0) {
         gprintf(" writen1: using alarm that was already set to %d",\
            aset);
         nc();
      }
      while(OK && nleft>0) {
         if((nwritten=write(fd, ptr, nleft)) <= 0) { 
            if(errno==EINTR) { /* errno 4 */
               nwritten=0; /* and call write() again */
               if(!test_socket_open(fd)) { /* will not reset alarm */
                  OK=0;
                  test=0;
               }
            }
            else {
               if(errno==ECONNRESET) { /* errno 104 */
                  gprintf(
                     " writen1: client reset connection on socket %d",\
                     fd);
               }
               else {
                  gprintf(" writen1: errno %d writing to socket %d",\
                     errno,fd);
               }
               nc();
               OK=0;
            }
         }
         if(ERRALRM) {
            gprintf(\
               " writen1: timeout writing %d bytes to socket %d",\
               nleft,fd);
            nc();
            OK=0;
         }
         if(OK) {
            nleft -= nwritten;
            ptr   += nwritten;
         }
         else nwritten=-1;
      }
      alarm(0);                /* turn off the alarm */
      signal(SIGALRM,sigfunc); /* restore signal handler */
      ERRALRM=0;

      if(!nleft && nwritten!=-1) nwritten=len;
      else nwritten=-1; /* error */
   }
   else test=0;

#if OPENSSL
   }
#endif

   if(nwritten==-1) {
      if(test) {
         time(&time2);
         if((float)(time2-time1)>(float)0.9*nsec) {
            gprintf(" writen1: probable timeout of %d second alarm",
               nsec);
            nc();
         }
      }
      else {
         if(NTRACE) {
            gprintf(" writen1: socket %d is not open, client closed",\
               fd);
            nc();
         }
         pushint(fd);
         sclose(); /* officially close and remove client */
      }
   }
   return(nwritten);
}

#endif /* NET */
