/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author and copyright holder of fftw_driver.c:  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#if defined(FFTW) || defined(FFTW3)
/* fftw_driver.c

   Albert Danial  September 5 2001

   Compute 1-D forward and inverse FFT's by calling FFTW library routines.
   Enable the FFTW pragma for linking with the FFTW version 2 library or
   FFTW3 pragma for linking with the FFTW version 3 library.  (Don't
   enable both pragmas!)
*/

#include <stdio.h>
#include <stdlib.h>

#if defined(FFTW)
#include <rfftw.h>
#include <fftw.h>
#else
#include <fftw3.h>
#endif

#include "fftw_driver.h"
#include "stk.h"
#include "mem.h"

/* real-only forward and inverse transforms: */
int fft()  /*  fft  ( hA --- hFr hFi ) {{{1 */
/* 
 * man entry:  fft {{{2
 * ( hA --- hFr hFi ) 
Computes the forward discrete Fourier transform of the first column in the input matrix.  Returns real and imaginary vectors of the  transform.  Since the input is real-only, the transform will be symmetric.  The original first column in hA can therefore be reconstructed with ifft, the real-only inverse transform.  Uses the FFTW function rfftw_one() and FFTW_ESTIMATE to create the plan.
 * category: math::fft
 * related:  icfft, cfft, ifft
 * 2}}}
 */

{
    int           nPts, nCol, i, half_nPts;
    double       *A, *Fr, *Fi;
#if defined(FFTW)
    fftw_real    *in, *out;  /* to be dynamically allocated */
    rfftw_plan    p;
#else
    double       *in, *out;  /* to be dynamically allocated */
    fftw_plan     p;
#endif

    if (tos->typ != MAT) {
        stkerr(" fft: ", MATNOT);
        return 0;
    }
    A    = tos->mat;
    nPts = tos->row;
    if(!nPts) {
       stkerr(" fft: ", "input matrix has no rows");
       return 0;
    }
    if (nPts % 2) {
        stkerr(" fft : Real fft must have even sized input. ",STKNOT);
        return 0;
    }
    nCol = tos->col;
    
#if defined(FFTW)
    in   = (fftw_real *) malloc(nPts * sizeof(fftw_real));
    out  = (fftw_real *) malloc(nPts * sizeof(fftw_real));
#else
    in   =  fftw_malloc(        nPts * sizeof(double));
    out  =  fftw_malloc(        nPts * sizeof(double));
#endif
    if (in  == NULL) {
        stkerr(" fft (in array): ",MEMNOT);
        return 0;
    }
    if (out == NULL) {
        stkerr(" fft (out array): ",MEMNOT);
        return 0;
    }
 
    /* populate the FFTW input structure */
    for (i = 0; i < nPts; i++) {
        in[i] = A[i];
    }
    drop(); /* dropping A from the stack */

#if defined(FFTW)
    /* make a plan */
    p = rfftw_create_plan(nPts, FFTW_FORWARD,  FFTW_ESTIMATE);

    /* compute the transform */
    rfftw_one(p, in, out);
#else
    /* make a plan */
    p = fftw_plan_r2r_1d(nPts, in, out, FFTW_FORWARD,  FFTW_ESTIMATE);

    /* compute the transform */
    fftw_execute(p);
#endif

    /* make arrays to contain the Re,Im parts of the transform */
    if(!matstk(nPts, 1, "_Fr")) return 0;
    Fr = tos->mat;
    if(!matstk(nPts, 1, "_Fi")) return 0;
    Fi = tos->mat;

    /* populate tops arrays with values from FFTW output structure */
    half_nPts =  (nPts+1)/2;
    for (i = 0;   i < nPts;     i++)
         out[i] /= (double) nPts;
    /* first half */
    for (i = 0;   i <= half_nPts;   i++) {
         Fr[i] = out[i];
         Fi[i] = out[nPts - i];
         if (i == half_nPts) Fi[i] = 0.0;
    }
    /* second half is mirror of first half */
    for (i = half_nPts + 1;  i <  nPts;  i++) {
         Fr[i] =  out[nPts - i];
         Fi[i] = -out[i];
    }
    Fi[0] = 0.0;

#if defined(FFTW)
    rfftw_destroy_plan(p);
    free(out);
    free(in);
#else
    fftw_destroy_plan(p);
    fftw_free(out);
    fftw_free(in);
#endif

    return 1;
} /* 1}}} */
int ifft() /* ifft ( hFr hFi --- hAr ) {{{1 */
/*
 * man entry:  ifft {{{2
 * ( hFr hFi --- hAr ) Computes the inverse discrete Fourier transform using the real and imaginary vectors on the stack.  These inputs are presumed to be symmetric, that is, they are the transform of a real-only signal.  Only the first N/2 terms are used from hFr and hFi, and from them a real-only inverse transform is constructed.  Uses the FFTW function rfftw_one() and FFTW_ESTIMATE to create the plan.
 * category: math::fft
 * related:  cfft, icfft, fft
 * 2}}}
 */
{
    int           nPts, i, half_nPts;
    double       *Ar, *Fr, *Fi;
#if defined(FFTW)
    fftw_real    *in, *out;  /* to be dynamically allocated */
    rfftw_plan    p;
#else
    double       *in, *out;  /* to be dynamically allocated */
    fftw_plan     p;
#endif

    if ((tos->typ != MAT) || ((tos-1)->typ != MAT)) {
        stkerr(" ifft: ", MATNOT2);
        return 0;
    }
    if (tos->row != (tos-1)->row) {
        stkerr(" ifft: ", MATSNOTC);
        return 0;
    }
    Fr = (tos-1)->mat;
    Fi = (tos  )->mat;

    nPts = tos->row;
    if (nPts % 2) {
        stkerr(" ifft : real ifft must have even sized input ",STKNOT);
        return 0;
    }
    
#if defined(FFTW)
    in   = (fftw_real *) malloc(nPts * sizeof(fftw_real));
    out  = (fftw_real *) malloc(nPts * sizeof(fftw_real));
#else
    in   =  fftw_malloc(        nPts * sizeof(double));
    out  =  fftw_malloc(        nPts * sizeof(double));
#endif
    if (in  == NULL) {
        stkerr(" ifft (in array): ",MEMNOT);
        return 0;
    }
    if (out == NULL) {
        stkerr(" ifft (out array): ",MEMNOT);
        return 0;
    }
 
    /* populate the FFTW input structure */
    half_nPts =  (nPts+1)/2;
    for (i = 0; i < half_nPts; i++) {
        in[i] =  Fr[i];
        if (i) in[nPts - i] = Fi[i];
    }
    in[half_nPts] = Fr[half_nPts];

    drop(); /* dropping Fi from the stack */
    drop(); /* dropping Fr from the stack */

#if defined(FFTW)
    /* make a plan */
    p = rfftw_create_plan(nPts, FFTW_BACKWARD, FFTW_ESTIMATE);

    /* compute the transform */
    rfftw_one(p, in, out);
#else
    /* make a plan */
    p = fftw_plan_r2r_1d(nPts, in, out, FFTW_BACKWARD,  FFTW_ESTIMATE);

    /* compute the transform */
    fftw_execute(p);
#endif

    /* make arrays to contain the Re,Im parts of the inverse transform */
    if(!matstk(nPts, 1, "_Ar")) return 0;
    Ar = tos->mat;

    /* populate tops array with values from FFTW output structure */
    for (i = 0; i < nPts; i++) {
        Ar[i] = out[i];
    }

#if defined(FFTW)
    rfftw_destroy_plan(p);
    free(out);
    free(in);
#else
    fftw_destroy_plan(p);
    fftw_free(out);
    fftw_free(in);
#endif

    return 1;
} /* 1}}} */

/* complex forward and inverse transforms: */
int cfft()   /*  cfft   ( hAr hAi --- hFr hFi ) {{{1 */
/* 
 * man entry:  cfft {{{2
 * ( hAr hAi --- hFr hFi ) 
Computes the forward discrete Fourier transform from the real and imaginary vectors on the stack.  Returns real and imaginary vectors of the transform.  If the inputs are symmetric, that is, they come from a real-only signal, then the imaginary component of the output will be all zeros.  Uses FFTW function fftw_one() and FFTW_ESTIMATE to create the plan.
 * category: math::fft
 * related:  ifft, fft, icfft
 * 2}}}
 */

{
    int           nPts, nCol, i;
    double       *Ar, *Ai, *Fr, *Fi;
    fftw_complex *in, *out;  /* to be dynamically allocated */
    fftw_plan     p;

    if (tos->typ != MAT) {
        stkerr(" cfft: Imaginary part ", MATNOT);
        return 0;
    }
    if ((tos-1)->typ != MAT) {
        stkerr(" cfft: Real part ", MATNOT);
        return 0;
    }
    Ar   = (tos-1)->mat;
    Ai   = (tos  )->mat;
    nPts = tos->row;
    nCol = tos->col;
    
#if defined(FFTW)
    in   = (fftw_complex *) malloc(nPts * sizeof(fftw_complex));
#else
    in   =  fftw_malloc(           nPts * sizeof(fftw_complex));
#endif
    if (in  == NULL) {
        stkerr(" cfft (in array): ",MEMNOT);
        return 0;
    }
#if defined(FFTW)
    out  = (fftw_complex *) malloc(nPts * sizeof(fftw_complex));
#else
    out  =  fftw_malloc(           nPts * sizeof(fftw_complex));
#endif
    if (out == NULL) {
        stkerr(" cfft (out array): ",MEMNOT);
        return 0;
    }
 
    /* populate the FFTW input structure */
    for (i = 0; i < nPts; i++) {
#if defined(FFTW)
        in[i].re = Ar[i];
        in[i].im = Ai[i];
#else
        in[i][0] = Ar[i];
        in[i][1] = Ai[i];
#endif
    }
    drop(); /* dropping Ai from the stack */
    drop(); /* dropping Ar from the stack */

#if defined(FFTW)
    /* make a plan */
    p = fftw_create_plan(nPts, FFTW_FORWARD,  FFTW_ESTIMATE);

    /* compute the transform */
    fftw_one(p, in, out);
#else
    /* make a plan */
    p = fftw_plan_dft_1d(nPts, in, out, FFTW_FORWARD,  FFTW_ESTIMATE);

    /* compute the transform */
    fftw_execute(p);
#endif

    /* make arrays to contain the Re,Im parts of the transform */
    if(!matstk(nPts, 1, "_Fr")) return 0;
    Fr = tos->mat;
    if(!matstk(nPts, 1, "_Fi")) return 0;
    Fi = tos->mat;

    /* populate tops arrays with values from FFTW output structure */
    for (i = 0; i < nPts; i++) {
#if defined(FFTW)
        Fr[i] = out[i].re / (double) nPts;
        Fi[i] = out[i].im / (double) nPts;
#else
        Fr[i] = out[i][0] / (double) nPts;
        Fi[i] = out[i][1] / (double) nPts;
#endif
    }

    fftw_destroy_plan(p);
#if defined(FFTW)
    free(out);
    free(in);
#else
    fftw_free(out);
    fftw_free(in);
#endif

    return 1;
} /* 1}}} */
int icfft()  /* icfft  ( hFr hFi --- hAr hAi ) {{{1 */
/*
 * man entry:  icfft {{{2
 * ( hFr hFi --- hAr hAi ) Computes the inverse discrete Fourier transform using the real and imaginary vectors on the stack.  If hFr and hFi are transforms of a real-only signal, then the imaginary output vector hAi should contain only machine zero terms.  Uses FFTW function fftw_one() and FFTW_ESTIMATE to create the plan.
 * category: math::fft
 * related:  fft, ifft, cfft
 * 2}}}
 */
{
    int           nPts, i;
    double       *Ar, *Ai, *Fr, *Fi;
    fftw_complex *in, *out;  /* to be dynamically allocated */
    fftw_plan     p;

    if ((tos->typ != MAT) || ((tos-1)->typ != MAT)) {
        stkerr(" icfft: ", MATNOT2);
        return 0;
    }
    if (tos->row != (tos-1)->row) {
        stkerr(" icfft: ", MATSNOTC);
        return 0;
    }
    Fr = (tos-1)->mat;
    Fi = (tos  )->mat;

    nPts = tos->row;
    
#if defined(FFTW)
    in   = (fftw_complex *) malloc(nPts * sizeof(fftw_complex));
    out  = (fftw_complex *) malloc(nPts * sizeof(fftw_complex));
#else
    in   =  fftw_malloc(           nPts * sizeof(fftw_complex));
    out  =  fftw_malloc(           nPts * sizeof(fftw_complex));
#endif
    if (in  == NULL) {
        stkerr(" icfft (in array): ",MEMNOT);
        return 0;
    }
    if (out == NULL) {
        stkerr(" icfft (out array): ",MEMNOT);
        return 0;
    }
 
    /* populate the FFTW input structure */
    for (i = 0; i < nPts; i++) {
#if defined(FFTW)
        in[i].re = Fr[i];
        in[i].im = Fi[i];
#else
        in[i][0] = Fr[i];
        in[i][1] = Fi[i];
#endif
    }
    drop(); /* dropping Fi from the stack */
    drop(); /* dropping Fr from the stack */

#if defined(FFTW)
    /* make a plan */
    p = fftw_create_plan(nPts, FFTW_BACKWARD, FFTW_ESTIMATE);

    /* compute the transform */
    fftw_one(p, in, out);
#else
    /* make a plan */
    p = fftw_plan_dft_1d(nPts, in, out, FFTW_BACKWARD,  FFTW_ESTIMATE);

    /* compute the transform */
    fftw_execute(p);
#endif

    /* make arrays to contain the Re,Im parts of the inverse transform */
    if(!matstk(nPts, 1, "_Ar")) return 0;
    Ar = tos->mat;
    if(!matstk(nPts, 1, "_Ai")) return 0;
    Ai = tos->mat;

    /* populate tops arrays with values from FFTW output structure */
    for (i = 0; i < nPts; i++) {
#if defined(FFTW)
        Ar[i] = out[i].re;
        Ai[i] = out[i].im;
#else
        Ar[i] = out[i][0];
        Ai[i] = out[i][1];
#endif
    }

    fftw_destroy_plan(p);
#if defined(FFTW)
    free(out);
    free(in);
#else
    fftw_free(out);
    fftw_free(in);
#endif

    return 1;
} /* 1}}} */

#endif
