package TopsFEA;
require Exporter;
use YAML qw( LoadFile );
use DBI;
use DBD::SQLite 1.00;
use Data::Dumper;
use strict;
our @ISA       = qw(Exporter);

our $SQLITE            = "sqlite3"; # SQLite command line tool
our $TRANSACTION_SIZE  =   10_000;  # number of insert's per db commit
our %nNodes_in_element = (
                          'beam2' => 2,
                          'tri3'  => 3,
                          'quad4' => 4,
                         );
our @EXPORT_OK = qw(
                    load_csv_file  write_csv_file sql_insert
                    build_empty_db dump_db        file_extension
                    compute_CSTM
                    $SQLITE $TRANSACTION_SIZE %nNodes_in_element
                   );
use vars qw ( $SQLITE $TRANSACTION_SIZE %nNodes_in_element );
my  $EPSILON      = 1.0E-15;
my  $SQRT_EPSILON = sqrt($EPSILON);

sub load_csv_file { # {{{1
    my (
        $csv_file       ,  # in
        $rs_schema_file ,  # out (name of csv schema filename embedded in .csv file)
        $rhhah_csv_data ,  # out
       ) = @_;

    my $schema;
    my $E_N_seq = 1;  # seq_no for element_node table
    open  IN, $csv_file or die "Cannot read $csv_file $!\n";
    while (<IN>) {
        next if /^\s*#/ or /^\s*$/;
        chomp;
        my @entry = split(",");
        map { s/^\s+//; s/\s+$//; } @entry;  # remove leading, trailing ws
        map { s/^\[.*?\]//      ; } @entry;  # remove embedded [..] comments
        # print "=>@entry\n";
        my $table = shift @entry;
        if ($table eq "schema") {
            ${$rs_schema_file} = $entry[0];
            # print "found schema file [${$rs_schema_file}]\n";
            if (-r ${$rs_schema_file}) {
                $schema = LoadFile(${$rs_schema_file}) 
            } else {
                die "Line $. of $csv_file defines a CSV schema file" .
                    " '${$rs_schema_file}'\nbut this file cannot be read.\n";
            }
            next;
        }
        die "Table '$table' is not defined in schema file ${$rs_schema_file}\n"
            unless defined %{$schema->{$table}};
        my $id = "";
        foreach my $row (@{$schema->{$table}{rows}}) {
#print "r=$row\n" if $table eq "tri3";
            # $row is a string of three comma separated terms: 
            #       column name, database type, description
            # eg:  'id          , A,   Node identifier.'
            $row =~ m/^\s*(\w+)\s+/;
            my $field_name = $1;
            if (!$field_name) {
                print Dumper($schema->{$table});
                die "Unable to extract column name for table $table" 
                    unless $field_name;
            }
            if ($field_name eq "id") {
                $id = shift @entry;
                next;
            }
            die "Table '$table' no id line $. of file $csv_file\n" unless $id;
#print "E=@entry\n" if $table eq "tri3";
            if (@entry) {
# print "Table '$table' line $. assigning $field_name -> $entry[0]\n";
#               if ($nNodes_in_element{$table}) {
#                   my $nNodes = $nNodes_in_element{$table};
#                   my @nodes  = @entry[($#entry-$nNodes+1)..$#entry];
#                   foreach (@nodes) {
#                       $rhhah_csv_data->{'element_node'}{$E_N_seq}{$id} = $_;
#                       ++$E_N_seq;
#                       pop @entry;
#                   }
#               } 
                push @{$rhhah_csv_data->{$table}{$id}}, 
                            { $field_name, shift @entry };
            } else {
                warn "Table '$table' no data for $field_name (line $.), " .
                     "will be null\n";
            }
        }
    }
    close IN;
}
# 1}}}
sub sql_insert { # {{{1
    #
    # Input is a single line of model data in comma separated value format.
    #
    # Output is one or more SQL insert statements that put the line of model
    # data in the database.
    #
    my (
        $csv_input    ,  # in      line of csv text with model data
        $rh_schema    ,  # in      schema definition hash
        $rs_nInsert   ,  # in/out  number of inserts this transaction
        $rs_E_seq     ,  # in/out  seq_no for element table
        $rs_E_N_seq   ,  # in/out  seq_no for element_node table
        $verbose      ,  # in
       ) = @_;

#die Dumper(\@{$rh_schema->{material}{rows}});

    $csv_input .= " "; # needed for lines that end with ,
                       # Without the comma @entry will miss last null term.
    my @entry = split(",", $csv_input);
    map { s/\[.*?\]//       ; } @entry;  # remove embedded [..] comments
    map { s/^\s+//; s/\s+$//; } @entry;  # remove leading, trailing ws
#print "sql_insert => {", join("}{", @entry), "}\n";
    my $table = shift @entry;

    die "Table '$table' is not defined in database schema\n"
        unless defined %{$rh_schema->{$table}};

    my $id     = shift @entry;
    my @insert = ();
    my $SQL    = "insert into $table values ( NULL," .   # seq_no
                                            "'$id', "    # id
        unless $nNodes_in_element{$table};
    # A NULL inserted into a field which is an integer primary key
    # has the effect of autonumbering the field from 1..N

    if ($nNodes_in_element{$table}) { # this is an element table
        # $SQL contains the insert into the element type table, ie, 'tri3'
        $SQL = "insert into $table values ( '${$rs_E_seq}', ";
        push @insert, "insert into element (" .
                         "'seq_no', 'id', 'type') values (" .
                         "${$rs_E_seq}, '$id', '$table');\n";
        # An element's nodes go in the element_node table, not in
        # the table for the element itself.  As element_node needs
        # the canonical node IDs and these can't be known until the
        # node table is completely populated, have to put the data
        # in a temporary tmp_element_node table and create the real
        # element_node table as a postprocessing step.
        my $pid    = shift @entry;
        $SQL      .= " '$pid');\n";
        foreach (@entry) { # the remaining entries are node ID's
            push @insert, "insert into tmp_element_node (" .
                             "'seq_no', 'eid', 'nid') values (" .
                             "${$rs_E_N_seq}, ${$rs_E_seq}, '$_');\n";
#XX         push @insert, "insert into tmp_element_node (" .
#XX                          "'eid', 'nid') values (${$rs_E_seq}, '$_');\n";
            ++${$rs_E_N_seq};
            ++${$rs_nInsert};
        }
        ++${$rs_E_seq};  # canonical element ID; starts at zero

    } else { # not an element table; insert everything from the csv here

        my $nFields = scalar @{$rh_schema->{$table}{rows}};
#warn "$table has $nFields fields\n";
#warn "map $table -> [" . join("][", @entry), "]\n";
        for (my $i = 2; $i < $nFields; $i++) {
            next unless defined $entry[$i-2];
            my ($field, $type, $description) = 
                split_row_field_type_desc( $rh_schema->{$table}{rows}[$i] );
            if ($entry[$i-2] =~ /^\s*$/) {
                $entry[$i-2] = "NULL";
            } elsif ($type =~ /^U?A/) {   # text entries must be quoted
                $entry[$i-2] = "'$entry[$i-2]'";
            } elsif ($type =~ /^U?R/) {
                # real numbers are only recognized by SQLite if they
                # have leading and trailing decimal values.
                #   .XX  must be changed to 0.XX
                #   XX   must be changed to XX.0
                $entry[$i-2] = "0" .  $entry[$i-2] if $entry[$i-2] =~ /^\./;
                $entry[$i-2] .= ".0" if $entry[$i-2] !~ /\./;
            }
#print "T=$table  entry=$entry[$i-2] : $field/$type/$description\n" if $verbose;
        }
        map { s/^''$/NULL/; }   @entry;  # convert null strings to NULL
        $SQL .= join(",", @entry);
        if ($table eq "node") { # node table:  put placeholder NULL's in for
                                # coordinates in the basic coordinate system
            $SQL .= ", NULL, NULL, NULL";
        }
        $SQL .= ");\n";
    }
    print "SQL=[$SQL]\n" if $verbose;
    push @insert, $SQL;

    ++${$rs_nInsert};

    return @insert;

}
# 1}}}
sub build_empty_db { # {{{1
    my ($schema,      # in 
        $schema_file, # in 
        $verbose,     # in 
       ) = @_;
    #
    # Return an array of SQL statements used to create the database.
    #

    my @SQL = ("BEGIN TRANSACTION;\n");
    push @SQL, "create table seq_no_cleanup(t_name text);\n";
    # Table 'seq_no_cleanup' contains names of tables which have a
    # field 'seq_no' containing a bogus -1 entry that must be deleted
    # before any real SQL operations are done on the database.  The
    # 'seq_no_cleanup' table itself can be dropped after this cleanup.
    my $constraint_ID = 0;
    foreach my $table (sort keys %{$schema}) {
        my @F   = ();
        my $first_insert = "insert into $table values(";
        my $have_seq_no  = 0;
        # first_insert puts a -1 into tables which have an integer primary
        # key field called seq_no so that subsequent NULL inserts into
        # seq_no begin at zero.  These null rows must be deleted before
        # the database is used for real.
        foreach my $row (@{$schema->{$table}{rows}}) {
            my ($field, $type, $description) = split_row_field_type_desc($row);
            # SQLite is typeless but does distinguish between numbers
            # (stored as doubles) and text (stored as text).
            #    P - integer primary key      (must be unique)
            #    I - integer
            #    R - float
            #    A - text
            #   UI - unique integer
            #   UR - unique float
            #   UA - unique text
            #   x/a(b) - x is a foreign key that references table a, field b
            my $unique = "";
            my $FK     = "";
            if  ($type =~ m{^(\w+)\s*/\s*(\w+)\s*\(\s*(\w+)\s*\)$}) {
                # format is x/a(b) meaning x is foreign key to table a, field b
                $type = $1;
                my $fk_table = $2;
                my $fk_field = $3;
                (my $this_field = $field) =~ s/\s+$//;
                $FK   = sprintf " -- ,constraint R_%03d foreign key (%s) " .
                                "references %s(%s)",
                                ++$constraint_ID,
                                $this_field ,
                                $fk_table,
                                $fk_field;

            }
            if      (uc $type =~ /^U(\w)$/) {          # unique
                $unique = "unique";
                $type   = $1;
            }
            if      (uc $type =~ /^P\b/) {           # integer primary key
                push @F, "$field integer primary key $FK";
            } elsif (uc $type =~ /^I\b/) {           # integer
                push @F, "$field integer $unique $FK";
            } elsif (uc $type =~ /^R\b/) {           # real
                push @F, "$field float   $unique $FK";
            } else {                                 # text
                push @F, "$field text    $unique $FK";
            }
            if ($field =~ /seq_no\s*$/ and $type =~ /^P\b/) {
                $first_insert .= "-1,";
                $have_seq_no   = 1;
            } else {
                $first_insert .= "NULL,";
            }
            printf "%-16s %-16s %s %s\n", 
                $table, $field, $type, join(", ", @F) if $verbose;
        }
        $first_insert =~ s/,\s*$/);\n/;

        my $create = "create table $table ( -- {{{\n\t " .
                      join("\n\t,", @F) .
                      "\n\t); -- }}}\n";

        push @SQL, $create;
        if ($have_seq_no) {
            push @SQL, $first_insert;
            push @SQL, "insert into seq_no_cleanup values('$table');\n";
        }
        print $create if $verbose;
        if (($table eq "node"      ) or 
            ($table eq "nodal_load")) {
            # dropped after coordinate xforms
            my $create_tmp = "create table tmp_$table ( -- {{{\n\t" . 
                             join("\n\t,", @F) . " ); -- }}}\n";
            print $create_tmp if $verbose;
            push @SQL, $create_tmp;
        }

    }
    push @SQL, "COMMIT;\n";
    return @SQL;
} # 1}}}
sub split_row_field_type_desc { # {{{1
    my ($row, ) = @_;
    my ($field, $type, $description) = split(',', $row, 3);
    $type =~ s/^\s+//; $type =~ s/\s+$//;
    return ($field, $type, $description);
} # 1}}}
sub write_csv_file { # {{{1
    my (
        $root_name       ,
        $schema_file     ,
        $rhhah_model_data,
        $verbose         ,
       ) = @_;

    my $fea_file = $root_name . ".csv";

    my ($schema, );

    open  OUT, ">$fea_file" or die "Cannot write $fea_file $!\n";
    print OUT "schema, $schema_file\n";

    foreach my $table (keys %{$rhhah_model_data}) {
#my $seq_no = 0;
        foreach my $id (keys %{$rhhah_model_data->{$table}}) {
#++$seq_no;
            my @csv    = ($table, $id, );
            my @keys   = ( "'id'" );
            my @values = ("'$id'" );
#print "T=$table id=$id\n";
            foreach my $row (@{$rhhah_model_data->{$table}{$id}}) {
                foreach my $col (keys %{$row}) {
                    push @keys  , "'$col'"        ;
                    push @values, "'$row->{$col}'";
                    push @csv   ,   $row->{$col}  ;
                }
            }
            print OUT join(",", @csv), "\n";
        }
    }

    close(OUT);
    print "Wrote $fea_file\n";
}
# 1}}}
sub dump_db { # {{{1
    #
    # Requires a well defined schema table in the database.
    #
    my (
        $db_file,         # in  (deletes the file if it exists)
        $schema ,         # in  1=print schema and exit
        $headers,         # in  1=print headers and exit
        $table  ,         # in  0=all tables, otherwise the name of a table to dump
        $id     ,         # in  0=all rows  , otherwise the row with ID = $id
       ) = @_;

    my $dbh = DBI->connect("dbi:SQLite:dbname=$db_file","","");
    my ($rc, $st, $create_table, $insert, );

    my $select = "select * from schema";
    my $hash_ref = $dbh->selectall_hashref($select, 'name');
    die "Empty schema table in $db_file; cannot dump.\n" unless defined %{$hash_ref};

    if ($schema) {
        printf "Schema file : %s\n", $hash_ref->{schema}{value};
        system("$SQLITE $db_file '.schema'");
        return;
    }

    if ($headers) {
        printf "Schema file : %s\n", $hash_ref->{schema}{value};
        foreach my $table (sort keys %{$hash_ref}) {
            # skip the schema file entry, and tables which are empty
            next if $table eq "schema" or !$hash_ref->{$table}{value};
            printf "%-12s: %5d %s\n", $table, 
                                      $hash_ref->{$table}{value},
                                      $hash_ref->{$table}{desc};
        }
        $rc = $dbh->disconnect;
        return;
    }

    foreach my $T (sort keys %{$hash_ref}) {
        # skip the schema file entry, and tables which are empty
        next if $T eq "element_node";
        next if $T eq "schema"; # or !$hash_ref->{$T}{value};
        next if $table and $table ne $T;
        $select = "select * from $T order by id";
        my $ary_ref  = $dbh->selectall_arrayref($select);
        foreach my $row (@{$ary_ref}) {
            # for elements, ${$row}[0] is the element ID
            next if $id and $id ne ${$row}[0];
            print $T;
            foreach my $col (@{$row}) {
                $col = "(null)" unless defined $col;
                printf " %-12s", $col;
            }
            if ($nNodes_in_element{$T}) {
                # If this is an element table, need to look up the nodes
                # for it by querying the element_node table.
                $select = "select * from element_node " . 
                          "where eid='${$row}[0]' order by seq_no";
                my $elem_node_ref  = $dbh->selectall_arrayref($select);
                # @elem_node_ref = [[seq_no, elem id, node id], # node 1
                #                     ...
                #                   [seq_no, elem id, node id]] # node n
                foreach my $e_n (@{$elem_node_ref}) {
                    printf " %-12s", $e_n->[2];
                }
            }
            print "\n";
        }
    }

    $rc = $dbh->disconnect;
} # 1}}}
sub file_extension { # {{{1
    my ($file, ) = @_;
    my $extension = "";
    my $root_name = "";
    if ($file =~ /^(.*?)\.(\w+)$/) {
        $root_name = $1;
        $extension = $2;
    }
    return ($extension, $root_name);
} # 1}}}
sub compute_CSTM { # {{{1
    #
    # Compute the coordinate system transformation matrices.
    #
    # Return an array of SQL statements which insert the CSTM into the 
    # coord_xfer table of the database.
    #
    my ($dbh    ,     # in handle to database
        $verbose,     # in 
       ) = @_;

    my $select      = "select * from coord";
    my $coord_table = $dbh->selectall_hashref($select, 'id');
    #
    # coord_table->{id}{seq_no}
    #                  {id_relative}
    #                  {type}        1=rect     2=cyl     3=sph
    #                  {id_A}        if null, use A1,A2,A3
    #                  {A1}
    #                  {A2}
    #                  {A3}
    #                    :   repeat id_A,A1,A2,A3 with B and C
    #
     
    my %known_coords = ( '0' => 1 ); # basic coordinate system, ID=0, is known

    my $n_unresolved = scalar keys %{$coord_table}; # n coord systems to figure out
    my $nIter        = 0;
    my %local_cstm   = (); # local_cstm{id} = [][] rotation matrix
    while (1) {  # compute local CSTM's going from ID_relative to ID {{{2
        ++$nIter;
        my $id_to_resolve = 0;
        foreach my $id (sort {$a <=> $b} keys %{$coord_table}) {
            next if $known_coords{$id};
            $id_to_resolve = $id;
            last;
        }
        last if !$id_to_resolve; # nothing left to figure out
        die "compute_CSTM:  too many iterations ($nIter)\n"
            if $nIter > (scalar keys %{$coord_table})**2;

        my (@A, @B, @C); # Coordinates of points that define the system;
                         # same definition as A,B,C in Nastran CORD2R entry.
        if ($coord_table->{$id_to_resolve}{id_A}) {
            die "compute_CSTM: no code to use nodal ID as orientation vector A";
        } else {
            $A[0] = $coord_table->{$id_to_resolve}{A1};
            $A[1] = $coord_table->{$id_to_resolve}{A2};
            $A[2] = $coord_table->{$id_to_resolve}{A3};
        }
        if ($coord_table->{$id_to_resolve}{id_B}) {
            die "compute_CSTM: no code to use nodal ID as orientation vector B";
        } else {
            $B[0] = $coord_table->{$id_to_resolve}{B1};
            $B[1] = $coord_table->{$id_to_resolve}{B2};
            $B[2] = $coord_table->{$id_to_resolve}{B3};
        }
        if ($coord_table->{$id_to_resolve}{id_C}) {
            die "compute_CSTM: no code to use nodal ID as orientation vector C";
        } else {
            $C[0] = $coord_table->{$id_to_resolve}{C1};
            $C[1] = $coord_table->{$id_to_resolve}{C2};
            $C[2] = $coord_table->{$id_to_resolve}{C3};
        }

#print "A, B, C coordinates for system $id_to_resolve\n";
#print "A: "; for (0..2) { printf "% 15.9e ", $A[$_]; } print "\n";
#print "B: "; for (0..2) { printf "% 15.9e ", $B[$_]; } print "\n";
#print "C: "; for (0..2) { printf "% 15.9e ", $C[$_]; } print "\n";

        my @AB     = Unit( Sub( @B, @A ) );
        my @AC     = Unit( Sub( @C, @A ) );
#print "AB:"; for (0..2) { printf "% 15.9e ", $AB[$_]; } print "\n";
#print "AC:"; for (0..2) { printf "% 15.9e ", $AC[$_]; } print "\n";
        die "compute_CSTM: ID $id_to_resolve A, B, C are colinear"
            if abs(Dot(@AB, @AC)) >= (1.0 - $SQRT_EPSILON);

        my @unit_Z = @AB ;
        my @unit_Y = Unit( Cross ( @AB, @AC ) );
#print "Uy:"; for (0..2) { printf "% 15.9e ", $unit_Y[$_]; } print "\n";
        my @unit_X = Cross ( @unit_Y, @unit_Z );

#print "Unit vectors for system $id_to_resolve\n";
#print "X: "; for (0..2) { printf "% 15.9e ", $unit_X[$_]; } print "\n";
#print "Y: "; for (0..2) { printf "% 15.9e ", $unit_Y[$_]; } print "\n";
#print "Z: "; for (0..2) { printf "% 15.9e ", $unit_Z[$_]; } print "\n";

        my $len_X_xy = sqrt($unit_X[0]**2 + $unit_X[1]**2);
        my $len_Y_xy = sqrt($unit_Y[0]**2 + $unit_Y[1]**2);

        my ($cos_psi, $sin_psi, $cos_theta, $sin_theta, $cos_phi, $sin_phi);

        if ($unit_Z[2] > $SQRT_EPSILON) {
            $cos_psi = $unit_X[0] / $len_X_xy;
            $sin_psi = $unit_X[1] / $len_X_xy;
        } else {  # psi is zero
            $cos_psi = 1;
            $sin_psi = 0;
        }

        $cos_theta =  $len_X_xy;
        $sin_theta = -$unit_X[2];

        if ($len_Y_xy > $SQRT_EPSILON) {
            $cos_phi = $len_Y_xy;
            $sin_phi = $unit_Y[2];
        } else {  # phi is 90 degrees because unit_Y is parallel to Z_basis
            $cos_phi = 0;
            $sin_phi = 1;
        }
#printf "cos_psi   = % 15.9e    sin_psi   = % 15.9e\n", $cos_psi,   $sin_psi  ;
#printf "cos_theta = % 15.9e    sin_theta = % 15.9e\n", $cos_theta, $sin_theta;
#printf "cos_phi   = % 15.9e    sin_phi   = % 15.9e\n", $cos_phi,   $sin_phi  ;

        @{$local_cstm{$id_to_resolve}} = rot_matrix($cos_psi  , $sin_psi  ,
                                                    $cos_theta, $sin_theta,
                                                    $cos_phi  , $sin_phi  );
        $known_coords{$id_to_resolve}  = 1;
#printf "$id_to_resolve\n"; print_3x3($local_cstm{$id_to_resolve});
        --$n_unresolved;
    }
    # 2}}}

#print "\$coord_table = ", Dumper($coord_table);
#print "\%local_cstm= ", Dumper(\%local_cstm);

    my %global_cstm = ();
    my @SQL         = ();
    foreach my $id (sort {$a <=> $b} keys %{$coord_table}) {
        # Recursively descend the tree of chained coordinate systems
        # multiplying local CSTM's together to arrive at the global
        # CSTM that ties back to the basic coordinate frame.
        my @coord_chain = ();
        coord_sys_chain($id, $coord_table, \@coord_chain);
#print "Chain $id: @coord_chain\n";
        my @cstm_trans = ();
        my @cstm_rot   = ();
        global_cstm( $id, \%local_cstm, $coord_table, \@coord_chain, 
                    \@cstm_trans, 
                    \@cstm_rot);
        my $seq_no = $coord_table->{$id}{seq_no};
        @{$global_cstm{$seq_no}{trans}} = @cstm_trans;
        @{$global_cstm{$seq_no}{rot}}   = @cstm_rot;
#print "- " x 20, "\n";
        my $insert = "insert into coord_xfer values ( '$seq_no', " 
                   . "'$cstm_trans[0]','$cstm_trans[1]','$cstm_trans[2]', " 
                   . "'$cstm_rot[0][0]','$cstm_rot[1][0]','$cstm_rot[2][0]', " 
                   . "'$cstm_rot[0][1]','$cstm_rot[1][1]','$cstm_rot[2][1]', " 
                   . "'$cstm_rot[0][2]','$cstm_rot[1][2]','$cstm_rot[2][2]');" 
                   . "\n" ;
        push @SQL, $insert;
        print "$insert\n" if $verbose;
    }

    return @SQL;

} # 1}}}
sub Cross {  #  {{{1
    my ($Ax, $Ay, $Az, $Bx, $By, $Bz) = @_;

    my $Cx = $Ay * $Bz  -  $By * $Az;
    my $Cy = $Bx * $Az  -  $Ax * $Bz;
    my $Cz = $Ax * $By  -  $Bx * $Ay;

    return ($Cx, $Cy, $Cz);
} #  1}}}
sub Dot {  #  {{{1
    my ($Ax, $Ay, $Az, $Bx, $By, $Bz) = @_;

    return $Ax*$Bx  +  $Ay*$By  +  $Az*$Bz;
} #  1}}}
sub Sub {  #  {{{1
    my ($Ax, $Ay, $Az, $Bx, $By, $Bz) = @_;
    #  Returns   A - B
    my $Cx = $Ax - $Bx;
    my $Cy = $Ay - $By;
    my $Cz = $Az - $Bz;

    return ($Cx, $Cy, $Cz);
} #  1}}}
sub Add {  #  {{{1
    my ($Ax, $Ay, $Az, $Bx, $By, $Bz) = @_;
    #  Returns   A + B
    my $Cx = $Ax + $Bx;
    my $Cy = $Ay + $By;
    my $Cz = $Az + $Bz;

    return ($Cx, $Cy, $Cz);
} #  1}}}
sub Mag {  #  {{{1
    my ($Ax, $Ay, $Az) = @_;
    # Returns the length of vector A
    #
    return sqrt( $Ax**2  +  $Ay**2  +  $Az**2 );
} #  1}}}
sub Unit {  #  {{{1
    my ($Ax, $Ay, $Az) = @_;

    my $len = Mag($Ax, $Ay, $Az);
    return ($Ax/$len, $Ay/$len, $Az/$len );
} #  1}}}
sub rot_matrix {  #  {{{1
    my ($cos_psi  , $sin_psi  ,
        $cos_theta, $sin_theta,
        $cos_phi  , $sin_phi  ,
       ) = @_;
    #
    # _Principles of Dynamics_, D.T. Greenwood, 2nd ed., 1988, pp. 356-357
    #
    my @R;

    $R[0][0] =  $cos_psi*$cos_theta;
    $R[1][0] = -$sin_psi*$cos_phi + $cos_psi*$sin_theta*$sin_phi;
    $R[2][0] =  $sin_psi*$sin_phi + $cos_psi*$sin_theta*$cos_phi;

    $R[0][1] =  $sin_psi*$cos_theta;
    $R[1][1] =  $cos_psi*$cos_phi + $sin_psi*$sin_theta*$sin_phi;
    $R[2][1] = -$cos_psi*$sin_phi + $sin_psi*$sin_theta*$cos_phi;

    $R[0][2] = -$sin_theta;
    $R[1][2] =  $cos_theta*$sin_phi;
    $R[2][2] =  $cos_theta*$cos_phi;

    return @R;
} #  1}}}
sub global_cstm {  #  {{{1
    my ($id             ,  # in  coordinate system ID 
        $rhaa_local_cstm,  # in  local CSTM's tying id_relative to id (3x3
                           #     rotation matrix only)
        $rhh_coord_data ,  # in  contents of the database's 'coord' table
        $ra_coord_chain ,  # in  list of coord system ID's that tie $id
                           #     to the basic system
        $ra_global_trans,  # out x,y,z coordinates of global translation vect.
        $raa_global_rot ,  # out 3x3 global rotation matrix
       ) = @_;

    # details {{{2
    # The hash $rhaa_local_cstm contains local coordinate system transformation 
    # matrices (CSTM).  Consider a sequence of coordinate systems, each one
    # relative to the previous one.  The first one is B, the basic system,
    # the second is c which is relative to B, et cetera:
    #
    #   B -> c -> d -> e .... -> k
    #
    # A local CSTM supplies the transformation needed to go from coordinate 
    # system N back to coordinate system N-1 upon which N is based.  The
    # transformation looks like this:
    #
    #  { x_(N-1) } =  { T_(N-1)->(N) } +  [ R_(N-1)->(N) ]{ x_(N) }
    #
    # where {x_q}    is the coordinate vector in system q
    #       {T_q->r} is the translation vector to go from system q to r
    #       [R_q->r] is the rotation matrix to go from system q to r
    #                as returned by rot_matrix, given sines and cosines
    #                of the rotation angles psi, theta, and phi that
    #                system r is rotated away from q
    #
    # then to go from system k to j:
    #   { x_j } =  { T_j->k } +  [ R_j->k ]{ x_k }              I
    # and similarly to go from i to j:
    #   { x_i } =  { T_i->j } +  [ R_i->j ]{ x_j }              II
    #
    # By substituting I into II we get the expression for going from k
    # back to i:
    #
    #   { x_i } =  { T_i->j } +  [ R_i->j ]( { T_j->k } +  [ R_j->k ]{ x_k } )
    # or
    #   { x_i } =  { T_i->j } +  [ R_i->j ] { T_j->k }  +
    #              [ R_i->j ][ R_j->k ]{ x_k }
    #
    # thus the translation vector to go from k to i is
    #   { T_i->k } =  { T_i->j } +  [ R_i->j ] { T_j->k } 
    # and the rotation matrix from k to i is
    #   [ R_i->k ] =  [ R_i->j ][ R_j->k ]{ x_k }
    #
    # Ultimately we need to get the each coordinate systems global CSTM--the 
    # {T} and [R] that link these systems back to the Basic system.
    # The global CSTM's are found by chaining together the local CSTM's
    # as shown in the equations above.
    #
    # 2}}}

    my $id_relative = $rhh_coord_data->{$id}{id_relative};

    # get the translation offset
    die "global_cstm:  no code to process {id_A} " 
        if defined $rhh_coord_data->{$id_relative}{id_A};
    my @translation = (
                        $rhh_coord_data->{$id}{A1},
                        $rhh_coord_data->{$id}{A2},
                        $rhh_coord_data->{$id}{A3},
                      );
    my @rotation    = @{$rhaa_local_cstm->{$id}};

#printf "T{$id}\n"; print_3(\@translation);
#printf "R{$id}\n"; print_3x3(\@rotation);

    foreach my $chain_id (@{$ra_coord_chain}) {
        next unless $chain_id;  # skip the basic system
        die "global_cstm:  no code to process {id_A} " 
            if defined $rhh_coord_data->{$chain_id}{id_A};
        my @trans_next = (
                            $rhh_coord_data->{$chain_id}{A1},
                            $rhh_coord_data->{$chain_id}{A2},
                            $rhh_coord_data->{$chain_id}{A3},
                          );
        my @rot_next   = @{$rhaa_local_cstm->{$chain_id}};
#printf "T{$chain_id}\n"; print_3(\@trans_next);
#printf "R{$chain_id}\n"; print_3x3(\@rot_next);
        @translation = Add(@trans_next,
                           mult_3x1(\@rot_next, \@translation));

        @rotation    = mult_3x3(   \@rotation,  \@rot_next );

#printf "updated trans\n"; print_3(\@translation);
#printf "updated rot\n";   print_3x3(\@rotation);
    }
    @{$ra_global_trans} = @translation;
    @{$raa_global_rot}  = @rotation;

} #  1}}}
sub coord_sys_chain {  #  {{{1
    # Return an array of coordinate system ID's that link the given system
    # back to the basic system.
    my ($id      , # in  ID of the coordinate system of interest
        $rhh_coord_data , # in 
        $ra_chain       , # out 
       ) = @_;
    
    my $id_based_on = $rhh_coord_data->{$id}{id_relative};
    push @{$ra_chain}, $id_based_on;
    if ($id_based_on) { # descend tree if it isn't the basic system (id=0)
        coord_sys_chain($id_based_on, $rhh_coord_data, $ra_chain);
    }

} #  1}}}
sub mult_3x3 {  #  {{{1
    my ($raa_A, $raa_B) = @_;

    # [A] and [B] must be 3x3 matrices; returns [A]*[B]

    my @C = ();
    for my $i (0..2) {
        for my $j (0..2) {
            $C[$i][$j] = $raa_A->[$i][0]*$raa_B->[0][$j] +
                         $raa_A->[$i][1]*$raa_B->[1][$j] +
                         $raa_A->[$i][2]*$raa_B->[2][$j];
        }
    }

    return @C;

} #  1}}}
sub mult_3x1 {  #  {{{1
    my ($raa_A, $ra_b) = @_;

    # [A] must be a 3x3 matrices
    # {b} must be an array with 3 terms; returns [A]*{B}

    my @C = ();
    for my $i (0..2) {
        $C[$i] = $raa_A->[0][$i]*$ra_b->[0] +
                 $raa_A->[1][$i]*$ra_b->[1] +
                 $raa_A->[2][$i]*$ra_b->[2];
    }

    return @C;

} #  1}}}
sub print_3x3 {  #  {{{1
    my ($raa_A) = @_;

    for my $row (0..2) {
        for my $col (0..2) {
            printf " % 15.9e", $raa_A->[$row][$col];
        }
        printf "\n";
    }

} #  1}}}
sub print_3 {  #  {{{1
    my ($ra_A) = @_;

    for my $j (0..2) {
        printf " % 15.9e", $ra_A->[$j];
    }
    printf "\n";

} #  1}}}

# SQL queries
sub queries {  #  {{{1
    my ($file, ) = @_;

    my %Queries = (
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - #
"
in:  canonical element ID
out: renumbered node IDs and their coordinates in basic system
" =>
"
    select R.new_id, N.x1, N.x2, N.x3 
        from element_node EN, node N, renumbered_nid R 
        where EN.eid=%d and N.seq_no=EN.nid and N.seq_no=R.orig_id 
            order by EN.seq_no;
",
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - #
"
in:  canonical ID of an isotropic shell element
out: Young's modulus, Poisson's ratio, density, thickness
" =>
"
    select M.E, M.nu, M.rho, P.thick 
        from tri3 T, shell_prop P, material M 
        where T.id=%d and T.shell_prop=P.id and M.id=P.material_id;
",
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - #
"
out:  renumbered_node_node (from renumbered_nid and node_node tables)
" =>
"
    insert into renumbered_node_node
    select R1.new_id, R2.new_id from node_node         NN,
                                     renumbered_nid    R1,
                                     renumbered_nid    R2
            where  R1.orig_id  =  NN.nid_a and
                   R2.orig_id  =  NN.nid_b and
                   R1.new_id  <=  R2.new_id
    union
    select R1.new_id, R1.new_id from renumbered_nid R1
        order by R1.new_id, R2.new_id;

",
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - #
"
out:  element_node table (created from tmp_element_node table)
" =>
"
         create index idx_node on node (id);
         create index idx1_t_e_n on tmp_element_node (nid);
         create index idx2_t_e_n on tmp_element_node (eid);
         insert into element_node (seq_no, eid, nid)
             select TEN.seq_no, TEN.eid, N.seq_no from
                 node N, tmp_element_node TEN where N.id = TEN.nid;
         drop table tmp_element_node;

",
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - #
"
out:  node_node table (created from element_node table)
" =>
"
   insert into node_node (nid_a, nid_b)
        select distinct EN1.nid, EN2.nid from
             element_node EN1, element_node EN2, node N
             where EN1.nid =  N.seq_no and
                   EN2.nid <> N.seq_no and
                   EN1.eid =  EN2.eid
             order by EN1.nid, EN2.nid;
",
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - #
"
out:  updated node table with values for basic coordinate system
" =>
"
    insert into tmp_node
    select N.id,
           N.seq_no ,
           N.coord_in,
           N.x1_in   ,
           N.x2_in   ,
           N.x3_in   ,
           N.coord_out,
           CX.T1 + CX.R11*N.x1_in + CX.R12*N.x2_in + CX.R13*N.x3_in,
           CX.T2 + CX.R21*N.x1_in + CX.R22*N.x2_in + CX.R23*N.x3_in,
           CX.T3 + CX.R31*N.x1_in + CX.R32*N.x2_in + CX.R33*N.x3_in
               from node as N,coord as C,coord_xfer as CX
               where N.coord_in = C.id and C.seq_no  = CX.seq_no
                   order by N.seq_no;
    delete from node;
    insert into node select * from tmp_node;
    drop table tmp_node;
",
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - #
"
out:  demo identifying boundary nodes given a (metis-created?)
      partitioning table 'membership' with u=node ID and v=set ID
" =>
"
    select distinct M1.u, M2.v 
        from node_node NN, membership M1, membership M2
        where NN.nid_a =  M1.u and
              NN.nid_b =  M2.u and
              M1.v     <> M2.v;
",
# - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - #
    );

    return %Queries;

} #  1}}}

1;
__END__
.csv file:  comma separated values, with optional in-line comments within
            brackets
