/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* term.c  September 1999

Copyright (c) 1999   D. R. Williamson

This file contains terminal(), which runs the program's event loop.

Function terminal() detects key inputs when running interactively, 
runs the multitasker, runs a TCP/IP server, and handles TCP/IP 
clients.  When X11 graphics is present, terminal() uses dispatcher()
to handle graphics events in various displayed windows. 

The lower portion of this file contains functions for X11 related
to running the terminal.  Graphics related X11 functions are in
file xterm.c.

X11 references:

  Redhat 5.2 Linux: /usr/doc/HOWTO/XFree86-HOWTO
  Linux and Unix:
     xterm -help
     man xterm
     man XXX for X11 command XXX, i.e., man XrmMergeDatabases

Some X11 elements adapted from: 

   Gnuplot, gplt_x11.c, distributed with Redhat 5.2 Linux
   Copyright 1986 - 1993, 1998   Thomas Williams, Colin Kelley

Network references:

   1. Stevens, W. R., "UNIX Network Programming Volume 1, Networking
      APIs: Sockets and XTI," second edition, Prentice Hall, 1998.

   2. Stevens, W. R., "UNIX Network Programming Volume 2, Inter-
      process Communications," second edition, Prentice Hall, 1999.

   3. Unix Socket FAQ: http://www.developerweb.net/forum
      Search on Maholski for comments on how to check for readable
      socket, and to determine how many bytes are waiting to be read.

   Source code for all examples in References 1 and 2 can be obtained
   from: http://www.kohala.com/start/unpv12e.html

Note:
   Source code for all examples in References 1 and 2 can be obtained
   from: http://www.kohala.com/start/unpv12e.html

   Values for errno are found in these places:
      GNU/Linux: /usr/include/asm/errno.h
      AIX: /usr/include/errno.h

----------------------------------------------------------------------*/

#undef  __STRICT_ANSI__
#define __STRICT_ANSI__
#undef  _POSIX_SOURCE
#define _POSIX_SOURCE /* GNU C Library, features.h (sets __USE_POSIX) */

#define _XOPEN_SOURCE_EXTENDED 1 /* 1 for AIX, for time.h timeval */
#include <time.h> /* put this before stdio.h in AIX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define __USE_BSD /* for typedef __caddr_t caddr_t; */
#include <sys/types.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "key.h"
#include "mat.h"
#include "math1.h"
#include "lib.h"
#include "mem.h"
#include "prs.h"
#include "sys.h"
#include "term.h"
#include "tex.h"

fd_set rset;    /* the current descriptor bits used for select() */
fd_set Rset;    /* the complete set of descriptor bits for select() */
fd_set Bset;    /* only the pending read socket for select() */
fd_set KEYset;  /* only the bits for STDIN, In, for select() */
int maxfd_init; /* file descriptor of max value after terminit() */
int Listenfd;   /* file descriptor of the server listening socket */

/* Max file descriptor number, used by select(), is:
      maxfd=MAX(maxfd_init, Listenfd, MAX(*client(i),i=1,clindx)) */
int maxfd; /* file descriptor (socket) of max value */

/* Vectors to handlers: */
void (*hAlarm)(int); /* handler for SIGALRM signals */
void (*hTimeOut)(int); /* handler on select() timeout */

/* multitasker */
struct ticker *tasklast=NULL;
struct ticker *tasktable=NULL;
static catitem *ONCAT_BUSY=NULL;
int timtic;
double freqtic;
double tictime0,tictime1;
#define FPS_LONG 0.5

int newcli_alarm; /* shared between conn_alarm() and clientmake() */
int NEWCLI_TIMEOUT;

/* ping() */
int PING,nSEC;
time_t ptime0=0,ptime1=0;

#ifdef OPENSSL
   int MAX_SOCK_SSL=100; 
#endif

int TIMEDOUT=0;

/*--------------------------------------------------------------------*/

#ifdef NET
int block(int sockfd, double sec)
/* This function blocks execution until client socket, sockfd, becomes 
   readable.  Here is a description of how it works:

   Define Bset bits only for sockfd, so only sockfd will cause select()
   in terminal() to return.  Then lock the program so no words are run 
   and the keyboard is locked (if running interactively). 

   In locking the program, function LOCK1() called below will jump to a 
   new run level and run terminal().  Within terminal(), select() will 
   respond only to a read-enable of sockfd, as dictated by Bset.  

   When sockfd becomes readable, sockfd will be read in the client loop
   of terminal(), where the item we are waiting for will be drained from
   the socket and appear on the stack.  

   At the bottom of the client loop in terminal(), after sockfd is read,
   function unblock() is called to return the program to its former 
   state when this function was entered.  

   The program will be unlocked, and operation will finally return to 
   this function (and run level) through LOCK1() in the code below.

   This function will return 0 (error) if the alarm to unblock had to
   be used, rather than a successful read of sockfd occurring first in
   the client loop of terminal().

   During the blocked time period (given by incoming sec), select() 
   continues to return on periodic timeouts, so the multitasker keeps 
   running. */
{
   int cn;

   if(sockfd<0 || sockfd>FD_SETSIZE-1 ||
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
/*    Ignore closed or invalid socket: */    
      return 1;
   }
   SO_PEND=sockfd; /* socket with pending read */
   BLOCKED=1;
   TIMEDOUT=0;

   FD_ZERO(&Bset);
   FD_SET(SO_PEND,&Bset); /* select() will respond only to SO_PEND */

   if(WTRACE) {
      gprintf(" block: blocking for read enable on socket %d",SO_PEND);
      nc();
   }
   return( 
   /* Lock with alarm in case SO_PEND does not become readable: */
      pushstr("unblock,") && /* ALARM word to run later to unblock */
      pushd(sec) &&          /* seconds to idle */
      LOCK1()                /* lock the program */
   );
}
#endif

#ifdef NET
int block1() /* BLOCK (nSocket nSec --- ) */
/* Block Socket for up to Sec seconds. */
{
   int sockfd;
   double sec;

   if(tos->typ!=NUM || (tos-1)->typ!=NUM) {
      stkerr(" BLOCK: ",NUMS2NOT);
      return 0;
   }
   if(!popd(&sec)) return 0;
   sec=MAX(0,sec);

   if(!popint(&sockfd)) return 0;

/* Quietly ignore closed socket */
   if(sockfd<0 || sockfd>FD_SETSIZE-1) return 1;

   return(block(sockfd,sec));
}
#endif

#ifdef NET
   int blockhold() /* HOLD (f --- ) */
/* If f is true, as in "yes HOLD," blocked socket SO_PEND waiting to 
   become readable will continue to be blocked until "no HOLD" is 
   obtained. */
{
   int f;

   if(!popint(&f)) return 0;

   HOLD=(f!=0);
   return 1;
}
#endif

#ifdef NET
int clientclose(int sockfd, int errflag)
/* Close socket descriptor sockfd, and remove its client from the list
   of clients.

   For a connection between two instances of this program, connection
   types LOCLI and LOSERV, write runflag=0 to sockfd.  This will cause 
   the remote end to close itself when its drain() reads runflag=0. */
{
   const int runflag=0; /* flag=0 for remote drain() to close socket */
   int cn=0,k=0;
   time_t tnow;

   if(sockfd<1) return 1; /* never try to close socket 0 */
   tnow=time(&tnow)+DELTA_T;

   if(sockfd==SO_PEND) { /* unblock if closing socket is pending */
   /* Only one socket, SO_PEND, is active if the system is BLOCKED.
      If SO_PEND is closing, unblock the system and turn HOLD off. */
      HOLD=0;
      BLOCKED=0; /* essential: set this before unblock() */
      unblock();
   }
   cn=*(sockCLI+sockfd);

   if(cn>clindx || cn<0) {
      if(WTRACE) {
         gprintf(" clientclose: socket %d client not found ",sockfd);
         gprintf("%s",datetime());
         nc();
      }
      close(sockfd);
      FD_CLR(sockfd,&Rset);
      return 1;
   }
   if(_client_open(sockfd)) {

      if(*(contyp+cn)==LOCLI || *(contyp+cn)==LOSERV) {
      /* Function write4n() below sends runflag=0 to this program at 
         the other end, making it close the connection by entering this
         very same section of code where it will send runflag=0 back to 
         here.

         By then the socket here is probably closed, so why doesn't 
         that program obtain an error from write4n()?  It has to do 
         with the TIME_WAIT state that is entered when close(sockfd) 
         performs what is called an "active close" (Reference 1, p. 
         40).  The connections linger until requisite acknowledgements
         are obtained at both ends.

         This program closes connections to itself by having each end 
         perform an active close (by running close()).  This produces
         clean disconnections, with no TIME_WAIT lingering of ports 
         afterward.  

         No TIME_WAIT lingering means the very same IP addresses and 
         ports can be reconnected without delay. */

         if(test_socket_open(sockfd)) {
            if(write4n(sockfd,runflag)!=sizeof(int)) {
            /* Ignore any write4n() error; socket probably just closed. 
               gprintf(" clientclose: error writing to socket %d",\
                  sockfd);
               stkerr("","");
               nc();
            */
            }
         }
         if(*(contyp+cn)==LOSERV) sleep(1); /* client closes sooner */
      }
      if(errflag || WTRACE) {
         if(errflag) {
            gprintf(" clientclose: socket %d closing on flag %d, ",\
               sockfd,errflag);
            gprintf("%s",datetime());
            nc();
            if(NTRACE) {
               gprintf(" clientclose: flags:"); nc();
               gprintf("    1: drainf() closing FOREIGN or NATIVE"); 
               nc();
               gprintf("    2: drain() closing LOCLI or LOSERV"); nc();
               gprintf("    3: drainf() closing NEWCLI"); nc();
               gprintf(\
                "    4: conn_alarm() closing NEWCLI grace period over");
               nc();
            }
         }
         gprintf(" clientclose: socket %d, port %5d,",sockfd,\
            *(clport+cn));
         if(       *(contyp+cn)==LOCLI) gprintf(" conn C>S");
         else if( *(contyp+cn)==NATIVE) gprintf(" conn C>F");
         else if( *(contyp+cn)==LOSERV) gprintf(" conn S<C");
         else if(*(contyp+cn)==FOREIGN) gprintf(" conn S<F");
         else if( *(contyp+cn)==NEWCLI) gprintf(" conn S<NEWCLI");
         else gprintf(" contyp unknown: %d",*(contyp+cn));
         gprintf(" is closed");
         nc();
      }
   }
   else {
      if(WTRACE) {
         gprintf(" clientclose: socket %d is already closed ",sockfd);
         gprintf("%s",datetime());
         nc();
      }
   }
   pushd(sockfd);
   pushd(xTRUE);
   setselect(); /* running: sockfd yes SELECT */

#ifdef OPENSSL
   if(*(clientSSL+cn)) {
      SSL_shutdown((SSL *)*(clientSSL+cn));
      SSL_set_shutdown(*(clientSSL+cn),SSL_SENT_SHUTDOWN);
      SSL_free((SSL *)*(clientSSL+cn));
   }
   *(clientSSL+cn)=NULL;
   *(clientSSL_PENDING+cn)=0;
#endif

   close(sockfd); /* call close() even if test_socket_open() is false */
   FD_CLR(sockfd,&Rset);

/* Remove sockfd from the list in SERVE_F: */
   if(*(contyp+cn)==NEWCLI || *(contyp+cn)==FOREIGN) {
      pushint(sockfd);
      extract1("SERVE_F","REM"); /* inline SERVE_F.REM is run */
   }
   if(*(clientIP+cn)) mallfree((void *)(clientIP+cn));
   *(clientIP+cn)=NULL;
   if(*(clientLOGIN+cn)) mallfree((void *)(clientLOGIN+cn));
   *(clientLOGIN+cn)=NULL;

   *(client+cn)=-1;
   *(clport+cn)=-1;
   *(clitim+cn)=-1; 
   *(clitimoff+cn)=tnow;
   *(contyp+cn)=NEWCLI;

/* Adjust maxfd if necessary: */
   if(sockfd==maxfd) {
      maxfd=MAX(maxfd_init,Listenfd);
      while(k<clindx+1) {
         if(*(client+k)>-1) maxfd=MAX(maxfd,*(client+k));
         k++;
      }
   }
   if(*(clsptr+cn)) { /* run ptrCls(sockfd) */
      pushint(sockfd);      /* nSocket to stack */
      pushd(*(clsptr+cn));  /* ptrCls to stack */
      *(clsptr+cn)=0; /* zero this before running ptrCls, so this
                         branch cannot be taken again */
      exe();          /* running word ptrCls; stk: nSocket ptrCls */

   /* Depending on ptrCls, may never return to here. */
   }
   if(sockfd==REMOTE_SOCKET) BYE=1;
   *(sockCLI+sockfd)=-1; /* do this after run ptrCls */

   return 1;
}
#endif

#ifdef NET
#ifdef OPENSSL
int clientmake(int sockfd, int port, SSL *ssl, int type, double ptr,\
   char *IPaddr, char *LOGIN)
#else
int clientmake(int sockfd, int port, int type, double ptr,\
   char *IPaddr, char *LOGIN)
#endif
/* Make a client by adding socket descriptor, sockfd, to the select() 
   descriptor bits in Rset, by adding sockfd to the client list, port 
   to the clport list, connection type to the contyp list, and ptr to 
   cliptr list.

   These are the connection types in contyp:
     type=-1 NEWCLI: initially for new clients connected to local server
     type=0 LOCLI: local client, connected to this prog remote server
     type=1 NATIVE: local client, connected to remote foreign server
     type=2 LOSERV: remote client this prog, connected to local server
     type=3 FOREIGN: remote client foreign, connected to local server */
{
   int cn;
   time_t tnow;

   tnow=time(&tnow)+DELTA_T;

   for(cn=0;cn<FD_SETSIZE;cn++) { /* find an unused client number */
      if(*(client+cn)<0) {
         *(client+cn)=sockfd; /* socket */
         *(clport+cn)=port;   /* port */
         *(contyp+cn)=type;   /* type (see list above) */
         *(cliptr+cn)=ptr;    /* ptr to run in drainf() if FOREIGN */
         *(clitim+cn)=tnow;   /* time of connection */
         *(clitimoff+cn)=-1;  /* initialize disconnect time */

         if(*(clientIP+cn)) mallfree((void *)(clientIP+cn));
         *(clientIP+cn)=memgetn(IPaddr,strlen(IPaddr));  /* IP addr */

         if(*(clientLOGIN+cn)) mallfree((void *)(clientLOGIN+cn));
         *(clientLOGIN+cn)=memgetn(LOGIN,strlen(LOGIN)); /* Login id */

         *(sockCLI+sockfd)=cn; /* gives client number for sockfd */
#ifdef OPENSSL
         if(*(clientSSL+cn)) SSL_free((SSL *)*(clientSSL+cn));
         *(clientSSL+cn)=ssl; /* ssl connection object */
#endif
         break;
      }
   }
   if(cn==FD_SETSIZE) {
      gprintf(" clientmake: too many clients");
      nc();
      return 0;
   }
/* Adding sockfd to Rset, the set of select() descriptors: */
   FD_SET(sockfd,&Rset);
   maxfd=MAX(maxfd,sockfd);

/* If client type is NEWCLI, it isn't known yet if the connecting pro-
   gram is type FOREIGN or type LOSERV.  Add sockfd to the list in word
   SERVE_F by running inline SERVE_F.ADD.  

   If later in drainf() it is determined this is not a foreign client, 
   the entry will be removed by running inline SERVE_F.REM.  

   SERVE_F and its local inlines ADD and REM are in file net.v.  Word 
   SERVE_F is central to running this program's server with foreign
   clients. */
   if(*(contyp+cn)==NEWCLI || *(contyp+cn)==FOREIGN) {
      pushint(sockfd);           /* sockfd on stack */
      extract1("SERVE_F","ADD"); /* running inline SERVE_F.ADD */
   }
/* Starting alarm period for NEWCLI: */
   if(type==NEWCLI && !newcli_alarm) {
      pushint(NEWCLI_TIMEOUT);
      pushstr("conn_alarm,");
      taskalarm();
      newcli_alarm=1;
   }
   if(cn>clindx)
      clindx=cn; /* the max index in lists client, clport, and contyp */

   return 1;
}
#endif

#ifdef NET
int conn_alarm() /* conn_alarm, ( --- ) */
/* Alarm to disconnect pending NEWCLI clients that have sent no bytes 
   in the period NEWCLI_TIMEOUT since connection was established.

   Connecting NEWCLI clients that send bytes within the alarm period 
   NEWCLI_TIMEOUT will cause drainf() to run, and there NEWCLI types
   will become either type LOSERV or FOREIGN and will not be discon-
   nected by this alarm.

   Normally, this function is called by the multitasker when a timeout
   really occurs.  When this alarm goes off, the oldest NEWCLI client
   is closed and the alarm is reset for the next one.

   But this function is called by drainf() when a NEWCLI becomes type 
   LOSERV or type FOREIGN, to allow the alarm to be omitted if there 
   are no more NEWCLIs.  In this case, since the alarm did not really
   go off, if there are other NEWCLIs and the oldest time span, dt, is 
   less than NEWCLI_TIMEOUT, the alarm is reset for its time remaining. 

   This function is defined in word.p as word "conn_alarm," so it 
   appears in the catalog to be found when taskalarm() is run in 
   clientmake() to start ALARM. 

   Because of the comma in its name, this word cannot be run directly 
   from the interactive prompt or from a file. */
{
   int cn=0,cn1=-1;
   int dt=0,dt1=-1;
   time_t tnow;

   tnow=time(&tnow)+DELTA_T;

/* Finding the oldest NEWCLI: */
   for(;cn<clindx+1;cn++) {
      if(*(client+cn)>-1 && *(contyp+cn)==NEWCLI) {
         if((dt=tnow-*(clitim+cn))>dt1) {
            dt1=dt;
            cn1=cn;
         }
      }
   }
   if(cn1>-1) {
      if(dt1<NEWCLI_TIMEOUT) {
      /* No timeout yet; this is probably a call from drainf().
         Reset alarm for oldest NEWCLI: */
         newcli_alarm=1;
         return(
            pushint(MAX(1,NEWCLI_TIMEOUT-dt1)) &&
            pushstr("conn_alarm,") &&
            taskalarm()
         );
      }
   /* Closing the oldest NEWCLI: */
      nc();
      gprintf(" conn_alarm: socket %d has not responded ",\
         *(client+cn1));
      gprintf("%s",datetime());
      nc();
      clientclose(*(client+cn1),4);

   /* Finding the next NEWCLI: */
      cn1=-1;
      dt1=-1;
      for(cn=0;cn<clindx+1;cn++) {
         if(*(client+cn)>-1 && *(contyp+cn)==NEWCLI) {
            if((dt=tnow-*(clitim+cn))>dt1) {
               dt1=dt;
               cn1=cn;
            }
         }
      }
      if(cn1>-1) {
      /* Starting alarm for next NEWCLI: */
         newcli_alarm=1;
         return(
            pushint(MAX(1,NEWCLI_TIMEOUT-dt1)) &&
            pushstr("conn_alarm,") &&
            taskalarm()
         );
      }
   }
/* No NEWCLIs.  Turn off conn_alarm: */
   newcli_alarm=0;
   return(
      pushstr("conn_alarm,") &&
      taskomit1()
   );
}
#endif

#ifdef NET
int daemon_init()
{
   pid_t pid=-1;

/* Fork the parent process and terminate it: */
   if((pid=fork())<0) {
      stkerr(" daemon_init: ","first fork failed");
      return 0;
   }
/* Parent terminates, first child continues: */
   if(pid>0) {
      if(WTRACE) {
         gprintf(" daemon_init: terminating parent process");
         nc();
      }
      _exit(OK_EXIT);
   }
/* First child becomes session leader of new session, and process 
   leader of new process, and has no controlling terminal (Ref 1, 
   p. 335): */
   if(setsid()<0) {
      stkerr(" daemon_init: ","set session id failed");
      return 0;
   }
/* Fork again to get a second child that is not a session leader, so 
   it cannot acquire a controlling terminal if it should later open a
   terminal device.  SIGHUP must be ignored because the terminating 
   first child will send SIGHUP to the second child: */
   signal(SIGHUP,SIG_IGN); /* ignore SIGHUP signal */
   if((pid=fork())<0) {
      stkerr(" daemon_init: ","second fork failed");
      return 0;
   }
/* First child terminates, second continues */
   if(pid>0) {
      if(WTRACE) {
         gprintf(" daemon_init: terminating first child process");
         nc();
      }
      _exit(OK_EXIT);
   }
   if(chdir("/tmp")<0) {
      stkerr(" daemon_init: ","changing to directory /tmp failed");
      return 0;
   }
   umask(0); /* clear file mode creation mask */
   return 1;
}
#endif

int disassemble() /* disassemble (hFile key --- hT) */
/* Decode file.  Usage:
      'ecard.bin' 'ec' old binary file, ec 70.7 disassemble, ec fclose
*/
{
   int f;
   char *name="_disassemble";

   if(tos->typ==NUM && (tos-1)->typ==MAT) {
      over();
      fileid();
      popint(&f);
      if(f==0) {
         stkerr(" disassemble: ",FILHNDNOT);
         return 0;
      }
      swap();
      pushstr("fload"); xmain(0);
      typmat2vol();
      pushstr(name); naming();
      swap();
      tdecode();
      if(tos->typ==NUM) { /* tdecode failed */
         drop();
         drop();
         return 0;
      }
      pushstr(name); naming();
      return 1;
   }
   stkerr(" disassemble: ",STKNOT);
   return 0;
}

#ifdef NET
int FOREVER_set() /* FOREVER (f --- ) */
/* Set flag that when true starts SERVER listening forever when the
   program is controlled by a script (not running interactively). 

   The initial value of FOREVER is true (1).

   If FOREVER is true, processing of the script ends at word SERVER 
   and the program continues running.

   If FOREVER is false, the script will be processed until it runs to 
   the end. */
{
   int f;

   if(!popint(&f)) return 0;

   FOREVER=(f!=0);
   return 1;
}
#endif

int frate() /* frate (n --- ) */
/* Specifying multitasker timer speed, n frames (tics) per second. */
{
   double n;

   if(!popd(&n)) return 0;

   n=MAX(1.1,n); /* 1.1 Hz or higher (AIX fails on 1 Hz or less) */

   USEC=MAX(1,(int)(0.5+1000000./n)); /* microseconds per tic at n Hz */

   return 1;
}

int ftic() /* ftic ( --- n) */
/* Fetching multitasker timer current frame speed.  This is the number
   of tics per second (Hz) measured over a 4 second interval. */
{
   return pushd(freqtic);
}

int LOCK1() /*  LOCK (qS d --- ) or (0 --- ) */
/* Locking the program for d seconds, d>0.  Quoted name, S, is the name
   of the ALARM word to be used to unlock the program.  The stack dia-
   gram of a word for an ALARM word, such as S, is ( --- ).

   When d=0, there is no accompanying quoted name and this function 
   unlocks the program.

   Patterned after wait1(). */
{
   char ch;
   int any_locked,k,RLHALT;
   static int locked[NBUF]={0};

   if(tos->typ!=NUM) {
      stkerr(" LOCK: ",NUMNOT);
      return 0;
   }
   if(tos->real) {

      if((tos-1)->typ!=STR) {
         stkerr(" LOCK: ",STRNOT);
         return 0;
      }
      if(!bufup1()) return 0; /* bufup1() preserves text file pointers,
                                 so words after LOCK are run */

      setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

      if(!*(jmpready+onbuf)) { /* falls through first time */

         *(jmpready+onbuf)=ENDLOCK;
         *(locked+onbuf)=1;

      /* Starting the ALARM: */
         if(!(swap() && taskalarm())) {
            bufdn1();
            return 0;
         }
         if(WTRACE) {
            gprintf(" LOCK: locked on run level %d",onbuf);
            nc();
#ifdef NET
            if(NTRACE) jmptable();
#endif
         }
         LOCKED=1;

         while(*(locked+onbuf)) {
#ifdef NET
            LOCK_PEND=0;
#endif
            terminal(); /* returns only on a key press, socket In */
            read(In,&ch,1); /* drain a byte from socket In */

            if(TRACE) gprintf("%c",ch); /* display key pressed */
         }
      }
      if(WTRACE) {
         gprintf(" LOCK: unlock run level %d",onbuf);
         nc();
      }
      *(locked+onbuf)=0;
      bufdn1();

   /* See if there are more locked levels below: */
      RLHALT=rlhalt();
      any_locked=0;
      k=onbuf;
      while(!any_locked && k>RLHALT) {
         any_locked=*(locked+k);
         k--;
      }
      if(!any_locked) {
         LOCKED=0;
         BUSY=0;
      }
   }
   else {
      drop(); /* drop d=0; there is no qS on stack in this case */

#ifdef NET
      if(NTRACE) {
         if(LOCK_PEND) {
            gprintf(\
               " LOCK: flag to unlock ignored on run level %d",onbuf);
         }
         else {
            gprintf(\
               " LOCK: flag to unlock received on run level %d",onbuf);
         }
         nc();
      }
      if(LOCK_PEND) return 1;
#else
      if(WTRACE) {
         gprintf(\
            " LOCK: flag to unlock received on run level %d",onbuf);
         nc();
      }
#endif

   /* Unwind down to highest ENDLOCK level using bufabort(): */
      RLHALT=rlhalt();
      any_locked=0;
      k=onbuf;
      while(!any_locked && k>RLHALT) {
         any_locked=*(locked+k);
         k--;
      }
      if(any_locked) bufabort(1+k); /* longjmp back to ENDLOCK */
      else {
         LOCKED=0;
         BUSY=0;
      }
   }
   return(!TIMEDOUT);
}

int LOCKED1() /*  LOCKED ( --- f) */
{
   return(pushint(LOCKED*xTRUE));
}

#ifdef NET
int new_client_timeout() /* new_client_timeout (s --- ) */
/* Set the time in seconds within which a new client connecting must 
   send something. */
{
   int s;

   if(!popint(&s)) return 0;

   NEWCLI_TIMEOUT=s;
   return 1;
}

int NEW_CLIENT_TIMEOUT() /* NEW_CLIENT_TIMEOUT ( --- s) */
/* Current value of new client timeout period. */
{
   return(pushint(NEWCLI_TIMEOUT));
}
#endif

#ifdef NET
int nextport()  /* nextport (nPORT --- nPORT1) */
/* Find the next available port starting with number PORT.  

   Tries port numbers from PORT to PORT+20, and returns PORT1 equal to 
   the port number closest to PORT, or PORT1=-1 if no available port is
   found in the range. */
{
   struct sockaddr_in servaddr;
   int bind_error=1,k=0,nports=20,PORT,sockfd;

   if(!popint(&PORT)) return 0;

   while(k<nports && bind_error) {
      sockfd=socket(AF_INET,SOCK_STREAM,0); /* create a TCP socket */

      if(sockfd<0) {
         stkerr(" nextport: ","socket error");
         return 0;
      }
      memset(&servaddr,0,sizeof(servaddr));
      servaddr.sin_family=AF_INET;

      servaddr.sin_port=htons(PORT);
      servaddr.sin_addr.s_addr=htonl(INADDR_ANY);

      if(bind(sockfd,(struct sockaddr*)&servaddr,
         sizeof(servaddr))<0) PORT++;

      else bind_error=0;
      close(sockfd);
      k++;
   }
   if(bind_error) pushint(-1);
   else pushint(PORT);

   return 1;
}
#endif

void noop1(int dummy) { }

#ifdef NET
int ntrace() /* ntrace ( --- ) */
{
   NTRACE=1;
   WTRACE=1;
   return 1;
}
#endif

#ifdef NET
int ntraceoff() /* nontrace ( --- ) */
{
   NTRACE=0;
   WTRACE=0;
   return 1;
}
#endif

#ifdef NET
int ntraceflag() /* NTRACE ( --- f) */
/* True if ntrace flag is on. */
{
   if(NTRACE) return(pushint(xTRUE));
   return(pushint(xFALSE));
}
#endif

int ping() /* ping (n --- ) */
/* Measuring multitasker timer frame speed for n seconds. */
{
   if(!popint(&nSEC)) return 0;

   if(!KEYS) {
      stkerr(" ping: ",KEYONLY);
      return 0;
   }
   PING=1;
   ptime0=0;
   time(&ptime1);
   ptime1++;
   return 1;
}

#ifdef KEYBOARD
int prompter() /* prompter (ptrRun qPrompt --- f) */
/* Text input at a prompt, then running word with ptrRun.

   Displays Prompt, and runs word with ptr equal to ptrRun whenever
   the Enter key is pressed.

   Word of ptrRun has the following stack diagram: (qText --- f).

   Flag f from ptrRun is true to continue at the Prompt, false to
   exit. */
{
   int flag=xTRUE,more;
   char *p[NBUF],*s;
   double ptrRun;

   if(tos->typ!=STR) {
      stkerr(" prompter: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=NUM) {
      stkerr(" prompter: ",NUMNOT);
      return 0;
   }
   if((s=(char *)memgetn(tos->tex,tos->col))==NULL) {
      return 0;
   }
   drop();
   popd(&ptrRun);

   if(bufup()) { 

      *(p+onbuf)=s;

      if(!*(jmpready+onbuf)) {

         *(jmpready+onbuf)=PROMPTER;
         setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
         flag=xTRUE;

         while(1) { /* keyboard input loop */

            pushstr(*(p+onbuf));
            accept_keys();
            popint(&more);
            if(!more) pushq2("",0);

            pushd(ptrRun);

            exe(); /* function ptrRun must leave flag on stack */

            if(tos->typ!=NUM) {
               flag=xFALSE;
               break;
            }
            popint(&more); /* pop more */

            if(more!=xTRUE && more!=xFALSE) { /* more must be boolean */
               flag=xFALSE;
               break;
            }
            if(!more) break; /* more is false to break */
         }
         pushint(flag);
      }
      mallfree((void *)(p+onbuf));
      bufdn();

      return 1;
   }
   return 0;
}
#endif

#ifdef NET
#ifdef KEYBOARD
int remoteprompter() /* remoteprompter (nSocket qPrompt --- ) */
/* Gather text keyed locally at Prompt, then run the text on an 
   instance of this program running remotely at Socket.

   Displays Prompt, and when the Enter key is pressed, keyed text is 
   run by this program at remote Socket.  Text normally displayed at 
   remote Socket will appear locally.

   Esc-q causes a return to the normal program prompt and commands
   again run the local program. 

   Words used by this function, remoteprompt_run and remoteprompt_stk,
   are defined in file net.v, and a high level word that uses this
   function, remoteprompt(), is also in net.v. */
{
   int depth,sockfd,temp;
   char *prompt;

   if(tos->typ!=STR) {
      stkerr(" remoteprompter: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=NUM) {
      stkerr(" remoteprompter: ",NUMNOT);
      return 0;
   }
   if((prompt=(char *)memgetn(tos->tex,tos->col))==NULL) {
      return 0;
   }
   REMOTE_PROMPT=prompt;
   drop();

   popint(&sockfd);
   REMOTE_SOCKET=sockfd;

   depth=stkdepth();

/* Keyboard input loop */
   while(_client_open(sockfd)) { 
      pushstr(prompt);
      accept_keys(); /* receiving keyboard input */

      nc(); 
      popint(&temp); /* flag -1 if have key input */

      if(temp) { /* have qS on stack to run on remote: */

      /* Add 2nd phrase, R, to stack after S to also run on remote: */
         pushstr("remoteprompt_run"); /* qR */

      /* Running S on remote and then running R on remote: */
         pushint(sockfd); /* nSocket */
         remoterun2(); /* qS qR nSocket */

      /* Blocking for null char to be sent by remoteprompt_run: */
         while(
            !(stkdepth()-depth) || /* change is zero */
            tos->typ!=STR ||       /* not a STR */
            tos->col!=1 ||         /* not a single char */
            *tos->tex!='\0'        /* not a null char */
            ) {
               pushint(sockfd);
               pushint(PROMPT_WAIT);
               if(!block1()) { /* quit prompter if time out */

               /* See if socket is still open: */
                  gprintf(" remoteprompter: socket %d",sockfd);
                  if(test_socket_open(sockfd))
                     gprintf(" is still open");
                  else 
                     gprintf(" has been closed");
                  nc();
                  BYE=1; /* simulate Esc-q to exit */
                  break;
               }
         }
         if(BYE) break;
         drop(); /* drop the VOL with null char */

      /* Phrase remoteprompt_run captured what the remote would 
         have displayed when S was run: */
         anyq(); /* anything in VOL from remote? */
         popint(&temp);
         if(temp) {
            dot(); /* printing what S would display on remote */
            nc();
         }
         else nc(); /* mimic "nl [Enter]" in word console */
      }
      else if(BYE) break; /* Esc-q was hit, exiting */
        
   /* Get a display of the remote stack: */
      pushstr("remoteprompt_stk");
      pushint(sockfd);
      remoterun();
      block(sockfd,MAXBLOCK); /* block for VOL from word >stk */

      anyq(); /* anything in VOL? */
      popint(&temp);
      if(temp) {
         if(tos->tex!=NULL &&
            strcmp(tos->tex," stack is empty")!=0) {
            dot(); /* print the remote stack display */
            nc();
         }
         else drop();
      }
   }
   BYE=0; /* turning off so levels below this won't exit */
   mallfree((void *)&prompt);
   REMOTE_PROMPT=NULL;
   REMOTE_SOCKET=-1;

   return 1;
}
#endif
#endif

#ifdef NET
int _server() 
/* Start a TCP/IP server on this machine, responding to IPaddr on port 
   nPort. */
{
   struct sockaddr_in servaddr;
   unsigned char iplocal[4]; /* local Internet IP address: */
   int i=0;

   if(!(tos->typ==NUM || (tos-1)->typ==STR || (tos-1)->typ==VOL)) {
      stkerr(" SERVER: ",STKNOT);
      return 0;
   }
   if(SERVER) {
      gprintf(" SERVER: busy listening on port %d",PORT);
      nc();
      return(drop2());
   }
   Listenfd=socket(AF_INET,SOCK_STREAM,0); /* create a TCP socket */
   if(Listenfd<0) {
      stkerr(" SERVER: ","socket error");
      return 0;
   }
   memset(&servaddr,0,sizeof(servaddr));
   servaddr.sin_family=AF_INET;

   popint(&PORT);
   if(PORT<1) PORT=SERV_PORT;
   servaddr.sin_port=htons(PORT);

   for(;i<4;i++) *(iplocal+i)=0;
   strchop();

   if(tos->col==0 || (tos->col==1 && *(tos->tex)=='*')) {
   /* Allowing any Internet address: */
      servaddr.sin_addr.s_addr=htonl(INADDR_ANY);
   }
   else {

#if (IPLOCAL==1)
   /* Conventional TCP/IP use of iplocal as servaddr.sin_addr.s_addr: */
      ip4_scan(tos->tex,iplocal); /* convert IP text to 4 numbers */

      if(WTRACE) {
         gprintf(" SERVER: scanned numbers of iplocal: %d %d %d %d",\
            *iplocal,*(iplocal+1),*(iplocal+2),*(iplocal+3));
         nc();
      }
      servaddr.sin_addr.s_addr=(in_addr_t)iplocal;
#endif

#if (IPLOCAL==0)
   /* The program will listen for INADDR_ANY, but will only connect to 
      IP addresses given in qIPaddr (a STR or VOL) presently on top of 
      the stack.

      This list of IP addresses is now appended to the table called
      "clients" in the library of word CLIENT_ALLOW (defined by net.v 
      at start up).  When a client connects, SERVER will check its IP
      address against table CLIENT_ALLOW.clients. */

      lpush();
      pushstr("CLIENT_ALLOW");
      pushstr("clients");
      dup2s();
      extract();
      lpeek();
      pile(); /* append to existing table */
      rev();
      implant(); /* banking CLIENT_ALLOW.clients */
      lpull();

   /* Allowing any Internet address: */
      servaddr.sin_addr.s_addr=htonl(INADDR_ANY);
#endif
   }
   if(bind(Listenfd,(struct sockaddr*)&servaddr,sizeof(servaddr))<0) {

      if(*(iplocal)) {
         stkerr("","");
         gprintf("%s",datetime());
         gprintf(" SERVER: bind error: IP address %s is not local",\
            tos->tex);
         nc();
      }
      else {
      /* stkerr("",""); don't report error; used by word nextport */ 
         gprintf("%s",datetime());
         gprintf(" SERVER: bind error: port %d is in use",PORT);
         gprintf(" or has not timed out yet");
         nc();
      }
      drop(); /* qIPaddr off stack */
      return 0;
   }
   drop(); /* qIPaddr off stack */

/* The program runs away in select() if listen() is not run: */
   if(listen(Listenfd,LISTENQ) < 0) {
      stkerr(" SERVER: ","listen error");
      return 0;
   }
/* Adding Listenfd to Rset, the set of select() descriptors: */
   FD_SET(Listenfd,&Rset);
   maxfd=MAX(maxfd,Listenfd);
   SERVER=1;

/* Run the phrase
      'exe_remote' '2drop' exe_remote 
   to disable the ability to substitute words from remoteprompt: */
   pushstr("'exe_remote' '2drop' exe_remote");
   xmain(0);

   if(!KEYS && !DSERVER) { /* not interactive; must be running script */
      if(WTRACE) {
         gprintf(" SERVER: running server on port %d, socket %d",\
            PORT,Listenfd); 
         nc();
      }
      if(FOREVER) {
         wait1(xTRUE); /* run forever as background server */
      }
   }
   return 1;
}
#endif

#ifdef NET
int server() /* SERVER (qIPaddr nPort --- ) */
/* Start a TCP/IP server on this machine, responding to IPaddr on port 
   nPort. */
{
   SECURE=0;
   return(_server());
}
#endif

#ifdef NET
int server1() /* DSERVER (qIPaddr nPort --- ) */
/* Start a daemon TCP/IP server on this machine, responding to IPaddr 
   on port nPort. */
{
   int f=0,wtrace=0;
   double const WAIT=0.1;

   if(!(tos->typ==NUM || (tos-1)->typ==STR || (tos-1)->typ==VOL)) {
      stkerr(" DSERVER: ",STKNOT);
      return 0;
   }
   if(KEYS) {
      stkerr(
         " DSERVER: ","cannot start daemon when keyboard is active");
      return 0;
   }
   if(SERVER) { /* close the present SERVER */
      wtrace=WTRACE;
      WTRACE=1;
      serverclose();
      WTRACE=wtrace;
   }
   sysout1(); /* push SYSOUT qFile to stack */
   fileq();
   popint(&f);
   if(!f) { /* no sysout log file has been defined; make one: */
      pushstr("/tmp/");
      pushstr("runid");
      xmain(0);
      cat();
      pushstr("_DSERVER.LOG");
      cat();
      set_sysout();

      sysout1(); /* make sure log file exists: */
      fileq();
      popint(&f);
      if(!f) { 
         stkerr(" DSERVER: ","cannot open log file");
         return 0;
      }
   }
   if(!daemon_init()) return 0;

#ifdef X11
   if(Dpy) XCloseDisplay(Dpy);
#endif
   Dpy=0;

   DSERVER=1; /* never is zero this while daemon is running */

/* Close standard file descriptors before opening the listening server.
   The listening server socket to be opened next must come before all 
   the client sockets to be opened later.

   The following sockets are possibly open:
      Socket Name  Description
        0     In   stdin, opened and named In in terminit()
        1     -    stdout
        2     -    stderr
        3     -    AIX curses newterm opened in keywin()
        4     Cn   X-Windows events, opened and named Cn in terminit()
   
   Some sockets open, readable and writable can be seen by running:

      "socket   open    readable writable" . nl
      9 0  
      DO list: I I socket_open I socket_readable I socket_writable ;
      LOOP 10 parkn bend mtext left justify 2 indent .

      [tops@clacker] ready > ww
      socket   open    readable writable
        0       -1        0       -1
        1       -1        0       -1
        2       -1        0       -1
        3        0       -1        0
        4       -1        0       -1
        5        0        0       -1
        6        0       -1        0
        7        0       -1        0
        8        0       -1        0
        9        0       -1        0
      [tops@clacker] ready > 

      Starting a server:

      [tops@clacker] ready > "" 9877 SERVER

      [tops@clacker] ready > ww
      socket   open    readable writable
        0       -1        0       -1
        1       -1        0       -1
        2       -1        0       -1
        3       -1        0        0 <<<<<<<<<<<<<<
        4       -1        0       -1
        5        0        0       -1
        6        0       -1        0
        7        0       -1        0
        8        0       -1        0
        9        0       -1        0
      [tops@clacker] ready > 
     
      A client connecting from another window (by running IPloop 9877 
      CLIENT):

      [tops@clacker] ready > 
      Thu Aug 25 07:25:17 PDT 2005 SERVER: connection from 127.0.0.1
      [tops@clacker] ready > ww
      socket   open    readable writable
        0       -1        0       -1
        1       -1        0       -1
        2       -1        0       -1
        3       -1        0        0
        4       -1        0       -1
        5        0        0       -1
        6       -1        0       -1 <<<<<<<<<<<<<<
        7        0       -1        0
        8        0       -1        0
        9        0       -1        0
      [tops@clacker] ready > 
*/
   if(In>-1) { 
      close(In); /* STDIN */
      FD_CLR(In,&Rset);
   }
   In=-1; /* WARNING: do not send In=-1 to FD_ISSET */

   fflush(stdout);
   close(STDOUT_FILENO); /* STDOUT */

   fflush(stderr);
   close(STDERR_FILENO); /* STDERR */

   if(Cn>-1) { 
      close(Cn); /* X-Windows events */
      FD_CLR(Cn,&Rset);
   }
   Cn=-1; /* WARNING: do not send Cn=-1 to FD_ISSET */

   if(!(_server())) {
      stkerr(" DSERVER: ","failed to start daemon server");
      return 0;
   }
   if(SECURE) 
      gprintf(\
         " DSERVER_SSH: running daemon server on port %d, socket %d",
         PORT,Listenfd); 
   else
      gprintf(" DSERVER: running daemon server on port %d, socket %d",\
         PORT,Listenfd); 
   nc();

/* Essential to reliable start up: */
   pushd(WAIT);
   pushstr("idle");
   xmain(0);

   terminal(); 
   stkerr(" DSERVER: ","daemon server returning");
   return 0;
}
#endif

#ifdef NET
#ifdef OPENSSL
int server2() /* SERVER_SSL (qIPaddr nPort --- ) */
/* Start a TCP/IP server on this machine, responding to IPaddr on port
   nPort.

   Communication with clients on nPort uses encryption on a connection
   secured by Transport Layer Security (TLS) and Secure Sockets Layer
   (SSL)--a TLS/SSL connection. */
{
   if(!server_SSL()) return 0;
   SECURE=1;
   return(_server());
}

int server3() /* DSERVER_SSL (qIPaddr nPort --- ) */
/* Start a daemon TCP/IP server on this machine, responding to IPaddr
   on port nPort. 

   Communication with clients on nPort uses encryption on a connection
   secured by Transport Layer Security (TLS) and Secure Sockets Layer
   (SSL)--a TLS/SSL connection. */
{
   if(!server_SSL()) return 0;
   SECURE=1;
   return(server1());
}
#endif
#endif

#ifdef NET
int serverclose() /* serverclose ( --- ) */
/* Close the listening socket opened by word SERVER.  Connected sockets
   remain connected. */
{
   if(SERVER) {
      if(WTRACE) {
         gprintf(" serverclose: listening port %d closed",PORT);
         nc();
      }
      close(Listenfd);
      FD_CLR(Listenfd,&Rset);
      Listenfd=-1; /* WARNING: do not send Listenfd=-1 to FD_ISSET */
      SERVER=0; 
      PORT=-1;
   }
   return 1;
}
#endif

#ifdef NET
int setready() /* READY (f --- ) */
/* If f=0, clear the bit for server Listenfd in Rset.  Server will 
   ignore connections from select() in terminal() until this function
   runs with f!=0. */
{
   int r;
   static int cleared=0;

   if(!popint(&r)) return 0;
   if(Listenfd<0) return 1;

   READY=(r!=0);

   if(READY && cleared) {
      FD_SET(Listenfd,&Rset); /* set Listenfd */
      cleared=0;
   }
   else if(!READY && !cleared) {
      FD_CLR(Listenfd,&Rset); /* clear Listenfd */
      cleared=1;
   }
   return 1;
}
#endif

#ifdef NET
int setselect() /* SELECT (nSocket f --- ) */
/* If f=0, clear the bit for client Socket in Rset, causing select() in 
   terminal() to let inputs to Socket go unread.  If f!=0 and the bit
   for Socket has been cleared, it will be reset.

   At start up, *(SELECT+i) equals 1 for all client sockets, i, to 
   select input as soon as it arrives.  Only in this function is any 
   element of SELECT ever set to 0 to skip reading the corresponding 
   socket. */
{
   int cn,f,sockfd;

   if(!popint(&f)) return 0;
   if(!popint(&sockfd)) return 0;

   if(sockfd<0 || sockfd>FD_SETSIZE-1 ||
      (cn=*(sockCLI+sockfd))>clindx || cn<0) {
/*    stkerr(" SELECT: ","invalid client socket descriptor");
      return 0;

      Ignore closed or invalid socket:
*/    return 1;
   }
   *(SELECT+cn)=(f!=0);

   if(*(SELECT+cn) && *(SELECT_CLR+cn)) {
      FD_SET(sockfd,&Rset); /* set the bit for sockfd again */
      *(SELECT_CLR+cn)=0;
   }
   else if(!(*(SELECT+cn)) && !(*(SELECT_CLR+cn))) {
      FD_CLR(sockfd,&Rset); /* clear the bit for sockfd */
      *(SELECT_CLR+cn)=1;
   }
   return 1;
}
#endif

int socket_open() /* socket_open (nSocket --- f) */
{
   int sockfd;

   if(!popint(&sockfd)) return 0;

   if(test_socket_open(sockfd))
      pushint(xTRUE); /* connection is valid */

   else /* not connected */
      pushint(xFALSE);

   return 1;
}

int taskadd(struct ticker *new, struct ticker **table,
            struct ticker **end)
/* Adding a new element to linked list of multitasker tasks.

   Making a doubly linked list in sorted order, after:
      Schildt, H., "C: The Complete Reference,"
      Osborne McGraw-Hill, 1995

   Fancier than required, but useful as a model for general
   linked lists. */
{
   register struct ticker *old,*p;

   if(*end==NULL) { /* very 1st element in list */
      new->nex=NULL;
      new->pre=NULL;
      *end=new;
      *table=new;
      return 1;
   }
   p=*table;
   old=NULL;
   while(p) {
      if(strcmp(p->cat->nam,new->cat->nam)<0) {
         old=p;
         p=p->nex;
      }
      else { /* error if already have nam */
         if(strcmp(p->cat->nam,new->cat->nam)==0) {
            stkerr(" taskadd: ",TASKDUPE);
            return 0;
         }
         if(p->pre) { /* inserting in front of p */
            p->pre->nex=new;
            new->nex=p;
            new->pre=p->pre;
            p->pre=new;
            return 1;
         }
         new->nex=p; /* inserting in front of 1st element */
         new->pre=NULL;
         p->pre=new;
         *table=new;
         return 1;
      }
   }
   old->nex=new; /* putting at end */
   new->nex=NULL;
   new->pre=old;
   *end=new;
   return 1;
}

int taskalarm() /* ALARM (d qS --- ) */
/* Define and start an alarm to run word S when d seconds (d>0)  have 
   elapsed. */
{
   struct ticker *p;
   double d,tnow;

   if((tos-1)->typ!=NUM) {
      stkerr(" ALARM: ",NUMNOT);
      return 0;
   }
   if(tos->typ!=STR) {
      stkerr(" ALARM: ",STRNOT);
      return 0;
   }
   strchop();

/* Defining the period of the alarm and its related frequency: */
   d=(tos-1)->real;

   if(!(d>0)) return(drop2());

   pushd((double)1/d); /* rate, fps, is one over the period d */

   over(); /* (fps qS) stk is now set up for taskdef(), word TASK */

   if(!(
      taskdef() &&
      ((p=taskfind(tos->tex,tasktable))!=NULL)
   )) return 0;

   p->typ=1; /* setting type to alarm */

   tnow=timed()+DELTA_T;    /* true GMT=machineGMT + DELTA_T */
   p->tout=tnow+1.0/p->fps; /* alarm future time-out */

   return( /* (d qS) */
      taskwake() && /* an ALARM starts running immediately */
      drop()
   );
}

int taskdef() /* TASK (w qS --- ) */
/* Defining a background task that executes catalog function S at a
   rate of w times per second. */
{
   struct ticker *p;
   char *nam;

   if(tos->typ!=STR) {
      stkerr(" TASK: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=NUM) {
      stkerr(" TASK: ",NUMNOT);
      return 0;
   }
   strchop();

   if((p=malloc(sizeof(struct ticker)))==NULL) {
      stkerr(" taskdef: ",MEMNOT);
      return 0;
   }
   if((nam=(char *)memgetn(tos->tex,tos->col))==NULL) {
      mallfree((void *)&p);
      return 0;
   }
   if(taskfind(nam,tasktable)) taskomit(); /* delete old task */
   else drop();

   p->typ=0; /* default typ is not an alarm */

   p->fps=pop()->real; /* rate, fps */

   p->tout=0; /* if alarm, to be set by ALARM */

   p->tic=-1; /* initially asleep */

   p->cat=(catitem *)catfetch(nam);
   mallfree((void *)&nam);
   if(p->cat && taskadd(p,&tasktable,&tasklast)) return 1; 

   mallfree((void *)&p);
   return 0;
}

int taskdisp() /* tasks ( --- ) */
/* Displaying a list of background tasks. */
{
   if(tasktable) tasklist(tasktable);
   else {
      gprintf(TASKSNOT);
      nc();
   }
   return 1;
}

void tasker(int signal)
/* Multitasker.  Decrementing task tic counts and running tasks with
   tickers that have expired. */
{
   static int secs=0,pings=0;
   double freq;
   struct ticker *c=NULL,*c0;
   static int BCOUNT=0;
   const int BCMAX=500; /* about 20 seconds at 25 Hz */

   if(PING) { /* not accurate for frequencies 2 Hz and less */
      time(&ptime0);
      if(ptime0==ptime1) pings++;
      else {
         if(!pings) return;

         gprintf(".");
         secs++;
         if(secs>=nSEC) {
            freq=(double)pings/(double)secs;

         /* Must press Enter because right now the program is waiting 
            for a key in function key(): */
            gprintf(" %3.1f Hz (press Enter to continue)",freq);

            pings=0;
            secs=0;
            PING=0; /* turn PING off */
            timtic=4*freqtic; /* smooths resumption of 4-sec ave */
         }
         else ptime1=ptime0;
      }
      return; /* this means tasks are suspended while ping runs */
   }
   timtic++;
   tictime0=timed();
   if(tictime0>tictime1) { /* update the running estimate of freqtic */

      freqtic=(double)timtic/4; /* divide by 4 for 4-second average */
      tictime1=4+tictime0;      /* update again in 4 seconds */

      timtic=0;
   }
   if(BUSY) {
      BCOUNT++;
      if(BCOUNT>BCMAX) { /* always interrupts once BCOUNT is reached */
         if(ONCAT_BUSY)
            gprintf(" tasker: busy task %s interrupted ",\
               ONCAT_BUSY->nam);
         else gprintf(" tasker: busy task interrupted ");
         gprintf("%s",datetime());
         nc();
         BUSY=0;
      }
      else return;
   }
/* NOTE: Behavior is unpredictable if tasktable is modified (due to 
   running taskadd() or taskrem()) while traversing the tasks list in
   the loop below, since local pointer c will become invalid.  If task-
   table pointer changes so it no longer equals c0, exit the loop. */
   c0=c=tasktable;

   while(c && c0==tasktable) { /* Traversing tasktables's linked list.*/

      if(c->tic>0) (c->tic)--;
      else {

         if(c->fps<FPS_LONG) { 

         /* Updating tic for tasks with long periods */
            if(c->tout>(tictime0+DELTA_T)) {

               if(c->tic>-1) {

               /* Improved estimate of tics until time-out: */
                  freq=2/((c->tout)-tictime0-DELTA_T);

                  c->tic=tics(freq);
               }
            }
         } 
         if((c->tic)==0) { 

         /* The countdown has ended for this c. */
         /* Working on this catalog item; note that oncat is global: */
            oncat=ONCAT_BUSY=c->cat;

            if(c->typ) { /* Running an ALARM word. */

            /* Alarm words are removed from the task list before 
               running them: */
/* Debug output worth keeping:
gprintf(" tasker alarm: %s %s\n",c->cat->nam,datetime());
*/
               taskrem(c->cat->nam,&tasktable,&tasklast);
               
            /* Running the alarm word: */
               (*(unsigned long (*)())oncat->exe)();
               return; /* must return; running taskrem() has changed
                          tasktable, so c is no longer valid */
            }
            else { /* Running a TASK word. */

               ticadvance(c); /* advance tic */

            /* Running the cataloged task word: */
               BUSY=1;
/* Debug output worth keeping:
gprintf(" tasker task: %s %s\n",c->cat->nam,datetime());
*/
               (*(unsigned long (*)())oncat->exe)();
               BUSY=0;
/* Debug output worth keeping:
if(c0!=tasktable) 
gprintf(" tasker c0: %x tasktable: %x %s\n",c0,tasktable,datetime());
*/
               if(c0!=tasktable) return; /* must return if changed */ 
            }
         }
      }
      c=c->nex; /* on the final list element, c becomes NULL here */
   }
}

struct ticker *taskfind(char *nam, struct ticker *table)
/* Finding an item in linked list of tasks. */
{
   register struct ticker *p=0;

   if(caton2(nam)) {
      p=table;
      while(p) {
         if(strcmp(p->cat->nam,oncat->nam)==0) break;
         p=p->nex;
      }
   }
   return p;
}

void tasklist(struct ticker *tasks)
/* Displaying items in linked list of tasks. */
{
   struct ticker *p;
   catitem *c;
   int wait;
   time_t tnow;

   p=tasks;
   gprintf(" Multitasker tasks:");
   nc();
      
   while(p) {
      c=p->cat;
      if(p->tic==-1)
         gprintf("  %s %g Hz; asleep",untagged(c->nam),p->fps);
      else {
         if(p->typ) { /* alarm, waiting for time tout: */
            time(&tnow);
            wait= MAX(0,
               ((double)p->tout - ((time_t)tnow + (int)DELTA_T)));
            gprintf("  %s alarm period %g seconds; remaining %d",
               c->nam,(1./p->fps),wait);
         }
         else { /* periodic task, counting tics down to zero: */
            wait=(int)(p->tic); /* tic counting down */
            gprintf("  %s task running at %g Hz; tics remaining %d",
               c->nam,p->fps,wait);
         }
      }
      nc();
      p=p->nex;
   }
}

int taskomit() /* OMIT (qS --- ) */
/* Deleting background task S from the multitasker's list. */
{
   if(tos->typ!=STR) {
      stkerr(" OMIT: ",STRNOT);
      return 0;
   }
   strchop();

   return(
      taskrem(tos->tex,&tasktable,&tasklast) &&
      drop()
   );
}

int taskomit1() /* -ALARM (qS --- ) */
/* Deleting background task S if it is in the multitasker's list,
   returning silently if it is not. */
{
   struct ticker *p;

   if(tos->typ!=STR) {
      stkerr(" -ALARM: ",STRNOT);
      return 0;
   }
   strchop();

   if(!(p=taskfind(tos->tex,tasktable))) {
      return(drop());
   }
   return(
      taskrem(tos->tex,&tasktable,&tasklast) &&
      drop()
   );
}

int taskrate() /* RATE (w qS --- ) */
/* Changing the rate of background task S. */
{
   struct ticker *p;

   if(tos->typ!=STR) {
      stkerr(" taskrate: ",STRNOT);
      return 0;
   }
   if((tos-1)->typ!=NUM) {
      stkerr(" taskrate: ",NUMNOT);
      return 0;
   }
   strchop();

   p=taskfind(tos->tex,tasktable);

   if(!p) {
      stkerr(" taskrate: ",TASKNOT);
      return 0;
   }
   if(p->typ==1) {
      stkerr(" taskrate: ","alarm rate cannnot be changed");
      return 0;
   }
   p->fps=(tos-1)->real;

   if(p->tic==-1)  return(drop2()); /* return if asleep */

/* Immediate reset; put to sleep, then awaken: */
   p->tic=-1;
   return( /* (w qS) */
      taskwake() && drop()
   );
}

int taskrem(char *nam, struct ticker **table, struct ticker **end)
/* Removing an item from the linked list of tasks. */
{
   struct ticker *p;

   p=taskfind(nam,*table);

   if(!p) {
      gprintf(" taskrem: task %s not found",nam);
      nc();
      stkerr("","");
      return 0;
   }
   p->tic=-1; /* SLEEP */

   if(*table==p) {
      *table=p->nex;
      if(*table) (*table)->pre=NULL;
      else *end=NULL;
   }
   else {
      p->pre->nex=p->nex;
      if(p!=*end) p->nex->pre=p->pre;
      else *end=p->pre;
   }
   mallfree((void *)&p);
   return 1;
}

int tasks_omit() /* tasks_omit ( --- ) */
/* Omit all tasks and alarms from the multitasker. */
{
   struct ticker *p;

   p=tasktable;
   while(p) {
      pushstr(p->cat->nam);
      taskomit1();
      p=p->nex;
   }
   return 1;
}

int tasksleep() /* SLEEP (qS --- ) */
/* Making background task S inactive. */
{
   struct ticker *p;

   if(tos->typ!=STR) {
      stkerr(" tasksleep: ",STRNOT);
      return 0;
   }
   strchop();

   p=taskfind(tos->tex,tasktable);

   if(!p) {
      stkerr(" tasksleep: ",TASKNOT);
      return 0;
   }
   if(p->typ==1) /* SLEEP omits an alarm: */
      return(taskomit());
   else {
      p->tic=-1;
      if(p->cat==ONCAT_BUSY) BUSY=0;
      return(drop());
   }
}

int taskwake() /* WAKE (qS --- ) */
/* Making background task S active. */
{
   struct ticker *p;

   if(tos->typ!=STR) {
      stkerr(" taskwake: ",STRNOT);
      return 0;
   }
   strchop();

   p=taskfind(tos->tex,tasktable);

   if(!p) {
      stkerr(" taskwake: ",TASKNOT);
      return 0;
   }
/* A task to awaken is one that is asleep; tasks that are asleep
   have p->tic equal to -1: */

   if(p->tic==-1) {

      ticadvance(p);

   /* Start short-period task (but not alarm) immediately: */
      if(p->fps>FPS_LONG && p->typ==0) p->tic=0; 
   }
   return(drop());
}

int tdecode() /* decode (hT1 key --- hT) if success,
                 decode (hT1 key --- hT1 key) if failure

   Note: non-ascii characters are still in the text returned. */
{
   char *name="_decode";
   int pdp=3412,trace;
   double key1,key2;

   if(tos->typ==NUM && (tos-1)->typ==VOL) {

      trace=TRACE;

      TRACE=0;

      popd(&key1);
      key1=MAX(2,key1);
      pushd(key1);

      over();
      typvol2mat();
      swap();

      tkey();

   /* key from last element: */
      key2=*((tos->mat)+(tos->row)-1);
      pushd(key2);

   /* key into 1x1 MAT, then into VOL type for import8: */
      hand(); typmat2vol();

   /* undoing tencode() work: endian to stack, and import key: */
      pushint(pdp); import8(); key2=*(tos->mat); drop();

   /* put zero where key used to be: */
      *((tos->mat)+(tos->row)-1)=0;

      if(key1==key2) {
         typmat2vol();
         pushstr(name);
         naming();
         lop();
      }
      else {
         drop();
         pushd(key1);
         gprintf(" tdecode: invalid key");
         nc();
      }
      TRACE=trace;
   }
   return 1;
}

int tencode() /* encode (hT key --- hT1) */
{
   char *name="_encode";
   int pdp=3412,trace;
   double key;

   if(tos->typ==NUM && (tos-1)->typ==VOL) {

      trace=TRACE;

      TRACE=0;

      popd(&key);
      key=MAX(2,key);

      dup1s();

      textput();
      textget0();

      pushd(key);
      over();

   /* VOL into STR into MAT: */
      textput();
      typstr2mat();

   /* key to 1x1 MAT, export8 makes into VOL of endian type: */
      over(); hand(); pushint(pdp); export8();

   /* key VOL back to MAT, pile onto the rest: */
      typvol2mat(); pile(); swap();

      tkey();

   /* verify decoding */
      typmat2vol();
      dup1s();
      rev();
      pushd(key);
      tdecode();
      if(tos->typ==NUM) drop();
      textget0();
      if(tos->row==(tos-1)->row) {
         pushstr("match"); xmain(0);
         totals();
         if(*tos->mat) {
            gprintf(" tencode: invalid key; please try another");
            nc();
            return(drop() && drop());
         }
         drop();
         lop();
      }
      else {
         gprintf(" tencode: invalid key; please try another");
         nc();
         return(drop() && drop() && drop());
      }
      pushstr(name);
      naming();

      TRACE=trace;
   }
   else drop();

   return 1;
}

void terminal()
/* Running the machine:
    - running the multitasker
    - keyboard processing when interactive console is running
    - listening as a TCP/IP server
    - handling TCP/IP clients 
    - X11 window dispatcher handling events in created windows

   When the program is locked, this function is reentered. */
{
   int nf; /* returned by select: number of active file descriptors */
   struct timeval tv;


#ifdef X11
   XEvent xevent;
#endif

#ifdef NET
   int cn,connfd,connshow,k,nodot,port,sockfd;
   socklen_t clilen;
   struct sockaddr_in cliaddr;
   char *IPaddr;
#endif

#ifdef OPENSSL
   #define LEN 1024
   #define WAIT 0
   #define uWAIT 10000
   int len,n,ret,reused,ssl_err,ssl_ret;
   char *T;
   SSL *ssl=NULL;
   BIO *sbio;
   SSL_SESSION *session;
#endif

   while(1) {

#ifdef X11
      if(Dpy) XFlush(Dpy); /* empty the X11 output buffer */
#endif

      if(!LOCKED) rset=Rset;

#ifdef NET
      else { /* program is locked and possibly blocked: */

         if(BLOCKED) /* one socket for select() to service: */
            rset=Bset; 

         else rset=KEYset; /* locked: respond to keys and drop them */
      }
#else
      else rset=KEYset; /* locked: respond to keys and drop them */
#endif

   /* Setting the timeout value because select() may have changed it: */
      tv.tv_sec=0;
      tv.tv_usec=USEC;

   /* The program sits here in select(), waiting for something to do. */
      nf=select(maxfd+1,&rset,(fd_set *)NULL,(fd_set *)NULL,&tv);

   /* The order of the following if() tests is important. */
      if(nf<0) {
         if(errno==EINTR) { /* a non-blocked signal was caught */
            hAlarm(0); /* SIGALRMs are handled by hAlarm */
            goto bottom;
         }
         else {
#ifdef NET
            if(errno==EBADF && BLOCKED) {
            /* Bad file descriptor while BLOCKED.  This is probably 
               a timeout while waiting for SO_PEND, the only active 
               socket while BLOCKED.  Unblock the system and keep 
               running: */
               BLOCKED=0; /* essential: set this before unblock() */
               unblock();
               goto bottom;
            }
#endif
            if(errno==EBADF) {
               gprintf(" terminal: select() failed on EBADF:");
               gprintf(" bad file descriptor");
            }
            else {
               if(errno==ENOMEM) {
                  gprintf(" terminal: select() failed on ENOMEM:");
                  gprintf(" out of memory");
               }
               else
                  gprintf(" terminal: select() failed, errno=%d",errno);
            }
            nc();
            gprintf(" terminal: exiting on fault in select() "); 
            gprintf("%s",datetime());
            nc();
            pushstr("exit");
            xmain(0);
         }
      }
      if(nf==0) { 
      /* Processing a tv timeout.  

         Timeouts from select() give the program a heartbeat.  

         Default setting is 25 timeouts per sec.  The number of time-
         outs per second is called the multitasker frame rate, and can
         be set by word frate. */

         hTimeOut(0); /* timeouts are handled by hTimeOut */

#ifdef OPENSSL
         if(SERVER && SECURE) {
         /* Check client sockets for SSL bytes pending, and if there 
            are any, simulate a select() return.  

            Passing this point about 25 times a second enables those
            SSL bytes previously moved out of a socket by SSL_read(),
            and into an SSL buffer, to be picked up. 

            And server COG-on-READ_F works transparently, as if the 
            bytes came from the socket emptied by SSL_read(). */

            for(cn=0;cn<=clindx;cn++)  
               nf+=SSL_select(cn); /* simulate a select() return */

            if(!nf) goto bottom;   
         /* else go process nf SSL clients as if select() returned. */
         }
         else goto bottom;
#else
         goto bottom;
#endif
      }
   /* Below here, nf is 1 or greater.  The order of the following if() 
      tests is important. */

#ifdef NET
   /* This section is adapted from the select listening socket loop 
      shown in Reference 1, p. 166.

      The number of clients served is limited to the maximum number
      of selectors allowed for select(), minus those being used for
      standard Unix (stdin, stdout, stderr), keyboard input, the
      listening socket, and X11 windows.  

      The maximum number of selectors may be compiled into the kernel, 
      and is given by FD_SETSIZE.  It is probably 256 or greater and 
      can be viewed in the start up log by saving it using the -l 
      switch. */

      if(SERVER) {
         if(Listenfd>-1 && FD_ISSET(Listenfd,&rset)) { 
         /* Register a new client. */
            clilen=sizeof(cliaddr);
            if((connfd=Accept(Listenfd, \
               (struct sockaddr *)&cliaddr,&clilen)) < 0) {
               gprintf(" SERVER: accept error");
               nc();
               goto endnet;
            }  
            if((IPaddr=sock_ntop((struct sockaddr *)&cliaddr,clilen))\
               ==NULL) {
               gprintf(" SERVER: cannot resolve IP, closing socket %d",\
                  connfd); 
               nl_prompt();
               close(connfd);
               goto endnet;
            }
            port=ntohs(cliaddr.sin_port);
            k=strlen(IPaddr);
            nodot=1;
            while(k>-1 && nodot) { /* remove port number from IPaddr */
               k--;
               if(*(IPaddr+k)=='.') nodot=0;
            }
            *(IPaddr+k)='\0';

            extract1("SERVE_F","CONNSHOW");
            popint(&connshow);
            if(WTRACE || connshow) {
               nc();
               gprintf("%s",datetime());
               gprintf(" SERVER: %s connect",IPaddr);
               nc();
               memprobe();
               gprintf(
                  "bytes delta: memprobe socket %d connect",connfd);
               nc();
            }
         /* Verify client IP address using word CLIENT_ALLOW (net.v), 
            where the table of allowed IP addresses is kept: */
            pushstr(IPaddr);
            pushstr("CLIENT_ALLOW");
            xmain(0);
            popint(&k); /* close connection if flag k is false */
            if(k==xFALSE) {

            /* Send something so wget will quit trying: */
               pushstr("CRLF");
               xmain(0);
               writen(connfd,tos->tex,tos->col);
               drop();

            /* Close the connection: */
               close(connfd);

               gprintf("%s",datetime());
               gprintf(" SERVER: %s connect refused",IPaddr);
               nl_prompt();

            /* Ignore new connections for RFSEC: */
               extract1("SERVE_F","RFSEC"); /* put seconds on stack */
               pushstr("SERVER_WAIT");
               xmain(0);

               goto endnet;
            }
#ifdef OPENSSL
            if(connfd>FD_SETSIZE) {
               gprintf(
                  " SERVER: cannot use socket number %d for index",
                  connfd);
               nc();
               nl_prompt();
               close(connfd);
               goto endnet;
            }
            if(SECURE) {
               ssl=SSL_new(ctx);
               sbio=BIO_new_socket(connfd,BIO_NOCLOSE);
               SSL_set_bio(ssl,sbio,sbio);

               if(WTRACE) {
                  timeprobe();
                  gprintf("microsec delta: begin SSL_accept_timeo");
                  nc();
               }
               if((ssl_ret=SSL_accept_timeo(ssl,SSL_CONNTO)<=0)) {
                  gprintf(
                     " SERVER: SSL_accept failed on handshake to %s",
                     IPaddr);
                  nc();
                  ssl_err=SSL_get_error(ssl,ssl_ret);
                  gprintf(" SERVER: SSL_accept error = %d",ssl_err);
                  nl_prompt();
                  SSL_free(ssl);

                  ret=readable_timeo(connfd,WAIT,uWAIT,&len);
                  len=MIN(len,LEN);
                  #undef WAIT
                  #undef uWAIT
                  #undef LEN

                  if(ret>0 && len>0 && strstk(len,"_connfd")) {
                     T=tos->tex;
                     n=readn1(connfd,T,len);
                     if(n>-1) { 
                        dup1s();
                        pushstr("HTTP"); /* look for HTTP in tos->tex */
                        grepr();
                        if(tos->row) {
                           drop(); /* drop grepr result */
                           gprintf(" SERVER: insecure request:");
                           nc();
                           textput();
                           asciify();
                           notrailing();
                           *(tos->tex+tos->col)='\0';
                           gprintf(" %s",tos->tex);
                           nc();
                           gprintf(\
                              " SERVER: closing insecure connection ");
                               gprintf("%s",datetime());
                           nc();
                           pushstr("<HTML><BODY>");
                           pushstr(\
                              "Closing insecure connection\r\n\r\n");
                           cat();
                           pushstr("</BODY></HTML>");
                           cat();
                           writen1(connfd,tos->tex,tos->col);
                           drop(); /* drop HTML */
                        }
                        else drop(); /* drop grepr result */
                     }
                     drop(); /* T off stack */
                  }
                  close(connfd);
                  gprintf(" SERVER: socket %d closed ",connfd);
                  gprintf("%s",datetime());
                  nc();
                  goto endnet;
               }
            /* SSL performs a rehandshake after 304 seconds; to revise,
               use SSL_SESSION_set_timeout() (see notes in ssl.c). */

               reused=SSL_session_reused(ssl);
               if(!reused) {
                  session=SSL_get_session(ssl);
                  if(session) SSL_set_session(ssl,session);
               }
               if(WTRACE) {
                  timeprobe();
                  gprintf("microsec delta: ");
                  if(reused) 
                    gprintf("end SSL_accept_timeo reused");
                  else gprintf("end SSL_accept_timeo with handshake");
                  nc();
               }
            }
#endif
         /* Make a new client: */
#ifdef OPENSSL
            if(!clientmake(connfd,port,ssl,NEWCLI,0,IPaddr,"")) {
#else
            if(!clientmake(connfd,port,NEWCLI,0,IPaddr,"")) {
#endif
               close(connfd);
               goto endnet;
            }
            if(WTRACE) {
               gprintf(" SERVER: new client on socket %d ",connfd);
               gprintf("%s",datetime());
               nc();
            }
            if(--nf <=0) goto endnet; /* no more to do */
         }
      }
   /* Client loop.  Checking all clients, remote and local, for 
      incoming data, and draining socket if there is any. */

      for(cn=0;cn<=clindx;cn++) { 
         if((sockfd=*(client+cn))<0) continue;
#ifdef OPENSSL
         if(!(SSL_select(cn) || FD_ISSET(sockfd,&rset))) continue;
#else
         if(!FD_ISSET(sockfd,&rset)) continue; 
#endif
      /* These are connection types defined in net.h:
            NEWCLI -1 new just connected from there to this prog here

            When this program is at the client end of a connection:
            LOCLI   0 from this prog here to this prog there
            NATIVE  1 from this prog here to foreign prog there 

            When this program is at the server end of a connection: 
            LOSERV  2 to this prog here from this prog there
            FOREIGN 3 to this prog here from foreign prog there

         Word clientsockets shows connection types.
      */
         SOCKFD=sockfd; /* global SOCKFD used by remotefd() */

         switch(*(contyp+cn)) {

            case NATIVE:  /* this program to foreign program */
            case FOREIGN: /* foreign program to this program */
               if(!drainf(sockfd,*(contyp+cn),*(cliptr+cn)))
                  clientclose(sockfd,1);
            break;

            case LOCLI:  /* this program to this program remote */
            case LOSERV: /* this program remote to this program */
               if(!drain(sockfd)) clientclose(sockfd,2);
            break;

            case NEWCLI: /* new remote, may be FOREIGN or LOSERV */
                  if(!drainf(sockfd,NEWCLI,0)) clientclose(sockfd,3);
            break;

            default:
               gprintf(" terminal: invalid conn type: %d",\
                  *(contyp+cn));
               nc();
               stkerr("","");
         }
         if(--nf <=0) break; /* no more to do */
      }
      if(BLOCKED && !HOLD) {
      /* Only one socket, SO_PEND, is active if BLOCKED, so this one 
         must be it.  Unblock the system since HOLD is not on: */
         HOLD=0;
         BLOCKED=0; /* essential: set this before unblock() */
         unblock();
      }
      SOCKFD=-1;
      endnet:
#endif
      
#ifdef X11
      if(Cn>-1 && Dpy && FD_ISSET(Cn,&rset)) { 
      /* Processing window event: */
         XNoOp(Dpy);
         while(XPending(Dpy)) { /* collecting events (1 of each): */
         /* NOTE: just collecting one of each means some multiple 
            occurrences of the same event are missed.  
            Rethink this. */
            XNextEvent(Dpy,&xevent);
            *(Events+xevent.type)=xevent;
         }
         dispatcher(); /* processing window events */
      }
#endif
      if(In>-1 && FD_ISSET(In,&rset)) {
      /* Return to process a console key press.  

         For good flow, it is important that key processing follow 
         client socket processing in order to first read incoming 
         socket bytes that should precede a key press. */
         return; 
      }
      bottom:
      continue; /* AIX requires this */
   }
}

int terminit()
/* Initializing variables for function terminal().  

   Output using gprintf() will go to the start up log, which is saved 
   using the -l switch. */
{
#ifdef NET
   int i;
#endif

#ifdef OPENSSL
   SSL *ssl;
#endif

   maxfd=-1;
   Listenfd=-1; /* WARNING: do not send Listenfd=-1 to FD_ISSET */

/* Initializing the select() items in event loop of terminal(): */
   FD_ZERO(&Rset);
   FD_ZERO(&Bset);
   FD_ZERO(&KEYset);

   gprintf(" FD_SETSIZE: %d",FD_SETSIZE); /* value for this machine */
   nc();

   In=fileno(stdin); /* inputs from keys */
   gprintf(" stdin socket, In: %d",In);

   if(test_socket_open(In)) {
      gprintf(", socket is open");
      FD_SET(In,&Rset);
      KEYset=Rset;
      maxfd=MAX(maxfd,In);
   }
   else {
      gprintf(", socket is not open");
      In=-1;
   }
   nc();

   Out=fileno(stdout); /* standard output */
   gprintf(" stdout socket, Out: %d",Out);

   if(test_socket_open(Out)) gprintf(", socket is open");
   else {
      gprintf(", socket is not open");
      Out=-1;
   }
   nc();

   WINKEY=0;

#ifdef X11
   if(Dpy) {
      Cn=ConnectionNumber(Dpy); /* events from XWindows */
      gprintf(" Xwindows events socket, Cn: %d",Cn);

      if(test_socket_open(Cn)) {
         gprintf(", socket is open");
         FD_SET(Cn,&Rset);
         maxfd=MAX(maxfd,Cn);
      }
      else {
         gprintf(", socket is not open");
         Cn=-1;
      }
      nc();
   }
#else
   Dpy=0;
#endif

   BUSY=0;
   PTRACE=0;
   REMOTE_PROMPT=NULL;
   WTRACE=0;

#ifdef NET
   BLOCKED=0;
   clindx=-1;
   COG=0;
   DSERVER=0;
   FOREVER=1;
   HOLD=0;
   LOCK_PEND=0;
   MAXBLOCK=6;
   NET_ENDIAN=NET_ENDIAN_DEF;
   NEWCLI_TIMEOUT=3;
   newcli_alarm=0;
   NTRACE=0;
   PORT=-1;
   READ_F=0;
   READY=1;
   REMOTERUN_DELAY=0;
   SECURE=0;
   SERVER=0;
   SOCKFD=-1;
   TIMEDOUT=0;

   for(i=0;i<FD_SETSIZE;i++) {
      *(client+i)=-1;
      *(clport+i)=-1;
      *(cliptr+i)=0;
      *(clitim+i)=-1; 
      *(clitimoff+i)=-1; 
      *(clsptr+i)=0;
      *(contyp+i)=NEWCLI;

      *(SELECT+i)=1; /* default: bit for select() read will be set */
      *(SELECT_CLR+i)=0; /* 0= never turned *(SELECT+i) off */

      *(sockCLI+i)=-1;
   }
#endif

#ifdef OPENSSL
   memset(clientSSL,0,FD_SETSIZE*sizeof(ssl));
   memset(clientSSL_PENDING,0,FD_SETSIZE*sizeof(int));
   SSL_CONNTO=20; /* seconds */
#endif

/* Multitasker */
   hTimeOut=(void (*)())tasker;
   hAlarm=(void (*)())noop1;

   tictime0=timed();
   tictime1=-1;

   freqtic=MAX(1,(1e6/USEC)); /* multitasker frequency */
   timtic=4*freqtic; /* 4 second sum for average in tasker() */

   maxfd_init=maxfd;
   return 1;
}

int test_socket_open(int sock)
/* Adapted from test_socket_open() of GNU Wget.

   Note: Select() will return a false indication of closed if there 
   are bytes to read.  See test_socket_openw() in net.c.

   See examples in term.c under: int server1(). */
{
  fd_set check_set;
  struct timeval to;
  int ret=0;
  
  if(sock<0 || sock>FD_SETSIZE-1) {
    return 0;
  }

/* Check if we still have a valid (non-EOF) connection.  From Andrew
   Maholski's code in the Unix Socket FAQ. */

  FD_ZERO (&check_set);
  FD_SET (sock, &check_set);

  /* Wait one microsecond */
  to.tv_sec = 0;
  to.tv_usec = 1;

/* If we get a timeout, then that means still connected */
  ret=(select (sock + 1, &check_set, NULL, NULL, &to) == 0);

  return(ret);
}

void ticadvance(struct ticker *p)
/* Set the initial tic count for task p, ready to begin the tic count-
   down in tasker(). 

   For accuracy, tasker() treats tasks with periods greater than two 
   seconds (frequencies less than FPS_LONG) by halving the remaining 
   time and adjusting countdown tics accordingly, until tics drives to 
   zero.

   In this way, the time between executions of a task with very long 
   period is accurate to within one second. */
{
   double tnow;

/* Use tictime0 from tasker() for latest time: */
   tnow=tictime0+DELTA_T; /* true GMT=machineGMT + DELTA_T */

   if(p->fps>FPS_LONG) {
      p->tout=tnow;
      p->tic=tics(p->fps);
   }
   else { /* long period tasks: */
      p->tout=tnow+(1.0/p->fps); /* ultimate time-out */
      p->tic=tics(p->fps)/2; /* tasker() recalibrates after half */
   }
}

int tics(double ftask)
/* Number of multitasker tics corresponding to frequency ftask of task.
   Assumes an efficiency factor. */
{
   const double eff=0.996;
   return(MAX(1,eff*freqtic/ftask));
}

int timtic1() /* timtic ( --- n) */
{
   return(pushint(timtic));
}

int tkey() /* tkey (hA key --- hA1) */
/* Encode A.  Incoming A is a matrix, key is a NUM. */
{
   char *name="_tkey";
   double key,seed;
   int cols,rows,trace;

   if(tos->typ==NUM && (tos-1)->typ==MAT) {

      trace=TRACE;

      TRACE=0;

      popd(&key);

      rows=tos->row;
      cols=tos->col;
      pushint(rows); pushint(cols*sizeof(double));

      seed=SEED;
      pushd(SEED0/key);
      seedset(); /* use this so SEED stays in bounds */
      trandom();
      SEED=seed;

      typvol2mat(); bend(); xor();

      pushstr(name); naming();

      TRACE=trace;
   }
   else drop();

   return 1;
}

int tload() /* bload (hT key --- hT1) */
{
   int trace;

   if(tos->typ==NUM && (tos-1)->typ==VOL) {

      trace=TRACE;

      TRACE=0;

      tdecode(); /* successful if VOL on stack */

      if(tos->typ!=VOL) {
         drop2();
         stkerr(" bload: ",FILNOSO);
         return 0;
      }

      TRACE=trace;

      return 1;
   }
   else {
      stkerr(" bload: ",STKNOT);
      return 0;
   }
}

int tload1() /* brun (hT key --- ) */
{
      return(tload() && xmain(0));
}

int tmake() /* bmake (hT key --- hT1) */
{
   if(tos->typ==NUM && (tos-1)->typ==VOL) {
      return(tencode());
   }
   else {
      stkerr(" bmake: ",STKNOT);
      return 0;
   }
}

#ifdef NET
int unblock() /* unblock, ( --- ) */
/* Unblock and unlock the program when SO_PEND becomes readable or when
   the alarm for MAXBLOCK seconds has timed out.

   This function is defined in word.p as word "unblock," so it appears
   in the catalog to be found when taskalarm() is run in block() to set
   ALARM, and when taskomit1() is run here to eliminate ALARM.

   Because of the comma in its name, this word cannot be run directly 
   from the interactive prompt or from a file. */
{
   if(WTRACE) {
      gprintf(" unblock: unblocking socket %d",SO_PEND);
      nc();
   }
   pushstr("unblock,");
   taskomit1(); /* turning off ALARM set in block() to run 'unblock,' */

   if(BLOCKED) { /* ALARM has timed out. */
      gprintf(" unblock: alarm to read from socket %d has timed out",
         SO_PEND);
      stkerr("","");
      nc();
      BLOCKED=0;
      TIMEDOUT=1;
   }
   SO_PEND=-1;

   UNLOCK();
   return 1;
}
#endif

int UNLOCK() /* UNLOCK ( --- ) */
{
   return(pushint(0) && LOCK1());
}

int winfocus() /* winfocus ( --- nFocus) */
/* The number of the window that has focus.  Focus is an XWindow number
   like the first element in a window control block.  To view Focus as 
   a hex int, run: (nFocus) int '%X' format. */
{
#ifdef X11
   int revert_to_return;
   Window focus_return;

   if(Dpy) {
      XGetInputFocus(Dpy, &focus_return, &revert_to_return);
      return(pushint((int)focus_return));
   }
   else {
      return(pushint(xFALSE));
   }
#else
   return(pushint(xFALSE));
#endif
}

int winkey() /* winkey ( --- nWin) */
/* Window number of interactive keyboard, equal to zero if no inter-
   active keyboard. */
{
   return(pushd(WINKEY));
}

int wtrace() /* wtrace ( --- ) */
{
#ifdef NET
   ntraceoff();
#endif
   WTRACE=1;
   return 1;
}

int wtraceoff() /* nowtrace ( --- ) */
{
   WTRACE=0;
   return 1;
}

int _X11() /* X11 ( --- f) */
/* Note: Even if X11 graphics is present, this will return false 
   until Dpy is set, which is when xterminit() is run during start
   up.  Since this is after boot.v is sourced, running word X11 in 
   boot.v (or any of the files that boot.v may source) will return 
   f=false. */
{
#ifdef X11
   if(Dpy) return(pushint(xTRUE));
   else return(pushint(xFALSE));
#else
   return(pushint(xFALSE));
#endif
}
  
/* End of terminal functions */

/*--------------------------------------------------------------------*/

/* Begin xterminal functions */

#ifdef X11

/* xterminit() */
char *Class,*Name;
KeyCode qkeycode;
int Raise=1;

/* wcolor() */
Colormap Cmap;
Bool Clear=0,Gray=0,Mono=0,Rv=0;
#define Ncolors 13
unsigned long colors[Ncolors];
char color_keys[Ncolors][30] =   {
   "background", "bordercolor", "text", "border", "axis",
   "line1", "line2", "line3",  "line4",
   "line5", "line6", "line7",  "line8"
};
char color_values[Ncolors][30] = {
   "white", "black",  "black",  "black",  "black",
   "red",   "green",  "blue",   "magenta",
   "cyan",  "sienna", "orange", "coral"
};
char gray_values[Ncolors][30] = {
   "black",   "white",  "white",  "gray50", "gray50",
   "gray100", "gray60", "gray80", "gray40",
   "gray90",  "gray50", "gray70", "gray30"
};

/* dbget() */
static XrmDatabase dbCmd,dbApp,dbDef,dbEnv,db=(XrmDatabase)0;
XrmValue value;

/* options */
static XrmOptionDescRec options[]={
   {"-mono",             ".mono",             XrmoptionNoArg,   (caddr_t) "on" },
   {"-gray",             ".gray",             XrmoptionNoArg,   (caddr_t) "on" },
   {"-clear",            ".clear",            XrmoptionNoArg,   (caddr_t) "on" },
   {"-tvtwm",            ".tvtwm",            XrmoptionNoArg,   (caddr_t) "on" },
   {"-pointsize",        ".pointsize",        XrmoptionSepArg,  (caddr_t) NULL },
   {"-display",          ".display",          XrmoptionSepArg,  (caddr_t) NULL },
   {"-name",             ".name",             XrmoptionSepArg,  (caddr_t) NULL },
   {"-geometry",         "*geometry",         XrmoptionSepArg,  (caddr_t) NULL },
   {"-background",       "*background",       XrmoptionSepArg,  (caddr_t) NULL },
   {"-bg",               "*background",       XrmoptionSepArg,  (caddr_t) NULL },
   {"-foreground",       "*foreground",       XrmoptionSepArg,  (caddr_t) NULL },
   {"-fg",               "*foreground",       XrmoptionSepArg,  (caddr_t) NULL },
   {"-bordercolor",      "*bordercolor",      XrmoptionSepArg,  (caddr_t) NULL },
   {"-bd",               "*bordercolor",      XrmoptionSepArg,  (caddr_t) NULL },
   {"-borderwidth",      ".borderwidth",      XrmoptionSepArg,  (caddr_t) NULL },
   {"-bw",               ".borderwidth",      XrmoptionSepArg,  (caddr_t) NULL },
   {"-font",             "*font",             XrmoptionSepArg,  (caddr_t) NULL },
   {"-fn",               "*font",             XrmoptionSepArg,  (caddr_t) NULL },
   {"-reverse",          "*reverseVideo",     XrmoptionNoArg,   (caddr_t) "on" },
   {"-rv",               "*reverseVideo",     XrmoptionNoArg,   (caddr_t) "on" },
   {"+rv",               "*reverseVideo",     XrmoptionNoArg,   (caddr_t) "off"},
   {"-iconic",           "*iconic",           XrmoptionNoArg,   (caddr_t) "on" },
   {"-synchronous",      "*synchronous",      XrmoptionNoArg,   (caddr_t) "on" },
   {"-xnllanguage",      "*xnllanguage",      XrmoptionSepArg,  (caddr_t) NULL },
   {"-selectionTimeout", "*selectionTimeout", XrmoptionSepArg,  (caddr_t) NULL },
   {"-title",            ".title",            XrmoptionSepArg,  (caddr_t) NULL },
   {"-xrm",              NULL,                XrmoptionResArg,  (caddr_t) NULL },
   {"-raise",            "*raise",            XrmoptionNoArg,   (caddr_t) "on" },
   {"-noraise",          "*raise",            XrmoptionNoArg,   (caddr_t) "off"},
   {"-persist",          "*persist",          XrmoptionNoArg,   (caddr_t) "on" },
   {"-proc",             ".proc",             XrmoptionSepArg,  (caddr_t) NULL },
   {"-draw",             ".draw",             XrmoptionSepArg,  (caddr_t) NULL },
   {"-xorigin",          ".xorigin",          XrmoptionSepArg,  (caddr_t) NULL },
   {"-yorigin",          ".yorigin",          XrmoptionSepArg,  (caddr_t) NULL },
   {"-width",            ".width",            XrmoptionSepArg,  (caddr_t) NULL },
   {"-height",           ".height",           XrmoptionSepArg,  (caddr_t) NULL }
};
#define Nopt (sizeof(options)/sizeof(options[0]))


/* wcreate() */
#define On(v) (!strcmp(v,"on") || !strcmp(v,"true") || \
               !strcmp(v,"On") || !strcmp(v,"True") || \
               !strcmp(v,"ON") || !strcmp(v,"TRUE"))

/* wfont() */
Cursor cursor;
#define DefaultFont "fixed"
XFontStruct *font;
int vchar;
#define XC_crosshair 34

/* wwidth() */
#define Nwidths 10 
/* border, axis, and 8 lines: */
unsigned int widths[Nwidths] = {2,0,0,0,0,0,0,0,0,0};

int atomget() /* XAtom (n --- qS) */
/* Fetch the string corresponding to nth atom defined in XAtom.h. 
   Atoms correspond to window properties, which can be obtained with
   GetWindowProperty().
   File lnxXatom.txt shows the 248 atom strings for linux, but only
   the first 68 are common to AIX. 
   This function is restricted to the common, predefined atoms. */
{
   Atom atomMax,atomMin=1;
   int atom,ret;
   char *p;
   const char *lastpredefined="WM_TRANSIENT_FOR";
   const int maxpredefined=68;

   if(!Dpy) {
      stkerr(" atomget: ",TERMNOT);
      return 0;
   }
   atomMax=XInternAtom(Dpy,lastpredefined,True);
   if(atomMax==None) atomMax=maxpredefined;

   if(!popint(&atom)) return 0;

   if(atom<atomMin || atom>atomMax) {
      stkerr(" atomget: ",ATOMOUT);
      return 0;
   }
   p=XGetAtomName(Dpy,atom);
   ret=pushstr(p);
   XFree(p);

   return ret;
}

char *dbget(XrmDatabase db, char *resource)
/* Fetch database resource of type="String" */
{
   char name[256]={0},class[256]={0};
   char *type;

   (void)sprintf(name,"%s.%s",Name,resource);
   (void)sprintf(class,"%s.%s",Class,resource);

   if(XrmGetResource(db,name,class,&type,&value)) {
      if(strcmp(type,"String")!=0) return((char *)0);
      return (char *)value.addr;
   }
   return((char *)0);
}

int dbget1() /* Xresource (qOpt --- qDb) */
{
   char *p,*r;

   if(tos->typ!=STR) {
      stkerr(" dbget1: ",STRNOT);
      return 0;
   }
   if((p=(char *)memgetn(tos->tex,tos->col))==NULL) {
      stkerr(" dbget1: ",MEMNOT);
      return 0;
   }
   r=dbget(db,p);
   mallfree((void *)&p);

   if(r) return(drop() && pushstr(r));
   else return(drop() && pushq2("",0));
}

int dheight() /* dheight ( --- h) */
/* Height of display in pixels. */
{
   return(pushint(DisplayHeight(Dpy,Scr)));
}

int dheightMM() /* dheightMM ( --- hMM) */
/* Height of display in millimeters. */
{
   return(pushint(DisplayHeightMM(Dpy,Scr)));
}  

void dispatcher()
/* Handling events in windows. */
{ 
/* dispatcher() attends to these messages in the order shown: */
   const int ncom=10;
   int Messages[10]={
      ClientMessage, /* may be a message to close, so this is first */
      Expose,
      VisibilityNotify,
      ConfigureNotify,
      KeyPress,
      KeyRelease,
      ButtonPress,
      MotionNotify,
      ButtonRelease
   };
   const char *BP="BP",*BR="BR",*CE="CE",*CM="CM",*EE="EE";
   const char *KP="KP",*KR="KR",*ME="ME",*VE="VE";
   double *EV;
   register int i=0;
   XEvent event;

   for(;i<ncom;i++) {
      event=Events[*(Messages+i)];
      Events[*(Messages+i)].type=0;

      if(event.type) {

         switch(event.type) {

            case ButtonPress:
               if(!matstk(BPsize,1,(char *)BP)) {
                  stkerr(" dispatcher, ButtonPressedEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+BPwin)=event.xbutton.window;
               *(EV+BPx)=event.xbutton.x;
               *(EV+BPy)=event.xbutton.y;
               *(EV+BPrx)=event.xbutton.x_root;
               *(EV+BPry)=event.xbutton.y_root;
               *(EV+BPst)=event.xbutton.state;
               *(EV+BPbu)=event.xbutton.button;
               XEventHandler(ButtonPress);
            break;

            case ButtonRelease:
               if(!matstk(BRsize,1,(char *)BR)) {
                  stkerr(" dispatcher, ButtonReleasedEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+BRwin)=event.xbutton.window;
               *(EV+BRx)=event.xbutton.x;
               *(EV+BRy)=event.xbutton.y;
               *(EV+BRrx)=event.xbutton.x_root;
               *(EV+BRry)=event.xbutton.y_root;
               *(EV+BRst)=event.xbutton.state;
               *(EV+BRbu)=event.xbutton.button;
               XEventHandler(ButtonRelease);
            break;

            case ClientMessage:
               if(!matstk(CMsize,1,(char *)CM)) {
                  stkerr(" dispatcher, ClientMessageEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+CMwin)=event.xclient.window;
               *(EV+CMtyp)=event.xclient.message_type;
               *(EV+CMfmt)=event.xclient.format;
               *(EV+CMl0)=event.xclient.data.l[0];
               *(EV+CMl1)=event.xclient.data.l[1];
               *(EV+CMl2)=event.xclient.data.l[2];
               *(EV+CMl3)=event.xclient.data.l[3];
               *(EV+CMl4)=event.xclient.data.l[4];
               XEventHandler(ClientMessage);
            break;

            case ConfigureNotify:
               if(!matstk(CEsize,1,(char *)CE)) {
                  stkerr(" dispatcher, ConfigureEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+CEwin)=event.xconfigure.window;
               *(EV+CEx)=event.xconfigure.x;
               *(EV+CEy)=event.xconfigure.y;
               *(EV+CEw)=event.xconfigure.width;
               *(EV+CEh)=event.xconfigure.height;
               *(EV+CEb)=event.xconfigure.border_width;
               XEventHandler(ConfigureNotify);
            break;

            case Expose:
               if(!matstk(EEsize,1,(char *)EE)) {
                  stkerr(" dispatcher, ExposeEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+EEwin)=event.xexpose.window;
               *(EV+EEx)=event.xexpose.x;
               *(EV+EEy)=event.xexpose.y;
               *(EV+EEw)=event.xexpose.width;
               *(EV+EEh)=event.xexpose.height;
               XEventHandler(Expose);
            break;
   
            case KeyPress:
               if(!matstk(KPsize,1,(char *)KP)) {
                  stkerr(" dispatcher, KeyPressedEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+KPwin)=event.xkey.window;
               *(EV+KPx)=event.xkey.x;
               *(EV+KPy)=event.xkey.y;
               *(EV+KPrx)=event.xkey.x_root;
               *(EV+KPry)=event.xkey.y_root;
               *(EV+KPst)=event.xkey.state;
               *(EV+KPkc)=event.xkey.keycode;
               XEventHandler(KeyPress);
            break;

            case KeyRelease:
               if(!matstk(KRsize,1,(char *)KR)) {
                  stkerr(" dispatcher, KeyReleasedEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+KRwin)=event.xkey.window;
               *(EV+KRx)=event.xkey.x;
               *(EV+KRy)=event.xkey.y;
               *(EV+KRrx)=event.xkey.x_root;
               *(EV+KRry)=event.xkey.y_root;
               *(EV+KRst)=event.xkey.state;
               *(EV+KRkc)=event.xkey.keycode;
               XEventHandler(KeyRelease);
            break;

            case MotionNotify:
               if(!matstk(MEsize,1,(char *)ME)) {
                  stkerr(" dispatcher, MotionEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+MEwin)=event.xmotion.window;
               *(EV+MEx)=event.xmotion.x;
               *(EV+MEy)=event.xmotion.y;
               *(EV+MErx)=event.xmotion.x_root;
               *(EV+MEry)=event.xmotion.y_root;
               *(EV+MEst)=event.xmotion.state;
               *(EV+MEh)=event.xmotion.is_hint;
               XEventHandler(MotionNotify);
            break;

            case VisibilityNotify:
               if(!matstk(VEsize,1,(char *)VE)) {
                  stkerr(" dispatcher, VisibilityEvent: ",MEMNOT);
                  break;
               }
               EV=tos->mat;
               *(EV+VEwin)=event.xvisibility.window;
               *(EV+VEst)=event.xvisibility.state;
               XEventHandler(VisibilityNotify);
            break;
/*
         #ifdef EXPORT_SELECTION
            case SelectionNotify:
            case SelectionRequest:
               handle_selection_event(event);
            break;
         #endif
*/
         }
      }
   }
}

int dprops() /* dprops ( --- ) */
/* Properties of the display that are common to all windows. */
{
   XPixmapFormatValues *XP;
   int count_return,i=0;
   int *depths;
   unsigned long black,white;

   if(Dpy) {

      gprintf(" Default properties of display common to all windows");
      nc();
      gprintf("    Display: %X",Dpy); nc();
      gprintf("    Root: %X",Root); nc();
      gprintf("    DefaultScreen: %X",Scr); nc();
      gprintf("    DefaultVisual: %X",Vis); nc();
      gprintf("    DefaultDepth: %X",Dep); nc();

      depths=XListDepths(Dpy,Scr,&count_return);
      gprintf("    Depths: ");
      for(;i<count_return;i++) gprintf("%d ",*(depths+i)); nc();
      XFree(depths);

      gprintf("    DisplayPlanes: %d",DisplayPlanes(Dpy,Scr)); nc();
      gprintf("    ColorCells: %d",DisplayCells(Dpy,Scr)); nc();
      black=BlackPixel(Dpy,Scr);
      white=WhitePixel(Dpy,Scr);
      gprintf("    Pixels: black=%d, white=%d",black,white); nc();
      gprintf("    Display pixels: %d by %d", \
         DisplayWidth(Dpy,Scr),DisplayHeight(Dpy,Scr)); nc();

      XP=XListPixmapFormats(Dpy,&count_return);
      gprintf("    Pixmap formats supported: %d",count_return);nc();
      gprintf("    Pixmap depth: %d",XP->depth);nc();
      gprintf("    Pixmap bits per pixel: %d",XP->bits_per_pixel);nc();
      gprintf("    Pixmap scanline pad: %d",XP->scanline_pad);nc();
      XFree(XP);

      gprintf("    ImageByteOrder: %d",ImageByteOrder(Dpy));nc();
      gprintf("    BitmapBitOrder: %d",BitmapBitOrder(Dpy));nc();
      gprintf("    (byte and bit order: %d=LSBFirst, %d=MSBFirst)", \
         LSBFirst,MSBFirst); nc();
      gprintf("    BitmapPad: %d",BitmapPad(Dpy));nc();
      gprintf("    BitmapUnit: %d",BitmapUnit(Dpy));nc();
   }
   else {
      gprintf(" No initialized windows"); nc();
   }
   return 1;
}

int dwidth() /* dwidth ( --- w) */
/* Width of display in pixels. */
{
   return(pushint(DisplayWidth(Dpy,Scr)));
}

int dwidthMM() /* dwidthMM ( --- wMM) */
/* Width of display in millimeters. */
{
   return(pushint(DisplayWidthMM(Dpy,Scr)));
}

int wcolor() 
/* Initial settings for window color. */
{
   int n=0;
   char option[20],color[30],*v,*ctype;
   unsigned long black,white;
   double intensity=-1;
   XColor xcolor;

   black=BlackPixel(Dpy,Scr); /* =0 */
   white=WhitePixel(Dpy,Scr); /* =1 */

/*
   dbget(db,"mono") && On(value.addr) && Mono++;
   dbget(db,"gray") && On(value.addr) && Gray++;
   dbget(db,"reverseVideo") && On(value.addr) && Rv++;
*/

   if (!Gray && 
      (Vis->class==GrayScale || Vis->class==StaticGray)) Mono++;

   if(!Mono) {
      Cmap=DefaultColormap(Dpy,Scr);
      ctype=(Gray) ? "Gray" : "Color";

      for(;n<Ncolors;n++) {
         *option='\0';
         strcat(option,color_keys[n]);

         if(n>1) strcat(option,ctype);

         if((v=dbget(db,option))==NULL) {
            if(Gray) v=gray_values[n];
            else v=color_values[n];
         }
         strcpy(color,v);

         if(!XParseColor(Dpy,Cmap,color,&xcolor)) {
            stkerr(" wcolor: ","error parsing color");
            return 0;
         }
         if(WTRACE) {
            gprintf(" %d color: %s %s",n,option,color); nc();
            if(!KEYS) {
               gprintf(" %d color: %s %s\n\r",n,option,color); 
            }
         }
         intensity=1; /* default intensity=1; can have 0<intensity<1 */

         xcolor.red*=intensity;
         xcolor.green*=intensity;
         xcolor.blue*=intensity;

         if(XAllocColor(Dpy,Cmap,&xcolor)) {
            colors[n]=xcolor.pixel;
         }
         else {
            colors[n]=black;
         }
      }
   }
   else {
      colors[0]=(Rv) ? black : white ;
      for(n=1;n<Ncolors;n++) colors[n]=(Rv) ? white : black;
   }
   return 1;
}

Window wcreate(char *name, unsigned int flags, int x, int y, 
   unsigned int width, unsigned int height, unsigned long CWmask,
   XSetWindowAttributes *watt)
{
   Window win; 
   static XSizeHints hints;

   if(!Dpy) {
      stkerr(" wcreate: ",TERMNOT);
      return 0;
   }
   win=XCreateWindow(Dpy,Root,x,y,width,height,BorderWidth,
      CopyFromParent,InputOutput,CopyFromParent,CWmask,watt);

/* Sets up for ClientMessage to close: */
   XChangeProperty(Dpy,win,WM_PROTOCOLS,XA_ATOM,32,PropModeReplace,
      (unsigned char *)&WM_DELETE_WINDOW,1);

   if(name==NULL) name=Name;
   if(name==NULL) name=dbget(db,"title");
   XStoreName(Dpy,win,name);

   XSelectInput(Dpy,win,
   /*   event mask              XEvent struct type name and id */
        0                    /* ClientMessage 33 */
      | StructureNotifyMask  /* ConfigureNotify 32 */
      | VisibilityChangeMask /* VisibilityNotify 15 */
      | ExposureMask         /* Expose 12 */
      | KeyPressMask         /* KeyPress 2 */
      | KeyReleaseMask       /* KeyRelease 3 */
      | ButtonPressMask      /* ButtonPress 4 */
      | ButtonReleaseMask    /* ButtonRelesase 5 */
      | ButtonMotionMask     /* MotionNotify 6 (with button down */
/*    The following bogs down a slow machine; try to live without it: */
/*    | PointerMotionMask */ /* MotionNotify 6 (any pointer motion) */
   );

   hints.flags=gFlags;
   hints.x=gX; 
   hints.y=gY;
   hints.width=gW;
   hints.height=gH;

   XSetNormalHints(Dpy,win,&hints);

/*
   Stuff that is not necessary:

   int Tvtwm=0;
   XWMHints wmh;

   dbget(db,"clear") && On(value.addr) && Clear++;
   dbget(db,"tvtwm") && On(value.addr) && Tvtwm++;

   if(!Tvtwm) hints.flags = flags;
   else hints.flags = flags & ~USPosition | PPosition;


   if(dbget(db,"iconic") && On(value.addr)) {
      wmh.flags=StateHint;
      wmh.initial_state=IconicState;
      XSetWMHints(Dpy,win,&wmh); 
   }
*/
   return win;
}

int wfont() 
/* Initial settings for window font. */
{
   char *fontname;

   fontname=dbget(db,"font");
   if(!fontname) fontname=DefaultFont;

   font=XLoadQueryFont(Dpy,fontname);
   if(!font) {
      gprintf(" wfont: can't load font %s",fontname); nc();
      gprintf(" wfont: using default font%s",DefaultFont); nc();
      font=XLoadQueryFont(Dpy,DefaultFont);
      if (!font) {
         gprintf(" wfont: can't load font '%s'\n", DefaultFont); nc();
         return 0;
      }
   }
   vchar=font->ascent+font->descent;
   cursor=XCreateFontCursor(Dpy,XC_crosshair);
   return 1;
}

int wgeometry() 
/* Initial settings for window geometry. */
{
   char *geometry;
   int x,y,flags;
   unsigned int w,h;

   BorderWidth=2;

   geometry=dbget(db,"geometry");
   if(geometry) {
      flags=XParseGeometry(geometry,&x,&y,&w,&h);
      if(flags & WidthValue)  gW=w;
      if(flags & HeightValue) gH=h;
      if(flags & (WidthValue | HeightValue)) 
         gFlags=(gFlags & ~PSize) | USSize;
      if(flags & XValue)
         gX=(flags & XNegative) 
            ? x + DisplayWidth(Dpy,Scr) - gW - BorderWidth*2 : x;
      if (flags & YValue)
         gY=(flags & YNegative) 
            ? y + DisplayHeight(Dpy,Scr) - gH - BorderWidth*2 : y;
      if (flags & (XValue | YValue))
         gFlags=(gFlags & ~PPosition) | USPosition;
   }
   else {
      gX=10;
      gY=10;
      gW=320;
      gH=225;
      gFlags=PSize;
   }
   return 1;
}

int wwidth() 
/* Initial settings for window widths. */
{
   char wkeys[Nwidths][30]={
      "border","axis",
      "line1","line2","line3","line4",
      "line5","line6","line7","line8"};
   char option[32]={0};
   register char *v;
   register int n=0;

   for(;n<Nwidths;n++) {
      *option='\0';
      strcat(option,wkeys[n]);
      strcat(option,"Width");
      if((v=dbget(db,option))) {
         if(*v<'0' || *v>'4' || strlen(v)>1) {
            gprintf(" wwidth: "," invalid width %s:%s",option,v); nc();
            return 0;
         }
         else widths[n] = (unsigned int)atoi(v);
      }
   }
   return 1;
}

int xterminit()
/* Initializing display and Xwindow database.

   Here's a typical trace during xterminit():
      Display: :0.0
      Home: /home/dale
      Name: tops
      Class: tops
      AppDefDir: /usr/lib/X11/app-defaults/Tops
      Server from file: /home/dale/.Xdefaults
      Environment from file-host: /home/dale/.Xdefaults-gutter */
{
   register int i;
   char *arg,**argv;
   char *appdef,buf[256],*display,*env,*home,host[MAXHOSTNAMELEN],
        *xserver,*x11cmd;
   int argc,flag;

   if(Dpy) return 1;

   if(WTRACE || TRACE) {
      gprintf(" initializing xterm"); nc();
   }
   if(!(
   /* X11 cmd string from boot.v to stk: */
      pushstr("9.0 _bin") &&
      xmain(0)
   )) {
      stkerr(" xterminit: ",STRNOT);
      return 0;
   }
   if((x11cmd=(char *)memgetn(tos->tex,tos->col))==NULL) {
      stkerr(" xterminit: ",MEMNOT);
      return 0;
   }
   if(!(
      dup1s() && words() && rows() && 
      popint(&argc) /* number of words in string */
   )) {
      mallfree((void *)&x11cmd);
      return 0;
   }
   if((argv=malloc(1+argc*sizeof(long)))==NULL) {
      stkerr(" xterminit: ",MEMNOT);
      mallfree((void *)&x11cmd);
      return 0;
   }
   strarray(); /* turning stack string into string array

   Making array of pointers to X11 command line words, argv[] style: */
   arg=tos->tex;
   *(argv)=arg;
   for(i=1;i<argc;i++) {
      *(argv+i)=1+arg+strlen(arg);
      arg=1+arg+strlen(arg);
   } /* Now argv points to X11 command line text on stk from boot.v */

/* Definitions from boot.v: */
   Name=binif(9.1);
   Class=binif(9.2);
   appdef=binif(9.3);

/* Parsing X11 command line words from argv: */
   XrmInitialize();
   XrmParseCommand(&dbCmd,options,Nopt,Name,&argc,argv);

   if(argc>1) {
      gprintf(" xterminit invalid option: %s",*(argv+1)); nc();
      stkerr("","");
      mallfree((void *)&x11cmd);
      mallfree((void *)&argv);
      mallfree((void *)&Name);
      mallfree((void *)&Class);
      mallfree((void *)&appdef);
      return 0;
   }
   mallfree((void *)&argv); /* free pointers to argv text */
   drop(); /* free argv text */

   XrmMergeDatabases(dbCmd,&db);

   if((display=getenv("DISPLAY"))==NULL) display="";
   if((home=getenv("HOME"))==NULL) home="";

   if(dbget(db,"display")) display=(char *)value.addr;
   if(dbget(db,"raise")) Raise=(On(value.addr));

   if(WTRACE) {
      gprintf(" Default X11 command: %s",x11cmd); nc();
      gprintf(" Display: %s",display); nc();
      gprintf(" Home: %s",home); nc();
      gprintf(" Name: %s",Name); nc();
      gprintf(" Class: %s",Class); nc();
      gprintf(" AppDefDir: %s",appdef); nc();
      if(!KEYS) {
         gprintf(" Default X11 command: %s\n\r",x11cmd);
      }
   }
   mallfree((void *)&x11cmd);

   Dpy=XOpenDisplay(display); /* open display: */

   if(!Dpy) {
      gprintf(" xterminit: unable to open display %s",display); 
      nc();

/*    if(sysout!=stderr) {
         fprintf(stderr," xterminit: unable to open display %s\n",
            display); 
      }
*/
      mallfree((void *)&Name);
      mallfree((void *)&Class);
      mallfree((void *)&appdef);
      return 1;
   }
   Root=DefaultRootWindow(Dpy);
   Scr=DefaultScreen(Dpy);
   Vis=DefaultVisual(Dpy,Scr);
   Dep=DefaultDepth(Dpy,Scr);

   qkeycode=XKeysymToKeycode(Dpy,XK_q); /* quit key */

/* XInternAtom() returns the id number for atom quoted; see file
   Xatoms.txt created with word XAtom for a list. */
   WM_PROTOCOLS=XInternAtom(Dpy,"WM_PROTOCOLS",False);
   WM_DELETE_WINDOW=XInternAtom(Dpy,"WM_DELETE_WINDOW",False);

/* Application default subdirectory */
   dbApp=XrmGetFileDatabase(appdef);
   XrmMergeDatabases(dbApp,&db);
   mallfree((void *)&appdef);

/* Server defaults */
   xserver=XResourceManagerString(Dpy);
   if(xserver) {
      if(WTRACE) {
         gprintf(" Server from resource: %s",xserver); nc();
      }
      dbDef=XrmGetStringDatabase(xserver);
   }
   else {
      sprintf(buf,"%s/.Xdefaults",home);
      if(WTRACE) {
         gprintf(" Server from file: %s",buf); nc();
      }
      dbDef=XrmGetFileDatabase(buf);
   }
   XrmMergeDatabases(dbDef,&db);

/* Getting XENVIRONMENT database: */
   if((env=getenv("XENVIRONMENT"))) {
      if(WTRACE) {
         gprintf(" Environment: %s",env); nc();
      }
      dbEnv=XrmGetFileDatabase(env);
   }
   else {
      if(GP_SYSTEMINFO(host)<0) {
         gprintf(" xterminit: %s failed",SYSINFO_METHOD); nc();
         stkerr("","");
         return 0;
      }
      sprintf(buf,"%s/.Xdefaults-%s",home,host);
      if(WTRACE) {
         gprintf(" Environment from file-host: %s",buf); nc();
      }
      dbEnv=XrmGetFileDatabase(buf);
   }
   XrmMergeDatabases(dbEnv,&db);

/* Sourcing file xterm.v: */
   if(!(
      pushstr("xterm.v") && 
      filefound() &&
      popint(&flag)
   )) return 0;

   if(flag) {
      if(!source()) {
         stkerr(" xterminit, file xterm.v: ",XTERMNOT);
         return 0;
      }
   }
   return(
   /* Initial settings for any window: */
      wgeometry() &&
      wwidth() &&
      wfont() &&
      wcolor()
   );
}
#endif
