/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* lib.c  April 1999

Copyright (c) 1999  D. R. Williamson
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "hash.h"
#include "inpo.h"
#include "key.h"
#include "lib.h"
#include "mat.h"
#include "math1.h"
#include "mem.h"
#include "sys.h"
#include "tag.h"
#include "tex.h"

catitem *oncatlink; /* Global ptr to last link found by caton() */

char *defword; /* pointer to text of the word being defined */

/* Parameters for BEGIN loop as it is being defined: */
int onbegin1; /* current level in stk of BEGIN loops (0 to NDBUF-1) */
int whilehit1[NDBUF];

int FENCE; /* catalog items below (inside) fence cannot be redefined */

int begind() /* BEGIN ( --- ) */
/* Runs a BEGIN ... UNTIL or BEGIN ... WHILE ... REPEAT loop.
   This is a skeleton that runs while a word is being defined. */
{
   int ret;

   if(!bufup()) return 0;

/* Simulating beginup: */
   onbegin1++;
   *(whilehit1+onbegin1)=0;

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=UNTIL;
      performd();
   }
   bufdn();
   onbegin1--; /* simulating begindn */

   if(ret==ABORT) longjmp(abortbuf,ABORT);

   return 1; /* <<-- returning to where begind was called */
}

int book(int v) /* (hA qS --- ) */ 
/* Puts stack item A into the catalog as catalog item S.  When S is 
   later seen in text, S will be found in the catalog and its stack
   item, A, will be pushed to the stack. */
{
   char *lib=NULL,*nam,stktok[16+1]={"(null)\0         \0"};
   char cattyp[5],stktyp[4];
   double x;
   int ret;

   if(stkdepth()<2) {
      stkerr(" book: ",NEEDTWO);
      return 0;
   }
   if(tos->typ!=STR) {
      stkerr(" book: ",STRNOT);
      return 0;
   }
   strchop();
   nam=tos->tex;

   if(number(nam,&x)) { 
      stkerr(" book: ",BADNAME);
      drop();
      return 0;
   }
/* If nam is to go into main (either CODE__ or DATA__), here are some 
   restrictions. */
   if((strcmp(*(cattag+onbuf),"CODE__")==0 
          && caton(tagged(nam,"DATA__"))) ||
      (strcmp(*(cattag+onbuf),"DATA__")==0 
          && caton(tagged(nam,"CODE__"))) ) {

   /* Cannot replace name that is inside FENCE: */
      if(oncat->seq<FENCE) {
         stkerr(" book: ",NAMFENCE);
         gprintf("  conflict with %s at catalog offset %d",\
            nam,oncat->seq);
         nc();
         if(INFIX)
            gprintf(\
            " book: to view the entire catalog, run: eview(catitems)");
         else
            gprintf(\
            " book: to view the entire catalog, run: catitems eview");
         nc();
         drop();
         return 0;
      }
   /* Cannot book nam into main if there is a native or constant 
      that matches: */
      if(oncat->typ==NATI) { 
         stkerr(" book: ",NATICANT);
         drop();
         return 0;
      }
      if(oncat->typ==CONS) {
         stkerr(" book: ",CONSCANT);
         drop();
         return 0;
      }
   /* The next two rules mean that an item name going into CODE__ 
      cannot match an item name in DATA__, and an item name going
      into DATA__ cannot match one in CODE__. */

   /* 1. Cannot book nam into main if it matches a catalog DEFN or
      INLI and nam type is a stack NUM, MAT, STR, or VOL: */
      if((
         oncat->typ==DEFN ||
         oncat->typ==INLI
         )&&(
         (tos-1)->typ==NUM ||
         (tos-1)->typ==MAT ||
         (tos-1)->typ==STR ||
         (tos-1)->typ==VOL
         )) {
         if(oncat->typ==DEFN) stkerr(" book: ",BOOKCANT);
         if(oncat->typ==INLI) stkerr(" book: ",BOOKICANT);
         drop();
         return 0;
      }
   /* 2. Cannot book nam into main if it matches a catalog VARI, 
      MATR, STRI or VOLU and nam type is a stack DEF or PTR: */
      if((
         oncat->typ==VARI ||
         oncat->typ==MATR ||
         oncat->typ==STRI ||
         oncat->typ==VOLU
         )&&(
         (tos-1)->typ==DEF ||
         (tos-1)->typ==PTR 
         )) {
         stkerr(" book: ",BOOKDCANT);
         drop();
         return 0;
      }
   }
   swap(); /* hA now on top of stack, nam next below */

   switch(tos->typ) {

   case DEF: /* Definitions look like natives, have main tags. */
      lib=tagged(nam,"CODE__"); /* DEF into main */

      tos->typ=VOL; /* DEF stkitem type is VOL of text */
      ret=catadd(DEFN,lib,(unsigned long (*)())exedefn,tos,v);
      strcpy(cattyp,"DEFN");
      strcpy(stktyp,"VOL");
   break;

   case NUM:
      cop(); /* this makes a copy if tos hA is already in the catalog */
      if(strcmp(*(cattag+onbuf),MAINLIB)==0)
         lib=tagged(nam,"DATA__"); /* NUM into main */
      else
         lib=tagged(nam,*(cattag+onbuf)); /* NUM into a word */

      ret=catadd(VARI,lib,(unsigned long (*)())exenumber,tos,v);
      strcpy(cattyp,"VARI");
      strcpy(stktyp,"NUM");
   break;

   case MAT:
      cop(); /* this makes a copy if tos hA is already in the catalog */
      if(strcmp(*(cattag+onbuf),MAINLIB)==0)
         lib=tagged(nam,"DATA__"); /* MAT into main */
      else
         lib=tagged(nam,*(cattag+onbuf)); /* MAT into a word */

      ret=catadd(MATR,lib,(unsigned long (*)())exestkitem,tos,v);
      strcpy(cattyp,"MATR");
      strcpy(stktyp,"MAT");
   break;

   case STR:
      cop(); /* this makes a copy if tos hA is already in the catalog */
      if(strcmp(*(cattag+onbuf),MAINLIB)==0)
         lib=tagged(nam,"DATA__"); /* STR into main */
      else
         lib=tagged(nam,*(cattag+onbuf)); /* STR into a word */

      ret=catadd(STRI,lib,(unsigned long (*)())exestring,tos,v);
      strcpy(cattyp,"STRI");
      strcpy(stktyp,"STR");
   break;

   case VOL:
      cop(); /* this makes a copy if tos hA is already in the catalog */
      if(strcmp(*(cattag+onbuf),MAINLIB)==0)
         lib=tagged(nam,"DATA__"); /* VOL into main */
      else
         lib=tagged(nam,*(cattag+onbuf)); /* VOL into a word */

      ret=catadd(VOLU,lib,(unsigned long (*)())exestkitem,tos,v);
      strcpy(cattyp,"VOLU");
      strcpy(stktyp,"VOL");
   break;

   case PTR:
      if(IN_WORD)
         lib=tagged(nam,*(cattag+onbuf)); /* inline in a word */
      else 
         lib=tagged(nam,"CODE__"); /* inline in CODE__ */
       
      if(IMMEDIATE)
         ret=catadd(INLI,lib,(unsigned long (*)())exeinline,tos,v);
      else
         ret=catadd(INLI,lib,(unsigned long (*)())exestkitem,tos,v);
      strcpy(cattyp,"INLI");
      strcpy(stktyp,"PTR");
   break;

   default:
      stkerr(" book: ",TYPNOT);
      ret=0;
   break;
   }
   if(TRACE && ret) {
      if(tos->tok) strncpy(stktok,tos->tok,16);
      gprintf(
         " word %s into catalog, type %s; stack type %s, token %s",
         lib,cattyp,stktyp,stktok); 
      nc();
   }
   lop(); /* nam gone */

   if(ret) drop();

   return ret;
}

int book1() /* book (hA qS --- ) */
{
   return(book(TRACE));
}

int bump() /* bump (x X --- ) */
/* The value of X, a NUM stored in the catalog, will be incremented
   by number x. */
{
   if(tos->typ!=NUM || (tos-1)->typ!=NUM) {
      stkerr(" bump: ",NUMS2NOT);
      return 0;
   }
   if(tos->nam==NULL || !caton2(tos->nam)) {
      gprintf(" bump: top of stack number is not in the catalog");
      nc();
      stkerr("","");
      return 0;
   }
/*
   The code here circumvents some of the program's mechanisms, leaving 
   other instances of catalog item X that are already on the stack in 
   an indeterminate state: they say they are X, but they have the value
   prior to bumping.  

   Since the program does not go by what a stack items's name is (nam 
   is in the stkitem only for convenience) this should not be a pro-
   blem--it just looks like one.

   May 2005.
   With new infix work, and parsing, the name of a stack item can be
   useful so in the future the program may rely upon a stack item's
   name.

   The revision below removes any hint of a problem, or any future
   problem, by rebooking X.

   oncat->stk->real+=(tos-1)->real;
   oncat->stk->imag+=(tos-1)->imag;

   if(is_complex(tos) || is_complex(tos-1)) set_complex(oncat->stk);

   return(drop2());
*/
   return(
      plus() &&
      pushstr(oncat->nam) &&
      book(0)
   );
}

int catadd(int type, char *word, unsigned long (*exe)(), \
   stkitem *stk, int verbose)
/* Adds catitem type, word, and elements to the catalog. */
{
   catitem *p;
   int new=0;

/* Setting nam, seq, and nex: */
   if((p=catloc(word,&new))==NULL) return 0;

   if(type!=NATI && type!=CONS) {
      if(p->typ==NATI) {
         stkerr(" catadd: ",NATICANT);
         return 0;
      }
      if(p->typ==CONS) {
         stkerr(" catadd: ",CONSCANT);
         return 0;
      }
      if(verbose && CATMSG) {
         if(new) {
         /* Showing only if !TRACE, since book() will show if TRACE: */
            if(!TRACE) {
               if(!strcmp(*(cattag+onbuf),MAINLIB)) {
                  gprintf(" word %s into catalog",untagged(word)); 
               }
               else {
                  gprintf(" word %s into catalog",word); 
               }
               nc();
            }
         }
         else {
            if(!TRACE) {
               if(!strcmp(*(cattag+onbuf),MAINLIB)) {
                  gprintf(" word %s is redefined",untagged(word));
               }
               else {
                  gprintf(" word %s is redefined",word);
               }
               nc();
            }
         }
      } 
   }
   if(!new) {
   /* Freeing catalog stkitem p.stk, which is being replaced by 
      incoming stk: */
      catfree(&p,1);
   }
/* Setting typ, exe, and stk: */
   p->typ=type;
   p->exe=exe;
   if(stk==NULL) {
      p->stk=NULL;
      return 1; /* returning if no stack item */
   }
   (*(stk->cntc))++; /* incrementing stkitem's catalog count */

/* Copying stack item into catalog item: */
   if((p->stk=(stkitem *)malloc(sizeof(stkitem)))==NULL) {
      stkerr(" catadd: ",MEMNOT);
      return 0;
   }
   memcpy(p->stk,stk,sizeof(stkitem));
   return 1;
}

int catbins() /* catbins ( --- hT) */
/* Displaying the contents of the catalog, by bins. */
{
   register int sp=2,k=0,wid=70;
   register catitem *c;
   register char *s;
   char *cbins="_catbins";
   int nchars;
   FILE *fp;

   if((fp=fopen(SYSCRCH,"w+"))==NULL) {
      gprintf(" catbins: cannot open file %s",SYSCRCH); nc();
      stkerr("","");
      return 0;
   }
   fprintf(fp,
      " The catalog contains %d words dispersed among %d hash bins",
      catwords,CATBINS
   );
   for(;k<CATBINS;k++) {
      if((c=catalog+k)->nam) {

         nchars=0;
         nchars+=fprintf(fp,"\n\r");

         s=c->nam;

         if(*s) nchars+=fprintf(fp," bin %d words: ",k);

         while(c && (s=c->nam)) {
            if(sp+(nchars+strlen(s))>wid) {

               fprintf(fp,"\n\r");
               nchars=0;
               nchars+=fprintf(fp,"   ");

            }
         /* sp spaces follow s: */
            if(*s) nchars+=fprintf(fp,"%s  ",s); 

            c=(catitem *)c->nex;
         }
      }
   }
   fclose(fp);

   return(
      pushstr(SYSCRCH) && 
      dup1s() && 
      asciiload() &&
      notrailing() &&
      swap() &&
      delete() &&
      pushstr(cbins) &&
      naming()
   );
}

void catfree(catitem **catptr, int len)
/* This function, and function drop(), control the freeing of all mal-
   locked memory. 

   Frees the stkitems associated with each catalog item in a vector of 
   catalog items.  Each freed catalog catitem remains in the linked list
   of the catalog, but is made inactive.  Future use of the item's name
   for an item being added to the catalog will reuse it.

   Allocated memory for a stkitem is freed if no copies of it are on the
   stack and if no copies of it are elsewhere in the catalog, i.e., when
   stack count, cnt, and catalog count, cntc, both equal zero.

   If a stkitem's catalog count, cntc, decrements to zero here but its
   stack count, cnt, is nonzero, it will not be freed.  It will be freed
   later by drop() when its stack count finally decrements to zero as
   it falls off the stack. */ 
{
   int i=0;
   catitem *p;
   stkitem *stk;

   for(;i<len;i++) {
      p=*(catptr+i);

      if((stk=p->stk)!=NULL) {

         (*(stk->cntc))--; /* decrementing catalog count */

      /* Freeing stack item if its catalog and stack counts are
         both zero: */
         if(*(stk->cntc)==0 && *(stk->cnt)==0) {

            if(TRACE) {
               gprintf(
                  " catfree: freeing mem of stack item %s in word %s", \
                  stk->tok,p->nam);
               nc();
            }
         /* Freeing the stack item's elements as in drop(): */
            mallfree((void *)&stk->tok);
            mallfree((void *)&stk->mat);
            mallfree((void *)&stk->tex);
            mallfree((void *)&stk->ptr);
            mallfree((void *)&stk->cnt);
            mallfree((void *)&stk->cntc);
         /* Never free S.nam: it is a catalog name, and names are 
            never deleted from the catalog. */
         }
         mallfree((void *)&stk);
         p->stk=NULL;
      }
   /* This typ and exe make catitem inactive: */
      p->typ=UNK;
      p->exe=(unsigned long (*)())exefree;
   }
}

catitem *catfetch(char *name)
/* Fetching catalog pointer for name. */
{
   if(!caton2(name)) { /* caton(name) points oncat to the catitem */

   /* Failed to find name.  Add main data tag before giving up: */
      if(!caton(tagged(name,"DATA__"))) {
         gprintf(" %s is not in the catalog",name); nc();
         return NULL;
      }
   } /* oncat, set by caton(), is the catitem sought: */
   return oncat;
}

void catinit()
/* Initializes the catalog. */
{
   register int k=0;

   if(TRACE) {
      gprintf(" initializing catalog"); nc();
   }
   *DEFNAM='\0';
   FENCE=0;

   catwords=0;

   for(;k<CATBINS;k++) { /* first entry in each hash bin: */
      catalog[k].typ=UNK;
      catalog[k].nam=NULL;
      catalog[k].exe=NULL;
      catalog[k].stk=NULL;
      catalog[k].seq=-1;
      catalog[k].nex=NULL; /* next in linked list */
   }
   ptrnum=patnew(128,16,sizeof(unsigned long));

   *(LIBID)=',';
   *(LIBID+1)=' '; /* this byte is set in libset() */
   *(LIBID+2)=':';
   *(LIBID+3)='\0';
}

int catitems() /* _catitems (k --- hT) */
/* Create a volume showing the type and name of the kth thru last items
   in the catalog.  Items are ordered in the sequence they were placed 
   into the catalog. */
{
   register catitem *c;
   register char *s,*T;
   register double *N,*Typ;
   register int i=0,k=0;
   const int width=80;
   int k0;
   char *catnam="_catitems";

/* This list matches enum cattyp in lib.h: */
   char *types="UNK,NATI,DEFN,CONS,VARI,MATR,STRI,VOLU,INLI";

   if(!popint(&k0)) return 0;

   if(!volstk(catwords,width,catnam)) return 0;
   T=tos->tex;
   memset(T,' ',catwords*width);

   if(!matstk(catwords,1,"_cattyps")) return 0;
   Typ=tos->mat;

   if(!matstk(catwords,1,"_catnums")) return 0;
   N=tos->mat;

   for(;k<CATBINS;k++) {
      if((c=catalog+k)->nam) {
         while(c && (s=c->nam)) {
            if((c->seq)>=k0) {
               memcpy(T,s,MIN(strlen(s),width));
               T+=width;
               *(N+i)=XBASE+(c->seq);
               *(Typ+i)=XBASE+(c->typ);
               i++;
            }
            c=(catitem *)c->nex;
         }
      }
   }
   if(i<=0) {
      return(
         drop2() &&
         drop() &&
         volstk(0,0,catnam)
      );
   }
   else { 
      tos->row=i;
      (tos-1)->row=i;
      (tos-2)->row=i;
      return(
   /* sequence */
      pushint(XBASE) &&
      pushint(i) &&
      items() &&
      park() &&
      pushint(xTRUE) &&
      sort1() &&

   /* sequence list */
      pushint(XBASE+1) &&
      catch() &&
      rev() &&

  /* type names */
      pushstr(types) &&
      words() &&
      swap() &&
      reach() && /* names of types */
      other() &&
      reach() &&  /* names of types in sequence */
      notrailing() &&
      pushint(5) &&
      blpad() && /* to width of 5 chars */
      rev() &&

      swap() &&
      reach() && /* words in sequence */
      asciify() &&
      notrailing() &&

      park() && /* park type and word */
      pushint(tos->row) &&

   /* word count, 0-based like catalog.  Running this phrase:
      "0 swap items '%.0f' format chop right justify" */
      pushint(0) &&
      swap() &&
      items() &&
      pushstr("%.0f") &&
      format() &&
      chop() &&
      true() &&
      justify() &&

      pushstr(" ") &&
      tail() &&
      swap() &&
      park() && /* park leading count */

      pushstr(catnam) &&
      naming()
   );
   }
}

catitem *catloc(char *word, int *new)
/* If word is found in catalog: sets global catalog pointer, oncat, to 
   catalog item of word, sets new to 0 and returns oncat.

   If word is not in catalog: leaves global oncat alone, adds word to 
   catalog, sets new to 1, and returns catalog ptr to new word in the
   catalog. */
{
   catitem *c;
   char *nam;

   if(caton(word)) {
      *new=0;
      return oncat;
   }
   if(oncatlink==NULL) {
      stkerr(" catloc: ",WORDEMPT);
      return NULL;
   }
   if((nam=(char *)memgetn(word,strlen(word)))==NULL) {
      stkerr(" catloc: ",MEMNOT);
      return NULL;
   }
   if(oncatlink->nam) {
      if((c=(catitem *)malloc(sizeof(catitem)))==NULL) {
         stkerr(" catloc: ",MEMNOT);
         return NULL;
      }
      oncatlink->nex=c; /* setting pointer to new catitem link */
   }
   else c=oncatlink; /* oncatlink is first in this hash bin */

   c->typ=UNK;
   c->nam=nam;
   c->exe=NULL;

   c->exe1=NULL;
   c->nam2=NULL;

   c->stk=NULL;
   c->seq=catwords;
   c->nex=NULL;

   catwords++;
   *new=1;
   return c;
}

int catnames() /* catnames ( --- hT) */
/* Creating a volume of all item names in the catalog.  Names are 
   ordered in their sequence in the catalog hash bins, a random 
   ordering. */
{
   register catitem *c;
   register char *s,*T;
   register int k=0;
   const int width=80;
   char *cnames="_catnames";

   if(!volstk(catwords,width,"_catnames")) return 0;
   T=tos->tex;
   memset(T,' ',catwords*width);

   for(;k<CATBINS;k++) {
      if((c=catalog+k)->nam) {
         while(c && (s=c->nam)) {
            if(*s) {
               memcpy(T,s,MIN(strlen(s),width));
               T+=width;
            }
            c=(catitem *)c->nex;
         }
      }
   }
   return(
      notrailing() && 
      noblanklines() && 
      pushstr(cnames) &&
      naming()
   );
}

int caton(char *word)
/* Incoming word is a name with a connected catalog tag.  If word is in
   the catalog, global catitem pointer, oncat, is set to the catitem 
   struct.  If word is not in the catalog, oncat is unchanged.

   Sets global pointer oncat to the pointer of catitem word if it is in
   the catalog, and returns 1.  

   Returns 0 if word not found: oncat is unchanged, and global pointer
   oncatlink is set to the last catalog pointer in the link.  

   When 0 is returned, a calling function can test name, oncatlink.nam,
   for null.  

   If oncatlink.nam is null, oncatlink is the beginning link in an (as
   yet) empty hash chain, and is where the first entry would go.  

   Otherwise, oncatlink.nam is the name of the last item currently in
   the hash chain, and a new item can be linked to its .nex element,
   oncatlink.nex, as is done in catloc(). */
{
   register catitem *c,*c0;

   if(word) {
/*
      if(TRACE) gprintf(" caton: %s\n",word);
*/
      c0=catalog+binh(word,CATBINS);
      c=c0;
      while(c && c->nam) { /* traversing catalog bin's linked list */

         if(strcmp(word,c->nam)==0) {
            oncat=c;
            return 1;
         }
         c0=c;
         c=(catitem *)c0->nex; /* eventually, c becomes NULL here */
      }
      oncatlink=c0; /* last item in hash chain */
      return 0;
   }
   oncatlink=NULL;
   return 0;
}

int caton2(char *item)
/* Implements the program's search strategy: 

      Looks first for item booked in the local library defined by 
      the tag at the current run level, cattag+onbuf.  

      Failing to find item in the local library, looks next in the
      CODE__ region of main for a native word, constant, define or 
      inline word to fire.

   Main consists of two regions: DATA__ and CODE__.  By making DATA__
   the "local" library of main, this function also works for data 
   stored in main.

   Incoming item is a catalog name with no catalog tag.

   When running in main, the context tag, cattag+onbuf, is set to 
   DATA__.  Here is context being set in ctrl.h, where MAINLIB has
   been equated to DATA__ in lib.h:

      *(cattag+onbuf)=MAINLIB; // context() can reset this default

   In this case, caton2() works by looking in DATA__ and then in 
   CODE__.

   For example, with trace activated (and trace code in caton() uncom-
   mented) this shows operation during start up while running in main:

       next word: fontadd
       caton: fontadd,1:DATA__
       caton: fontadd,1:CODE__
       exec definition fontadd,1:CODE__, lib tag DATA__
      ---- Begin run level 3; stack depth = 1 -----
       next word: strchop
       caton: strchop,1:DATA__
       caton: strchop,1:CODE__
       next word: this
       caton: this,1:DATA__
       caton: this,1:CODE__
       next word: fontfcb
       caton: fontfcb,1:DATA__
       caton: fontfcb,1:CODE__
       exec definition fontfcb,1:CODE__, lib tag DATA__
      ---- Begin run level 4; stack depth = 2 -----
       next word: strchop
       caton: strchop,1:DATA__
       caton: strchop,1:CODE__
       next word: lowercase

   This shows that by having MAINLIB set to DATA__, caton2() first
   looks in DATA__ (because it is the "local" library for main)
   and then looks in CODE__. 

   Function book() ensures that the same name is never placed into
   both DATA__ and CODE__. */
{
   return(item!=NULL && (strlen(item)>0) && (

   /* Booked item stored in local: */
      caton(tagged(item,*(cattag+onbuf))) || 

   /* Native, constant, define, or inline (all are in CODE__): */
      caton(tagged(item,"CODE__")))
   );
}

int caton3(char *item)
/* Incoming item is a catalog name with no catalog tag.

   Same as caton2() except also searches in DATA__ for booked stack 
   items.

   Searches for item in the current local library, then in DATA__,
   and then looks for a native word, constant, define, or inline in 
   CODE__.

   This special word does an exhaustive search of main, and is used 
   by brak() and list() in ctrl.c, and catseq() and ptr() in lib.c. */
{
   return(item!=NULL && (strlen(item)>0) && (

   /* Booked item stored in local: */
      caton(tagged(item,*(cattag+onbuf))) ||

   /* Booked item stored in DATA__ */
      caton(tagged(item,"DATA__")) || 

   /* Native, constant, define, or inline (all are in CODE__): */
      caton(tagged(item,"CODE__")))
   );
}

int catsearch1() /* catsearch (qS --- hT) */
/* Search the catalog for matches to S, and return them in T. */
{
   char *S;
   int len;

   if(tos->typ!=STR) {
      stkerr(" catsearch: ",STRNOT);
      return 0;
   }
   strchop();
   S=tos->tex;
   len=tos->col;

   pushint(0);
   catitems(); /* table of catalog names on stack */
   pushint(2+XBASE); /* index to 3rd string */
   string(); 
   drop(); /* ignore flag from string() */
   dup1s();
   pushq2(S,len);
   grepr(); /* searching for S */
   reach(); /* get rows that contain a match */
   notrailing();
   if(!tos->row) {
      drop();
      pushint(1);
      pushint(0);
      blockofblanks(); /* return empty T if no match */
   }
   return(
      lop() && /* S off the stack */
      pushstr("_catsearch") &&
      naming()
   );
}

int catseq() /* catseq (qS --- n) or (hT --- hA) */
/* Location n of word S within the catalog sequence; n is a 0-based 
   index.  Location n is a number in the sequence of names given by 
   word catitems. 
   If a VOL of names is incoming in T, returned A is an array of
   sequence numbers. */
{
   char *S,*T;
   double *A;
   int chars,k=0,rows;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" catseq: ",STRORVOLNOT);
      return 0;
   }
   words();
   T=tos->tex;
   chars=tos->col;
   rows=tos->row;

   if((S=memgetc(1,chars))==NULL) return 0;

   if(!matstk(rows,1,"_catseq")) return 0;
   A=tos->mat;

   for(;k<rows;k++) {
      memcpy(S,T,chars);
      strchop1(S);

      if(caton3(S)) *A=oncat->seq;
      else *A=-1;

      T+=chars;
      A++;
   }
   if(rows==1) {
      pushint(*(tos->mat));
      lop();
   }
   mallfree((void *)&S);
   return(lop());
}

int constant(char *word, char *num)
/* Adds constant number and its word to the main library.  Like a native
   word, this constant number cannot be replaced in the catalog. */
{
   double d;

   if(!number(num,&d)) {
      stkerr(" constant: ",STRNOTNUM);
      return 0;
   }
   return(
      push(NUM,NULL,NOTAG,d,NULL,NULL,0,0,NULL) &&
      catadd(CONS,tagnative(word),\
         (unsigned long (*)())exenumber,tos,0) && drop() 
   );
}

int container() /* _container (qWord --- ) */
/* This word creates a container word.

   A container word has the stack diagram ( --- qWord), that is, it
   simply puts its name on the stack when it runs. 

   Example: see man container. 

   Container words were called "library words" made by word library.
   But since all words have libraries, container is more descriptive.  
   Word library remains as a synonym for container. */
{
   char *name="container";

/* Here is the phrase that makes the inline: */
   char *make="'container' inlinex";

/* Here is the text of the inline being made, word container: */
   char *text="\
      \"[ defname (qWord) 'NAME' book ] NAME\" (qT)\
      swap (qWord) (qT qWord) inlinex"; 

   if(!caton(tagnative(name))) {
      pushstr(text);
      pushstr(make);
      xmain(0); /* making inline called container */
   }
   return(
      pushstr(name) && /* running inline called container */
      xmain(0)
   );
}

void context(char *tag)
/* Setting tag for current run level, onbuf.  All stack items placed 
   into the catalog will have tag appended to their names.

   Tag has been set previously to the name of the word that is running
   at this run level.

   After setting context (cattag) here, the calls to function tagged() 
   that use it will restrict catalog searches to just this word's local
   library. 

   Warning: function public() resets context to MAINLIB, and therefore 
   should not be used during creation of a word. */
{
   if(tag) {
      *(cattag+onbuf)=(char *)tag;
   } else {
      *(cattag+onbuf)=MAINLIB;
   }
}

int cprops() /* cprops (qS --- ) */
/* Showing properties of any catalog item name given in quote-string S.
   If the catalog item is in the local library of a word, its full
   library name is required in string S, like this example

      [tops@gutter] ready > "ok?,@strinp" cprops
       ok?,@strinp is a string; catalog ptr: 83E90E8; exe ptr: 8054B7C
       STR: tok (null), val 0, mat 0, tex 83E9060
            row 1, col 10, ptr 0, cnt 1, cntc 1, nam ok?,@strinp

   showing props of stkitem ok? in the local library of word strinp.

   And strinp is a defined word as this shows:

      [tops@gutter] ready > 'strinp' cprops
       strinp is a defined word; catalog ptr: 83E7538; exe ptr: 805490C
       VOL: tok strinp, val 0, mat 0, tex 83E7468
            row 1, col 105, ptr 0, cnt 1, cntc 1, nam strinp

   Defined words, as strinp above shows, contain a VOL stack item which
   is the text of the word. */
{
   char *S;
   int CTYPE;

   if(tos->typ!=STR) {
      stkerr(" cprops: ",STRNOT);
      return 0;
   }
   strchop();
   S=tos->tex;

   dup1s();
   if(!ctype() || !popint(&CTYPE)) {
      drop();
      return 0;
   }
   switch(CTYPE) {

      case NATI:
         gprintf(" %s is a native word:",oncat->nam);
         nc();
         gprintf("   seq: %d; cat ptr: %lX; exe ptr: %lX", \
            oncat->seq,oncat,oncat->exe);
         nc();
      return(drop());

      case DEFN:
         gprintf(" %s is a defined word:",oncat->nam);
      break;

      case CONS:
         gprintf(" %s is a native constant:",oncat->nam);
      break;

      case VARI:
         gprintf(" %s is a number variable:",oncat->nam);
      break;

      case MATR:
         gprintf(" %s is a matrix:",oncat->nam);
      break;

      case STRI:
         gprintf(" %s is a string:",oncat->nam);
      break;

      case VOLU:
         gprintf(" %s is a volume:",oncat->nam);
      break;

      case INLI:
         gprintf(" %s is an inline:",oncat->nam);
      break;

      case UNK:
         gprintf(" %s is not in library %d",S,onlib);
         nc();
         pushint(0);
         catitems(); /* table of catalog names on stack */
         dup1s();
         pushint(2+XBASE); /* index to 3rd string */
         word(); 
         drop(); /* ignore flag from word() */
         pushstr(S);
         grepr(); /* searching for S */
         reach(); /* get rows that contain a match */
         notrailing();
         if(tos->row) {
            gprintf(\
               " pattern %s matches in the following catalog items:",S);
            nc();
            pushint(3);
            indent();
            dot();
            nc();
         }
         else drop();
         return(drop()); /* done with qS */
      break;
   }
   nc();
   gprintf("   seq: %d; cat ptr: %lX; exe ptr: %lX", \
      oncat->seq,oncat,oncat->exe);
   nc();
   return(
      drop() && /* done with qS */
      exestkitem() && /* pushing oncat->stkitem to stack */
      props() /* showing props of item on stack too */
   );
}

int ctype() /* ctype (qS --- N) */
/* Catalog type of name S; returned N matches a value in the enum:
      enum cattyp {UNK=0,NATI,DEFN,CONS,VARI,MATR,STRI,VOLU,INLI}; */
{
   int found=0;

   if(tos->typ!=STR) {
      stkerr(" ctype: ",STRNOT);
      return 0;
   }
   strchop();

   if(tagon(tos->tex)) found=caton(tos->tex);
   else found=caton2(tos->tex);

   if(!found) {
      pushint(UNK);
      return(lop());
   }
/* From caton(), oncat is catitem, oncat->stk is its stkitem */
   if((oncat->typ)==NATI) {
   /* A complication from stkitems posing as NATI: a true native 
      word has no stack item; if its does (as does INLI ARGV, made 
      by setargs() in main.c), assume an INLI type: */

      if((oncat->stk)!=NULL) {
         pushint(INLI);
      }
      else {
         pushint(NATI);
      }
   }
   else {
      pushint(oncat->typ);
   }
   return(lop());
}

int define() /* define: ( --- ) */
{
   return(define1(0));
}

int define1(int inli) 
/* Creates a definition from upcoming tokens until word end, and places
   it into the catalog. 

   Incoming inli = 0 if call is from define() (for word define:) 
                 = 1 if call is from inline2() (for word inline:) */
{
   int pend[4]={LOOP,THEN,UNTIL,REPEAT};
   char n=4,*s,c;
   int err,i,len,ret;
   char text[30]={' ','d','e','f','i','n','e',':',' ',
                  '\0','\0','\0','\0','\0','\0','\0','\0',
                  '\0','\0','\0','\0','\0','\0','\0','\0',
                  '.','.','.','\0'};

   if(!bufup()) return 0;  

   if(!wordadd(NULL)) { /* initializing function wordadd */
      bufdn(); 
      return 0;
   }
   err=stkerrcount;
   onbegin1=-1;
   tdepth1=0;

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=DEFINE;
      ret=ERRC;
      performd(); /* returns here, not above, when error */
   }
   memcpy((text+9),DEFNAM,(len=MIN(16,strlen(DEFNAM))));

   if(ret!=DEFINE) {
      if(ret==ABORT) {
         bufdn();
         mallfree((void *)&defword); 
         longjmp(abortbuf,ABORT);
      }
      stkerr(text,DEFNOT);
      c=0;
      for(i=0;i<n;i++) if(*(jmpready+onbuf)==*(pend+i)) c=1; 
      if(c) stkerr("   ",PENDING);
      stkerr("   ",NEEDEND); 
      bufdn();
      mallfree((void *)&defword); 
      return 0;
   }
   if(tdepth1 && STKBAL) {
      stkerr(" define: ",LOCSTKNOT);
   }
   if(err==stkerrcount) { 
      bufdn();
      if((s=(char *)memgetn(defword,strlen(defword)))==NULL) {
         stkerr(" define: ",MEMRESIZE);
         mallfree((void *)&defword); 
         return 0;
      }
      mallfree((void *)&defword); /* volume DEFTAG onto stack: */
      if(!pushtex((char *)memgetn(DEFTAG,strlen(DEFTAG)),s,1,strlen(s)) 
         ) return 0
      ;
      if(!inli) {
         tos->typ=DEF; /* the only time this is type DEF; after booking
                          into catalog it will be stack type VOL */
         ret=( /* word DEFNAM into catalog: */
            pushstr(DEFNAM) && 
            book(1)
         );
         if(!ret) {
            stkerr(text,DEFNOT);
            drop();
            stkerrcount=err;
            return 0; 
         }
         if(TRACE) { /* displaying VOL stkitem text of new catitem */
            if(!caton(tagnative(DEFNAM))) {
               stkerr(" define1: ","new catitem not found");
               return 0;
            }
            pushstr(oncat->stk->tex);
            gprintf(" text of word %s: ",DEFNAM);
            asciify(); sp(); dot(); nc(); 
         }
         *DEFNAM='\0'; /* deactivate name; inline1() looks at it */
         return ret;
      }
      else { /* put name on stack, send volume to inline1() */
         ret=(ret && pushstr(DEFNAM));
         *DEFNAM='\0'; /* inline1() not working above define: */
         return ret;
      }
   }
   stkerr(text,DEFNOT);
   bufdn();
   mallfree((void *)&defword); 
   stkerrcount=err;
   return 0;
}

int definenot() /* define: ( --- ) */
/* Halts creation of a definition when nesting is incorrect. */
{
   stkerr(" define: ",DEFNEST);
   ip=ipend; /* jump to end of source buffer */
   return 0;
}

int defname() /* defname ( --- qS) */
{
   return(pushstr(DEFNAM));
}

int dojmpd() /* LOOP ( --- ) or +LOOP (n --- ) */
/* Jumps to doloopd; never returns */
{
   if(*(jmpready+onbuf)==LOOP) {
      longjmp(*(jmpenv+onbuf),LOOP);
   }
   stkerr(" LOOP or +LOOP: ",BADLOOP);
   return 0;
}

int doloopd() /* DO (n2 n1 --- ) and ?DO (n2 n1 --- ) */
/* Runs a DO ... LOOP from index n1 to index n2-1 (0-based).
   This is a skeleton that runs while a word is being defined. */
{
   int ret;

   if(!bufup()) return 0;

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=LOOP;
      performd(); /* when error, will return here */
   }
   bufdn();
   if(ret==ABORT) longjmp(abortbuf,ABORT);
   return 1; /* <<-- returning to where doloopd was called */
}

int end1d() /* end ( --- ) */
/* Determines action of word end while a definition is being created.
   Calls end1() to set endbuf[onbuf] to 1.  If endbuf[onbuf] is 0 (not
   valid end), run levels are unwound to get back to the level of
   define(). */
{
   end1();

   if(!*(endbuf+onbuf)) {
      stkerr(" end1d: ",PENDING);
      report(token); /* not end while define: */
      bufunwind(DEFINE,DEFINE,0); /* makes longjmp to define() */
   }
   return 1;
}

int _exists(char *S)
/* Returns true if S is a booked catalog item in the local library of 
   the running word (if not in a running word, then local library is the
   main library), or if S is the name of a native word, a constant or a
   defined word (these three types are always in the main library). */
{
   catitem *csav;
   int f=0;

   csav=oncat;
   f=caton2(S); /* performing the search described in exists() */
   oncat=csav;
   return f;
}

int exists() /* exists? (qS --- f) */
/* Pushes true flag to stack if S is a booked catalog item in the local
   library of the running word (if not in a running word, then local li-
   brary is the main library), or if S is the name of a native word, a 
   constant or a defined word (these three types are always in the main
   library). */
{
   if(tos->typ==STR) {
      strchop();
      if(_exists(tos->tex)) pushint(xTRUE);
      else pushint(xFALSE);
      return(lop());
   }
   stkerr(" exists?: ",STRNOT);
   return 0;
}

int extract() /* extract (qW qS --- hA) yank (qW qS --- hA) */
/* Extract stack item A that is stored in the library of word W under
   the name S.  Equivalent to phrase: tagsep rot cat cat ptr exe.

   Local substitutions for S and W are allowed, causing this behavior
   (analogous to a pointer to a pointer):
      Strings named W and S in the local library of the word running 
      this function (not word W) take precedence, i.e., if there is a 
      string in the local library called S with chars ABC, then ABC 
      will be sought in word W; or, if there is a string called W with 
      chars MOV, then the item named S will be sought from word MOV. */
{
   char *p;
   int len,lensep=strlen(LIBID),ret=0;

   if(tos->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" extract: ",STRSNOT2);
      return 0;
   }
   swap();
   strchop();
   strlocal(*(cattag+onbuf)); /* local substitution for string W */

/* Make local substitution for string S if there is not an item S in
   word W: */
   swap();
   strchop();
   if(!caton(tagged(tos->tex,(tos-1)->tex))) 
      strlocal(*(cattag+onbuf)); /* local substitution for string S */

   len=lensep+tos->col+(tos-1)->col;
   if((p=malloc(1+len))==NULL) {
      stkerr(" extract: ",MEMNOT);
      return 0;
   }
   *p='\0';

/* Making catalog name like A+LIBID+W: */
   strcat(p,tos->tex);
   strcat(p,LIBID);
   strcat(p,(tos-1)->tex);

   ret=(
      drop2() &&
      pushq2(p,len) &&
      ptr() &&
      exe()
   );
   mallfree((void *)&p);
   return(ret);
}

int fence() /* fence ( --- ) */
{
   FENCE=catwords;
   return 1;
}

int fence_at() /* fence_at ( --- n) */
{
   return(pushint(FENCE));
}

int global() /* global (qA --- ) */
/* Declare A to be a global variable.  Global variables are stored in
   the library of word DATA__, and are seen by all inlines. 
   This word stores a 0-by-0 MAT for the initial A. */
{
   if(tos->typ!=STR) {
      stkerr(" global: ",STRNOT);
      return 0;
   }
   if(!matstk(0,0,tos->tex)) return 0;

   return(
      swap() &&
      pushstr("DATA__") &&
      swap() &&
      implant()
   );
}

int ifbranchd() /* IF (f --- ) */
/* Runs an IF ... ELSE ... THEN structure.
   This is a skeleton that runs while a word is being defined. 
   It looks at both branches--the words that follow IF and the words
   that follow ELSE--so errors in either may be caught. */
{
   int ret;

   if(!bufup()) return 0;

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */

   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=THEN;
      *(ctrlactive+onbuf)=0; /* force to run the words that follow IF */
      performd(); /* when IF error, will return here */
   }
   if(ret==THEN) {
      bufdn();
      return 1; /* <<-- returning to where ifbranchd was called */
   }
   if(ret==ELSE) {
      *(ctrlactive+onbuf)=0; /* now force to run words after ELSE */
   }
   if(ret==ABORT) {
      bufdn();
      longjmp(abortbuf,ABORT);
   }
   performd(); /* when ELSE error, will return here */
   return 1;
}

int implant() /* implant (hA qW qS --- )  bank (hA qW qS --- ) */
/* Literal implant.
   Works the same as book except A goes into the library of word W,
   under the name S, instead into of the library of the running word. */
{
   char *context_save[NBUF];
   int fence,IMMsav,in_word_sav,ret;

   if(tos->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" implant: ",STRSNOT2);
      return 0;
   }
   in_word_sav=IN_WORD;
   IMMsav=IMMEDIATE; 

   IN_WORD=1; /* required if A it type PTR */
   IMMEDIATE=1; /* if A is type PTR, it will run when yanked like a 
                   macro, even if it was made with function inline(). */
   strchop();
   swap();
   strchop();
   swap();

   *(context_save+onbuf)=*(cattag+onbuf);

   if(!caton(tagnative((tos-1)->tex))) {
      gprintf(" implant: %s: ",tagnative((tos-1)->tex));
      gprintf(CATNOTW);
      nc();
      stkerr("","");
      *(cattag+onbuf)=*(context_save+onbuf);
      return 0;
   }
   if(TRACE) {
      gprintf(" implant into lib of %s, lib tag %s", \
         oncat->nam,oncat->stk->tok);
      nc();
   }
   context(oncat->stk->tok);

/* To allow changing items in the library of protected words,
   lower the fence: */
   fence=FENCE;
   FENCE=0;
   ret=(
      lop() &&
      book(0)
   );
   FENCE=fence;

   *(cattag+onbuf)=*(context_save+onbuf);
   IN_WORD=in_word_sav;
   IMMEDIATE=IMMsav;

   return ret;
}
   
int implant2() /* bank* (hA qW qS --- ) */
/* Context implant, defined by forms A.B from parser.

   Works the same as book except A is placed into the library of word 
   W, under the name S, instead of into the library of the running word.

   Local substitutions for S and W are allowed, causing this behavior
   (analogous to a pointer to a pointer):
      Strings named W and S in the local library of the word running 
      this function (not word W where A is being placed) take prece-
      dence, i.e., if there is a string in the local library called S 
      with chars ABC, then ABC will be the name of A placed in word W; 
      or, if there is a string called W with chars MOV, then A will be 
      placed into word MOV under the name S. */
{
   char *context_save[NBUF];
   int fence,ret;

   if(tos->typ!=STR || (tos-1)->typ!=STR) {
      stkerr(" implant1: ",STRSNOT2);
      return 0;
   }
   strlocal(*(cattag+onbuf)); /* local substitution for string S */
   swap();

   strlocal(*(cattag+onbuf)); /* local substitution for string W */
   swap();

/* The rest works the same as implant(): */

   *(context_save+onbuf)=*(cattag+onbuf);

   if(!caton(tagnative((tos-1)->tex))) {
      gprintf(" implant1: %s: ",tagnative((tos-1)->tex));
      gprintf(CATNOTW);
      nc();
      stkerr("","");
      *(cattag+onbuf)=*(context_save+onbuf);
      return 0;
   }
   if(TRACE) {
      gprintf(" implant1 into lib of %s, lib tag %s", \
         oncat->nam,oncat->stk->tok);
      nc();
   }
   context(oncat->stk->tok);

/* To allow changing items in the library of protected words,
   lower the fence: */
   fence=FENCE;
   FENCE=0;
   ret=(
      lop() &&
      book(0)
   );
   FENCE=fence;

   *(cattag+onbuf)=*(context_save+onbuf);
   return ret;
}

int inline2() /* inline: ( --- ) */
/* Uses define1() to make a volume until just before it is booked into 
   the catalog, when the definition's volume and name are on the stack.
   Then calls inlinex() to transform the volume and quote string into 
   an inline, and place that into the catalog. 

   Turns STRICT off so inline1() will make unresolved references into
   quote strings that are fired by word main. */
{
   int ret=0,strict;

   strict=STRICT;
   strictoff();

   ret=(define1(1) && inlinex());

   STRICT=strict;
   return(ret);
}

int into() /* into (hA --- ) */
/* Puts stack item A into the catalog with name S that follows next in
   the source input.  

   Example:
      vector: 1 2 3 ; into S

   When S is seen later in text, the stack item of catalog item A will
   be pushed to the stack. 

   Word book preceded by S in quotes is equivalent to the example above:
      vector: 1 2 3 ; "S" book
*/ {
   if(feed()) {
      return(book(TRACE));
   }
   stkerr(" into: ",FEEDNOT);
   return 0;
}

int lcatitems() /* lcatitems (nLib --- hT) */
/* Catalog items in library nLib. */
{
   int ret=0;
   char tag[33];

   if(tos->typ!=NUM) {
      stkerr(" lcatitems: ",NUMNOT);
      return 0;
   }
/* Run word tagsep in library nLib: */
   ret=(
      pushstr("tagsep") &&
      swap() &&
      xcatlib() 
   );
   if(!ret) return 0;

   *tag='\0';
   strncat(tag,tos->tex,32);
   drop();

   ret=(
      pushint(0) &&
      catitems() &&
      dup1s() &&
      pushstr(tag) &&
      grepr()
   );
   if(ret) {
      if(tos->row) ret=reach();
      else { 
         ret=(
            drop2() &&
            pushint(0) &&
            dup1s() &&
            blockofblanks()
         );
      }
   }
   return(ret);
}

int lexists() /* lexists? (qS nLib --- f) */
/* Pushes true flag to stack if S is a catalog item in library number 
   nLib. */
{
   if(tos->typ!=NUM) {
      stkerr(" lexists?: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR) {
      stkerr(" lexists?: ",STRNOT);
      return 0;
   }
   return(
      pushstr("exists?") &&
      swap() &&
      xcatlib() 
   );
}

int libget() /* libget (qS nLib --- hA) */
/* From library number nLib, get stack item A that is stored under 
   the name S. */
{
   if(tos->typ!=NUM) {
      stkerr(" libget: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR) {
      stkerr(" libget: ",STRNOT);
      return 0;
   }
   return(
      pushstr("main") &&
      swap() &&
      xcatlib() 
   );
}

int libinit(int lib)
/* Initialize library number lib and set context to it (libset()). 
   This function is run by nativewords(). */
{
   int catmsg,i=0;

   if(TRACE) {
      gprintf(" initializing library %d",lib); 
      nc();
   }
   libset(lib); /* the context of lib */

/* Put three library words into the catalog.  These are the very first 
   catalog items in lib (word lcatitems displays the catalog of lib): */

   catmsg=CATMSG;
   CATMSG=0;

   pushstr("NUMBERS__"); /* constants that inlines use */
   pushstr("CODE__");    /* executable words in main */
   pushstr("DATA__");    /* cataloged stack items in main */

   for(;i<3;i++) {
      if(!caton(tagnative(tos->tex))) {
         pushstr("");
         swap();
         inlinex(); /* library word into catalog */
      }
      else drop();
   }
   CATMSG=catmsg;

   return 1;
}

int libpry() /* libpry (qS --- hA) */
/* Find stack item S in local library and push it to stack.  Returned
   A is a purged volume if S is not found.  For static lib items like
   NUM, MAT, VOL, PTR this word is equivalent to the phrase: ptr exe.  

   For lib items that fire, like definitions and inlines, this word
   will simply fetch their contents while 'ptr exe' will fire them.

   Native words have null stack items; returned A will be purged. */
{ 
   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" libpry: ",STRORVOLNOT);
      return 0;
   }
   strchop();

   if(caton2(tos->tex)) {
      if(oncat->typ==NATI) {
      /* A true native word has no stack item; otherwise it is a 
         stkitem posing as NATI.  See note in ctype(). */
         if((oncat->stk)==NULL) {
            /*
            stkerr(" libpry: ",NATNOT);
            return 0;
            */
            return(
               volstk(0,0,"_libpry") && /* push purged vol to stack */
               lop()
            );
         }
      }
      return(
         exestkitem() && /* pushing caton oncat to stack */
         lop()
      );
   }
   else 
      return(
         volstk(0,0,"_libpry") && /* pushing purged volume to stack */
         lop()
      );
}

int libput() /* libput (X qS nLib --- )*/
/* Put stack item X into library number nLib under name S. */
{
   if(tos->typ!=NUM) {
      stkerr(" libput: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR) {
      stkerr(" libput: ",STRNOT);
      return 0;
   }
   return(
      pushstr("book") &&
      swap() &&
      xcatlib() 
   );
}

int libset(int lib)
/* Set the global library number, onlib, and set the corresponding 
   identifier LIBID.

   String LIBID was initialized in libinit()). */
{
   if(TRACE) {
      gprintf(" libset: onlib: %d, onbuf: %d",lib,onbuf);
      nc();
   }
   if(lib<LIB0 || lib>NLIBS) {
      gprintf(" libset: library number %d is out of bounds",lib);
      nc();
      stkerr("","");
      return 0;
   }
   onlib=lib; /* set global variable */
   *(catlib+onbuf)=onlib;

/* Set the 2nd character in LIBID to the one-character symbol that is 
   the current library number, onlib. */

   switch(onlib) {
      case LIB0: *(LIBID+1)='\x30'; /* 0 */
      break;
      case LIB1: *(LIBID+1)='\x31'; /* 1 */
      break;
      case LIB2: *(LIBID+1)='\x32'; /* 2 */
      break;
      case LIB3: *(LIBID+1)='\x33'; /* 3 */
      break;
      case LIB4: *(LIBID+1)='\x34'; /* 4 */
      break;
   }
   return 1;
}

int listd() /* list: ( --- hA) */
/* Gathers upcoming numbers into a matrix column, or upcoming quote-
   strings into a 1-row volume.
   This is a skeleton that runs while a word is being defined. */
{
   int ret;

   if(!bufup()) return 0;

   ret=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp will land here */
   if(!*(jmpready+onbuf)) {
      *(jmpready+onbuf)=ENDLIST;
      performd(); /* when error, will return here */
   }
   if(ret!=ENDLIST) {
      bufdn();
      if(ret==ABORT) longjmp(abortbuf,ABORT);

      stkerr(" list: ",NEEDEND); 
      return 0;
   }
   bufdn();
   return 1; /* <<-- returning to where listd was called */
}

int localq() /* local? (qS --- f) */
/* Pushes true flag to stack if S is a booked catalog item in the local
   library of the running word. */
{
   catitem *csav;
   char *p;
   int f=0;

   if(tos->typ==STR) {
      if((p=(char *)memgetn(tos->tex,tos->col))==NULL) {
         stkerr(" local?: ", MEMNOT);
         return 0;
      }
      strchop();
      drop();
      csav=oncat;

   /* Booked item stored in local: */
      f=caton(tagged(p,*(cattag+onbuf))); 

      mallfree((void *)&p);
      oncat=csav;
      if(f) return(pushint(xTRUE));
      else return(pushint(xFALSE));
   }
   stkerr(" local?: ",STRNOT);
   return 0; 
}

int localref() /* localref (qWord qName --- qS) */
/* Make a catalog name with local library tag, for use with names sent 
   to words that make words.

   Made from this inline (but upgraded to use local substitutions 
   (like extract()), as shown in C code below):

   inline: localref (qWord qName --- qS) \ catalog S for Name in Word
      strchop tagsep rot, strchop any?
      IF cat ELSE drop tagmain THEN cat
   end */
{
   int f;

   swap();
   strchop();
   strlocal(*(cattag+onbuf)); /* local substitution for Word */

   swap();
   strchop();
   if(!caton(tagged(tos->tex,(tos-1)->tex)))
      strlocal(*(cattag+onbuf)); /* local substitution for Name */

   tagsep();
   rot();
   anyq();
   popint(&f);

   if(f) cat();
   else {
      drop();
      pushstr("tag");
      xmain(0); 
   }
   return(cat());
}

int matptr() /* matptr (hA --- ptr) */
/* Books matrix A into the catalog--giving it a unique name based on 
   its memory address--and returns ptr of stack type NUM.  Later, 
   putting ptr on the stack and firing exe will cause hA to reappear 
   on the stack.  
   Since ptr is of type NUM, it can be placed into lists of numbers 
   which are structures of various types of items.

   Primary use of matptr is for items not already in the catalog that 
   only need to be referenced through a numerical ptr and not by name.  

   Use word ptr to obtain a similar numerical ptr for a named item that
   has already been placed into the catalog. */
{
   char *fX="%X#",name[18]={0};

   if(tos->typ!=MAT && tos->typ!=VOL) {
      stkerr(" matptr: ",MATORVOLNOT);
      return 0;
   }
/* Name looks like: 80696381# for 32-bit addresses; name has a trailing
   # so it isn't mistaken for a number string; it also has a trailing
   null. */

/* Name based on mem address: */
   if(tos->typ==MAT) sprintf(name,fX,tos->mat); 
   if(tos->typ==VOL) sprintf(name,fX,tos->tex); 
   return(
      pushstr(name) &&
      book(TRACE) &&
      pushstr(name) &&
      ptr()
   );
}

int native(char *word, unsigned long (*exe)())
/* Adds native word and its execution address to the main library. */
{
/* if(TRACE) gprintf(" native: %s",tagnative(word)); */

   return(catadd(NATI,tagnative(word),exe,NULL,0));
}

int notag() /* notag (hT --- hT1) */
/* Removes the library tag from a list of names on stack. */
{ 
   char *p,*T,*T1;
   char* name="_notag";
   int chars,i=0,len;

   if(tos->typ==STR) {
      strchop();
      p=strstr(tos->tex,LIBID);

      if(p) chars=p-tos->tex;
      else chars=tos->col;

      return(pushq2(tos->tex,chars) && lop());
   }
   if(tos->typ==VOL) {

      chop();
      T=tos->tex;

      if(!volstk(tos->row,(chars=tos->col),name)) return 0;
      T1=tos->tex;

      for(;i<tos->row;i++) {
         memset(T1,' ',chars);

         p=strstr(T,LIBID);

         if(p) len=p-T;
         else len=chars;

         memcpy(T1,T,len);

         T+=chars;
         T1+=chars;
      }
      return(lop());
   }
   stkerr(" notag: ", STRORVOLNOT);
   return 0;
}

void performd()
/* Runs the control words when making a definition, skipping over
   most words. */
{
   char *token[NBUF];

   while(!*(endbuf+onbuf) && (*(token+onbuf)=tokenget())!=NULL) {

      if(TRACE) { 
         gprintf(" def next word: %s",*(token+onbuf)); nc(); }

      if(strmatch(*(token+onbuf),"}") || strmatch(*(token+onbuf),")")) {
         goto bottom; }

      if(*(token+onbuf)==0) { 
         token0(); goto bottom; }

      if(strmatch(*(token+onbuf),"[")) {
         if(!brak()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"]")) {
         if(!brakend()) report(*(token+onbuf)); goto bottom; }

   /* From tokenget(), strings and text are in mallocked tokenq. */
      if(tokenq) {
         if(!wordadd(tokenq)) {
            report(tokenq); 
            mallfree((void *)&tokenq); 
            goto bottom;
         }
         mallfree((void *)&tokenq);
      }
      else {
         if(!wordadd(*(token+onbuf))) { 
            report(*(token+onbuf)); goto bottom; 
         }
      }
      if(strmatch(*(token+onbuf),"list:")) {
         if(!listd()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"push")) {
         tdepth1++; goto bottom; }

      if(strmatch(*(token+onbuf),"pull")) {
         tdepth1--; goto bottom; }

      if(strmatch(*(token+onbuf),"DO")) {
         if(!doloopd()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"?DO")) {
         if(!doloopd()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"LOOP")) {
         if(!dojmpd()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"+LOOP")) {
         if(!dojmpd()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"define:")) {
         if(!definenot()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"IF")) {
         if(!ifbranchd()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"THEN")) {
         if(!ifthen()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"ELSE")) {
         if(!ifelse()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"BEGIN")) {
         if(!begind()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"WHILE")) {
         if(!while1d()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"REPEAT")) {
         if(!repeatd()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"UNTIL")) {
         if(!untild()) report(*(token+onbuf)); goto bottom; }

      if(strmatch(*(token+onbuf),"end") || \
         strmatch(*(token+onbuf),";")) {
         end1d(); goto bottom; }

      bottom: continue;
   }
   if(!*(endbuf+onbuf)) end1d(); /* must close out any jumps */

   if(TRACE) {
      gprintf(" Returning from performd on word %s",*(token+onbuf));
      nc();
   }
}

int ptr() /* ptr (qS --- ptr) */
/* Obtains a catalog pointer to the catalog item named S and creates a
   ptr NUM.  

   The pointer obtained, ptr, is stack type NUM.  Note that ptr is not 
   type PTR, a list of execution addresses, but ptr could be a pointer 
   to type PTR (or to any other type stored in the catalog).

   Since ptr is type NUM (an ordinary number), it can be stored in vec-
   tors and matrices with other numbers.  When placed on the stack, ptr
   can be fired by word exe (see function exe()) to produce the same 
   result as saying word S.

   To handle 64-bit addresses, the actual ptr is placed in a pattern 
   table and its 32-bit offset in the pattern table is placed into NUM.

   Functions for pattern tables use a patitem struct and are: patadd(),
   patget(), patloc(), patnew(), patprops().  The global table pointer,
   ptrnum, is the patitem table for ptr NUMs.

   The remaining 32 bits of NUM are used to authenticate it as a ptr 
   NUM.

   The token of ptr, which can be viewed with word props, will contain
   name S.

   If catalog item S is changed, NUM ptr, perhaps stored in a matrix of
   pointers, will reflect the revision when it is fired later since it 
   still points to the same place. */
{
   union {
      double x;
      catitem *oncat[sizeof(double)];
      int i[sizeof(double)];
   } X={0};
   char *tok;
   int loc;

   if(tos->typ!=STR) {
      stkerr(" ptr: ",STRNOT);
      return 0;
   }
   strchop();

/* Catalog address is returned in global catitem *oncat: */
   if(!caton3(tos->tex)) { 
      gprintf(" ptr: no catalog entry for %s",tos->tex); nc();
      stkerr("","");
      HALT(); /* stopping everything here prevents seg fault */
      return 0;
   }

/* To make it easier to spot an invalid pointer--since executing one
   is disastrous--the 32-bit locator for oncat is placed in the portion
   of the 8-byte double that does not contain the exponent.  In the por-
   tion that does contain the exponent, 31 bits are null and the sign 
   bit is set to 1.  
   To qualify a ptr before executing it, exe() will look for the non-
   zero 32-bit locator and the 32-bit pattern with the sign bit. */

/* Aids to authentication:
      kzero=0x80000000 has 31 zero bits and sign bit set to 1 
      oplus is 0 for little endian, 1 for big, offset to 32-bit loc 
      kplus is 1 for little endian, 0 for big, offset to kzero */ 

   patadd(&oncat,ptrnum,&loc); /* oncat addr into global table ptrnum */
   *(X.i+oplus)=loc; /* 32-bit loc in global table into X.i */

   *(X.i+kplus)=kzero; /* 32-bit authentication into X.i */
/*
   if(TRACE) // trace shows the entire pattern table: //
      patprops(ptrnum); 
*/
   if((tok=(char *)memgetn(tos->tex,tos->col))==NULL) return 0;

   pushd(X.x);      /* tos->tok for this new d is initially NULL */
   tos->tok=tok;    /* giving ptr the tok name of catalog item S */ 
   set_catptr(tos); /* set tag to catalog ptr type */

   return(lop());
}

int ptrq() /* ptr? (ptr --- f) */
/* Flag f is true if item on stack is a catalog ptr NUM. */
{
   catitem *c;
   union {
      double x;
      int i[sizeof(double)];
   } X={0};
   int k=-1;

   if(tos->typ!=NUM) {
      if(stkdepth()) drop();
      return(pushint(xFALSE));
   }
   if(is_catptr(tos)) return(drop() && pushint(xTRUE));

   else {
      popd(&X.x);

      k=*(X.i+kplus); /* k=kzero should be null except for sign bit */
      if(k!=kzero) return(pushint(xFALSE));

/*    Fetching pattern of catitem pointer oncat; c may be a 64-bit
      pointer: */
      c=(catitem *)patget((int)*(X.i+oplus),ptrnum);
      if(!c) return(pushint(xFALSE));

      memcpy(&oncat,c,sizeof(catitem *));

      if(oncat==0) return(pushint(xFALSE));

      return(pushint(xTRUE));
   }
}

int ptrtable() /* ptrtable ( --- ) */
/* Displaying table of ptr NUMs. */
{
   patitem *p;
   char *P;
   int bytes,k=0,rows;

   p=ptrnum;
   P=(char *)p->pat;

   gprintf(" Pattern table of ptr addresses:");
   nc();
   gprintf("  rows in use: %d",(rows=p->use));
   nc();
   gprintf("  bytes width: %d",(bytes=p->wid));
   nc();
   if(rows) {
      gprintf("  contents:");
      nc();
      for(;k<rows;k++) {
         memcpy((char *)&oncat,(char *)P,sizeof(catitem *));

         gprintf("   Row %3d: ",k);
         if(oncat) gprintf("%8lX  %s",oncat,oncat->nam);
         else gprintf("%8lX",oncat);
         nc();

         P+=bytes;
      }
   }
   return 1;
}

int ptrtok() /* ptrtok (ptr --- qS) */
/* Fetch the token name for ptr.  S is empty string if item on stack
   is not a ptr. */
{
   catitem *c;
   union {
      double x;
      int i[sizeof(double)];
   } X={0};
   int k=-1;

   if(tos->typ!=NUM) {
      if(stkdepth()) drop();
      return(pushq2("",0));
   }
   popd(&X.x);

   k=*(X.i+kplus); /* k=kzero should be null except for sign bit */
   if(k!=kzero) return(pushq2("",0));

/* Fetching pattern of catitem pointer oncat; oncat may be a 64-bit
   pointer: */
   c=(catitem *)patget((int)*(X.i+oplus),ptrnum);
   if(!c) return(pushq2("",0));

   memcpy(&oncat,c,sizeof(catitem *));
   if(oncat==0) return(pushq2("",0));

   return(pushstr(oncat->nam));
}

int quoted(char *word, char *token)
/* Adds quote-string, token, to the catalog after removing surrounding
   quotes.

   Quotes surrounding token are removed by pushq1() when token is
   pushed to the stack. */
{
   if(token && pushq1(token)) {

      return(
         pushstr(word) &&
         book(TRACE)
      );
   }
   stkerr(" quoted: ",STRNOT);
   return 0;
}

int repeatd() /* REPEAT ( --- ) */
/* Jumps to begind(); never returns.
   This is a skeleton that runs while a word is being defined. */
{
   if(*(jmpready+onbuf)==REPEAT) {
      longjmp(*(jmpenv+onbuf),REPEAT);
   }
   stkerr(" REPEAT: ",BADLOOP);
   return 0;
}

void strlocal(char *local_context) /* (qS --- qS || qS1) */
/* If string S is booked in local lib, run it and return any string S1 
   it leaves on the stack; otherwise, just return with S. */
{
   if(!tagon(tos->tex)) {
      if(caton(tagged(tos->tex,local_context))) {
         dup1s();
         local();
         if(tos->typ==STR) lop(); /* if string, use it and drop S */
         else drop();             /* keep incoming S */
      }
   }
}

int tag() /* tag ( --- qS) */
{
   char *t;

   t=tagged("\0",*(cattag+onbuf));
   return(pushstr(t));
}

char *tagged(char *word, char *CONTEXT)
/* Tags word for searching in the local library of current run level.
   Note: Contents of returned pointer change continuously; copy to
   safe storage if required. */
{
   static char wordtag[3+2*DEFNC];
   static int LEN=3+2*DEFNC;

   if(!word || !CONTEXT) return(NULL);

   if(tagon(word)) return(word); /* don't tag a tagged word */

   *(wordtag)='\0';
   strncat(wordtag,word,LEN);
   strncat(wordtag,LIBID,LEN);
   return((char *)strncat(wordtag,CONTEXT,LEN));
}

void tagmake(char *tag, char *text)
/* Makes text into a tag for the current local library.
   Note: This function is for use only during creation of an item to go
   into the catalog, as in inline1() and define() (using wordadd()).  
   Also see context(). */
{
   if(PUBLIC) {
      strcpy(tag,MAINLIB);
   }
   else {
      *tag='\0';
      strcat(tag,text);
   }
}

char *tagnative(char *word)
/* Tags word for searching for native, constant, define or inline.

   Note: Contents of returned pointer change continuously; copy to
   safe storage if required. */
{
/* Native, constant, define, or inline (all are in CODE__): */
   return(tagged(word,"CODE__"));
}

int tagon(char *word)
/* Returns 1 if word has a tag. */
{
   if(word) return(strstr(word,LIBID)!=0);
   else return 0;
}

int tagsep() /* tagsep ( --- qTAG) */
/* Pushes the program's stack item tag seperator string to the stack. */
{
   return(pushstr(LIBID));
}
 
char *untagged(char *word)
/* Remove catalog tag from word. 
   Note: Contents of returned pointer change continuously; copy to safe
   storage if required. */
{
   char *p;
   int len;
   static char notag[3+2*DEFNC];
   static int LEN=3+2*DEFNC;

   if(!tagon(word)) return(word); 

   *(notag)='\0';
   strncat(notag,word,LEN);
   p=strstr(notag,LIBID);

   if(p) {
      len=p-notag;
      *(notag+len)='\0';
   }
   return((char *)notag);
}

int untild() /* UNTIL (f --- ) */
/* Jumps to begind(); never returns.
   This is a skeleton of until() that runs while a word is being 
   defined. */
{
   if(!*(whilehit1+onbegin1) && *(jmpready+onbuf)==UNTIL) {
      longjmp(*(jmpenv+onbuf),UNTIL);
   }
   stkerr(" UNTIL: ",BADLOOP);
   return 0;
}

int variable(char *word, double d)
/* Adds number d and its word to the catalog.  Like a definition, this
   number can be redefined. */
{
   return(
      push(NUM,NULL,NOTAG,d,NULL,NULL,0,0,NULL) &&
      catadd(VARI,word,(unsigned long (*)())exenumber,tos,0) &&
      drop()
   );
}

int while1d() /* WHILE (f --- ) */
/* Controls jump in BEGIN ... WHILE ... REPEAT loop.
   This is a skeleton that runs while a word is being defined. */
{
   *(whilehit1+onbegin1)=1;
   if(*(jmpready+onbuf)==UNTIL) {
      return 1;
   }
   stkerr(" WHILE: ",BADBRANCH);
   return(bufunwind(UNTIL,ERRC,0));
}

int wordadd(char *token)
/* Adds word token to the text of the definition being created.

   First used with token==NULL to start a new definition.  */
{
   static int ipw;
   static unsigned char *crlf;
   int len;

   if(token==NULL) { /* initializes when token==NULL */
      if((defword=(char *)memgetc(1,DEFCH))==NULL) {
         stkerr(" wordadd: ",MEMNOT); 
         return 0;
      }
      ipw=-1;
      crlf=iprect; /* line of def name */
      return 1;
   }
   len=strlen(token);
   if((ipw+len+2)>DEFCH) {
      gprintf(" error during creation of word %s",DEFNAM);
      nc();
      stkerr( " wordadd: ",DEFCHMAX);
      mallfree((void *)&defword); 
/*
      This really gets lost because the program attempts to interpret 
      the remaining phrases in this lost word and then the rest of the 
      file.  

      Just long-jump quit and don't try to unwind thru define(): */ 
      halt(); 
      return 0;
   }
   if(ipw<0) {
      ipw=0; /* first token is the name of the definition */
      strncpy(DEFNAM,token,MIN(DEFNC,len)); 
      DEFNAM[MIN(DEFNC,len)]='\0';
      if(caton(DEFNAM)) {
         if(oncat->typ==NATI) {
            stkerr(" define: ",NATICANT);
            return 0;
         }
         if(oncat->typ==CONS) {
            stkerr(" define: ",CONSCANT);
            return 0;
         }
      }
   /* First token is also the local library tag: */
      tagmake(DEFTAG,DEFNAM);
      context(DEFTAG);

      return 1;
   }
   if(crlf!=iprect) { /* if new line, adding a new line char, \n */
      *(defword+ipw)='\n';
      ipw++;
      crlf=iprect;
   }
   memcpy(defword+ipw,token,len);
   ipw+=len+1;
   *(defword+ipw)='\0';
   return 1;
}
