#!/usr/bin/env perl
use warnings;
use strict;
use Getopt::Long;
use File::Basename;
# use Data::Dumper::Simple;
my $script = basename $0;
my %env  = ( # Use these environment variables if they are defined.
    "CC"     => 1, # Will be overridden if given --with-cc=<C compiler>
    "CFLAGS" => 1,
    "F77"    => 1, # Will be overridden if given --with-f77=<fortran77 compiler>
    "FFLAGS" => 1,
    "F_LIB"  => 1,
           );
# 'config' option variables {{{1
my %config = (
    "prefix"     => { 
                        "default" => "/usr/local" ,
                        "desc" => "Directory to install this package.",
                    },
    "with_cc"    => { 
                        "default" => "gcc"        ,
                        "desc" => "C compiler.",
                    },
    "with_f77"   => { 
                        "default" => "g77"        ,
                        "desc" => "Fortran compiler.",
                    },
    "with_f_lib" => { 
                        "default" => "-lg2c"      ,
                        "desc" => "Fortran runtime library.",
                    },
             );
# 1}}}
# 'with'   option variables {{{1
my %with = (
    "arpack"   => { 
                    "flag"    => "ARPACK"   , 
                    "default" => "no"       ,
                    "libs"    => "-larpack" , 
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-arpack:s" => 
                        "ARPACK eigenvalue library", 
                                 },
                  } ,
    "blas"     => { 
                    "flag"    => "BLAS"     , 
                    "default" => "yes"      ,
                    "libs"    => "-lblas"   ,
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-blas:s" => 
                        "Basic Linear Algebra Subprogram library.", 
                                 },
                  } ,
    "curses"     => { 
                    "flag"    => "CURSES"  , 
                    "default" => "yes"      ,
                    "libs"    => "-lcurses"  ,
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-curses:s" => 
                        "Command line manipulation library.", 
                                 },
                  } ,
    "dyn_load" => { 
                    "flag"    => "DYNLOAD"  , 
                    "default" => "yes"      ,
                    "libs"    => "-ldl"     ,
                    "incl"    => ""         , 
                    "extra"   => "LDFLAGS= -rdynamic -Wl,-export-dynamic",
                    "args"    => { 
                        "with-dyn_load:s" => 
                        "Dynamic load library.", 
                                 },
                  } ,
    "essl"     => { 
                    "flag"    => "ESSL"     , 
                    "default" => "no"       ,
                    "libs"    => "-lessl"   ,
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-essl:s" => 
                        "IBM's Engineering and Science Subprogram Library.", 
                                 },
                  } ,
    "fftw2"    => { 
                    "flag"    => "FFTW"     , 
                    "default" => "no"       ,
                    "libs"    => "-lrfftw -lfftw"   ,
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-fftw2:s" => 
                        "Fast Fourier Transform library (version 2).",
                        "with-fftw2_libs=s" => 
                        "Fast Fourier Transform library (version 2), library directory.",
                        "with-fftw2_incl=s" => 
                        "Fast Fourier Transform library (version 2), header file directory.",
                                 },
                  } ,
    "fftw3"    => { 
                    "flag"    => "FFTW3"    , 
                    "default" => "no"       ,
                    "libs"    => "-lfftw3"  ,
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-fftw3:s" => 
                        "Fast Fourier Transform library (version 3).",
                        "with-fftw3_libs=s" => 
                        "Fast Fourier Transform library (version 3), library directory.",
                        "with-fftw3_incl=s" => 
                        "Fast Fourier Transform library (version 3), header file directory.",
                                 },
                  } ,
    "keyboard" => { 
                    "default" => "yes"      ,
                    "libs"    => ""         ,
                    "incl"    => ""         , 
                    "args"    => { 
                        "with-keyboard:s" => 
                        "Enable keyboard handling operations. (X=yes/no)",
                                 },
                  } ,
    "lapack"   => { 
                    "flag"    => "LAPACK"   , 
                    "default" => "yes"      ,
                    "libs"    => "-llapack" ,
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-lapack:s" => 
                        "Enable functions that call the Linear Algebra library (version 3).", 
                                 },
                  } ,
    "matlab"   => { 
                    "flag"    => "MATLAB"   , 
                    "default" => "no"       ,
                    "libs"    => ""  ,
                    "incl"    => ""         , 
                    "args"    => { 
                        "with-matlab:s" => 
                        "Enable MATLAB API and gateway functions.", 
                                 },
                  } ,
    "metis"    => { 
                    "flag"    => "METIS"    , 
                    "default" => "no"       ,
                    "libs"    => "-lmetis"  ,
                    "incl"    => ""         , 
                    "args"    => { 
                        "with-metis:s" => 
                        "Enable MeTiS graph partitioning commands.", 
                        "with-metis_libs=s" => 
                        "MeTiS graph partitioning library directory.", 
                        "with-metis_incl=s" => 
                        "MeTiS graph partitioning header file directory.", 
                                 },
                  } ,
    "net"      => { 
                    "flag"    => "NET"      , 
                    "default" => "yes"      ,
                    "libs"    => ""         ,
                    "incl"    => ""         , 
                    "args"    => { 
                        "with-net:s" => 
                        "Enable networking functions. (X=yes/no)",
                                 },
                  } ,
    "sqlite"   => { 
                    "flag"    => "SQLITE"   , 
                    "default" => "no"       ,
                    "libs"    => "-lsqlite3",
                    "incl"    => ""         , 
                    "args"    => { 
                        "with-sqlite:s" => 
                        "Enable SQL commands via the SQLite v3 database engine.",
                        "with-sqlite_libs=s" => 
                        "SQLite v3 database library directory.",
                        "with-sqlite_incl=s" => 
                        "SQLite v3 database header file directory.",
                                 },
                  } ,
    "ssl"      => { 
                    "flag"    => "OPENSSL"  , 
                    "default" => "no"       ,
                    "libs"    => "-lssl -lcrypto"  ,
                    "incl"    => ""         , 
                    "args"    => { 
                        "with-ssl:s" => 
                        "OpenSSL installation directory.",
                        "with-ssl_libs=s" => 
                        "OpenSSL library directory.",
                        "with-ssl_incl=s" => 
                        "OpenSSL header file directory.",
                                 },
                  } ,
    "umfpack"  => { 
                    "flag"    => "UMFPACK"  , 
                    "default" => "no"       ,
                    "libs"    => "-lumfpack -lamd",
                    "incl"    => ""         , 
                    "args"    => { 
                        "with-umfpack:s" => 
                        "UMFPACK v4.3 sparse matrix factoring installation directory.",
                        "with-umfpack_libs=s" => 
                        "UMFPACK v4.3 sparse matrix factoring library directory.",
                        "with-umfpack_incl=s" => 
                        "UMFPACK v4.3 sparse matrix factoring header file directory.",
                                 },
                  } ,
    "vmo"      => { 
                    "flag"    => "VMO"      , 
                    "default" => "no"       ,
                    "libs"    => "-L/usr/local/lib/voice -lmgsm -lpvf -lvoice -lutil -lmgetty -lvgetty"   ,
                    "incl"    => "-I/usr/local/include/voice", 
                    "args"    => { 
                        "with-vmo:s" => 
                        "Enable Speech synthesis commands.", 
                        "with-vmo_libs=s" => 
                        "Speech synthesis library directory.", 
                        "with-vmo_incl=s" => 
                        "Speech synthesis header file directory.", 
                                 },
                  } ,
    "vtparse"  => { 
                    "flag"    => "VT"       , 
                    "default" => "no"       ,
                    "libs"    => "-L/usr/local/vtparse -lvtparse"  ,
                    "incl"    => "-I/usr/local/vtparse" ,  
                    "args"    => { 
                        "with-vtparse:s" => 
                        "Xterm character handling installation directory.", 
                        "with-vtparse_libs=s" => 
                        "Xterm character handling library directory.", 
                        "with-vtparse_incl=s" => 
                        "Xterm character handling header file directory.", 
                                 },
                  },
    "X11"      => { 
                    "flag"    => "X11"      , 
                    "default" => "yes"      ,
                    "libs"    => "-L/usr/X11R6/lib -lX11"   , 
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-X11:s" => 
                        "X11 graphics library.",
                                 },
                  },
    "xdb"      => { 
                    "flag"    => "XDB"      , 
                    "default" => "no"       ,
                    "libs"    => ""         , 
                    "incl"    => ""         ,
                    "args"    => { 
                        "with-xdb:s" => 
                        "Nastran .xdb file reading library.", 
                                 },
                  },
            ); # 1}}}
my $options = "";
foreach my $k (sort keys %config) {
    $options .= sprintf "  --%-20s %s\n%s[%s]\n", 
                $k . "=X", $config{$k}{desc},
                ' ' x 25, $config{$k}{default};
}
my @with_arg_list = ();
foreach my $opt (sort keys %with) {
    foreach my $arg (sort keys %{$with{$opt}{args}}) {
        # strip off =s, =i, :s, :i argument types
        ( my $clean = $arg ) =~ s/[:=]\w$//;  
        my $default = $with{$opt}{libs};
        $default = "" if $clean =~ m{_(libs|incl)$};  # these have no defaults
        $options .= sprintf "  --%-20s %s\n", 
                    $clean . "=X", $with{$opt}{args}{$arg};
        $options .= sprintf "%s[%s]\n", ' ' x 25, $with{$opt}{libs}
            if $default;
        push @with_arg_list, $arg;
    }
}
my %args = ();
GetOptions (  # {{{1
           \%args, 
            "help"                ,
            "prefix=s"            ,
            "with-cc=s"           ,
            "with-f77=s"          ,
            "with-f_lib=s"        ,
            @with_arg_list,
            ); # 1}}}
# {{{1
my $Usage = 
"Usage: $script [options]
Options: [defaults in brackets after descriptions]
Configuration:
  --cache-file=FILE       cache test results in FILE
  --help                  print this message
  --no-create             do not create output files
  --quiet, --silent       do not print `checking...' messages

Features and packages:
$options
"; 
print $Usage if $args{help};
# 1}}}

my @D_flags       = ();  # -D defines
my @L_flags       = ();  # libraries
my @I_flags       = ();  # includes
foreach my $opt (sort keys %with) {
    my $with_opt      = "with-" . $opt;
    my $with_opt_libs = "with-" . $opt . "_libs";
    my $with_opt_incl = "with-" . $opt . "_incl";
    next if defined $args{$with_opt} and
                    $args{$with_opt} =~ /^no$/i;

    # defines
    push @D_flags, "-D" . $with{$opt}{flag} 
        if defined $with{$opt}{flag} and
          (defined $args{$with_opt} or 
           defined $args{$with_opt_libs});

    # libraries
    my $Lib = "";
    if (defined $args{$with_opt_libs}) {
        # first choice: with-XXX_libs=
        $Lib = "-L" .  $args{$with_opt_libs};
    } elsif (defined $args{$with_opt}) {
        # second choice: with-XXX/lib   or   -lXXX
        if ($args{$with_opt} and $args{$with_opt} !~ /^yes$/i) {
            if ($args{$with_opt} =~ /^-l/i or
                $args{$with_opt} =~ /\.([oa]|so(\.\d+)*)$/) {
                # was given -lXXX, -LXXX, library file, or object file
                $Lib = $args{$with_opt};
            } else {
                # presumably a directory, prefix with -L, postfix with /lib
                $Lib = "-L" .  $args{$with_opt} . "/lib";
            }
        }
    } elsif ($with{$opt}{libs}) {          
        # third choice:  default library
        $Lib = $with{$opt}{libs};
    }
    push @L_flags, "$with{$opt}{flag}_LIB=$Lib" if $Lib;

    # includes
    my $Inc = "";
    if (defined $args{$with_opt_incl}) {
        # first choice: with-XXX_incl=
        $Inc = "-I" .  $args{$with_opt_incl};
    } elsif (defined $args{$with_opt}) {
        # second choice: with-XXX/incl   or   -IXXX
        if ($args{$with_opt} and $args{$with_opt} !~ /^yes$/i) {
            if ($args{$with_opt} =~ /^-I/) {
                $Inc = $args{$with_opt};
            } elsif ($args{$with_opt} !~ /^-l/i and
                     $args{$with_opt} !~ /\.([oa]|so(\.\d+)*)$/) {
                # presumably a directory, prefix with -I, postfix with /include
                $Inc = "-I" .  $args{$with_opt} . "/include";
            }
        }
    } elsif ($with{$opt}{incl}) {          
        # third choice:  default include
        $Inc = $with{$opt}{incl};
    }
    push @I_flags, "$with{$opt}{flag}_INC=$Lib" if $Inc;
}
# print Dumper(%args, @D_flags, @L_flags, @I_flags);
my $prefix = $config{prefix}; # default
   $prefix = $args{prefix} if defined $args{prefix};
set_syspath("src/sys.c", $prefix);

my %compiler = ();
if ($args{'with-cc'}) {
    $compiler{CC} = $args{'with-cc'};
} elsif ($ENV{CC}) {
    $compiler{CC} = $ENV{CC};
} else { 
    $compiler{CC} = $config{with_cc}{default};
}
if ($args{'with-f77'}) {
    $compiler{F77} = $args{'with-f77'};
} elsif ($ENV{F77}) {
    $compiler{F77} = $ENV{F77};
} else {
    $compiler{F77} = $config{with_f77}{default};
}
if ($args{'with-f_lib'}) {
    $compiler{F_LIB} = $args{'with-f_lib'};
} elsif ($ENV{F_LIB}) {
    $compiler{F_LIB} = $ENV{F_LIB};
} else {
    $compiler{F_LIB} = $config{with_f_lib}{default};
}
if ($ENV{CFLAGS}) {
    $compiler{CFLAGS} = $ENV{CFLAGS};
} else {
    $compiler{CFLAGS} = "-fPIC -g -O4 -Wall -DFORT_UDSC";
}
if ($ENV{FFLAGS}) {
    $compiler{FFLAGS} = $ENV{FFLAGS};
} else {
    $compiler{FFLAGS} = "-O";
}
            
write_make_inc(\@D_flags, \@L_flags, \@I_flags, \%compiler);

sub set_syspath { # {{{1
    my $file = shift;
    my $path = shift;

    my $lines        = "";
    my $found_marker = 0;
    open  IN, $file or die "Cannot read $file:  $!\n";
    while (<IN>) {
        $found_marker = 1 if /TOPS_SYSPATH_DEFAULT/; 
        if ($found_marker  and /IF\s+'.*?'\s+THEN/) {
            $found_marker  = 0; 
            s{IF\s+'(.*?)'\s+THEN}{IF '$path' THEN};
        }
        $lines .= $_;
    }
    close IN;

    open  OUT, ">$file" or die "Cannot update $file:  $!\n";
    print OUT $lines;
    close OUT;
} # 1}}}
sub write_make_inc { # {{{1
    my ($ra_D_flags,
        $ra_L_flags,
        $ra_I_flags,
        $rh_compiler) = @_;

    foreach my $x (@{$ra_D_flags}) {  $rh_compiler->{CFLAGS} .= " $x"; }

    my $file = "make.inc";
    open  OUT, ">$file" or die "Cannot write to $file  $!\n";
    foreach my $x (sort keys %{$rh_compiler}) {
        print OUT "$x=$rh_compiler->{$x}\n";
    }
    foreach my $x (@{$ra_I_flags}) { print OUT "$x\n"; }
    foreach my $x (@{$ra_L_flags}) { print OUT "$x\n"; }
    close OUT;
} # 1}}}
__END__
./configure.pl  --with-blas                                          \
                --with-lapack=/usr/lib/atlas/liblapack.so.3.0        \
                --with-sqlite_incl=/usr/local/sqlite-3.2.7/include   \
                --with-sqlite_libs=/usr/local/sqlite-3.2.7/lib       \
                --with-fftw3=/usr/local/fftw-3.0.1                   \
                --with-umfpack_libs=/home/al/UMFPACKv4.3/UMFPACK/Lib \
                --with-umfpack_incl=/home/al/UMFPACKv4.3/UMFPACK/Inc \
                --with-X11                                           \
                --with-curses="-lncurses"                            \
                --with-dyn_load                                      \
                --with-keyboard                                      \
                --with-net                                           \
                --prefix=/usr/local/tops-3.1

