/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#ifndef TOP10_UTILXMLDUMPABLE_HH
#define TOP10_UTILXMLDUMPABLE_HH

#include <tinyxml.h>
#include "Log.hh"

namespace top10 {
namespace util {

//! Virtual base class for all classes capable of saving/loading to/from XML
class XmlDumpable {
public:
  XmlDumpable(std::string node_name);
  
  inline std::string getNodeName() const { return node_name; }
  
  //! Returns true if node->Value() is node_name.
  bool handles(const TiXmlElement* node) const;
  
  //! Load the state of this object from an XML node.
  /*! \param node the Xml node from which to load, its value must be node_name 
      \return 0  in case of success <0 otherwise. */
  int load(const TiXmlElement* node);
  
  //! Save the state of this object to an XML node.
  /*! \param node output argument. Its value and children will be overwritten.
      \return 0 in case of success, <0 otherwise.
  */
  int save(TiXmlElement*) const;
  
  //! Reset the state of this instance
  virtual void clearState() =0;
  
  virtual ~XmlDumpable();
  
protected:
  //! Does the job of loading, called by load.
  /*! Should not check the value of the node. */
  virtual int loadXml(const TiXmlElement*) =0;

  //! Does the job of saving, called by save
  /*! Should not modify the value of the node. */
  virtual int saveXml(TiXmlElement*) const =0;
  
  //! Dispatch a message in the log telling that a loadXml() method was invoked with the wrong kind of node
  void logUnexpectedSection(std::string expected, const TiXmlElement* got) const;
      
  //! Dispatch a message in the log telling something was wrong with a node (missing attribute...)
  void logXmlNodeError(std::string err, const TiXmlElement*, top10::util::Log::Level = top10::util::Log::Warning) const;

private:
  std::string node_name;
};

      
//! Dispatch a message in the log telling something was wrong with a node (missing attribute...)
void logXmlNodeError(std::string err, const TiXmlElement*, top10::util::Log::Level = top10::util::Log::Warning);


//! Unary function object that inserts XML nodes under a provided node.
/*!
  Suitable for use with std::for_each.
*/
struct SaveInserterFunction
{
  SaveInserterFunction(TiXmlElement* top);
  void operator()(const top10::util::XmlDumpable& item);

  TiXmlElement* m_top;
};

}
}

#endif
