/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_UTIL_VALUE_HANDLE
#define TOP10_UTIL_VALUE_HANDLE

#include <string>

namespace top10
{
  namespace util
  {

    //! Base type for all value handles
    /*! A value handle is a wrapper around a bool, double or a color adding a name and a description.
        Intended for use in GUIs to set "constants" used in various places.
    */
    class ValueHandle
    {
    public:
      std::string getName() const;
      std::string getDescription() const;
      void setName(const std::string& name);
      void setDescription(const std::string& name);

      virtual void reset() =0;
      virtual void setValueAsText(const std::string&) =0;
      virtual std::string getValueAsText() const =0;
      virtual std::string getDefaultValueAsText() const =0;
      virtual ~ValueHandle();

    private:
      std::string name;
      std::string description;
    };

    class BoolValueHandle : public ValueHandle
    {
    public:
      BoolValueHandle(bool* value_ptr, bool default_value);

      void reset();
      void setValueAsText(const std::string&);
      std::string getValueAsText() const;
      std::string getDefaultValueAsText() const;

      void setValue(bool);
      bool getValue() const;

    private:
      bool *value_ptr;
      bool default_value;
    };

    class DoubleValueHandle : public ValueHandle
    {
    public:
      DoubleValueHandle(double* value_ptr, double default_value);

      void reset();
      void setValueAsText(const std::string&);
      std::string getValueAsText() const;
      std::string getDefaultValueAsText() const;

      void setValue(double);
      double getValue() const;

    private:
      double *value_ptr;
      double default_value;
    };

    class ColorValueHandle : public ValueHandle
    {
    public:
      ColorValueHandle(unsigned char* r_ptr,
	unsigned char* g_ptr,
	unsigned char* b_ptr,
	unsigned char r_default,
	unsigned char g_default,
	unsigned char b_default);

      void reset();
      void setValueAsText(const std::string&);
      std::string getValueAsText() const;
      std::string getDefaultValueAsText() const;

      void setRedValue(unsigned char);
      void setGreenValue(unsigned char);
      void setBlueValue(unsigned char);
      
      unsigned char getRedValue() const;
      unsigned char getGreenValue() const;
      unsigned char getBlueValue() const;

    private:
      unsigned char* r_ptr;
      unsigned char* g_ptr;
      unsigned char* b_ptr;
      unsigned char r_default;
      unsigned char g_default;
      unsigned char b_default;

    private:
      static bool parseStr(const std::string& str, unsigned char& out_r, unsigned char& out_g, unsigned char& out_b);
      static std::string makeStr(unsigned char out_r, unsigned char out_g, unsigned char out_b);
    };

  }
}
#endif
