/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "ValueHandle.hh"
#include <sstream>

namespace top10
{
namespace util
{

  std::string ValueHandle::getName() const
  {
    return name;
  }

  std::string ValueHandle::getDescription() const
  {
    return description;
  }

  ValueHandle::~ValueHandle()
  {
  }

  void ValueHandle::setName(const std::string& n)
  {
    name = n;
  }

  void ValueHandle::setDescription(const std::string& d)
  {
    description = d;
  }


  /*
   * BoolValueHandle
   */

  BoolValueHandle::BoolValueHandle(bool* value_ptr, bool default_value)
    : value_ptr(value_ptr),
    default_value(default_value)
  {
  }

  void BoolValueHandle::reset()
  {
    *value_ptr = default_value;
  }

  void BoolValueHandle::setValue(bool b)
  {
    *value_ptr = b;
  }

  bool BoolValueHandle::getValue() const
  {
    return *value_ptr;
  }

  void BoolValueHandle::setValueAsText(const std::string& s)
  {
    if (s == "No" || s == "no" || s == "NO" || s == "0")
      setValue(false);
    else if (s == "Yes" || s == "yes" || s == "YES" || s == "1")
      setValue(true);
  }

  std::string BoolValueHandle::getValueAsText() const
  {
    if (*value_ptr)
      return "Yes";
    else return "No";
  }

  std::string BoolValueHandle::getDefaultValueAsText() const
  {
    if (default_value)
      return "Yes";
    else return "No";
  }

  /*
   * DoubleValueHandle
   */

  DoubleValueHandle::DoubleValueHandle(double* value_ptr, double default_value)
    : value_ptr(value_ptr),
    default_value(default_value)
  {
  }

  void DoubleValueHandle::reset()
  {
    *value_ptr = default_value;
  }

  void DoubleValueHandle::setValue(double b)
  {
    *value_ptr = b;
  }

  double DoubleValueHandle::getValue() const
  {
    return *value_ptr;
  }

  void DoubleValueHandle::setValueAsText(const std::string& s)
  {
    std::istringstream buf(s.c_str());
    buf >> *value_ptr;
  }

  std::string DoubleValueHandle::getValueAsText() const
  {
    std::ostringstream buf;
    buf << *value_ptr;
    return buf.str();
  }

  std::string DoubleValueHandle::getDefaultValueAsText() const
  {
    std::ostringstream buf;
    buf << default_value;
    return buf.str();
  }

  /*
   * ColorValueHandle
   */

  ColorValueHandle::ColorValueHandle(unsigned char* r_ptr,
                                     unsigned char* g_ptr,
				     unsigned char* b_ptr,
				     unsigned char r_default,
				     unsigned char g_default,
				     unsigned char b_default)
    : r_ptr(r_ptr),
    g_ptr(g_ptr),
    b_ptr(b_ptr),
    r_default(r_default),
    g_default(g_default),
    b_default(b_default)
  {
  }

  void ColorValueHandle::reset()
  {
    *r_ptr = r_default;
    *g_ptr = g_default;
    *b_ptr = b_default;
  }

  void ColorValueHandle::setRedValue(unsigned char v)
  {
    *r_ptr = v;
  }

  void ColorValueHandle::setGreenValue(unsigned char v)
  {
    *g_ptr = v;
  }

  void ColorValueHandle::setBlueValue(unsigned char v)
  {
    *b_ptr = v;
  }

  unsigned char ColorValueHandle::getRedValue() const
  {
    return *r_ptr;
  }

  unsigned char ColorValueHandle::getGreenValue() const
  {
    return *g_ptr;
  }

  unsigned char ColorValueHandle::getBlueValue() const
  {
    return *b_ptr;
  }

  void ColorValueHandle::setValueAsText(const std::string& s)
  {
    parseStr(s, *r_ptr, *g_ptr, *b_ptr);
  }

  std::string ColorValueHandle::getValueAsText() const
  {
    return makeStr(*r_ptr, *g_ptr, *b_ptr);
  }

  std::string ColorValueHandle::getDefaultValueAsText() const
  {
    return makeStr(r_default, g_default, b_default);
  }

  bool ColorValueHandle::parseStr(const std::string& str, unsigned char& out_r, unsigned char& out_g, unsigned char& out_b)
  {
    unsigned char r =0, g =0, b =0;

    unsigned int pos = 0;
    enum {ParseRed, ParseGreen, ParseBlue} state = ParseRed;
    int num = 0;

    while (pos < str.size())
    {
      switch (state)
      {
      case ParseRed:
	if (str[pos] >= '0' && str[pos] <= '9')
	  num = num*10 + (str[pos] - '0');
	else if (str[pos] == ':')
	{
	  if (num >= 0 && num < 256)
	    r = num;
	  else
	    return false;

	  state = ParseGreen;
	  num = 0;
	}
	break;

      case ParseGreen:
	if (str[pos] >= '0' && str[pos] <= '9')
	  num = num*10 + (str[pos] - '0');
	else if (str[pos] == ':')
	{
	  if (num >= 0 && num < 256)
	    g = num;
	  else
	    return false;

	  state = ParseBlue;
	  num = 0;
	}
	break;

      case ParseBlue:
	if (str[pos] >= '0' && str[pos] <= '9')
	  num = num*10 + (str[pos] - '0');
	break;
      }

      ++pos;
    }

    if (num >= 0 && num < 256)
      b = num;
    else
      return false;

    out_r = r;
    out_g = g;
    out_b = b;

    return true;
  }

  std::string ColorValueHandle::makeStr(unsigned char out_r, unsigned char out_g, unsigned char out_b)
  {
    std::ostringstream buf;
    buf<<(int)out_r<<":"<<(int)out_g<<":"<<(int)out_b;
    return buf.str();
  }

}
}

