/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#ifndef TOP10_UTILRESPONSIBILITYCHAIN_HH
#define TOP10_UTILRESPONSIBILITYCHAIN_HH

#include <list>

namespace top10 {
namespace util {

//! Convenience class template for implicit use with ResponsibilityChain
template< typename HandlerT >
class DefaultRespChainAcc {
public:
  //! Check if a handler is capable of handling a request
  template< typename RequestT >
    inline bool handles(HandlerT h, const RequestT& req) const { return h->handles(req); }
  
  //! Tell a handler to do its job on a request
  template< typename RequestT >
    inline int handle(HandlerT h, RequestT& req) const { return h->handle(req); }
};
    
template< typename HandlerT, typename AccessorT = DefaultRespChainAcc< HandlerT > >
class ResponsibilityChain {
public:
  ResponsibilityChain(AccessorT acc): acc(acc) {}
  
  /*!
    \return 0 if there was a handler capable of handling the request, and the handler succeded,
            1 if no handler could handle the request
            <0 if a handler was supposed to be able to handle the request, but failed to complete its task
  */
  template< typename RequestT >
      int handle(RequestT req) {
        for (typename std::list<HandlerT>::iterator it = handlers.begin(); it != handlers.end(); ++it)
        {
          if (acc.handles(*it, req)) return acc.handle(*it, req);
        }
        return 1;
      }
  
  inline void addHandler(HandlerT h) { handlers.push_back(h); }
  virtual ~ResponsibilityChain() {}

private:
  AccessorT acc;
  std::list<HandlerT> handlers;
};

}
}

#endif
