#ifndef TOP10_UTILREFCOUNT_HH
#define TOP10_UTILREFCOUNT_HH

#include <cassert>
#include <vector>

#define REF_DEBUG 1

namespace top10 {
namespace util {

class RefCount
{
public:
  RefCount();
  RefCount(const RefCount&);
  virtual ~RefCount();
  
  void use() const;
  bool release() const;

  inline int getCount() const { return ref_count; }
  unsigned long getSalt() const { return salt;}

  inline void setDeallocateFlag(bool b) { deallocate = b; }
  inline bool getDeallocateFlag() const { return deallocate; }

  inline void check() const
  {
#if REF_DEBUG
    assert(ref_count >= 0);
    assert(salt == 0x900df00d);
#endif
  }

private:
  mutable int ref_count;
  mutable bool first_use;
  bool deallocate;
  unsigned long salt;
};


//! A safe pointer wrapper, does not update reference counts.
template<typename T>
class Ptr
{
public:
  //! Default value: null pointer
  Ptr(): m_ptr(0) {}

  //! Init from a raw pointer
  Ptr(T* ptr): m_ptr(ptr) { if (m_ptr) m_ptr->check(); }

  //! Destructor: check
  ~Ptr() { if (m_ptr) m_ptr->check(); }

  //! Check, then convert to a raw pointer
  inline T* getPtr() const { if (m_ptr) m_ptr->check(); return m_ptr; }

  //! Emulate raw pointers (does a check)
  inline T* operator->() const { return getPtr(); }

  //! Emulate raw pointers (does a check)
  inline T& operator*() const { return *getPtr(); }
  
  //! Do a check, check if not null.
  inline bool isValid() const { if (m_ptr) { m_ptr->check(); return true; } return false; }

  //! Do a check, emulate a dynamic cast.
  template<typename T2> inline T2* dyncast() const {
      if (m_ptr) m_ptr->check();
      return dynamic_cast<T2*>(m_ptr);
  }

private:
  T* m_ptr;
};


//! A pointer wrapper used to maintain reference counts automatically.
template<typename T>
class Ref {
public:
  explicit Ref(T* ptr): ptr(ptr) { if (ptr) ptr->use(); }
  Ref(): ptr(0) {}
  Ref(const Ref<T>& other): ptr(other.getPtr()) { if (ptr) ptr->use(); }

  inline Ref<T>& operator=(const Ref<T>& other)
  {
    if (ptr == other.ptr)
      return *this;

    if (ptr && ptr->release())
      destroy();

    ptr = other.getPtr();

    if (ptr)
      ptr->use();

    return *this;
  }

  inline
  Ref<T>& operator=(T* optr)
  {
    if (optr == ptr)
      return *this;
    
    if (ptr && ptr->release())
      destroy();

    ptr = optr;

    if (ptr)
      ptr->use();

    return *this;
  }

  template<typename T2>
  inline
    T2* dyncast() const {
      if (ptr)
	ptr->check();

      return dynamic_cast<T2*>(ptr);
    }
  
  inline
  bool isValid() const {
    if (ptr){
      ptr->check();
      return true;
    }
    return false;
  }

  inline
  T* getPtr() const {
    if (ptr)
      ptr->check();
    
    return ptr;
  }
  
  inline
  const T* getConstPtr() const {
    if (ptr)
      ptr->check();

    return ptr;
  }

  inline
  T& operator*() const {
    if (ptr)
      ptr->check();
    return *ptr;
  }

  inline
  T* operator->() const {
    if (ptr)
      ptr->check();
    return ptr;
  }

  inline
  bool operator==(const Ref<T>& other) const {
    return ptr == other.ptr;
  }

  inline
  bool operator!=(const Ref<T>& other) const {
    return ptr != other.ptr;
  }

  inline
  bool operator<(const Ref<T>& other) const {
    return ptr < other.ptr;
  }

  inline
  void discard() {
    if (ptr)
      ptr->check();
    
    if (ptr && ptr->release())
      destroy();
    
    ptr=0;
  }
  
  ~Ref() {
    discard();
  }

private:

  inline void destroy() const {
    if (ptr->getDeallocateFlag())
    {
      delete(ptr);
    }
    else
    {
      ptr->T::~T();
    }
  }

  T* ptr;
};


}
}

#endif
