/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "PathFinder.hh"
#include <cstdlib>
#include <sys/types.h>
#include <sys/stat.h>

using namespace top10;
using namespace top10::util;

std::string PathFinder::find(std::string filename)
{
  struct stat buf;

  // Replace all spaces in the filename by underscores
  for (std::string::size_type i=0; i<filename.size(); ++i) {
    if (filename[i] == ' ') filename[i]='_';
  }

  std::string name(filename);
  if (isAbsolute(name)) {
    if (stat(name.c_str(), &buf) == 0) return cleanSep(name);
    name = strip(name);
  }

  while (!name.empty()) {
    for (Paths::const_iterator path = paths.begin();
	 path != paths.end();
	 ++path) {
      std::string tmp = *path + std::string("/") + name;
      if (stat(tmp.c_str(), &buf) == 0) return cleanSep(tmp);
    }
    name = strip(name);
  }

  return cleanSep(name);
}

std::string PathFinder::cleanSep(std::string name)
{
  // Replace all / and \ by the seperator corresponding to the OS
  for (std::string::size_type i=0; i<name.size(); ++i) {
    if (name[i] == '/' || name[i] == '\\')
#ifdef WIN32
      name[i] = '\\';
#else
      name[i] = '/';
#endif
  }

  return name;
}

std::string PathFinder::getPath(std::string name)
{
  std::string ret;

  // Look for the last '/', and remove everything on its right
  std::string::size_type pos = name.find_last_of("/\\");
  if (pos != std::string::npos)
    ret = name.substr(0, pos);

  return ret;
}

bool PathFinder::isAbsolute(std::string name)
{
#ifdef WIN32
  return (name.size()>1 && ':' == name[1]) || (name.size()>0 && ('\\' == name[0] || '/' == name[0]));
#else
  return '/' == name[0];
#endif
}

std::string PathFinder::strip(std::string name)
{
  std::string ret;

  // Look for the first '/', and remove everything on its left
  std::string::size_type pos = name.find_first_of("/\\", 0);
  if (pos != std::string::npos)
    ret = name.substr(pos+1, name.size() - pos -1);

  return ret;
}

void PathFinder::addPath(std::string new_path)
{
  for (Paths::const_iterator path = paths.begin();
       path != paths.end();
       ++path)
    if (*path == new_path) return;

  paths.push_back(new_path);
}

std::string PathFinder::basename(std::string name)
{
  std::string ret;

  // Look for the last '/', and remove everything on its left
  std::string::size_type pos = name.find_last_of("/\\");
  if (pos != std::string::npos)
    ret = name.substr(pos+1, name.size() - pos -1);
  else ret = name;
  return ret;
}

#include "util/UserFile.hh"

// Preprocessor stringification magic...
#define xstr(s) str(s)
#define str(s) #s

PathFinder PathFinder::defaultPathFinder()
{
  PathFinder ret;
  std::string paths[] = {"data", xstr(DATA_DIR), ""};

  for (int i=0; !paths[i].empty(); ++i) {
    ret.addPath(paths[i]);
    ret.addPath(paths[i] +"/textures");
    ret.addPath(paths[i] +"/meshes");
    ret.addPath(paths[i] +"/tracks");
    ret.addPath(paths[i] +"/sounds");
    ret.addPath(paths[i] +"/kart_data");
    ret.addPath(paths[i] +"/fonts");
    ret.addPath(paths[i] +"/track_data");
  }

  ret.addPath(top10::util::UserFile::getPath());
  top10::util::UserFile::makePath(); // Just in case it does not exist.

  for (int i=0; !paths[i].empty(); ++i) {
    ret.addPath(paths[i] + "/defaults");
  }
  
  return ret;
}
