/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/


#ifndef TOP10_UTILGRAPH_HH
#define TOP10_UTILGRAPH_HH

#include <map>
#include <vector>
#include <set>
#include <list>

namespace top10 {
namespace util {

class Graph
{
public:
  struct VertexId {
    VertexId(int n): id(n) {}
    int id;
    inline bool operator<(const VertexId& other) const { return id < other.id; }
    inline bool operator==(const VertexId& other) const { return id == other.id; }
  };
  typedef std::vector<VertexId> Path;
  
public:
  Graph();

  inline void addVertex(VertexId id) { addVertexPrivate(id); }
  void removeVertex(VertexId);
  int getVertices(std::vector<VertexId>&) const;
  bool hasVertex(VertexId) const;
  
  void addEdge(VertexId, VertexId, float weight = 1.0);
  void addArc(VertexId, VertexId, float weight = 1.0);
  int getSuccessors(VertexId, std::vector<VertexId>&) const;
  int countSuccessors(VertexId) const;
  
  //! Find a vertex with maximum degree in an undirected graph
  /*! \return true if successful, false if all vertices are isolated. */
  bool findStar(VertexId& out_v, const std::set<VertexId>& exclude) const;
  
  //! Remove as few vertices as possible so that all vertices are isolated in an undirected graph
  /*! This is useful to remove points on a curve whose normals cross each other. This
    is needed to avoid nasty loops in the inner outline of a curve.
    \return the number of vertices removed. */
  int separate(const std::set<VertexId>& exclude);
  
  //! Extract smallest loops
  /*! Assumes a planar graph with non-intersecting edges, where each node keeps an ordered list
    of successors. The successors are ordered by their angle with the node */
  int getSmallestLoops(std::list<Path>& out) const;
  
  ~Graph();

  //! Test this class
  static void test();
  
protected:
  typedef std::pair<float, VertexId> WeightedArc;
  typedef std::vector<WeightedArc> WeightedArcs;
  
  struct FindById {
    FindById(VertexId v): v(v) {}
    inline bool operator()(WeightedArc a) {
      return a.second == v;
    }
    
    VertexId v;
  };
  
  /*! Mapping from nodes to ougoing edges. Encodes the successor relation */
  typedef std::map<VertexId, WeightedArcs> Map;
 
  Map::iterator addVertexPrivate(VertexId);
   
  //! Mapping from vertices to their successors/neighbours
  Map graph;
  
private:
  typedef std::set< std::pair< VertexId, VertexId > > ArcSet;
  
  Path getLoop(ArcSet& visited, Map::const_iterator vertex, int succ_pos) const;
  //! Find the left-most ougoing edge of vertex when entering from prev
  /*! This method assumes weights are to be interpreted as angles */
  VertexId findNext(VertexId prev, VertexId vertex) const;
};

}
}

#endif
