/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "Graph.hh"
#include "Log.hh"     // for assert
#include <algorithm>  // for std::find

namespace top10 {
namespace util {

Graph::Graph()
{}


Graph::~Graph()
{}

Graph::Map::iterator Graph::addVertexPrivate(VertexId v)
{
  Map::iterator find_it = graph.find(v);
  
  // Not there yet, add it
  if (find_it == graph.end()) {
    // It's already clear, we just want to add an entry to the graph
    graph[v].clear();
    find_it = graph.find(v);
  }
  
  assert(find_it != graph.end());
  return find_it;
}

bool Graph::hasVertex(VertexId v) const
{
  return graph.find(v) != graph.end();
}

void Graph::removeVertex(VertexId v)
{
  Map::iterator v_it = graph.find(v);
  if (v_it == graph.end()) return;
  
  // Also remove all edges between v and its neighbours
  for (WeightedArcs::const_iterator ngb_it = v_it->second.begin();
      ngb_it != v_it->second.end();
      ++ngb_it)
  {
    Map::iterator find_ngb = graph.find(ngb_it->second);
    assert(find_ngb != graph.end());
    
    // Find v in ngb's list of neighbours
    WeightedArcs::iterator v_pos = std::find_if(find_ngb->second.begin(), find_ngb->second.end(), FindById(v));
    assert(v_pos != find_ngb->second.end());
    
    find_ngb->second.erase(v_pos);
  }
  
  // Remove v itself
  graph.erase(v_it);
}

/*!
  \note neighbours are sorted by increasing weight
*/
void Graph::addEdge(VertexId v1, VertexId v2, float weight)
{
  Map::iterator v1_it = addVertexPrivate(v1);
  Map::iterator v2_it = addVertexPrivate(v2);
  
  WeightedArcs::iterator pos_it;
  for (pos_it = v1_it->second.begin(); pos_it != v1_it->second.end() && pos_it->first >= weight; ++pos_it);
  v1_it->second.insert(pos_it, WeightedArc(weight, v2));
  
  for (pos_it = v2_it->second.begin(); pos_it != v2_it->second.end() && pos_it->first >= weight; ++pos_it);
  v2_it->second.insert(pos_it, WeightedArc(weight, v1));
}

/*!
  \note neighbours are sorted by increasing weight
*/
void Graph::addArc(VertexId v1, VertexId v2, float weight)
{
  Map::iterator v1_it = addVertexPrivate(v1);
  
  WeightedArcs::iterator pos_it;
  for (pos_it = v1_it->second.begin(); pos_it != v1_it->second.end() && pos_it->first >= weight; ++pos_it);
  v1_it->second.insert(pos_it, WeightedArc(weight, v2));
}

int Graph::getVertices(std::vector<VertexId>& out) const
{
  Map::const_iterator it;
  for (it = graph.begin(); it != graph.end(); ++it) out.push_back(it->first);
  return graph.size();
}

int Graph::getSuccessors(VertexId v, std::vector<VertexId>& out) const
{
  Map::const_iterator v_it = graph.find(v);
  if (v_it == graph.end()) return 0;
  
  for (WeightedArcs::const_iterator it = v_it->second.begin(); it != v_it->second.end(); ++it)
    out.push_back(it->second);
  return v_it->second.size();
}

int Graph::countSuccessors(VertexId v) const
{
  Map::const_iterator v_it = graph.find(v);
  if (v_it != graph.end()) return v_it->second.size();
  else return 0;  
}

bool Graph::findStar(VertexId& v_out, const std::set<VertexId>& exclude ) const
{
  int deg_max = 0;
  
  for (Map::const_iterator it = graph.begin(); it != graph.end(); ++it) {
    if ((int)it->second.size() > deg_max && exclude.find(it->first) == exclude.end()) {
      deg_max = it->second.size();
      v_out = it->first;
    }
  }
  
  return deg_max != 0;
}

int Graph::separate(const std::set<VertexId>& exclude)
{
  int c=0;
  VertexId v=0;
  while (findStar(v, exclude)) {
    ++c;
    removeVertex(v);
  }
  return c;
}

int Graph::getSmallestLoops(std::list<Path>& out) const
{
  int count = 0;
  ArcSet visited;

  // Look for nodes of degree != 2
  bool found = false;
  for (Map::const_iterator it = graph.begin(); it != graph.end(); ++it) {
    Map::const_iterator it2 = it;
    ++it2;
    if (it->second.size() != 2 || (!found && it2 == graph.end())) {
      found = true;
      // Follow each outgoing edge...
      for (int succ_pos = 0; succ_pos < it->second.size(); ++succ_pos) {
        // ... to build the loop
        Path p = getLoop(visited, it, succ_pos);
        if (!p.empty()) {
          out.push_back(p);
          ++count;
        }
      }
    }
  }
  
  return count;
}
  
Graph::Path Graph::getLoop(ArcSet& visited, Map::const_iterator vertex, int succ_pos) const
{
  Path ret;

  assert(vertex != graph.end());
  assert(vertex->second.size() > succ_pos);
  
  VertexId succ = vertex->second[succ_pos].second;
  
  // Follow the left-most (relative to the incoming edge) edge until we loop
  while (visited.find(std::make_pair(vertex->first, succ)) == visited.end()) {
    ret.push_back(vertex->first);
    visited.insert(std::make_pair(vertex->first, succ));
    VertexId succ2 = findNext(vertex->first, succ);
    vertex = graph.find(succ);
    assert(vertex != graph.end());
    succ = succ2;
  }
  if (!ret.empty()) ret.push_back(vertex->first);
  
  return ret;
}

Graph::VertexId Graph::findNext(VertexId prev, VertexId vertex) const
{
  Map::const_iterator find_v = graph.find(vertex);
  assert(find_v != graph.end());
  
  WeightedArcs::const_iterator find_prev = std::find_if(find_v->second.begin(), find_v->second.end(), FindById(prev));
  assert(find_prev != find_v->second.end());
  
  int pos = (find_prev - find_v->second.begin() + 1)%find_v->second.size();
  return find_v->second[pos].second;
}

void Graph::test()
{
  bool passed = true;
  
  std::cout<<"Testing Graph::separate():"<<std::endl;
  Graph G;
  G.addEdge(0,1);
  G.addEdge(0,2);
  G.addEdge(0,3);
  G.addEdge(2,0);
  std::set<VertexId> empty;
  int removed = G.separate(empty);
  
  if (removed != 1) {
    std::cout<<" Failed (removed = "<<removed<<")"<<std::endl;
    passed = false;
  }
  if (G.hasVertex(0)) {
    std::cout<<" Failed (vertex 0 was not removed)"<<std::endl;
    passed = false;
  }
  if (passed) std::cout<<" Passed"<<std::endl;
  
  passed = true;
  std::cout<<"Testing Graph::getSmallestLoops():"<<std::endl;
  Graph G2;
  
  G2.addArc(1,2,90.0);
  G2.addArc(2,3,135.0);
  G2.addArc(3,4,0.0);
  G2.addArc(4,2,225.0);
  G2.addArc(1,5,270.0);
  G2.addArc(5,6,225.0);
  G2.addArc(6,7,0.0);
  G2.addArc(7,5,135.0);
  
  G2.addArc(2,1,360.0-90.0-180.0);
  G2.addArc(3,2,360.0-135.0-180.0);
  G2.addArc(4,3,360.0-0.0-180.0);
  G2.addArc(2,4,360.0-225.0-180.0);
  G2.addArc(5,1,360.0-270.0-180.0);
  G2.addArc(6,5,360.0-225.0-180.0);
  G2.addArc(7,6,360.0-0.0-180.0);
  G2.addArc(5,7,360.0-135.0-180.0);
  
  std::list< Graph::Path > loops;
  int n_loops = G2.getSmallestLoops(loops);
  
  if (n_loops != loops.size()) {
    std::cout<<" Failed: n_loops ("<<n_loops<<") != loops.size() ("<<loops.size()<<")"<<std::endl;
    passed = false;
  }
  if (n_loops != 3) {
    std::cout<<" Failed: n_loops ("<<n_loops<<") != 3"<<std::endl;
    passed = false;
  }
  for (std::list< Graph::Path >::const_iterator path_it = loops.begin(); path_it != loops.end(); ++path_it) {
    for (Graph::Path::const_iterator it = path_it->begin(); it != path_it->end(); ++it) {
      std::cout<<" "<<it->id;
    }
    std::cout<<std::endl;
  }
  if (passed) std::cout<<" Passed"<<std::endl;
  
}

}
}
