/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

//My stuff
#include "Options.hh"
#include "UIPanel.hh"
#include "MainMenu.hh"
#include "Controls.hh"
#include "util/PathFinder.hh"
#include "util/UserFile.hh"
#include "util/SlowTime.hh"
#include "util/GlobalOptions.hh"
#include "helpers/initGL.hh"
#include "sound/AudioContext.hh"

//C++ stuff
#include <iostream>
#include <fstream>

//SDL Stuff
#include <SDL/SDL.h>

// PLIB fonts
#include <plib/fnt.h>

using namespace top10::ui_interactive;
using namespace std;

top10::track::Track* parseNewTrack(std::string);

static void close_joy(int, void* joy)
{
  SDL_JoystickClose((SDL_Joystick*)joy);
}

/// Deinitialize SDL when destroyed
class SDL_Destroyer
{
public:
  static const int quit =1;
  static const int joystick =2;

public:
  SDL_Destroyer(): do_quit(false), do_joy(false) {};

  /// Tell what parts of the sdl to deinit
  void enable(int what)
  {
    if (what & quit) do_quit = true;
    if (what & joystick) do_joy = true;
  }

  ~SDL_Destroyer()
  {
    if (do_quit) SDL_Quit();
  }

private:
  bool do_quit;
  bool do_joy;
};


int main(int argc, char** argv)
{
  using namespace top10::ui_interactive::options;
  using top10::util::Error;

  try {
    // Create the user-specific application data directory if necessary
    top10::util::UserFile::makePath();
    top10::util::UserFile::makePath( "laps" );

    SDL_Destroyer sdl_closer;  // Close sdl when destroyed
    SDL_Surface* screen;
    top10::util::PathFinder path_finder = top10::util::PathFinder::defaultPathFinder();

    // Load options
    try {
      top10::util::GlobalOptions::getGlobalOptions()->load("options.xml");
    }
    catch (top10::util::Error& e) {
      std::cerr<<"Error while reading options: "<<e<<std::endl;
    }
    
    parse(argc, argv);
    top10::util::global_time.setSlowFactor(slow_factor);

    /* Initialize the SDL library */
    if ( SDL_Init(SDL_INIT_NOPARACHUTE | SDL_INIT_TIMER | SDL_INIT_VIDEO | SDL_INIT_JOYSTICK) < 0 ) {
      throw Error(string("Couldn't initialize SDL: ") + string(SDL_GetError()));
    }
    
    // Open all joysticks
    int n_sticks = SDL_NumJoysticks();
    std::vector<SDL_Joystick*> joysticks;
    for (int i=0; i<n_sticks; ++i) {
      SDL_Joystick* joy = SDL_JoystickOpen(i);
      joysticks.push_back(joy);
      if (joy) {
	std::cout<<"Opened joystick "<<i<<std::endl;
	std::cout<<"name: "<<SDL_JoystickName(i)<<" ";
	std::cout<<"Axes: "<<SDL_JoystickNumAxes(joy)<<" ";
	std::cout<<"Buttons: "<<SDL_JoystickNumButtons(joy)<<" ";
	std::cout<<"Balls: "<<SDL_JoystickNumBalls(joy)<<" ";
      }
      else std::cout<<"Failed top open joystick "<<i<<std::endl;
    }

    /* Clean up on exit, exit on window close and interrupt */
    sdl_closer.enable(SDL_Destroyer::quit);
  
    // Initialize GL stuff
    screen = top10::helpers::initGL(w, h, full_screen);
    top10::graphX::Renderer render;
    render.initGL(w,h);
    top10::graphX::CameraNode* camera = new top10::graphX::CameraNode;
    render.setCamera(camera);

    top10::helpers::setGrabInput(true);

    // First load the default settings
    std::string kart_path = path_finder.find("default/default.physx");
    if (kart_path.empty()) throw Error("Could not find default physics settings");
    else {
      std::ifstream kart_in(kart_path.c_str());
      if (!kart_in) throw Error("Could not open ")+kart_path;
      top10::physX::Kart::initDefaultParameters(kart_in);
    }
    // Then load the specific settings
    if (!kart_filename.empty()) {
      kart_path = path_finder.find(kart_filename + "/" + kart_filename + ".physx");
      if (kart_path.empty()) throw Error("Could not find ") + kart_filename + " physics settings";
      else {
	std::ifstream kart_in(kart_path.c_str());
	if (!kart_in) throw Error("Could not open ")+kart_path;
	top10::physX::Kart::initDefaultParameters(kart_in);
      }
    }

    // Init font lib
    fntInit();

    // Audio
    top10::sound::AudioContext* audio_device(0);
    top10::util::Ref<top10::sound::SourceAllocator> audio_alloc(0);
    if (use_audio) {
      audio_device = new top10::sound::AudioContext;
      audio_alloc = new top10::sound::SourceAllocator(audio_device);
    }
    
    // Load keyboard mapping
    ActionMapping actions;
    try {
      ActionMapping tmp_actions;
      if (!path_finder.find("inputs").empty())
	tmp_actions = loadActionMapping("inputs");
      actions = tmp_actions;
    }
    catch (std::string err) {
      std::cerr<<"Could not load action mapping: "<<err<<std::endl;
    }
    
    UIPanel panel(audio_alloc.getPtr());
    MainMenuRef menu(new MainMenu(&panel, &actions, &render, audio_alloc.getPtr()));
    panel.setFrontMenu(menu.getPtr());
    panel.resizeEvent(w,h);
    panel.setMenu(menu.getPtr());

    Menu::makeSounds(audio_alloc.getPtr());

    /* Main loop */
    SDL_Event event;
    bool quit = false;
    while (!menu->getQuit() && !quit)
    {

      // Handle events
      while (!quit && SDL_PollEvent(&event))
      {
	if (event.type == SDL_QUIT)
	  quit = true;
	else if (event.type == SDL_KEYDOWN)
	{
	  if ((event.key.keysym.mod & KMOD_CTRL) &&
	    (event.key.keysym.sym == SDLK_g))
	  {
	    top10::helpers::toggleGrabInput();
	  }
	  else if ((event.key.keysym.mod & KMOD_CTRL) &&
	    (event.key.keysym.sym == SDLK_f))
	  {
	    screen = top10::helpers::toggleFullscreen(screen);
	  }
	  else panel.event(event);
	}
	else
	  panel.event(event);	
      }

      if (quit) continue;

      // Update game and display it
      panel.updateFrontData();
      render.clearAllGL();
      panel.renderGL();
      panel.renderAL();

      // render the menu if the panel is shown.
      if (panel.getNode())
      {
	render.clearDepthGL();
	top10::graphX::RenderList rl;
	render.buildList(panel.getNode(), &rl);
	render.renderHudGL(rl);
      }

      SDL_GL_SwapBuffers();
    }
    saveActionMapping(top10::util::UserFile::getPath()+"/inputs", actions);
    top10::util::GlobalOptions::getGlobalOptions()->save(top10::util::UserFile::getPath()+"/options.xml");
  }
  catch (const std::string& e) {
    cerr<<"Fatal error caught: "<<e<<endl;
  }
#if 0
  catch (...) {
    cerr<<"Unknown error caught"<<endl;
  }
#endif


 cout<<"Exit"<<endl;
 
 return 0;
}
