/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "UIPanel.hh"
#include "Menu.hh"
#include "util/PathFinder.hh"
#include "util/Log.hh"
#include "util/GlobalOptions.hh"

#include <cstdio>
#include <vorbis/vorbisfile.h>

using namespace top10;
using namespace ui_interactive;

/*
 * Class: UIPanel
 */

UIPanel::UIPanel(top10::sound::SourceAllocator* audio_alloc): currentMenu(0), frontMenu(0), audio_alloc(audio_alloc)
{
  loadMusic();
  playMusic();
}



void UIPanel::loadMusic()
{
  using top10::util::Log;

  const top10::util::GlobalOptions* opt = top10::util::GlobalOptions::getGlobalOptions();

  const std::string filename = opt->getStrOption("Sound.Menu.Music.File");
  if (filename.empty())
    return;

  std::string path = top10::util::PathFinder::defaultPathFinder().find(filename);
  if (path.empty())
  {
    Log::getSingle()->send(Log::Error, "MainMenu/loadMusic", "Could not find " + filename);
    return;
  }

  FILE* f = fopen("data/menus.ogg", "rb");
  
  if (!f)
  {
    Log::getSingle()->send(Log::Error, "MainMenu/loadMusic", "Could not load " + filename);
    return;
  }

  vorbis_info *pInfo;
  OggVorbis_File oggFile;
  int status = ov_open(f, &oggFile, NULL, 0);
  if (status < 0)
  {
    Log::getSingle()->send(Log::Error, "MainMenu/loadMusic", "Could not open " + filename + " as an Ogg file");
    fclose(f);
    return;
  }

  // Get some information about the OGG file
  pInfo = ov_info(&oggFile, -1);
  if (!pInfo)
  {
    Log::getSingle()->send(Log::Error, "MainMenu/loadMusic", "Could not open " + filename + " as an Ogg file");
    ov_clear(&oggFile); // Also calls fclose(f)
    return;
  }

  // Check the number of channels... always use 16-bit samples
  ALenum format;
  if (pInfo->channels == 1)
    format = AL_FORMAT_MONO16;
  else
    format = AL_FORMAT_STEREO16;
  // end if

  // The frequency of the sampling rate
  ALsizei freq = pInfo->rate;

  const unsigned int BUFFER_SIZE = 32768;
  char array[BUFFER_SIZE];    // Local fixed size array
  std::vector<char> buffer;
  int bitStream = 0;
  long bytes = 0;
  do {
    // Read up to a buffer's worth of decoded sound data
    bytes = ov_read(&oggFile, array, BUFFER_SIZE, 0 /*endianess*/, 2, 1, &bitStream);
    // Append to end of buffer
    buffer.insert(buffer.end(), array, array + bytes);
  } while (bytes > 0);

  ov_clear(&oggFile);

  alGenBuffers(1, &music_buffer);

  // Upload sound data to buffer
  alBufferData(music_buffer, format, &buffer[0], static_cast < ALsizei > (buffer.size()), freq);

  music_source = audio_alloc->getNew(music_buffer);
  music_source->setLooping(true);
  music_source->setGain(opt->getNumOption("Sound.Menu.Music.Volume"));

  Log::getSingle()->send(Log::Info, "UIPanel/loadMusic", "Loaded " + filename);
}



void UIPanel::playMusic()
{
  if (music_source.isValid())
    music_source->play();
}



void UIPanel::stopMusic()
{
  if (music_source.isValid())
    music_source->pause();
}



void UIPanel::event(SDL_Event event)
{
  if (event.type == SDL_KEYDOWN && event.key.keysym.sym == SDLK_ESCAPE)
  {
    if (currentMenu.isValid())
      setMenu(0);
    else
      setMenu(frontMenu.getPtr());
  }
  else
  {
    if (currentMenu.isValid())
      currentMenu->event(event);
    else if (frontMenu.isValid())
      frontMenu->handleFrontEvent(event);
  }
}

void UIPanel::setMenu(Menu* m)
{
  currentMenu = m;
  if (currentMenu.isValid())
  {
    SDL_EnableUNICODE(1);

    if (frontMenu.isValid())
      frontMenu->on_panelShowed();
  }
  else
  {
    if (frontMenu.isValid())
      frontMenu->on_panelHidden();

    SDL_EnableUNICODE(-1);
  }
}

void UIPanel::setFrontMenu(Menu* m)
{
  if (frontMenu.isValid())
    frontMenu->on_panelFrontOff();
  frontMenu = m;
  if (m)
    m->on_panelFrontOn();
}

top10::graphX::Node* UIPanel::getNode()
{
  if (!currentMenu.isValid()) return 0;
  return currentMenu->getNode();
}

void UIPanel::updateFrontData()
{
  if (frontMenu.isValid())
    frontMenu->updateFrontData();
}

void UIPanel::renderGL()
{
  if (frontMenu.isValid())
    frontMenu->renderGL();
}

void UIPanel::renderAL()
{
  if (frontMenu.isValid())
    frontMenu->renderAL();
}

Menu* UIPanel::getCurrentMenu()
{
  return currentMenu.getPtr();
}
      
const Menu* UIPanel::getCurrentMenu() const
{
  return currentMenu.getPtr();
}
