/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "TrainingMenu.hh"
#include "UIPanel.hh"
#include "util/GlobalOptions.hh"
#include "util/Log.hh"
#include "util/strconv.hh"

namespace top10
{
namespace ui_interactive
{

  TrainingMenu::TrainingMenu(UIPanel* panel,
			     Menu* parent, ActionMapping* actions,
			     top10::graphX::Renderer* renderer,
			     top10::sound::SourceAllocator* sources,
			     top10::racing::LapRecordDb* laps):
  FrontMenu(panel, parent),
    m_session(renderer, sources, laps),
    m_actions(actions),
    m_renderer(renderer),
    m_steer_amount(0),
    m_is_shown(false)
  {
    m_setup = new SetupMenu(panel, this, &m_session, renderer);
    unsigned int id = addMenu(m_setup.getPtr());
    setMenuText(id, "Setup");

    setExitText("End session");

    updateNode();

    // When using the keyboard, steer events should not steer 100%
    m_steer_amount = (int)(getOptD("Control.Steer.Factor")*32767);
    if (m_steer_amount < 0 || m_steer_amount >32767)
    {
      top10::util::Log::getSingle()->send(top10::util::Log::Error, "TrainingMenu", 
	"Steer amount " + top10::util::toString(m_steer_amount) + " out of range. Forcing in-range value");
      m_steer_amount = 16000;
    }
  }



  TrainingMenu::~TrainingMenu()
  {
  }



  void TrainingMenu::handleFrontEvent(SDL_Event ev)
  {
    ActionMapping::const_iterator action = m_actions->find(ev);
	  
    if (action != m_actions->end())
    {
      switch(action->second)
      {
	  using namespace top10::racing;

	// Analog controls
      case Steering:
      case Acceleration:
      case Braking:
      case AccelerationAndBraking:
      case BrakingAndAcceleration:
	m_session.sendEvent(action->second, Control::getValue(ev));
	break;

	// Digital controls
      case SteerLeft:
	m_session.sendEvent(Steering, action->first.invertedEvent(ev)?0:-m_steer_amount);
	break;

      case SteerRight:
	m_session.sendEvent(Steering, action->first.invertedEvent(ev)?0:m_steer_amount);
	break;

      case Accelerate:
	m_session.sendEvent(Acceleration, action->first.invertedEvent(ev)?-32767:32767);
	break;

      case Brake:
	m_session.sendEvent(Braking, action->first.invertedEvent(ev)?-32767:32767);
	break;

      case GearDown:
	m_session.sendEvent(GearDown, action->first.invertedEvent(ev)?-32767:32767);
	break;

      case GearUp:
	m_session.sendEvent(GearUp, action->first.invertedEvent(ev)?-32767:32767);
	break;

      case ResetKart:
	m_session.sendEvent(ResetKart, 0);
	break;

	// Cameras
      case NextCamera:
	if (!action->first.invertedEvent(ev))  // We do not want to change the camera on key release
	  m_session.nextCamera();
	break;

      case PrevCamera:
	if (!action->first.invertedEvent(ev))
	  m_session.prevCamera();
	break;

      case NextKart:
	if (!action->first.invertedEvent(ev))
	  m_session.nextKart();
	break;

      case PrevKart:
	if (!action->first.invertedEvent(ev))
	  m_session.prevKart();
	break;

	// Switch to the free camera
      case FreeCamera:
	if (!action->first.invertedEvent(ev))
	  m_session.toggleFreeCamera();
	break;
              
	// Free moving camera movements
      case FreeCameraHorizontal:
	if (Control::getValue(ev) > 8000)
	  m_session.freeCamera()->rotateY((Control::getValue(ev)-8000)/-8000.0);
	else if (Control::getValue(ev) < -8000)
	  m_session.freeCamera()->rotateY((Control::getValue(ev)+8000)/-8000.0);

	m_session.freeCamera()->update();
	break;
              
      case FreeCameraVertical:
	if (Control::getValue(ev) > 8000)
	  m_session.freeCamera()->rotateX((Control::getValue(ev)-8000)/-8000.0);
	else if (Control::getValue(ev) < -8000)
	  m_session.freeCamera()->rotateX((Control::getValue(ev)+8000)/-8000.0);

	m_session.freeCamera()->update();
	break;

      case FreeCameraMoveX:
	if (Control::getValue(ev) > 8000) {
	  top10::math::Vector T((Control::getValue(ev)-8000)/-20000.0, 0.0, 0.0);
	  m_session.freeCamera()->translateL(T);
	}
	else if (Control::getValue(ev) < -8000) {
	  top10::math::Vector T((Control::getValue(ev)-8000)/-20000.0, 0.0, 0.0);                
	  m_session.freeCamera()->translateL(T);
	}
              
	m_session.freeCamera()->update();
	break;

      case FreeCameraMoveZ:
	if (Control::getValue(ev) > 8000) {
	  top10::math::Vector T(0.0, 0.0, (Control::getValue(ev)-8000)/-20000.0);
	  m_session.freeCamera()->translateL(T);
	}
	else if (Control::getValue(ev) < -8000) {
	  top10::math::Vector T(0.0, 0.0, (Control::getValue(ev)-8000)/-20000.0);                
	  m_session.freeCamera()->translateL(T);
	}

	m_session.freeCamera()->update();
	break;
              
      }
    }

  }



  void TrainingMenu::updateFrontData()
  {
    if (!m_is_shown)
      m_session.updateGame(m_clock.getTicks());
  }



  void TrainingMenu::renderGL()
  {
    m_session.renderGL();
  }



  void TrainingMenu::renderAL()
  {
    m_session.renderAL();
  }



  void TrainingMenu::startSession(std::string track_name, std::string kart_model,
                                  std::string kart_physx, bool rainy_weather)
  {
    m_session.createTrainingSession(getOptI("TrainingSession.NumVehicles"),
				    track_name, kart_model, kart_physx, rainy_weather);
    m_session.startGame(m_clock.getTicks());
  }



  void TrainingMenu::killSession()
  {
    m_session.terminateGame();
  }


  void TrainingMenu::on_panelShowed()
  {
    m_is_shown = true;
    m_clock.setSlowFactor(0.0);
    m_session.toggleSimulationSound(false);
  }


  void TrainingMenu::on_panelHidden()
  {
    m_is_shown = false;
    m_clock.setSlowFactor(1.0);
    m_session.toggleSimulationSound(true);
  }


  void TrainingMenu::on_panelFrontOn()
  {
    panel->stopMusic();
  }


  void TrainingMenu::on_panelFrontOff()
  {
    panel->playMusic();
  }

}
}
