/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "TrackScene.hh"
#include "util/GlobalOptions.hh"

namespace top10
{
  namespace ui_interactive
  {



    TrackScene::TrackScene(top10::graphX::Renderer* render):
      m_render(render)
    {
      // Lighting: using the default light (TODO: use a user-specified light instead)
      m_light = new top10::graphX::LightNode;

      m_static_shadow_method = top10::graphX::ShadowVolumeNode::DynamicOutline;
      m_dynamic_shadow_method = top10::graphX::ShadowVolumeNode::DynamicOutline;
    }




    TrackScene::~TrackScene()
    {
    }



    void TrackScene::setTrack(top10::track::Track* track)
    {
      if (m_track.isValid())
	m_light->removeChild(m_track->getTopNode());

      m_track = track;
      m_light->addChild(m_track->getTopNode());
      if (m_static_shadows.isValid())
      {
	m_static_shadows->addChild(
	  top10::graphX::makeProxy(track->getTopNode(),
				   new top10::graphX::ShadowVolumeNode::MyProxyOperation(m_static_shadow_method),
				   true) );
      }
    }



    void TrackScene::setSkyBox(const std::string& path)
    {
      if (path.empty())
	m_skybox = new top10::graphX::SkyBoxNode;
      else
	m_skybox = new top10::graphX::SkyBoxNode(path);
    }



    void TrackScene::disableSkyBox()
    {
      m_skybox.discard();
    }



    void TrackScene::setTrackShadows(top10::graphX::ShadowVolumeNode::Method m)
    {
      m_static_shadow_method = m;
      m_static_shadows = new top10::graphX::GroupNode;
      m_static_shadows->addChild(
	top10::graphX::makeProxy(m_track->getTopNode(),
				 new top10::graphX::ShadowVolumeNode::MyProxyOperation(m_static_shadow_method),
				 true) );

      // Setup the shadow volumes
      top10::graphX::Light light = m_light->getLight();
      top10::math::Vector light_pos;
      if (light.getType() == top10::graphX::Light::PointLight)
	light_pos = light.getPosition();
      else
	light_pos = -100.0 * light.getDirection();
      top10::graphX::ShadowVolumeNode::recurseUpdate(light_pos, m_static_shadows.getPtr());
    }



    void TrackScene::setKartShadows(top10::graphX::ShadowVolumeNode::Method m)
    {
      m_dynamic_shadow_method = m;
      m_dynamic_shadows = new top10::graphX::GroupNode;
      for (int i=0; i<m_transforms.size(); ++i)
      {
	m_dynamic_shadows->addChild(
	  top10::graphX::makeProxy(m_transforms[i].getPtr(),
				   new top10::graphX::ShadowVolumeNode::MyProxyOperation(m_dynamic_shadow_method),
				   true) );
      }
    }



    unsigned int TrackScene::newKart()
    {
      unsigned int ret = m_transforms.size();

      top10::graphX::TransformNodeRef trans( new top10::graphX::TransformNode );
      m_transforms.push_back( trans );

      top10::graphX::AlphaNodeRef alpha( new top10::graphX::AlphaNode );
      m_alphas.push_back( alpha );
      alpha->addChild( trans.getPtr() );

      top10::graphX::SwitchNodeRef sw( new top10::graphX::SwitchNode );
      m_switches.push_back( sw );
      sw->enable(0);
      sw->addChild( alpha.getPtr() );

      m_light->addChild( sw.getPtr() );

      m_karts.push_back( KartNodeRef(0) );

      return ret;
    }



    void TrackScene::showKart(unsigned int id)
    {
      m_switches[id]->enable(0);
    }



    void TrackScene::hideKart(unsigned int id)
    {
      m_switches[id]->enable(1);
    }



    void TrackScene::setKart(unsigned int id, const std::string& model)
    {
      top10::util::Ref<KartNode>
	kart_model(KartManager::getInstance()->getKart(model));

      top10::util::Ref<KartNode> kart_node(kart_model->clone());

      // Remove all existing nodes under the transform node.
      top10::graphX::Node* child;
      while ( (child = m_transforms[id]->getChild(0)) )
	m_transforms[id]->removeChild(child);

      m_transforms[id]->addChild(kart_node.getPtr());
      m_karts[id] = kart_node.getPtr();
    }



    void TrackScene::toggleGhost(unsigned int id, bool b)
    {
      if (b)
	m_alphas[id]->setAlpha(getOptUC("Render.Ghost.Alpha"));
      else
	m_alphas[id]->setAlpha(255);
    }



    void TrackScene::setKartTransform(unsigned int id,
				      const top10::math::Matrix4& M)
    {
      m_transforms[id]->setToWorld(M);
    }



    void TrackScene::setKartState(unsigned int id,
				  double steer, double left_steer, double right_steer,
				  double h_fl, double h_fr, double h_rl, double h_rr,
				  double rot_fl, double rot_fr, double rot_rl, double rot_rr)
    {
      m_karts[id]->update(steer, left_steer, right_steer,
			  h_fl, h_fr, h_rl, h_rr,
			  rot_fl, rot_fr, rot_rl, rot_rr);
    }



    void TrackScene::renderGL()
    {
      // The sky
      if (m_skybox.isValid())
      {
	top10::graphX::RenderList skybox_list;
	m_render->buildList(m_skybox.getPtr(), &skybox_list);
	m_render->renderSkyBoxGL(skybox_list);
	m_render->clearDepthGL();
      }

      // The track and the karts
      top10::graphX::RenderList rl;
      m_render->buildList(m_light.getPtr(), &rl);
      m_render->setAlphaTest(GL_GREATER, 0.5f);	  // Quick hack: never draw non-opaque fragments
      m_render->renderGL(rl);
      m_render->setAlphaTest(GL_ALWAYS, 0.0f);

      // Update the dynamic shadow volumes
      top10::util::Ref<top10::graphX::GroupNode> shadows(new top10::graphX::GroupNode);
      if (m_dynamic_shadows.isValid())
      {
	top10::graphX::Light light = m_light->getLight();
	top10::math::Vector light_pos;
	if (light.getType() == top10::graphX::Light::PointLight)
	  light_pos = light.getPosition();
	else
	  light_pos = -100.0 * light.getDirection();
	top10::graphX::ShadowVolumeNode::recurseUpdate(light_pos, m_dynamic_shadows.getPtr());

	shadows->addChild(m_dynamic_shadows.getPtr());
      }
      if (m_static_shadows.isValid())
	shadows->addChild(m_static_shadows.getPtr());

      // Draw the shadows
      if (shadows->getChild(0))
      {
	top10::graphX::ShadowVolumeNode::renderShadows(
	  m_render,
	  shadows.getPtr(),
	  getOptUC("Render.Shadow.Alpha")
	  );
      }
    }


  }
}
