#include "StaticScene.hh"
#include "helpers/MeshFactory.hh"
#include "util/Log.hh"
#include "graphX/GroupNode.hh"
#include <algorithm>

namespace
{
  struct MeshInserterFunction
  {
    MeshInserterFunction(top10::graphX::Node* node,
			 top10::helpers::MeshEditor* ed)
      : m_node(node),
	m_ed(ed)
      {
      }

    void operator()(const std::string& name) const
      {
	m_ed->gotoMesh(name);
	m_node->addChild(m_ed->getCurrentNode());
      }

    top10::graphX::Node* m_node;
    top10::helpers::MeshEditor* m_ed;
  };
}

namespace top10
{
  namespace ui_interactive
  {
    using top10::util::Log;
    typedef top10::util::XmlDumpable XmlT;

    const char * StaticScene::s_nodename = "static_scene";

    StaticScene::StaticScene():
      XmlT(s_nodename)
    {
      clearState();
    }



    void StaticScene::clearState()
    {
      m_meshes.clearState();
      m_views.clear();
      setLightNode( top10::math::Vector(0.0, 0.0, 0.0) );
    }



    int StaticScene::loadXml(const TiXmlElement* el)
    {
      unsigned int num_load_mesh = 0;
      unsigned int num_load_cam = 0;
      const TiXmlElement* child = el->FirstChildElement();
      while (child)
      {
	if (m_meshes.handles(child))
	{
	  ++num_load_mesh;
	  m_meshes.load(child);
	}
	else if (child->Value() == std::string("camera_matrix"))
	{
	  ++num_load_cam;
	  loadCameraMatrix(child);
	}
	else if (child->Value() == std::string("camera_lookat"))
	{
	  ++num_load_cam;
	  loadCameraLookAt(child);
	}
	else if (child->Value() == std::string("light"))
	{
	  loadLight(child);
	}
	else
	  Log::getSingle()->send(
	    Log::Warning,
	    "StaticScene/loadXml",
	    std::string("Skipping node ") + child->Value() );

	child = child->NextSiblingElement();
      }

      if (num_load_mesh == 0)
      {
	Log::getSingle()->send(
	  Log::Error,
	  "StaticScene/loadXml",
	  "Found no mesh editor." );
      }
      else if (num_load_mesh > 1)
      {
	Log::getSingle()->send(
	  Log::Error,
	  "StaticScene/loadXml",
	  "Found too many mesh editors." );
      }

      if (num_load_cam == 0)
      {
	Log::getSingle()->send(
	  Log::Error,
	  "StaticScene/loadXml",
	  "Found no camera." );
      }

      return 0;
    }



    void StaticScene::loadCameraMatrix(const TiXmlElement* el)
    {
      top10::math::Matrix4 M;
      top10::math::loadXml(el, &M);

      top10::math::Vector pos(0.0, 0.0, 0.0);
      top10::math::Vector up(0.0, 1.0, 0.0);
      top10::math::Vector dir(1.0, 0.0, 0.0);

      pos = M*pos;
      up = M*up;
      dir = M*dir;

      top10::math::Frustum view;
      view.setFOV(35.0);
      view.setNear(0.1);
      view.setFar(100.0);
      view.setCenter(pos);
      view.setDirection(dir, up);
      view.update();
      
      m_views.push_back(view);
    }



    void StaticScene::loadCameraLookAt(const TiXmlElement* el)
    {
      top10::math::Vector pos, look_at;

      const TiXmlElement* child = el->FirstChildElement();
      while (child)
      {
	if (child->Value() == std::string("position"))
	  pos.loadXml(child);
	else if (child->Value() == std::string("lookat"))
	  look_at.loadXml(child);

	child = child->NextSiblingElement();
      }

      top10::math::Vector dir = look_at - pos;
      double sz = dir.size();
      if (sz > SMALL_VECTOR)
	dir /= sz;
      else
	dir = top10::math::Vector(0.0, 0.0, -1.0);

      top10::math::Frustum view;

      view.setFOV(35.0);
      view.setNear(0.1);
      view.setFar(100.0);
      view.setCenter(pos);
      view.setDirection(dir, top10::math::Vector(0.0, 1.0, 0.0));
      view.update();
      
      m_views.push_back(view);      
    }



    void StaticScene::loadLight(const TiXmlElement* el)
    {
      top10::math::Vector pos;
      pos.loadXml(el);
      setLightNode(pos);
    }



    void StaticScene::setLightNode(const top10::math::Vector& pos)
    {
      top10::graphX::Light light(top10::graphX::Light::PointLight);
      light.setPosition(pos);

      m_light = new top10::graphX::LightNode;

      std::list<std::string> mesh_names = m_meshes.getMeshNames();
      MeshInserterFunction f(m_light.getPtr(), &m_meshes);
      std::for_each(mesh_names.begin(), mesh_names.end(), f);

      m_light->setLight(light);
    }



    int StaticScene::saveXml(TiXmlElement* el) const
    {
      TiXmlElement child("");
      m_meshes.save(&child);
      el->InsertEndChild(child);

      for (std::vector<top10::math::Frustum>::const_iterator it = m_views.begin();
	   it != m_views.end();
	   ++it)
      {
	saveCamera(el, *it);
      }

      return 0;
    }



    void StaticScene::saveCamera(TiXmlElement* el,
				 const top10::math::Frustum& view) const
    {
      TiXmlElement child("camera_matrix");

      top10::math::Matrix4 M = view.getOrient();
      top10::math::Translation4 T(-view.getCenter());
      top10::math::Matrix4 R = T*M;
      top10::math::saveXml(&child, R);
      
      el->InsertEndChild(child);
    }



    top10::math::Frustum
    StaticScene::getView(unsigned int id) const
    {
      if (id >= m_views.size())
	return top10::math::Frustum();

      return m_views.at(id);
    }



    unsigned int
    StaticScene::getNumViews() const
    {
      return m_views.size();
    }



    const top10::graphX::Node*
    StaticScene::getNode() const
    {
      return m_light.getPtr();
    }
  }
}
