#include "SetupScreen.hh"
#include "graphX/Renderer.hh"
#include "graphX/TextNode.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/CameraNode.hh"
#include "physX/Kart.hh"
#include "util/GlobalOptions.hh"
#include "util/PathFinder.hh"

namespace top10
{
  namespace ui_interactive
  {
    SetupScreen::SetupScreen(top10::graphX::Renderer* render,
			     fntFont* font):
      m_render(render),
      m_font(font),
      m_curr_comp(top10::physX::SettingsState::COMP_FIRST)
    {
      m_settings = new top10::physX::Settings;
      loadSettings();
      m_state.m_settings = m_settings.getConstPtr();
      
      loadScene();

      propagate();
    }



    void SetupScreen::loadSettings()
    {
      const std::string filename = getOptS("Setup.ConfigFile");
      top10::util::PathFinder finder = 
	top10::util::PathFinder::defaultPathFinder();

      const std::string path = finder.find(filename);
      if (path.empty())
      {
	top10::util::Log::getSingle()->send(
	  top10::util::Log::Critical,
	  "SetupScreen/loadSettings",
	  "Could not find file " + filename );

	m_settings.discard();

	return;
      }

      TiXmlDocument doc(path);
      if (!doc.LoadFile(path))
      {
	top10::util::Log::getSingle()->send(
	  top10::util::Log::Critical,
	  "SetupScreen/loadSettings",
	  "Could not open file " + path );

	m_settings.discard();

	return;
      }

      int status = m_settings->load(doc.RootElement());
      if (status != 0)
	m_settings.discard();
    }



    void SetupScreen::loadScene()
    {
      const std::string filename = getOptS("Setup.Scene");
      top10::util::PathFinder finder = 
	top10::util::PathFinder::defaultPathFinder();

      const std::string path = finder.find(filename);
      if (path.empty())
      {
	top10::util::Log::getSingle()->send(
	  top10::util::Log::Critical,
	  "SetupScreen/loadScene",
	  "Could not find file " + filename );

	return;
      }

      TiXmlDocument doc(path);
      if (!doc.LoadFile(path))
      {
	top10::util::Log::getSingle()->send(
	  top10::util::Log::Critical,
	  "SetupScreen/loadScene",
	  "Could not open file " + path );

	return;
      }

      int status = m_scene.load(doc.RootElement());
      if (status != 0)
	return;

      m_scene_node = m_scene.getNode();
    }



    void SetupScreen::update()
    {
      // TODO: camera movements to center on the component
    }



    void SetupScreen::renderGL()
    {
      if (!m_settings.isValid())
	return;

      // Render the scene
      m_render->setCamera(getCamera());
      top10::graphX::RenderList rl;
      m_render->buildList(m_scene_node.getPtr(), &rl);
      m_render->renderGL(rl);

      // Render the text
      top10::util::Ref<top10::graphX::TextNode> setting_text;
      setting_text = new top10::graphX::TextNode;

      setting_text->setText( getStr(m_curr_comp) + ": " + 
			     m_state.getDesc(m_curr_comp) );

      setting_text->setPointSize(0.075);
      setting_text->setFont(m_font);
      setting_text->setPos(-0.9, -0.9);

      top10::util::Ref<top10::graphX::MaterialNode> color;
      color = new top10::graphX::MaterialNode;
      color->r = color->g = 0;
      color->b = 200;
      color->addChild(setting_text.getPtr());

      rl.clear();
      m_render->buildList(color.getPtr(), &rl);
      m_render->renderHudGL(rl);
    }

    

    void SetupScreen::increase()
    {
      if (!m_settings.isValid())
	return;

      m_state.incr(m_curr_comp);

      propagate();
    }



    void SetupScreen::decrease()
    {
      if (!m_settings.isValid())
	return;

      m_state.decr(m_curr_comp);

      propagate();
    }



    void SetupScreen::nextComponent()
    {
      if (!m_settings.isValid())
	return;

      if (m_curr_comp == top10::physX::SettingsState::COMP_LAST)
	m_curr_comp = top10::physX::SettingsState::COMP_FIRST;
      else
	m_curr_comp = top10::physX::SettingsState::Components(m_curr_comp +1);
    }



    void SetupScreen::prevComponent()
    {
      if (!m_settings.isValid())
	return;

      if (m_curr_comp == top10::physX::SettingsState::COMP_FIRST)
	m_curr_comp = top10::physX::SettingsState::COMP_LAST;
      else
	m_curr_comp = top10::physX::SettingsState::Components(m_curr_comp -1);
    }



    void SetupScreen::propagate()
    {
      if (!m_settings.isValid())
	return;

      typedef top10::physX::SettingsState T;

      //TODO: the "breadth" value are not currently used.
      top10::physX::Kart::BACK_LONG_K =
	m_settings->m_tyre_settings.at(
	  m_state.m_idxs[T::COMP_REAR_TYRE_STICKYNESS] ).m_long_k;

      top10::physX::Kart::BACK_SIDE_K =
	m_settings->m_tyre_settings.at(
	  m_state.m_idxs[T::COMP_REAR_TYRE_STICKYNESS] ).m_side_k;

      top10::physX::Kart::FRONT_LONG_K =
	m_settings->m_tyre_settings.at(
	  m_state.m_idxs[T::COMP_FRONT_TYRE_STICKYNESS] ).m_long_k;

      top10::physX::Kart::FRONT_SIDE_K =
	m_settings->m_tyre_settings.at(
	  m_state.m_idxs[T::COMP_FRONT_TYRE_STICKYNESS] ).m_side_k;
      
      top10::physX::Kart::FRONT_STIFFNESS =
	m_settings->m_stiff_settings.at(
	  m_state.m_idxs[T::COMP_FRONT_STIFFNESS] ).m_stiff;

      top10::physX::Kart::BACK_STIFFNESS =
	m_settings->m_stiff_settings.at(
	  m_state.m_idxs[T::COMP_REAR_STIFFNESS] ).m_stiff;

      top10::physX::Kart::ENGINE_RPS_OPT =
	m_settings->m_engine_settings.at(
	  m_state.m_idxs[T::COMP_ENGINE_USAGE] ).m_opt;
    }



    top10::graphX::CameraNode*
    SetupScreen::getCamera() const
    {
      top10::graphX::CameraNode* ret = new top10::graphX::CameraNode;
      ret->setView( m_scene.getView(m_curr_comp) );
      return ret;
    }
  }
}
