/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "ReplayRenderer.hh"
#include "FogValues.hh"
#include "trackedit/TrackFactory.hh"
#include "util/GlobalOptions.hh"
#include <fstream>

namespace top10
{
  namespace ui_interactive
  {



    ReplayRenderer::ReplayRenderer(top10::graphX::Renderer* render):
      m_render(render)
    {
    }



    void ReplayRenderer::setReplay(const top10::racing::LapRecord& lap,
				   const std::string& track_name,
				   const std::string& kart_name,
				   bool foggy)
    {
      m_lap = lap;

      // Create the track
      top10::track::TrackRef track;
      track = top10::tracked::TrackFactory::getSingle()->loadTrack(track_name);

      m_track_scene = new TrackScene(m_render);
      m_track_scene->setTrack(track.getPtr());

      // Shadows by the track.
      const bool track_shadow_enabled =
	getEnabledOpt("Render.Track.Shadow.Quality");
      if (track_shadow_enabled)
      {
	const std::string method_str =
	  getOptS("Render.Track.Shadow.Quality");

	top10::graphX::ShadowVolumeNode::Method method =
	  top10::graphX::ShadowVolumeNode::toMethod(method_str);

	m_track_scene->setTrackShadows(method);
      }

      // The skybox
      if (!foggy)
      {
	m_track_scene->setSkyBox(
	  top10::tracked::TrackFactory::getSingle()->getSkyboxName(track_name));
      }
      
      // Fog values
      FogValues fog;
      if (foggy)
	fog = FogValues::createRainy();
      else
	fog = FogValues::createClear();

      m_render->setClearColor(fog.getRed(), fog.getGreen(), fog.getBlue());
      m_render->setFog(fog.getDensity(), fog.getNear(), fog.getFar());

      // Create a kart
      m_kart_id = m_track_scene->newKart();
      m_track_scene->setKart(m_kart_id, kart_name);

      // Kart shadows
      const bool kart_shadow_enabled =
	getEnabledOpt("Render.Kart.Shadow.Quality");
      if (kart_shadow_enabled)
      {
	const std::string method_str =
	  getOptS("Render.Kart.Shadow.Quality");

	top10::graphX::ShadowVolumeNode::Method method =
	  top10::graphX::ShadowVolumeNode::toMethod(method_str);

	m_track_scene->setKartShadows(method);
      }

      // Update time markers
      m_last_update = top10::util::global_time.getTicks();
      m_timer.setSlowFactor(0.0);
      m_last_frame = m_timer.getTicks();
      m_start_frame = m_last_frame;

      // Set camera movement rates
      m_rot_rate_h = 0.0;
      m_rot_rate_v = 0.0;
      m_move_speed = 0.0;
      m_zoom_rate = 1.0;

      cameraPreset(POS_BACK);
      gotoStart();
      pause();
    }



    void ReplayRenderer::update()
    {
      if (!m_track_scene.isValid())
	return;

      using top10::math::Vector;

      Uint32 now = top10::util::global_time.getTicks();
      Uint32 now_frame = m_timer.getTicks();

      double dt = 0.001 * (now - m_last_update);

      m_rot_h += m_rot_rate_h *dt;
      m_rot_v += m_rot_rate_v *dt;
      m_dist += m_move_speed *dt;
//      m_fov *= m_zoom_rate *dt;


      // Set position and direction according to the kart's position

      top10::racing::KartState s =
	m_lap.getKartState(now_frame - m_start_frame);
      
      Vector dir = s.orient * (Vector(1.0, 0.0, 1.0) | m_local_dir);
      dir.y = m_local_dir.y;
      double dir_sz = dir.size();
      if (dir_sz > SMALL_VECTOR)
	dir /= dir_sz;
      else
	dir = Vector(1.0, 0.0, 0.0);

      top10::math::Rotation3 R0(M_PI*m_rot_h/180.0, Vector(0.0, 1.0, 0.0));
      top10::math::Rotation3 R1(M_PI*m_rot_v/180.0, dir ^ Vector(0.0, 1.0, 0.0));
      dir = R0 * (R1 * dir);
      Vector pos = s.translation -m_dist*dir;

      
      // Set the camera
      top10::math::Frustum view(pos, dir, Vector(0.0, 1.0, 0.0), 1.0,
				m_fov, 0.05, 1000.0);

      top10::util::Ref<top10::graphX::CameraNode> cam;
      cam = new top10::graphX::CameraNode;
      cam->setView(view);
      
      m_render->setCamera(cam.getPtr());

      m_last_update = now;
      m_last_frame = now_frame;


      // Update the track scene
      m_track_scene->setKartTransform(m_kart_id, s.getTransform());
      m_track_scene->setKartState(m_kart_id,
				  s.steer, s.steer, s.steer,
				  0.0, 0.0, 0.0, 0.0,
				  0.0, 0.0, 0.0, 0.0);

      // Stop the replay if the replay has reached the end.
      if (now_frame-m_start_frame >= m_lap.getTime())
	pause();

      if (now_frame < m_start_frame)
	pause();
    }



    void ReplayRenderer::renderGL()
    {
      if (!m_track_scene.isValid())
	return;

      m_track_scene->renderGL();
    }



    void ReplayRenderer::pause()
    {
      m_timer.setSlowFactor(0.0);
    }



    void ReplayRenderer::forward(double speed)
    {
      if (m_last_frame-m_start_frame > m_lap.getTime())
	return;

      m_timer.setSlowFactor(1.0/speed);
    }



    void ReplayRenderer::backward(double speed)
    {
      if (m_last_frame <= m_start_frame)
	return;

      m_timer.setSlowFactor(-1.0/speed);
    }



    void ReplayRenderer::gotoStart()
    {
      m_start_frame = m_timer.getTicks();
      m_last_frame = m_start_frame;
    }



    void ReplayRenderer::gotoEnd()
    {
      m_timer.setSlowFactor(0.001);
    }



    void ReplayRenderer::cameraRotateH(double rad_per_sec)
    {
      m_rot_rate_h = rad_per_sec;
    }



    void ReplayRenderer::cameraRotateV(double rad_per_sec)
    {
      m_rot_rate_v = rad_per_sec;
    }



    void ReplayRenderer::cameraMove(double meters_per_sec)
    {
      m_move_speed = meters_per_sec;
    }



    void ReplayRenderer::cameraZoom(double factor_per_sec)
    {
      m_zoom_rate = factor_per_sec;
    }



    void ReplayRenderer::cameraPreset(PresetPosition pos)
    {
      using top10::math::Vector;

      if (!m_track_scene.isValid())
	return;

      switch (pos)
      {
      case POS_FRONT:
	m_local_dir = Vector(-1.0, -0.2, 0.0);
	break;

      case POS_BACK:
	m_local_dir = Vector(1.0, -0.2, 0.0);
	break;

      case POS_LEFT:
	m_local_dir = Vector(0.0, -0.2, 1.0);
	break;

      case POS_RIGHT:
	m_local_dir = Vector(0.0, -0.2, -1.0);
	break;

      case POS_TOP:
	m_local_dir = Vector(0.0, -1.0, 0.0);
	break;
      }

      m_rot_h = 0.0;
      m_rot_v = 0.0;
      m_dist = 10.0;
      m_fov = 60.0;
    }


  }
}
