/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "ReplayMenu.hh"
#include "YesNoMenu.hh"
#include "util/Log.hh"
#include "util/GlobalOptions.hh"

namespace
{
  static const double ROTATE_RATE = 30.0;
  static const double MOVE_SPEED = 1.0;
  static const double ZOOM_RATE = 0.01;
}

namespace top10
{
  namespace ui_interactive
  {

    ReplayMenu::ReplayMenu(UIPanel* panel, Menu* parent, top10::graphX::Renderer* render, top10::racing::LapRecordDb* laps):
      FrontMenu(panel, parent),
      m_laps(laps),
      m_render(render)
    {
      for (unsigned int i=0; i < m_laps->getNumEntries(); ++i)
      {
	unsigned int id = addMenu(0);
	setMenuText(id, m_laps->getEntry(i).getRepr());
      }

      setExitText("End session");

      setScaleX(getOptD("Playback.Menu.Scale.X"));
      setScaleY(getOptD("Playback.Menu.Scale.Y"));
      setPosX(getOptD("Playback.Menu.Position.X"));
      setPosY(getOptD("Playback.Menu.Position.Y"));
      setCursorX(getOptD("Playback.Menu.Cursor.X"));

      updateNode();
    }



    void ReplayMenu::handleFrontEvent(SDL_Event e)
    {
      if (!m_session.isValid())
	return;

      if (e.type == SDL_KEYDOWN || e.type == SDL_KEYUP)
      {
	double enable = 0.0;
	if (e.type == SDL_KEYDOWN)
	  enable = 1.0;

	switch (e.key.keysym.sym)
	{
	case SDLK_UP:
	  m_session->cameraRotateV(enable * ROTATE_RATE);
	  break;

	case SDLK_DOWN:
	  m_session->cameraRotateV(-enable * ROTATE_RATE);
	  break;

	case SDLK_LEFT:
	  m_session->cameraRotateH(-enable * ROTATE_RATE);
	  break;

	case SDLK_RIGHT:
	  m_session->cameraRotateH(enable * ROTATE_RATE);
	  break;

	case SDLK_PAGEUP:
	  m_session->cameraMove(-enable * MOVE_SPEED);
	  break;

	case SDLK_PAGEDOWN:
	  m_session->cameraMove(enable * MOVE_SPEED);
	  break;

	case SDLK_z:
	  m_session->cameraZoom(1.0 + enable * ZOOM_RATE);
	  break;

	case SDLK_x:
	  m_session->cameraZoom(1.0 -enable * ZOOM_RATE);
	  break;

	case SDLK_KP8:
	  if (e.type == SDL_KEYDOWN)
	    m_session->cameraPreset(ReplayRenderer::POS_FRONT);
	  break;

	case SDLK_KP2:
	  if (e.type == SDL_KEYDOWN)
	    m_session->cameraPreset(ReplayRenderer::POS_BACK);
	  break;

	case SDLK_KP4:
	  if (e.type == SDL_KEYDOWN)
	    m_session->cameraPreset(ReplayRenderer::POS_LEFT);
	  break;

	case SDLK_KP6:
	  if (e.type == SDL_KEYDOWN)
	    m_session->cameraPreset(ReplayRenderer::POS_RIGHT);
	  break;

	case SDLK_SPACE:
	  if (e.type == SDL_KEYDOWN)
	    m_session->forward(1.0);
	  break;

	case SDLK_PAUSE:
	  if (e.type == SDL_KEYDOWN)
	    m_session->pause();
	  break;

	case SDLK_HOME:
	  if (e.type == SDL_KEYDOWN)
	  {
	    m_session->gotoStart();
	    m_session->pause();
	  }
	  break;

	case SDLK_END:
	  if (e.type == SDL_KEYDOWN)
	  {
	    m_session->gotoEnd();
	    m_session->pause();
	  }
	  break;
	  
	case SDLK_BACKSPACE:
	  if (e.type == SDL_KEYDOWN)
	  {
	    m_session->backward(2.0);
	  }
	  break;

	case SDLK_s:
	  if (e.type == SDL_KEYDOWN)
	  {
	    m_session->forward(0.25);
	  }
	  break;

	case SDLK_f:
	  if (e.type == SDL_KEYDOWN)
	  {
	    m_session->forward(2.0);
	  }
	  break;

	default:
	  break;
	}
      }
    }



    void ReplayMenu::updateFrontData()
    {
      if (m_session.isValid())
	m_session->update();
    }



    void ReplayMenu::renderGL()
    {
      if (m_session.isValid())
	m_session->renderGL();
    }



    void ReplayMenu::renderAL()
    {
    }



    void ReplayMenu::startSession(const top10::racing::LapRecordEntry& lap)
    {
      if (m_session.isValid())
	m_session.discard();

      try
      {
	m_session = new ReplayRenderer(m_render);
	m_session->setReplay(lap.getLap(), lap.getTrackName(), lap.getKartName(), lap.isFoggy());
      }
      catch(const std::string& err)
      {
	top10::util::Log::getSingle()->send(top10::util::Log::Warning,
					    "ReplayMenu/startSession",
					    "Failed to load replay: "+
					    err);
	m_session.discard();
      }
    }



    void ReplayMenu::killSession()
    {
      if (m_session.isValid())
	m_session.discard();
    }



    void ReplayMenu::selected()
    {
      if (getEntryN() +1 == getNumEntries())
	FrontMenu::selected();
      else
	startSession( m_laps->getEntry(getEntryN()) );
    }

  }
}
