#ifndef TOP10_UI_MENU_HH
#define TOP10_UI_MENU_HH

#include "util/RefCount.hh"
#include "util/NoCopy.hh"
#include "extras/GLee.h"
#include "graphX/GroupNode.hh"
#include "sound/SourceAllocator.hh"

#include <vector>
#include <string>
#include <SDL/SDL.h>

class fntFont;

namespace top10
{
  namespace ui_interactive
  {

    class UIPanel;

    //! A menu which can be shown on a UIPanel.
    class Menu: public top10::util::RefCount, public top10::util::NoCopy
    {
    public:
      Menu(UIPanel*);
      static void makeSounds(top10::sound::SourceAllocator*);
      virtual ~Menu();

      virtual unsigned int addMenu(Menu*);
      void setMenuText(unsigned int, const std::string& text);
      const std::string& getMenuText(unsigned int) const;
      Menu* getMenu(unsigned int) const;

      virtual bool isFrontType() const;

      //! Called when a panel is showed and this menu is the front one.
      /*! Does nothing by default. */
      virtual void on_panelShowed();

      //! Called when a panel is hidden and this menu is the front one.
      /*! Does nothing by default. */
      virtual void on_panelHidden();

      //! Called when this menu is made the front menu of a panel.
      /*! Does nothing by default. */
      virtual void on_panelFrontOn();

      //! Called when this menu is taken away as the front menu in a panel.
      /*! Does nothing by default. */
      virtual void on_panelFrontOff();

      //! By default: show linked menu.
      virtual void selected();

      //! By default: does nothing.
      virtual	void updateFrontData();
      //! By default: does nothing.
      virtual	void renderGL();
      //! By default: does nothing.
      virtual void renderAL();

      /*! By default:
      Key up -> move to the previous menu entry
      Key down -> ... next entry
      Space -> Select current entry
      */
      virtual void event(SDL_Event);

      /*! By default: does nothing. */
      virtual void handleFrontEvent(SDL_Event);

      //! Move to next entry.
      int nextEntry();

      //! Move to previous entry.
      int prevEntry();

      //! Get the current entry number.
      inline int getEntryN() const { return entry_n; }

      //! Get the node showing this menu.
      top10::graphX::Node* getNode();

      //! Set the font used by all menus.
      static void setFont(const std::string&);

      //! Set the horizontal scale factor
      inline void setScaleX(double s) { scale_x = s; }

      //! Set the vertical scale factor
      inline void setScaleY(double s) { scale_y = s; }

      //! Set the horizontal position of the menu.
      inline void setPosX(double x) { start_x = x; }

      //! Set the vertical position of the menu.
      inline void setPosY(double y) { start_y = y; }

      //! Set the horizontal position of the cursor.
      inline void setCursorX(double x) { cursor_x = x; }

    protected:
      unsigned int num_rows;

      //! Panel showing this menu.
      UIPanel* panel;

      //! Recreate the nodes showing this menu.
      virtual void updateNode();

      //! Set the font singleton.
      static fntFont* getFont();

    protected: // For FrontMenu
      struct Entry
      {
	std::string text;
	top10::util::Ref<Menu> link;
      };

      std::vector<Entry> entries;

    protected:
      unsigned int getNumEntries() const;
      void setSelected(const Entry& entry);

    private:
      //! Current entry.
      int entry_n;

      unsigned char text_r;
      unsigned char text_g;
      unsigned char text_b;
      double scale_x;
      double scale_y;
      double start_x;
      double start_y;
      double cursor_x;
      std::string cursor_text;
      double point_sz;
      double spacing;

      /*!
      The parent of a list of TextNodes (one for each entry in the menu),
      and a pointer showing the current node.
      */
      top10::util::Ref< top10::graphX::GroupNode > node;

      //! The font used by all menus.
      static fntFont* font;

      //! The sound played when moving to an entry
      static top10::sound::Source* src_navigate;

      //! The sound played when selecting an entry
      static top10::sound::Source* src_select;
    };
    typedef top10::util::Ref<Menu> MenuRef;

    //! A menu which is set to the front menu when selected.
    /*!
      The menu has a last entry which returns to the parent menu when selected.
      */
    class FrontMenu: public Menu
    {
    public:
      FrontMenu(UIPanel*, Menu* parent);
      void setExitText(const std::string&);
      unsigned int addMenu(Menu* m);

      //! Delegates to Menu if the selected entry is not the last one.
      void selected();

      //! Returns true.
      bool isFrontType() const;

    private:
      //! End the session handled by this menu.
      /*! Does nothing by default. */
      virtual void killSession();

      Menu* m_parent; // Not a Ref to avoid creating cycles.
      std::string m_text;
    };
  }
}

#endif
