#include "Menu.hh"
#include "UIPanel.hh"

#include "util/GlobalOptions.hh"
#include "util/PathFinder.hh"
#include "util/Log.hh"

#include "graphX/TextNode.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/TransformNode.hh"
#include "sound/BufferManager.hh"
#include "sound/Listener.hh"
#include <iostream>
#include <plib/fnt.h>

namespace top10
{
  namespace ui_interactive
  {

    fntFont* Menu::font = 0;
    top10::sound::Source* Menu::src_navigate = 0;
    top10::sound::Source* Menu::src_select = 0;

    Menu::Menu(UIPanel* _panel):
    panel(_panel),
      entry_n(0)
    {
      num_rows = getOptI("Menu.NumEntries");
      text_r = getOptUC("Menu.Text.Color.Red");
      text_g = getOptUC("Menu.Text.Color.Green");
      text_b = getOptUC("Menu.Text.Color.Blue");
      scale_x = getOptD("Menu.Scale.X");
      scale_y = getOptD("Menu.Scale.Y");
      start_x = getOptD("Menu.Position.X");
      start_y = getOptD("Menu.Position.Y");
      cursor_x = getOptD("Menu.Cursor.X");
      cursor_text = getOptS("Menu.Cursor.String");
      point_sz = getOptD("Menu.Text.PointSize");
      spacing = getOptD("Menu.Text.Spacing");

      top10::util::RefCount::setDeallocateFlag(false);
    }


    Menu::~Menu()
    {
    }


    void Menu::makeSounds(top10::sound::SourceAllocator* alloc)
    {
      using namespace top10::sound;
      BufferManager* manager = BufferManager::getSingle();

      top10::util::PathFinder finder = top10::util::PathFinder::defaultPathFinder();
      std::string path = finder.find("navigate.wav");
      if (!path.empty())
      {
	ALuint buf = manager->load(path);
	src_navigate = alloc->getNew(buf);
      }
      else
      {
	top10::util::Log::getSingle()->send(top10::util::Log::Error, "UIPanel/makeSounds", "Failed to find navigate.wav");
      }

      path = finder.find("select.wav");
      if (!path.empty())
      {
	ALuint buf = manager->load(path);
	src_select = alloc->getNew(buf);
      }
      else
      {
	top10::util::Log::getSingle()->send(top10::util::Log::Error, "UIPanel/makeSounds", "Failed to find select.wav");
      }

    }

    unsigned int Menu::addMenu(Menu* m)
    {
      unsigned int id = entries.size();

      Entry new_one;
      new_one.link = m;
      new_one.text = "???";

      entries.push_back(new_one);

      return id;
    }

    void Menu::setMenuText(unsigned int id, const std::string& text)
    {
      entries.at(id).text = text;
    }

    const std::string& Menu::getMenuText(unsigned int id) const
    {
      return entries.at(id).text;
    }

    Menu* Menu::getMenu(unsigned int id) const
    {
      return entries.at(id).link.getPtr();
    }

    unsigned int Menu::getNumEntries() const
    {
      return entries.size();
    }

    void Menu::selected()
    {
      assert(entries.empty() || (entry_n >= 0 && entry_n < (int)entries.size()));
      if (!entries.empty())
      {
	if (src_select)
	  src_select->play();

	setSelected(entries.at(entry_n));
      }
    }

    void Menu::setSelected(const Entry& entry)
    {
      if (entry.link.isValid() && entry.link->isFrontType())
	panel->setFrontMenu(entry.link.getPtr());

      panel->setMenu(entry.link.getPtr());
    }

    void Menu::event(SDL_Event event)
    {
      if (event.type == SDL_KEYDOWN)
      {
	switch (event.key.keysym.sym)
	{
	case SDLK_SPACE:
	case SDLK_RETURN:
	  selected();
	  break;

	case SDLK_UP:
	  prevEntry();
	  break;

	case SDLK_DOWN:
	  nextEntry();
	  break;

	default:
	  break;
	}
      }
    }

    int Menu::nextEntry()
    {
      if (entries.empty()) {
	assert(entry_n==0);
	return entry_n;
      }

      if (entry_n == ((int)entries.size())-1) entry_n = 0;
      else entry_n++;

      if (src_navigate)
      {
	src_navigate->stop();
	src_navigate->play();
      }

      updateNode();

      return entry_n;
    }

    int Menu::prevEntry()
    {
      if (entries.empty()) {
	assert(entry_n==0);
	return entry_n;
      }

      if (entry_n == 0) entry_n = entries.size()-1;
      else entry_n--;

      if (src_navigate)
      {
	src_navigate->stop();
	src_navigate->play();
      }

      updateNode();

      return entry_n;
    }

    void Menu::setFont(const std::string& font_name)
    {
      if (font)
      {
	delete font;
	font = 0;
      }

      std::string font_path = top10::util::PathFinder::defaultPathFinder().find(font_name);
      if (font_path.empty())
	throw std::string("Could not load font ") + font_name;

      font = new fntTexFont;
      font->load(font_path.c_str());
    }

    fntFont* Menu::getFont()
    {
      if (!font)
	setFont("default.txf");

      return font;
    }

    void Menu::updateNode()
    {
      int first_entry = entry_n - num_rows/2;
      if (first_entry < 0)
	first_entry = 0;

      int last_entry = first_entry + num_rows;
      if (last_entry > (int)entries.size() -1)
	last_entry = (int)entries.size() -1;

      top10::graphX::MaterialNode* color = new top10::graphX::MaterialNode;
      color->r = text_r;
      color->g = text_g;
      color->b = text_b;

      top10::graphX::TransformNode* transform = new top10::graphX::TransformNode;
      top10::math::Identity4 scale;
      scale(0, 0) = scale_x;
      scale(1, 1) = scale_y;
      transform->setToWorld(scale);
      transform->addChild(color);

      node = transform;

      int i = first_entry;
      double y = start_y;
      for (; i <= last_entry; ++i)
      {
	top10::graphX::TextNode* text_node = new top10::graphX::TextNode;
	text_node->setFont(getFont());
	text_node->setText(entries.at(i).text);
	text_node->setPos(start_x, y);
	text_node->setPointSize(point_sz);
	color->addChild(text_node);

	if (i == entry_n)
	{
	  text_node = new top10::graphX::TextNode;
	  text_node->setFont(getFont());
	  text_node->setText(cursor_text);
	  text_node->setPos(cursor_x, y);
	  text_node->setPointSize(point_sz);
	  color->addChild(text_node);
	}

	y -= spacing;
      }
    }

    void Menu::on_panelShowed()
    {
    }

    void Menu::on_panelHidden()
    {
    }

    void Menu::updateFrontData()
    {
    }

    void Menu::renderGL()
    {
    }

    void Menu::renderAL()
    {
    }

    void Menu::handleFrontEvent(SDL_Event)
    {
    }

    void Menu::on_panelFrontOn()
    {
    }

    void Menu::on_panelFrontOff()
    {
    }

    bool Menu::isFrontType() const
    {
      return false;
    }


    top10::graphX::Node* Menu::getNode()
    {
      return node.getPtr();
    }

    /*
     * FrontMenu
     */
    FrontMenu::FrontMenu(UIPanel* panel, Menu* parent)
      : Menu(panel),
      m_parent(parent),
      m_text("Exit")
    {
      unsigned int id = Menu::addMenu(0);
      setMenuText(id, m_text);
    }



    void FrontMenu::setExitText(const std::string& s)
    {
      assert( getNumEntries() > 0 );
      m_text = s;
      setMenuText(getNumEntries() -1, m_text);
    }



    unsigned int FrontMenu::addMenu(Menu* m)
    {
      assert(getNumEntries() > 0);

      unsigned int id = Menu::entries.size() -1;
      unsigned int id2 = Menu::addMenu(m);

      std::swap(Menu::entries.at(id2), Menu::entries.at(id));

      return id;
    }



    bool FrontMenu::isFrontType() const
    {
      return true;
    }



    void FrontMenu::selected()
    {
      if (getEntryN() +1 != getNumEntries())
	Menu::selected();
      else
      {
	killSession();
	Menu::panel->setFrontMenu(m_parent);
	Menu::panel->setMenu(m_parent);
      }
    }



    void FrontMenu::killSession()
    {
    }



  }
}
