/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_UI_KART_HH
#define TOP10_UI_KART_HH

#include "extras/GLee.h"
#include <iostream>

#include "graphX/TransformNode.hh"
#include "graphX/CameraNode.hh"
#include "physX/Kart.hh"
#include "sound/EngineSoundInstance.hh"
#include "sound/RollingSoundInstance.hh"
#include "sound/SkiddingSoundInstance.hh"

namespace top10 {
  namespace ui_interactive {

    using top10::math::Vector;
    
    //! Class used to draw a kart
    class KartNode: public top10::graphX::GroupNode {
    public:
      //! Update the position of the kart and its elements
      void update(double steer_angle,
		  double steer_fl, double steer_fr,
		  double h_fl, double h_fr,
		  double h_rl, double h_rr,
		  double ang_fl, double ang_fr,
		  double ang_rl, double anf_rr) const;

      KartNode* clone() const;
      
    private:
      KartNode(std::string model_filename);
      
      static void parsePart(std::istream&, const char* keyword,
                            std::string* alias,
			    std::string* object, top10::math::Matrix4* M);
      
      //! Indexes in the transforms and models arrays
      static const int BODY = 0;
      static const int WHEEL_FL = 1;
      static const int WHEEL_FR = 2;
      static const int WHEEL_RL = 3;
      static const int WHEEL_RR = 4;
      static const int STEER = 5;
      static const int LAST = 5;

      //! The static transformation to apply to each part
      top10::util::Ref<top10::graphX::TransformNode> static_transforms[LAST+1];
      
      //! The transformations to apply according to the state of the kart
      top10::util::Ref<top10::graphX::TransformNode> dynamic_transforms[LAST+1];

      /*! Transform that brings the (statically transformed) components to some normal position, where the
          dynamic transformartions are performed. After the that, the reverse transform is done to bring the
          component back to its location on the go-kart */
      top10::math::Matrix4 component_to_normal[LAST+1];
      //! The reverse...
      top10::math::Matrix4 normal_to_component[LAST+1];
            
      friend class KartManager;
    };
    typedef top10::util::Ref< KartNode > KartNodeRef;
    typedef std::vector< KartNodeRef > KartNodeRefs;

    //! Class responsible for loading karts
    class KartManager {
    public:
      static KartManager* getInstance();
      
      KartNode* getKart(std::string model_name);
      
    private:
      static KartManager* instance;
      KartManager();
      std::map< std::string, top10::util::Ref<KartNode> > cache;
    };

    //! Extends a physX::Kart with various UI-related information
    class Kart: public top10::physX::Kart
    {
    public:
      /**
	 Creates a new kart.
	 \param starting_pos Where to place the kart
	 \param orientation The initial orientation of the kart
      */
      Kart(const std::string& name, top10::track::Track* track, top10::sound::SourceAllocator*);

      /**
	 Update the kart.
	 This member does not update the physics related data. Instead, it
	 accumulates data useful to generate sounds
      */
      void updateUI(double dt);
       
      inline double getLeftSteer() const  { return wheel_fl.getAngle(); }
      inline double getRightSteer() const { return wheel_fr.getAngle(); }
      inline double getSteer() const      { return steer_angle; }
      inline double getHeightFL() const   { return wheel_fl.getHeight(); }
      inline double getHeightFR() const   { return wheel_fr.getHeight(); }
      inline double getHeightRL() const   { return wheel_rl.getHeight(); }
      inline double getHeightRR() const   { return wheel_rr.getHeight(); }

      inline double getAngPosFL() const { return wheel_fl.getAngularPos(); }
      inline double getAngPosFR() const { return wheel_fr.getAngularPos(); }
      inline double getAngPosRL() const { return wheel_rl.getAngularPos(); }
      inline double getAngPosRR() const { return wheel_rr.getAngularPos(); }

      top10::graphX::CameraNode* getCamera(int) const;
      int getNCameras() const;

      void toggleSound(bool);

    private:
      top10::util::Ref<top10::sound::EngineSoundInstance> engine_sound;

      top10::util::Ref<top10::sound::RollingSoundInstance> rolling_sound_fl;
      top10::util::Ref<top10::sound::RollingSoundInstance> rolling_sound_fr;
      top10::util::Ref<top10::sound::RollingSoundInstance> rolling_sound_rl;
      top10::util::Ref<top10::sound::RollingSoundInstance> rolling_sound_rr;

      top10::util::Ref<top10::sound::SkiddingSoundInstance> skidding_sound_fl;
      top10::util::Ref<top10::sound::SkiddingSoundInstance> skidding_sound_fr;
      top10::util::Ref<top10::sound::SkiddingSoundInstance> skidding_sound_rl;
      top10::util::Ref<top10::sound::SkiddingSoundInstance> skidding_sound_rr;

      top10::util::Ref<top10::sound::SkiddingSoundInstance> side_skidding_sound_fl;
      top10::util::Ref<top10::sound::SkiddingSoundInstance> side_skidding_sound_fr;
      top10::util::Ref<top10::sound::SkiddingSoundInstance> side_skidding_sound_rl;
      top10::util::Ref<top10::sound::SkiddingSoundInstance> side_skidding_sound_rr;

    };

    typedef top10::util::Ref< Kart > KartRef;
    typedef std::vector< KartRef > KartRefs;
  };
};

#endif
