/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include <fstream>
#include <cmath>

#include "Kart.hh"
#include "helpers/MeshFactory.hh"
#include "util/error.hh"
#include "util/PathFinder.hh"
#include "util/Parse.hh"
#include "util/Log.hh"
#include "track/Track.hh"
#include "math/Matrix.hh"
#include "graphX/AutoCameraNode.hh"
#include "graphX/AimMarkNode.hh"

using namespace top10::ui_interactive;
using namespace top10::math;
using namespace std;

using top10::util::Error;
using top10::util::Log;

/*
 * KartManager
 */

KartManager* KartManager::instance = 0;

KartManager::KartManager() {}

KartManager* KartManager::getInstance()
{
  if (instance) return instance;
  else return (instance = new KartManager);
}

KartNode* KartManager::getKart(std::string model_filename)
{
  // See if we already loaded this kind of kart
  std::map<std::string, top10::util::Ref<KartNode> >::iterator cache_f = cache.find(model_filename);
  if (cache_f != cache.end()) return cache_f->second.getPtr();
  
  // If not, load it now
  KartNode* ret = new KartNode(model_filename);
  cache[model_filename] = ret;
  ret->setName("KART["+model_filename+"]");
  
  return ret;
}

/*
 * KartNode
 */

#define EXPECT(str) if (token != str) throw Error("Expected "+std::string(str)+", got "+token)

KartNode::KartNode(std::string model_filename)
{
  for (int i=0; i<=LAST; ++i) {
    static_transforms[i] = new top10::graphX::TransformNode;
    dynamic_transforms[i] = new top10::graphX::TransformNode;
    dynamic_transforms[i]->addChild(static_transforms[i].getPtr());
    addChild(dynamic_transforms[i].getPtr());
  }
  
  // All 3d models that are part of the kart
  std::map<std::string, top10::util::Ref<top10::graphX::Node> > model_map;
  
  top10::util::PathFinder finder = top10::util::PathFinder::defaultPathFinder();
  std::string path = finder.find(model_filename);
  if (path.empty()) throw Error("Could not find ") + model_filename;
  std::ifstream in(path.c_str());
  if (!in) throw Error("Could not open "+path);
  finder.addPath(top10::util::PathFinder::getPath(path));
  
  /* Expected:
     BEGIN MODELS
     <model filename> <alias name>
     ...
     END
  */

  std::string token;
  in>>token;
  EXPECT("BEGIN");
  in>>token;
  EXPECT("MODELS");
  for (;;) {
    token = top10::util::parseString(in);
    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");

    path = finder.find(token);
    if (path.empty()) throw Error("Could not find "+token);
	top10::graphX::Node* model = top10::helpers::MeshFactory::getSingle()->load(path);
    if (!model) throw Error("Could not load "+path);

    token = top10::util::parseString(in);
    model_map[token] = model;
  }

  /* Expected:
     BEGIN BODY
     <alias> <object name>
     TRANSFORM <4x4 matrix (16 floats)>
     END
  */
  std::map<std::string, top10::util::Ref<top10::graphX::Node> >::iterator f;

  const char* keywords[] = {"BODY", "WHEEL_FL", "WHEEL_FR", "WHEEL_RL", "WHEEL_RR", "STEERING_WHEEL"};
  std::string aliases[LAST+1];
  std::string objects[LAST+1];
  top10::math::Matrix4 transforms[LAST+1];
  
  for (int i=0; i<=LAST; ++i) {
    in>>token;
    EXPECT("BEGIN");
    parsePart(in, keywords[i], &(aliases[i]), &(objects[i]), &(transforms[i]));
    static_transforms[i]->setToWorld(transforms[i]);
    if (i == KartNode::STEER) {
      Vector steer_center, steer_axis;
      
      in>>token;
      EXPECT("CENTER");
      in>>steer_center.x;
      in>>steer_center.y;
      in>>steer_center.z;

      in>>token;
      EXPECT("AXIS");
      in>>steer_axis.x;
      in>>steer_axis.y;
      in>>steer_axis.z;
      
      double axis_size = steer_axis.size();
      if (axis_size < SMALL_VECTOR) throw Error("Axis of steering wheel is too small");
      steer_axis /= axis_size;
    
      // Compute the transformations that will bring the wheel to lay horizontal around (0,0,0)
      Vector z(0,0,1);
      Vector x = steer_axis^z;
      Matrix4 M(x, steer_axis, z);
      Translation4 T(steer_center);
      M = T*M;
      normal_to_component[KartNode::STEER] = M;
      component_to_normal[KartNode::STEER] = top10::math::inverse(M);
    }
    else if (i == KartNode::BODY) {
      // Nothing, parsePart did everything already.
    }
    else {
      Vector attach;
      
      in>>token;
      EXPECT("CENTER");
      in>>attach.x>>attach.y>>attach.z;
    
      normal_to_component[i] = Translation4(attach);
      component_to_normal[i] = Translation4(-attach);

      top10::graphX::TransformNode* dbg = new top10::graphX::TransformNode;
      dbg->setToWorld( Translation4(attach) );
      dbg->addChild( new top10::graphX::AimMarkNode );
      addChild(dbg);
    }

    in>>token;
    EXPECT("END");

    f = model_map.find(aliases[i]);
    if (f != model_map.end())
    {
      top10::graphX::Node* object_node = f->second->find(objects[i]);
      if (object_node != 0)
        static_transforms[i]->addChild(object_node);
      else
        Log::getSingle()->send(Log::Warning, "KartNode", "Could not find object "+objects[i]+" in model "+aliases[i]);
    }
    else Log::getSingle()->send(Log::Warning, "KartNode", "Could not find model "+aliases[i]);
  }
}

KartNode* KartNode::clone() const
{
  KartNode* ret = new KartNode(*this);
  for (int i=0; i<=LAST; ++i) {
    ret->removeChild(dynamic_transforms[i].getPtr());
    ret->dynamic_transforms[i] = new top10::graphX::TransformNode;
    ret->dynamic_transforms[i]->addChild(ret->static_transforms[i].getPtr());
    ret->addChild(ret->dynamic_transforms[i].getPtr());
  }
  return ret;
}

void KartNode::parsePart(std::istream& in, const char* keyword, std::string* alias, std::string* object, top10::math::Matrix4* M)
{
  using top10::util::parseString;

  std::string token;

  in>>token;
  EXPECT(keyword);
  in>>token;
  EXPECT("MODEL");
  *alias = parseString(in);
  *object = parseString(in);
  in>>token;
  EXPECT("TRANSFORM");
  in>>(*M);
}

void KartNode::update(double steer_angle,
                      double steer_fl, double steer_fr,
                      double h_fl, double h_fr,
                      double h_rl, double h_rr,
		      double rot_fl, double rot_fr,
		      double rot_rl, double rot_rr) const
{
  top10::math::Matrix4 R = top10::math::Identity4();
  double c = cos(steer_angle);
  double s = sin(steer_angle);
  R(0,0) = c;
  R(2,0) = -s;
  R(0,2) = s;
  R(2,2) = c;
  dynamic_transforms[STEER]->setToWorld(normal_to_component[STEER]*R*component_to_normal[STEER]);
  
  top10::math::Matrix4 R0 = top10::math::Identity4();
  c = cos(rot_fl);
  s = sin(rot_fl);
  R0(0,0) = c;
  R0(1,0) = -s;
  R0(0,1) = s;
  R0(1,1) = c;

  c = cos(steer_fl);
  s = sin(steer_fl);
  R(0,0) = c;
  R(2,0) = -s;
  R(0,2) = s;
  R(2,2) = c;
  R(1,3) = h_fl;
  dynamic_transforms[WHEEL_FL]->setToWorld(normal_to_component[WHEEL_FL]*R*R0*component_to_normal[WHEEL_FL]);

  c = cos(rot_fr);
  s = sin(rot_fr);
  R0(0,0) = c;
  R0(1,0) = -s;
  R0(0,1) = s;
  R0(1,1) = c;

  c = cos(steer_fr);
  s = sin(steer_fr);
  R(0,0) = c;
  R(2,0) = -s;
  R(0,2) = s;
  R(2,2) = c;
  R(1,3) = h_fr;
  dynamic_transforms[WHEEL_FR]->setToWorld(normal_to_component[WHEEL_FR]*R*R0*component_to_normal[WHEEL_FR]);

  c = cos(rot_rl);
  s = sin(rot_rl);
  R0(0,0) = c;
  R0(1,0) = -s;
  R0(0,1) = s;
  R0(1,1) = c;
  
  R(0,0) =  R(2,2) = 1;
  R(2,0) = R(0,2) = 0;
  R(1,3) = h_rl;
  dynamic_transforms[WHEEL_RL]->setToWorld(normal_to_component[WHEEL_RL]*R*R0*component_to_normal[WHEEL_RL]);

  c = cos(rot_rr);
  s = sin(rot_rr);
  R0(0,0) = c;
  R0(1,0) = -s;
  R0(0,1) = s;
  R0(1,1) = c;

  R(1,3) = h_rr;
  dynamic_transforms[WHEEL_RR]->setToWorld(normal_to_component[WHEEL_RR]*R*R0*component_to_normal[WHEEL_RR]);
}

/*
 * Kart
 */

Kart::Kart(const std::string& name,
	   top10::track::Track* track,
	   top10::sound::SourceAllocator* audio_alloc):
  top10::physX::Kart(track, name)
{
  if (audio_alloc)
  {
    engine_sound = new top10::sound::EngineSoundInstance(top10::sound::buildKartSound(), audio_alloc);

    top10::sound::RollingSound* rs = top10::sound::makeRollingSound();

    /* BUG: Commented out to avoid triggering bug #22063 */ 
/*    rolling_sound_fl = new top10::sound::RollingSoundInstance(rs, audio_alloc);
    rolling_sound_fl->setEnabled(false);

    rolling_sound_fr = new top10::sound::RollingSoundInstance(rs, audio_alloc);
    rolling_sound_fr->setEnabled(false);

    rolling_sound_rl = new top10::sound::RollingSoundInstance(rs, audio_alloc);
    rolling_sound_rl->setEnabled(false);

    rolling_sound_rr = new top10::sound::RollingSoundInstance(rs, audio_alloc);
    rolling_sound_rr->setEnabled(false);
*/
    top10::sound::SkiddingSound* skid = top10::sound::buildSkiddingSound();

    skidding_sound_fl = new top10::sound::SkiddingSoundInstance(skid, audio_alloc);
    skidding_sound_fl->setEnabled(false);

    skidding_sound_fr = new top10::sound::SkiddingSoundInstance(skid, audio_alloc);
    skidding_sound_fr->setEnabled(false);

    skidding_sound_rl = new top10::sound::SkiddingSoundInstance(skid, audio_alloc);
    skidding_sound_rl->setEnabled(false);

    skidding_sound_rr = new top10::sound::SkiddingSoundInstance(skid, audio_alloc);
    skidding_sound_rr->setEnabled(false);

    top10::sound::SkiddingSound* side_skid = top10::sound::buildSideSkiddingSound();

    side_skidding_sound_fl = new top10::sound::SkiddingSoundInstance(side_skid, audio_alloc);
    side_skidding_sound_fl->setEnabled(false);

    side_skidding_sound_fr = new top10::sound::SkiddingSoundInstance(side_skid, audio_alloc);
    side_skidding_sound_fr->setEnabled(false);

    side_skidding_sound_rl = new top10::sound::SkiddingSoundInstance(side_skid, audio_alloc);
    side_skidding_sound_rl->setEnabled(false);

    side_skidding_sound_rr = new top10::sound::SkiddingSoundInstance(side_skid, audio_alloc);
    side_skidding_sound_rr->setEnabled(false);

  }
}

void Kart::updateUI(double dt)
{
  if (dt == 0.0) return;
  
  if (engine_sound.isValid())
  {
    engine_sound->update
      (engine.getAngularVelocity(),
      engine.getAccel() > 0.5,
      getPos(),
      getSpeed());
  }

  if (rolling_sound_fl.isValid())
  {
    rolling_sound_fl->setEnabled(wheel_fl.getSurface().bumps > 0.01);
    rolling_sound_fl->update(dt, wheel_fl.getPos(), getSpeed());
  }

  if (rolling_sound_fr.isValid())
  {
    rolling_sound_fr->setEnabled(wheel_fr.getSurface().bumps > 0.01);
    rolling_sound_fr->update(dt, wheel_fr.getPos(), getSpeed());
  }

  if (rolling_sound_rl.isValid())
  {
    rolling_sound_rl->setEnabled(wheel_rl.getSurface().bumps > 0.01);
    rolling_sound_rl->update(dt, wheel_rl.getPos(), getSpeed());
  }

  if (rolling_sound_rr.isValid())
  {
    rolling_sound_rr->setEnabled(wheel_rr.getSurface().bumps > 0.01);
    rolling_sound_rr->update(dt, wheel_rr.getPos(), getSpeed());
  }

  if (skidding_sound_fl.isValid())
  {
    skidding_sound_fl->setEnabled(wheel_fl.getSurface().grip > 0.9);
    skidding_sound_fl->update(wheel_fl.getLoad(), wheel_fl.getLongFriction(), wheel_fl.getPos(), getSpeed());
  }

  if (skidding_sound_fr.isValid())
  {
    skidding_sound_fr->setEnabled(wheel_fr.getSurface().grip > 0.9);
    skidding_sound_fr->update(wheel_fr.getLoad(), wheel_fr.getLongFriction(), wheel_fr.getPos(), getSpeed());
  }

  if (skidding_sound_rl.isValid())
  {
    skidding_sound_rl->setEnabled(wheel_rl.getSurface().grip > 0.9);
    skidding_sound_rl->update(wheel_rl.getLoad(), wheel_rl.getLongFriction(), wheel_rl.getPos(), getSpeed());
  }

  if (skidding_sound_fr.isValid())
  {
    skidding_sound_rr->setEnabled(wheel_rr.getSurface().grip > 0.9);
    skidding_sound_rr->update(wheel_rr.getLoad(), wheel_rr.getLongFriction(), wheel_rr.getPos(), getSpeed());
  }

  if (side_skidding_sound_fl.isValid())
  {
    side_skidding_sound_fl->setEnabled(wheel_fl.getSurface().grip > 0.9);
    side_skidding_sound_fl->update(wheel_fl.getLoad(), wheel_fl.getSideFriction(), wheel_fl.getPos(), getSpeed());
  }

  if (side_skidding_sound_fr.isValid())
  {
    side_skidding_sound_fr->setEnabled(wheel_fr.getSurface().grip > 0.9);
    side_skidding_sound_fr->update(wheel_fr.getLoad(), wheel_fr.getSideFriction(), wheel_fr.getPos(), getSpeed());
  }

  if (side_skidding_sound_rl.isValid())
  {
    side_skidding_sound_rl->setEnabled(wheel_rl.getSurface().grip > 0.9);
    side_skidding_sound_rl->update(wheel_rl.getLoad(), wheel_rl.getSideFriction(), wheel_rl.getPos(), getSpeed());
  }

  if (side_skidding_sound_fr.isValid())
  {
    side_skidding_sound_rr->setEnabled(wheel_rr.getSurface().grip > 0.9);
    side_skidding_sound_rr->update(wheel_rr.getLoad(), wheel_rr.getSideFriction(), wheel_rr.getPos(), getSpeed());
  }

}

int Kart::getNCameras() const
{
  return 6;
}

top10::graphX::CameraNode* Kart::getCamera(int i) const
{
  const top10::math::Matrix3& orient(body.getOrientation());
  Vector right = orient * Vector(0,0,1);
  Vector forward = orient * Vector(1,0,0);
  Vector up = orient * Vector(0,1,0);
  Vector center = body.getMassCenter();
  double fov = 30;

  Vector pos, dir, dup;
  switch(i) {
  case 0: //In
    pos = center - forward*0.5 + Vector(0, 0.45,0);
    dir = forward - up*0.2;
    dup = Vector(0,1,0);
    fov = 45;
    break;
  case 1: //Front
    pos = center + forward*5 + Vector(0,1,0);
    dir = -forward - Vector(0,0.2,0);;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case 2: //Back
    pos = center - forward*5 + Vector(0,1.5,0);
    dir = forward - Vector(0,0.2,0);
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case 3: //left
    pos = center - right*15 + Vector(0,1,0);
    dir = right;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case 4: //right
    pos = center + right*15 + Vector(0,1,0);
    dir = -right;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case 5: //top
    pos = center - Vector(0,100,0);
    dir = Vector(0,-1,0);
    fov = 50;
    dup = forward;
    break;
  default: return 0;
  };

  top10::graphX::AutoCameraNode* camera = new top10::graphX::AutoCameraNode;
  camera->accessView().setCenter(pos);
  camera->accessView().setDirection(dir, dup);
  camera->accessView().setFOV(fov);
  camera->accessView().setNear(0.7);
  camera->accessView().setFar(500);
  camera->accessView().setRatio(1.4); //FIXME
  camera->accessView().update();
  
  return camera;
}


void Kart::toggleSound(bool b)
{
  if (b)
  {
    if (engine_sound.isValid())
      engine_sound->unmute();
    
    if (rolling_sound_fl.isValid())
      rolling_sound_fl->unmute();

    if (rolling_sound_fr.isValid())
      rolling_sound_fr->unmute();

    if (rolling_sound_rl.isValid())
      rolling_sound_rl->unmute();

    if (rolling_sound_rr.isValid())
      rolling_sound_rr->unmute();

    if (skidding_sound_fl.isValid())
      skidding_sound_fl->unmute();

    if (skidding_sound_fr.isValid())
      skidding_sound_fr->unmute();

    if (skidding_sound_rl.isValid())
      skidding_sound_rl->unmute();

    if (skidding_sound_rr.isValid())
      skidding_sound_rr->unmute();

    if (side_skidding_sound_fl.isValid())
      side_skidding_sound_fl->unmute();

    if (side_skidding_sound_fr.isValid())
      side_skidding_sound_fr->unmute();

    if (side_skidding_sound_rl.isValid())
      side_skidding_sound_rl->unmute();

    if (side_skidding_sound_rr.isValid())
      side_skidding_sound_rr->unmute();
  }
  else
  {
    if (engine_sound.isValid())
      engine_sound->mute();
    
    if (rolling_sound_fl.isValid())
      rolling_sound_fl->mute();

    if (rolling_sound_fr.isValid())
      rolling_sound_fr->mute();

    if (rolling_sound_rl.isValid())
      rolling_sound_rl->mute();

    if (rolling_sound_rr.isValid())
      rolling_sound_rr->mute();

    if (skidding_sound_fl.isValid())
      skidding_sound_fl->mute();

    if (skidding_sound_fr.isValid())
      skidding_sound_fr->mute();

    if (skidding_sound_rl.isValid())
      skidding_sound_rl->mute();

    if (skidding_sound_rr.isValid())
      skidding_sound_rr->mute();

    if (side_skidding_sound_fl.isValid())
      side_skidding_sound_fl->mute();

    if (side_skidding_sound_fr.isValid())
      side_skidding_sound_fr->mute();

    if (side_skidding_sound_rl.isValid())
      side_skidding_sound_rl->mute();

    if (side_skidding_sound_rr.isValid())
      side_skidding_sound_rr->mute();
  }
}
