/*
  Top 10, a racing simulator
  Copyright (C) 2003-2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "GameManager.hh"
#include "FogValues.hh"
#include "util/GlobalOptions.hh"
#include "util/Log.hh"
#include "util/error.hh"
#include "graphX/LightNode.hh"
#include "graphX/AlphaNode.hh"
#include "graphX/CullNode.hh"
#include "graphX/Wireframe.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/ShadowVolumeNode.hh"
#include "trackedit/TrackFactory.hh"
#include "sound/Listener.hh"
#include "util/strconv.hh"
#include "util/UserFile.hh"

#include <fstream>

#include <plib/fnt.h>

using top10::util::Log;

namespace top10 {
namespace ui_interactive {

GameManager::GameManager(top10::graphX::Renderer* render, 
			 top10::sound::SourceAllocator* audio_alloc,
			 top10::racing::LapRecordDb* laps): 
  render(render),
    audio_alloc(audio_alloc),
    laps(laps),
    is_rainy(false),
    started(false),
    use_freecam(false),
    render_hud(false)
{
  free_camera = new top10::graphX::FreeCameraNode;

  std::string font_name = getOptS("Render.Font");    
  std::string font_path = top10::util::PathFinder::defaultPathFinder().find("fonts/"+font_name+".txf");
  if (font_path.empty())
    throw std::string("Could not load font fonts/"+font_name+".txf");
  font = new fntTexFont(font_path.c_str());

  sound_update_period = (unsigned int)(getOptD("Sound.UpdatePeriod")*1000.0);
}

void GameManager::createTrainingSession(int n_karts,
					std::string track_name, std::string kart_model_name,
                                        std::string kart_physx_name, bool wet_weather)
{
  using top10::util::Log;

  terminateGame();
  
  try {
    track = top10::tracked::TrackFactory::getSingle()->loadTrack(track_name);
    this->track_name = track_name;
    this->kart_name = kart_model_name;

    // Load the physics parameters of the vehicle
    if (!kart_physx_name.empty())
    {
      top10::util::PathFinder finder = top10::util::PathFinder::defaultPathFinder();
      std::string path = finder.find(kart_physx_name);
      if (path.empty())
	throw top10::util::Error("Could not find ")+kart_physx_name;

      std::ifstream kart_in(path.c_str());
      if (!kart_in)
	throw top10::util::Error("Could not open ")+path;

      top10::physX::Kart::initDefaultParameters(kart_in);
    }

    // Create all karts
    createKarts(n_karts);
    std::vector<top10::physX::KartRef> physx_karts;
    for (int i=0; i<karts.size(); ++i)
    {
      top10::physX::KartRef kref( karts.at(i).getPtr() );
      physx_karts.push_back(kref);
    }

    // Create the simulation
    simulation = new top10::racing::Simulation(track.getPtr(), physx_karts, laps);
    simulation->setWet(wet_weather);
    is_rainy = wet_weather;
    
    // Setup the rendering
    track_scene = new TrackScene(render);
    track_scene->setTrack(track.getPtr());

    // Skybox
    if (!is_rainy)
    {
      track_scene->setSkyBox(
	top10::tracked::TrackFactory::getSingle()->getSkyboxName(track_name));
    }

    // Shadows by the track.
    const bool track_shadow_enabled = getEnabledOpt("Render.Track.Shadow.Quality");
    if (track_shadow_enabled)
    {
      const std::string method_str = getOptS("Render.Track.Shadow.Quality");

      top10::graphX::ShadowVolumeNode::Method method =
	top10::graphX::ShadowVolumeNode::toMethod(method_str);

      track_scene->setTrackShadows(method);
    }
    
    // The karts
    Kart* kart;
    int i=0;
    while ( (kart = dynamic_cast<Kart*>(simulation->getKart(i++))) ) {
      assert(kart);

      unsigned int id = track_scene->newKart();
      kart_ids.push_back(id);

      track_scene->setKart(id, kart_model_name);
      track_scene->setKartTransform(id, getKartTransform(kart));
    }

    // Kart shadows
    const bool kart_shadow_enabled = getEnabledOpt("Render.Kart.Shadow.Quality");
    if (kart_shadow_enabled)
    {
      const std::string method_str = getOptS("Render.Kart.Shadow.Quality");
      top10::graphX::ShadowVolumeNode::Method method =
	top10::graphX::ShadowVolumeNode::toMethod(method_str);

      track_scene->setKartShadows(method);
    }
  
    // The ghost
    ghost_id = track_scene->newKart();
    track_scene->setKart(ghost_id, kart_model_name);
    track_scene->hideKart(ghost_id);
    track_scene->toggleGhost(ghost_id, true);
    
    // The hud needs to be slightly transparent, otherwise transparent bits in the font texture won't show at all
    top10::graphX::AlphaNode* hud_alpha = new top10::graphX::AlphaNode;
    hud_alpha->setAlpha(getOptUC("Render.HUD.Alpha"));
    hud = hud_alpha;
    
    // Use screen pixel coordinates
    top10::graphX::TransformNode* hud_transform = new top10::graphX::TransformNode;
    top10::math::Matrix4 translate = top10::math::Translation4(top10::math::Vector(-1.0, -1.0, 0.0));
    top10::math::Matrix4 scale = top10::math::Scaling3(1.0/render->getScreenWidth(), 1.0/render->getScreenHeight(), 1.0);
    translate *= scale;
    hud_transform->setToWorld(translate);
    hud->addChild(hud_transform);
    
    // The timers, showing the current and best times
    current_time = new top10::graphX::TextNode;
    current_time->setFont(font);
    current_time->setPos(getOptD("Render.HUD.CurrentTime.Pos.X"), getOptD("Render.HUD.CurrentTime.Pos.Y"));
    current_time->setPointSize(getOptD("Render.HUD.CurrentTime.Font.Size"));
    hud_transform->addChild(current_time.getPtr());
    
    best_time = new top10::graphX::TextNode;
    best_time->setFont(font);
    best_time->setPos(getOptD("Render.HUD.BestTime.Pos.X"), getOptD("Render.HUD.BestTime.Pos.Y"));
    best_time->setPointSize(getOptD("Render.HUD.BestTime.Font.Size"));
    hud_transform->addChild(best_time.getPtr());
    
    last_time = new top10::graphX::TextNode;
    last_time->setFont(font);
    last_time->setPos(getOptD("Render.HUD.LastTime.Pos.X"), getOptD("Render.HUD.LastTime.Pos.Y"));
    last_time->setPointSize(getOptD("Render.HUD.LastTime.Font.Size"));
    hud_transform->addChild(last_time.getPtr());

    stat_fps = new top10::graphX::TextNode;
    stat_fps->setFont(font);
    stat_fps->setPos(getOptD("Render.HUD.Fps.Pos.X"), getOptD("Render.HUD.Fps.Pos.Y"));
    stat_fps->setPointSize(getOptD("Render.HUD.Fps.Font.Size"));
    hud_transform->addChild(stat_fps.getPtr());
  }
  catch(std::string e) {
    top10::util::Log::getSingle()->send(top10::util::Log::Error, "GameManager", e);
    terminateGame();
  }
  catch(...) {
    terminateGame();
    Log::getSingle()->send(Log::Error, "GameManager", "Failed to create a new session for an unknown reason");
  }
}

void GameManager::createKarts(int n_karts)
{
  for (int i=0; i<n_karts; ++i)
  {
    karts.push_back(KartRef(new Kart(kart_name, track.getPtr(), audio_alloc.getPtr())));
  }
}

void GameManager::resetControlledKart()
{
  top10::physX::Kart::State s = karts[camera_kart]->getKartState();
  Kart* new_kart = new Kart(kart_name, track.getPtr(), audio_alloc.getPtr());
  new_kart->setPosOnTrack(s.translation, s.orient * top10::math::Vector(1.0, 0.0, 0.0));
  karts[camera_kart] = new_kart;
  simulation->setKart(kart_ids.at(camera_kart), karts.at(camera_kart).getPtr());
}

void GameManager::updateGame(unsigned int now)
{
  if (!started) return;
  
  simulation->update(now - time_start);

  Kart* kart;
  int i=0;
  while ( (kart = dynamic_cast<Kart*>(simulation->getKart(i))) )
  {
    assert(kart);
    unsigned int kart_id = kart_ids.at(i);

    track_scene->setKartTransform(kart_id, getKartTransform(kart));
    track_scene->setKartState(kart_id,
			      kart->getSteer(), kart->getLeftSteer(), kart->getRightSteer(),
			      kart->getHeightFL(), kart->getHeightFR(), kart->getHeightRL(), kart->getHeightRR(),
			      kart->getAngPosFL(), kart->getAngPosFR(), kart->getAngPosRL(), kart->getAngPosRR());
    kart->updateUI((now - last_update)/1000.0);
    ++i;
  }
  
  const top10::racing::KartTimer* timer = simulation->getTimer(camera_kart);
  if (timer) {
    if (ghost_hidden && timer->getBestTime() != 0) {
      ghost_hidden = false;
      track_scene->showKart(ghost_id);
    }
    
    current_time->setTime(timer->getCurrentTime());
    best_time->setTime(timer->getBestTime());
    last_time->setTime(timer->getLastTime());

    // Update the ghost
    if (simulation->getBestLap().size() != 0) {
      top10::racing::KartState s = simulation->getBestLap().getKartState(timer->getCurrentTime());
      track_scene->setKartTransform(ghost_id, s.getTransform());
      track_scene->setKartState(ghost_id, s.steer, s.steer, s.steer, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0);
    }
  }

  // Update FPS statistics every second.
  ++fps_count;
  if (now - last_fps_mark >= 1000) {
    double fps_val = fps_count * 1000.0 / (now - last_fps_mark);
    stat_fps->setNum(fps_val, 3);
    last_fps_mark = now;
    fps_count = 0;
  }

  last_update = now;
}

top10::math::Matrix4 GameManager::getKartTransform(const Kart* kart) const
{
  top10::racing::KartState s = kart->getKartState();
  top10::math::Matrix4 T = top10::math::Translation4(-kart->getMassCenterL());
  top10::math::Matrix4 T_inv = top10::math::Translation4(kart->getMassCenterL());
  top10::math::Matrix4 O = s.orient;
  top10::math::Matrix4 P = top10::math::Translation4(s.translation);
  return P*T_inv*O*T;
}

void GameManager::startGame(unsigned int now)
{
  if (simulation.isValid())
  {
    started = true;
    last_sound_update = last_update = time_start = last_fps_mark = now;
    fps_count = 0;

    FogValues fog;
    if (is_rainy)
    {
      fog = FogValues::createRainy();
    }
    else
    {
      fog = FogValues::createClear();
    }

    if (getEnabledOpt("Render.HUD"))
      render_hud = true;

    render->setClearColor(fog.getRed(), fog.getGreen(), fog.getBlue());
    render->setFog(fog.getDensity(), fog.getNear(), fog.getFar());
  }
}

void GameManager::terminateGame()
{
  started = false;
  ghost_hidden = true;
  camera_kart = camera = 0;
  kart_ids.clear();
  karts.clear();
  track_scene.discard();
}


void GameManager::sendEvent(Action a, Sint16 value)
{
  if (simulation.isValid())
    simulation->sendEvent(camera_kart, a, value);    
}

void GameManager::renderGL()
{
  if (!simulation.isValid()) return;
  
  render->clearAllGL();
  render->setCamera(getCamera());

  track_scene->renderGL();

  if (render_hud)
  {
    top10::graphX::RenderList hud_list;
    render->buildList(hud.getPtr(), &hud_list);
    render->clearDepthGL();
    render->renderHudGL(hud_list);
  }
}

void GameManager::nextCamera()
{
  if (!simulation.isValid()) return;
  
  ++camera;
  Kart* kart = dynamic_cast<Kart*>(simulation->getKart(camera_kart));
  if (!kart) return;
  
  if (camera == kart->getNCameras()) camera = 0;
}

void GameManager::prevCamera()
{
  if (!simulation.isValid()) return;
  
  --camera;
  if (camera < 0) camera = 0;
}

void GameManager::nextKart()
{
  if (!simulation.isValid()) return;
  
  ++camera_kart;
  if (camera_kart == karts.size()) camera_kart = 0;   
}

void GameManager::prevKart()
{
  if (!simulation.isValid()) return;
  
  --camera_kart;
  if (camera_kart < 0) camera_kart = 0;    
}

top10::graphX::CameraNode* GameManager::getCamera()
{
  if (!simulation.isValid()) return 0;
  
  if (use_freecam) return free_camera.getPtr();
  
  Kart* kart = dynamic_cast<Kart*>(simulation->getKart(camera_kart));
  assert(kart);

  return kart->getCamera(camera);
}

void GameManager::renderAL()
{
  if (!simulation.isValid()) return;

  if (last_update - last_sound_update < sound_update_period)
    return;
  last_sound_update = last_update;

  top10::sound::Listener l;
  if (use_freecam)
  {
    l.setView(free_camera->getView());
  }
  else
  {
    Kart* kart = dynamic_cast<Kart*>(simulation->getKart(camera_kart));
    assert(kart);

    top10::util::Ref<top10::graphX::CameraNode> cam(kart->getCamera(camera));
    l.setView(cam->getView());
//    l.setSpeed(simulation->getKart(camera_kart)->getBody().getSpeedAt(cam->getView()->getCenter()));
    l.setSpeed(simulation->getKart(camera_kart)->getSpeed());
  }
  l.setGain(getOptD("Sound.Master.Volume"));
  l.setALState();
}

void GameManager::toggleSimulationSound(bool b)
{
  for (int i=0; i < karts.size(); ++i)
    karts[i]->toggleSound(b);
}

}
}
