/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include <SDL/SDL.h>   // To load samples

#include "EngineSound.hh"
#include "util/PathFinder.hh"
#include "util/Log.hh"
#include "helpers/initAL.hh"

using top10::util::Log;

namespace top10
{
  namespace sound
  {
    EngineRpmRange::EngineRpmRange(ALuint buffer, double rpm_min, double rpm_start, double rpm_end, double rpm_max)
      : al_buffer(buffer),
      rpm_min(rpm_min),
      rpm_start(rpm_start),
      rpm_end(rpm_end),
      rpm_max(rpm_max)
    {
    }

    ALuint EngineRpmRange::getBuffer() const
    {
      return al_buffer;
    }

    double EngineRpmRange::getRpmEnd() const
    {
      return rpm_end;
    }

    double EngineRpmRange::getRpmStart() const
    {
      return rpm_start;
    }

    double EngineRpmRange::getRpmMin() const
    {
      return rpm_min;
    }

    double EngineRpmRange::getRpmMax() const
    {
      return rpm_max;
    }


    EngineSound::EngineSound()
    {
    }

    void EngineSound::insertRangeOn(top10::sound::EngineRpmRange r)
    {
      std::vector<EngineRpmRange>::iterator it;
      for (it = ranges_on.begin(); it != ranges_on.end() && it->getRpmStart() < r.getRpmStart(); ++it);
      ranges_on.insert(it, r);
    }

    void EngineSound::insertRangeOff(top10::sound::EngineRpmRange r)
    {
      std::vector<EngineRpmRange>::iterator it;
      for (it = ranges_off.begin(); it != ranges_off.end() && it->getRpmStart() < r.getRpmStart(); ++it);
      ranges_off.insert(it, r);
    }

    std::vector<ALuint> EngineSound::getBuffersOn() const
    {
      std::vector<ALuint> ret;
      for (std::vector<EngineRpmRange>::const_iterator it = ranges_on.begin(); it != ranges_on.end(); ++it)
	ret.push_back(it->getBuffer());
      return ret;
    }

    std::vector<ALuint> EngineSound::getBuffersOff() const
    {
      std::vector<ALuint> ret;
      for (std::vector<EngineRpmRange>::const_iterator it = ranges_off.begin(); it != ranges_off.end(); ++it)
	ret.push_back(it->getBuffer());
      return ret;
    }

    void EngineSound::getSourceParameters(double rpm, bool on_load, std::vector<double> &amplitudes, std::vector<double> &pitches) const
    {
      const std::vector<EngineRpmRange>* ranges = 0;
      if (on_load) ranges = &ranges_on;
      else         ranges = &ranges_off;

      for (std::vector<EngineRpmRange>::const_iterator it = ranges->begin(); it != ranges->end(); ++it)
      {
	// For all ranges finishing before rpm, disable the source.
	if (it->getRpmMax() < rpm)
	{
	  amplitudes.push_back(0.0);
	  pitches.push_back(0.0);
	}

	// For all ranges fading, compute the attenuation of the amplitude.
	else if (it->getRpmEnd() < rpm)
	{
	  double f = rpm - it->getRpmEnd();
	  f /= it->getRpmMax() - it->getRpmEnd();
	  f = 1.0 - f;
	  if (f < 0.0)
	    f = 0.0;
	  if (f > 1.0)
	    f = 1.0;
	  amplitudes.push_back(f);

	  pitches.push_back(getPitch(rpm, it->getRpmMin(), it->getRpmMax()));
	}

	// For ranges in full, set full amplitude
	else if (it->getRpmStart() < rpm)
	{
	  amplitudes.push_back(1.0);

	  pitches.push_back(getPitch(rpm, it->getRpmMin(), it->getRpmMax()));
	}

	// For rising ranges
	else if (it->getRpmMin() < rpm)
	{
	  double f = rpm - it->getRpmMin();
	  f /= it->getRpmStart() - it->getRpmMin();
	  if (f < 0.0)
	    f = 0.0;
	  if (f > 1.0)
	    f = 1.0;
	  amplitudes.push_back(f);

	  pitches.push_back(getPitch(rpm, it->getRpmMin(), it->getRpmMax()));
	}

	// Ranges not enabled yet
	else
	{
	  amplitudes.push_back(0.0);
	  pitches.push_back(0.0);
	}
      }
    }

    double EngineSound::getPitch(double rpm, double rpm_min, double rpm_max)
    {
      double f;
      f = rpm - rpm_min;
      f /= rpm_max-rpm_min;
      f /= 2.0;
      f += 0.5;
      return f;
    }

    EngineSoundInstance::EngineSoundInstance(const top10::sound::EngineSound* model)
      : model(model), error_reported(false)
    {
      std::vector<ALuint> buffers;
      std::vector<ALuint>::const_iterator it;

      buffers = model->getBuffersOn();

      ALenum al_err;
      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "EngineSoundInstance",
	std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " before creating sources ");

      for (it = buffers.begin(); it != buffers.end(); ++it)
      {
	ALuint new_source = 42;
	alGenSources(1, &new_source);

	alSourcei(new_source, AL_BUFFER, *it);

	alSourcei(new_source, AL_LOOPING, 1);

	al_sources_on.push_back(new_source);
      }

      buffers = model->getBuffersOff();
      for (it = buffers.begin(); it != buffers.end(); ++it)
      {
	ALuint new_source = 43;
	alGenSources(1, &new_source);

	alSourcei(new_source, AL_BUFFER, *it);

	alSourcei(new_source, AL_LOOPING, 1);

	al_sources_off.push_back(new_source);
      }

      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "EngineSoundInstance",
	std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " after creating sources ");

    }

    EngineSoundInstance::~EngineSoundInstance()
    {
      std::vector<ALuint>::iterator it;
      for (it = al_sources_on.begin(); it != al_sources_on.end(); ++it)
      {
	alDeleteSources(1, &(*it));
      }
      for (it = al_sources_off.begin(); it != al_sources_off.end(); ++it)
      {
	alDeleteSources(1, &(*it));
      }
    }

    void EngineSoundInstance::update(double rpm, bool on_load, const top10::math::Vector& pos, const top10::math::Vector& speed)
    {
      std::vector<ALuint>* active;
      std::vector<ALuint>* inactive;
    
      if (on_load)
      {
	active = &al_sources_on;
	inactive = &al_sources_off;
      }
      else
      {
	active = &al_sources_off;
	inactive = &al_sources_on;
      }

      ALenum al_err;
      al_err = alGetError();
      if (!error_reported && al_err != AL_NO_ERROR)
      {
	error_reported = true;
	Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
	std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " before alSourceStop (inactive).");
      }

      for (std::vector<ALuint>::iterator it = inactive->begin(); it != inactive->end(); ++it)
      {
	alSourceStop(*it);
      }

      std::vector<double> gains;
      std::vector<double> pitches;
      model->getSourceParameters(rpm, on_load, gains, pitches);

      std::vector<double>::size_type i = 0;
      for (std::vector<ALuint>::iterator it = active->begin(); it != active->end(); ++it, ++i)
      {
	if (gains.at(i) > 0.0 && pitches.at(i) > 0.0)
	{
	  al_err = alGetError();
	  if (!error_reported && al_err != AL_NO_ERROR)
	  {
	    error_reported = true;
	    Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
	    std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " before alSource (GAIN/PITCH).");
	  }

	  // Gain and pitch
	  alSourcef(*it, AL_GAIN, gains.at(i));
	  alSourcef(*it, AL_PITCH, pitches.at(i));
	  
	  // Position and velocity
	  alSource3f(*it, AL_POSITION, pos.x, pos.y, pos.z);
	  alSource3f(*it, AL_VELOCITY, speed.x, speed.y, speed.z);

	  al_err = alGetError();
	  if (!error_reported && al_err != AL_NO_ERROR)
	  {
	    error_reported = true;
	    Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
	    std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " before alGetSource(AL_SOURCE_STATE).");
	  }

	  ALint is_playing;
	  alGetSourcei(*it, AL_SOURCE_STATE, &is_playing);
	  if (is_playing != AL_PLAYING)
	  {
	    al_err = alGetError();
	    if (!error_reported && al_err != AL_NO_ERROR)
	    {
	      error_reported = true;
	      Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
	      std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " before alSourcePlay.");
	    }

	    alSourcePlay(*it);

	    al_err = alGetError();
	    if (!error_reported && al_err != AL_NO_ERROR)
	    {
	      error_reported = true;
	      Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
	      std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " after alSourcePlay.");
	    }
	  }
	}
	else
	{
	  al_err = alGetError();
	  if (!error_reported && al_err != AL_NO_ERROR)
	  {
	    error_reported = true;
	    Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
	    std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " before alSourceStop.");
	  }

	  alSourceStop(*it);

	  al_err = alGetError();
	  if (!error_reported && al_err != AL_NO_ERROR)
	  {
	    error_reported = true;
	    Log::getSingle()->send(Log::Warning, "EngineSoundInstance/update",
	    std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " after alSourceStop.");
	  }
	}
      }
    }

    EngineRpmRange EngineSound::loadData(const char *filename, double rpm_min, double rpm_start, double rpm_end, double rpm_max)
    {
      /* sound samples specs */
      SDL_AudioSpec wanted;
      wanted.freq = 22050;
      wanted.format = AUDIO_S16;
      wanted.channels = 1;    /* 1 = mono, 2 = stereo */
      Uint32 len;
      Uint8* buff;

      ALuint al_buffer = 0;

      std::string path = top10::util::PathFinder::defaultPathFinder().find(filename);
      if (path.empty())
	throw top10::util::Error(std::string("Could not find ") + std::string(filename));

      if (!SDL_LoadWAV(path.c_str(), &wanted, &buff, &len))
	throw top10::util::Error(SDL_GetError());

      ALenum al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "EngineSound/loadData",
	std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " before loading " + filename);

      alGenBuffers(1, &al_buffer);
      alBufferData(al_buffer, AL_FORMAT_MONO16, (void*)buff, len, wanted.freq);

      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "EngineSound/loadData",
	std::string("AL error ") + top10::helpers::makeErrorStr(al_err) + " after loading " + filename);

//      SDL_FreeWAV(buff);

      return EngineRpmRange(al_buffer, rpm_min, rpm_start, rpm_end, rpm_max);
    }

    EngineSound* buildKartSound()
    {
      EngineSound* ret = new EngineSound;

      try {
	EngineRpmRange rng = EngineSound::loadData("idle_off.wav", 0.0, 1.0, 40.0, 50.0);
	ret->insertRangeOff(rng);
      }
      catch(const top10::util::Error& err)
      {
	Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading idle_off.wav: ") + err);
      }

      try {
	EngineRpmRange rng = EngineSound::loadData("low_off.wav", 40.0, 50.0, 90.0, 100.0);
	ret->insertRangeOff(rng);
      }
      catch(const top10::util::Error& err)
      {
	Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading low_off.wav: ") + err);
      }

      try {
	EngineRpmRange rng = EngineSound::loadData("mid_off.wav", 90.0, 100.0, 150.0, 160.0);
	ret->insertRangeOff(rng);
      }
      catch(const top10::util::Error& err)
      {
	Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading mid_off.wav: ") + err);
      }

      try {
	EngineRpmRange rng = EngineSound::loadData("hi_off.wav", 150.0, 160.0, 200.0, 210.0);
	ret->insertRangeOff(rng);
      }
      catch(const top10::util::Error& err)
      {
	Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading hi_off.wav: ") + err);
      }


      try {
	EngineRpmRange rng = EngineSound::loadData("idle_on.wav", 0.0, 1.0, 40.0, 50.0);
	ret->insertRangeOn(rng);
      }
      catch(const top10::util::Error& err)
      {
	Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading idle_on.wav: ") + err);
      }

      try {
	EngineRpmRange rng = EngineSound::loadData("low_on.wav", 40.0, 50.0, 90.0, 100.0);
	ret->insertRangeOn(rng);
      }
      catch(const top10::util::Error& err)
      {
	Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading low_on.wav: ") + err);
      }

      try {
	EngineRpmRange rng = EngineSound::loadData("mid_on.wav", 90.0, 100.0, 150.0, 160.0);
	ret->insertRangeOn(rng);
      }
      catch(const top10::util::Error& err)
      {
	Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading mid_on.wav: ") + err);
      }

      try {
	EngineRpmRange rng = EngineSound::loadData("hi_on.wav", 150.0, 160.0, 200.0, 210.0);
	ret->insertRangeOn(rng);
      }
      catch(const top10::util::Error& err)
      {
	Log::getSingle()->send(Log::Warning, "buildKartSound", std::string("loading hi_on.wav: ") + err);
      }

      return ret;
    }
  };
};