/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "ControlsMenu.hh"
#include <sstream>
#include <iostream>

using namespace top10::ui_interactive;

const Action ControlsMenu::actions[] = {
  top10::racing::Steering, top10::racing::Acceleration,
  top10::racing::Braking, top10::racing::AccelerationAndBraking,
  top10::racing::BrakingAndAcceleration,
  top10::racing::SteerLeft, top10::racing::SteerRight,
  top10::racing::Accelerate, top10::racing::Brake,
  top10::racing::GearDown, top10::racing::GearUp,
  top10::racing::ResetKart,
  top10::racing::NextCamera, top10::racing::PrevCamera,
  top10::racing::NextKart, top10::racing::PrevKart,
  top10::racing::FreeCamera, top10::racing::FreeCameraHorizontal,
  top10::racing::FreeCameraVertical, top10::racing::FreeCameraMoveX,
  top10::racing::FreeCameraMoveZ
};

const int ControlsMenu::n_entries = sizeof(ControlsMenu::actions)/sizeof(Action);
const char* ControlsMenu::action_texts[] = {
  "Steering:", "Acceleration:", "Braking:", "Accel + Braking:", "Braking + Accel:",
  "Steer Left:", "Steer Right:", "Accelerate:", "Brake:", "Gear down:", "Gear up:",
  "Reset Kart:",
  "Next Cam:", "Prev Cam:", "Next Kart:", "Prev Kart:", "FreeCam:", "FreeCam H:", "FreeCam V:",
  "FeeCam X:", "FreeCam Z:"
};

ControlsMenu::ControlsMenu(UIPanel* panel, ActionMapping* _digs):
  Menu(panel),
  digs(_digs),
  isAssigning(false)
{
  // Create the right number of entries.
  for (int i=0; i<n_entries; ++i) {
    addMenu(0);
  }

  // Put the right text in each entry
  initEntries();

  updateNode();
}

int ControlsMenu::findActionIdx(Action action) const
{
  for (int i=0; i<getNumEntries(); ++i) {
    if (actions[i] == action) return i;
  }
  return -1;
}

void ControlsMenu::initEntries()
{
  for (unsigned int i=0; i<getNumEntries(); ++i) {
    setMenuText(i, std::string(action_texts[i]));
  }

  for (ActionMapping::const_iterator it = digs->begin();
       it != digs->end();
       ++it) {
    int idx = findActionIdx(it->second);
    assert(idx != -1);
    updateEntry(idx, it->first);
  }

  updateNode();
}

void ControlsMenu::updateEntry(int idx, Control ctrl)
{
  std::string entry = getMenuText(idx);
  std::string::size_type pos = entry.find(':', 0);
  assert(pos != std::string::npos);
  std::string new_str = entry.substr(0, pos+1);
  std::ostringstream buf;
  buf<<ctrl;
  new_str += buf.str();
  setMenuText(idx, new_str);
}

void ControlsMenu::event(SDL_Event event)
{
  if (!isAssigning) {
    Menu::event(event);
    return;
  }

  if (event.type == SDL_KEYUP && event.key.keysym.sym == SDLK_SPACE) return;

  if (Control::accepted(event)) {
    if (Control::getValue(event) > 16000 || Control::getValue(event) < -16000) {
      Control ctrl(event);
      updateEntries(ctrl);
      isAssigning = false;
    }
  }
}

void ControlsMenu::selected()
{
  isAssigning = true;
}

void ControlsMenu::updateEntries(Control ctrl)
{
  // Add the new mapping
  digs->insert(ctrl, actions[getEntryN()]);

  // Update text of all menu entries
  initEntries();
}
