#include "WallDialog.hh"
#include <QtGui/QVBoxLayout>
#include <QtGui/QFileDialog>
#include <QtGui/QLabel>


const char* WallDialog::WALL_LABEL = "Thickness";
const char* WallDialog::MESH_LABEL = "Upper height";

WallDialog::WallDialog(QWidget *parent)
: QDialog(parent),
m_editor(0),
m_wall_names(0),
m_points_list(0)
{
  // The tabs  
  m_tabs = new QTabWidget;
  m_tab1 = new QWidget;
  m_tab2 = new QWidget;
  m_tabs->insertTab(0, m_tab1, "Properties");
  m_tabs->insertTab(1, m_tab2, "Texture");

  // The texture widget
  m_texture = new TextureWidget;
  connect(m_texture->Texture_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_Texture_lineEdit_editingFinished()));
  connect(m_texture->ScaleLat_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_ScaleLat_lineEdit_editingFinished()));
  connect(m_texture->ScaleLong_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_ScaleLong_lineEdit_editingFinished()));
  connect(m_texture->TranslateLat_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_TranslateLat_lineEdit_editingFinished()));
  connect(m_texture->TranslateLong_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_TranslateLong_lineEdit_editingFinished()));
  connect(m_texture->FileSelect_pushButton, SIGNAL(pressed()), this, SLOT(on_FileSelect_pushButton_pressed()));

  // The extrusion path and the properties
  m_props = new QWidget;
  m_props_skel.setupUi(m_props);  

  m_wall_names = new top10::qt::ComboBoxEdit;
  m_points_list = new top10::qt::ListPickWidget;

  connect(m_wall_names, SIGNAL(itemInserted(const QString&)), this, SLOT(on_m_wall_names_itemCreated(const QString&)));
  connect(m_wall_names, SIGNAL(itemRemoved(const QString&)), this, SLOT(on_m_wall_names_itemDeleted(const QString&)));
  connect(m_wall_names, SIGNAL(itemSelected(const QString&)), this, SLOT(on_m_wall_names_itemSelected(const QString&)));
  connect(m_wall_names, SIGNAL(itemRenamed(const QString&)), this, SLOT(on_m_wall_names_itemRenamed(const QString&)));

  connect(m_points_list, SIGNAL(itemIncluded(int, const QString&)), this, SLOT(on_m_points_list_itemIncluded(int, const QString&)));
  connect(m_points_list, SIGNAL(itemExcluded(int, const QString&)), this, SLOT(on_m_points_list_itemExcluded(int, const QString&)));

  connect(m_props_skel.m_height, SIGNAL(returnPressed()), this, SLOT(on_m_props_height_returnPressed()));
  connect(m_props_skel.m_thickness, SIGNAL(returnPressed()), this, SLOT(on_m_props_thickness_returnPressed()));
  connect(m_props_skel.m_slant, SIGNAL(returnPressed()), this, SLOT(on_m_props_slant_returnPressed()));
  connect(m_props_skel.m_wall_radio, SIGNAL(toggled(bool)), this, SLOT(on_m_props_wall_toggled(bool)));
  connect(m_props_skel.m_mesh_radio, SIGNAL(toggled(bool)), this, SLOT(on_m_props_mesh_toggled(bool)));

  // Layout of the extrusion path and the properties
  QVBoxLayout* layout = new QVBoxLayout;
  layout->addWidget(m_points_list);
  layout->addWidget(m_props);
  m_tab1->setLayout(layout);

  // Layout of the texture widget
  QVBoxLayout* layout2 = new QVBoxLayout;
  layout2->addWidget(m_texture);
  m_tab2->setLayout(layout2);

  // Layout of the tabs
  QVBoxLayout* layout3 = new QVBoxLayout;
  layout3->addWidget(m_wall_names);
  layout3->addWidget(m_tabs);
  setLayout(layout3);
}



WallDialog::~WallDialog()
{
}



void WallDialog::setEditor(top10::tracked::WallEditor* ed)
{
  if (ed)
    ed->freeClient();

  ed->setClient(this);

  m_editor = ed;
}



void WallDialog::updateDialog()
{
  if (m_editor)
  {
    updateNames();
    updatePoints();
    updateProps();
    updateTexture();
  }
}



void WallDialog::updateNames()
{
  if (m_editor)
  {
    m_wall_names->clear();

    std::list<std::string> names;
    m_editor->getWallNames(&names);
    for (std::list<std::string>::const_iterator it = names.begin(); it != names.end(); ++it)
    {
      m_wall_names->insertItemNoEmit(*it);
    }
    m_wall_names->setCurrentItemNoEmit( m_editor->getWallName() );
  }
}



void WallDialog::updateTexture()
{
  if (m_editor && m_editor->getTexture())
  {
    m_texture->init(*m_editor->getTexture());
    updateImage();
  }
}



void WallDialog::updatePoints()
{
  m_points_list->clearNoEmit();

  std::list<std::string> all_points;
  m_editor->getAllPointNames(&all_points);
  for (std::list<std::string>::const_iterator it = all_points.begin(); it != all_points.end(); ++it)
  {
    m_points_list->insertItemNoEmit(*it);
  }

  std::list<std::string> points;
  m_editor->getPointNames(&points);
  for (std::list<std::string>::const_iterator it = points.begin(); it != points.end(); ++it)
  {
    m_points_list->includeItemNoEmit(*it, true /* insert_last */);
  }
}



void WallDialog::updateProps()
{
  m_props_skel.m_height->setText(QString().setNum(m_editor->getHeight()));
  m_props_skel.m_slant->setText(QString().setNum(m_editor->getSlant()));
  m_props_skel.m_thickness->setText(QString().setNum(m_editor->getThickness()));
  
  if (m_editor->isTypeWall())
  {
    m_props_skel.m_wall_radio->toggle();
    m_props_skel.m_thickness_label->setText(WALL_LABEL);
  }
  else if (m_editor->isTypeMesh())
  {
    m_props_skel.m_mesh_radio->toggle();
    m_props_skel.m_thickness_label->setText(MESH_LABEL);
  }
}



void WallDialog::updateUi()
{
  updateDialog();
}



void WallDialog::on_m_wall_names_itemCreated(const QString& item)
{
  bool status = true;

  const std::string new_name = m_editor->newWall();

  // Set the name chosen by the editor.
  status = status && m_wall_names->setCurrentItemNoEmit( item.toStdString() );
  status = status && m_wall_names->setCurrentNameNoEmit( new_name );

  updateDialog();

  if (!status)
    top10::util::Log::getSingle()->send(top10::util::Log::Critical, "WallDialog/WallCreated", "Failed to name new item " + new_name);
}



void WallDialog::on_m_wall_names_itemDeleted(const QString& item)
{
  bool status = m_editor->setCurrentWall(item.toStdString());

  if (status)
  {
    m_editor->deleteWall();
    updatePoints();
    updateProps();
    updateTexture();
  }
  else
    top10::util::Log::getSingle()->send(top10::util::Log::Critical, "WallDialog/WallDeleted", "No item " + item.toStdString());
}



void WallDialog::on_m_wall_names_itemSelected(const QString& item)
{
  bool status = m_editor->setCurrentWall(item.toStdString());

  if (status)
  {
    updatePoints();
    updateProps();
    updateTexture();
  }
  else
    top10::util::Log::getSingle()->send(top10::util::Log::Critical, "WallDialog/WallSelected", "No item " + item.toStdString());
}



void WallDialog::on_m_wall_names_itemRenamed(const QString& new_name)
{
  bool status = m_editor->renameWall(new_name.toStdString());
  
  if (!status)
    top10::util::Log::getSingle()->send(
      top10::util::Log::Critical,
      "WallDialog/WallRenamed",
      "Cannot rename " + m_editor->getWallName() + " to " + new_name.toStdString() );
}



void WallDialog::on_m_points_list_itemIncluded(int pos, const QString& item)
{
  m_editor->insertPoint(pos, item.toStdString());
  updatePoints();
}



void WallDialog::on_m_points_list_itemExcluded(int pos, const QString& item)
{
  m_editor->removePoint(item.toStdString());
  updatePoints();
}



void WallDialog::on_m_props_height_returnPressed()
{
  double value = m_props_skel.m_height->text().toDouble();
  m_editor->setHeight(value);
  m_props_skel.m_height->setText(QString().setNum(value));
}



void WallDialog::on_m_props_thickness_returnPressed()
{
  double value = m_props_skel.m_thickness->text().toDouble();
  m_editor->setThickness(value);
  m_props_skel.m_thickness->setText(QString().setNum(value));
}



void WallDialog::on_m_props_slant_returnPressed()
{
  double value = m_props_skel.m_slant->text().toDouble();
  m_editor->setSlant(value);
  m_props_skel.m_slant->setText(QString().setNum(value));
}



void WallDialog::on_m_props_wall_toggled(bool b)
{
  if (b)
  {
    m_editor->setTypeWall();
    m_props_skel.m_thickness_label->setText(WALL_LABEL);
  }
}



void WallDialog::on_m_props_mesh_toggled(bool b)
{
  if (b)
  {
    m_editor->setTypeMesh();
    m_props_skel.m_thickness_label->setText(MESH_LABEL);
  }
}



void WallDialog::on_Texture_lineEdit_editingFinished( )
{
  m_editor->getTexture()->setFilename(m_texture->Texture_lineEdit->text().toStdString());
  updateImage();
}



void WallDialog::on_ScaleLat_lineEdit_editingFinished( )
{
  m_editor->getTexture()->setScaleLat(m_texture->ScaleLat_lineEdit->text().toFloat());
  updateImage();
}



void WallDialog::on_ScaleLong_lineEdit_editingFinished( )
{
  m_editor->getTexture()->setScaleLong(m_texture->ScaleLong_lineEdit->text().toFloat());
  updateImage();
}



void WallDialog::on_TranslateLat_lineEdit_editingFinished( )
{
  m_editor->getTexture()->setTranslateLat(m_texture->TranslateLat_lineEdit->text().toFloat());
  updateImage();
}



void WallDialog::on_TranslateLong_lineEdit_editingFinished( )
{
  m_editor->getTexture()->setTranslateLong(m_texture->TranslateLong_lineEdit->text().toFloat());
  updateImage();
}



void WallDialog::on_FileSelect_pushButton_pressed( )
{  
  // Open the file selector
  QString s = QFileDialog::getOpenFileName(this, tr("Choose a file to open"), "./", tr("Texture (*.tga *.jpg *.png)"));
  if (s.isEmpty()) return;
  
  m_texture->Texture_lineEdit->setText(s);
  
  std::string name = s.toStdString();
  m_editor->getTexture()->setFilename(name);
  
  updateImage();
}



void WallDialog::updateImage()
{
  if (m_texture->Texture_lineEdit->text().isEmpty())
    m_texture->erase();
  else
    m_texture->loadFile(m_texture->Texture_lineEdit->text());
  
  QDialog::update();
}
