/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "SectionsDlg.hh"
#include "util/Log.hh"
#include "util/strconv.hh"

using top10::util::Log;

SectionsDlg::SectionsDlg():
  ed(0), viewer(0)
{
  setupUi(this);
  tabWidget->addTab(&texture, "Texture");
  connect(texture.Texture_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_Texture_lineEdit_editingFinished()));
  connect(texture.ScaleLat_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_ScaleLat_lineEdit_editingFinished()));
  connect(texture.ScaleLong_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_ScaleLong_lineEdit_editingFinished()));
  connect(texture.TranslateLat_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_TranslateLat_lineEdit_editingFinished()));
  connect(texture.TranslateLong_lineEdit, SIGNAL(editingFinished()), this, SLOT(on_TranslateLong_lineEdit_editingFinished()));
  connect(texture.FileSelect_pushButton, SIGNAL(pressed()), this, SLOT(on_FileSelect_pushButton_pressed()));

  connect(Path_comboBox->lineEdit(), SIGNAL(textEdited(const QString&)), this, SLOT(on_Path_comboBox_editTextEdited(const QString&)));
}

void SectionsDlg::fullUpdate()
{
  assert(ed);
  
  QString tmp;
  
  // Populate the list of paths
  Path_comboBox->clear();
  std::list<std::string> path_names = ed->getPathNames();
  for (std::list<std::string>::const_iterator it = path_names.begin();
      it != path_names.end();
      ++it)
  {
    Path_comboBox->addItem(QString(it->c_str()));
    if (ed->getPathName() == *it) {
      Path_comboBox->setCurrentIndex(Path_comboBox->count()-1);
    }
  }

  // Texture of the current path
  texture.Texture_lineEdit->setText(QString(ed->getTexture().c_str()));
  texture.ScaleLat_lineEdit->setText(QString().setNum(ed->getTextureScaleLat()));
  texture.ScaleLong_lineEdit->setText(QString().setNum(ed->getTextureScaleLong()));
  texture.TranslateLat_lineEdit->setText(QString().setNum(ed->getTextureTranslateLat()));
  texture.TranslateLong_lineEdit->setText(QString().setNum(ed->getTextureTranslateLong()));
  updateImage();
  
  // Sections in the current path
  int idx = ed->getCurrentControlIndex();
  if (idx == -1)
    CurrentControl_lineEdit->clear();
  else
    CurrentControl_lineEdit->setText(QString().setNum(idx));

  idx = ed->getLastControlIndex();
  if (idx == -1)
    MaxControl_label->clear();
  else
    MaxControl_label->setText(QString().setNum(idx));
  
  // Looping
  if (ed->getLoops()) Loop_checkBox->setCheckState(Qt::Checked);
  else Loop_checkBox->setCheckState(Qt::Unchecked);

  // Length of section
  double section_length = ed->getCurrentPathLength();
  QString length_string = QString("Length: ") + QString().number(section_length) + " m";
  Length_label->setText(length_string);

  // Waypoint combo box
  Waypoint_comboBox->clear();
  for (std::vector<top10::tracked::PathEditor::WayPoint>::const_iterator it = ed->getWaypointsEditor()->waypoints.begin();
       it != ed->getWaypointsEditor()->waypoints.end();
       ++it)
  {
    Waypoint_comboBox->addItem(it->name.c_str());
    if (ed->getWaypointName() == it->name) {
      Waypoint_comboBox->setCurrentIndex(Waypoint_comboBox->count()-1);
    }
  }
  
  TgFactorIn_doubleSpinBox->setValue(ed->getTgFactorIn());
  TgFactorOut_doubleSpinBox->setValue(ed->getTgFactorOut());
  LRoadWidth_doubleSpinBox->setValue(ed->getLRoadWidth());
  RRoadWidth_doubleSpinBox->setValue(ed->getRRoadWidth());
  SecondaryTg_checkBox->setCheckState(ed->getTgUse()==1? Qt::Checked : Qt::Unchecked);

  // Kerbs: TODO
}

void SectionsDlg::showEvent(QShowEvent* e)
{
  update();   
}

void SectionsDlg::on_Path_comboBox_editTextEdited(const QString& s)
{
  assert(ed);
  ed->setPathName(s.toStdString());
  update();
}

void SectionsDlg::on_Path_comboBox_activated(const QString& s)
{
  assert(ed);
  ed->gotoPath(s.toStdString());
  update();
  viewer->updateGL();
}

void SectionsDlg::on_PathDelete_pushButton_pressed()
{
  assert(ed);
  ed->removePath();
  update();
  viewer->updateGL();
}

void SectionsDlg::on_PathAdd_pushButton_pressed()
{
  assert(ed);
  ed->addPath();
  update();  
  viewer->updateGL();
}

void SectionsDlg::on_PrevControl_Btn_pressed()
{
  assert(ed);
  ed->gotoPrevControl();
  update();
  viewer->updateGL();
}

void SectionsDlg::on_NextControl_Btn_pressed()
{
  assert(ed);
  ed->gotoNextControl();
  update();  
  viewer->updateGL();
}

void SectionsDlg::on_CurrentControl_lineEdit_editingFinished()
{
  assert(ed);
  bool ok = false;
  int idx = -1;
  idx = CurrentControl_lineEdit->text().toInt(&ok);
  if (ok)
    ed->gotoControl(idx);
  update();
  viewer->updateGL();
}

void SectionsDlg::on_DeleteControl_Btn_pressed()
{
  assert(ed);
  ed->removeControl();
  update();
  viewer->updateGL();
}

void SectionsDlg::on_AddControl_Btn_pressed()
{
  assert(ed);
  ed->addControl();
  update();
  viewer->updateGL();    
}

void SectionsDlg::on_LRoadWidth_doubleSpinBox_valueChanged(double x)
{
  assert(ed);
  ed->setLRoadWidth(x);
  update();
  viewer->updateGL(); 
}

void SectionsDlg::on_RRoadWidth_doubleSpinBox_valueChanged(double x)
{
  assert(ed);
  ed->setRRoadWidth(x);
  update();
  viewer->updateGL(); 
}

void SectionsDlg::on_TgFactorIn_doubleSpinBox_valueChanged(double x)
{
  assert(ed);
  ed->setTgFactorIn(x);
  update();
  viewer->updateGL();
}

void SectionsDlg::on_TgFactorOut_doubleSpinBox_valueChanged(double x)
{
  assert(ed);
  ed->setTgFactorOut(x);
  update();
  viewer->updateGL();
}

void SectionsDlg::on_SecondaryTg_checkBox_stateChanged(int state)
{
  assert(ed);

  switch(state)
  {
  case Qt::Checked: ed->setTgUse(1); break;
  case Qt::Unchecked: ed->setTgUse(0); break;
  }

  viewer->updateGL();
}

void SectionsDlg::on_Loop_checkBox_stateChanged(int state)
{
  assert(ed);
  if ((state != Qt::Unchecked) == ed->getLoops()) return;
  
  ed->setLoops((state != Qt::Unchecked));
  update();
  viewer->updateGL();
}

void SectionsDlg::on_Waypoint_comboBox_activated(int idx)
{
  assert(ed);
  ed->setWaypoint(Waypoint_comboBox->itemText(idx).toStdString());
  update();
  viewer->updateGL();
}

void SectionsDlg::on_Fetch_pushButton_pressed()
{
  assert(ed);
  ed->setWaypoint(ed->getWaypointsEditor()->getCurrentName());
  update();
  viewer->updateGL();
}

void SectionsDlg::on_AutoTg_pushButton_pressed()
{
  assert(ed);
  ed->smoothTangents();
  viewer->updateGL();
}

void SectionsDlg::on_Texture_lineEdit_editingFinished( )
{
  assert(ed);
  ed->setTexture(texture.Texture_lineEdit->text().toStdString());
  updateImage();
}

void SectionsDlg::on_ScaleLat_lineEdit_editingFinished( )
{
  assert(ed);
  ed->setTextureScaleLat(texture.ScaleLat_lineEdit->text().toFloat());
  updateImage();
}

void SectionsDlg::on_ScaleLong_lineEdit_editingFinished( )
{
  assert(ed);
  ed->setTextureScaleLong(texture.ScaleLong_lineEdit->text().toFloat());
  updateImage();
}

void SectionsDlg::on_TranslateLat_lineEdit_editingFinished( )
{
  assert(ed);
  ed->setTextureTranslateLat(texture.TranslateLat_lineEdit->text().toFloat());
  updateImage();
}

void SectionsDlg::on_TranslateLong_lineEdit_editingFinished( )
{
  assert(ed);
  ed->setTextureTranslateLong(texture.TranslateLong_lineEdit->text().toFloat());
  updateImage();
}

void SectionsDlg::on_FileSelect_pushButton_pressed( )
{
  assert(ed);
  
  // Open the file selector
  QString s = QFileDialog::getOpenFileName(this, tr("Choose a file to open"), "./", tr("Texture (*.tga *.jpg *.png)"));
  if (s.isEmpty()) return;
  
  texture.Texture_lineEdit->setText(s);
  
  std::string name = s.toStdString();
  ed->setTexture(name);
  
  updateImage();
}

void SectionsDlg::updateImage()
{
  if (texture.Texture_lineEdit->text().isEmpty())
    texture.erase();
  else
    texture.loadFile(texture.Texture_lineEdit->text());
  
  QDialog::update();
}
