/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "QGLView.hh"
#include "math/Vertex.hh"
#include "math/Matrix.hh"
#include "util/PathFinder.hh"

#include <cassert>
#include <math.h>

using top10::math::Vector;

QGLView::QGLView(QWidget* parent, QGLWidget* share):
  QGLWidget(parent, share),
  pan_speed(50.0),
  old_x(-1), old_y(-1), camera(0), renderer(0), hud_r(255), hud_g(0), hud_b(255)
{
  setMouseTracking(true);
  setFocusPolicy(Qt::ClickFocus);
}

QGLView::~QGLView()
{
  if (renderer) delete renderer;
}

void QGLView::addDrawable(top10::tracked::Drawable* d)
{
  if (renderer)
    top_node->addChild(d->getSwitchNode());
  drawables.push_back(d);
}

void QGLView::initializeGL()
{
  if (!renderer) {
    renderer = new top10::graphX::Renderer;
    
    // The scene graph of the track
    top_node = new top10::graphX::MaterialNode;
    top_node->r = top_node->g = top_node->b = 255;
      
    light_node = new top10::graphX::LightNode;
    light_node->addChild(top_node.getPtr());
  
    // SG of the skybox
    //skybox = new top10::graphX::SkyBoxNode;
      
    // SG of the HUD
    hud_node = new top10::graphX::MaterialNode;
    hud_node->r = hud_r;
    hud_node->g = hud_g;
    hud_node->b = hud_b;
    mark_node = new top10::graphX::AimMarkNode;
    hud_node->addChild(mark_node.getPtr());  
    
    for (std::vector<top10::tracked::Drawable*>::iterator it = drawables.begin(); it != drawables.end(); ++it) {
      top_node->addChild((*it)->getSwitchNode());
    }
  }
  
  renderer->initGL(800, 600);

  std::string font_path = top10::util::PathFinder::defaultPathFinder().find("default.txf");
  if (!font_path.empty())
    font.load(font_path.c_str());
}

void QGLView::resizeGL(int w, int h)
{
  glViewport(0, 0, w, h);
  renderer->resizeGL(w, h);
  if (camera) {
      camera->setRatio((double)w/h);
      camera->update();
  }
}

void QGLView::paintGL()
{
  renderer->clearAllGL();
  
  if (camera) {
    renderer->setCamera(camera);

    //renderer->clearList();
    //renderer->buildList(skybox.getPtr());
    //renderer->renderSkyBoxGL();
    
    renderer->clearDepthGL();
    top10::graphX::RenderList main_rl;
    renderer->buildList(/*light_node*/top_node.getPtr(), &main_rl);
    renderer->renderGL(main_rl);
    
    renderer->clearDepthGL();
    top10::graphX::RenderList hud_rl;
    renderer->buildList(hud_node.getPtr(), &hud_rl);
    renderer->renderHudGL(hud_rl);
  }
}

void QGLView::mouseMoveEvent(QMouseEvent* e)
{
  using top10::math::Vector;
  // Rotate the camera
  if (e->buttons() & Qt::RightButton) {
    if (old_x >= 0) {
      double f_x = 180.0/width();
      double f_y = 90.0/height();

      if (camera) {
	camera->rotateY((e->globalX() - old_x)*f_x);
	camera->rotateX((e->globalY() - old_y)*f_y);
	camera->update();
	updateGL();
      }
    }
  }
  // Move the camera (pan)
  else if (e->buttons() & Qt::LeftButton) {
    if (old_x >= 0) {
      double f_x = pan_speed/width();
      double f_y = pan_speed/height();
      if (camera) {
	Vector flat_dir = camera->getView().getDirection();
	flat_dir.y = 0.0;
	double s = flat_dir.size();
	if (s > SMALL_VECTOR)
	  flat_dir /= s;
	else {
	  flat_dir = camera->getView().getUp();
	  flat_dir.y = 0;
	  s = flat_dir.size();
	  assert(s > SMALL_VECTOR);
	  flat_dir /= s;
	}
	camera->translate((e->globalX()-old_x)*f_x*camera->getView().getRight() - (e->globalY()-old_y)*f_y*flat_dir);
	camera->update();
	updateGL();
      }
    }
  }
  else if (e->buttons() & Qt::MidButton) {
    if (old_x >= 0) {
      double f_y = pan_speed/height();
      if (camera) {
	camera->translate(Vector(0.0, (e->globalY() - old_y) * f_y, 0.0));
	camera->update();
	updateGL();
      }
    }
  }

  old_x = e->globalX();
  old_y = e->globalY();
}

void QGLView::keyPressEvent(QKeyEvent* ev)
{
  switch(ev->key()) {
    case Qt::Key_Plus: camera->zoom(1.1); camera->update(); updateGL(); break;
    case Qt::Key_Minus: camera->zoom(1.0/1.1); camera->update(); updateGL(); break; 
    default: ev->ignore();
  }
}

fntFont* QGLView::getFont()
{
  return &font;
}

float QGLView::getPanSpeed() const
{
  return pan_speed;
}

void QGLView::setPanSpeed(float s)
{
  pan_speed = s;
}

float QGLView::getFarDist() const
{
  if (camera)
    return camera->getView().getFar();
  return 0.0;
}

void QGLView::setFarDist(float d)
{
  if (camera)
    camera->setFar(d);
}
