/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include "QGLMeshView.hh"
#include "graphX/GridNode.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/TransformNode.hh"

QGLMeshView::QGLMeshView(QWidget* parent, QGLWidget* share)
  : QGLWidget(parent, share),
  mesh_ed(0),
  mode(ModelView),
  old_x(-1),
  old_y(-1),
  renderer(0)
{
  setMouseTracking(true);
}


QGLMeshView::~QGLMeshView()
{
  delete renderer;
}

void QGLMeshView::setMeshEditor( top10::helpers::MeshEditor* ed )
{
  mesh_ed = ed;
  makeTopNode();
}

void QGLMeshView::setMode( Mode m )
{
  mode = m;
}

void QGLMeshView::initializeGL( )
{
  if (!renderer) {
    renderer = new top10::graphX::Renderer;
    makeTopNode();
  } 
  renderer->initGL(800, 600);
  
  camera = new top10::tracked::MainCamera;
}

void QGLMeshView::makeTopNode( )
{
  top_node = new top10::graphX::GroupNode;

  top10::graphX::MaterialNode* mat_node = new top10::graphX::MaterialNode;
  mat_node->r = grid_r;
  mat_node->g = grid_g;
  mat_node->b = grid_b;
  top_node->addChild(mat_node);
  
  top10::graphX::GridNode* grid_node = new top10::graphX::GridNode;
  grid_node->setGridUnit(2.0);
  grid_node->setSizeX(100.0);
  grid_node->setSizeZ(100.0);
  mat_node->addChild(grid_node);
  
  if (mesh_ed && mesh_ed->getCurrentNode())
    mat_node->addChild(mesh_ed->getCurrentNode());
}

void QGLMeshView::resizeGL( int w, int h )
{
  assert(renderer);
  assert(camera.isValid());
  
  renderer->resizeGL(w, h);
  camera->setRatio((double)w/h);
  camera->update();
}

void QGLMeshView::paintGL( )
{
  if (!renderer) return;
  
  makeTopNode();
  
  renderer->clearAllGL();
  renderer->setCamera(camera.getPtr());
  
  top10::graphX::RenderList main_rl;
  renderer->buildList(top_node.getPtr(), &main_rl);
  renderer->renderGL(main_rl);
}

void QGLMeshView::mouseMoveEvent( QMouseEvent* mouse_ev )
{
  switch (mode)
  {
    case ModelView: modelView(mouse_ev); break;
    case ModelTranslateXZ: modelTranslateXZ(mouse_ev); break;
    case ModelTranslateXY: modelTranslateXY(mouse_ev); break;
    case ModelRotateX: modelRotateX(mouse_ev); break;
    case ModelRotateY: modelRotateY(mouse_ev); break;
    case ModelRotateZ: modelRotateZ(mouse_ev); break;
  }
}

void QGLMeshView::keyPressEvent( QKeyEvent * ev )
{
  switch (mode)
  {
    case ModelView: modelView(ev); break;
    default: /* do nothing */ break; 
  }
}

void QGLMeshView::modelView( QMouseEvent * e )
{
  using top10::math::Vector;
  // Rotate the camera
  if (e->buttons() & Qt::RightButton) {
    if (old_x >= 0) {
      double f_x = 180.0/width();
      double f_y = 90.0/height();

      camera->rotateY((e->globalX() - old_x)*f_x);
      camera->rotateX((e->globalY() - old_y)*f_y);
      camera->update();
      updateGL();
    }
  }
  // Move the camera (pan)
  else if (e->buttons() & Qt::LeftButton) {
    if (old_x >= 0) {
      double f_x = 50.0/width();
      double f_y = 50.0/height();
	
      Vector flat_dir = camera->getView().getDirection();
      flat_dir.y = 0.0;
      double s = flat_dir.size();
      if (s > SMALL_VECTOR)
        flat_dir /= s;
      else {
        flat_dir = camera->getView().getUp();
        flat_dir.y = 0;
        s = flat_dir.size();
        assert(s > SMALL_VECTOR);
        flat_dir /= s;
      }
      camera->translate((e->globalX()-old_x)*f_x*camera->getView().getRight() - (e->globalY()-old_y)*f_y*flat_dir);
      camera->update();
      updateGL();
    }
  }
  else if (e->buttons() & Qt::MidButton) {
    if (old_x >= 0) {
      double f_y = 50.0/height();
      camera->translate(Vector(0.0, (e->globalY() - old_y) * f_y, 0.0));
      camera->update();
      updateGL();
    }
  }

  old_x = e->globalX();
  old_y = e->globalY();
}

void QGLMeshView::modelView( QKeyEvent * ev )
{
  switch(ev->key()) {
    case Qt::Key_Plus: camera->zoom(1.1); camera->update(); updateGL(); break;
    case Qt::Key_Minus: camera->zoom(1.0/1.1); camera->update(); updateGL(); break; 
    default: ev->ignore();
  }
}

void QGLMeshView::modelTranslateXZ( QMouseEvent * e )
{
  using top10::math::Vector;

  // Move the object
  if (e->buttons() & Qt::LeftButton) {
    if (old_x >= 0) {
      double f_x = 50.0/width();
      double f_y = 50.0/height();

      mesh_ed->translate((e->globalX()-old_x)*f_x*camera->getView().getRight() - (e->globalY()-old_y)*f_y*camera->getView().getDirection());
      
      updateGL();
    }
  }

  old_x = e->globalX();
  old_y = e->globalY();
}

void QGLMeshView::modelTranslateXY( QMouseEvent * e )
{
  using top10::math::Vector;

  // Move the object
  if (e->buttons() & Qt::LeftButton) {
    if (old_x >= 0) {
      double f_x = 50.0/width();
      double f_y = 50.0/height();

      mesh_ed->translate((e->globalX()-old_x)*f_x*camera->getView().getRight() - (e->globalY()-old_y)*f_y*camera->getView().getUp());
      
      updateGL();
    }
  }

  old_x = e->globalX();
  old_y = e->globalY();
}

void QGLMeshView::modelRotateX( QMouseEvent * e )
{
  using top10::math::Vector;

  // Rotate the object
  if (e->buttons() & Qt::LeftButton) {
    if (old_x >= 0) {
      mesh_ed->rotate(top10::math::X, 180.0*(e->globalX()-old_x)/width());      
      updateGL();
    }
  }

  old_x = e->globalX();
  old_y = e->globalY();
}

void QGLMeshView::modelRotateY( QMouseEvent * e )
{
  using top10::math::Vector;

  // Rotate the object
  if (e->buttons() & Qt::LeftButton) {
    if (old_x >= 0) {
      mesh_ed->rotate(top10::math::Y, 180.0*(e->globalX()-old_x)/width());      
      updateGL();
    }
  }

  old_x = e->globalX();
  old_y = e->globalY();
}

void QGLMeshView::modelRotateZ( QMouseEvent * e )
{
  using top10::math::Vector;

  // Rotate the object
  if (e->buttons() & Qt::LeftButton) {
    if (old_x >= 0) {
      mesh_ed->rotate(top10::math::Z, 180.0*(e->globalX()-old_x)/width());      
      updateGL();
    }
  }

  old_x = e->globalX();
  old_y = e->globalY();
}


