/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "PathDlg.hh"
#include <Qt/qlabel.h>
#include <Qt/qlineedit.h>
#include <Qt/qlcdnumber.h>
#include <Qt/qpushbutton.h>

PathDlg::PathDlg():
  viewer(0), ed(0), rotation_step(0.15),
  in_update(false)
{
  setupUi(this);
  
  connect(AddBtn, SIGNAL(pressed()), this, SLOT(insertPoint()));
  connect(DelBtn, SIGNAL(pressed()), this, SLOT(deletePoint()));
  connect(PrevBtn, SIGNAL(pressed()), this, SLOT(prevPoint()));
  connect(NextBtn, SIGNAL(pressed()), this, SLOT(nextPoint()));
  connect(PickBtn, SIGNAL(pressed()), this, SLOT(pickPoint()));
  connect(XEditCP1, SIGNAL(editingFinished()), this, SLOT(setPoint()));
  connect(YEditCP1, SIGNAL(editingFinished()), this, SLOT(setPoint()));
  connect(ZEditCP1, SIGNAL(editingFinished()), this, SLOT(setPoint()));
  connect(TangentX_lineEdit, SIGNAL(editingFinished()), this, SLOT(setTangent()));
  connect(TangentY_lineEdit, SIGNAL(editingFinished()), this, SLOT(setTangent()));
  connect(TangentZ_lineEdit, SIGNAL(editingFinished()), this, SLOT(setTangent()));
  connect(TangentX2_lineEdit, SIGNAL(editingFinished()), this, SLOT(setTangent()));
  connect(TangentY2_lineEdit, SIGNAL(editingFinished()), this, SLOT(setTangent()));
  connect(TangentZ2_lineEdit, SIGNAL(editingFinished()), this, SLOT(setTangent()));
  connect(RotateCCW_pushButton, SIGNAL(pressed()), this, SLOT(rotateTangentCCW()));
  connect(RotateCW_pushButton, SIGNAL(pressed()), this, SLOT(rotateTangentCW()));
  connect(RotateCCW2_pushButton, SIGNAL(pressed()), this, SLOT(rotateTangentCCW2()));
  connect(RotateCW2_pushButton, SIGNAL(pressed()), this, SLOT(rotateTangentCW2()));
  connect(Name_lineEdit, SIGNAL(editingFinished()), this, SLOT(setName()));
}

void PathDlg::insertPoint()
{
  assert(ed);
  assert(viewer);
  ed->addNew();
  viewer->updateGL();
  update();
}

void PathDlg::deletePoint()
{
  assert(ed);
  assert(viewer);
  ed->remove();
  viewer->updateGL();
  update();  
}

void PathDlg::nextPoint()
{
  assert(ed);
  assert(viewer);
  ed->next();
  viewer->updateGL();
  update();
}

void PathDlg::prevPoint()
{
  assert(ed);
  assert(viewer);
  ed->previous();
  viewer->updateGL();
  update();
}

void PathDlg::on_Current_lineEdit_editingFinished()
{
  assert(ed);
  int idx = 0;
  bool ok = false;
  idx = Current_lineEdit->text().toInt(&ok);
  if (ok)
  {
    ed->gotoIdx(idx);
    viewer->updateGL();
    update();
  }
}

void PathDlg::pickPoint()
{
  assert(ed);
  assert(viewer);
  ed->pick();
  viewer->updateGL();
  update();  
}

void PathDlg::update()
{
  in_update = true;
  
  assert(ed);
  top10::math::Vector cp = ed->getCurrentPos();
  QString tmp;
  tmp.setNum(cp.x);
  XEditCP1->setText(tmp);
  tmp.setNum(cp.y);
  YEditCP1->setText(tmp);
  tmp.setNum(cp.z);
  ZEditCP1->setText(tmp);
  
  cp = ed->getCurrentTangent();
  tmp.setNum(cp.x);
  TangentX_lineEdit->setText(tmp);
  tmp.setNum(cp.y);
  TangentY_lineEdit->setText(tmp);
  tmp.setNum(cp.z);
  TangentZ_lineEdit->setText(tmp);

  cp = ed->getCurrentTangent2();
  tmp.setNum(cp.x);
  TangentX2_lineEdit->setText(tmp);
  tmp.setNum(cp.y);
  TangentY2_lineEdit->setText(tmp);
  tmp.setNum(cp.z);
  TangentZ2_lineEdit->setText(tmp);

  Num_label->setText(QString().setNum((int)ed->waypoints.size()));
  int idx = ed->getCurrentIndex();
  if (idx == -1) {
    Current_lineEdit->clear();
    PickBtn->setEnabled(false);
    XEditCP1->setEnabled(false);
    YEditCP1->setEnabled(false);
    ZEditCP1->setEnabled(false);
    TangentX_lineEdit->setEnabled(false);
    TangentY_lineEdit->setEnabled(false);
    TangentZ_lineEdit->setEnabled(false);
    TangentX2_lineEdit->setEnabled(false);
    TangentY2_lineEdit->setEnabled(false);
    TangentZ2_lineEdit->setEnabled(false);
    DelBtn->setEnabled(false);
    Name_lineEdit->setEnabled(false);
    Name_lineEdit->clear();
    BankAngle_doubleSpinBox->setEnabled(false);
  }
  else {
    Current_lineEdit->setText(QString().setNum(idx));
    PickBtn->setEnabled(true);
    XEditCP1->setEnabled(true);
    YEditCP1->setEnabled(true);
    ZEditCP1->setEnabled(true);
    TangentX_lineEdit->setEnabled(true);
    TangentY_lineEdit->setEnabled(true);
    TangentZ_lineEdit->setEnabled(true);
    TangentX2_lineEdit->setEnabled(true);
    TangentY2_lineEdit->setEnabled(true);
    TangentZ2_lineEdit->setEnabled(true);
    DelBtn->setEnabled(true);
    Name_lineEdit->setEnabled(true);
    tmp = ed->getCurrentName().c_str();
    Name_lineEdit->setText(tmp);
    BankAngle_doubleSpinBox->setEnabled(true);
    BankAngle_doubleSpinBox->setValue(ed->getCurrentBankAngle());
  }  
  in_update = false;
}

void PathDlg::setPoint()
{
  if (in_update) return;
  
  assert(ed);
  assert(viewer);
  top10::math::Vector V;
  V.x = XEditCP1->text().toDouble();
  V.y = YEditCP1->text().toDouble();
  V.z = ZEditCP1->text().toDouble();
  ed->setCurrentPos(V); 
  
  viewer->updateGL();
  update();
}

void PathDlg::setTangent()
{
  if (in_update) return;
  
  assert(ed);
  assert(viewer);
  top10::math::Vector V;
  V.x = TangentX_lineEdit->text().toDouble();
  V.y = TangentY_lineEdit->text().toDouble();
  V.z = TangentZ_lineEdit->text().toDouble();
  ed->setCurrentTangent(V); 

  V.x = TangentX2_lineEdit->text().toDouble();
  V.y = TangentY2_lineEdit->text().toDouble();
  V.z = TangentZ2_lineEdit->text().toDouble();
  ed->setCurrentTangent2(V); 

  viewer->updateGL();
  update();
}

void PathDlg::rotateTangentCCW()
{
  if (in_update) return;
  
  assert(ed);
  assert(viewer);
  top10::math::Rotation3 R(rotation_step, top10::math::Vector(0.0, 1.0, 0.0));
  ed->setCurrentTangent( R*ed->getCurrentTangent() );
  
  viewer->updateGL();
  update();
}

void PathDlg::rotateTangentCCW2()
{
  if (in_update) return;
  
  assert(ed);
  assert(viewer);
  top10::math::Rotation3 R(rotation_step, top10::math::Vector(0.0, 1.0, 0.0));
  ed->setCurrentTangent2( R*ed->getCurrentTangent2() );
  
  viewer->updateGL();
  update();
}

void PathDlg::rotateTangentCW()
{
  if (in_update) return;
  
  assert(ed);
  assert(viewer);
  top10::math::Rotation3 R(rotation_step, top10::math::Vector(0.0, -1.0, 0.0));
  ed->setCurrentTangent( R*ed->getCurrentTangent() );
  
  viewer->updateGL();
  update();
}

void PathDlg::rotateTangentCW2()
{
  if (in_update) return;
  
  assert(ed);
  assert(viewer);
  top10::math::Rotation3 R(rotation_step, top10::math::Vector(0.0, -1.0, 0.0));
  ed->setCurrentTangent2( R*ed->getCurrentTangent2() );
  
  viewer->updateGL();
  update();
}

void PathDlg::setName()
{
  if (in_update) return;
  
  assert(ed);
  
  std::string s = Name_lineEdit->text().toStdString();
  ed->setCurrentName(s);
  update();
}

void PathDlg::on_BankAngle_doubleSpinBox_valueChanged(double val)
{
  if (in_update) return;
  
  assert(ed);
  
  ed->setCurrentBankAngle((float)val);
  viewer->updateGL();    
}

