/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include <QtGui/QtGui>
#include "MyMainWidow.hh"

#include "util/ValueHandle.hh"

MyMainWindow::MyMainWindow()
{
  // The main view
  trackView = new QGLView;
  meshes_dlg = new MeshLibDlg(trackView);
  setCentralWidget(trackView);

  // The log window
  logWindow = new QTextEdit;
  logWindow->setReadOnly(true);

  // Receives messages from the logger
  receiver = new StatusBarReceiver;
  receiver->setLogWindow(logWindow);

  // Put each dialog in a dock
  docks.push_back(new QDockWidget("Starting areas", this));
  docks.back()->setWidget(&sa_dlg);
  docks.push_back(new QDockWidget("Waypoints", this));
  docks.back()->setWidget(&path_dlg);
  docks.push_back(new QDockWidget("Sections", this));
  docks.back()->setWidget(&sections_dlg);
  docks.push_back(new QDockWidget("Checkpoints", this));
  docks.back()->setWidget(&cps_dlg);
  docks.push_back(new QDockWidget("Layout", this));
  docks.back()->setWidget(&layout_dlg);
  docks.push_back(new QDockWidget("Hulls", this));
  docks.back()->setWidget(&hull_dlg);
  docks.push_back(new QDockWidget("Mesh library", this));
  docks.back()->setWidget(meshes_dlg);
  docks.push_back(new QDockWidget("Triangulation", this));
  docks.back()->setWidget(&tri_dlg);
  docks.push_back(new QDockWidget("Textures", this));
  docks.back()->setWidget(&texture_dlg);
  docks.push_back(new QDockWidget("Log", this));
  docks.back()->setWidget(logWindow);
  docks.push_back(new QDockWidget("Distance", this));
  docks.back()->setWidget(&dist_dlg);
  docks.push_back(new QDockWidget("Mesh instances", this));
  docks.back()->setWidget(&mesh_instance_dlg);
  docks.push_back(new QDockWidget("Arc tool", this));
  docks.back()->setWidget(&arc_tool_dlg);
  docks.push_back(new QDockWidget("Walls", this));
  docks.back()->setWidget(&wall_dlg);

  for (std::list<QDockWidget*>::iterator it = docks.begin(); it != docks.end(); ++it) {
    (*it)->hide();
    (*it)->setFloating(true);
    addDockWidget(Qt::BottomDockWidgetArea, *it);
  }
  
  // Widgets in the main window, actions
  createActions();
  createMenus();
  createToolBars();
  createStatusBar();
  readSettings();
  setCurrentFile("");
  
  // Register all editors to the main editor
  main_ed.addEditor(&cps_ed);
  main_ed.addEditor(&sa_ed);
  main_ed.addEditor(&sections_ed);
  main_ed.addEditor(&path_ed);
  main_ed.addEditor(&hull_ed);
  main_ed.addEditor(&tri_ed);
  main_ed.addEditor(&layout_ed);
  main_ed.addEditor(&meshes_ed);
  main_ed.addEditor(&texture_ed);
  main_ed.addEditor(&mesh_instance_ed);
  main_ed.addEditor(&wall_ed);

  // Setup connections between editors
  path_ed.setSectionsEditor(&sections_ed);
  sections_ed.setWaypointsEditor(&path_ed);
  hull_ed.setWaypointEditor(&path_ed);
  tri_ed.setSectionsEditor(&sections_ed);
  mesh_instance_ed.setMeshEditor(&meshes_ed);
  wall_ed.setPathEditor(&path_ed);

  // Some editors need a camera to find which object is under the reticle
  camera = new top10::tracked::MainCamera;
  camera->setFar(10000.0);
  camera->translate(top10::math::Vector(0.0, 100.0, 0.0));
  camera->rotateX(-90.0);
  path_ed.setCamera(camera.getPtr());
  cps_ed.setCamera(camera.getPtr());
  sa_ed.setCamera(camera.getPtr());

  trackView->setMainCamera(camera.getPtr());
  trackView->addDrawable(&cps_ed);
  trackView->addDrawable(&sa_ed);
  trackView->addDrawable(&path_ed);
  trackView->addDrawable(&sections_ed);
  trackView->addDrawable(&layout_ed);
  trackView->addDrawable(&hull_ed);
  trackView->addDrawable(&tri_ed);
  trackView->addDrawable(&mesh_instance_ed);
  trackView->addDrawable(&wall_ed);

  // Needed to update the view after changes. TODO: The editors should actually take care of that
  cps_dlg.setViewer(trackView);
  sa_dlg.setViewer(trackView);
  path_dlg.setViewer(trackView);
  sections_dlg.setViewer(trackView);
  layout_dlg.setViewer(trackView);
  mesh_instance_dlg.setViewer(trackView);

  // Each dialog must be connected to its editor
  cps_dlg.setEditor(&cps_ed);
  sa_dlg.setEditor(&sa_ed);
  path_dlg.setEditor(&path_ed);
  sections_dlg.setEditor(&sections_ed);
  layout_dlg.setEditor(&layout_ed);
  hull_dlg.setEditor(&hull_ed);
  meshes_dlg->setEditor(&meshes_ed);
  tri_dlg.setEditor(&tri_ed);
  texture_dlg.setEditor(&texture_ed);
  dist_dlg.setEditor(&path_ed);
  mesh_instance_dlg.setMeshInstanceEditor(&mesh_instance_ed);
  arc_tool_dlg.setWaypointsEditor(&path_ed);
  wall_dlg.setEditor(&wall_ed);

  // Set the font to use to render text in the main view
  path_ed.setFont(trackView->getFont());

  updateDlgs();
  updateEditors();
  
  sections_listener = new top10::tracked::SectionsListener(&sections_ed);
  path_ed.addListener(sections_listener.getPtr());

  setOptionsDlg();
}

void MyMainWindow::closeEvent(QCloseEvent *event)
{
  if (maybeSave()) {
      writeSettings();
      event->accept();
  } else {
      event->ignore();
  }
  
  cps_dlg.close();
  sa_dlg.close();
  path_dlg.close();
  sections_dlg.close();
  layout_dlg.close();
  hull_dlg.close();
  meshes_dlg->close();
  tri_dlg.close();
  texture_dlg.close();
  dist_dlg.close();
  mesh_instance_dlg.close();
  options_dlg.close();
  arc_tool_dlg.close();
  wall_dlg.close();
}

void MyMainWindow::newFile()
{
  if (maybeSave()) {
    main_ed.clearAll();
    setCurrentFile("");
    updateEditors();
    updateDlgs();
  }
}

void MyMainWindow::open()
{
  if (maybeSave()) {
    QString fileName = QFileDialog::getOpenFileName(this);
    if (!fileName.isEmpty())
      loadFile(fileName);
  }
}

bool MyMainWindow::save()
{
  if (filename.empty()) {
    return saveAs();
  } else {
    return saveFile(filename.c_str());
  }
}

bool MyMainWindow::saveAs()
{
  QString fileName = QFileDialog::getSaveFileName(this);
  if (fileName.isEmpty())
    return false;

  return saveFile(fileName);
}

void MyMainWindow::about()
{
  QMessageBox::about(this, tr("TOP10 track editor"),
            tr("Copyright (C) 2005-2007 Johann Deneux <johann.deneux@gmail.com>"));
}

void MyMainWindow::showPrefs()
{
  options_dlg.show();
}

void MyMainWindow::documentWasModified()
{
  setWindowModified(true);
}

void MyMainWindow::createActions()
{
  newAct = new QAction(QIcon(":/filenew.png"), tr("&New"), this);
  newAct->setShortcut(tr("Ctrl+N"));
  newAct->setStatusTip(tr("Create a new file"));
  connect(newAct, SIGNAL(triggered()), this, SLOT(newFile()));

  openAct = new QAction(QIcon(":/fileopen.png"), tr("&Open..."), this);
  openAct->setShortcut(tr("Ctrl+O"));
  openAct->setStatusTip(tr("Open an existing file"));
  connect(openAct, SIGNAL(triggered()), this, SLOT(open()));

  saveAct = new QAction(QIcon(":/filesave.png"), tr("&Save"), this);
  saveAct->setShortcut(tr("Ctrl+S"));
  saveAct->setStatusTip(tr("Save the document to disk"));
  connect(saveAct, SIGNAL(triggered()), this, SLOT(save()));

  saveAsAct = new QAction(tr("Save &As..."), this);
  saveAsAct->setStatusTip(tr("Save the document under a new name"));
  connect(saveAsAct, SIGNAL(triggered()), this, SLOT(saveAs()));

  optionsAct = new QAction(tr("Preferences"), this);
  optionsAct->setStatusTip(tr("Set preferences"));
  connect(optionsAct, SIGNAL(triggered()), this, SLOT(showPrefs()));

  exitAct = new QAction(tr("E&xit"), this);
  exitAct->setShortcut(tr("Ctrl+Q"));
  exitAct->setStatusTip(tr("Exit the application"));
  connect(exitAct, SIGNAL(triggered()), this, SLOT(close()));

  aboutAct = new QAction(tr("&About"), this);
  aboutAct->setStatusTip(tr("Show the application's About box"));
  connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

  aboutQtAct = new QAction(tr("About &Qt"), this);
  aboutQtAct->setStatusTip(tr("Show the Qt library's About box"));
  connect(aboutQtAct, SIGNAL(triggered()), qApp, SLOT(aboutQt()));
  
  showStartAreaAct = new QAction(QIcon(":/startarea.png"), tr("Show Start Area"), this);
  showStartAreaAct->setStatusTip(tr("Show the starting area in the main view"));
  showStartAreaAct->setCheckable(true);
  connect(showStartAreaAct, SIGNAL(toggled(bool)), this, SLOT(showStartAreaView(bool)));
  
  showCheckpointsAct = new QAction(QIcon(":/checkpoint.png"), tr("Show checkpoints"), this);
  showCheckpointsAct->setStatusTip(tr("Show all checkpoints in the main view"));
  showCheckpointsAct->setCheckable(true);
  connect(showCheckpointsAct, SIGNAL(toggled(bool)), this, SLOT(showCheckpointsView(bool)));
  
  showPathAct = new QAction(QIcon(":/waypoint.png"), tr("Show waypoints"), this);
  showPathAct->setStatusTip(tr("Show waypoints in the main view"));
  showPathAct->setCheckable(true);
  connect(showPathAct, SIGNAL(toggled(bool)), this, SLOT(showPathView(bool)));
  showPathAct->setChecked(true);
  
  showSectionsAct = new QAction(QIcon(":/section.png"), tr("Show sections"), this);
  showSectionsAct->setStatusTip(tr("Show the sections in the main view"));
  showSectionsAct->setCheckable(true);
  connect(showSectionsAct, SIGNAL(toggled(bool)), this, SLOT(showSectionsView(bool)));
  showSectionsAct->setChecked(true);
  
  showLayoutAct = new QAction(QIcon(":/layout.png"), tr("Show layout"), this);
  showLayoutAct->setStatusTip(tr("Show the layout in the main view"));
  showLayoutAct->setCheckable(true);
  connect(showLayoutAct, SIGNAL(toggled(bool)), this, SLOT(showLayoutView(bool)));
  showLayoutAct->setChecked(true);
  
  showHullAct = new QAction(QIcon(":/hull_view.png"), tr("Show hulls"), this);
  showHullAct->setStatusTip(tr("Show the current hull in the main view"));
  showHullAct->setCheckable(true);
  connect(showHullAct, SIGNAL(toggled(bool)), this, SLOT(showHullView(bool)));
  showHullAct->setChecked(true);

  showTriangulationAct = new QAction(QIcon(":/triangulation.png"), tr("Show the triangulation"), this);
  showTriangulationAct->setStatusTip(tr("Show the triangulated track in the main view"));
  showTriangulationAct->setCheckable(true);
  connect(showTriangulationAct, SIGNAL(toggled(bool)), this, SLOT(showTriangulationView(bool)));
  showTriangulationAct->setChecked(false);
  
  updateOutlineAct = new QAction(QIcon(":/gear.png"), tr("Update outline"), this);
  updateOutlineAct->setStatusTip(tr("Update the outline of the track"));
  connect(updateOutlineAct, SIGNAL(triggered()), this, SLOT(updateOutline()));

  showMeshInstancesAct = new QAction(QIcon(":/mesh_insts.png"), tr("Show mesh instances"), this);
  showMeshInstancesAct->setStatusTip(tr("Show all mesh instances in the main view"));
  showMeshInstancesAct->setCheckable(true);
  connect(showMeshInstancesAct, SIGNAL(toggled(bool)), this, SLOT(showMeshInstancesView(bool)));
  showMeshInstancesAct->setChecked(true);

  showWallsAct = new QAction(tr("Show walls"), this);
  showWallsAct->setStatusTip(tr("Show walls"));
  showWallsAct->setCheckable(true);
  connect(showWallsAct, SIGNAL(toggled(bool)), this, SLOT(showWalls(bool)));
  showWallsAct->setChecked(true);
}

void MyMainWindow::createMenus()
{
    fileMenu = menuBar()->addMenu(tr("&File"));
    fileMenu->addAction(newAct);
    fileMenu->addAction(openAct);
    fileMenu->addAction(saveAct);
    fileMenu->addAction(saveAsAct);
    fileMenu->addAction(optionsAct);
    fileMenu->addSeparator();
    fileMenu->addAction(exitAct);

    menuBar()->addSeparator();

    toolsMenu = menuBar()->addMenu(tr("Tools"));
    for (std::list<QDockWidget*>::iterator it = docks.begin(); it != docks.end(); ++it)
      toolsMenu->addAction((*it)->toggleViewAction());
    
    menuBar()->addSeparator();
    
    viewMenu = menuBar()->addMenu(tr("Views"));
    viewMenu->addAction(showStartAreaAct);
    viewMenu->addAction(showCheckpointsAct);
    viewMenu->addAction(showPathAct);
    viewMenu->addAction(showSectionsAct);
    viewMenu->addAction(showLayoutAct);
    viewMenu->addAction(showHullAct);
    viewMenu->addAction(showTriangulationAct);
    viewMenu->addAction(showMeshInstancesAct);
    viewMenu->addAction(showWallsAct);
    viewMenu->addSeparator();
    viewMenu->addAction(updateOutlineAct);
    menuBar()->addSeparator();
    
    helpMenu = menuBar()->addMenu(tr("&Help"));
    helpMenu->addAction(aboutAct);
    helpMenu->addAction(aboutQtAct);
}

void MyMainWindow::createToolBars()
{
    fileToolBar = addToolBar(tr("File"));
    fileToolBar->addAction(newAct);
    fileToolBar->addAction(openAct);
    fileToolBar->addAction(saveAct);
        
    viewToolBar = addToolBar(tr("View"));
    viewToolBar->addAction(showStartAreaAct);
    viewToolBar->addAction(showCheckpointsAct);
    viewToolBar->addAction(showPathAct);
    viewToolBar->addAction(showSectionsAct);
    viewToolBar->addAction(showLayoutAct);
    viewToolBar->addAction(showHullAct);
    viewToolBar->addAction(showTriangulationAct);
//    viewToolBar->addAction(showMeshInstancesAct);
    viewToolBar->addSeparator();
    viewToolBar->addAction(updateOutlineAct);
}

void MyMainWindow::createStatusBar()
{
    statusBar()->showMessage(tr("Ready"));
    receiver->setStatusBar(statusBar());
    top10::util::Log::getSingle()->addReceiver(receiver.getPtr());
}

void MyMainWindow::readSettings()
{
  QSettings settings("Top10", "Track Editor");
  QPoint pos = settings.value("pos", QPoint(200, 200)).toPoint();
  QSize size = settings.value("size", QSize(400, 400)).toSize();
  resize(size);
  move(pos);

  trackView->pan_speed = settings.value("pan_speed", 50.0).toDouble();
  path_ed.label_point_size = settings.value("waypoints_label_size", 2.0).toDouble();
  path_ed.label_r = settings.value("waypoints_label_red", 255).toInt();
  path_ed.label_g = settings.value("waypoints_label_green", 255).toInt();
  path_ed.label_b = settings.value("waypoints_label_blue", 255).toInt();
}

void MyMainWindow::writeSettings()
{
    QSettings settings("Top10", "Track Editor");
    settings.setValue("pos", pos());
    settings.setValue("size", size());

    settings.setValue("pan_speed", trackView->pan_speed);
    settings.setValue("waypoints_label_size", path_ed.label_point_size);
    settings.setValue("waypoints_label_red", (int)path_ed.label_r);
    settings.setValue("waypoints_label_green", (int)path_ed.label_g);
    settings.setValue("waypoints_label_blue", (int)path_ed.label_b);
}

bool MyMainWindow::maybeSave()
{
    if (main_ed.needSave()) {
        int ret = QMessageBox::warning(this, tr("Top10 Track Editor"),
                      tr("The document has been modified.\n"
                        "Do you want to save your changes?"),
                      QMessageBox::Yes | QMessageBox::Default,
                      QMessageBox::No,
                      QMessageBox::Cancel | QMessageBox::Escape);
        if (ret == QMessageBox::Yes)
            return save();
        else if (ret == QMessageBox::Cancel)
            return false;
    }
    return true;
}

void MyMainWindow::updateDlgs()
{
  cps_dlg.update();
  sa_dlg.update();
  path_dlg.update();
  sections_dlg.update();
  layout_dlg.update();
  hull_dlg.update();
  meshes_dlg->update();
  tri_dlg.update();
  texture_dlg.update();
  dist_dlg.populate();
  dist_dlg.computeDist();
  mesh_instance_dlg.update();
  wall_dlg.update();
}

void MyMainWindow::updateEditors()
{
  sections_ed.updateView();
  tri_ed.updateView();
}

void MyMainWindow::loadFile(const QString &fileName)
{
  try {
    trackView->makeCurrent();
    main_ed.loadXml(fileName.toStdString());
    updateDlgs();
    updateEditors();
    setCurrentFile(fileName);
    statusBar()->showMessage(tr("File loaded"), 2000);
  }
  catch (std::string err) {
    statusBar()->showMessage(err.c_str(), 2000);
  }
  catch (...) {
    statusBar()->showMessage(tr("Unknown error"), 2000);
  }
}

bool MyMainWindow::saveFile(const QString &fileName)
{
  try {
    QApplication::setOverrideCursor(Qt::WaitCursor);
    main_ed.saveXml(fileName.toStdString());
    QApplication::restoreOverrideCursor();
    setCurrentFile(fileName);
    statusBar()->showMessage(tr("File saved"), 2000);
  }
  catch (std::string err) {
    statusBar()->showMessage(err.c_str(), 2000);
    return false;
  }
  catch (...) {
    statusBar()->showMessage(tr("Unknown error"), 2000);
    return false;
  }
  
  return true;
}

void MyMainWindow::setCurrentFile(const QString &fileName)
{
  filename = fileName.toStdString();
  
  main_ed.setNeedSave(false);  
  setWindowModified(false);

  QString shownName;
  if (filename.empty())
    shownName = "untitled.xml";
  else
    shownName = strippedName(fileName);

  setWindowTitle(tr("%1[*] - %2").arg(shownName).arg(tr("Top10 track editor")));
}

QString MyMainWindow::strippedName(const QString &fullFileName)
{
  return QFileInfo(fullFileName).fileName();
}

void MyMainWindow::showCheckpointsView(bool b)
{
  cps_ed.setVisible(b);    
}

void MyMainWindow::showStartAreaView(bool b)
{
  sa_ed.setVisible(b);
}

void MyMainWindow::showPathView(bool b)
{
  path_ed.setVisible(b);
}

void MyMainWindow::showSectionsView(bool b)
{
  sections_ed.setVisible(b);
}

void MyMainWindow::showLayoutView(bool b)
{
  layout_ed.setVisible(b);
}

void MyMainWindow::showHullView(bool b)
{
  hull_ed.setVisible(b);
}

void MyMainWindow::showTriangulationView(bool b)
{
  tri_ed.setVisible(b);
}

void MyMainWindow::showMeshInstancesView(bool b)
{
  mesh_instance_ed.setVisible(b);
}

void MyMainWindow::showWalls(bool b)
{
  wall_ed.setVisible(b);
}

void MyMainWindow::updateOutline()
{
  tri_ed.updateView();
}

void MyMainWindow::setOptionsDlg()
{
  using top10::util::DoubleValueHandle;
  using top10::util::ColorValueHandle;

  DoubleValueHandle* d_handle;
  ColorValueHandle* c_handle;

  d_handle = new DoubleValueHandle(&(trackView->pan_speed), 50.0);
  d_handle->setName("Pan speed");
  d_handle->setDescription("Affects the speed at which the camera in the main view is translated."
    " Comes into effect immediately.");
  options_dlg.addValueHandle(d_handle);

  d_handle = new DoubleValueHandle(&(path_ed.label_point_size), 2.0);
  d_handle->setName("Waypoints label size");
  d_handle->setDescription("Sets the size of labels showing the name of waypoints in the main view."
    " Comes into effect after next change to waypoints.");
  options_dlg.addValueHandle(d_handle);

  c_handle = new ColorValueHandle(&(path_ed.label_r), &(path_ed.label_g), &(path_ed.label_b),
                                  255, 100, 200);
  c_handle->setName("Waypoints label color");
  c_handle->setDescription("Sets the red, green and blue components (0-255) of the color of waypoints labels."
    " Comes into effect after next change to waypoints.");
  options_dlg.addValueHandle(c_handle);
}
