/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include "MeshLibDlg.hh"
#include <QtGui/QVBoxLayout>
#include <QtGui/QCheckBox>

MeshLibDlg::MeshLibDlg(QGLWidget* share):
  mesh_ed(0),
  view(0, share)
{
  setupUi(this);
  connect(Name_comboBox->lineEdit(), SIGNAL(editingFinished()), this, SLOT(_on_Name_comboBox_textEdited()));
  
  while (tabWidget->count() > 0)
    tabWidget->removeTab(0);

  tabWidget->addTab(&view, tr("View"));

  Show_front_CheckBox = new QCheckBox(tr("Show front faces"));
  Show_back_CheckBox = new QCheckBox(tr("Show back faces"));
  Shadow_CheckBox = new QCheckBox(tr("Enable shadow volumes"));
  Collision_CheckBox = new QCheckBox(tr("Enable collisions"));

  QLayout* properties_layout = new QVBoxLayout;
  properties_layout->addWidget(Show_front_CheckBox);
  properties_layout->addWidget(Show_back_CheckBox);
  properties_layout->addWidget(Shadow_CheckBox);
  properties_layout->addWidget(Collision_CheckBox);

  QWidget* Properties_widget = new QWidget;
  Properties_widget->setLayout(properties_layout);

  tabWidget->addTab(Properties_widget, tr("Properties"));

  connect( Show_front_CheckBox, SIGNAL(stateChanged(int)), this, SLOT(on_Show_front_CheckBox_stateChanged(int)) );
  connect( Show_back_CheckBox, SIGNAL(stateChanged(int)), this, SLOT(on_Show_back_CheckBox_stateChanged(int)) );
  connect( Shadow_CheckBox, SIGNAL(stateChanged(int)), this, SLOT(on_Shadow_CheckBox_stateChanged(int)) );
  connect( Collision_CheckBox, SIGNAL(stateChanged(int)), this, SLOT(on_Collision_CheckBox_stateChanged(int)) );
}


MeshLibDlg::~MeshLibDlg()
{
}

void MeshLibDlg::_on_Name_comboBox_textEdited()
{
  QString new_name = Name_comboBox->lineEdit()->text();
  if (!new_name.isEmpty())
    mesh_ed->setName(new_name.toStdString());
}

void MeshLibDlg::on_Name_comboBox_activated( const QString& name )
{
  mesh_ed->gotoMesh(name.toStdString());
  update();
}

void MeshLibDlg::set3dsTransform()
{
  top10::math::Matrix4 T;
  T(0,0) = -1.0;
  T(2,1) = 1.0;
  T(1,2) = 1.0;
  T(3,3) = 1.0;
  mesh_ed->setTransform(T);
}

void MeshLibDlg::on_Filename_lineEdit_editingFinished( )
{
  std::string filename = Filename_lineEdit->text().toStdString();
  if (!filename.empty())
  {
    mesh_ed->loadMesh(filename);
    
    /* If we are loading a 3ds file and the transform matrix is the identity,
    we change the transformation matrix so that the Z of the object is the vertical axis */
    if (filename.rfind(".3ds") == filename.size()-4 &&
        mesh_ed->getTransform() == top10::math::Identity4())
      set3dsTransform();

    update();
  }
}

void MeshLibDlg::on_View_radioButton_toggled( bool b )
{
  if (b)
    view.setMode(QGLMeshView::ModelView);
}

void MeshLibDlg::on_TranslateXZ_radioButton_toggled( bool b )
{
  if (b)
    view.setMode(QGLMeshView::ModelTranslateXZ);
}

void MeshLibDlg::on_TranslateXY_radioButton_toggled( bool b )
{
  if (b)
    view.setMode(QGLMeshView::ModelTranslateXY);
}

void MeshLibDlg::on_RotateX_radioButton_toggled( bool b )
{
  if (b)
    view.setMode(QGLMeshView::ModelRotateX);
}

void MeshLibDlg::on_RotateY_radioButton_toggled( bool b )
{
  if (b)
    view.setMode(QGLMeshView::ModelRotateY);
}

void MeshLibDlg::on_RotateZ_radioButton_toggled( bool b )
{
  if (b)
    view.setMode(QGLMeshView::ModelRotateZ);
}

void MeshLibDlg::on_FS_pushButton_pressed( )
{
  QString fileName = QFileDialog::getOpenFileName(this);
  std::string filename = fileName.toStdString();
  if (!filename.empty()) {
    mesh_ed->loadMesh(filename);

    /* If we are loading a 3ds file and the transform matrix is the identity,
    we change the transformation matrix so that the Z of the object is the vertical axis */
    if (filename.rfind(".3ds") == filename.size()-4 &&
        mesh_ed->getTransform() == top10::math::Identity4())
      set3dsTransform();

    update();
  }
}

/* You gotta love GUI code. It's beautiful, original and non-repetetive
*/

void MeshLibDlg::on_M00_lineEdit_editingFinished( )
{
  transformEdited(0, 0, M00_lineEdit->text());
}

void MeshLibDlg::on_M10_lineEdit_editingFinished( )
{
  transformEdited(1, 0, M10_lineEdit->text());
}

void MeshLibDlg::on_M20_lineEdit_editingFinished( )
{
  transformEdited(2, 0, M20_lineEdit->text());
}

void MeshLibDlg::on_M01_lineEdit_editingFinished( )
{
  transformEdited(0, 1, M01_lineEdit->text());
}

void MeshLibDlg::on_M11_lineEdit_editingFinished( )
{
  transformEdited(1, 1, M11_lineEdit->text());
}

void MeshLibDlg::on_M21_lineEdit_editingFinished( )
{
  transformEdited(2, 1, M21_lineEdit->text());
}

void MeshLibDlg::on_M02_lineEdit_editingFinished( )
{
  transformEdited(0, 2, M02_lineEdit->text());
}

void MeshLibDlg::on_M12_lineEdit_editingFinished( )
{
  transformEdited(1, 2, M12_lineEdit->text());
}

void MeshLibDlg::on_M22_lineEdit_editingFinished( )
{
  transformEdited(2, 2, M22_lineEdit->text());
}

void MeshLibDlg::on_M03_lineEdit_editingFinished( )
{
  transformEdited(0, 3, M03_lineEdit->text());
}

void MeshLibDlg::on_M13_lineEdit_editingFinished( )
{
  transformEdited(1, 3, M13_lineEdit->text());
}

void MeshLibDlg::on_M23_lineEdit_editingFinished( )
{
  transformEdited(2, 3, M23_lineEdit->text());
}

void MeshLibDlg::transformEdited( int i, int j, const QString & text )
{
  double x;
  bool status = false;
  x = text.toDouble(&status);
  if (status)
  {
    top10::math::Matrix4 M = mesh_ed->getTransform();
    M(i,j) = x;
    try {
      mesh_ed->setTransform(M);
      view.updateGL();
    }
    catch(top10::math::NonInvertible<4>& e) {}
  }
}

void MeshLibDlg::on_Show_front_CheckBox_stateChanged(int s)
{
  if (s == Qt::Checked)
    mesh_ed->setFlag(top10::helpers::MeshEditor::FACE_SHOW_FRONT);
  else
    mesh_ed->clearFlag(top10::helpers::MeshEditor::FACE_SHOW_FRONT);

  view.updateGL();
}

void MeshLibDlg::on_Show_back_CheckBox_stateChanged(int s)
{
  if (s == Qt::Checked)
    mesh_ed->setFlag(top10::helpers::MeshEditor::FACE_SHOW_BACK);
  else
    mesh_ed->clearFlag(top10::helpers::MeshEditor::FACE_SHOW_BACK);

  view.updateGL();
}

void MeshLibDlg::on_Shadow_CheckBox_stateChanged(int s)
{
  if (s == Qt::Checked)
    mesh_ed->setFlag(top10::helpers::MeshEditor::SHADOW_VOLUME);
  else
    mesh_ed->clearFlag(top10::helpers::MeshEditor::SHADOW_VOLUME);
}

void MeshLibDlg::on_Collision_CheckBox_stateChanged(int s)
{
  if (s == Qt::Checked)
    mesh_ed->clearFlag(top10::helpers::MeshEditor::NO_COLLISION);
  else
    mesh_ed->setFlag(top10::helpers::MeshEditor::NO_COLLISION);
}

void MeshLibDlg::update( )
{
  if (!mesh_ed || mesh_ed->getMeshName().empty()) {
    M00_lineEdit->setEnabled(false);
    M10_lineEdit->setEnabled(false);
    M20_lineEdit->setEnabled(false);
    
    M01_lineEdit->setEnabled(false);
    M11_lineEdit->setEnabled(false);
    M21_lineEdit->setEnabled(false);
    
    M02_lineEdit->setEnabled(false);
    M12_lineEdit->setEnabled(false);
    M22_lineEdit->setEnabled(false);
    
    M03_lineEdit->setEnabled(false);
    M13_lineEdit->setEnabled(false);
    M23_lineEdit->setEnabled(false);
    
    Filename_lineEdit->setEnabled(false);
    FS_pushButton->setEnabled(false);

    Show_front_CheckBox->setEnabled(false);
    Show_back_CheckBox->setEnabled(false);
    Shadow_CheckBox->setEnabled(false);
    Collision_CheckBox->setEnabled(false);
  }
  else {
    M00_lineEdit->setEnabled(true);
    M10_lineEdit->setEnabled(true);
    M20_lineEdit->setEnabled(true);
    
    M01_lineEdit->setEnabled(true);
    M11_lineEdit->setEnabled(true);
    M21_lineEdit->setEnabled(true);
    
    M02_lineEdit->setEnabled(true);
    M12_lineEdit->setEnabled(true);
    M22_lineEdit->setEnabled(true);
    
    M03_lineEdit->setEnabled(true);
    M13_lineEdit->setEnabled(true);
    M23_lineEdit->setEnabled(true);
    
    Filename_lineEdit->setEnabled(true);
    FS_pushButton->setEnabled(true);

    Show_front_CheckBox->setEnabled(true);
    Show_back_CheckBox->setEnabled(true);
    Shadow_CheckBox->setEnabled(true);
    Collision_CheckBox->setEnabled(true);
  }
  
  updateMatrix();
  updateMeshName();
  updateFileName();
  view.updateGL();
  updateProperties();
}

void MeshLibDlg::updateMatrix( )
{
  top10::math::Matrix4 M = mesh_ed->getTransform();
  
  M00_lineEdit->setText(QString().setNum(M(0,0)));
  M10_lineEdit->setText(QString().setNum(M(1,0)));
  M20_lineEdit->setText(QString().setNum(M(2,0)));
  
  M01_lineEdit->setText(QString().setNum(M(0,1)));
  M11_lineEdit->setText(QString().setNum(M(1,1)));
  M21_lineEdit->setText(QString().setNum(M(2,1)));
  
  M02_lineEdit->setText(QString().setNum(M(0,2)));
  M12_lineEdit->setText(QString().setNum(M(1,2)));
  M22_lineEdit->setText(QString().setNum(M(2,2)));
  
  M03_lineEdit->setText(QString().setNum(M(0,3)));
  M13_lineEdit->setText(QString().setNum(M(1,3)));
  M23_lineEdit->setText(QString().setNum(M(2,3)));
}

void MeshLibDlg::updateMeshName( )
{
  Name_comboBox->clear();
  std::list<std::string> names = mesh_ed->getMeshNames();
  for (std::list<std::string>::const_iterator it = names.begin(); it != names.end(); ++it)
  {
    Name_comboBox->addItem(it->c_str());
  }
  int idx = Name_comboBox->findText(QString(mesh_ed->getMeshName().c_str()));
  if (idx != -1) Name_comboBox->setCurrentIndex(idx);
}

void MeshLibDlg::updateFileName( )
{
  Filename_lineEdit->setText(QString(mesh_ed->getMeshFilename().c_str()));
}

void MeshLibDlg::updateProperties()
{
  if (mesh_ed->getFlags() & top10::helpers::MeshEditor::FACE_SHOW_FRONT)
    Show_front_CheckBox->setCheckState( Qt::Checked );
  else
    Show_front_CheckBox->setCheckState( Qt::Unchecked );

  if (mesh_ed->getFlags() & top10::helpers::MeshEditor::FACE_SHOW_BACK)
    Show_back_CheckBox->setCheckState( Qt::Checked );
  else
    Show_back_CheckBox->setCheckState( Qt::Unchecked );

  if (mesh_ed->getFlags() & top10::helpers::MeshEditor::SHADOW_VOLUME)
    Shadow_CheckBox->setCheckState( Qt::Checked );
  else
    Shadow_CheckBox->setCheckState( Qt::Unchecked );

  if (mesh_ed->getFlags() & top10::helpers::MeshEditor::NO_COLLISION)
    Collision_CheckBox->setCheckState( Qt::Unchecked );
  else
    Collision_CheckBox->setCheckState( Qt::Checked );
}

void MeshLibDlg::on_Add_pushButton_pressed( )
{
  mesh_ed->addMesh();
  update();
}

void MeshLibDlg::on_Del_pushButton_pressed( )
{
  mesh_ed->delMesh();
  update();
}

MeshLibDlg::Listener::Listener(MeshLibDlg* view):
view(view)
{
}

void MeshLibDlg::Listener::notifyTransform(const top10::math::Matrix4& M)
{
  // We assume M and view->mesh_ed->getTransform() are identical.
  view->updateMatrix();
}

void MeshLibDlg::setEditor(top10::helpers::MeshEditor *ed)
{
  mesh_ed = ed;
  view.setMeshEditor(ed);
  ed->addListener(new Listener(this));
}